
#include <linux/clk.h>
#include <linux/io.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <mach/platform.h>
#include <mach/gpio.h>
#include <asm/atomic.h>
#include <linux/spinlock.h>


#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/soc.h>
#include <sound/soc-dapm.h>
#include <linux/proc_fs.h>
#include <sound/dmw-tdm.h>
#include <sound/gxp22xx-boards.h>
#include <linux/delay.h>
#include "dmw-tdm-mstr.h"
#include "dmw-pcm.h"
#include "../codecs/tlv320aic3x.h"
#ifdef CONFIG_SND_DMW_SOC_BOARD_CSR8811
#include "../codecs/csr8811.h"
#endif
#define DRIVER_NAME "snd-gxp22xx-mach-drv"
#define DEVICE_NAME "soc-audio"

//#define V_DEBUG 
/*****************************************************************************/
#undef pr_debug
#define pr_debug(fmt, ...) \
	printk(KERN_DEBUG pr_fmt(fmt), ##__VA_ARGS__)

#ifdef V_DEBUG
	#ifndef DEBUG 
		#define DEBUG
	#endif
	#define vdbg_prt(fmt, ...) \
		pr_debug("gxv22xx-aic asoc Mach-drv: %s(): " fmt, __func__, ##__VA_ARGS__)
#else
	#define vdbg_prt(fmt, ...)
#endif


#ifdef DEBUG
	#define dbg_prt(fmt, ...) \
		pr_debug("gxv22xx-aic asoc Mach-drv: %s(): " fmt, __func__, ##__VA_ARGS__)

	#define DIR_STR(stream_dir)  ((stream_dir == 0) ? "for Playback" : "for Capture")
	
#else
	#define dbg_prt(fmt, ...)	
	#define DIR_STR(stream_dir)
#endif

struct gxp_snd_gpios
{
	uint mic_sw;
	uint speak_sw;
};

static struct gxp_snd_gpios  g_brd_snd_gpios;

static int g_dmw_tel_state = 0;

static void dmw_set_tel_state(void);

struct snd_soc_codec *g_codec;

/*****************************************************************************/
#define GXV22XX_HDS_AMP_GPIO 0
#define GXV22XX_SPK_AMP_GPIO GPIO_PORTF(28)
#define GXV22XX_MIC_AMP_GPIO GPIO_PORTF(6)


#define CODEC_CLOCK 	12000000
#ifdef CONFIG_SOC_CODEC_TDM_MASTER
#define AUDIO_FORMAT (SND_SOC_DAIFMT_I2S | \
	SND_SOC_DAIFMT_CBM_CFM |\
	SND_SOC_DAIFMT_NB_NF)
#else
	#define AUDIO_FORMAT (SND_SOC_DAIFMT_I2S | \
	SND_SOC_DAIFMT_CBS_CFS |\
	SND_SOC_DAIFMT_NB_NF)
#endif

static struct clk * g_machdrv_dp_clk = NULL;
static atomic_t g_dp_clk_opencnt = { 0 };
static struct timer_list sw_delay;
static struct work_struct restore_unmuted;
/* Note: dp clock reference-counting here is for debugging purpose.
 *	 Otherwise, the clock framework's internal reference counting
 *	 is adequate.
 */
static void brd_dp_clk_enable(void)
{
	uint clk_ref_after_incr;

	dbg_prt();

	clk_ref_after_incr = atomic_inc_return(&g_dp_clk_opencnt);
	if (clk_ref_after_incr - 1 == 0) {
		vdbg_prt("%s %u\n", "Enabling DP clock. Ref-count after:",
			clk_ref_after_incr);

		clk_enable(g_machdrv_dp_clk);
	} else {
		vdbg_prt("%s %u\n",
			"DP clock already Enabled. Ref-count after:",
			clk_ref_after_incr);
	}
}

static void brd_dp_clk_disable(void)
{
	uint clk_ref_after_decr;

	dbg_prt();

	clk_ref_after_decr = atomic_dec_return(&g_dp_clk_opencnt);
	if (clk_ref_after_decr == 0) {
		vdbg_prt("%s %u\n", "Disabling DP clock. Ref-count after:",
			clk_ref_after_decr);

		clk_disable(g_machdrv_dp_clk);
	} else {
		vdbg_prt("%s %u\n", "DP clock still On. Ref-count after:",
			clk_ref_after_decr);
	}
}

static int brd_gxv22xx_startup(struct snd_pcm_substream *substream)
{
	dbg_prt("%s\n", DIR_STR(substream->stream));
	
	brd_dp_clk_enable();

	return 0;
}

static void brd_gxv22xx_shutdown(struct snd_pcm_substream *substream)
{
	dbg_prt("%s\n", DIR_STR(substream->stream));

	brd_dp_clk_disable();
}

static int check_valid_board( void )
{
	dbg_prt();

	/* Check for EVB96 board with Wolfson chip WM8976 */
	if (dmw_board_is("gxp22xx")) {
		printk("Assuming wm8976 sound device on evb96 board\n");
		return true;
	}

	return false;
}

static int gxp22xx_brd_init_sound_gpios(struct gxp22xx_board_pdata *pgpio)
{
	int ret;
	if(pgpio == NULL)
	{
		return -ENXIO;
	
	}
	/*default speank on*/
	ret = gpio_request(pgpio->speak_sw, DEVICE_NAME);
	if (ret != 0)
		return  -EBUSY;
	gpio_direction_output(pgpio->speak_sw, 1);
	g_brd_snd_gpios.speak_sw = pgpio->speak_sw;
	/*default mic input handset*/
	ret = gpio_request(pgpio->mic_sw, DEVICE_NAME);
	if (ret != 0)
		return -EBUSY;
	gpio_direction_output(pgpio->mic_sw, 1);
	g_brd_snd_gpios.mic_sw = pgpio->mic_sw;
	return 0;
}

static int brd_gxv22xx_hw_params(struct snd_pcm_substream *substream,
				 struct snd_pcm_hw_params *params)
{
		struct snd_soc_pcm_runtime *rtd = substream->private_data;
		struct snd_soc_dai *codec_dai = rtd->codec_dai;
		struct snd_soc_dai *cpu_dai = rtd->cpu_dai;
		int ret;
		/* Set codec DAI configuration */
		ret = snd_soc_dai_set_fmt(codec_dai, AUDIO_FORMAT);
		if (ret < 0) {
			printk(KERN_ERR "can't set codec DAI configuration\n");
			return ret;
		}
	
		/* Set cpu DAI configuration */
		ret = snd_soc_dai_set_fmt(cpu_dai, AUDIO_FORMAT);
		if (ret < 0) {
			printk(KERN_ERR "can't set cpu DAI configuration\n");
			return ret;
		}
	
		/* Set the codec system clock for DAC and ADC */
		ret = snd_soc_dai_set_sysclk(codec_dai, 0,
				CODEC_CLOCK, SND_SOC_CLOCK_OUT);
		if (ret < 0) {
			printk(KERN_ERR "can't set codec system clock\n");
			return ret;
		}
		return 0;

}

static int brd_gxv22xx_prepare(struct snd_pcm_substream *substream)
{
	dbg_prt("%s\n", DIR_STR(substream->stream));
	vdbg_prt("%s\n", "Setting Tel state");
	dmw_set_tel_state();
	return 0;
}

/*****************************************************************************/

static const char *switch_function[] = {"Off", "On"};
static const struct soc_enum tel_enum = SOC_ENUM_SINGLE_EXT(2, switch_function);
static const struct soc_enum mic_enum = SOC_ENUM_SINGLE_EXT(2, switch_function);
static const char *micr_sel_st[] = {"Differential", "Single-ended"};
static const struct soc_enum micr_sel_enum = SOC_ENUM_SINGLE_EXT(2, micr_sel_st);

static int dmw_get_tel(struct snd_kcontrol *kcontrol,
		       struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = g_dmw_tel_state;

	vdbg_prt("%s %s\n", "TEL state is:", (g_dmw_tel_state == 0) ? "Off" : "On");
	
	return 0;
}

static int dmw_set_tel(struct snd_kcontrol *kcontrol,
		       struct snd_ctl_elem_value *ucontrol)
{
	if (g_dmw_tel_state == ucontrol->value.integer.value[0]) {
		vdbg_prt("%s %s\n", "TEL state Unchanged:", (g_dmw_tel_state == 0) ? "Off" : "On");
		return 0;
	}

	g_dmw_tel_state = ucontrol->value.integer.value[0];

	vdbg_prt("%s %s\n", "Updating TEL state to:", (g_dmw_tel_state == 0) ? "Off" : "On");
	
	return 0;
}

/* Input Analog Switch: */
static int gxp_get_mic_sel(struct snd_kcontrol *kcontrol,
                    struct snd_ctl_elem_value *ucontrol)
{
        int mic_sel_st;
        const uint p_gpio = g_brd_snd_gpios.mic_sw;
        
        if (p_gpio == -1) {
                vdbg_prt("%s\n", "Mic selection is not supported in this board:");
                return 0;
        }

        mic_sel_st = gpio_get_value(p_gpio);
        vdbg_prt("%s %s\n", "Mic selection is:",
                micr_sel_st[mic_sel_st]);
        ucontrol->value.integer.value[0] = mic_sel_st;
        return 0;
        
}

static int gxp_set_mic_sel(struct snd_kcontrol *kcontrol,
                    struct snd_ctl_elem_value *ucontrol)
{
        int mic_sel_st;
        const uint p_gpio = g_brd_snd_gpios.mic_sw;
        if (p_gpio == -1) {
                vdbg_prt("%s\n", "Mic selection is not supported in this board:");
                return 0;
        }
        mic_sel_st = gpio_get_value(p_gpio);
        if (mic_sel_st == ucontrol->value.integer.value[0]) {
                vdbg_prt("%s %s\n", "Mic selection Unchanged:",
                        micr_sel_st[mic_sel_st]);
                return 0;
        }
        
        mic_sel_st = ucontrol->value.integer.value[0];
        vdbg_prt("%s %s\n", "Setting Mic selection to:",
                micr_sel_st[mic_sel_st]);
	if(g_codec)
	{
		snd_soc_write(g_codec, 15, 0x80|snd_soc_read(g_codec, 15));
		mdelay(1);
		snd_soc_write(g_codec, 16, 0x80|snd_soc_read(g_codec, 16));
		mdelay(1);

	}

        gpio_set_value(p_gpio, mic_sel_st);
	if(g_codec)
	{
		//mdelay(1000);
		//snd_soc_write(g_codec, 15, 0x7f&snd_soc_read(g_codec, 15));
		//snd_soc_write(g_codec, 16, 0x7f&snd_soc_read(g_codec, 16));
		//mdelay(10);


		mod_timer(&sw_delay, jiffies + msecs_to_jiffies(500));
	}
        return 1;

}
static int gxp_get_mic1_pwr(struct snd_kcontrol *kcontrol,
                    struct snd_ctl_elem_value *ucontrol)
{
	int value = 0;
       	if(g_codec)
	{
		value = snd_soc_read(g_codec, 15);
		printk("value %0x\n", value);
		ucontrol->value.integer.value[0] = (value & 0x80)?1:0;
	}
	return 0;
}

static int gxp_set_mic1_pwr(struct snd_kcontrol *kcontrol,
                    struct snd_ctl_elem_value *ucontrol)
{
	int value = 0;
       	if(g_codec)
	{
		value = snd_soc_read(g_codec, 15);
		printk("value %0x , set %d\n", value, ucontrol->value.integer.value[0]);
		mdelay(1);
		if(ucontrol->value.integer.value[0])
			snd_soc_write(g_codec, 15, value|0x80);
		else
		{
			snd_soc_write(g_codec, 15, value&(~0x80));
		}	
	}
	return 0;

}
static int gxp_get_mic2_pwr(struct snd_kcontrol *kcontrol,
                    struct snd_ctl_elem_value *ucontrol)
{
        int value = 0;
       	if(g_codec)
	{
		value = snd_soc_read(g_codec, 16);
		ucontrol->value.integer.value[0] = (value & 0x80)?1:0;
	}
	return 0;

}

static int gxp_set_mic2_pwr(struct snd_kcontrol *kcontrol,
                    struct snd_ctl_elem_value *ucontrol)
{
	int value = 0;
       	if(g_codec)
	{
		value = snd_soc_read(g_codec, 16);
		
		if(ucontrol->value.integer.value[0])
			snd_soc_write(g_codec, 16, value|0x80);
		else
			snd_soc_write(g_codec, 16, value&(~0x80));
			
	}

	return 0;
}

static const struct snd_kcontrol_new gxp_controls[] = {
	SOC_ENUM_EXT("Telephony", tel_enum, dmw_get_tel, dmw_set_tel),
	SOC_ENUM_EXT("Mic-Type-Select", micr_sel_enum, gxp_get_mic_sel, gxp_set_mic_sel),
	SOC_ENUM_EXT("Mic1-mute", mic_enum, gxp_get_mic1_pwr, gxp_set_mic1_pwr),
	SOC_ENUM_EXT("Mic2-mute", mic_enum, gxp_get_mic2_pwr, gxp_set_mic2_pwr),
};

static int gxv22xx_spk_event(struct snd_soc_dapm_widget *w,
			  struct snd_kcontrol *k, int event)
{

	if (SND_SOC_DAPM_EVENT_ON(event))
	{
		gpio_direction_output(GXV22XX_SPK_AMP_GPIO, 1);
	}
	else
	{
		gpio_direction_output(GXV22XX_SPK_AMP_GPIO, 0);
	}
	return 0;
}

static int gxv22xx_hds_event(struct snd_soc_dapm_widget *w,
			  struct snd_kcontrol *k, int event)
{
	if (SND_SOC_DAPM_EVENT_ON(event))
	{
		gpio_direction_output(GXV22XX_HDS_AMP_GPIO, 1);
	}
	else
	{
		gpio_direction_output(GXV22XX_HDS_AMP_GPIO, 0);
	}
	return 0;
}
#if 0
static int gxv22xx_mic_event(struct snd_soc_dapm_widget *w,
			  struct snd_kcontrol *k, int event)
{
	int tmp_value;
	if(g_codec)
	{
		tmp_value=snd_soc_read(g_codec, 24);
		snd_soc_write(g_codec, 24, (tmp_value&0x83)|(0x8<<2));
	}
	
	if (SND_SOC_DAPM_EVENT_ON(event))
	{
		gpio_direction_output(GXV22XX_MIC_AMP_GPIO, 1);
	}
	else
	{
		gpio_direction_output(GXV22XX_MIC_AMP_GPIO, 0);
	}
	mdelay(200);
	snd_soc_write(g_codec, 24, tmp_value);
	snd_soc_write(g_codec, 19, 0x4); 
	return 0;
}
#endif
static const struct snd_soc_dapm_widget tlv320aic3x_dapm_widgets[] = {
	//SND_SOC_DAPM_HP("Headset", gxv22xx_hds_event),
	SND_SOC_DAPM_SPK("Speaker", gxv22xx_spk_event),
	//SND_SOC_DAPM_MIC("Mic", gxv22xx_mic_event),
	//SND_SOC_DAPM_MIC("Hd_Mic", NULL),
};

static const struct snd_soc_dapm_route audio_map[] = {
	
	//{"Headphone", NULL, "HPLOUT"},
	//{"Headphone", NULL, "HPROUT"},
	//{"Headset", NULL, "LLOUT"},
	{"Speaker", NULL, "RLOUT"},
	//{"Mic", NULL, "LINE1L"},
	//{"Hd_Mic", NULL, "LINE1R"},
	
	//{"Speaker", NULL, "LLOUT"},
	//{"Speaker", NULL, "Right Line Out"},
};

/*****************************************************************************/

static struct snd_soc_ops brd_gxv22xx_ops = {
	.hw_params = brd_gxv22xx_hw_params,
	.prepare   = brd_gxv22xx_prepare,
};


static int brd_gxv22xx_dai_link_init(struct snd_soc_pcm_runtime *rtd)
{
	int err;
 	struct snd_soc_codec *codec = rtd->codec;
	struct snd_soc_dapm_context * dapm = &codec->dapm;

	/* Note: this is the "probe" function */
	vdbg_prt("%s\n", "Adding Telephony control");
	err = snd_soc_add_controls(codec, gxp_controls,
				ARRAY_SIZE(gxp_controls));
	if (err < 0) {
		return err;
	}

	vdbg_prt("%s\n", "Adding dapm controls brd_dapm_widgets");
	err = snd_soc_dapm_new_controls(dapm, tlv320aic3x_dapm_widgets,
					ARRAY_SIZE(tlv320aic3x_dapm_widgets));
	if (err < 0) {
		return err;
	}

	vdbg_prt("%s\n", "Adding dapm routes brd_audio_route");
	err = snd_soc_dapm_add_routes(dapm, audio_map,
				      ARRAY_SIZE(audio_map));
	if (err < 0) {
		return err;
	}
	g_codec = codec;
	return 0;
}

/*****************************************************************************/

#ifdef CONFIG_SND_DMW_SOC_BOARD_CSR8811
static int brd_csr8811_hw_params(struct snd_pcm_substream *substream,
				 struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *codec_dai = rtd->codec_dai;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;

	unsigned int dai_format;
	struct clk * dp_clk = NULL;
	unsigned long dp_clk_rate;
	int ret;

#ifndef CONFIG_SOC_CODEC_TDM_MASTER
	uint alsa_Fs;
	struct snd_soc_codec *codec = rtd->codec;
	struct snd_soc_card *card = codec->card;
	struct tdm_master_coeff * coeff;
	unsigned long act_dp_clk_rate;
#endif /* ! CONFIG_SOC_CODEC_TDM_MASTER */

	dbg_prt("%s\n", DIR_STR(substream->stream));

	/*
	 * Note: dmw96 TDM-machine HW limitations:
	 *
	 *  - as Master:
	 *    ----------
	 *  You can configure the timing of the Rising edge of the generated
	 *  FSync signal (at either rising or falling SCLK edge), but you cannot
	 *  directly configure the timing of its Falling edge.
	 *  Thus, for generating true I2S LRC clock, we need to Shift
	 *  the Rising of the FSync in such a way that the Falling edge would take
	 *  place one bit before the start of data bits.
	 *
	 *  - as Slave:
	 *    ---------
	 *  It syncs only on Rising edge of FSync.
	 *  A solution would be to invert the FSync polarity, generated at
	 *  the Csr. The inversion is not available on the csr8811,
	 *  So we can't use this solution.
	 *  This limitation yields left/right channel swap on playback. 
	 */


	/* Note: (!)
	 * Do NOT change the order of the following configuration steps !
	 *
	 * This is due to the Codec (Csr) Driver code dependency:
	 * It must know the Master/Slave configuration before calculating
	 * its PLL settings.
	 */

	/* STEP 1: */

	/* Set DAI's TDM configuration: */
	dai_format = 0;
	dai_format |= SND_SOC_DAIFMT_I2S; /* TODO changes for CSR8811 !!! */

#ifdef CONFIG_SOC_CODEC_TDM_MASTER
	/* DMW is TDM Slave */
	dai_format |= SND_SOC_DAIFMT_NB_NF;	/* Normal SCLK(BCLK) & FSync(LRC) polarity */ 
	dai_format |= SND_SOC_DAIFMT_CBM_CFM;	/* Codec(!) point-of-view - Csr is Master */
	vdbg_prt("%s\n", "DMW96 TDM is Slave, Csr is Master");
#else
	/* DMW is TDM Master */
	dai_format |= SND_SOC_DAIFMT_NB_NF;	/* Normal SCLK(BCLK) & FSync(LRC) polarity */ 
	dai_format |= SND_SOC_DAIFMT_CBS_CFS;	/* Codec(!) point-of-view - Csr is Slave */
	vdbg_prt("%s\n", "DMW96 TDM is Master, Csr is Slave");
#endif /* !CONFIG_SOC_CODEC_TDM_MASTER */

	/* Set cpu DAI configuration */
	vdbg_prt("%s", "Calling CPU DAI set_fmt(): ");
	ret = snd_soc_dai_set_fmt(cpu_dai, dai_format);
	if (ret < 0) {
		return ret;
	}

	/* Set codec DAI configuration */
	vdbg_prt("%s\n", "Calling CODEC DAI set_fmt(): ");
	ret = snd_soc_dai_set_fmt(codec_dai, dai_format);
	if (ret < 0) {
		return ret;
	}

	/* STEP 2: */

	/* Get DP clk rate
	 * Note: the ASOC Platform driver (dmw-tdm.c) is now the DP clock "owner".
	 * 	 See mach-dmw/clock.c
	 */
	dp_clk = clk_get_sys("dmw-tdm", "dp");
	if (dp_clk < 0) {
		return (int)dp_clk;
	}
	dp_clk_rate = clk_get_rate(dp_clk);
	vdbg_prt("%s %lu %s\n", "DP clk running at:", dp_clk_rate, "Hz");

#ifndef CONFIG_SOC_CODEC_TDM_MASTER
	alsa_Fs = params_rate(params);
	coeff = dmw_as_master(alsa_Fs);
#endif /* ! CONFIG_SOC_CODEC_TDM_MASTER */

	return 0;
}

static int brd_csr8811_hw_free(struct snd_pcm_substream *substream)
{
	dbg_prt("%s %s\n",
		DIR_STR(substream->stream),
		"Currently doing nothing ---->");

	return 0;
}

static int brd_csr8811_prepare(struct snd_pcm_substream *substream)
{
	dbg_prt("%s\n", DIR_STR(substream->stream));

	/* Note:
	 * From the asoc sound-driver point of view, the 'Telephony' control
	 * gets updated (from the /etc/asound.conf file setting,
	 * according to the Currently Used Android sound 'device'),
	 * After brd_wm897X_hw_params() and Before brd_wm897X_prepare() are
	 * being called.
	 *
	 * The Machine-Driver's _prepare() (this dmw_pcm_prepare() function),
	 * is called Before the _prepare()s of the Platform-Drivers
	 * (PCM & TDM), so here is a good point to pass the info to them.
	 *
	 * This also means that the final TDM configuration, especially the
	 * _tdm_grant() issue, should be made in the cpu-dai's (TDM)
	 * _prepare() function !
	 */
	vdbg_prt("%s\n", "Setting TEL state");
	
	dmw_set_tel_state();

	return 0;
}

static int brd_csr8811_startup(struct snd_pcm_substream *substream)
{
	dbg_prt("%s %s\n",
		DIR_STR(substream->stream),
		"Currently doing nothing ---->");

	return 0;
}

static void brd_csr8811_shutdown(struct snd_pcm_substream *substream)
{
	dbg_prt("%s %s\n",
		DIR_STR(substream->stream),
		"Currently doing nothing ---->");
}

static int brd_csr8811_trigger(struct snd_pcm_substream *substream, int cmd)
{
	dbg_prt("%s %s\n",
		DIR_STR(substream->stream),
		"Currently doing nothing ---->");

	return 0;
}

static struct snd_soc_ops g_brd_csr8811_ops = {
	.startup	= brd_csr8811_startup,
	.shutdown	= brd_csr8811_shutdown,
	.hw_params	= brd_csr8811_hw_params,
	.hw_free	= brd_csr8811_hw_free,
	.prepare	= brd_csr8811_prepare,
	.trigger	= brd_csr8811_trigger,
};

static int brd_csr8811_dai_init(struct snd_soc_pcm_runtime *rtd)
{
	int ret;
	struct snd_soc_codec *codec = rtd->codec;
	struct snd_soc_dapm_context * dapm = &codec->dapm;

	// Do Not delete - possible future use !
	//struct snd_soc_card *card = codec->card;
	//struct dspg_drv_priv_data *priv_data = snd_soc_card_get_drvdata(card);
	//void *pltfrm_data = priv_data->pdata;
	
	dbg_prt();

	vdbg_prt("%s\n", "Syncing dapm");
	ret = snd_soc_dapm_sync(dapm);
	if (ret) {
		vdbg_prt("%s\n", "Dapm syncing Failed");
		return ret;
	}

	return 0;
}
#endif /* CONFIG_SND_DMW_SOC_BOARD_CSR8811 */
static struct snd_soc_dai_link brd_gxv22xx_dai_link[] = {
	{
		.name = "TLV320AIC3X",
		.stream_name = "AIC3X",
		
		/* Note: in DSPG boards, the Third dmw96's TDM machine
		 *       is connected to the Wolfson codec chip
		 */
		.cpu_dai_name = "dmw-tdm.2",
		.platform_name = "dmw-pcm-audio",
		.codec_dai_name = "tlv320aic3x-hifi",
		.codec_name = "tlv320aic3x-codec.1-0018",
		.init = brd_gxv22xx_dai_link_init,
		.ops = &brd_gxv22xx_ops,
	},
#ifdef CONFIG_SND_DMW_SOC_BOARD_CSR8811
	{
	/* Index 1 : CSR8811 conneted to DMW96 over TDM 0 : */

		.name = "dmw96-csr8811_Dai_Link",
		.stream_name = "dmw96-csr8811_I2S",

		/* asoc Codec device name */
		.codec_name = "BT-csr8811",

		/* asoc Platform driver (PCM) name */
		.platform_name = "dmw-pcm-audio",
		
		/* asoc Cpu-Dai (TDM) device name */
		.cpu_dai_name = "dmw-tdm.0",
		
		/* asoc Codec-Dai device name */
		.codec_dai_name = "csr8811-hifi",
		 
		.init = brd_csr8811_dai_init,
		.ops = &g_brd_csr8811_ops,
	},
#endif /* CONFIG_SND_DMW_SOC_BOARD_CSR8811 */
};


static void dmw_set_tel_state(void)
{
	struct snd_soc_dai_link *dai_link = &brd_gxv22xx_dai_link[0];
	const char *cpu_dai_name = dai_link->cpu_dai_name;

	dbg_prt();

	vdbg_prt("%s %s\n", "Requested TEL state is",
		(g_dmw_tel_state) ? "ON" : "OFF");

	vdbg_prt("%s\n", "Calling dmw_tdm_set_telephony()");
	dmw_tdm_set_telephony(cpu_dai_name, g_dmw_tel_state);

	vdbg_prt("%s\n", "Calling dmw_pcm_set_telephony()");
	dmw_pcm_set_telephony(g_dmw_tel_state);
}

/*****************************************************************************/

static struct snd_soc_card brd_gxp22xx_card = {
	.name = "DSPG-BOARD-AIC3104",
	//.platform = &dmw_soc_platform,    /* Platform interface; sound/soc/dmw/dmw-pcm.c */
	.dai_link = brd_gxv22xx_dai_link,  /* DAI_link array */
	.num_links = ARRAY_SIZE(brd_gxv22xx_dai_link),
};

static int switch_tel(
	struct file *file,
	const char *buffer,
	unsigned long count,
	void *data
)
{

    char        state_string[12] = {'\0'};
    char        *endp;
    unsigned long newValue = 0;

    if ((count > sizeof(state_string) - 1))
        return (-EINVAL);

    if (copy_from_user(state_string, buffer, count))
        return (-EFAULT);

    state_string[count] = '\0';

    newValue = simple_strtoul(state_string, &endp, 10);
    if(newValue)
	g_dmw_tel_state = 1;
    else
	g_dmw_tel_state = 0;
    dmw_set_tel_state();   
    return count;
}



static int aic_write(
	struct file *file,
	const char *buffer,
	unsigned long count,
	void *data
)
{

    char        state_string[12] = {'\0'};
    char        *endp;
    unsigned int reg=0;
    unsigned long newValue = 0;
    char val_buf[10]={0}, reg_buf[10]={0};

    if ((count > sizeof(state_string) - 1))
        return (-EINVAL);

    if (copy_from_user(state_string, buffer, count))
        return (-EFAULT);

    state_string[count] = '\0';
    sscanf(state_string, "%s %s", reg_buf, val_buf);
    reg = simple_strtoul(reg_buf, &endp, 10);
    newValue = simple_strtoul(val_buf, &endp, 16);
    printk("write reg %d, value %lx\n", reg, newValue);
    snd_soc_write(g_codec, reg, newValue);
    return count;
}

static void restore_unmuted_handle(struct work_struct *work)
{
	if(g_codec)
	{
		snd_soc_write(g_codec, 15, 0x7f&snd_soc_read(g_codec, 15));
		mdelay(1);
		snd_soc_write(g_codec, 16, 0x7f&snd_soc_read(g_codec, 16));
	}
}

static void sw_delay_handle(unsigned long data)
{
	schedule_work(&restore_unmuted);
}



static int gxp22xx_snd_probe(struct platform_device *pdev)
{
	int ret=0;
	struct snd_soc_card *card = &brd_gxp22xx_card;


	struct gxp22xx_board_pdata *pdata;
	dbg_prt();
	/* Get DP clock handle (for Suspend/Resume operations) */
	g_machdrv_dp_clk = clk_get_sys("dmw-tdm", "dp");
	if (g_machdrv_dp_clk) {
		vdbg_prt("%s\n", "Got DP clk");

	} else {
		vdbg_prt("%s\n", "Can't get DP clk");
		return (int)g_machdrv_dp_clk;
	}
	
	pdata = (struct gxp22xx_board_pdata *)pdev->dev.platform_data;
	
	gxp22xx_brd_init_sound_gpios(pdata);
	
	/*add MCLK TBD*/

	card->dev = &pdev->dev;
	platform_set_drvdata(pdev, card);
	ret = snd_soc_register_card(card);
	if(ret)
	{
		vdbg_prt("%s %i\n", "snd_soc_register_card Failed", ret);
		goto ERR_CLEAR;
	}
	{
	struct proc_dir_entry *ent;
	ent = create_proc_entry("switch_tel", S_IFREG|S_IRUGO, NULL);
	if(!ent)
		return 0;
	ent->write_proc=switch_tel;
	ent->read_proc=NULL;
	}
	
	{
	struct proc_dir_entry *ent;
	ent = create_proc_entry("aic_write", S_IFREG|S_IRUGO, NULL);
	if(!ent)
		return 0;
	ent->write_proc=aic_write;
	ent->read_proc=NULL;
	}
	setup_timer(&sw_delay, sw_delay_handle, 0);
	INIT_WORK(&restore_unmuted, restore_unmuted_handle);
	
	return 0;
ERR_CLEAR:

	snd_soc_card_set_drvdata(card, NULL);
	platform_set_drvdata(pdev, NULL);
	card->dev = NULL;
	clk_put(g_machdrv_dp_clk);
	g_machdrv_dp_clk = NULL;
	return ret; 
}

static int gxp22xx_snd_remove(struct platform_device *pdev)
{
	struct snd_soc_card *card = platform_get_drvdata(pdev);
	dbg_prt();
	snd_soc_unregister_card(card);

	/* Cleanup: */
	snd_soc_card_set_drvdata(card, NULL);
	platform_set_drvdata(pdev, NULL);
	card->dev = NULL;

	if (g_machdrv_dp_clk) {
		clk_put(g_machdrv_dp_clk);
		g_machdrv_dp_clk = NULL;
	}
	
	return 0;

}


static struct platform_driver g_brd_gxp22xx_snd_drv = {
	.driver = {
		.name = DRIVER_NAME,
		.owner = THIS_MODULE,
	},
	.probe = gxp22xx_snd_probe,
	.remove = __devexit_p(gxp22xx_snd_remove),
};

static int __init brd_gxv22xx_mod_init(void)
{
	int err;

	dbg_prt();

	/* Check board matching */
	if (!check_valid_board()) {
		vdbg_prt("%s\n", "Not my board");
		return -ENODEV;
	}
	vdbg_prt("%s\n", "Valid board found");


	vdbg_prt("%s\n", "Registrating asoc machine driver");

	err = platform_driver_register(&g_brd_gxp22xx_snd_drv);
	if (err) {
		vdbg_prt("%s\n", "asoc machine driver registration Failed !!!");
	}

	return err;
}

static void __exit brd_gxv22xx_mod_exit(void)
{
	dbg_prt();

	vdbg_prt("%s\n", "UnRegistrating platform driver");
	platform_driver_unregister(&g_brd_gxp22xx_snd_drv);
}

module_init(brd_gxv22xx_mod_init);
module_exit(brd_gxv22xx_mod_exit);

MODULE_DESCRIPTION("ASoC machine driver for GXV board with AIC3104");
MODULE_AUTHOR("GXV Group");
MODULE_LICENSE("GPL");

