/*
 * dspg-wm897X.c -- ASoC Machine Driver for DMW-96 DSPG-style board,
 * with Wolfson WM8976 or WM8978.
 *
 * 9 November, 2011:
 * If 'CONFIG_SND_DMW_SOC_BOARD_CSR8811' is defined, it also handles
 * the CSR csr8811 Bluetooth codec.
 *
 * 20 December, 2011:
 * If 'CONFIG_SND_DMW_SOC_EVB96_SI3050' is defined, it also handles
 * the SI3050 FXO codec (on EVB96 board only).
 * By Murali Mohan TD Mohan <Murali.Mohan@dspg.com>
 *
 *  Copyright (C) 2011 DSPG Technologies GmbH
 *
 * 16 October, 2011:
 * Adapted to kernel 2.6.39 by Avi Miller <Avi.Miller@dspg.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/clk.h>
#include <linux/io.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <mach/platform.h>
#include <mach/gpio.h>
#include <asm/atomic.h>

#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/soc.h>
#include <sound/soc-dapm.h>

#include <sound/dmw-tdm.h>
#include <sound/dmw-boards.h>
#include "dmw-tdm-mstr.h"
#include "dmw-pcm.h"

//#include "../codecs/wm897X_mod.h"

//#include "dspg-wm897X.h"

#define DRIVER_NAME "gxe51xx-tdm-mach-drv"
#define DEVICE_NAME "dmw-tdm-1"

void dmw_clk_print_active(void);



#define V_DEBUG 
/*****************************************************************************/
#undef pr_debug
#define pr_debug(fmt, ...) \
        printk(KERN_DEBUG pr_fmt(fmt), ##__VA_ARGS__)

#ifdef V_DEBUG
        #ifndef DEBUG
                #define DEBUG
        #endif
        #define vdbg_prt(fmt, ...) \
                pr_debug("asoc Pltfrm PCM: %s(): " fmt, __func__, ##__VA_ARGS__)
#else
        #define vdbg_prt(fmt, ...)
#endif

#ifdef V_DEBUG
        #ifndef DEBUG 
                #define DEBUG
        #endif
        #define vdbg_prt(fmt, ...) \
                pr_debug("gxe51xx-pstn asoc Mach-drv: %s(): " fmt, __func__, ##__VA_ARGS__)
#else
        #define vdbg_prt(fmt, ...)
#endif


#ifdef DEBUG
        #define dbg_prt(fmt, ...) \
                pr_debug("gxe51xx-pstn asoc Mach-drv: %s(): " fmt, __func__, ##__VA_ARGS__)

        #define DIR_STR(stream_dir)  ((stream_dir == 0) ? "for Playback" : "for Capture")

#else
        #define dbg_prt(fmt, ...)       
        #define DIR_STR(stream_dir)
#endif


/*****************************************************************************/

#ifndef ARGUSE
  #define ARGUSED(x) (void)(x)
#endif

/*****************************************************************************/

static int g_dmw_tel_state = 0;

#include "sound/si3050.h"
static int g_dmw_fxo_audiostate = 1;


static struct dmw_tdm_pdata * gxe51xx_tdm_pdata_ptr = NULL;

typedef enum
{
	SI3050_DAI,
	NUM_OF_DAI
} gxe51xx_dai_e;

static struct snd_soc_dai_link gxe51xx_board_dai_link[NUM_OF_DAI];

/*****************************************************************************/

static struct clk * g_machdrv_dp_clk = NULL;

static atomic_t g_dp_clk_opencnt = { 0 };

static const char *tel_function[] = {"Off", "On"};
static const struct soc_enum tel_enum = SOC_ENUM_SINGLE_EXT(2, tel_function);

/* Note: dp clock reference-counting here is for debugging purpose.
 *	 Otherwise, the clock framework's internal reference counting
 *	 is adequate.
 */
static void brd_dp_clk_enable(void)
{
	uint clk_ref_after_incr;

	dbg_prt();

	clk_ref_after_incr = atomic_inc_return(&g_dp_clk_opencnt);
	if (clk_ref_after_incr - 1 == 0) {
		vdbg_prt("%s %u\n", "Enabling DP clock. Ref-count after:",
			clk_ref_after_incr);

		clk_enable(g_machdrv_dp_clk);
	} else {
		vdbg_prt("%s %u\n",
			"DP clock already Enabled. Ref-count after:",
			clk_ref_after_incr);
	}
}

static void brd_dp_clk_disable(void)
{
	uint clk_ref_after_decr;

	dbg_prt();

	clk_ref_after_decr = atomic_dec_return(&g_dp_clk_opencnt);
	if (clk_ref_after_decr == 0) {
		vdbg_prt("%s %u\n", "Disabling DP clock. Ref-count after:",
			clk_ref_after_decr);

		clk_disable(g_machdrv_dp_clk);
	} else {
		vdbg_prt("%s %u\n", "DP clock still On. Ref-count after:",
			clk_ref_after_decr);
	}
}


/*****************************************************************************/

/*****************************************************************************/


static int dmw_get_fxoaudio(struct snd_kcontrol *kcontrol,
               struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = g_dmw_fxo_audiostate;
	vdbg_prt("%s %s\n", "FXO Audio state:",
		(g_dmw_fxo_audiostate == 0) ? "Off" : "On");

	return 0;
}

static int dmw_set_fxoaudio(struct snd_kcontrol *kcontrol,
               struct snd_ctl_elem_value *ucontrol)
{
	if (g_dmw_fxo_audiostate == ucontrol->value.integer.value[0]) {
		vdbg_prt("%s %s\n", "FXO Audio state Unchanged:",
			(g_dmw_fxo_audiostate == 0) ? "Off" : "On");
		return 0;
	}

	g_dmw_fxo_audiostate = ucontrol->value.integer.value[0];
	vdbg_prt("%s %s\n", "FXO Audio state:",
		(g_dmw_fxo_audiostate == 0) ? "Off" : "On");
	return 0;
}

static const struct snd_kcontrol_new fxo_audiocontrols[] = {
    SOC_ENUM_EXT("FXO_AUDIO", tel_enum, dmw_get_fxoaudio, dmw_set_fxoaudio),
};


static void dmw_update_fxo_audiostate(void)
{
	struct snd_soc_dai_link *dai_link = &gxe51xx_board_dai_link[SI3050_DAI];
	const char *cpu_dai_name = dai_link->cpu_dai_name;

	dbg_prt();

	vdbg_prt("%s %s\n", "Requested FXO Audio state is",
		(g_dmw_fxo_audiostate) ? "ON" : "OFF");

	vdbg_prt("%s %s\n",
		"Calling dmw_tdm_set_telephony() for cpu-dai",
		cpu_dai_name);
	dmw_tdm_set_telephony(cpu_dai_name, g_dmw_fxo_audiostate);

	vdbg_prt("%s\n", "Calling dmw_pcm_set_telephony()");
	dmw_pcm_set_telephony(g_dmw_fxo_audiostate);
}

static int brd_si3050_hw_params(struct snd_pcm_substream *substream,
                 struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *codec_dai = rtd->codec_dai;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;
	
	unsigned int dai_format;
	struct clk * dp_clk = NULL;
	unsigned long dp_clk_rate;
	int ret;

#ifndef CONFIG_SOC_CODEC_TDM_MASTER
	uint alsa_Fs;
	struct snd_soc_codec *codec = rtd->codec;
	struct snd_soc_card *card = codec->card;
//	struct wm897X_setup_data *setup_data = snd_soc_card_get_drvdata(card);
	struct tdm_master_coeff * coeff;
	unsigned long act_dp_clk_rate;
#endif /* ! CONFIG_SOC_CODEC_TDM_MASTER */

	dbg_prt("%s\n", DIR_STR(substream->stream));

	/*
	 * Note: dmw96 TDM-machine HW limitations:
	 *
	 *  - as Master:
	 *    ----------
	 *  You can configure the timing of the Rising edge of the generated
	 *  FSync signal (at either rising or falling SCLK edge), but you cannot
	 *  directly configure the timing of its Falling edge.
	 *  Thus, for generating true I2S LRC clock, we need to Shift
	 *  the Rising of the FSync in such a way that the Falling edge would take
	 *  place one bit before the start of data bits.
	 *
	 *  - as Slave:
	 *    ---------
	 *  It syncs only on Rising edge of FSync.
	 *  A solution would be to invert the FSync polarity, generated at
	 *  the Wolfson. However, it turns out that in this case the Wolfson also
	 *  puts the ADC (capture) data on the Right channel, instead of on the
	 *  Left one. So we can't use this solution.
	 *  This limitation yields left/right channel swap on playback. 
	 *  Another soultion would be to swap the channels by swapping the Wolfson
	 *  L/R DAC's output.
	 */


	/* Note: (!)
	 * Do NOT change the order of the following configuration steps !
	 *
	 * This is due to the Codec (Wolfson) Driver code dependency:
	 * It must know the Master/Slave configuration before calculating
	 * its PLL settings.
	 */

	/* STEP 1: */

	/* Set DAI's TDM configuration: */
	dai_format = 0;
	dai_format |= SND_SOC_DAIFMT_I2S;

#ifdef CONFIG_SOC_CODEC_TDM_MASTER
	/* DMW96 is TDM Slave */
//#error why am i here?
	dai_format |= SND_SOC_DAIFMT_NB_NF; /* Normal SCLK(BCLK) & FSync(LRC) polarity */ 
	dai_format |= SND_SOC_DAIFMT_CBM_CFM;   /* Codec(!) point-of-view - si3050 is Master */
	vdbg_prt("%s\n", "DMW96 TDM is Slave, SI3050 is Master");
#else
	/* DMW96 is TDM Master */

	dai_format |= SND_SOC_DAIFMT_NB_NF; /* Normal SCLK(BCLK) & FSync(LRC) polarity */ 
	dai_format |= SND_SOC_DAIFMT_CBS_CFS;   /* Codec(!) point-of-view - si3050 is Slave */
	vdbg_prt("%s\n", "DMW96 TDM is Master, SI3050 is Slave");
#endif /* !CONFIG_SOC_CODEC_TDM_MASTER */

	/* Set cpu DAI's TDM configuration */
	vdbg_prt("%s", "Calling CPU DAI set_fmt(): ");
	ret = snd_soc_dai_set_fmt(cpu_dai, dai_format);
	if (ret < 0) {
		vdbg_prt("%s\n", "CPU DAI tdm-format setting Failed !!!");
		return ret;
	}

	/* Set codec DAI's TDM configuration */
	vdbg_prt("%s\n", "Calling CODEC DAI set_fmt(): ");
	ret = snd_soc_dai_set_fmt(codec_dai, dai_format);
	if (ret < 0) {
		vdbg_prt("%s\n", "CODEC DAI tdm-format setting Failed !!!");
		return ret;
	}

	/* STEP 2: */

	/* Get DP clk rate
	 * Note: the ASOC Platform driver (dmw-tdm.c) is now the DP clock "owner".
	 * 	 See mach-dmw/clock.c
	 */
	dp_clk = clk_get_sys("dmw-tdm", "dp");
	if (dp_clk < 0) {
		return (int)dp_clk;
	}
	dp_clk_rate = clk_get_rate(dp_clk);
	vdbg_prt("%s %lu %s\n", "clk running at:", dp_clk_rate, "Hz");

#ifndef CONFIG_SOC_CODEC_TDM_MASTER
	alsa_Fs = params_rate(params);
	coeff = dmw_as_master(alsa_Fs);
//	setup_data->lrc_fs_x100 = coeff->lrc_fs_x100;
	vdbg_prt("%s %u %s %u\n",
		"alsa_Fs:", alsa_Fs,
		"lrc_Fs_x100:", coeff->lrc_fs_x100);

	act_dp_clk_rate = dp_clk_rate;
	act_dp_clk_rate /= 480000; /* Teoretical sysclk, in KHz */
	act_dp_clk_rate *= 480019; /* Actual sysclk, in KHz */
	vdbg_prt("%s %lu %s %lu\n",
		"Theoretical TDM clk:", dp_clk_rate,
		"Actual TDM clk:", act_dp_clk_rate);
	dp_clk_rate = act_dp_clk_rate;
#endif /* ! CONFIG_SOC_CODEC_TDM_MASTER */

	clk_put(dp_clk);

	/* Set Wolfson's PLL */
	vdbg_prt("%s\n", "Calling CODEC DAI set_sysclk()");
	ret = snd_soc_dai_set_sysclk(codec_dai, 0/*WM897X_PLL*/, (unsigned int)dp_clk_rate,
			SND_SOC_CLOCK_IN);
	if (ret < 0) {
		vdbg_prt("%s\n", "CODEC DAI sysclock setting (Wolfson internal PLL) Failed !!!");
		return ret;
	}

	/* Set Wolfson's BCLK rate.
	 * Note: Relevant only when Wolfson is the TDM Master!
	 * For Wolfson as Master, we always use 16 bits per time-slot,
	 * for All sample-rates (Fs)!
	 */
	vdbg_prt("%s\n", "Calling CODEC DAI set_clkdiv()");
	/* Divide 256*Fs by 8 to get Wolfson BCLK == dmw96 SCLK = 2*16*Fs:
	 * Reg 6 bits 4:2, BCLKDIV
	 */
	ret = snd_soc_dai_set_clkdiv(codec_dai, 1 /*WM897X_BCLKDIV*/, 0x3 << 2);
	if (ret < 0) {
		vdbg_prt("%s\n", "CODEC DAI clkdiv setting (Wolfson BCLK output) Failed !!!");
		return ret;
	}
	return 0;

}


static int brd_si3050_hw_free(struct snd_pcm_substream *substream){
	dbg_prt("%s %s\n",
		DIR_STR(substream->stream),
		"Currently doing nothing ---->");

	return 0;
}


static int brd_si3050_prepare(struct snd_pcm_substream *substream){
	dbg_prt("%s\n", DIR_STR(substream->stream));
	vdbg_prt("%s\n", "Setting TEL state");
	
	dmw_update_fxo_audiostate();
	return 0;
}

static int brd_si3050_startup(struct snd_pcm_substream *substream){
	dbg_prt("%s %s\n",
		DIR_STR(substream->stream),
		"Currently doing nothing ---->");

	brd_dp_clk_enable();

	return 0;
}

static void brd_si3050_shutdown(struct snd_pcm_substream *substream){
	dbg_prt("%s %s\n",
		DIR_STR(substream->stream),
		"Currently doing nothing ---->");

	brd_dp_clk_disable();

}

static int brd_si3050_trigger(struct snd_pcm_substream *substream, int cmd){
	dbg_prt("%s %s\n",
		DIR_STR(substream->stream),
		"Currently doing nothing ---->");

	return 0;
}


static struct snd_soc_ops g_brd_si3050_ops = {
	.startup    = brd_si3050_startup,
	.shutdown   = brd_si3050_shutdown,
	.hw_params  = brd_si3050_hw_params,
	.hw_free    = brd_si3050_hw_free,
	.prepare    = brd_si3050_prepare,
	.trigger    = brd_si3050_trigger,
};


/*****************************************************************************/

static int dmw_get_tel(struct snd_kcontrol *kcontrol,
		       struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = g_dmw_tel_state;

	vdbg_prt("%s %s\n", "TEL state is:",
		(g_dmw_tel_state == 0) ? "Off" : "On");
	
	return 0;
}

static int dmw_set_tel(struct snd_kcontrol *kcontrol,
		       struct snd_ctl_elem_value *ucontrol)
{
	if (g_dmw_tel_state == ucontrol->value.integer.value[0]) {
		vdbg_prt("%s %s\n",
			"TEL state Unchanged:",
			(g_dmw_tel_state == 0) ? "Off" : "On");
		return 0;
	}

	g_dmw_tel_state = ucontrol->value.integer.value[0];

	vdbg_prt("%s %s\n", "Updating TEL state to:",
		(g_dmw_tel_state == 0) ? "Off" : "On");
	
	return 1;
}


static const struct snd_kcontrol_new tel_controls[] = {
	SOC_ENUM_EXT("Telephony", tel_enum, dmw_get_tel, dmw_set_tel),
};

/*****************************************************************************/


static int brd_si3050_dai_link_init(struct snd_soc_pcm_runtime *rtd)
{
	struct snd_soc_codec *codec = rtd->codec;
	struct snd_soc_dai *codec_dai = rtd->codec_dai;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;
        unsigned int dai_format;
        struct clk * dp_clk = NULL;
        unsigned long dp_clk_rate;

#ifndef CONFIG_SOC_CODEC_TDM_MASTER
        uint alsa_Fs;
        struct snd_soc_card *card = codec->card;
//      struct wm897X_setup_data *setup_data = snd_soc_card_get_drvdata(card);
        struct tdm_master_coeff * coeff;
        unsigned long act_dp_clk_rate;
#endif /* ! CONFIG_SOC_CODEC_TDM_MASTER */

	int ret;

	/* Note: this is the "probe" function */
	dbg_prt();

	vdbg_prt("%s\n", "Adding FXO control");
	ret = snd_soc_add_controls(codec, fxo_audiocontrols, ARRAY_SIZE(fxo_audiocontrols));
	if (ret) {
		vdbg_prt("%s\n", "FXO Audio control adding Failed");
		return ret;
	}



	/* STEP 1: */

	/* Set DAI's TDM configuration: */
	dai_format = 0;
	dai_format |= SND_SOC_DAIFMT_I2S;

#ifdef CONFIG_SOC_CODEC_TDM_MASTER
	/* DMW96 is TDM Slave */
//#error why am i here?
	dai_format |= SND_SOC_DAIFMT_NB_NF; /* Normal SCLK(BCLK) & FSync(LRC) polarity */ 
	dai_format |= SND_SOC_DAIFMT_CBM_CFM;   /* Codec(!) point-of-view - si3050 is Master */
	vdbg_prt("%s\n", "DMW96 TDM is Slave, SI3050 is Master");
#else
	/* DMW96 is TDM Master */

	dai_format |= SND_SOC_DAIFMT_NB_NF; /* Normal SCLK(BCLK) & FSync(LRC) polarity */ 
	dai_format |= SND_SOC_DAIFMT_CBS_CFS;   /* Codec(!) point-of-view - si3050 is Slave */
	vdbg_prt("%s\n", "DMW96 TDM is Master, SI3050 is Slave");
#endif /* !CONFIG_SOC_CODEC_TDM_MASTER */

	/* Set cpu DAI's TDM configuration */
	vdbg_prt("%s", "Calling CPU DAI set_fmt(): ");
	ret = snd_soc_dai_set_fmt(cpu_dai, dai_format);
	if (ret < 0) {
		vdbg_prt("%s\n", "CPU DAI tdm-format setting Failed !!!");
		return ret;
	}

	/* Set codec DAI's TDM configuration */
	vdbg_prt("%s\n", "Calling CODEC DAI set_fmt(): ");
	ret = snd_soc_dai_set_fmt(codec_dai, dai_format);
	if (ret < 0) {
		vdbg_prt("%s\n", "CODEC DAI tdm-format setting Failed !!!");
		return ret;
	}

        dp_clk = clk_get_sys("dmw-tdm.2", NULL);
        if (dp_clk < 0) {
                return (int)dp_clk;
        }
        dp_clk_rate = clk_get_rate(dp_clk);
        vdbg_prt("%s %lu %s\n", "TDM clk running at:", dp_clk_rate, "Hz");
        
#ifndef CONFIG_SOC_CODEC_TDM_MASTER
//        alsa_Fs = params_rate(params);
//        coeff = dmw_as_master(alsa_Fs);
        coeff = dmw_as_master(8000);
//      setup_data->lrc_fs_x100 = coeff->lrc_fs_x100;
        vdbg_prt("%s %u %s %u\n",
                "alsa_Fs:", alsa_Fs,
                "lrc_Fs_x100:", coeff->lrc_fs_x100);
        
//        act_dp_clk_rate = dp_clk_raate;
//        act_dp_clk_rate /= 480000; /* Teoretical sysclk, in KHz */
//        act_dp_clk_rate *= 480019; /* Actual sysclk, in KHz */
        vdbg_prt("%s %lu %s %lu\n",
                "Theoretical DP clk:", dp_clk_rate,
                "Actual DP clk:", act_dp_clk_rate);
        dp_clk_rate = act_dp_clk_rate;
#endif /* ! CONFIG_SOC_CODEC_TDM_MASTER */ 

        clk_put(dp_clk);
        
        /* Set Wolfson's PLL */
        vdbg_prt("%s\n", "Calling CODEC DAI set_sysclk()");
        ret = snd_soc_dai_set_sysclk(codec_dai, 0/*WM897X_PLL*/, (unsigned int)dp_clk_rate,
                        SND_SOC_CLOCK_OUT);

	brd_dp_clk_enable();

	return 0;
}


static struct snd_soc_dai_link gxe51xx_board_dai_link[] = {
	{
	/* Index 1 : SI3050 conneted to DMW96 over TDM 2 : */

		.name = "dmw96-si3050_dai_link",
		.stream_name = "dmw96-si3050_I2S",

		/* asoc Codec device name */
		.codec_name = "spi1.0",

		/* asoc Platform driver (PCM) name */
		.platform_name = "dmw-pcm-audio",

		/* asoc Cpu-Dai (TDM) device name */
		.cpu_dai_name = "dmw-tdm.2",

		/* asoc Codec-Dai device name */
		.codec_dai_name = "si3050-fxo",

		.init = brd_si3050_dai_link_init,
		.ops = &g_brd_si3050_ops,
	},
};

/*****************************************************************************/

static int brd_wm897X_suspend_pre(struct snd_soc_card *card)
{
	dbg_prt("%s\n", "Currently doing nothing ---->");

	return 0;
}

static int brd_wm897X_suspend_post(struct snd_soc_card *card)
{
	dbg_prt("%s\n", "Currently doing nothing ---->");

	return 0;
}

static int brd_wm897X_resume_pre(struct snd_soc_card *card)
{
	dbg_prt("%s\n", "Currently doing nothing ---->");

	return 0;
}

static int brd_wm897X_resume_post(struct snd_soc_card *card)
{
	dbg_prt("%s\n", "Currently doing nothing ---->");

	return 0;
}

static struct snd_soc_card gxe51xx_si3050_card = {

	.name		= "gxe51xx-card-si3050",
	.suspend_pre	= brd_wm897X_suspend_pre,
	.suspend_post	= brd_wm897X_suspend_post,
	.resume_pre	= brd_wm897X_resume_pre,
	.resume_post	= brd_wm897X_resume_post,
	.dai_link	= gxe51xx_board_dai_link,  /* DAI-link Array(!) */
	.num_links	= ARRAY_SIZE(gxe51xx_board_dai_link),

	/* Note: new to ASoC:
	 * Card's (not Codec's !) set_bias_level().
	 * May be used for certain activities, e.g. diasbling
	 * card's power amplifiers or enabling them just upon BIAS_ON;
	 * this is if they are Not declared as card's DAPM Widgests,
	 * as part of the audio Routes.
	 */
	.set_bias_level		= NULL,
	.set_bias_level_post	= NULL,
};

/*****************************************************************************/

static int dspg_snd_probe(struct platform_device *pdev)
{
	int ret;
	struct snd_soc_card *card = &gxe51xx_si3050_card;
	//void *pltfrm_data = pdev->dev.platform_data;

	dbg_prt();

	gxe51xx_tdm_pdata_ptr = (struct dmw_tdm_pdata*)pdev->dev.platform_data;
#if 0 
	ret = gxe51xx_init_tdm_gpios();

	if (ret) {
		vdbg_prt("%s %i\n", "GPIOs init Failed:", ret);
		return ret;
	}
#endif
	/* Get DP clock handle (for Suspend/Resume operations) */
	g_machdrv_dp_clk = clk_get_sys("dmw-tdm.2", NULL);
	if (g_machdrv_dp_clk) {
		vdbg_prt("%s\n", "Got TDM clk");

	} else {
		vdbg_prt("%s\n", "Can't get TDM clk");
		return (int)g_machdrv_dp_clk;
	}

	/* Book keeping : */
	
	/* note: platform_set_drvdata() here saves pointer to the card's data
	 * on the device's 'struct device_private *p'->driver_data
	 */
	card->dev = &pdev->dev;
	platform_set_drvdata(pdev, card);

	/* note: this sets card->drvdata = priv_data */
	snd_soc_card_set_drvdata(card, NULL);

	/* Register ASoC sound Card */
	vdbg_prt("%s\n", "Registaring asoc Card");
	ret = snd_soc_register_card(card);
	if (ret) {
		vdbg_prt("%s %i\n", "snd_soc_register_card Failed", ret);
		goto ERR_CLEAR;
	}
	vdbg_prt("%s\n", "asoc Card registred OK");

	return 0;

ERR_CLEAR:

	snd_soc_card_set_drvdata(card, NULL);
	platform_set_drvdata(pdev, NULL);
	card->dev = NULL;
	clk_put(g_machdrv_dp_clk);
	g_machdrv_dp_clk = NULL;
		
	return ret; 
}

static int dspg_snd_remove(struct platform_device *pdev)
{
	struct snd_soc_card *card = platform_get_drvdata(pdev);

	dbg_prt();
printk("-->dspg_snd_remove\n");

	snd_soc_unregister_card(card);

	/* Cleanup: */
	snd_soc_card_set_drvdata(card, NULL);
	platform_set_drvdata(pdev, NULL);
	card->dev = NULL;

	if (g_machdrv_dp_clk) {
		clk_put(g_machdrv_dp_clk);
		g_machdrv_dp_clk = NULL;
	}

	return 0;	
}

static struct platform_driver gxe51xx_tdm_drv = {
	.driver = {
		.name = DRIVER_NAME,
		.owner = THIS_MODULE,

		/* A.M. New ASoC feature - to test */
		.pm = &snd_soc_pm_ops,
	},
	.probe = dspg_snd_probe,
	.remove = __devexit_p(dspg_snd_remove),
};

/*****************************************************************************/

static int __init gxe51xx_tdm_mod_init(void)
{
	int err;

	dbg_prt();

	printk("Registrating asoc machine driver\n");

	err = platform_driver_register(&gxe51xx_tdm_drv);
	if (err) {
		printk("%s asoc machine driver registration Failed !!! (%d)\n", &gxe51xx_tdm_drv.driver.name, err);
	}

	return err;
}

static void __exit gxe51xx_tdm_mod_exit(void)
{
	dbg_prt();

	printk("UnRegistrating asoc machine driver\n");

	platform_driver_unregister(&gxe51xx_tdm_drv);
}

module_init(gxe51xx_tdm_mod_init);
module_exit(gxe51xx_tdm_mod_exit);

MODULE_DESCRIPTION("TDM machine driver for GXE51XX ");
MODULE_AUTHOR("Penio Radev");
MODULE_LICENSE("GPL");
