/*
 * dspg-wm897X.c -- ASoC Machine Driver for DMW-96 DSPG-style board,
 * with Wolfson WM8976 or WM8978.
 *
 * 9 November, 2011:
 * If 'CONFIG_SND_DMW_SOC_BOARD_CSR8811' is defined, it also handles
 * the CSR csr8811 Bluetooth codec.
 *
 * 20 December, 2011:
 * If 'CONFIG_SND_DMW_SOC_EVB96_SI3050' is defined, it also handles
 * the SI3050 FXO codec (on EVB96 board only).
 * By Murali Mohan TD Mohan <Murali.Mohan@dspg.com>
 *
 *  Copyright (C) 2011 DSPG Technologies GmbH
 *
 * 16 October, 2011:
 * Adapted to kernel 2.6.39 by Avi Miller <Avi.Miller@dspg.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/clk.h>
#include <linux/io.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <mach/platform.h>
#include <mach/gpio.h>
#include <asm/atomic.h>

#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/soc.h>
#include <sound/soc-dapm.h>

#include <sound/dmw-tdm.h>
#include <sound/dmw-boards.h>
#include "dmw-tdm-mstr.h"
#include "dmw-pcm.h"

#include "../codecs/wm897X_mod.h"

#include "dspg-wm897X.h"

#ifdef CONFIG_SND_DMW_SOC_BOARD_CSR8811
#include "../codecs/csr8811.h"
#endif

/*****************************************************************************/

#ifndef ARGUSED
  #define ARGUSED(x) (void)(x)
#endif

/*****************************************************************************/

#define DEVICE_NAME "soc-audio"

static int g_dmw_tel_state = 0;

#ifdef CONFIG_SND_DMW_SOC_EVB96_SI3050
#include "sound/si3050.h"
static int g_dmw_fxo_audiostate = 1;
#endif


static struct dmw_board_pdata * g_brd_snd_gpios = NULL;

static struct snd_soc_dai_link g_brd_wm897X_dai_link[NUM_OF_DAIs];

/*****************************************************************************/

static struct clk * g_machdrv_dp_clk = NULL;

static atomic_t g_dp_clk_opencnt = { 0 };

static const char *tel_function[] = {"Off", "On"};
static const struct soc_enum tel_enum = SOC_ENUM_SINGLE_EXT(2, tel_function);

/* Note: dp clock reference-counting here is for debugging purpose.
 *	 Otherwise, the clock framework's internal reference counting
 *	 is adequate.
 */
static void brd_dp_clk_enable(void)
{
	uint clk_ref_after_incr;

	dbg_prt();

	clk_ref_after_incr = atomic_inc_return(&g_dp_clk_opencnt);
	if (clk_ref_after_incr - 1 == 0) {
		vdbg_prt("%s %u\n", "Enabling DP clock. Ref-count after:",
			clk_ref_after_incr);

		clk_enable(g_machdrv_dp_clk);
	} else {
		vdbg_prt("%s %u\n",
			"DP clock already Enabled. Ref-count after:",
			clk_ref_after_incr);
	}
}

static void brd_dp_clk_disable(void)
{
	uint clk_ref_after_decr;

	dbg_prt();

	clk_ref_after_decr = atomic_dec_return(&g_dp_clk_opencnt);
	if (clk_ref_after_decr == 0) {
		vdbg_prt("%s %u\n", "Disabling DP clock. Ref-count after:",
			clk_ref_after_decr);

		clk_disable(g_machdrv_dp_clk);
	} else {
		vdbg_prt("%s %u\n", "DP clock still On. Ref-count after:",
			clk_ref_after_decr);
	}
}

static int dspg_brd_init_sound_gpios(void)
{
	int err;
	uint gpio;

	dbg_prt();
	
	/* Init board-specific GPIOs: */

	if (g_brd_snd_gpios == NULL) {
		return -ENXIO;
	}


	gpio = g_brd_snd_gpios->pa_shutdn;
	if (gpio != -1) {
		err = gpio_request(gpio, DEVICE_NAME);
		if (err < 0) {
			vdbg_prt("%s\n", "Can't get the PA Shutdown gpio");
			return err;
		}
		gpio_direction_output(gpio, 0);
		gpio_set_enable(gpio, 1);
		gpio_set_pull_enable(gpio, 0);
	}


	gpio = g_brd_snd_gpios->pa_gain;
	if (gpio != -1) {
		err = gpio_request(gpio, DEVICE_NAME);
		if (err < 0) {
			vdbg_prt("%s\n", "Can't get the PA Gain gpio");
			return err;
		}
		gpio_direction_output(gpio, 1);
		gpio_set_enable(gpio, 1);
		gpio_set_pull_enable(gpio, 0);
	}

	gpio = g_brd_snd_gpios->in_ana_sw;
	if (gpio != -1) {
		err = gpio_request(gpio, DEVICE_NAME);
		if (err < 0) {
			vdbg_prt("%s\n", "Can't get the Input Analog-Switch gpio");
			return err;
		}
		gpio_direction_output(gpio, 1);
		gpio_set_enable(gpio, 1);
		gpio_set_pull_enable(gpio, 0);
	}

	gpio = g_brd_snd_gpios->out_ana_sw;
	if (gpio != -1) {	
		err = gpio_request(gpio, DEVICE_NAME);
		if (err < 0) {
			vdbg_prt("%s\n", "Can't get the Output Analog-Switch gpio");
			return err;
		}
		gpio_direction_output(gpio, 0);
		gpio_set_enable(gpio, 1);
		gpio_set_pull_enable(gpio, 0);
	}

	return 0;
}

static void dspg_brd_free_sound_gpios(void)
{
	uint gpio;

	dbg_prt();
	
	if (g_brd_snd_gpios != NULL) {

		gpio = g_brd_snd_gpios->pa_shutdn;
		if (gpio != -1) {
			/* Shutdown Pwr-Amp */
			gpio_set_value(gpio, 0);
			gpio_free(gpio);
		}

		gpio = g_brd_snd_gpios->pa_gain;
		if (gpio != -1) {

			gpio_free(gpio);
		}

		gpio = g_brd_snd_gpios->in_ana_sw;
		if (gpio != -1) {
			gpio_free(gpio);
		}

		gpio = g_brd_snd_gpios->out_ana_sw;
		if (gpio != -1) {
			gpio_free(gpio);
		}
	}
}

static void dmw_set_tel_state(void)
{
	/* Note: dai-link 0 discribes the 'connection' of the Wolfson codec
	 * to TDM 2 (third dmw96's TDM machine) on the EVB96 board.
	 */
	struct snd_soc_dai_link *dai_link = &g_brd_wm897X_dai_link[0];
	const char *cpu_dai_name = dai_link->cpu_dai_name;

	dbg_prt();

	vdbg_prt("%s %s\n", "Requested TEL state is",
		(g_dmw_tel_state) ? "ON" : "OFF");

	vdbg_prt("%s %s\n",
		 "Calling dmw_tdm_set_telephony() for cpu-dai",
		 cpu_dai_name);
	dmw_tdm_set_telephony(cpu_dai_name, g_dmw_tel_state);

	vdbg_prt("%s\n", "Calling dmw_pcm_set_telephony()");
	dmw_pcm_set_telephony(g_dmw_tel_state);
}

/*****************************************************************************/

static int brd_wm897X_hw_params(struct snd_pcm_substream *substream,
				 struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *codec_dai = rtd->codec_dai;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;
	
	unsigned int dai_format;
	struct clk * dp_clk = NULL;
	unsigned long dp_clk_rate;
	int ret;

#ifndef CONFIG_SOC_CODEC_TDM_MASTER
	uint alsa_Fs;
	struct snd_soc_codec *codec = rtd->codec;
	struct snd_soc_card *card = codec->card;
	struct wm897X_setup_data *setup_data = snd_soc_card_get_drvdata(card);
	struct tdm_master_coeff * coeff;
	unsigned long act_dp_clk_rate;
#endif /* ! CONFIG_SOC_CODEC_TDM_MASTER */

	dbg_prt("%s\n", DIR_STR(substream->stream));

	/*
	 * Note: dmw96 TDM-machine HW limitations:
	 *
	 *  - as Master:
	 *    ----------
	 *  You can configure the timing of the Rising edge of the generated
	 *  FSync signal (at either rising or falling SCLK edge), but you cannot
	 *  directly configure the timing of its Falling edge.
	 *  Thus, for generating true I2S LRC clock, we need to Shift
	 *  the Rising of the FSync in such a way that the Falling edge would take
	 *  place one bit before the start of data bits.
	 *
	 *  - as Slave:
	 *    ---------
	 *  It syncs only on Rising edge of FSync.
	 *  A solution would be to invert the FSync polarity, generated at
	 *  the Wolfson. However, it turns out that in this case the Wolfson also
	 *  puts the ADC (capture) data on the Right channel, instead of on the
	 *  Left one. So we can't use this solution.
	 *  This limitation yields left/right channel swap on playback. 
	 *  Another soultion would be to swap the channels by swapping the Wolfson
	 *  L/R DAC's output.
	 */


	/* Note: (!)
	 * Do NOT change the order of the following configuration steps !
	 *
	 * This is due to the Codec (Wolfson) Driver code dependency:
	 * It must know the Master/Slave configuration before calculating
	 * its PLL settings.
	 */

	/* STEP 1: */

	/* Set DAI's TDM configuration: */
	dai_format = 0;
	dai_format |= SND_SOC_DAIFMT_I2S;

#ifdef CONFIG_SOC_CODEC_TDM_MASTER
	/* DMW96 is TDM Slave */

	/* This setting doesn't work, due to Wolfson ADC data on the R channel. See note above.
	 * dai_format |= SND_SOC_DAIFMT_NB_IF;	// Normal SCLK(BCLK), Inverted FSync(LRC) polarity
	 */
	 
	dai_format |= SND_SOC_DAIFMT_NB_NF;	/* Normal SCLK(BCLK) & FSync(LRC) polarity */ 
	dai_format |= SND_SOC_DAIFMT_CBM_CFM;	/* Codec(!) point-of-view - Wolfson is Master */
	vdbg_prt("%s\n", "DMW96 TDM is Slave, Wolfson is Master");
#else
	/* DMW96 is TDM Master */

	dai_format |= SND_SOC_DAIFMT_NB_NF;	/* Normal SCLK(BCLK) & FSync(LRC) polarity */ 
	dai_format |= SND_SOC_DAIFMT_CBS_CFS;	/* Codec(!) point-of-view - Wolfson is Slave */
	vdbg_prt("%s\n", "DMW96 TDM is Master, Wolfson is Slave");
#endif /* !CONFIG_SOC_CODEC_TDM_MASTER */

	/* Set cpu DAI's TDM configuration */
	vdbg_prt("%s", "Calling CPU DAI set_fmt(): ");
	ret = snd_soc_dai_set_fmt(cpu_dai, dai_format);
	if (ret < 0) {
		vdbg_prt("%s\n", "CPU DAI tdm-format setting Failed !!!");
		return ret;
	}

	/* Set codec DAI's TDM configuration */
	vdbg_prt("%s\n", "Calling CODEC DAI set_fmt(): ");
	ret = snd_soc_dai_set_fmt(codec_dai, dai_format);
	if (ret < 0) {
		vdbg_prt("%s\n", "CODEC DAI tdm-format setting Failed !!!");
		return ret;
	}

	/* STEP 2: */

	/* Get DP clk rate
	 * Note: the ASOC Platform driver (dmw-tdm.c) is now the DP clock "owner".
	 * 	 See mach-dmw/clock.c
	 */
	dp_clk = clk_get_sys("dmw-tdm", "dp");
	if (dp_clk < 0) {
		return (int)dp_clk;
	}
	dp_clk_rate = clk_get_rate(dp_clk);
	vdbg_prt("%s %lu %s\n", "DP clk running at:", dp_clk_rate, "Hz");

#ifndef CONFIG_SOC_CODEC_TDM_MASTER
	alsa_Fs = params_rate(params);
	coeff = dmw_as_master(alsa_Fs);
	setup_data->lrc_fs_x100 = coeff->lrc_fs_x100;
	vdbg_prt("%s %u %s %u\n",
		"alsa_Fs:", alsa_Fs,
		"lrc_Fs_x100:", setup_data->lrc_fs_x100);

	act_dp_clk_rate = dp_clk_rate;
	act_dp_clk_rate /= 480000; /* Teoretical sysclk, in KHz */
	act_dp_clk_rate *= 480019; /* Actual sysclk, in KHz */
	vdbg_prt("%s %lu %s %lu\n",
		"Theoretical DP clk:", dp_clk_rate,
		"Actual DP clk:", act_dp_clk_rate);
	dp_clk_rate = act_dp_clk_rate;
#endif /* ! CONFIG_SOC_CODEC_TDM_MASTER */

	clk_put(dp_clk);

	/* Set Wolfson's PLL */
	vdbg_prt("%s\n", "Calling CODEC DAI set_sysclk()");
	ret = snd_soc_dai_set_sysclk(codec_dai, WM897X_PLL, (unsigned int)dp_clk_rate,
			SND_SOC_CLOCK_IN);
	if (ret < 0) {
		vdbg_prt("%s\n", "CODEC DAI sysclock setting (Wolfson internal PLL) Failed !!!");
		return ret;
	}

	/* Set Wolfson's BCLK rate.
	 * Note: Relevant only when Wolfson is the TDM Master!
	 * For Wolfson as Master, we always use 16 bits per time-slot,
	 * for All sample-rates (Fs)!
	 */
	vdbg_prt("%s\n", "Calling CODEC DAI set_clkdiv()");
	/* Divide 256*Fs by 8 to get Wolfson BCLK == dmw96 SCLK = 2*16*Fs:
	 * Reg 6 bits 4:2, BCLKDIV
	 */
	ret = snd_soc_dai_set_clkdiv(codec_dai, WM897X_BCLKDIV, 0x3 << 2);
	if (ret < 0) {
		vdbg_prt("%s\n", "CODEC DAI clkdiv setting (Wolfson BCLK output) Failed !!!");
		return ret;
	}

	return 0;
}

static int brd_wm897X_hw_free(struct snd_pcm_substream *substream)
{
	dbg_prt("%s %s\n",
		DIR_STR(substream->stream),
		"Currently doing nothing ---->");

	return 0;
}

static int brd_wm897X_prepare(struct snd_pcm_substream *substream)
{
	dbg_prt("%s\n", DIR_STR(substream->stream));

	/* Note:
	 * From the asoc sound-driver point of view, the 'Telephony' control
	 * gets updated (from the /etc/asound.conf file setting,
	 * according to the Currently Used Android sound 'device'),
	 * After brd_wm897X_hw_params() and Before brd_wm897X_prepare() are
	 * being called.
	 *
	 * The Machine-Driver's _prepare() (this dmw_pcm_prepare() function),
	 * is called Before the _prepare()s of the Platform-Drivers
	 * (PCM & TDM), so here is a good point to pass the info to them.
	 *
	 * This also means that the final TDM configuration, especially the
	 * _tdm_grant() issue, should be made in the cpu-dai's (TDM)
	 * _prepare() function !
	 */
	vdbg_prt("%s\n", "Setting TEL state");
	
	dmw_set_tel_state();

	return 0;
}

static int brd_wm897X_startup(struct snd_pcm_substream *substream)
{
	dbg_prt("%s\n", DIR_STR(substream->stream));
	
	brd_dp_clk_enable();

	return 0;
}

static void brd_wm897X_shutdown(struct snd_pcm_substream *substream)
{
	dbg_prt("%s\n", DIR_STR(substream->stream));

	brd_dp_clk_disable();
}

static int brd_wm897X_trigger(struct snd_pcm_substream *substream, int cmd)
{
	dbg_prt("%s %s\n",
		DIR_STR(substream->stream),
		"Currently doing nothing ---->");

	return 0;
}

static struct snd_soc_ops brd_wm897X_ops = {
	.startup	= brd_wm897X_startup,
	.shutdown	= brd_wm897X_shutdown,
	.hw_params	= brd_wm897X_hw_params,
	.hw_free	= brd_wm897X_hw_free,
	.prepare	= brd_wm897X_prepare,
	.trigger	= brd_wm897X_trigger,
};

/*****************************************************************************/

#ifdef CONFIG_SND_DMW_SOC_BOARD_CSR8811

static int brd_csr8811_hw_params(struct snd_pcm_substream *substream,
				 struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *codec_dai = rtd->codec_dai;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;

	unsigned int dai_format;
	struct clk * dp_clk = NULL;
	unsigned long dp_clk_rate;
	int ret;

#ifndef CONFIG_SOC_CODEC_TDM_MASTER
	uint alsa_Fs;
	struct snd_soc_codec *codec = rtd->codec;
	struct snd_soc_card *card = codec->card;
	struct csr8811_setup_data * setup_data = snd_soc_card_get_drvdata(card);
	struct tdm_master_coeff * coeff;
	unsigned long act_dp_clk_rate;
#endif /* ! CONFIG_SOC_CODEC_TDM_MASTER */

	dbg_prt("%s\n", DIR_STR(substream->stream));

	/*
	 * Note: dmw96 TDM-machine HW limitations:
	 *
	 *  - as Master:
	 *    ----------
	 *  You can configure the timing of the Rising edge of the generated
	 *  FSync signal (at either rising or falling SCLK edge), but you cannot
	 *  directly configure the timing of its Falling edge.
	 *  Thus, for generating true I2S LRC clock, we need to Shift
	 *  the Rising of the FSync in such a way that the Falling edge would take
	 *  place one bit before the start of data bits.
	 *
	 *  - as Slave:
	 *    ---------
	 *  It syncs only on Rising edge of FSync.
	 *  A solution would be to invert the FSync polarity, generated at
	 *  the Csr. The inversion is not available on the csr8811,
	 *  So we can't use this solution.
	 *  This limitation yields left/right channel swap on playback. 
	 */


	/* Note: (!)
	 * Do NOT change the order of the following configuration steps !
	 *
	 * This is due to the Codec (Csr) Driver code dependency:
	 * It must know the Master/Slave configuration before calculating
	 * its PLL settings.
	 */

	/* STEP 1: */

	/* Set DAI's TDM configuration: */
	dai_format = 0;
	dai_format |= SND_SOC_DAIFMT_I2S; /* TODO changes for CSR8811 !!! */

#ifdef CONFIG_SOC_CODEC_TDM_MASTER
	/* DMW is TDM Slave */
	dai_format |= SND_SOC_DAIFMT_NB_NF;	/* Normal SCLK(BCLK) & FSync(LRC) polarity */ 
	dai_format |= SND_SOC_DAIFMT_CBM_CFM;	/* Codec(!) point-of-view - Csr is Master */
	vdbg_prt("%s\n", "DMW96 TDM is Slave, Csr is Master");
#else
	/* DMW is TDM Master */
	dai_format |= SND_SOC_DAIFMT_NB_NF;	/* Normal SCLK(BCLK) & FSync(LRC) polarity */ 
	dai_format |= SND_SOC_DAIFMT_CBS_CFS;	/* Codec(!) point-of-view - Csr is Slave */
	vdbg_prt("%s\n", "DMW96 TDM is Master, Csr is Slave");
#endif /* !CONFIG_SOC_CODEC_TDM_MASTER */

	/* Set cpu DAI configuration */
	vdbg_prt("%s", "Calling CPU DAI set_fmt(): ");
	ret = snd_soc_dai_set_fmt(cpu_dai, dai_format);
	if (ret < 0) {
		return ret;
	}

	/* Set codec DAI configuration */
	vdbg_prt("%s\n", "Calling CODEC DAI set_fmt(): ");
	ret = snd_soc_dai_set_fmt(codec_dai, dai_format);
	if (ret < 0) {
		return ret;
	}

	/* STEP 2: */

	/* Get DP clk rate
	 * Note: the ASOC Platform driver (dmw-tdm.c) is now the DP clock "owner".
	 * 	 See mach-dmw/clock.c
	 */
	dp_clk = clk_get_sys("dmw-tdm", "dp");
	if (dp_clk < 0) {
		return (int)dp_clk;
	}
	dp_clk_rate = clk_get_rate(dp_clk);
	vdbg_prt("%s %lu %s\n", "DP clk running at:", dp_clk_rate, "Hz");

#ifndef CONFIG_SOC_CODEC_TDM_MASTER
	alsa_Fs = params_rate(params);
	coeff = dmw_as_master(alsa_Fs);
	setup_data->lrc_fs_x100 = coeff->lrc_fs_x100;
	vdbg_prt("%s %u %s %u\n",
		"alsa_Fs:", alsa_Fs,
		"lrc_Fs_x100:", setup_data->lrc_fs_x100);

	act_dp_clk_rate = dp_clk_rate;
	act_dp_clk_rate /= 480000; /* Teoretical sysclk, in KHz */
	act_dp_clk_rate *= 480019; /* Actual sysclk, in KHz */
	vdbg_prt("%s %lu %s %lu\n",
		"Theoretical DP clk:", dp_clk_rate,
		"Actual DP clk:", act_dp_clk_rate);
	dp_clk_rate = act_dp_clk_rate;
#endif /* ! CONFIG_SOC_CODEC_TDM_MASTER */

	clk_put(dp_clk);

	/* Set Csr's PLL */
	vdbg_prt("%s\n", "Calling CODEC DAI set_sysclk()");
	ret = snd_soc_dai_set_sysclk(codec_dai, CSR8811_PLL, (unsigned int)dp_clk_rate,
			SND_SOC_CLOCK_IN);
	if (ret < 0) {
		vdbg_prt("%s\n", "CODEC DAI sysclock setting Failed !!!");
		return ret;
	}

	/* Set Csr's BCLK rate.
	 * Note: Relevant only when Csr is the TDM Master!
	 * For Csr as Master, we always use 16 bits per time-slot,
	 * for All sample-rates (Fs)!
	 */
	vdbg_prt("%s\n", "Calling CODEC DAI set_Clkdiv()");
	ret = snd_soc_dai_set_clkdiv(codec_dai, CSR8811_BCLKDIV, 0);
	if (ret < 0) {
		vdbg_prt("%s\n", "CODEC DAI clkdiv setting Failed !!!");
		return ret;
	}

	return 0;
}

static int brd_csr8811_hw_free(struct snd_pcm_substream *substream)
{
	dbg_prt("%s %s\n",
		DIR_STR(substream->stream),
		"Currently doing nothing ---->");

	return 0;
}

static int brd_csr8811_prepare(struct snd_pcm_substream *substream)
{
	dbg_prt("%s\n", DIR_STR(substream->stream));

	/* Note:
	 * From the asoc sound-driver point of view, the 'Telephony' control
	 * gets updated (from the /etc/asound.conf file setting,
	 * according to the Currently Used Android sound 'device'),
	 * After brd_wm897X_hw_params() and Before brd_wm897X_prepare() are
	 * being called.
	 *
	 * The Machine-Driver's _prepare() (this dmw_pcm_prepare() function),
	 * is called Before the _prepare()s of the Platform-Drivers
	 * (PCM & TDM), so here is a good point to pass the info to them.
	 *
	 * This also means that the final TDM configuration, especially the
	 * _tdm_grant() issue, should be made in the cpu-dai's (TDM)
	 * _prepare() function !
	 */
	vdbg_prt("%s\n", "Setting TEL state");
	
	dmw_set_tel_state();

	return 0;
}

static int brd_csr8811_startup(struct snd_pcm_substream *substream)
{
	dbg_prt("%s %s\n",
		DIR_STR(substream->stream),
		"Currently doing nothing ---->");

	return 0;
}

static void brd_csr8811_shutdown(struct snd_pcm_substream *substream)
{
	dbg_prt("%s %s\n",
		DIR_STR(substream->stream),
		"Currently doing nothing ---->");
}

static int brd_csr8811_trigger(struct snd_pcm_substream *substream, int cmd)
{
	dbg_prt("%s %s\n",
		DIR_STR(substream->stream),
		"Currently doing nothing ---->");

	return 0;
}

static struct snd_soc_ops g_brd_csr8811_ops = {
	.startup	= brd_csr8811_startup,
	.shutdown	= brd_csr8811_shutdown,
	.hw_params	= brd_csr8811_hw_params,
	.hw_free	= brd_csr8811_hw_free,
	.prepare	= brd_csr8811_prepare,
	.trigger	= brd_csr8811_trigger,
};

#endif /* CONFIG_SND_DMW_SOC_BOARD_CSR8811 */

/*****************************************************************************/

#ifdef CONFIG_SND_DMW_SOC_EVB96_SI3050

static int dmw_get_fxoaudio(struct snd_kcontrol *kcontrol,
               struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = g_dmw_fxo_audiostate;
	vdbg_prt("%s %s\n", "FXO Audio state:",
		(g_dmw_fxo_audiostate == 0) ? "Off" : "On");

	return 0;
}

static int dmw_set_fxoaudio(struct snd_kcontrol *kcontrol,
               struct snd_ctl_elem_value *ucontrol)
{
	if (g_dmw_fxo_audiostate == ucontrol->value.integer.value[0]) {
		vdbg_prt("%s %s\n", "FXO Audio state Unchanged:",
			(g_dmw_fxo_audiostate == 0) ? "Off" : "On");
		return 0;
	}

	g_dmw_fxo_audiostate = ucontrol->value.integer.value[0];
	vdbg_prt("%s %s\n", "FXO Audio state:",
		(g_dmw_fxo_audiostate == 0) ? "Off" : "On");
	return 0;
}

static const struct snd_kcontrol_new fxo_audiocontrols[] = {
    SOC_ENUM_EXT("FXO_AUDIO", tel_enum, dmw_get_fxoaudio, dmw_set_fxoaudio),
};


static void dmw_update_fxo_audiostate(void)
{
	struct snd_soc_dai_link *dai_link = &g_brd_wm897X_dai_link[SI3050_DAI];
	const char *cpu_dai_name = dai_link->cpu_dai_name;

	dbg_prt();

	vdbg_prt("%s %s\n", "Requested FXO Audio state is",
		(g_dmw_fxo_audiostate) ? "ON" : "OFF");

	vdbg_prt("%s %s\n",
		"Calling dmw_tdm_set_telephony() for cpu-dai",
		cpu_dai_name);
	dmw_tdm_set_telephony(cpu_dai_name, g_dmw_fxo_audiostate);

	vdbg_prt("%s\n", "Calling dmw_pcm_set_telephony()");
	dmw_pcm_set_telephony(g_dmw_fxo_audiostate);
}

static int brd_si3050_hw_params(struct snd_pcm_substream *substream,
                 struct snd_pcm_hw_params *params)
{
	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_dai *cpu_dai = rtd->cpu_dai;

	unsigned int dai_format;
	int ret;

	dbg_prt("%s\n", DIR_STR(substream->stream));

	/* STEP 1: */

	/* Set DAI's TDM configuration: */
	dai_format = 0;
	dai_format |= SND_SOC_DAIFMT_I2S;

#ifdef CONFIG_SOC_CODEC_TDM_MASTER
	/* DMW96 is TDM Slave */

	dai_format |= SND_SOC_DAIFMT_NB_NF; /* Normal SCLK(BCLK) & FSync(LRC) polarity */ 
	dai_format |= SND_SOC_DAIFMT_CBM_CFM;   /* Codec(!) point-of-view - si3050 is Master */
	vdbg_prt("%s\n", "DMW96 TDM is Slave, SI3050 is Master");
#else
	/* DMW96 is TDM Master */

	dai_format |= SND_SOC_DAIFMT_NB_NF; /* Normal SCLK(BCLK) & FSync(LRC) polarity */ 
	dai_format |= SND_SOC_DAIFMT_CBS_CFS;   /* Codec(!) point-of-view - si3050 is Slave */
	vdbg_prt("%s\n", "DMW96 TDM is Master, SI3050 is Slave");
#endif /* !CONFIG_SOC_CODEC_TDM_MASTER */

	/* Set cpu DAI's TDM configuration */
	vdbg_prt("%s", "Calling CPU DAI set_fmt(): ");
	ret = snd_soc_dai_set_fmt(cpu_dai, dai_format);
	if (ret < 0) {
		vdbg_prt("%s\n", "CPU DAI tdm-format setting Failed !!!");
		return ret;
	}

	return 0;
}


static int brd_si3050_hw_free(struct snd_pcm_substream *substream){
	dbg_prt("%s %s\n",
		DIR_STR(substream->stream),
		"Currently doing nothing ---->");

	return 0;
}


static int brd_si3050_prepare(struct snd_pcm_substream *substream){
	dbg_prt("%s\n", DIR_STR(substream->stream));
	vdbg_prt("%s\n", "Setting TEL state");
	
	dmw_update_fxo_audiostate();
	return 0;
}

static int brd_si3050_startup(struct snd_pcm_substream *substream){
	dbg_prt("%s %s\n",
		DIR_STR(substream->stream),
		"Currently doing nothing ---->");

	return 0;
}

static void brd_si3050_shutdown(struct snd_pcm_substream *substream){
	dbg_prt("%s %s\n",
		DIR_STR(substream->stream),
		"Currently doing nothing ---->");
}

static int brd_si3050_trigger(struct snd_pcm_substream *substream, int cmd){
	dbg_prt("%s %s\n",
		DIR_STR(substream->stream),
		"Currently doing nothing ---->");

	return 0;
}


static struct snd_soc_ops g_brd_si3050_ops = {
	.startup    = brd_si3050_startup,
	.shutdown   = brd_si3050_shutdown,
	.hw_params  = brd_si3050_hw_params,
	.hw_free    = brd_si3050_hw_free,
	.prepare    = brd_si3050_prepare,
	.trigger    = brd_si3050_trigger,
};

#endif /* CONFIG_SND_DMW_SOC_EVB96_SI3050 */

/*****************************************************************************/

static int dmw_get_tel(struct snd_kcontrol *kcontrol,
		       struct snd_ctl_elem_value *ucontrol)
{
	ucontrol->value.integer.value[0] = g_dmw_tel_state;

	vdbg_prt("%s %s\n", "TEL state is:",
		(g_dmw_tel_state == 0) ? "Off" : "On");
	
	return 0;
}

static int dmw_set_tel(struct snd_kcontrol *kcontrol,
		       struct snd_ctl_elem_value *ucontrol)
{
	if (g_dmw_tel_state == ucontrol->value.integer.value[0]) {
		vdbg_prt("%s %s\n",
			"TEL state Unchanged:",
			(g_dmw_tel_state == 0) ? "Off" : "On");
		return 0;
	}

	g_dmw_tel_state = ucontrol->value.integer.value[0];

	vdbg_prt("%s %s\n", "Updating TEL state to:",
		(g_dmw_tel_state == 0) ? "Off" : "On");
	
	return 1;
}


static const struct snd_kcontrol_new tel_controls[] = {
	SOC_ENUM_EXT("Telephony", tel_enum, dmw_get_tel, dmw_set_tel),
};

/*****************************************************************************/

/* board-specific controls: */

/* Output Power-Amplifier controls: */

static int g_pa_shutdn_cntrl = 0;
static int g_pa_shutdn_dapm = 0;

static int brd_apply_pwramp_shutdn(void)
{
	int pa_shutdn_st;
	int composite_pa_shutdn_st;
	const uint p_gpio = g_brd_snd_gpios->pa_shutdn;

	if (p_gpio == -1) {
		vdbg_prt("%s\n", "power amp shutdown is not supported in this board:");
		return 0;
	}

	pa_shutdn_st = gpio_get_value(p_gpio);
	composite_pa_shutdn_st = g_pa_shutdn_cntrl && g_pa_shutdn_dapm;
	if (pa_shutdn_st == composite_pa_shutdn_st) {
		vdbg_prt("%s %s\n", "Output PwrAmp shutdown Unchanged:",
			pwramp_shutdn_st[pa_shutdn_st]);
		return 0;
	}

	vdbg_prt("%s %s\n", "Setting PwrAmp shutdown to:",
		pwramp_shutdn_st[composite_pa_shutdn_st]);
	gpio_set_value(g_brd_snd_gpios->pa_shutdn, composite_pa_shutdn_st);
	return 1;
}

/* PA Shutdown: */
static int brd_get_pwramp_shutdn(struct snd_kcontrol *kcontrol,
			  struct snd_ctl_elem_value *ucontrol)
{
	int pa_shutdn_st;
	const uint p_gpio = g_brd_snd_gpios->pa_shutdn;

	if (p_gpio == -1) {
		vdbg_prt("%s\n", "power amp shutdown is not supported in this board:");
		return 0;
	}

	pa_shutdn_st = gpio_get_value(p_gpio);
	vdbg_prt("%s %s\n", "Output PwrAmp shutdown is:",
		pwramp_shutdn_st[pa_shutdn_st]);
	ucontrol->value.integer.value[0] = pa_shutdn_st;
	return 0;
}

static int brd_set_pwramp_shutdn(struct snd_kcontrol *kcontrol,
			struct snd_ctl_elem_value *ucontrol)
{
	g_pa_shutdn_cntrl = ucontrol->value.integer.value[0];
	vdbg_prt("%s %s\n", "PwrAmp shutdown: Control request to:",
		pwramp_shutdn_st[g_pa_shutdn_cntrl]);
	return brd_apply_pwramp_shutdn();
}

static int brd_dapm_pwramp_shutdn(struct snd_soc_dapm_widget *w,
			struct snd_kcontrol *k, int event)
{
	g_pa_shutdn_dapm = SND_SOC_DAPM_EVENT_ON(event) ? 1 : 0;
	vdbg_prt("%s %s\n", "PwrAmp shutdown: Dapm request to:",
		pwramp_shutdn_st[g_pa_shutdn_dapm]);
	return brd_apply_pwramp_shutdn();
}

/* PA Gain: */
static int brd_get_pwramp_gain(struct snd_kcontrol *kcontrol,
			struct snd_ctl_elem_value *ucontrol)
{
	int pa_gain_st;
	const uint p_gpio = g_brd_snd_gpios->pa_gain;

	if (p_gpio == -1) {
		vdbg_prt("%s\n", "power amp gain selection is not supported in this board:");
		return 0;
	}
	
	pa_gain_st = gpio_get_value(p_gpio);
	vdbg_prt("%s %s\n", "Output PwrAmp gain is:",
		pwramp_gain_st[pa_gain_st]);
	ucontrol->value.integer.value[0] = pa_gain_st;
	return 0;

}

static int brd_set_pwramp_gain(struct snd_kcontrol *kcontrol,
			struct snd_ctl_elem_value *ucontrol)
{
	int pa_gain_st;
	const uint p_gpio = g_brd_snd_gpios->pa_gain;

	if (p_gpio == -1) {
		vdbg_prt("%s\n", "power amp gain selection is not supported in this board:");
		return 0;
	}

	pa_gain_st = gpio_get_value(p_gpio);
	if (pa_gain_st == ucontrol->value.integer.value[0]) {
		vdbg_prt("%s %s\n", "Output PwrAmp gain Unchanged:",
			pwramp_gain_st[pa_gain_st]);
		return 0;
	}

	pa_gain_st = ucontrol->value.integer.value[0];
	vdbg_prt("%s %s\n", "Setting PwrAmp gain to:",
		pwramp_gain_st[pa_gain_st]);
	gpio_set_value(p_gpio, pa_gain_st);
	return 1;
}

/* Input Analog Switch: */
static int brd_get_mic_sel(struct snd_kcontrol *kcontrol,
		    struct snd_ctl_elem_value *ucontrol)
{
	int mic_sel_st;
	const uint p_gpio = g_brd_snd_gpios->in_ana_sw;
	
	if (p_gpio == -1) {
		vdbg_prt("%s\n", "Mic selection is not supported in this board:");
		return 0;
	}

	mic_sel_st = gpio_get_value(p_gpio);
	vdbg_prt("%s %s\n", "Mic selection is:",
		micr_sel_st[mic_sel_st]);
	ucontrol->value.integer.value[0] = mic_sel_st;
	return 0;
	
}

static int brd_set_mic_sel(struct snd_kcontrol *kcontrol,
		    struct snd_ctl_elem_value *ucontrol)
{
	int mic_sel_st;
	const uint p_gpio = g_brd_snd_gpios->in_ana_sw;

	if (p_gpio == -1) {
		vdbg_prt("%s\n", "Mic selection is not supported in this board:");
		return 0;
	}
	mic_sel_st = gpio_get_value(p_gpio);
	if (mic_sel_st == ucontrol->value.integer.value[0]) {
		vdbg_prt("%s %s\n", "Mic selection Unchanged:",
			micr_sel_st[mic_sel_st]);
		return 0;
	}
	
	mic_sel_st = ucontrol->value.integer.value[0];
	vdbg_prt("%s %s\n", "Setting Mic selection to:",
		micr_sel_st[mic_sel_st]);
	gpio_set_value(p_gpio, mic_sel_st);
	return 1;

}

/* Output Anlog Switch: */
static int brd_get_output_sel(struct snd_kcontrol *kcontrol,
		       struct snd_ctl_elem_value *ucontrol)
{
	int out_ana_sw_st;
	const uint p_gpio = g_brd_snd_gpios->out_ana_sw;

	if (p_gpio == -1) {
		vdbg_prt("%s\n", "output selection is not supported in this board:");
		return 0;
	}

	out_ana_sw_st = gpio_get_value(p_gpio);
	vdbg_prt("%s %s\n", "Output Analog Switch connects:",
		outdev_sel_st[out_ana_sw_st]);
	ucontrol->value.integer.value[0] = out_ana_sw_st;
	return 0;

}

static int brd_set_output_sel(struct snd_kcontrol *kcontrol,
		       struct snd_ctl_elem_value *ucontrol)
{
	int out_ana_sw_st;
	const uint p_gpio = g_brd_snd_gpios->out_ana_sw;

	if (p_gpio == -1) {
		vdbg_prt("%s\n", "output selection is not supported in this board:");
		return 0;
	}

	out_ana_sw_st = gpio_get_value(p_gpio);

	if (out_ana_sw_st == ucontrol->value.integer.value[0]) {
		vdbg_prt("%s %s\n", "Output Analog Switch Unchanged:",
			outdev_sel_st[out_ana_sw_st]);			
		return 0;
	}

	out_ana_sw_st = ucontrol->value.integer.value[0];
	vdbg_prt("%s %s\n", "Setting Analog Switch to:",
		outdev_sel_st[out_ana_sw_st]);
	gpio_set_value(p_gpio, out_ana_sw_st);
	return 1;
		
}

/*****************************************************************************/

static int brd_wm897X_dai_init(struct snd_soc_pcm_runtime *rtd)
{
	int ret;
	struct snd_soc_codec *codec = rtd->codec;
	struct snd_soc_dapm_context * dapm = &codec->dapm;

	// Do Not delete - possible future use !
	//struct snd_soc_card *card = codec->card;
	//struct dspg_drv_priv_data *priv_data = snd_soc_card_get_drvdata(card);
	//void *pltfrm_data = priv_data->pdata;
	
	dbg_prt();

	vdbg_prt("%s\n", "Adding Telephony control");
	ret = snd_soc_add_controls(codec, tel_controls,
				ARRAY_SIZE(tel_controls));
	if (ret) {
		vdbg_prt("%s\n", "Telephony control adding Failed");
		return ret;
	}

	vdbg_prt("%s\n", "Adding audio control");
	ret = snd_soc_add_controls(codec, brd_audio_controls,
				   len_of_control_array);
	if (ret) {
		vdbg_prt("%s\n", "Audio control adding Failed");
		return ret;
	}

	vdbg_prt("%s\n", "Adding dapm widgets");
	ret = snd_soc_dapm_new_controls(dapm, brd_dapm_widgets,
					len_of_widget_array);
	if (ret) {
		vdbg_prt("%s\n", "Dapm widgets adding Failed");
		return ret;
	}

	vdbg_prt("%s\n", "Adding dapm routes");
	ret = snd_soc_dapm_add_routes(dapm, brd_audio_route,
				      len_of_route_array);
	if (ret) {
		vdbg_prt("%s\n", "Dapm routes adding Failed");
		return ret;
	}

	/* Disable Wolfson unused input-pins Permanently */
	ret = 0;
	if (!ret) {
		ret = snd_soc_dapm_nc_pin(dapm, "AUXL");
	}
	if (!ret) {
		ret = snd_soc_dapm_nc_pin(dapm, "AUXR");
	}
	if (!ret) {
		ret = snd_soc_dapm_nc_pin(dapm, "L2");
	}
	if (!ret) {
		ret = snd_soc_dapm_nc_pin(dapm, "R2");
	}
	if (ret) {
		vdbg_prt("%s\n", "dapm pin disabling Failed");
		return ret;
	}

	vdbg_prt("%s\n", "Syncing dapm");
	ret = snd_soc_dapm_sync(dapm);
	if (ret) {
		vdbg_prt("%s\n", "Dapm syncing Failed");
		return ret;
	}

	return 0;
}

#ifdef CONFIG_SND_DMW_SOC_BOARD_CSR8811
static int brd_csr8811_dai_init(struct snd_soc_pcm_runtime *rtd)
{
	int ret;
	struct snd_soc_codec *codec = rtd->codec;
	struct snd_soc_dapm_context * dapm = &codec->dapm;

	// Do Not delete - possible future use !
	//struct snd_soc_card *card = codec->card;
	//struct dspg_drv_priv_data *priv_data = snd_soc_card_get_drvdata(card);
	//void *pltfrm_data = priv_data->pdata;
	
	dbg_prt();

	vdbg_prt("%s\n", "Syncing dapm");
	ret = snd_soc_dapm_sync(dapm);
	if (ret) {
		vdbg_prt("%s\n", "Dapm syncing Failed");
		return ret;
	}

	return 0;
}
#endif /* CONFIG_SND_DMW_SOC_BOARD_CSR8811 */

#ifdef CONFIG_SND_DMW_SOC_EVB96_SI3050
static int brd_si3050_dai_link_init(struct snd_soc_pcm_runtime *rtd)
{
	struct snd_soc_codec *codec = rtd->codec;
	int ret;

	/* Note: this is the "probe" function */
	dbg_prt();

	vdbg_prt("%s\n", "Adding FXO control");
	ret = snd_soc_add_controls(codec, fxo_audiocontrols,
			ARRAY_SIZE(fxo_audiocontrols));
	if (ret) {
		vdbg_prt("%s\n", "FXO Audio control adding Failed");
		return ret;
	}

	return 0;
}
#endif /* CONFIG_SND_DMW_SOC_EVB96_SI3050 */


static struct snd_soc_dai_link g_brd_wm897X_dai_link[] = {
	{
	/* Index 0 : WM897X conneted to DMW96 over TDM 2 : */
		
		.name = "dmw96-wm897X_Dai_Link",
		.stream_name = "dmw96-wm897X_I2S",

		/* asoc Codec device name + its I2C full address
		 * ('name'.'i2c-bus-#'-'4-digit-addr')
		 */
		.codec_name = "wm897X.0-001a",
		
		/* asoc Platform driver (PCM) name */
		.platform_name = "dmw-pcm-audio",
		
		/* asoc Cpu-Dai (TDM) device name */
		.cpu_dai_name = "dmw-tdm.2",
		
		/* asoc Codec-Dai device name */
		.codec_dai_name = "wm897X-hifi",

		.init = brd_wm897X_dai_init,
		.ops = &brd_wm897X_ops,
	},

#ifdef CONFIG_SND_DMW_SOC_BOARD_CSR8811
	{
	/* Index 1 : CSR8811 conneted to DMW96 over TDM 1 : */

		.name = "dmw96-csr8811_Dai_Link",
		.stream_name = "dmw96-csr8811_I2S",

		/* asoc Codec device name */
		.codec_name = "BT-csr8811",

		/* asoc Platform driver (PCM) name */
		.platform_name = "dmw-pcm-audio",
		
		/* asoc Cpu-Dai (TDM) device name */
		.cpu_dai_name = "dmw-tdm.1",
		
		/* asoc Codec-Dai device name */
		.codec_dai_name = "csr8811-hifi",
		 
		.init = brd_csr8811_dai_init,
		.ops = &g_brd_csr8811_ops,
	},
#endif /* CONFIG_SND_DMW_SOC_BOARD_CSR8811 */

#ifdef CONFIG_SND_DMW_SOC_EVB96_SI3050
	{
	/* Index 2 : SI3050 conneted to DMW96 over TDM 0 : */

		.name = "dmw96-si3050_dai_link",
		.stream_name = "dmw96-si3050_I2S",

		/* asoc Codec device name */
		.codec_name = "spi1.0",

		/* asoc Platform driver (PCM) name */
		.platform_name = "dmw-pcm-audio",

		/* asoc Cpu-Dai (TDM) device name */
		.cpu_dai_name = "dmw-tdm.0",

		/* asoc Codec-Dai device name */
		.codec_dai_name = "si3050-fxo",

		.init = brd_si3050_dai_link_init,
		.ops = &g_brd_si3050_ops,
	},
#endif /* CONFIG_SND_DMW_SOC_EVB96_SI3050 */

};

/*****************************************************************************/

static int brd_wm897X_suspend_pre(struct snd_soc_card *card)
{
	dbg_prt("%s\n", "Currently doing nothing ---->");

	return 0;
}

static int brd_wm897X_suspend_post(struct snd_soc_card *card)
{
	dbg_prt("%s\n", "Currently doing nothing ---->");

	return 0;
}

static int brd_wm897X_resume_pre(struct snd_soc_card *card)
{
	dbg_prt("%s\n", "Currently doing nothing ---->");

	return 0;
}

static int brd_wm897X_resume_post(struct snd_soc_card *card)
{
	dbg_prt("%s\n", "Currently doing nothing ---->");

	return 0;
}

static struct snd_soc_card g_dspg_wm897X_card = {

	.name		= "dspg-card-wm897X",
	.suspend_pre	= brd_wm897X_suspend_pre,
	.suspend_post	= brd_wm897X_suspend_post,
	.resume_pre	= brd_wm897X_resume_pre,
	.resume_post	= brd_wm897X_resume_post,
	.dai_link	= g_brd_wm897X_dai_link,  /* DAI-link Array(!) */
	.num_links	= ARRAY_SIZE(g_brd_wm897X_dai_link),

	/* Note: new to ASoC:
	 * Card's (not Codec's !) set_bias_level().
	 * May be used for certain activities, e.g. diasbling
	 * card's power amplifiers or enabling them just upon BIAS_ON;
	 * this is if they are Not declared as card's DAPM Widgests,
	 * as part of the audio Routes.
	 */
	.set_bias_level		= NULL,
	.set_bias_level_post	= NULL,
};

/*****************************************************************************/

static int dspg_snd_probe(struct platform_device *pdev)
{
	int ret;
	struct wm897X_setup_data *priv_data;
	struct snd_soc_card *card = &g_dspg_wm897X_card;
	//void *pltfrm_data = pdev->dev.platform_data;

	dbg_prt();

	/* Init board's sound-related GPIOs */
	g_brd_snd_gpios = (struct dmw_board_pdata*)pdev->dev.platform_data;

	ret = dspg_brd_init_sound_gpios();

	if (ret) {
		vdbg_prt("%s %i\n", "GPIOs init Failed:", ret);
		return ret;
	}

	/* Get DP clock handle (for Suspend/Resume operations) */
	g_machdrv_dp_clk = clk_get_sys("dmw-tdm", "dp");
	if (g_machdrv_dp_clk) {
		vdbg_prt("%s\n", "Got DP clk");

	} else {
		vdbg_prt("%s\n", "Can't get DP clk");
		return (int)g_machdrv_dp_clk;
	}

	/* Allocate mach-driver private data
	 * (for passing solution-specific info to
	 *    Codec & Platform(TDM) drivers):
	 *    - Tie-Off usage --> to Codec driver
	 *    - LRC (FSYnc) practical rate --> to TDM driver
	 * )
	 */
	priv_data = kzalloc(sizeof(struct wm897X_setup_data *), GFP_KERNEL);
	if (!priv_data) {
		vdbg_prt("%s\n", "Can't allocate private data");
		return -ENOMEM;
	}

	/* Book keeping : */
	
	/* note: platform_set_drvdata() here saves pointer to the card's data
	 * on the device's 'struct device_private *p'->driver_data
	 */
	card->dev = &pdev->dev;
	platform_set_drvdata(pdev, card);

	/* Keep copy of board's codec-private-data on the card */
	*priv_data = hw_data_for_codec;
	
	/* note: this sets card->drvdata = priv_data */
	snd_soc_card_set_drvdata(card, priv_data);

	/* Register ASoC sound Card */
	vdbg_prt("%s\n", "Registaring asoc Card");
	ret = snd_soc_register_card(card);
	if (ret) {
		vdbg_prt("%s %i\n", "snd_soc_register_card Failed", ret);
		goto ERR_CLEAR;
	}
	vdbg_prt("%s\n", "asoc Card registred OK");

	return 0;

ERR_CLEAR:

	snd_soc_card_set_drvdata(card, NULL);
	platform_set_drvdata(pdev, NULL);
	card->dev = NULL;
	kfree(priv_data);
	clk_put(g_machdrv_dp_clk);
	g_machdrv_dp_clk = NULL;
		
	return ret; 
}

static int dspg_snd_remove(struct platform_device *pdev)
{
	struct snd_soc_card *card = platform_get_drvdata(pdev);
	struct wm897X_setup_data* priv_data = snd_soc_card_get_drvdata(card);

	dbg_prt();

	snd_soc_unregister_card(card);

	/* Cleanup: */
	snd_soc_card_set_drvdata(card, NULL);
	platform_set_drvdata(pdev, NULL);
	card->dev = NULL;
	kfree(priv_data);

	dspg_brd_free_sound_gpios();

	if (g_machdrv_dp_clk) {
		clk_put(g_machdrv_dp_clk);
		g_machdrv_dp_clk = NULL;
	}
	
	return 0;	
}

static struct platform_driver g_brd_wm897X_snd_drv = {
	.driver = {
		.name = DRIVER_NAME,
		.owner = THIS_MODULE,

		/* A.M. New ASoC feature - to test */
		.pm = &snd_soc_pm_ops,
	},
	.probe = dspg_snd_probe,
	.remove = __devexit_p(dspg_snd_remove),
};

/*****************************************************************************/

static int __init brd_wm897X_mod_init(void)
{
	int err;

	dbg_prt();

	/* Check board matching */
	if (!check_valid_board()) {
		vdbg_prt("%s\n", "Not my board");
		return -ENODEV;
	}
	vdbg_prt("%s\n", "Valid board found");


	vdbg_prt("%s\n", "Registrating asoc machine driver");

	err = platform_driver_register(&g_brd_wm897X_snd_drv);
	if (err) {
		vdbg_prt("%s\n", "asoc machine driver registration Failed !!!");
	}

	return err;
}

static void __exit brd_wm897X_mod_exit(void)
{
	dbg_prt();

	vdbg_prt("%s\n", "UnRegistrating asoc machine driver");

	platform_driver_unregister(&g_brd_wm897X_snd_drv);
}

module_init(brd_wm897X_mod_init);
module_exit(brd_wm897X_mod_exit);

MODULE_DESCRIPTION("ASoC machine driver for DSPG board with WM8976 or WM8978");
MODULE_AUTHOR("DSP Group");
MODULE_LICENSE("GPL");
