/*
 * dmw-tdm.c -- DMW ASoC CPU DAI driver
 *
 *  Copyright (C) 2011 DSPG Technologies GmbH
 *
 * 16 October, 2011:
 * Adapted to kernel 2.6.39 by Avi Miller <Avi.Miller@dspg.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#define DEBUG
#define V_DEBUG

#include <linux/clk.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/module.h>
#include <linux/slab.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>

#include <mach/coma-tdm.h>

#include <sound/dmw-tdm.h>
#include "dmw-tdm-mstr.h"

/*****************************************************************************/

#ifdef V_DEBUG
	#ifndef DEBUG 
		#define DEBUG
	#endif
	#define vdbg_prt(fmt, ...) \
		pr_debug("asoc Pltfrm TDM: %s(): " fmt, __func__, ##__VA_ARGS__)
#else
	#define vdbg_prt(fmt, ...)
#endif


#ifdef DEBUG
	#define dbg_prt(fmt, ...) \
		pr_debug("asoc Pltfrm TDM: %s(): " fmt, __func__, ##__VA_ARGS__)

	#define DIR_STR(stream_dir)  ((stream_dir == 0) ? "for Playback" : "for Capture")
#else
	#define dbg_prt(fmt, ...)
	#define DIR_SRT(dir)
#endif

/*****************************************************************************/

/* Note: There are 3 TDM machines in the dmw-96 chip, each one having its
 *       own TDM-clock. However, there is a Single (global) DP-clock, driving
 *       the codec chip(or codec chips !) mounted on the board.
 */

#define DEVICE_NAME "dmw-tdm"

#define DRV_NAME "dmw-tdm"

#define NUM_DMW96_TDMs  3

#define SYSCLK__480_MHz	480000000UL
#define SYSCLK__246_MHz	246000000UL
/* Add other system-clock freq */
#define SYSCLK__DEFAULT	SYSCLK__480_MHz

enum eSYS_CLK {
	kSYS_CLK_480MHz,
	kSYS_CLK_246MHz,
	/* Add for other system-clock freq */
	kSYS_CLK_DEFAULT,
	kSYS_CLK_MAX
};

enum eFS_HZ {
	kFS_HZ_8000,
	kFS_HZ_11025,
	kFS_HZ_16000,
	kFS_HZ_22050,
	kFS_HZ_32000,
	kFS_HZ_44100,
	kFS_HZ_48000,
	kFS_HZ_MAX
};

struct tdm_clks {
	uint dp_clk_rate;
	struct tdm_master_coeff coeff[ kFS_HZ_MAX ];
};

static const struct tdm_clks tdm_clks_tab[ kSYS_CLK_MAX ] = {

    /* A.M. 14 July, 2011. Note:
     * Actual LRC/FSync frequency, as measured with scope on my EVB96 board, differs somewhat
     * from the calculated values. This means that the SYSCLK (PLL1 output frequency) slightly
     * differs from the theoretical 480.000MHz.
     * The 12MHz xTal, driving the PLL1 input, is +/-30 ppm only, so it might be wrong xTal
     * capacitors.
     *
     * The deviatation mesured is, for example, 48,001.9 (measured) Hz instead of
     * 48,000.00 (calculated) Hz, or 44,119.4 Hz instead of 44,117.65 Hz.
     * This implies that SYSCLK is actualy 480 MHz * (44,119.4 / 44,117.65) = 480.019 MHz !
     *
     * The DP clock is deviated as well - 12,000.476 Hz instead of 12,000.000 Hz.
     * The Wolfson internal PLL calculation, for generation 256*Fs, is based on its 
     * input MCLK (which is the DP clock) frequency, thus, for final tuning, I set
     * the .lrc_fs_x100 values with the Measured values
     * The .dp_clk_rate value will be adjusted to its Real value by the asoc Machine-driver,
     * before handling it to the Codec(Wolfson)-driver, for 256*Fs internal PLL calculations !
     */

    /* kSYS_CLK_480MHz */
#if 0
    /* Calculated LRC/FSync values */
    {	.dp_clk_rate = 12000000, /*19200000*/
	.coeff = {
	    { .fs =  8000, .N_bits_in_ts = 20, .sclk_divider = 1500, .lrc_fs_x100 =  800000 },
	    { .fs = 11025, .N_bits_in_ts = 19, .sclk_divider = 1146, .lrc_fs_x100 = 1102232 },
	    { .fs = 16000, .N_bits_in_ts = 20, .sclk_divider =  750, .lrc_fs_x100 = 1600000 },
	    { .fs = 22050, .N_bits_in_ts = 19, .sclk_divider =  573, .lrc_fs_x100 = 2204464 },
	    { .fs = 32000, .N_bits_in_ts = 20, .sclk_divider =  375, .lrc_fs_x100 = 3200000 },
	    { .fs = 44100, .N_bits_in_ts = 20, .sclk_divider =  272, .lrc_fs_x100 = 4411765 },
	    { .fs = 48000, .N_bits_in_ts = 20, .sclk_divider =  250, .lrc_fs_x100 = 4800000 },
	},
    },
#else
    /* Measured DP-CLK and LRC/FSync values */
    {   .dp_clk_rate = 12000476,
	.coeff = {
	    { .fs =  8000, .N_bits_in_ts = 20, .sclk_divider = 1500, .lrc_fs_x100 =  800032 },
	    { .fs = 11025, .N_bits_in_ts = 19, .sclk_divider = 1146, .lrc_fs_x100 = 1102280 },
	    { .fs = 16000, .N_bits_in_ts = 20, .sclk_divider =  750, .lrc_fs_x100 = 1600060 },
	    { .fs = 22050, .N_bits_in_ts = 19, .sclk_divider =  573, .lrc_fs_x100 = 2204550 },
	    { .fs = 32000, .N_bits_in_ts = 20, .sclk_divider =  375, .lrc_fs_x100 = 3200130 },
	    { .fs = 44100, .N_bits_in_ts = 20, .sclk_divider =  272, .lrc_fs_x100 = 4411940 },
	    { .fs = 48000, .N_bits_in_ts = 20, .sclk_divider =  250, .lrc_fs_x100 = 4800190 },
	},
    },
#endif

    /* kSYS_CLK_246MHz */
    {	.dp_clk_rate = 12300000,
	.coeff = {
	    { .fs =  8000, .N_bits_in_ts = 16, .sclk_divider =  961, .lrc_fs_x100 =  799948 },
	    { .fs = 11025, .N_bits_in_ts = 20, .sclk_divider =  558, .lrc_fs_x100 = 1102151 },
	    { .fs = 16000, .N_bits_in_ts = 18, .sclk_divider =  427, .lrc_fs_x100 = 1600312 },
	    { .fs = 22050, .N_bits_in_ts = 20, .sclk_divider =  279, .lrc_fs_x100 = 2204301 },
	    { .fs = 32000, .N_bits_in_ts = 17, .sclk_divider =  226, .lrc_fs_x100 = 3201458 },
	    { .fs = 44100, .N_bits_in_ts = 18, .sclk_divider =  155, .lrc_fs_x100 = 4408602 },
	    { .fs = 48000, .N_bits_in_ts = 20, .sclk_divider =  128, .lrc_fs_x100 = 4804688 },
	},
    },

    /* Add for other system-clock freq */

    /* kSYS_CLK_DEFAULT */
    {	.dp_clk_rate = 12000000,
	.coeff = {
	    { .fs =  8000, .N_bits_in_ts = 20, .sclk_divider = 1500, .lrc_fs_x100 =  800000 },
	    { .fs = 11025, .N_bits_in_ts = 19, .sclk_divider = 1146, .lrc_fs_x100 = 1102232 },
	    { .fs = 16000, .N_bits_in_ts = 20, .sclk_divider =  750, .lrc_fs_x100 = 1600000 },
	    { .fs = 22050, .N_bits_in_ts = 19, .sclk_divider =  573, .lrc_fs_x100 = 2204464 },
	    { .fs = 32000, .N_bits_in_ts = 20, .sclk_divider =  375, .lrc_fs_x100 = 3200000 },
	    { .fs = 44100, .N_bits_in_ts = 20, .sclk_divider =  272, .lrc_fs_x100 = 4411765 },
	    { .fs = 48000, .N_bits_in_ts = 20, .sclk_divider =  250, .lrc_fs_x100 = 4800000 },
	},
    },
};

struct tdm_master_coeff * dmw_as_master(uint fs)
{
	enum eSYS_CLK sysclk_ix;
	enum eFS_HZ fs_ix;
	ulong pll1_rate;
	struct clk * pll1_clk = NULL;
	struct tdm_master_coeff * entry;

	dbg_prt();

	pll1_clk = clk_get_sys(DEVICE_NAME, "pll1");
	if (pll1_clk) {
		pll1_rate = (long)clk_get_rate(pll1_clk);
	} else {
		return NULL;
	}

	switch (pll1_rate) {
	case SYSCLK__480_MHz:
		sysclk_ix = kSYS_CLK_480MHz;
		break;
	case SYSCLK__246_MHz:
		sysclk_ix = kSYS_CLK_246MHz;
		break;
	default:
		sysclk_ix = kSYS_CLK_DEFAULT;
		break;
	}

	for (fs_ix = kFS_HZ_8000; fs_ix < kFS_HZ_MAX; fs_ix++){
		if (fs == tdm_clks_tab[sysclk_ix].coeff[fs_ix].fs) {
			break;
		}
	}
	if (fs_ix >= kFS_HZ_MAX) {
		/* Sample-rate not found */
		return NULL;
	}

	entry = (struct tdm_master_coeff *)&(tdm_clks_tab[sysclk_ix].coeff[fs_ix]);
	clk_put(pll1_clk);
	return entry;
}
EXPORT_SYMBOL(dmw_as_master);

/*****************************************************************************/

static int find_cpu_dai_id_by_name(const char *cpu_dai_name);
static int g_tel_rqst[NUM_DMW96_TDMs];

static long g_pll1_rate = 0;
static struct clk * g_dp_clk = NULL;

/*****************************************************************************/

/* TDM registers: */
#define TDM_ENABLE                   0x0000
#define TDM_CONFIG_1                 0x0004
#define TDM_CONFIG_1_AUTO_SYNC       (1 << 13)
#define TDM_CONFIG_1_CLK_MSTR_SLV    (1 << 12)
#define TDM_CONFIG_1_FSYNC_MSTR_SLV  (1 << 11)
#define TDM_CONFIG_1_SYNC_EDGE       (1 << 10)
#define TDM_CONFIG_1_RX_EDGE         (1 <<  9)
#define TDM_CONFIG_1_TX_EDGE         (1 <<  8)
#define TDM_CONFIG_1_TIME_SLOTS      0x1F
#define TDM_CONFIG_2                 0x0008
#define TDM_CONFIG_2_SHIFT           (1 << 9)
#define TDM_CONFIG_2_TRUNCATE        (1 << 8)
#define TDM_CONFIG_2_BANK_SIZE       0x07
#define TDM_FSYNC                    0x0010
#define TDM_FSYNC_DUR                0x0014
#define TDM_FIFO_WM                  0x0018
#define TDM_INT_EN                   0x0020
#define TDM_INT_EN_RX_FULL           (1 << 3)
#define TDM_INT_EN_TX_EMPTY          (1 << 1)
#define TDM_INT_EN_TX_WM             (1 << 0)
#define TDM_INT_EN_RX_WM             (1 << 2)
#define TDM_INT_CLR                  0x0028
#define TDM_INT_CLR_TX_FLAGS         (1 << 3)
#define TDM_INT_CLR_FSYNC            (1 << 2)
#define TDM_INT_CLR_RX_OVRN          (1 << 1)
#define TDM_INT_CLR_TX_UNDRN         (1 << 0)
#define TDM_INT_STATUS               0x002C
#define TDM_INT_STATUS_RX_OVRN       (1 << 3)
#define TDM_INT_STATUS_TX_UNDRN      (1 << 1)
#define TDM_STATUS                   0x0030
#define TDM_TX_FIFO                  0x0038
#define TDM_RX_FIFO                  0x003C
#define TDM_TX_REG_0                 0x0A00
#define TDM_RX_REG_0                 0x1200
#define TDM_FDA_REG_00               0x1800
#define TDM_FDA_REG_01               0x1804
#define TDM_FDA_REG_02               0x1808
#define TDM_FDA_REG_03               0x180C

#define TDM_REGLEN 0x2000

/* The TDM FIFOs registers are 20 bits wide.
 * In the dmw-96 TDM machine, the 20 MSbits are taken
 * when writing/reading the Tx/Rx fifo with a u32 value.
 * Note that this is opposite to the dw-52/74 TDM machine - 20 LSbits.
 */

/* The TDM Tx & Rx FIFOs have room for total of 128 samples - 
 * e.g. for 64 TDM frames, where a frame consists of 2 time-slots
 * (Left+Right channels, interleaved).
 */
#define TDM_FIFO_DEPTH	64

/* When using DMA to write/read the TDM FIFOs,
 * we set the fifo water-mark level to trigger a dma
 * transaction. This is done by hardware. In this case,
 * the WM interrupts are not used, thus disabled.
 *
 * NOTE:
 * ----
 * TDM_FIFO_LVL_FOR_DMA Must(!) be >= N, where N is defined in
 * arch/arm/mach-dmw/devices.c, by the value of "transfer_length" fields
 * of the DMA channels used by the 3 TDM machines, as:
 *
 * 	.transfer_length = (2 * N) - 1,	// N audio frames (stereo samples)
 *
 * "transfer_length" is the DMA Burst length, in u32 units.
 *
 * TDM_FIFO_LVL_FOR_DMA Should be equal to N.
 * It can be larger then N, but it CANNOT be Smaller !!!
 * If it is smaller, pcm data in the TDM TX Fifo might be over-written!
 */
#define	TDM_FIFO_LVL_FOR_DMA	(TDM_FIFO_DEPTH / 4)

#define TDM_TX_FIFO_LEVEL  TDM_FIFO_LVL_FOR_DMA  /* emptiness level, in frames */
#define TDM_RX_FIFO_LEVEL  TDM_FIFO_LVL_FOR_DMA  /* fulness   level, in frames */

#define TDM_SLOT(_index, _cd, _width) \
	(((_index) << 8) | (0x03 << 6) | (!!(_cd) << 5) | (_width & 0x1F))

/* CSS code works vs. the TDM FIFOs using IRQ.
 * It passes 4 audio frames (stereo-samples) per 
 * Water-Mark interrupt.
 */
#define TDM_FIFO_LVL_FOR_CSS (TDM_FIFO_DEPTH - 4)

/*****************************************************************************/

/*
 * The structure is per TDM and shared between the playback and capture
 * substreams. Fortunately there's no need for a dedicated mutex because the
 * access to the struct is implicitly guarded by the 'pcm_mutex' of the ASoC
 * core.
 */
struct dmw_tdm_priv {
	uint dai_id;
	void *reg_base;
	resource_size_t iores_mem;
	int iores_irq;
	struct clk *clk;
	uint tdm_clk_active;

	tdm_reset_fp_t reset;

	int opencnt         : 2;
	int fs_master       : 1;
	int clk_master      : 1;
	int play_configured : 1;
	int capt_configured : 1;
	int play_active     : 1;
	int capt_active     : 1;

	int css_to_own_tdm  : 1; /* Pending request to TDM-ownership switch-over:
			          * '0 - to Cortex; '1' - to CSS
			          */
	int css_owns_tdm    : 1; /* Current state of TDM ownership:
				  * '0' - by Cortex; '1' - by CSS;
				  */

	int tdm_sample_bits; 	/* num SCLKs per pcm sample, over TDM bus */
	int alsa_sample_bits;	/* num bits per alsa pcm sample */
	int rate;		/* sample-rate (Fs), in Hz */
	int alsa_channels;	/* num channels (TDM time-slots) */

	struct snd_pcm_substream *play_substream;
	struct snd_pcm_substream *capt_substream;
};

static unsigned long tdm_read(struct dmw_tdm_priv *tdm, int reg)
{
	return readl(tdm->reg_base + reg);
}

static void tdm_write(struct dmw_tdm_priv *tdm, int reg, unsigned long value)
{
	writel(value, tdm->reg_base + reg);
}

static void dmw_tdm_reset(struct dmw_tdm_priv *tdm)
{
	dbg_prt();
	
	tdm_write(tdm, TDM_ENABLE, 0);

	/* TDM clock must be Disabled when doing the TDM Reset,
	 * Must wait(!) for clock to Cease: the worst case is
	 * with the slowest clock.
	 */
	clk_disable(tdm->clk);
	udelay(20); /* 20 uSec, according to latest spec */

	/* Reset TDM block (this reset is Asynchronous !) */
	tdm->reset(1);
	udelay(2);
	tdm->reset(0);

	clk_enable(tdm->clk);
}

static int dmw_tdm_configure(struct dmw_tdm_priv *tdm)
{
	unsigned int slots;
	unsigned int alsa_bps, tdm_bps;
	unsigned int tdm_reg;
	unsigned int tx_fifo_wm_lvl, rx_fifo_wm_lvl;
	ulong master_sclk_rate;
	struct tdm_master_coeff * master_coeff;
	unsigned int i;

	dbg_prt();

#ifdef CONFIG_CSS_TDM
	if (tdm->css_owns_tdm) {
		/* Resume Cortex ownership over the TDM machine,
		 * Before(!) touching the TDM machine.
		 */
		if (coma_tdm_revoke(tdm->dai_id) == 0) {
			tdm->css_owns_tdm = 0;
			vdbg_prt("%s %u %s\n",
				"tdm", tdm->dai_id, "Revoked OK");
		} else {
			vdbg_prt("%s %u %s\n",
				 "tdm", tdm->dai_id, "Revoke Failed");
		}
	}
#endif /* CONFIG_CSS_TDM */

	/* Note:
	 * "bps" here means Bits-per-Sample.
	 *
	 * When the DMW96 TDM machine is the Master, i.e sourcing out the
	 * TDM SCLK & FSync clocks, there might be cases in which we would like
	 * to produce More SCLK clocks than the number of bits in the pcm sample.
	 * This is to achive better SCLK(BCLK) frequency accuracy.
	 * The SCLK frequency depends on Sample-Rate;
	 * its accuracy depends on the driving PLL frequency setting,
	 * the clock-tree further dividers resolution and the Sample-Rate Family
	 * (Fs = 8/12/11.025KHz multiplications).
	 * Example: to Tx a Fs=44.1KHz 16-bit pcm sample, where PLL freq=147.456MHz,
	 * we'll use 19 sckls (instaed of 16).
	 * 
	 * Since the FIFO regs width is 20 bits, we are limitted to Tx/Rx
	 * 20 bits (20 sclks) in a tdm time-slot.
	 * Another HW restriction: minimal number of bits in a time-slot is 8 (8 sclks).
	 * (8 <= nun sclks per time-slot <= 20).
	 * However, the (ALSA) PCM sample may contain 16 (commonly), 20, 24, etc. bits.
	 *
	 * When there is a need to Tx/Rx more than 20 bits per time-slot,
	 * we can use Two (!) time-slots per pcm sample:
	 *  - the first  time-slot is of Audio/Voice type, having upto 20 bits.
	 *  - the second time-slot is of Control/Status type, having the rest of the bits,
	 *    but not less then 8.
	 * 
	 * Currently, we Do Not support more than 20 bits per pcm sample.
	 */

	/* Note: the term "bps" means here "Bits per Sample" ! */
	
	alsa_bps = tdm->alsa_sample_bits;
	if (alsa_bps > 20) {
		vdbg_prt("%s\n", "PCM format-width > 20 bits is Not supported");
		return -EINVAL;
	}

	if ((tdm->clk_master) && (tdm->fs_master)) {
		/* dmw96 is Master */
		master_coeff = dmw_as_master(tdm->rate); /* arg is alsa sample-rate */
		if (master_coeff == NULL) {
			return -EINVAL;
		}
		tdm_bps = master_coeff->N_bits_in_ts;
		master_sclk_rate = g_pll1_rate / master_coeff->sclk_divider;
		/* Set Actual SCLK (BCLK) frequency over TDM bus,
		 * generated by the dmw96, as TDM Master !
		 */
		clk_set_rate(tdm->clk, master_sclk_rate);

	} else {
		/* dmw96 is Slave */
		tdm_bps = tdm->tdm_sample_bits;
	}

	if (tdm_bps < alsa_bps) {
		tdm_bps = alsa_bps;
	}

	/*
	 * We should sw-reset the DMW96 TDM block perior to any TDM configuration
	 * change according to the data sheet.
	 */
	dmw_tdm_reset(tdm);

	/* Configure Frame Descriptor Array. Might insert some NULL samples if
	 * required.
	 */
	tdm_write(tdm, TDM_FDA_REG_00, TDM_SLOT(0, 0, tdm_bps));
	tdm_write(tdm, TDM_FDA_REG_01, TDM_SLOT(1, 0, tdm_bps));
	slots = 2;


	/* Note:
	 * - Here we set the TDM bus configuration:
	 *	DMW96 TDM Slave/Master;
	 *	I2s format Only(!) (for now).
	 *
	 * Remember the DMW96 TDM machine limitation:
	 *
	 *   - as Master:
	 *     ----------
	 *     it can produce only Rising FSync pulse signal,
	 *     (at rising or falling SCLK edge - ok), but Cannot(!) produce
	 *     falling FSync signal. Thus, for true I2S format, we need to Shift
	 *     the Rising of the FSync (which is LRC signal in I2S) in such way that the
	 *     Falling edge would take place one bit before the start of Frame,
	 *     which is start of the Left channel. This means:
	 *
	 *     - for 2 time-slots (up to 20 bits per sample), with N bits per time-slot :
	 *       Trigger the FSync on N-1 bit of the First time slot.
	 *       e.g. on bit 15 of the first time-slot, for 16-bits time-slots.
	 *
	 *   - as Slave:
	 *     ---------
	 *     it syncs only(!) on Rising edge of FSync.
	 *     So we need to Invert the FSync polarity, generated at the Wolfson.
	 *     However, we Cannot do this, since the Wolfson will put the ADC data
	 *     on the second tdm-slot (Right channel) - we can't live with that.
	 *     So we have Left/Right channel swap!
	 *
	 *
	 *     Slave or Master:
	 *     ----------------
	 *     Tx & Rx edges can, and shoud be configured, for I2S, as follows:
	 *     - Tx data at Falling edge of SCLK
	 *     - Rx data at Rising edge of SCLK
	 */

	/* Configure TDM as Master/Slave SYNC/TX/RX Edges, 2 time-slots */
	tdm_reg = 0x0000;
	tdm_reg |= slots - 1;			/* Number of time-slots in frame (2 or 4) */
	tdm_reg |= TDM_CONFIG_1_TX_EDGE;	/* Tx data on SCLK Falling edge */
	tdm_reg &= ~TDM_CONFIG_1_RX_EDGE;	/* Rx data on SCLK Rising edge */

	/* FSync rise at SCLK Falling edge:
	 *   Matters only if Master; Irrelevant for Slave.
	 */
	tdm_reg |= TDM_CONFIG_1_SYNC_EDGE;
	
	if (tdm->fs_master) {
		/* DMW96 TDM is FSync Master */
		tdm_reg |= TDM_CONFIG_1_FSYNC_MSTR_SLV; /* Source-out FSync clock */
	} else {
		/* DMW96 TDM is FSync Slave */
		tdm_reg &= ~TDM_CONFIG_1_FSYNC_MSTR_SLV; /* Sink-in FSync clock */
		tdm_reg |= TDM_CONFIG_1_AUTO_SYNC;	 /* Resync on FSync, after FSync loss/jitter */
	}
	if (tdm->clk_master) {
		tdm_reg |= TDM_CONFIG_1_CLK_MSTR_SLV;	/* Source-out SCLK clock */
	} else {
		tdm_reg &= ~TDM_CONFIG_1_CLK_MSTR_SLV;	/* Sink-in SCLK clock */
	}
	tdm_write(tdm, TDM_CONFIG_1, tdm_reg);

	/* Configure MSB fisrt, 64(32) banks (in a fifo line), 2(4) fifo lines */
	tdm_reg = 0x0000;
	tdm_reg |= TDM_CONFIG_2_SHIFT;	  /* Shift Out/In MSB first */
	tdm_reg |= TDM_CONFIG_2_TRUNCATE; /* On Tx/Rx - Allign data bits if less then 20 */
	if (slots == 2) {
		/* Configure 2 Fifo lines - for 2 time-slots in a frame.
		 * Each line has 64 regiters (banks).
		 */
		tdm_reg |= (0 & 0x0007);
	}
	tdm_write(tdm, TDM_CONFIG_2, tdm_reg);

	/* Configure LRC Rise on last bit of Left channel -
	 * to produce LRC Fall at 1 bit before data frame starts (standard I2S)
	 */
	tdm_reg = 0x0000;
	if (slots == 2) {
		tdm_reg |= ((tdm_bps - 1) << 8);	/* Last bit of */
		tdm_reg |= (0 & 0x001F);		/*   First time slot */
	}
	tdm_write(tdm, TDM_FSYNC, tdm_reg);
	
	/* Configure LRC length, in sclks */
	tdm_write(tdm, TDM_FSYNC_DUR, tdm_bps - 1);

	/* Configure TX & RX FIFOs Water-Mark */
	tdm_reg = 0x0000;
	if (!tdm->css_to_own_tdm) {
		tx_fifo_wm_lvl = TDM_TX_FIFO_LEVEL;
		rx_fifo_wm_lvl = TDM_RX_FIFO_LEVEL;
	}
	else {
		tx_fifo_wm_lvl = TDM_FIFO_LVL_FOR_CSS;
		rx_fifo_wm_lvl = TDM_FIFO_LVL_FOR_CSS;
	}

	/* Future: for more then 20 bits in per TDM channels,
	 * using two time-slots per L/R channel
	 */
	if (slots == 4) {
		tx_fifo_wm_lvl >>= 1; /* :2 */
		rx_fifo_wm_lvl >>= 1; /* :2 */
	}

	/* Latest documented changes in the dmw96 TDM machine:
	 * - RX_LVL field is bits 13-8.
	 * - TX_LVL and RX_LVL actually trigger DMA or WM-Interrupt
	 *   at field value + 1.
	 */
	tdm_reg |= (tx_fifo_wm_lvl & 0x003F) << 0;
	tdm_reg |= (rx_fifo_wm_lvl & 0x003F) << 8;    /* bits 13:8 */
	tdm_write(tdm, TDM_FIFO_WM, tdm_reg);

	/* "Clear" TDM FIFOs (!):
	 * Note: this is needed in order to Deassert the TDM Water-Mark
	 * interrupt indications, at the TDM machine, in case they are
	 * asserted.
	 * Otherwise, an interrupt will be asserted immediately after
	 * unmasking these interrupts, Even Though the machine is
	 * Disabled (TDM_ENABLE has 0)!!!
	 * This is because the TDM WM interrupts are of "Level" type
	 * (Not "Edge-trigger").
	 */
	/* Fill-in Tx fifo */
	i = tx_fifo_wm_lvl + 1;
	i *= slots / 2;
	while (i--) {
		tdm_write(tdm, TDM_TX_FIFO, 0x0000); /* Left */
		tdm_write(tdm, TDM_TX_FIFO, 0x0000); /* Right */
	}
	/* Empty Rx fifo */
	i = rx_fifo_wm_lvl + 1;
	i *= slots / 2;
	while (i--) {
		tdm_read(tdm, TDM_RX_FIFO); /* Left */
		tdm_read(tdm, TDM_RX_FIFO); /* Right */
	}
	
	/* TODO - add handing for Grant Timeout (CSS has curshed ) */

#ifdef CONFIG_CSS_TDM
	if (tdm->css_to_own_tdm) {
		if (!tdm->css_owns_tdm) {
			/* TDM ownership should be given to CSS.
			 *
			 * The TDM machine will now be handled by the CSS'es TDM driver.
			 * The Cortex now finishes the TDM machine configuration to meet
			 * the CSS needs and Ceases to handle it !
			 */

			/* Disable TDM IRQ (at the Cortex ICU !) */
			disable_irq(tdm->iores_irq);
			
			/* Enable Tx/Rx Water-Mark and Underrun/Overrun interrupts
			 * (to be handled by the CSS !)
			 */
			tdm_write(tdm, TDM_INT_EN, 
				TDM_INT_EN_TX_EMPTY | TDM_INT_EN_RX_FULL | /* For normal operation */
				TDM_INT_EN_TX_WM | TDM_INT_EN_RX_WM);	   /* To catch errors */
			
			/* Handover TDM ownership to CSS: */
			if (coma_tdm_grant(tdm->dai_id) == 0) {
				tdm->css_owns_tdm = 1;
				vdbg_prt("%s %u %s\n",
					 "tdm", tdm->dai_id, "Granted to CSS");
			} else {
				vdbg_prt("%s %u %s\n",
					 "tdm", tdm->dai_id, "Grant Failed");
			}
		}
		
	} else {
		/* TDM ownership should stay owned by the Cortex */
		tdm->css_owns_tdm = 0;
		vdbg_prt("%s %u\n", "Cortex owns tdm", tdm->dai_id);
	}
#endif /* CONFIG_CSS_TDM */

	return 0;

} /* end dmw_tdm_configure() */

/*****************************************************************************/

static void tdm_apply_constraints(struct dmw_tdm_priv *tdm,
				struct snd_pcm_substream *substream)
{
	dbg_prt("%s\n", DIR_STR(substream->stream));

	/* Set the constraints according to the already configured stream */
	snd_pcm_hw_constraint_minmax(substream->runtime,
				SNDRV_PCM_HW_PARAM_RATE,
				tdm->rate,
				tdm->rate);

	snd_pcm_hw_constraint_minmax(substream->runtime,
				SNDRV_PCM_HW_PARAM_SAMPLE_BITS,
				tdm->alsa_sample_bits,
				tdm->alsa_sample_bits);
}

static int tdm_check_constraints(struct dmw_tdm_priv *tdm,
					struct snd_pcm_hw_params *params)
{
	dbg_prt();

	/* Check the constraints according to the already configured stream */
	if ((params_rate(params) != tdm->rate) ||
	    (snd_pcm_format_width(params_format(params)) != tdm->alsa_sample_bits)) {
		vdbg_prt("%s\n", "Constraints matching Failed!");
		return -EINVAL;
	}

	return 0;
}

static int dmw_tdm_startup(struct snd_pcm_substream *substream,
				struct snd_soc_dai *cpu_dai)
{
	struct dmw_tdm_priv *tdm = snd_soc_dai_get_tdm_privdata(cpu_dai);

	dbg_prt("%s %s\n", cpu_dai->name, DIR_STR(substream->stream));

	if (tdm->opencnt++ == 0) {
		clk_enable(tdm->clk);
		tdm->tdm_clk_active = 1;
		vdbg_prt("%s %u %s\n",
			"TDM clock of tdm id", cpu_dai->id, "Enabled");
	}

	switch (substream->stream) {
	case SNDRV_PCM_STREAM_PLAYBACK:
		vdbg_prt("%s\n", "Playback stream startup");
		tdm->play_substream = substream;
		if (tdm->capt_configured) {
			tdm_apply_constraints(tdm, substream);
			vdbg_prt("%s\n", "While Capture already configured");
		}
		break;

	case SNDRV_PCM_STREAM_CAPTURE:
		vdbg_prt("%s\n", "Capture stream startup");
		tdm->capt_substream = substream;
		if (tdm->play_configured) {
			tdm_apply_constraints(tdm, substream);
			vdbg_prt("%s\n", "While Playback already configured");
		}
		break;
	}

	return 0;
}

static void dmw_tdm_shutdown(struct snd_pcm_substream *substream,
				struct snd_soc_dai *cpu_dai)
{
	struct dmw_tdm_priv *tdm = snd_soc_dai_get_tdm_privdata(cpu_dai);

	dbg_prt("%s %s\n", cpu_dai->name, DIR_STR(substream->stream));

	switch (substream->stream) {
	case SNDRV_PCM_STREAM_PLAYBACK:
		vdbg_prt("%s\n", "Playback shutdown");
		tdm->play_substream = NULL;
		break;

	case SNDRV_PCM_STREAM_CAPTURE:
		vdbg_prt("%s\n", "Capture shutdown");
		tdm->capt_substream = NULL;
		break;
	}

	if (--tdm->opencnt == 0) {
		clk_disable(tdm->clk);
		tdm->tdm_clk_active = 0;
		vdbg_prt("%s %u %s\n",
			"TDM clock of tdm id", cpu_dai->id, "Disabled");
	}
}

static int dmw_tdm_prepare(struct snd_pcm_substream *substream,
				struct snd_soc_dai *cpu_dai)
{
	struct dmw_tdm_priv *tdm = snd_soc_dai_get_tdm_privdata(cpu_dai);
	int cpu_dai_id;

	dbg_prt("%s\n", DIR_STR(substream->stream));

	cpu_dai_id = find_cpu_dai_id_by_name(cpu_dai->name);
	if (cpu_dai_id < 0) {
		vdbg_prt("%s %s\n",
		"Can't find cpu-dai id for", cpu_dai->name);
		return -EINVAL;
	}
	tdm->css_to_own_tdm = g_tel_rqst[cpu_dai_id];

	/*
	 * Configure TDM machine (including its Reset) just before the first
	 * direction is started.
	 * This is especially needed after an underrun/overrun has occured
	 * And
	 * when Dect Telephony is involved (ownership of the TDM machine by the CSS),
	 * since the state of the relevant asoc control, in asound.conf, is actually
	 * set After dmw_tdm_hw_params() is called and Before dmw_tdm_prepare() gets
	 * called.
	 */
	if ( (!tdm->play_active) &&
	     (!tdm->capt_active) &&
	     (!tdm->css_owns_tdm) ) {
		return dmw_tdm_configure(tdm);
	} else {
		vdbg_prt("%s\n\t\t%s %u %s %u %s %u\n",
			"dmw_tdm_configure()Skipped !!!",
			"play_active:",  (uint)(tdm->play_active),
			"capt_active:",  (uint)(tdm->capt_active),
			"css_owns_tdm:", (uint)(tdm->css_owns_tdm));
	}

	return 0;
}	

static int dmw_tdm_trigger(struct snd_pcm_substream *substream, int cmd,
		struct snd_soc_dai *cpu_dai)
{
	struct dmw_tdm_priv *tdm = snd_soc_dai_get_tdm_privdata(cpu_dai);
	
	dbg_prt("%s\n", DIR_STR(substream->stream));

	/* Note:
	 * Under DECT phone-call, where the TDM ownership has been granted to CSS,
	 * No triggers are expected. Will be handled anyway, just in case they come.
	 */

	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
	case SNDRV_PCM_TRIGGER_RESUME:
	case SNDRV_PCM_TRIGGER_PAUSE_RELEASE:
		if (tdm->css_owns_tdm) {
			vdbg_prt("%s\n", "START trigger: TDM Not owned by Cortex. Ignoring!");
			return -EPERM;
		}

		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
			vdbg_prt("%s\n", "Playback Start/Resume/unPuase");
			tdm->play_active = 1;
			if (tdm->capt_active) {
				break;
			}
		} else {
			vdbg_prt("%s\n", "Capture Start/Resume/unPuase");
			tdm->capt_active = 1;
			if (tdm->play_active) {
				break;
			}
		}
		
		/* Clear interrupt status (before enabling interrupts) */
		tdm_write(tdm, TDM_INT_CLR, 0xFFFF);

		/* Enable Tx & Rx Underrun/Overrun error interrupts */
		tdm_write(tdm, TDM_INT_EN,
			TDM_INT_EN_TX_EMPTY | TDM_INT_EN_RX_FULL);
		
		/* Enable TDM machine */
		tdm_write(tdm, TDM_ENABLE, 1);
		break;

	case SNDRV_PCM_TRIGGER_STOP:
	case SNDRV_PCM_TRIGGER_SUSPEND:
	case SNDRV_PCM_TRIGGER_PAUSE_PUSH:
		if (substream->stream == SNDRV_PCM_STREAM_PLAYBACK) {
			vdbg_prt("%s\n", "Playback Stop/Suspend/Puase");
			tdm->play_active = 0;
			if (tdm->capt_active) {
				break;
			}
		} else {
			vdbg_prt("%s\n", "Capture Stop/Suspend/Puase");
			tdm->capt_active = 0;
			if (tdm->play_active) {
				break;
			}
		}

		if (!tdm->css_owns_tdm) {
			/* Disable all TDM interrupts */
			tdm_write(tdm, TDM_INT_EN, 0);

			/* Disable TDM machine */
			tdm_write(tdm, TDM_ENABLE, 0);
		} else {
			vdbg_prt("%s\n", "STOP trigger: TDM Not owned by Cortex. Not disabling TDM!");
		}
		break;

	default:
		return -EINVAL;
	}

	return 0;
}

/* Note: this fuction may be called more then once for the same session ! */
static int dmw_tdm_hw_params(struct snd_pcm_substream *substream,
				struct snd_pcm_hw_params *params,
				struct snd_soc_dai *cpu_dai)
{
	struct dmw_tdm_priv *tdm = snd_soc_dai_get_tdm_privdata(cpu_dai);
	
	int ret;

	dbg_prt("%s\n", DIR_STR(substream->stream));

	/*
	 * The other direction might already be configured. All we can do in
	 * this case is to verify that the settings match as the TDM allows no
	 * dynamic reconfiguration.
	 */
	switch (substream->stream) {
	case SNDRV_PCM_STREAM_PLAYBACK:
		if (tdm->capt_configured) {
			ret = tdm_check_constraints(tdm, params);
			if (!ret) {
				tdm->play_configured = 1;
				vdbg_prt("%s\n", "Playback stream implicitly configured");
			} else {
				vdbg_prt("%s\n",
					 "Playback stream doesn't match Capture stream. Denied!");
			}
			return ret;
		}
		break;

	case SNDRV_PCM_STREAM_CAPTURE:
		if (tdm->play_configured) {
			ret = tdm_check_constraints(tdm, params);
			if (!ret) {
				tdm->capt_configured = 1;
				vdbg_prt("%s\n", "Capture stream implicitly configured");
			} else {
				vdbg_prt("%s\n",
					 "Capture stream doesn't match Playback stream. Denied!");
			}
			return ret;
		}
		break;
	}

	/* Save PCM parameters and Configure accordingly: */
	tdm->alsa_sample_bits = snd_pcm_format_width(params_format(params));
	tdm->rate = params_rate(params);
	tdm->alsa_channels = params_channels(params);

	vdbg_prt("%s\n\t\t%s %u  %s %u  %s %u\n",
		 "ALSA PCM params:",
		 "bits-per-sample:", tdm->alsa_sample_bits,
		 "sample-rate:", tdm->rate,
		 "num channels:", tdm->alsa_channels);
	
	/* If both playback and capture streams are open, and one of them
	 * is setting the hw parameters right now (since we are here), set
	 * constraints to the other stream to match the current one.
	 */
	switch (substream->stream) {
	case SNDRV_PCM_STREAM_PLAYBACK:
		if (tdm->capt_substream) {
			vdbg_prt("%s\n", "Apply constraints to Capture stream");
			tdm_apply_constraints(tdm, tdm->capt_substream);
		}
		tdm->play_configured = 1; 
		vdbg_prt("%s\n", "Playback stream explicitly configured");
		break;

	case SNDRV_PCM_STREAM_CAPTURE:
		if (tdm->play_substream) {
			vdbg_prt("%s\n", "Apply constraints to Playback stream");
			tdm_apply_constraints(tdm, tdm->play_substream);
		}
		tdm->capt_configured = 1;
		vdbg_prt("%s\n", "Capture stream explicitly configured");
		break;
	}

	return 0;
}

/* Note: this fuction may be called more then once for the same session ! */
static int dmw_tdm_hw_free(struct snd_pcm_substream *substream,
				struct snd_soc_dai *cpu_dai)
{
	struct dmw_tdm_priv *tdm = snd_soc_dai_get_tdm_privdata(cpu_dai);

	dbg_prt("%s\n", DIR_STR(substream->stream));

	switch (substream->stream) {
	case SNDRV_PCM_STREAM_PLAYBACK:
		tdm->play_configured = 0;
		vdbg_prt("%s\n", "Playback stream unconfigured");
		break;

	case SNDRV_PCM_STREAM_CAPTURE:
		tdm->capt_configured = 0;
		vdbg_prt("%s\n", "Capture stream unconfigured");
		break;
	}

#ifdef CONFIG_CSS_TDM
	if ((!tdm->play_configured) && (!tdm->capt_configured)) {
		if (tdm->css_owns_tdm) {

			/* TODO - add handing for Revoke Timeout (CSS has curshed ) */

			/* Resume Cortex ownership over the TDM machine */
			if (coma_tdm_revoke(tdm->dai_id) == 0) {
				tdm->css_owns_tdm = 0;
				vdbg_prt("%s %u %s\n",
					 "tdm", tdm->dai_id, "Revoked OK");
			} else {
				vdbg_prt("%s %u %s\n",
					 "tdm", tdm->dai_id, "Revoke Failed");
			}

			/* Entirely Reset the TDM machine:
			 * the TDM machine gets Disabled, its interrupts cleared,
			 * and all of its registers get their default values.
			 */
			dmw_tdm_reset(tdm);

			/* Enable TDM interrupt (at the Cortex ICU !) */
			enable_irq(tdm->iores_irq);
		}
	}
#endif /* CONFIG_CSS_TDM */

	return 0;
}

static int dmw_tdm_set_dai_fmt(struct snd_soc_dai *cpu_dai, unsigned int fmt)
{
	struct dmw_tdm_priv *tdm = snd_soc_dai_get_tdm_privdata(cpu_dai);
	
	dbg_prt();

	switch (fmt & SND_SOC_DAIFMT_FORMAT_MASK) {
	case SND_SOC_DAIFMT_I2S:
		break;
	default:
		return -EINVAL;
	}

	switch (fmt & SND_SOC_DAIFMT_MASTER_MASK) {
		
	/* Note: M/S setting is from CODEC point-of-view.
	 * 	 Here we set the Opposite !
	 */
	
	case SND_SOC_DAIFMT_CBM_CFM:
		tdm->clk_master = 0;
		tdm->fs_master = 0;
		vdbg_prt("%s\n", "DMW96 is TDM Slave");
		break;
	case SND_SOC_DAIFMT_CBS_CFM:
		tdm->clk_master = 1;
		tdm->fs_master = 0;
		break;
	case SND_SOC_DAIFMT_CBM_CFS:
		tdm->clk_master = 0;
		tdm->fs_master = 1;
		break;
	case SND_SOC_DAIFMT_CBS_CFS:
		tdm->clk_master = 1;
		tdm->fs_master = 1;
		vdbg_prt("%s\n", "DMW96 is TDM Master");
		break;
	default:
		return -EINVAL;
	}

	return 0;
}

static irqreturn_t dmw_tdm_irq(int irq, void *dev_id)
{
	struct dmw_tdm_priv *tdm = dev_id;
	irqreturn_t ret = IRQ_NONE;
	uint pending;
	unsigned long flags;

	pending = tdm_read(tdm, TDM_INT_STATUS) & tdm_read(tdm, TDM_INT_EN);

	if (pending) {
		if (pending & (TDM_INT_STATUS_RX_OVRN | TDM_INT_STATUS_TX_UNDRN)) {
			if (pending & TDM_INT_STATUS_TX_UNDRN) {
				printk(KERN_WARNING "TDM %u Underrun !!!\n",
					tdm->dai_id);

			}
			if (pending & TDM_INT_STATUS_RX_OVRN) {
				printk(KERN_WARNING "TDM %u Overrun !!!\n",
					tdm->dai_id);
			}

			/*
			 * DMA was too slow. We have to cancel both directions because
			 * the TDM has to be reset to recover.
			 */
			if (tdm->play_active) {
				vdbg_prt("%s\n", "TDM IRQ: Stopping Playback pcm stream !");
				snd_pcm_stream_lock_irqsave(tdm->play_substream, flags);
				snd_pcm_stop(tdm->play_substream, SNDRV_PCM_STATE_XRUN);
				snd_pcm_stream_unlock_irqrestore(tdm->play_substream, flags);
			}
			if (tdm->capt_active) {
				vdbg_prt("%s\n", "TDM IRQ: Stopping Capture pcm stream !");
				snd_pcm_stream_lock_irqsave(tdm->capt_substream, flags);
				snd_pcm_stop(tdm->capt_substream, SNDRV_PCM_STATE_XRUN);
				snd_pcm_stream_unlock_irqrestore(tdm->capt_substream, flags);
			}

			/* Clear X-RUN interrupts (in TDM machine) */
			tdm_write(tdm, TDM_INT_CLR, TDM_INT_CLR_RX_OVRN | TDM_INT_CLR_TX_UNDRN);

		} else {

			/* Other interrupts should not happen */
			// ... Unless machine is TDM Slave and FSYNC_INT enabled 
			// and FSync-jitter happens.

			/* Clear All TDM interrupts (in TDM machine) */
			tdm_write(tdm, TDM_INT_CLR, 0xFFFFF);

			printk(KERN_WARNING "Unexpected TDM interrupt !!! pending = 0x%X\n", pending);
		}

		ret = IRQ_HANDLED;
	}
	
	return ret;
}

static int dmw_tdm_dai_probe(struct snd_soc_dai *cpu_dai)
{
	struct dmw_tdm_priv *tdm;
	struct platform_device *pdev = to_platform_device(cpu_dai->dev);
	struct dmw_tdm_pdata *pdata = cpu_dai->dev->platform_data;
	struct resource *iores;
	int ret = 0;
	long actual_dp_clk = 0, desired_dp_clk = 0;

	dbg_prt();

	tdm = kzalloc(sizeof(*tdm), GFP_KERNEL);
	if (!tdm) {
		return -ENOMEM;
	}
	vdbg_prt("%s %u %s\n",
		 "cpu-dai(tdm) id", cpu_dai->id, "kzalloc OK");
	
	/* Note: pdata = cpu_dai->dev->platform_data (above) points to
	** 'struct platform_device tdmN_device'.platform_data,
	** which points to tdmN_pdata, of type 'struct dmw_tdm_pdata',
	** (which is defined in dmw-tdm.h).
	** There, the .reset field points to the 'reset_tdmN', which is	the
	** physical Reset-Function of TDM machine number N.
	** See arch/arm/mach-dmw/borad-XXX.c
	*/
	tdm->reset = pdata->reset; 

	tdm->clk = clk_get(cpu_dai->dev, NULL);
	if (IS_ERR(tdm->clk)) {
		ret = PTR_ERR(tdm->clk);
		dev_err(cpu_dai->dev, "No clock!\n");
		goto err_kfree;
	}
	vdbg_prt("%s %u %s %li\n",
		 "cpu-dai(tdm) id", cpu_dai->id, "clk_get OK", (long)clk_get_rate(tdm->clk));
#if 0
	desired_dp_clk = 4096000;

//clk_set_parent(tdm->clk,  clk_get_sys("pll3", NULL));

        actual_dp_clk = clk_round_rate(tdm->clk, desired_dp_clk);
        if (actual_dp_clk < 0) {
                vdbg_prt("%s\n", "Can't set desired rate of TDM clk");
		goto err_kfree;
        }
        clk_set_rate(tdm->clk, (unsigned long)actual_dp_clk);
	vdbg_prt("%s %u %s %li\n",
		 "cpu-dai(tdm) id", cpu_dai->id, "clk_set to ", (long)clk_get_rate(tdm->clk));


	desired_dp_clk = 12000000;
        actual_dp_clk = clk_round_rate(clk_get_sys("dmw-tdm", "dp"), desired_dp_clk);
        if (actual_dp_clk < 0) {
                printk("clk_round_rate(dp, %d): %ld\n", desired_dp_clk, actual_dp_clk);
                return;
        }
        if (clk_set_rate(clk_get_sys("dmw-tdm", "dp"), actual_dp_clk)) {
                printk("clk_set_rate(dp) failed!\n");
                return;
        }
        if (clk_enable(clk_get_sys("dmw-tdm", "dp"))) {
                printk("clk_enable(dp) failed!\n");
                return;
        }
        printk("TDM test: DP = %ldHz\n", actual_dp_clk);


	clk_enable(tdm->clk);
#endif
	tdm->iores_irq = platform_get_irq(pdev, 0);
	if (tdm->iores_irq < 0) {
		ret = tdm->iores_irq;
		dev_err(cpu_dai->dev, "No IRQ!\n");
		goto err_put;
	}
	vdbg_prt("%s %u %s\n",
		 "cpu-dai(tdm) id", cpu_dai->id, "platform_get_irq OK");

	iores = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!iores) {
		ret = -EINVAL;
		dev_err(cpu_dai->dev, "Mem iores missing!\n");
		goto err_put;
	}
	vdbg_prt("%s %u %s\n",
		 "cpu-dai(tdm) id", cpu_dai->id, "platform_get_resource OK");
	
	tdm->iores_mem = iores->start;

	iores = platform_get_resource(pdev, IORESOURCE_DMA, 0);
	if (!iores) {
		ret = -EINVAL;
		dev_err(cpu_dai->dev, "Capture DMA iores missing!\n");
		goto err_put;
	}
	vdbg_prt("%s %u %s\n",
		 "cpu-dai(tdm) id", cpu_dai->id, "platform_get_resource IORESOURCE_DMA 0 OK");
	
	cpu_dai->capture_dma_data = (void *)iores->start;

	iores = platform_get_resource(pdev, IORESOURCE_DMA, 1);
	if (!iores) {
		ret = -EINVAL;
		dev_err(cpu_dai->dev, "Playback DMA iores missing!\n");
		goto err_put;
	}
	vdbg_prt("%s %u %s\n",
		 "cpu-dai(tdm) id", cpu_dai->id, "platform_get_resource IORESOURCE_DMA 1 OK");

	cpu_dai->playback_dma_data = (void *)iores->start;

	if (!request_mem_region(tdm->iores_mem, TDM_REGLEN,
	                        pdev->dev.driver->name)) {
		ret = -EBUSY;
		dev_err(cpu_dai->dev, "Cannot reserve mem region!\n");
		goto err_put;
	}
	vdbg_prt("%s %u %s\n",
		 "cpu-dai(tdm) id", cpu_dai->id, "request_mem_region OK");

	tdm->reg_base = ioremap_nocache(tdm->iores_mem, TDM_REGLEN);
	if (!tdm->reg_base) {
		ret = -ENOMEM;
		dev_err(cpu_dai->dev, "Cannot remap registers!\n");
		goto err_release;
	}
	vdbg_prt("%s %u %s\n",
		 "cpu-dai(tdm) id", cpu_dai->id, "ioremap_nocache OK");

	ret = request_irq(tdm->iores_irq, dmw_tdm_irq, 0, "dmw_tdm_irq", tdm);
	if (ret) {
		dev_err(cpu_dai->dev, "Cannot request IRQ!\n");
		goto err_unmap;
	}
	vdbg_prt("%s %u %s\n",
		 "cpu-dai(tdm) id", cpu_dai->id, "request_irq OK");

	tdm->css_to_own_tdm = 0;
	tdm->css_owns_tdm = 0;
	
	tdm->alsa_sample_bits = 16; 	/* Defualt PCM sample width, in bits */
	tdm->tdm_sample_bits = 16;	/* Default TDM time-slot width, in SCLK cycles:
					 * (1 cycle per bit)
					 */
	tdm->dai_id = cpu_dai->id;

	snd_soc_dai_set_tdm_privdata(cpu_dai, tdm);

	vdbg_prt("%s %u %s\n",
		 "cpu-dai(tdm) id", cpu_dai->id, "probed ok");

	return 0;

err_unmap:
	vdbg_prt("%s %u %s\n",
		 "cpu-dai(tdm) id", cpu_dai->id, "err_unmap !!!");

	iounmap(tdm->reg_base);

err_release:
	vdbg_prt("%s %u %s\n",
		 "cpu-dai(tdm) id", cpu_dai->id, "err_release !!!");

	release_mem_region(tdm->iores_mem, TDM_REGLEN);
err_put:
	vdbg_prt("%s %u %s\n",
		 "cpu-dai(tdm) id", cpu_dai->id, "err_put !!!");

	clk_put(tdm->clk);
err_kfree:
	vdbg_prt("%s %u %s\n",
		 "cpu-dai(tdm) id", cpu_dai->id, "err_kfree !!!");

	kfree(tdm);

	vdbg_prt("%s %u %s\n",
		 "cpu-dai(tdm) id", cpu_dai->id, "probe Failed !!!");

	return ret;
}

static int dmw_tdm_dai_remove(struct snd_soc_dai *cpu_dai)
{
	struct dmw_tdm_priv *tdm = snd_soc_dai_get_tdm_privdata(cpu_dai);
	
	dbg_prt();

	free_irq(tdm->iores_irq, tdm);
	iounmap(tdm->reg_base);
	release_mem_region(tdm->iores_mem, TDM_REGLEN);
	clk_put(tdm->clk);
	kfree(tdm);

	return 0;
}

static snd_pcm_sframes_t dmw_tdm_delay_report(
		struct snd_pcm_substream *substream, struct snd_soc_dai *cpu_dai)
{
	/* Note:
	 * The DMA buffers, in dmw-pcm.c, are defined to hold
	 * one single max period (number of audio-frames), and dynamically hold
	 * excactly one Period; thus the delay (in number of frames) is 1 period
	 * to start with.
	 * In addition to that, there are audio-frames waiting to be TX in the
	 * TX-FIFO (after being DMA'ed) or accumulated in the RX_FIFO (before
	 * being DMA'ed).
	 */

	switch (substream->stream) {

	case SNDRV_PCM_STREAM_PLAYBACK:
		return substream->runtime->period_size +
			(TDM_FIFO_DEPTH - TDM_TX_FIFO_LEVEL);	/* 48 */

	case SNDRV_PCM_STREAM_CAPTURE:
		return substream->runtime->period_size +
			TDM_RX_FIFO_LEVEL;			/* 16 */
	}

	return 0;
}

static struct snd_soc_dai_ops dmw_tdm_dai_ops = {
	.set_fmt	= dmw_tdm_set_dai_fmt,
	.startup	= dmw_tdm_startup,
	.shutdown	= dmw_tdm_shutdown,
	.hw_params	= dmw_tdm_hw_params,
	.hw_free	= dmw_tdm_hw_free,
	.prepare	= dmw_tdm_prepare,
	.trigger	= dmw_tdm_trigger,
	.delay		= dmw_tdm_delay_report,

	/// A.M. Is symmetric_rates fixed in
	/// kernel 2.6.39 or still 'broken' ?	
	//.symmetric_rates = 1,
};

#define DMW_TDM_DAI(_id)						  \
{									  \
	.name = DEVICE_NAME"." __stringify(_id),			  \
	.id = (_id),  							  \
	.probe = dmw_tdm_dai_probe,					  \
	.remove = dmw_tdm_dai_remove,					  \
	.ops = &dmw_tdm_dai_ops,					  \
	.playback = {							  \
		.channels_min = 1,					  \
		.channels_max = 2,					  \
		.rates = SNDRV_PCM_RATE_8000_48000,	  		  \
		.formats = SNDRV_PCM_FMTBIT_S8 | SNDRV_PCM_FMTBIT_S16_LE, \
	},								  \
	.capture = {							  \
		.channels_min = 1,					  \
		.channels_max = 2,					  \
		.rates = SNDRV_PCM_RATE_8000_48000,	 		  \
		.formats = SNDRV_PCM_FMTBIT_S8 | SNDRV_PCM_FMTBIT_S16_LE, \
	},								  \
}

static struct snd_soc_dai_driver dmw_tdm_dai[NUM_DMW96_TDMs] = {
	DMW_TDM_DAI(0),
	DMW_TDM_DAI(1),
	DMW_TDM_DAI(2),
};

/*****************************************************************************/

static int find_cpu_dai_id_by_name(const char *cpu_dai_name)
{
	int i;
	struct snd_soc_dai_driver *p;
	
	dbg_prt();

	for (i = 0, p = &dmw_tdm_dai[0];
	     i < ARRAY_SIZE(dmw_tdm_dai);
	     i++, p++) {

		//vdbg_prt("[%i] %s %s %s\n",
		//	 i, "matching", cpu_dai_name, p->name);
		
		if (strcmp(cpu_dai_name, p->name) == 0) {
			return p->id;	
		}
	}

	/* Not found */
	return -1;
}

/* This function is currently Unused. Do not delete it! */
int dmw_tdm_set_bits_per_sample(struct snd_soc_dai *cpu_dai, unsigned int bps)
{
	struct dmw_tdm_priv *tdm = snd_soc_dai_get_tdm_privdata(cpu_dai);
	
	dbg_prt();

	if (tdm->capt_active || tdm->play_active) {
		return -EBUSY; /* too late */
	}

	tdm->tdm_sample_bits = bps;
	return dmw_tdm_configure(tdm);
}
EXPORT_SYMBOL(dmw_tdm_set_bits_per_sample);

void dmw_tdm_set_telephony(const char *cpu_dai_name, int tel_rqst)
{
	int cpu_dai_id;

	dbg_prt();

	cpu_dai_id = find_cpu_dai_id_by_name(cpu_dai_name);
	if (cpu_dai_id < 0) {
		vdbg_prt("%s %s %s\n",
			"Can't find matched id for cpu-dai",
			cpu_dai_name, "Aborting!");
		return;
	}

	vdbg_prt("%s %s %s.  %s %s\n",
		 "Pending TEL Request for cpu-dai",
		 cpu_dai_name,
		 ((tel_rqst) ? ": Tel to ON" : ": Tel to OFF"),
		 "Curr TDM Owner:", ((g_tel_rqst[cpu_dai_id]) ? "CSS" : "Cortex"));

	if (tel_rqst) {
		if (!g_tel_rqst[cpu_dai_id]) {
			vdbg_prt("%s\n", "Pending Cortex --> CSS switch");
	   	}

	} else {
		if (g_tel_rqst[cpu_dai_id]) {
			vdbg_prt("%s\n", "Pending CSS --> Cortex switch");
		}
	}

	g_tel_rqst[cpu_dai_id] = tel_rqst;
}
EXPORT_SYMBOL(dmw_tdm_set_telephony);

/*****************************************************************************/

static int __devinit dmw_tdm_dev_probe(struct platform_device *pdev)
{
	int err;
	struct snd_platform_data *pdata = pdev->dev.platform_data;

	dbg_prt();

	if (pdev->id < 0 || pdev->id >= ARRAY_SIZE(dmw_tdm_dai)) {
		dev_err(&pdev->dev, "Invalid DMW TDM device instance (%d)!\n",
			pdev->id);
		return -ENXIO;
	}

	if (!pdata) {
		dev_err(&pdev->dev, "Missing platform data!\n");
		return -EINVAL;
	}

	vdbg_prt("%s %u\n", "Registering cpu_dai", pdev->id);

	err = snd_soc_register_dai(&pdev->dev, &dmw_tdm_dai[pdev->id]);
	if (err) {
		vdbg_prt("%s %u %s\n", "Registering cpu-dai", pdev->id, "Failed !!!");
	}

	return err;
}

static int __devexit dmw_tdm_dev_remove(struct platform_device *pdev)
{
	dbg_prt();

	vdbg_prt("%s %u\n", "Unregister cpu-dai", pdev->id);

	snd_soc_unregister_dai(&pdev->dev);

	return 0;
}

static int dmw_tdm_dev_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct dmw_tdm_priv *tdm = dev_get_drvdata(&pdev->dev);

	dbg_prt();

	if (tdm) {
		if (!tdm->css_owns_tdm) {
			/* Disable dai's TDM machine clock,
			 * If (!) Is active upon 'suspend'
			 */
			if (tdm->tdm_clk_active) {
				vdbg_prt("%s %u\n",
					"Disabling TDM clock of TDM id",
					tdm->dai_id);
				
				clk_disable(tdm->clk);
			}
		} else {
			vdbg_prt("%s %u %s\n",
				"TDM id",
				tdm->dai_id,
				"owned by CSS. Ignoring!");
		}

	} else {
		vdbg_prt("%s\n", "No driver_data. Ignoring!");
	}

	return 0;
}

static int dmw_tdm_dev_resume(struct platform_device * pdev)
{
	struct dmw_tdm_priv *tdm = dev_get_drvdata(&pdev->dev);

	dbg_prt();

	if (tdm) {
		if (!tdm->css_owns_tdm) {
			if (tdm->tdm_clk_active) {
				/* Re-Enable dai's TDM machine clock,
				 * If (!) Was active upon 'suspend'
				 */
				vdbg_prt("%s %u\n",
					"Re-Enabling TDM clock of TDM id",
					tdm->dai_id);

				clk_enable(tdm->clk);
			}
		} else {
			vdbg_prt("%s %u %s\n",
				"TDM id",
				tdm->dai_id,
				"owned by CSS. Ignoring!");
		}

	} else {
		vdbg_prt("%s\n", "No driver_data. Ignoring!");
	}

	return 0;
}

static struct platform_driver dmw_tdm_driver = {
	.probe		= dmw_tdm_dev_probe,
	.remove		= __devexit_p(dmw_tdm_dev_remove),
	.suspend	= dmw_tdm_dev_suspend,
	.resume		= dmw_tdm_dev_resume,
	.driver		= {
		.name	= DRV_NAME,
		.owner	= THIS_MODULE,
	},
};

/*****************************************************************************/

static int __init dmw_tdm_init(void)
{
	long actual_dp_clk;
	long desired_dp_clk;
	struct clk * pll1_clk = NULL;
	int err;

	dbg_prt();

	/* Set dmw-96 DP clock, assuming source is PLL1 ("sysclk"): */
	g_dp_clk = clk_get_sys(DEVICE_NAME, "dp");
	if (g_dp_clk) {
		vdbg_prt("%s\n", "Got DP clk");
	} else {
		vdbg_prt("%s\n", "Can't get DP clk");
		
		return (int)g_dp_clk;
	}
		
	pll1_clk = clk_get_sys(DEVICE_NAME, "pll1");
	if (pll1_clk) {
		g_pll1_rate = (long)clk_get_rate(pll1_clk);
		vdbg_prt("%s %li %s\n", "PLL1 freq is:", g_pll1_rate, "Hz");
	} else {
		vdbg_prt("%s\n", "Can't get PLL1 clk");
		
		return (int)pll1_clk;
	}

	switch (g_pll1_rate) {
	case SYSCLK__480_MHz:
		desired_dp_clk = tdm_clks_tab[kSYS_CLK_480MHz].dp_clk_rate;
		break;
	case SYSCLK__246_MHz:
		desired_dp_clk = tdm_clks_tab[kSYS_CLK_246MHz].dp_clk_rate;
		break;
	/* Add for other system-clock freq */
	default:
		desired_dp_clk = tdm_clks_tab[kSYS_CLK_DEFAULT].dp_clk_rate;
		break;
	}

	actual_dp_clk = clk_round_rate(g_dp_clk, desired_dp_clk);
	if (actual_dp_clk < 0) {
		vdbg_prt("%s\n", "Can't set desired rate of DP clk");

		return (int)actual_dp_clk;
	}
	clk_set_rate(g_dp_clk, (unsigned long)actual_dp_clk);
	vdbg_prt("%s %li %s\n", "DP clk rate is:", actual_dp_clk, "Hz");
	
	vdbg_prt("%s\n", "Registrating platform driver");
	err = platform_driver_register(&dmw_tdm_driver);
	if (err) {
		vdbg_prt("%s\n", "platform_driver_register Failed !!!");
	}
	return err;
}
module_init(dmw_tdm_init);

static void __exit dmw_tdm_exit(void)
{
	dbg_prt();

	platform_driver_unregister(&dmw_tdm_driver);

	clk_put(g_dp_clk);
	g_dp_clk = NULL;
}
module_exit(dmw_tdm_exit);

MODULE_DESCRIPTION("DMW ASoC platform CPU DAI driver");
MODULE_AUTHOR("DSP Group");
MODULE_LICENSE("GPL");
MODULE_ALIAS("platform:" DRV_NAME);

