/*
 * si3050.c  --  Silicon Labs SI3050 ALSA SoC Audio Codec driver
 *
 * Copyright (C) 2011 DSPG (DSP Group)
 *
 * Based on wm8978.c:
 *  Copyright (C) 2009-2010 Guennadi Liakhovetski <g.liakhovetski@gmx.de>
 *  Copyright (C) 2007 Carlos Munoz <carlos@kenati.com>
 *  Copyright 2006-2009 Wolfson Microelectronics PLC.
 *  Based on wm8974 and wm8990 by Liam Girdwood <lrg@slimlogic.co.uk>
 *
 * Adapted by Murali TD. Mohan <Murali.Mohan@dspg.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

#define DEBUG
#define V_DEBUG

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/pm.h>
#include <linux/spi/spi.h>
#include <linux/platform_device.h>
#include <linux/slab.h>
#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>
#include <sound/soc-dapm.h>
#include <sound/initval.h>
#include <sound/tlv.h>
#include <asm/div64.h>
#include <linux/gpio.h>
#include <sound/si3050.h>
#include <linux/ioctl.h>
#include <linux/fs.h>
#include <linux/device.h>
#include <linux/err.h>
#include <linux/list.h>
#include <linux/errno.h>
#include <linux/mutex.h>

#include <linux/spi/spi.h>
#include <linux/spi/spidev.h>
#include <sound/si3050.h>
#include <asm/uaccess.h>

/*****************************************************************************/

#ifdef V_DEBUG
	#ifndef DEBUG 
		#define DEBUG
	#endif
	#define vdbg_prt(fmt, ...) \
		pr_debug("SI3050 asoc Cdec-drv: %s(): " fmt, __func__, ##__VA_ARGS__)
#else
	#define vdbg_prt(fmt, ...)
#endif


#ifdef DEBUG
	#define dbg_prt(fmt, ...) \
		pr_debug("SI3050 asoc Cdec-drv: %s(): " fmt, __func__, ##__VA_ARGS__)
#else
	#define dbg_prt(fmt, ...)	
#endif



/*** SPI USER SPACE INTERFACE *************************************************/


static DECLARE_BITMAP(minors, N_SPI_MINORS);
static LIST_HEAD(device_list);
static DEFINE_MUTEX(device_list_lock);

static unsigned bufsiz = 4096;

static int reset_inited = 0;

/*
 * We can't use the standard synchronous wrappers for file I/O; we
 * need to protect against async removal of the underlying spi_device.
 */
static void si3050_spidev_complete(void *arg)
{
	complete(arg);
}

static ssize_t
si3050_spidev_sync(struct si3050_spidev_data *si3050_spidev, struct spi_message *message)
{
	DECLARE_COMPLETION_ONSTACK(done);
	int status;

	message->complete = si3050_spidev_complete;
	message->context = &done;

	spin_lock_irq(&si3050_spidev->spi_lock);
	
	if (si3050_spidev->spi == NULL) {
		status = -ESHUTDOWN;
	} else {
		status = spi_async(si3050_spidev->spi, message);
	}

	spin_unlock_irq(&si3050_spidev->spi_lock);

	if (status == 0) {
		wait_for_completion(&done);
		status = message->status;
		if (status == 0)
			status = message->actual_length;
	}
	return status;
}

static inline ssize_t
si3050_spidev_sync_write(struct si3050_spidev_data *si3050_spidev, size_t len)
{
	struct spi_transfer t = {
		.tx_buf=si3050_spidev->buffer,
		.len=len,
	};
	struct spi_message m;

	spi_message_init(&m);
	spi_message_add_tail(&t, &m);
	return si3050_spidev_sync(si3050_spidev, &m);
}

static inline ssize_t
si3050_spidev_sync_read(struct si3050_spidev_data *si3050_spidev, size_t len)
{
	struct spi_transfer t = {
			.rx_buf	= si3050_spidev->buffer,
			.len	= len,
	};
	struct spi_message m;

	spi_message_init(&m);
	spi_message_add_tail(&t, &m);
	return si3050_spidev_sync(si3050_spidev, &m);
}

/* Read-only message with current device setup */
static ssize_t
si3050_spidev_read(struct file *filp, char __user *buf, size_t count, loff_t *f_pos)
{
	struct si3050_spidev_data *si3050_spidev;
	ssize_t status = 0;

	/* chipselect only toggles at start or end of operation */
	if (count > bufsiz) {
		return -EMSGSIZE;
	}

	si3050_spidev = filp->private_data;

	mutex_lock(&si3050_spidev->buf_lock);
	status = si3050_spidev_sync_read(si3050_spidev, count);
	if (status > 0) {
		unsigned long missing;

		missing = copy_to_user(buf, si3050_spidev->buffer, status);
		if (missing == status) {
			status = -EFAULT;
		} else {
			status = status - missing;
		}
	}
	mutex_unlock(&si3050_spidev->buf_lock);

	return status;
}

/* Write-only message with current device setup */
static ssize_t
si3050_spidev_write(struct file *filp, const char __user *buf,
			size_t count, loff_t *f_pos)
{
	struct si3050_spidev_data *si3050_spidev;
	ssize_t status = 0;
	unsigned long missing;

	/* chipselect only toggles at start or end of operation */
	if (count > bufsiz)
		return -EMSGSIZE;

	si3050_spidev = filp->private_data;

	mutex_lock(&si3050_spidev->buf_lock);
	missing = copy_from_user(si3050_spidev->buffer, buf, count);
	if (missing == 0) {
		status = si3050_spidev_sync_write(si3050_spidev, count);
	} else {
		status = -EFAULT;
	}
	mutex_unlock(&si3050_spidev->buf_lock);

	return status;
}

static int si3050_spidev_message(struct si3050_spidev_data *si3050_spidev,
        struct spi_ioc_transfer *u_xfers, unsigned n_xfers)
{
	struct spi_message msg;
	struct spi_transfer *k_xfers;
	struct spi_transfer *k_tmp;
	struct spi_ioc_transfer *u_tmp;
	unsigned int n, total;
	u8 *buf;
	int status = -EFAULT;

	spi_message_init(&msg);
	k_xfers = kcalloc(n_xfers, sizeof(*k_tmp), GFP_KERNEL);
	if (k_xfers == NULL) {
		return -ENOMEM;
	}

	/* Construct spi_message, copying any tx data to bounce buffer.
	* We walk the array of user-provided transfers, using each one
	* to initialize a kernel version of the same transfer.
	*/
	buf = si3050_spidev->buffer;
	total = 0;
	for (n = n_xfers, k_tmp = k_xfers, u_tmp = u_xfers;
	     n;
	     n--, k_tmp++, u_tmp++) {

		k_tmp->len = u_tmp->len;

		total += k_tmp->len;
		if (total > bufsiz) {
			status = -EMSGSIZE;
			goto done;
		}

		if (u_tmp->rx_buf) {
			k_tmp->rx_buf = buf;
			if (!access_ok( VERIFY_WRITE,
					(u8 __user *)(uintptr_t)u_tmp->rx_buf,
					u_tmp->len )) {
				goto done;
			}
		}
		if (u_tmp->tx_buf) {
			k_tmp->tx_buf = buf;
			if (copy_from_user( buf,
					    (const u8 __user *)(uintptr_t)u_tmp->tx_buf,
					    u_tmp->len )) {
				goto done;
			}
		}
		buf += k_tmp->len;

		k_tmp->cs_change = !!u_tmp->cs_change;
		k_tmp->bits_per_word = u_tmp->bits_per_word;
		k_tmp->delay_usecs = u_tmp->delay_usecs;
		k_tmp->speed_hz = u_tmp->speed_hz;


/* A.M. This section can't compile: */		
//#ifdef VERBOSE
#if 0
		dev_dbg(&si3050_spidev->spi->dev,
			"  xfer len %zd %s%s%s%dbits %u usec %uHz\n",
		u_tmp->len,
		u_tmp->rx_buf ? "rx " : "",
		u_tmp->tx_buf ? "tx " : "",
		u_tmp->cs_change ? "cs " : "",
		u_tmp->bits_per_word ? /*what val?*/ : si3050_spidev->spi->bits_per_word,
		u_tmp->delay_usecs,
		u_tmp->speed_hz ? /*what val?*/ : si3050_spidev->spi->max_speed_hz);
#endif
		spi_message_add_tail(k_tmp, &msg);
	}

	status = si3050_spidev_sync(si3050_spidev, &msg);
	if (status < 0) {
		goto done;
	}

	/* copy any rx data out of bounce buffer */
	buf = si3050_spidev->buffer;
	for (n = n_xfers, u_tmp = u_xfers; n; n--, u_tmp++) {
		if (u_tmp->rx_buf) {
			if (__copy_to_user( (u8 __user *)(uintptr_t)u_tmp->rx_buf,
					     buf,
					     u_tmp->len )) {
				status = -EFAULT;
				goto done;
			}
		}
		buf += u_tmp->len;
	}
	status = total;

done:
	kfree(k_xfers);
	return status;
}

static long
si3050_spidev_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	int err = 0;
	int retval = 0;
	struct si3050_spidev_data *si3050_spidev;
	struct spi_device *spi;
	u32 tmp;
	unsigned int n_ioc;
	struct spi_ioc_transfer *ioc;
	struct si3050_platform_data *pdata=NULL;

	/* Check type and command number */
	if (_IOC_TYPE(cmd) != SPI_IOC_MAGIC) {
		return -ENOTTY;
	}

	/* Check access direction once here; don't repeat below.
	* IOC_DIR is from the user perspective, while access_ok is
	* from the kernel perspective; so they look reversed.
	*/
	if (_IOC_DIR(cmd) & _IOC_READ) {
		err = !access_ok(VERIFY_WRITE, (void __user *)arg,
					_IOC_SIZE(cmd));
	}
	
	if (err == 0 && (_IOC_DIR(cmd) & _IOC_WRITE)) {
		err = !access_ok(VERIFY_READ,(void __user *)arg,
					_IOC_SIZE(cmd));
	}
	
	if (err) {
		return -EFAULT;
	}

	/* guard against device removal before, or while,
	* we issue this ioctl.
	*/
	si3050_spidev = filp->private_data;

	spin_lock_irq(&si3050_spidev->spi_lock);

	spi = spi_dev_get(si3050_spidev->spi);
	pdata = spi->dev.platform_data;

	spin_unlock_irq(&si3050_spidev->spi_lock);

	if (spi == NULL) {
		return -ESHUTDOWN;
	}

	/* use the buffer lock here for triple duty:
	*  - prevent I/O (from us) so calling spi_setup() is safe;
	*  - prevent concurrent SPI_IOC_WR_* from morphing
	*    data fields while SPI_IOC_RD_* reads them;
	*  - SPI_IOC_MESSAGE needs the buffer locked "normally".
	*/
	mutex_lock(&si3050_spidev->buf_lock);

	switch (cmd) {
		/* read requests */
		case SPI_IOC_RD_MODE:
			retval = __put_user(spi->mode & SPI_MODE_MASK,
						(__u8 __user *)arg);
		break;
		case SPI_IOC_RD_LSB_FIRST:
			retval = __put_user((spi->mode & SPI_LSB_FIRST) ?  1 : 0,
						(__u8 __user *)arg);
		break;
		case SPI_IOC_RD_BITS_PER_WORD:
			retval = __put_user(spi->bits_per_word,
						(__u8 __user *)arg);
		break;
		case SPI_IOC_RD_MAX_SPEED_HZ:
			retval = __put_user(spi->max_speed_hz,
					(__u32 __user *)arg);
		break;

		/* write requests */
		case SPI_IOC_WR_MODE:
			retval = __get_user(tmp, (u8 __user *)arg);
			if (retval == 0) {
				u8  save = spi->mode;

				if (tmp & ~SPI_MODE_MASK) {
					retval = -EINVAL;
					break;
				}

				tmp |= spi->mode & ~SPI_MODE_MASK;
				spi->mode = (u8)tmp;
				retval = spi_setup(spi);
				if (retval < 0) {
					spi->mode = save;
				} else {
					dev_dbg(&spi->dev,
						"spi mode %02x\n", tmp);
				}
			}
		break;
		case SPI_IOC_WR_LSB_FIRST:
			retval = __get_user(tmp, (__u8 __user *)arg);
			if (retval == 0) {
				u8  save = spi->mode;

				if (tmp) {
					spi->mode |= SPI_LSB_FIRST;
				} else {
					spi->mode &= ~SPI_LSB_FIRST;
				}
				retval = spi_setup(spi);
				if (retval < 0) {
					spi->mode = save;
				} else {
					dev_dbg(&spi->dev, "%csb first\n",
						tmp ? 'l' : 'm');
				}
			}
		break;
		case SPI_IOC_WR_BITS_PER_WORD:
			retval = __get_user(tmp, (__u8 __user *)arg);
			if (retval == 0) {
				u8  save = spi->bits_per_word;

				spi->bits_per_word = tmp;
				retval = spi_setup(spi);
				if (retval < 0) {
					spi->bits_per_word = save;
				} else {
					dev_dbg(&spi->dev,
						"%d bits per word\n", tmp);
				}
			}
		break;
		case SPI_IOC_WR_MAX_SPEED_HZ:
			retval = __get_user(tmp, (__u32 __user *)arg);
			if (retval == 0) {
				u32 save = spi->max_speed_hz;

				spi->max_speed_hz = tmp;
				retval = spi_setup(spi);
				if (retval < 0) {
					spi->max_speed_hz = save;
				} else {
					dev_dbg(&spi->dev, "%d Hz (max)\n", tmp);
				}
			}
		break;

		case SI3050_IOC_RESET:
			si3050_reset((unsigned int)pdata->reset);
		break;

		default:
			/* segmented and/or full-duplex I/O request */
			if ( (_IOC_NR(cmd) != _IOC_NR(SPI_IOC_MESSAGE(0))) ||
			     (_IOC_DIR(cmd) != _IOC_WRITE) ) {
				retval = -ENOTTY;
				break;
			}

			tmp = _IOC_SIZE(cmd);
			if ((tmp % sizeof(struct spi_ioc_transfer)) != 0) {
				retval = -EINVAL;
				break;
			}
			n_ioc = tmp / sizeof(struct spi_ioc_transfer);
			if (n_ioc == 0) {
				break;
			}

			/* copy into scratch area */
			ioc = kmalloc(tmp, GFP_KERNEL);
			if (!ioc) {
				retval = -ENOMEM;
				break;
			}
			if (__copy_from_user(ioc, (void __user *)arg, tmp)) {
				kfree(ioc);
				retval = -EFAULT;
				break;
			}

			/* translate to spi_message, execute */
			retval = si3050_spidev_message(si3050_spidev, ioc, n_ioc);
			kfree(ioc);
		break;
	}

	mutex_unlock(&si3050_spidev->buf_lock);

	spi_dev_put(spi);

	return retval;
}

static int si3050_spidev_open(struct inode *inode, struct file *filp)
{
	struct si3050_spidev_data *si3050_spidev;
	int status = -ENXIO;

	mutex_lock(&device_list_lock);

	list_for_each_entry(si3050_spidev, &device_list, device_entry) {
		if (si3050_spidev->devt == inode->i_rdev) {
			status = 0;
			break;
		}
	}

	if (status == 0) {
		if (!si3050_spidev->buffer) {
			si3050_spidev->buffer = kmalloc(bufsiz, GFP_KERNEL);
			if (!si3050_spidev->buffer) {
				dev_dbg(&si3050_spidev->spi->dev,
					"open/ENOMEM\n");
				status = -ENOMEM;
			}
		}
		if (status == 0) {
			si3050_spidev->users++;
			filp->private_data = si3050_spidev;
			nonseekable_open(inode, filp);
		}
	} else {
		pr_debug("si3050_spidev: nothing for minor %d\n",
			 iminor(inode));
	}

	mutex_unlock(&device_list_lock);

	return status;
}

static int si3050_spidev_release(struct inode *inode, struct file *filp)
{
	struct si3050_spidev_data *si3050_spidev;
	int status = 0;

	mutex_lock(&device_list_lock);
	
	si3050_spidev = filp->private_data;
	filp->private_data = NULL;

	/* last close? */
	si3050_spidev->users--;
	if (!si3050_spidev->users) {
		int     dofree;

		kfree(si3050_spidev->buffer);
		si3050_spidev->buffer = NULL;

		/* ... after we unbound from the underlying device? */
		spin_lock_irq(&si3050_spidev->spi_lock);
		dofree = (si3050_spidev->spi == NULL);
		spin_unlock_irq(&si3050_spidev->spi_lock);

		if (dofree) {
			kfree(si3050_spidev);
		}
	}

	mutex_unlock(&device_list_lock);

	return status;
}

static const struct file_operations si3050_spidev_fops = {
	.owner =    THIS_MODULE,
	/* REVISIT switch to aio primitives, so that userspace
	* gets more complete API coverage. It'll simplify things
	* too, except for the locking.
	*/
	.write		= si3050_spidev_write,
	.read		= si3050_spidev_read,
	.unlocked_ioctl	= si3050_spidev_ioctl,
	.open		= si3050_spidev_open,
	.release	= si3050_spidev_release,
};

/*---------------------------------------------------------------------------*/

/* The main reason to have this class is to make mdev/udev create the
 * /dev/si3050_spidevB.C character device nodes exposing our userspace API.
 * It also simplifies memory management.
 */

static struct class *si3050_spidev_class;

/*---------------------------------------------------------------------------*/

int si3050_spidev_probe(struct spi_device *spi)
{
	struct si3050_spidev_data *si3050_spidev;
	int status;
	unsigned long minor;
	
	/* Allocate driver data */
	si3050_spidev = kzalloc(sizeof(*si3050_spidev), GFP_KERNEL);
	if (!si3050_spidev) {
		return -ENOMEM;
	}

	/* Initialize the driver data */
	si3050_spidev->spi = spi;
	spin_lock_init(&si3050_spidev->spi_lock);
	mutex_init(&si3050_spidev->buf_lock);

	INIT_LIST_HEAD(&si3050_spidev->device_entry);

	/* If we can allocate a minor number, hook up this device.
	* Reusing minors is fine so long as udev or mdev is working.
	*/
	mutex_lock(&device_list_lock);

	minor = find_first_zero_bit(minors, N_SPI_MINORS);

	if (minor < N_SPI_MINORS) {
		struct device *dev;

		si3050_spidev->devt = MKDEV(SI3050_SPIDEV_MAJOR, minor);
		dev = device_create(si3050_spidev_class, &spi->dev, si3050_spidev->devt, si3050_spidev, "si3050spi%d.%d", spi->master->bus_num, spi->chip_select);
		status = IS_ERR(dev) ? PTR_ERR(dev) : 0;
	} else {
		dev_dbg(&spi->dev, "no minor number available!\n");
		status = -ENODEV;
	}

	if (status == 0) {
		set_bit(minor, minors);
		list_add(&si3050_spidev->device_entry, &device_list);
	}

	mutex_unlock(&device_list_lock);

	if (status == 0) {
		spi_set_drvdata(spi, si3050_spidev);
	} else {
		kfree(si3050_spidev);
	}

	return status;
}

int si3050_spidev_remove(struct spi_device *spi)
{
	struct si3050_spidev_data *si3050_spidev = spi_get_drvdata(spi);

	/* make sure ops on existing fds can abort cleanly */
	spin_lock_irq(&si3050_spidev->spi_lock);
	
	si3050_spidev->spi = NULL;
	spi_set_drvdata(spi, NULL);
	
	spin_unlock_irq(&si3050_spidev->spi_lock);

	/* prevent new opens */
	mutex_lock(&device_list_lock);
	
	list_del(&si3050_spidev->device_entry);
	device_destroy(si3050_spidev_class, si3050_spidev->devt);
	clear_bit(MINOR(si3050_spidev->devt), minors);
	if (si3050_spidev->users == 0) {
		kfree(si3050_spidev);
	}
	
	mutex_unlock(&device_list_lock);

	return 0;
}

int si3050_spidev_init(void)
{
	int status;

	/* Claim our 256 reserved device numbers.  Then register a class
	* that will key udev/mdev to add/remove /dev nodes.  Last, register
	* the driver which manages those device numbers.
	*/
	BUILD_BUG_ON(N_SPI_MINORS > 256);
	status = register_chrdev(SI3050_SPIDEV_MAJOR, "spi", &si3050_spidev_fops);
	if (status < 0) {
		return status;
	}

	si3050_spidev_class = class_create(THIS_MODULE, "si3050spi");
	if (IS_ERR(si3050_spidev_class)) {
		unregister_chrdev(SI3050_SPIDEV_MAJOR, "si3050spi");
		return PTR_ERR(si3050_spidev_class);
	}

	return status;
}

void si3050_spidev_exit(void)
{
	class_destroy(si3050_spidev_class);
	unregister_chrdev(SI3050_SPIDEV_MAJOR, "si3050spi");
}

/******************* END OF SPI USER SPACE INTERFACE **************************/

/* codec private data */
struct si3050_priv {
	struct snd_soc_codec codec;
	struct spi_device *spi;
	struct si3050_platform_data *pdata;
};

/*
 * Configure SI3050 clock dividers.
 */
static int si3050_dai_set_clkdiv(struct snd_soc_dai *codec_dai,
				 int div_id, int div)
{
	dbg_prt("%s\n", "Currently doing nothing ---->");
	return 0;
}

/*
 * @freq:	freq is codec's input-clock frequency
 */
static int si3050_dai_set_sysclk(struct snd_soc_dai *codec_dai, int clk_id,
				 unsigned int freq, int dir)
{
	dbg_prt("%s\n", "Currently doing nothing ---->");
	return 0;
}

static int si3050_dai_set_pll(struct snd_soc_dai *dai, int pll_id, int source,
			      unsigned int freq_in, unsigned int freq_out)
{
	dbg_prt("%s\n", "Currently doing nothing ---->");
	return 0;
}

/*
 * Set ADC and Voice DAC format.
 */
static int si3050_dai_set_fmt(struct snd_soc_dai *codec_dai, unsigned int fmt)
{
	dbg_prt("%s\n", "Currently doing nothing ---->");
	return 0;
}

/*
 * Set PCM DAI bit size and sample rate.
 */
static int si3050_dai_hw_params(struct snd_pcm_substream *substream,
			    	struct snd_pcm_hw_params *params,
			    	struct snd_soc_dai *dai)
{
	dbg_prt("%s\n", "Currently doing nothing ---->");
	return 0;
}

static int si3050_dai_hw_free(struct snd_pcm_substream *substream,
			      struct snd_soc_dai *cpu_dai)
{
	dbg_prt("%s\n", "Currently doing nothing ---->");
	return 0;
}

static int si3050_dai_prepare(struct snd_pcm_substream *substream,
			      struct snd_soc_dai *dai)
{
	dbg_prt("%s\n", "Currently doing nothing ---->");
	return 0;
}	

static int si3050_dai_trigger(struct snd_pcm_substream *substream, int cmd,
			      struct snd_soc_dai *dai)
{
	dbg_prt("%s\n", "Currently doing nothing ---->");
	return 0;
}	

static int si3050_dai_mute(struct snd_soc_dai *dai, int mute)
{
	dbg_prt("%s %s\n", "set DACs Soft mute", (mute == 0) ? "Off" : "On");
	return 0;
}

static int si3050_dai_startup(struct snd_pcm_substream *substream,
			      struct snd_soc_dai *dai)
{
	dbg_prt("%s\n", "Currently doing nothing ---->");
	return 0;
}

static void si3050_dai_shutdown(struct snd_pcm_substream *substream,
				struct snd_soc_dai *dai)
{
	dbg_prt("%s\n", "Currently doing nothing ---->");
}	

static struct snd_soc_dai_ops si3050_dai_ops = {
	.set_sysclk =	si3050_dai_set_sysclk,
	.set_pll =	si3050_dai_set_pll,
	.set_clkdiv =	si3050_dai_set_clkdiv,
	.set_fmt =	si3050_dai_set_fmt,
	.digital_mute =	si3050_dai_mute,
	.startup =	si3050_dai_startup,
	.shutdown =	si3050_dai_shutdown,
	.hw_params =	si3050_dai_hw_params,
	.hw_free =	si3050_dai_hw_free,
	.prepare =	si3050_dai_prepare,
	.trigger =	si3050_dai_trigger,
};

/*****************************************************************************/

static int si3050_drv_suspend(struct snd_soc_codec *codec, pm_message_t state)
{
	dbg_prt("%s\n", "Currently doing nothing ---->");
	return 0;
}

static int si3050_drv_resume(struct snd_soc_codec *codec)
{
	dbg_prt("%s\n", "Currently doing nothing ---->");
	return 0;
}

static int si3050_drv_probe(struct snd_soc_codec *codec)
{
	dbg_prt("%s\n", "Currently doing nothing ---->");
	return 0;
}

/* power down chip */
static int si3050_drv_remove(struct snd_soc_codec *codec)
{
	dbg_prt("%s\n", "Currently doing nothing ---->");
	return 0;
}

static struct snd_soc_codec_driver soc_codec_dev_si3050 = {
	.probe =		si3050_drv_probe,
	.remove =		si3050_drv_remove,
	.suspend =		si3050_drv_suspend,
	.resume =		si3050_drv_resume,
	.set_bias_level	=	NULL,
	.reg_cache_size	=	0,
	.reg_word_size	=	0,
	.reg_cache_default =	0,
};

/*****************************************************************************/

struct snd_soc_dai_driver si3050_dai_drv = {
	.name = "si3050-fxo",
	.playback = {
		.stream_name	= "Playback",
		.channels_min	= 1,
		.channels_max	= 1,	/* SI3050 has Single output-path */
		.rates		= SNDRV_PCM_RATE_8000, /* A.M. was 16KHz */
		.formats	= SI3050_FORMATS,
	},
	.capture = {
		.stream_name	= "Capture",
		.channels_min	= 1,
		.channels_max	= 1,    /* SI3050 has Single input-path */
		.rates		= SNDRV_PCM_RATE_8000, /* A.M. was 16KHz */
		.formats	= SI3050_FORMATS,
	},
	.ops = &si3050_dai_ops,
};



void si3050_reset(unsigned int gpio)
{
	dbg_prt();

	gpio_direction_output(gpio, 1);
	udelay(100);
	gpio_set_value(gpio, 0);
	udelay(2000);
	gpio_set_value(gpio, 1);
	udelay(2000);
}


static int si3050_spi_probe(struct spi_device *spi)
{
	struct si3050_platform_data *pdata = spi->dev.platform_data;
	struct si3050_priv *si3050;
	struct snd_soc_codec *codec;
	int ret;

	dbg_prt();

	if (!pdata) {
		dev_err(&spi->dev, "no platform data provided\n");
		return -ENODEV;
	}

	si3050 = kzalloc(sizeof(struct si3050_priv), GFP_KERNEL);
	if (si3050 == NULL) {
		return -ENOMEM;
	}

	codec = &si3050->codec;
	codec->dev = &spi->dev;

	si3050->pdata = pdata;
	spi_set_drvdata(spi, si3050);


	/*
	* setup spi parameters; this makes sure that parameters we request
	* are acceptable by the spi driver
	*/
	spi->mode = SPI_MODE_0; /* clk active high */
	spi->bits_per_word = 8;

	ret = spi_setup(spi);
	if (ret < 0) {
		dev_err(&spi->dev, "spi_setup() failed\n");
		goto err_si3050;
	}

	si3050->spi = spi;

	/* reset FXO chips only the first time. All are controlled by same reset pin */
	if (!reset_inited)
	{
		ret = gpio_request(si3050->pdata->reset, "si3050 reset");
		if (ret < 0) {
			dev_err(&spi->dev, "failed requesting reset gpio\n");
			goto err_si3050;
		}
	
		ret = gpio_get_direction(si3050->pdata->reset); /* A.M. Murali - is this correct? */
		si3050_reset(si3050->pdata->reset);
		
		reset_inited = 1;
	}

	dev_info(&spi->dev, "initialized\n");
	si3050_spidev_probe(spi);

	vdbg_prt("%s\n", "Registering asoc Codec driver and Dai");
	
	ret = snd_soc_register_codec(&spi->dev, &soc_codec_dev_si3050, &si3050_dai_drv, 1);
	if (ret < 0) {
		vdbg_prt("%s\n", "Asoc codec driver and dai registration Failed !!!");
		goto err_si3050;
	}
	return ret;


err_si3050:
	kfree(si3050);
	return ret;
}

static int __devexit si3050_spi_remove(struct spi_device *spi)
{
	struct si3050_priv *si3050 = spi_get_drvdata(spi);
	dbg_prt();

	snd_soc_unregister_codec(&spi->dev);
	si3050_spidev_remove(spi);

	kfree(si3050);

	return 0;
}

static struct spi_driver dmw96_si3050_spi_driver = {
	.driver = {
		.name	= "dmw96_si3050_spi",
		.bus	= &spi_bus_type,
		.owner	= THIS_MODULE,
	},
	.probe	= si3050_spi_probe,
	.remove	= __exit_p(si3050_spi_remove),
};

static int __init si3050_module_init(void)
{
	dbg_prt();

	si3050_spidev_init();
	return spi_register_driver(&dmw96_si3050_spi_driver);
}

static __devexit void si3050_module_exit(void)
{
	dbg_prt();

	si3050_spidev_exit();
	spi_unregister_driver(&dmw96_si3050_spi_driver);
}


module_init(si3050_module_init);
module_exit(si3050_module_exit);

MODULE_DESCRIPTION("ASoC SI3050 D.A.A. codec driver");
MODULE_AUTHOR("DSP Group");
MODULE_LICENSE("GPL");
