/*
 * dmw_wdt.c - DMW watchdog userspace interface
 *
 * Copyright (c) 2011 DSPG Technologies GmbH
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 */

#include <linux/fs.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/miscdevice.h>
#include <linux/module.h>
#include <linux/clk.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <linux/watchdog.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <mach/hardware.h>
#include <mach/watchdog.h>

MODULE_DESCRIPTION("DMW watchdog driver");
MODULE_LICENSE("GPL");
MODULE_ALIAS_MISCDEV(WATCHDOG_MINOR);

#define WD_CTRL     0x00
#define WD_ENABLE   0x04
#define WD_REFRESH  0x08
#define WD_INTCLR   0x1c
#define WD_INTSTAT  0x20

#define WD_CTRL_PERIOD_1s     (0 << 0)
#define WD_CTRL_PERIOD_10s    (1 << 0)
#define WD_CTRL_PERIOD_100s   (2 << 0)
#define WD_CTRL_PERIOD_1000s  (3 << 0)
#define WD_CTRL_INTEN         (1 << 2)

static void __iomem *dmw_wdt_base;
static int dmw_wdt_boot_status;
static unsigned long dmw_wdt_opened;

static void
dmw_wdt_enable(void)
{
	writew(0xabcd, dmw_wdt_base + WD_ENABLE);
	writew(0xabcd, dmw_wdt_base + WD_ENABLE);
}

static void
dmw_wdt_refresh(void)
{
	writew(0xa5a5, dmw_wdt_base + WD_REFRESH);
	writew(0x5a5a, dmw_wdt_base + WD_REFRESH);
}

static int
dmw_wdt_open(struct inode *inode, struct file *file)
{
	if (test_and_set_bit(0, &dmw_wdt_opened))
		return -EBUSY;

	dmw_wdt_enable();
	printk("DMW watchdog timer started\n");

	return nonseekable_open(inode, file);
}

static int
dmw_wdt_release(struct inode *inode, struct file *file)
{
	/*
	 * once the watchdog is running it cannot be disabled, we can refresh
	 * it once more, though
	 */
	dmw_wdt_refresh();

	clear_bit(0, &dmw_wdt_opened);
	return 0;
}

static ssize_t
dmw_wdt_write(struct file *file, const char *data, size_t len, loff_t * ppos)
{
	if (len)
		dmw_wdt_refresh();

	return len;
}

static struct watchdog_info info = {
	.identity = "DMW watchdog",
	.options = WDIOF_CARDRESET,
};

static int
dmw_wdt_ioctl(struct inode *inode, struct file *file,
              unsigned int cmd, unsigned long arg)
{
	int ret = -EOPNOTSUPP;
	void __user *argp = (void __user *)arg;
	int __user *p = argp;

	switch (cmd) {
	case WDIOC_GETSUPPORT:
		ret = copy_to_user(argp, &info,
		                   sizeof(info)) ? -EFAULT : 0;
		break;

	case WDIOC_GETSTATUS:
		ret = put_user(0, p);
		break;

	case WDIOC_GETBOOTSTATUS:
		ret = put_user(dmw_wdt_boot_status, p);
		break;

	case WDIOC_KEEPALIVE:
		dmw_wdt_refresh();
		ret = 0;
		break;
	}

	return ret;
}

static int
dmw_wdt_fsync(struct file *file, int datasync)
{
	return 0;
}

static const struct file_operations dmw_wdt_fops = {
	.owner		= THIS_MODULE,
	.llseek		= no_llseek,
	.write		= dmw_wdt_write,
	.compat_ioctl	= dmw_wdt_ioctl,
	.open		= dmw_wdt_open,
	.release	= dmw_wdt_release,
	.fsync		= dmw_wdt_fsync,
};

static struct miscdevice dmw_wdt_miscdev = {
	.minor		= WATCHDOG_MINOR,
	.name		= "watchdog",
	.fops		= &dmw_wdt_fops,
};

static irqreturn_t
dmw_wdt_isr(int irq, void *unused)
{
	return IRQ_HANDLED;
}

static int __init
dmw_wdt_probe(struct platform_device *pdev)
{
	int irq, ret;
	struct resource *res;
	struct dmw_wdt_platform_data *pdata = pdev->dev.platform_data;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res)
		return -EINVAL;

	irq = platform_get_irq(pdev, 0);
	if (irq < 0)
		return irq;

	if (!request_mem_region(res->start, res->end - res->start + 1,
	    pdev->name)) {
		dev_err(&pdev->dev, "failed to request memory resource\n");
		return -EBUSY;
	}

	dmw_wdt_base = ioremap_nocache(res->start, res->end - res->start + 1);
	if (!dmw_wdt_base) {
		dev_err(&pdev->dev, "failed to map registers\n");
		return -ENOMEM;
	}

	ret = request_irq(irq, dmw_wdt_isr, 0, NULL, NULL);
	if (ret) {
		dev_err(&pdev->dev, "error requesting irq\n");
		return -EIO;
	}

	/* was the last reset triggered by us? */
	if (pdata && pdata->get_wdt_reset_ind && pdata->get_wdt_reset_ind())
		dmw_wdt_boot_status |= WDIOF_CARDRESET;

	/* setup with 10 seconds period */
	writew(WD_CTRL_PERIOD_10s, dmw_wdt_base + WD_CTRL);

	ret = misc_register(&dmw_wdt_miscdev);
	if (ret)
		printk("%s(): error registering miscdev: %d\n", __func__, ret);

	printk("DMW watchdog timer initialized, not started\n");

	return ret;
}

static struct platform_driver watchdog_driver = {
	.driver = {
		.name = "dmw_wdt",
		.owner = THIS_MODULE,
	},
};

static int __init
dmw_wdt_init(void)
{
	return platform_driver_probe(&watchdog_driver, dmw_wdt_probe);
}

module_init(dmw_wdt_init);

