/*
 * linux/drivers/video/grandstream/lcdfb.c
 *
 *  Grandstream uc1701-spi LCD Driver
 *
 *  This file is subject to the terms and conditions of the GNU General Public
 *  License. See the file COPYING in the main directory of this archive for
 *  more details.
 */

#define CONFIG_HW_MODEL_GXP1400
#include <linux/version.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/string.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/fb.h>
#include <linux/init.h>
#include <linux/clk.h>
#include <linux/spi/spi.h>
#include <linux/platform_device.h>
#include <linux/dma-mapping.h>
#include <linux/semaphore.h>
#include <linux/completion.h>
#include <linux/proc_fs.h>
#include <linux/interrupt.h>
#include <asm/ioctl.h>
#include <asm/uaccess.h>
#include <linux/gpio.h>
#include <gs_icon.h>
#include <gs_misc.h>

#include "kgui.h"

#define NEW_LCD 1

#if NEW_LCD
#define LCD_SIZE_X  128
#define LCD_SIZE_Y  32
#else
#define LCD_SIZE_X  128
#define LCD_SIZE_Y  40 
#endif

#define SCREEN_SIZE_X 132
#define SCREEN_SIZE_Y LCD_SIZE_Y
#define SCREEN_BPP    1

#define DBG
#ifdef DBG
#define DEBUG(args...) printk(args)
#else
#define DEBUG(args...)
#endif

//#define GPIO_LCD_CD    GPIO_PORTG(17)
#define GPIO_LCD_RST   GPIO_PORTF(19)

#define LCD_SET_CMD( cd ) do { gpio_set_value( (cd), 0 ); } while ( 0 )
#define LCD_SET_DAT( cd ) do { gpio_set_value( (cd), 1 ); } while ( 0 )

/*
 * Driver data
 */
#define UC1701FB_NAME "uc1701fb-spi"

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,36)
static DECLARE_MUTEX(sem_lcd);
#else
static DEFINE_SEMAPHORE(sem_lcd);
#endif

/*
 * KGUI
 */

static void fb_sync( void );
static void fb_clear( void );

/*
 * LCD Configuration
 */

/*
 * Configure attached panel
 */

/*
 * This structure defines the hardware state of the graphics card. Normally
 * you place this in a header file in linux/include/video. This file usually
 * also includes register information. That allows other driver subsystems
 * and userland applications the ability to use the same header file to
 * avoid duplicate work and easy porting of software.
 */
struct uc1701fb_par
{
    unsigned char* fb_buffer;
    unsigned int fb_size;
    dma_addr_t dma_handle;
    struct spi_device * spi;
};

struct uc1701fb_par* dev_data;

int contrast_now = 50; 
/*
 * Here we define the default structs fb_fix_screeninfo and fb_var_screeninfo
 * if we don't use modedb. If we do use modedb see xxxfb_init how to use it
 * to get a fb_var_screeninfo. Otherwise define a default var as well.
 */
static struct fb_fix_screeninfo uc1701fb__devinitdata = {
    .id =       UC1701FB_NAME,
    .type =     FB_TYPE_PACKED_PIXELS,
    .visual =   FB_VISUAL_PSEUDOCOLOR,
    .xpanstep = 0,
    .ypanstep = 0,
    .ywrapstep = 0,
    .line_length = 132,
    .accel =    FB_ACCEL_NONE,
};

static struct fb_var_screeninfo uc1701fb_var_default = {
    .xres = 132,
    .yres = LCD_SIZE_Y,
    .xres_virtual = 132,
    .yres_virtual = LCD_SIZE_Y,
    .bits_per_pixel = 1,
    .grayscale = 0,
    .left_margin = 0,
    .right_margin = 0,
    .upper_margin = 0,
    .lower_margin = 0,
    .vmode = FB_VMODE_NONINTERLACED,
};

    /*
     *  Modern graphical hardware not only supports pipelines but some
     *  also support multiple monitors where each display can have its
     *  its own unique data. In this case each display could be
     *  represented by a separate framebuffer device thus a separate
     *  struct fb_info. Now the struct xxx_par represents the graphics
     *  hardware state thus only one exist per card. In this case the
     *  struct xxx_par for each graphics card would be shared between
     *  every struct fb_info that represents a framebuffer on that card.
     *  This allows when one display changes it video resolution (info->var)
     *  the other displays know instantly. Each display can always be
     *  aware of the entire hardware state that affects it because they share
     *  the same xxx_par struct. The other side of the coin is multiple
     *  graphics cards that pass data around until it is finally displayed
     *  on one monitor. Such examples are the voodoo 1 cards and high end
     *  NUMA graphics servers. For this case we have a bunch of pars, each
     *  one that represents a graphics state, that belong to one struct
     *  fb_info. Their you would want to have *par point to a array of device
     *  states and have each struct fb_ops function deal with all those
     *  states. I hope this covers every possible hardware design. If not
     *  feel free to send your ideas at jsimmons@users.sf.net
     */

    /*
     *  If your driver supports multiple boards or it supports multiple
     *  framebuffers, you should make these arrays, or allocate them
     *  dynamically using framebuffer_alloc() and free them with
     *  framebuffer_release().
     */

/*
 * uc1701-spi control -- to be split off into physical layer module
 */
#define lcd_par struct uc1701fb_par

//===========================================================================
static int uc1701fb_mmap(
    struct fb_info *info,
    struct vm_area_struct *vma
)
//===========================================================================
{
    unsigned long size = vma->vm_end - vma->vm_start;
    unsigned long offset = vma->vm_pgoff << PAGE_SHIFT;
    
    if (offset + size > info->fix.smem_len) {
        return -EINVAL;
    }

    offset += (unsigned long)dev_data->dma_handle;
    vma->vm_page_prot = pgprot_noncached (vma->vm_page_prot) ;

	if (remap_pfn_range(vma, vma->vm_start, offset >> PAGE_SHIFT,
			    size, vma->vm_page_prot))
		return -EAGAIN;

    return 0;
}

//===========================================================================
static void uc1701_update_screen(
    lcd_par* par,
    int x,
    int y,
    int width,
    int height
)
//===========================================================================
{
    u8 cmd_buf[32];
    int page;
    int page_offset;
    struct gs_lcd_platform_data *pdata = (struct gs_lcd_platform_data *)dev_data->spi->dev.platform_data;
    unsigned int cd_gpio = pdata->lcd_cd;

    down( &sem_lcd );

    page_offset = SCREEN_SIZE_X;
    for ( page = 0; page < 5; page++ )
    {
        LCD_SET_CMD(cd_gpio);
        cmd_buf[0] = 0x04;
        cmd_buf[1] = 0x10;
        cmd_buf[2] = 0xB1 + page;
        spi_write( par->spi, cmd_buf, 3 );
        LCD_SET_DAT(cd_gpio);
        spi_write( par->spi, par->fb_buffer + page * page_offset, SCREEN_SIZE_X );
    }

    up( &sem_lcd );
}

//===========================================================================
static int
uc1701fb_check_var(
    struct fb_var_screeninfo *var,
    struct fb_info *info
)
//===========================================================================
{
    /* The panel parameters are supplied by user-space, we only provide low-level
     * access in this driver.  It is assumed user-space configuration is correct.
     */

    return 0;
}

//===========================================================================
static int
uc1701fb_set_par(
    struct fb_info *info
)
//===========================================================================
{

    return 0;
}

//===========================================================================
static ssize_t uc1701fb_read(
    struct fb_info* info,
    char* buf,
    size_t count,
    loff_t* ppos
)
//===========================================================================
{
    unsigned long p = *ppos;
    unsigned int fb_mem_len;

    fb_mem_len = info->var.xres * info->var.yres * info->var.bits_per_pixel;

    if ( p >= fb_mem_len )
        return 0;

    if ( count >= fb_mem_len )
        count = fb_mem_len;

    if ( count + p > fb_mem_len )
        count = fb_mem_len - p;

    if ( count )
    {
        char* base_addr;

        base_addr = info->screen_base;
        count -= copy_to_user( buf, base_addr + p, count );

        if ( !count )
            return -EFAULT;
        *ppos += count;
    }

    return count;
}
//===========================================================================
static ssize_t uc1701fb_write(
    struct fb_info *info,
    const char* buf,
    size_t count,
    loff_t* ppos
)
//===========================================================================
{
    unsigned long p = *ppos;
    unsigned int fb_mem_len;
    int err = 0;
    int written;

    fb_mem_len = ( info->var.xres * info->var.yres ) / ( 8 / info->var.bits_per_pixel );

    if ( p > fb_mem_len )
        return -ENOSPC;
    if ( count >= fb_mem_len )
        count = fb_mem_len;
    if ( count + p > fb_mem_len )
    {
        count = fb_mem_len - p;
        err = -ENOSPC;
    }

    if ( count )
    {
        char *base_addr;

        base_addr = info->screen_base;
        count -= copy_from_user( base_addr + p, buf, count );
        *ppos += count;

        written = count;

        uc1701_update_screen( info->par, 0, 0, SCREEN_SIZE_X, SCREEN_SIZE_Y );

        return count;
    }

    return err;
}

//===========================================================================
static int uc1701fb_ioctl(
    struct fb_info *info,
    unsigned int cmd,
    unsigned long arg)
//===========================================================================
{
    int err = 0;
    struct fb_update_rect update_rect;
    struct fb_kgui fb_kgui;
        int progress;

    /* This is redundant right now since we use copy_from_user in the one
     * place a user pointer is used, but still good practice
     */
    if ( _IOC_DIR(cmd) & _IOC_WRITE )
        err = !access_ok( VERIFY_READ, (void __user *)arg, _IOC_SIZE( cmd ) );

    if ( err )
        return -EFAULT;

    switch( cmd )
    {
        case FBIO_UPDATE:
            if ( copy_from_user( &update_rect, (void*)arg, sizeof( update_rect ) ) != 0 )
                return -EIO;
            uc1701_update_screen( info->par, update_rect.x, update_rect.y, update_rect.width, update_rect.height );
            break;
        case FBIO_PROGRESS:
            if ( copy_from_user( &progress, (void*)arg, sizeof ( progress ) ) != 0 )
                return -EIO;
            kgui_progress( progress );
            break;
        case FBIO_KGUI:
            if ( copy_from_user( &fb_kgui, (void*)arg, sizeof( fb_kgui ) ) != 0 )
                return -EIO;
            if ( fb_kgui.clear )
            {
                fb_clear();
            }
            if ( fb_kgui.string.enabled )
            {
                kgui_string( &fb_kgui.string.string[0], fb_kgui.string.location );
            }
            if ( fb_kgui.throbber.enabled )
            {
                kgui_throbber( fb_kgui.throbber.location );
            }
            else if ( fb_kgui.throbber.cancelled )
            {
                kgui_cancel_throbber();
            }
            if ( fb_kgui.progress.enabled )
            {
                kgui_enable_progress( fb_kgui.progress.location, fb_kgui.progress.percent );
            }
            if ( fb_kgui.progress.cancelled )
            {
                kgui_cancel_progress();
            }
            if ( fb_kgui.sync )
            {
                fb_sync();
            }
            return 0;
        default:
            return -EINVAL;
    }

    return 0;
}


    /*
     *  Frame buffer operations
     */

static struct fb_ops uc1701fb_ops = {
    .owner          = THIS_MODULE,
    .fb_check_var   = uc1701fb_check_var,
    .fb_set_par     = uc1701fb_set_par,
    .fb_read        = uc1701fb_read,
    .fb_write       = uc1701fb_write,
    .fb_fillrect    = cfb_fillrect,     /* Needed !!! */
    .fb_copyarea    = cfb_copyarea, /* Needed !!! */
    .fb_imageblit   = cfb_imageblit,    /* Needed !!! */
    .fb_ioctl       = uc1701fb_ioctl,
    .fb_mmap        = uc1701fb_mmap,
};

//===========================================================================
static void fb_clear(
    void
)
//===========================================================================
{
    memset( dev_data->fb_buffer, 0, ( SCREEN_SIZE_X * SCREEN_SIZE_Y * SCREEN_BPP ) / 8 );
}

#if defined( CONFIG_HW_MODEL_GXP1400 ) || defined( CONFIG_HW_MODEL_GXP1150 )
//===========================================================================
static void icon_clear(
    void
)
//===========================================================================
{
    u8 cmd_buf[3];
    u8 dat_buf = 0x00;
    int i;
    struct gs_lcd_platform_data *pdata = (struct gs_lcd_platform_data *)dev_data->spi->dev.platform_data;
    unsigned int cd_gpio = pdata->lcd_cd;

    down( &sem_lcd );
    cmd_buf[0] = 0x04;
    cmd_buf[1] = 0x10;
    cmd_buf[2] = 0xB8;
    for ( i = 0; i < 4; i++ )
    {
        LCD_SET_CMD(cd_gpio);
        spi_write( dev_data->spi, cmd_buf, 3 );
        LCD_SET_DAT(cd_gpio);
        spi_write( dev_data->spi, &dat_buf, 1 );
        cmd_buf[0]++;
    }
    cmd_buf[0] = 0x0F;
    cmd_buf[1] = 0x17;
    LCD_SET_CMD(cd_gpio);
    spi_write( dev_data->spi, cmd_buf, 3 );
    LCD_SET_DAT(cd_gpio);
    spi_write( dev_data->spi, &dat_buf, 1 );
    cmd_buf[0] = 0x00;
    cmd_buf[1] = 0x18;
    for ( i = 0; i < 4; i++ )
    {
        LCD_SET_CMD(cd_gpio);
        spi_write( dev_data->spi, cmd_buf, 3 );
        LCD_SET_DAT(cd_gpio);
        spi_write( dev_data->spi, &dat_buf, 1 );
        cmd_buf[0]++;
    }

    up( &sem_lcd );
}
#endif

//===========================================================================
static void fb_sync(
    void
)
//===========================================================================
{
    uc1701_update_screen( dev_data, 0, 0, SCREEN_SIZE_X, SCREEN_SIZE_Y );
}

//===========================================================================
void fb_sync_region(
    int x, int y, int width, int height
)
//===========================================================================
{
    uc1701_update_screen( dev_data, x, y, width, height );
}

//===========================================================================
void gs_lcd_set_contrast(
    int percent
)
//===========================================================================
{
    unsigned short pm_val;
    u8 cmd_buf[2];
    struct gs_lcd_platform_data *pdata = (struct gs_lcd_platform_data *)dev_data->spi->dev.platform_data;
    unsigned int cd_gpio = pdata->lcd_cd;

#define PM_VAL_MIN 0
 //   pm_val = PM_VAL_MIN + ( ( percent / 10 ) *  6);
    if(percent > 99 )percent = 99;
    if(percent < 0 )percent = 0;
    pm_val = PM_VAL_MIN + percent * 16 / 25;
    
    down( &sem_lcd );
    LCD_SET_CMD(cd_gpio);
    cmd_buf[0] = 0x81;
    cmd_buf[1] = pm_val;
    spi_write( dev_data->spi, cmd_buf, 2 );
    up( &sem_lcd );
}

//===========================================================================
static int
uc1701_proc_set_contrast(
    struct file * file,
    const char * buffer,
    unsigned long count,
    void * data
)
//===========================================================================
{
    char *string_buff;
    unsigned long cpersent;
    
    string_buff = kzalloc (count + 1, GFP_KERNEL);
    if(!string_buff)
         return -ENOMEM;
    
    if ( copy_from_user( string_buff, buffer, count ) ) {
          return -EFAULT;
    }
    string_buff[count] = '\0';
    
    cpersent = simple_strtoul(string_buff, NULL, 10);
    kfree(string_buff);
    if(cpersent >= 100)cpersent = 100;
    contrast_now = cpersent;
    gs_lcd_set_contrast(cpersent);

    return count;
}

//===========================================================================
static int uc1701_proc_get_contrast (
    char *page, char **start,
    off_t off, int count, 
    int *eof, void *data
)
//===========================================================================
{
    int len = 0; 
    
    len = sprintf(page, "%d\n", contrast_now);

    if (len <= off+count) 
        *eof = 1; 
    *start = page + off; 
    len -= off; 
    if (len>count) 
        len = count;
    if (len<0) 
        len = 0; 

    return (len);
}

//===========================================================================
static int
uc1701_proc_icon(
    struct file * file,
    const char * buffer,
    unsigned long count,
    void * data
)
//===========================================================================
{
    u8 cmd_buf[3];
    u8 cmd;
    struct gs_lcd_platform_data *pdata = (struct gs_lcd_platform_data *)dev_data->spi->dev.platform_data;
    unsigned int cd_gpio = pdata->lcd_cd;

    if ( copy_from_user( &cmd, buffer, 1 ) )
    {
        return -EFAULT;
    }

    switch( cmd )
    {
        case LCD_ICON_NETWORK:
            cmd_buf[0] = 0x07;
            cmd_buf[1] = 0x10;
            break;
        case LCD_ICON_NETWORK_BORDER:
            cmd_buf[0] = 0x08;
            cmd_buf[1] = 0x10;
            break;
        case LCD_ICON_HANDSET:
            cmd_buf[0] = 0x06;
            cmd_buf[1] = 0x10;
            break;
        case LCD_ICON_SPKR:
            cmd_buf[0] = 0x05;
            cmd_buf[1] = 0x10;
            break;
        case LCD_ICON_HEADSET:
            cmd_buf[0] = 0x04;
            cmd_buf[1] = 0x10;
            break;
        case LCD_ICON_MUTE:
            cmd_buf[0] = 0x00;
            cmd_buf[1] = 0x18;
            break;
        case LCD_ICON_CALLFWD:
            cmd_buf[0] = 0x01;
            cmd_buf[1] = 0x18;
            break;
        case LCD_ICON_DND:
            cmd_buf[0] = 0x02;
            cmd_buf[1] = 0x18;
            break;
        case LCD_ICON_LOCK:
            cmd_buf[0] = 0x03;
            cmd_buf[1] = 0x18;
            break;
        default:
            return -EFAULT;
    }
    cmd_buf[2] = 0xB8;

    down( &sem_lcd );
    LCD_SET_CMD(cd_gpio);
    spi_write( dev_data->spi, cmd_buf, 3 );
    LCD_SET_DAT(cd_gpio);
    cmd_buf[0] = *(unsigned char *)data;
    spi_write( dev_data->spi, cmd_buf, 1 );
    up( &sem_lcd );

    return count;
}

//===========================================================================
static int
uc1701_proc_reset(
    struct file * file,
    const char * buffer,
    unsigned long count,
    void * data
)
//===========================================================================
{
    int i;
    struct gs_lcd_platform_data *pdata = (struct gs_lcd_platform_data *)dev_data->spi->dev.platform_data;
    unsigned int rst_gpio = pdata->reset;
    
    gpio_set_value( rst_gpio, 0 );
    udelay( 1000 );
    gpio_set_value( rst_gpio, 1 );
    for ( i = 0; i < 6; i++ )
    {
        udelay( 1000 );
    }

    return count;
}

//===========================================================================
static int
uc1701_procfs_write_data(
    struct file * file,
    const char * buffer,
    unsigned long count,
    void * data
)
//===========================================================================
{
    char cmd_buf[32];
    struct gs_lcd_platform_data *pdata = (struct gs_lcd_platform_data *)dev_data->spi->dev.platform_data;
    unsigned int cd_gpio = pdata->lcd_cd;

    if ( count > sizeof ( cmd_buf ) )
    {
        return -ENOSPC;
    }

    down( &sem_lcd );
    LCD_SET_DAT(cd_gpio);
    if ( copy_from_user( cmd_buf, buffer, count ) )
    {
        up( &sem_lcd );
        return -EFAULT;
    }
    if ( spi_write( dev_data->spi, cmd_buf, count ) < 0 )
    {
        up( &sem_lcd );
        return -EIO;
    }
    up( &sem_lcd );
    return count;
}

//===========================================================================
static int
uc1701_procfs_write(
    struct file * file,
    const char * buffer,
    unsigned long count,
    void * data
)
//===========================================================================
{
    char cmd_buf[32];
    struct gs_lcd_platform_data *pdata = (struct gs_lcd_platform_data *)dev_data->spi->dev.platform_data;
    unsigned int cd_gpio = pdata->lcd_cd;

    if ( count > sizeof ( cmd_buf ) )
    {
        return -ENOSPC;
    }

    down( &sem_lcd );
    LCD_SET_CMD(cd_gpio);
    if ( copy_from_user( cmd_buf, buffer, count ) )
    {
        up( &sem_lcd );
        return -EFAULT;
    }
    if ( spi_write( dev_data->spi, cmd_buf, count ) < 0 )
    {
        up( &sem_lcd );
        return -EIO;
    }
    up( &sem_lcd );
    return count;
}

//===========================================================================
static void panel_setup(
    void
)
//===========================================================================
{
    u8 cmd_buf[5];
    struct gs_lcd_platform_data *pdata = (struct gs_lcd_platform_data *)dev_data->spi->dev.platform_data;
    unsigned int cd_gpio = pdata->lcd_cd;

    down( &sem_lcd );

    LCD_SET_CMD(cd_gpio);
    /* Set MX */
    cmd_buf[0] = 0xA1;
    //cmd_buf[0] = 0xA0;  /* horizontal reverse */
    spi_write( dev_data->spi, cmd_buf, 1 );

    /* Set MY */
    cmd_buf[0] = 0xC0;
    //cmd_buf[0] = 0xC8; /* vertical reverse */
    spi_write( dev_data->spi, cmd_buf, 1 );

    /* Set Vlcd */
#if NEW_LCD
    cmd_buf[0] = 0x23;
#else
    cmd_buf[0] = 0x27;
#endif
    spi_write( dev_data->spi, cmd_buf, 1 );

    /* Set BR */
    cmd_buf[0] = 0xA2;
    spi_write( dev_data->spi, cmd_buf, 1 );

    /* Set PM */
    cmd_buf[0] = 0x81;
#if NEW_LCD
    cmd_buf[1] = 0x8;
#else
    cmd_buf[1] = 0x10;
#endif
    spi_write( dev_data->spi, cmd_buf, 2 );

    /* Set AFH */
    cmd_buf[0] = 0xAF;
    spi_write( dev_data->spi, cmd_buf, 1 );

    /* Set PC */
    cmd_buf[0] = 0x2F;
    spi_write( dev_data->spi, cmd_buf, 1 );

#if NEW_LCD
    cmd_buf[0] = 0x48;
    spi_write( dev_data->spi, cmd_buf, 1 );
#endif

    up( &sem_lcd );
}

/* ------------------------------------------------------------------------- */

    /*
     *  Initialization
     */

static void alloc_framebuffer( struct fb_info *info )
{
    struct fb_var_screeninfo *var = &info->var;
    struct uc1701fb_par* par = info->par;
    unsigned int fb_size;

    fb_size = (var->xres * var->yres * var->bits_per_pixel ) / 8;

    par->fb_size = PAGE_ALIGN( fb_size );
    par->fb_buffer = dma_alloc_coherent(
                        NULL,
                        par->fb_size,
                        &par->dma_handle,
                        GFP_KERNEL );
    info->fix.smem_start = (uint32_t)par->fb_buffer;
    info->fix.smem_len = par->fb_size;

    DEBUG(".fb_size = %d\n", par->fb_size);
    DEBUG(".fb_buffer = %08x\n", (uint32_t)par->fb_buffer);

    info->screen_base = par->fb_buffer;
}

static void free_framebuffer(struct fb_info *info)
{
    struct uc1701fb_par* par = info->par;
    
    dma_free_coherent(NULL, par->fb_size, par->fb_buffer,
		       par->dma_handle);
}

static int uc1701fb_probe( struct spi_device * dev )
{
    struct fb_info *info = NULL;
    struct uc1701fb_par* par;
    struct proc_dir_entry * procfs_root;
    struct proc_dir_entry * procfs_entry;
    struct gs_lcd_platform_data *pdata = (struct gs_lcd_platform_data *)(dev->dev.platform_data);
    int ret;

    DEBUG("uc1701FB_PROBE\n");

    /*
     * Dynamically allocate info and par
     */
    if ( dev_data == NULL )
    {
        info = framebuffer_alloc(sizeof(struct uc1701fb_par), &dev->dev);

        if (!info) {
            /* goto error path */
            return -EIO;
        }

        dev_data = par = info->par;
        memset( par, 0, sizeof( struct uc1701fb_par ) );

        par->spi = dev;

        /*
         * Here we set the screen_base to the virtual memory address
         * for the framebuffer. Usually we obtain the resource address
         * from the bus layer and then translate it to virtual memory
         * space via ioremap. Consult ioport.h.
         */
        info->screen_base = NULL;
        info->fbops = &uc1701fb_ops;
        info->fix = uc1701fb__devinitdata;  /* this will be the only time xxxfb_fix will be
                                     * used, so mark it as __devinitdata
                                     */
        info->pseudo_palette = NULL;/* The pseudopalette is an
                                     * 16-member array.  SCF - truecolor visual
                                     * has no pseudopallet.
                                     */
        /*
         * Set up flags to indicate what sort of acceleration your
         * driver can provide (pan/wrap/copyarea/etc.) and whether it
         * is a module -- see FBINFO_* in include/linux/fb.h
         *
         * If your hardware can support any of the hardware accelerated functions
         * fbcon performance will improve if info->flags is set properly.
         *
         * FBINFO_HWACCEL_COPYAREA - hardware moves
         * FBINFO_HWACCEL_FILLRECT - hardware fills
         * FBINFO_HWACCEL_IMAGEBLIT - hardware mono->color expansion
         * FBINFO_HWACCEL_YPAN - hardware can pan display in y-axis
         * FBINFO_HWACCEL_YWRAP - hardware can wrap display in y-axis
         * FBINFO_HWACCEL_DISABLED - supports hardware accels, but disabled
         * FBINFO_READS_FAST - if set, prefer moves over mono->color expansion
         * FBINFO_MISC_TILEBLITTING - hardware can do tile blits
         *
         * NOTE: These are for fbcon use only.
         */
        info->flags = FBINFO_DEFAULT;

        /*
         * The following is done in the case of having hardware with a static
         * mode. If we are setting the mode ourselves we don't call this.
         */
        info->var = uc1701fb_var_default;

        alloc_framebuffer( info );
        if (register_framebuffer(info) < 0) {
            ret = -EINVAL;
            goto error_uc1701;
        }
        printk(KERN_INFO "fb%d: %s frame buffer device\n", info->node,
           info->fix.id);

    }

    ret = gpio_request( pdata->reset, "uc1701fb" );
    if (ret < 0) {
        dev_err(&dev->dev, "failed requesting lcd reset gpio\n");
        goto error_uc1701;
    }
#if 0  //we have reset in uboot
    int i;
    gpio_direction_output( pdata->reset, 0 );
    udelay( 1000 );
    gpio_set_value( pdata->reset, 1 );
    for ( i = 0; i < 6; i++ )
    {
        udelay( 1000 );
    }

    ret = gpio_request( pdata->lcd_cd, "uc1701fb" );
    if (ret < 0) {
        dev_err(&dev->dev, "failed requesting lcd CD gpio\n");
        goto error_uc1701;
    }
    gpio_direction_output( pdata->lcd_cd, 1 );
	
	printk(KERN_INFO "reset uc1701\n");
#endif
    fb_clear();
    panel_setup();
#if defined( CONFIG_HW_MODEL_GXP1400 ) || defined( CONFIG_HW_MODEL_GXP1150 )
    icon_clear();
#endif
//  fb_sync(); //we show logo in uboot and we dont display bank on here. make display more coherent.

    kgui_init( dev_data->fb_buffer, SCREEN_SIZE_X, SCREEN_SIZE_Y, SCREEN_SIZE_X, SCREEN_SIZE_Y, 0, 1 );
#if defined( CONFIG_FB_LOGO )
	printk(KERN_INFO "start display kgui_logo\n");
    kgui_logo();
    fb_sync();
#else
    kgui_throbber( LOCATION_MID );
#endif
    gs_lcd_set_contrast(contrast_now);

    procfs_root = proc_mkdir( "uc1701", NULL );

    procfs_entry = create_proc_entry( "reg", S_IFREG | S_IRUGO, procfs_root );
    procfs_entry->write_proc = uc1701_procfs_write;
    procfs_entry = create_proc_entry( "dat", S_IFREG | S_IRUGO, procfs_root );
    procfs_entry->write_proc = uc1701_procfs_write_data;

    procfs_entry = create_proc_entry( "reset", S_IFREG | S_IRUGO, procfs_root );
    procfs_entry->write_proc = uc1701_proc_reset;

    procfs_entry = create_proc_entry( "contrast", S_IFREG | S_IRUGO, procfs_root );
    procfs_entry->write_proc = uc1701_proc_set_contrast;
    procfs_entry->read_proc = uc1701_proc_get_contrast;

    procfs_entry = create_proc_entry( "icon_set", S_IFREG | S_IRUGO, procfs_root );
    procfs_entry->write_proc = uc1701_proc_icon;
    procfs_entry->data = (void*)0x01;
    procfs_entry = create_proc_entry( "icon_clr", S_IFREG | S_IRUGO, procfs_root );
    procfs_entry->write_proc = uc1701_proc_icon;
    procfs_entry->data = (void*)0x00;

    return 0;
error_uc1701:
    free_framebuffer(info);
    framebuffer_release(info);
    return ret;
}

    /*
     *  Cleanup
     */
static int uc1701fb_remove( struct spi_device * dev )
{
    struct fb_info *info = dev_get_drvdata( &dev->dev );
    struct uc1701fb_par* par;

    if (info) {
        par = info->par;

        if ( par->fb_buffer )
        {
            dma_free_coherent(
                NULL,
                par->fb_size,
                par->fb_buffer,
                par->dma_handle
            );
        }

        unregister_framebuffer(info);
        framebuffer_release(info);
    }

    return 0;
}

static struct spi_driver uc1701fb_driver = {
    .driver = {
        .name = "uc1701fb",
        .bus = &spi_bus_type,
        .owner = THIS_MODULE,
    },
    .probe = uc1701fb_probe,
    .remove = __devexit_p(uc1701fb_remove),
};

static int __init uc1701fb_init(void)
{
    int ret;

    dev_data = NULL;

    DEBUG("uc1701FB_INIT\n");

    ret = spi_register_driver( &uc1701fb_driver );

    DEBUG("uc1701FB_INIT: %08x\n", ret);

    return ret;
}

static void __exit uc1701fb_exit(void)
{
    /* Should be free by now */
    dev_data = NULL;

    spi_unregister_driver( &uc1701fb_driver );
}

/* ------------------------------------------------------------------------- */


    /*
     *  Modularization
     */

module_init(uc1701fb_init);
module_exit(uc1701fb_exit);

EXPORT_SYMBOL(gs_lcd_set_contrast);
EXPORT_SYMBOL(fb_sync_region);
MODULE_LICENSE("GPL");

