/*
 * TODO:
 * 	- dynamically setup var and fix on probe
 * 	- switch backlight on/off
 */
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/mm.h>
#include <linux/fb.h>
#include <linux/init.h>
#include <linux/dma-mapping.h>
#include <linux/interrupt.h>
#include <linux/clk.h>
#include <linux/semaphore.h>
#include <linux/delay.h>
#include <asm/uaccess.h> /*needed for copy_form_user*/

#ifndef CONFIG_MACH_VERSATILE_BROADTILE
#include <asm/gpio.h>
#endif

#include <video/dw74fb.h>
#include <video/dw_bl.h>

MODULE_DESCRIPTION("DW74 LCD Controller framebuffer driver");
MODULE_LICENSE("GPL");

static unsigned underruns = 0;
module_param(underruns, uint, 0644);
MODULE_PARM_DESC(underruns, "Number of underruns that happened.");

static unsigned unhandled_ints = 0;
module_param(unhandled_ints, uint, 0644);
MODULE_PARM_DESC(unhandled_ints, "Number of unhandled interrupts.");

#define DW74FB_NAME "dw74fb"
static char dw74fb_name[] = DW74FB_NAME;

#ifdef CONFIG_FB_DW74_LOW_POWER
#define DW74FB_NAME_LOW_POWER "dw74fb_low_power"
#endif

static struct dw74fb_panel *panel = NULL;
static struct dw74fb *g_dw74fb = NULL;

static int lcdc_mode = DWFB_MODE_LCD;

const struct fb_videomode dw74fb_modedb[] = {
	{ "480p60a",  60,  720,  480, 37037,  58,  17, 30, 9, 63, 6, // 27.00 MHz
	  FB_SYNC_EXT, FB_VMODE_NONINTERLACED, FB_MODE_IS_STANDARD },
	{ "720p50a",  50, 1280,  720, 13468, 220, 400, 19, 6, 80, 5, // 74.25 MHz
	  FB_SYNC_EXT, FB_VMODE_NONINTERLACED, FB_MODE_IS_STANDARD },
	{ "720p60a",  60, 1280,  720, 13468, 220,  70, 19, 6, 80, 5, // 74.25 MHz
	  FB_SYNC_EXT, FB_VMODE_NONINTERLACED, FB_MODE_IS_STANDARD },
	{ "1080p24a",  24, 1920, 1080, 13468, 638, 148, 36, 4, 44, 5, // 74.25 MHz
	  FB_SYNC_EXT, FB_VMODE_NONINTERLACED, FB_MODE_IS_STANDARD },
	{ "1080p25a",  25, 1920, 1080, 13468, 528, 148, 36, 4, 44, 5, // 74.25 MHz
	  FB_SYNC_EXT, FB_VMODE_NONINTERLACED, FB_MODE_IS_STANDARD },
	{ "1080p30a",  30, 1920, 1080, 13468,  88, 148, 36, 4, 44, 5, // 74.25 MHz
	  FB_SYNC_EXT, FB_VMODE_NONINTERLACED, FB_MODE_IS_STANDARD },
	{ "480p60",   60,  720,  480, 37037,  58,  17, 30, 9, 63, 6, // 27.00 MHz
	  FB_SYNC_EXT, FB_VMODE_NONINTERLACED, FB_MODE_IS_STANDARD },
	{ "720p50",   50, 1280,  720, 13468, 220, 400, 19, 6, 80, 5, // 74.25 MHz
	  FB_SYNC_EXT, FB_VMODE_NONINTERLACED, FB_MODE_IS_STANDARD },
	{ "720p60",   60, 1280,  720, 13468, 220,  70, 19, 6, 80, 5, // 74.25 MHz
	  FB_SYNC_EXT, FB_VMODE_NONINTERLACED, FB_MODE_IS_STANDARD },
	{ "1080p24",  24, 1920, 1080, 13468, 638, 148, 36, 4, 44, 5, // 74.25 MHz
	  FB_SYNC_EXT, FB_VMODE_NONINTERLACED, FB_MODE_IS_STANDARD },
	{ "1080p25",  25, 1920, 1080, 13468, 528, 148, 36, 4, 44, 5, // 74.25 MHz
	  FB_SYNC_EXT, FB_VMODE_NONINTERLACED, FB_MODE_IS_STANDARD },
	{ "1080p30",  30, 1920, 1080, 13468,  88, 148, 36, 4, 44, 5, // 74.25 MHz
	  FB_SYNC_EXT, FB_VMODE_NONINTERLACED, FB_MODE_IS_STANDARD },
};

#define ANDROID_XRES 800
#define ANDROID_YRES 480

static int bpp_to_info(struct fb_var_screeninfo *var, int bits_per_pixel)
{
	int bpp = -1;

	if (bits_per_pixel == 16) {
		var->bits_per_pixel = 16;
		var->red.offset = 11;
		var->red.length = 5;
		var->green.offset = 5;
		var->green.length = 6;
		var->blue.offset = 0;
		var->blue.length = 5;
		bpp = 0;
	} else if ((bits_per_pixel == 18) || (bits_per_pixel == 24)) {
		var->bits_per_pixel = 24;
		var->red.offset = 16;
		var->red.length = 8;
		var->green.offset = 8;
		var->green.length = 8;
		var->blue.offset = 0;
		var->blue.length = 8;
		bpp = 1;
	}

	return bpp;
}

static int set_display_bpp(struct dw74fb *dw74fb, struct fb_info *info)
{
	unsigned int bpp = 0;

	if (info->var.bits_per_pixel == 16) {
		bpp = 0x0;
		/* This specify that this is a video*/
		if (lcdc_mode == DWFB_MODE_VIDEO_PAL || lcdc_mode == DWFB_MODE_VIDEO_NTSC)
			bpp = 0x2;
	} else if (info->var.bits_per_pixel == 24) {
		bpp = 0x1;
	} else if (info->var.bits_per_pixel == 2) {
		bpp = 0x3;
	} else {
		/* TODO: find a way to define YUV422 */
		printk(KERN_ERR "Fail to set bit per pixel value %d is not valid\n", info->var.bits_per_pixel);
		return -1;
	}

	dw74fb_writel(dw74fb,LCDC_REG_INTMR, bpp);

	/* input data transfer sizes */
	dw74fb_writel(dw74fb, LCDC_REG_INDTR, (( info->var.xres * info->var.bits_per_pixel ) / 32 ) - 1);

	dw74fb_writel(dw74fb, LCDC_REG_OFFAR0, info->var.xres * info->var.bits_per_pixel / 8);
	dw74fb_writel(dw74fb, LCDC_REG_OFFAR1, info->var.xres * info->var.bits_per_pixel / 8);

	return 0;
}

static int
dw74fb_set_videomode(struct dw74fb *dw74fb, int mode)
{
	struct fb_info *info = dw74fb->dwfb_info.info;
	struct fb_var_screeninfo var = info->var;
	const struct fb_videomode *hdmi_mode = NULL;
	int ret, xres, yres;
	unsigned int addr;
	int bpp = 1;
	int xpos = 0, ypos = 0, xend = ANDROID_XRES, yend = ANDROID_YRES;
	int android = 0;
	int dispir = 0x0001;
	int pancsr = 0x0000;
	unsigned long clk_rate = 0;

	if ((mode < DWFB_MODE_LCD) || (mode > DWFB_MODE_1080P30))
		return -1;

	if (mode == DWFB_MODE_LCD) {
		/* restore original settings */
		hdmi_mode = &panel->videomode;
		dispir    =  panel->dispir;
		pancsr    =  panel->pancsr;
		clk_rate  =  panel->clock_rate;
		bpp = bpp_to_info(&info->var, panel->bits_per_pixel);
	} else {
		hdmi_mode = &dw74fb_modedb[mode - DWFB_MODE_480P60A];
		xres = hdmi_mode->hsync_len + hdmi_mode->left_margin +
		       hdmi_mode->xres + hdmi_mode->right_margin;
		yres = hdmi_mode->vsync_len + hdmi_mode->upper_margin +
		       hdmi_mode->yres + hdmi_mode->lower_margin;
		clk_rate = xres * yres * hdmi_mode->refresh;
		bpp = bpp_to_info(&info->var, 24);
	}

	if ((mode != DWFB_MODE_LCD) && (hdmi_mode->name[strlen(hdmi_mode->name)-1] == 'a'))
		android = 1;

	dw74fb_writel(dw74fb, LCDC_REG_LCDCCR, 0);
	dw74fb_writel(dw74fb, LCDC_REG_DISPCR, 0);

	fb_videomode_to_var(&var, hdmi_mode);
	if (!android) {
		/* hdmi_mode describes the actual image we are displaying */
#ifdef CONFIG_FB_DW74_DOUBLEBUF
		var.yres_virtual = hdmi_mode->yres*2;
#endif
		info->fix.line_length = hdmi_mode->xres * (info->var.bits_per_pixel/8);
		info->screen_size = info->fix.line_length * var.yres_virtual;
		info->fix.smem_len = info->screen_size;

		var.activate = FB_ACTIVATE_FORCE | FB_ACTIVATE_NOW;
		ret = fb_set_var(info, &var);
		if (ret < 0) {
			printk("set_videomode failed %d\n", ret);
			return -1;
		}
	}
	clk_rate = clk_round_rate(dw74fb->clk, clk_rate);
	clk_set_rate(dw74fb->clk, clk_rate);

	dw74fb_writel(dw74fb, LCDC_REG_DISPIR, dispir);
	dw74fb_writel(dw74fb, LCDC_REG_PANCSR, pancsr);

	dw74fb_writel(dw74fb, LCDC_REG_INTMR,  bpp);

	dw74fb_writel(dw74fb, LCDC_REG_VSTR,   hdmi_mode->vsync_len - 1);
	dw74fb_writel(dw74fb, LCDC_REG_VFTR,   hdmi_mode->upper_margin - 1);
	dw74fb_writel(dw74fb, LCDC_REG_VATR,   hdmi_mode->yres - 1);
	dw74fb_writel(dw74fb, LCDC_REG_VETR,   hdmi_mode->lower_margin - 1);

	dw74fb_writel(dw74fb, LCDC_REG_HSTR,   hdmi_mode->hsync_len - 1);
	dw74fb_writel(dw74fb, LCDC_REG_HFTR,   hdmi_mode->left_margin - 1);
	dw74fb_writel(dw74fb, LCDC_REG_HADSTR, hdmi_mode->xres - 1);
	dw74fb_writel(dw74fb, LCDC_REG_HAPWR,  hdmi_mode->xres - 1);
	dw74fb_writel(dw74fb, LCDC_REG_HETR,   hdmi_mode->right_margin - 1);

	if (!android)
		dw74fb_writel(dw74fb, LCDC_REG_INDTR, ((hdmi_mode->xres * info->var.bits_per_pixel) / 32 ) - 1);

	dw74fb_writel(dw74fb, LCDC_REG_INDXSR, hdmi_mode->xres - 1);
	dw74fb_writel(dw74fb, LCDC_REG_INDYSR, hdmi_mode->yres - 1);

	/* display position */
	if (android && (hdmi_mode->xres > ANDROID_XRES))
		xpos = (hdmi_mode->xres - ANDROID_XRES)/2;
	if (android && (hdmi_mode->yres > ANDROID_YRES))
		ypos = (hdmi_mode->yres - ANDROID_YRES)/2;
	if (!android)
		xend = hdmi_mode->xres;
	else if (hdmi_mode->xres > ANDROID_XRES)
		xend = hdmi_mode->xres - xpos;
	if (!android)
		yend = hdmi_mode->yres;
	else if (hdmi_mode->yres > ANDROID_YRES)
		yend = hdmi_mode->yres - ypos;
	dw74fb_writel(dw74fb, LCDC_REG_DISPXSPOSR,  xpos);
	dw74fb_writel(dw74fb, LCDC_REG_DISPXEPOSR,  xend - 1);
	dw74fb_writel(dw74fb, LCDC_REG_DISPYSPOS1R, ypos);
	dw74fb_writel(dw74fb, LCDC_REG_DISPYEPOS1R, yend - 1);

	if (!android) {
		/* input buffer */
		dw74fb_writel(dw74fb, LCDC_REG_OFFAR0, info->fix.line_length);
#ifdef CONFIG_FB_DW74_DOUBLEBUF
		dw74fb_writel(dw74fb, LCDC_REG_OFFAR1, info->fix.line_length);
#endif
	}

	addr = info->fix.smem_start + (info->screen_size / 2);
	dw74fb_writel(dw74fb, LCDC_REG_MSBAHBA1R, addr >> 16);
	dw74fb_writel(dw74fb, LCDC_REG_LSBAHBA1R, addr & 0xffff);

	dw74fb_writel(dw74fb, LCDC_REG_PARUP,  0x0001);
	dw74fb_writel(dw74fb, LCDC_REG_INTR,   0x0001);
	dw74fb_writel(dw74fb, LCDC_REG_DISPCR, 0x0001);
	dw74fb_writel(dw74fb, LCDC_REG_LCDCCR, 0x0001);
	
	return 0;
}

#ifdef CONFIG_BACKLIGHT_DMW_AMOLED
static struct platform_device amoled_backlight_device = {
	.name = "dmw-amoled-bl",
	.num_resources = 0,
	.resource = NULL,
	.dev = {
		.platform_data = NULL,
	},
};
#endif

/*
 * hardware specifics
 */
unsigned long
dw74fb_readl(struct dw74fb *dw74fb, unsigned long addr)
{
	return readl(dw74fb->regs + addr);
}

void
dw74fb_writel(struct dw74fb *dw74fb, unsigned long addr, unsigned long val)
{
	writel(val, dw74fb->regs + addr);
}

static void dw74fb_hw_enable(struct dw74fb *dw74fb)
{
	dw74fb_writel(dw74fb, LCDC_REG_LCDCCR, 1);
}

static void dw74fb_hw_disable(struct dw74fb *dw74fb)
{
	dw74fb_writel(dw74fb, LCDC_REG_LCDCCR, 0);
}

static void
dw74fb_run_sequence(struct dw74fb *dw74fb, struct dw74fb_seq *seq)
{
	struct dw74fb_seq *cmd = seq;

	while (cmd->addr != LCDC_SEQ_END) {
		dw74fb_writel(dw74fb, cmd->addr, cmd->value);
		cmd++;
	}
}

void lcdc_own_hw(void){

	if (g_dw74fb != NULL){
		if (down_interruptible(&(g_dw74fb->lcdc_hw_busy)) != 0){
				printk(KERN_ERR "ili9325 LCD operation since it was signaled while waiting for frame to be finished\n");
		}
	}
}

void lcdc_release_hw(void)
{
	if (g_dw74fb != NULL)
		up(&g_dw74fb->lcdc_hw_busy);
}

#ifdef FB_DW74_PANEL_ILI9325
static struct platform_device dw_ili9325_device = {
	.name		= "dw-ili9325",
	.dev		= {
		.platform_data     = &g_dw74fb,
	},
};
#endif


static struct dw74fb_seq dw74fb_hw_init_seq[] = {
#ifdef CONFIG_FB_DW74_CPUTYPE
	{LCDC_REG_INTER,  0x30},
#else
	{LCDC_REG_INTER,  0x1c},  /* enable underrun & fifo empty irqs */
#endif
	{LCDC_REG_GCER,   0},     /* disable gamma correction */
	{LCDC_REG_CLPER,  0},     /* disable clipping */

	{LCDC_REG_BACKCPR, 0},    /* black background */

	{LCDC_REG_DISPYSPOS2R, 0}, /* only used for CCIR656 */
	{LCDC_REG_DISPYEPOS2R, 0}, /* only used for CCIR656 */

	{LCDC_REG_YCLPCR, 0xfe01},/* default clipping control for Y when use CCIR */
	{LCDC_REG_CCLPCR, 0xfe01},/* default clipping control for Cb and Cr when use CCIR */

#ifdef CONFIG_FB_DW74_CPUTYPE
	{LCDC_REG_LCDCCR, 0x02}, /* FIF0 Reset */
	{LCDC_REG_OSDMCR, 0x00}, /* Disable OSDM */
	{LCDC_REG_DISPIR, 0x24}, /* Set CPU Type, 80 Type, 16 bit RGB */
	{LCDC_REG_CTLTR0, 0x1000}, /*Set RS Polarity and Data Format, Note: RS is active LOW for ili9325 */
	{LCDC_REG_CMDFSR, 0x06}, /* Command Fifo Size */
#endif

	{LCDC_REG_DISPCR, 1}, /* enable display */
	{LCDC_REG_PARUP,  1}, /* update parameters on next frame */
	{LCDC_SEQ_END,    0},
};

static void set_display_location(struct dw74fb *dw74fb, struct dwfb_location* location)
{
	/* display position */
	dw74fb_writel(dw74fb, LCDC_REG_DISPXSPOSR,  location->x_start);
	dw74fb_writel(dw74fb, LCDC_REG_DISPXEPOSR,  location->x_end - 1);

	if (lcdc_mode == DWFB_MODE_VIDEO_PAL){
		dw74fb_writel(dw74fb, LCDC_REG_DISPYSPOS1R, 23 + (location->y_start / 2));
		dw74fb_writel(dw74fb, LCDC_REG_DISPYSPOS2R, 336 + location->y_start / 2 );
	} else if (lcdc_mode == DWFB_MODE_VIDEO_NTSC){
		dw74fb_writel(dw74fb, LCDC_REG_DISPYSPOS1R, 20 + (location->y_start / 2));
		dw74fb_writel(dw74fb, LCDC_REG_DISPYSPOS2R, 283 + location->y_start / 2 );
	} else  {
		dw74fb_writel(dw74fb, LCDC_REG_DISPYSPOS1R, location->y_start);
	}

	if (lcdc_mode == DWFB_MODE_VIDEO_PAL){
		dw74fb_writel(dw74fb, LCDC_REG_DISPYEPOS1R, 23 + location->y_end / 2  - 1);
		dw74fb_writel(dw74fb, LCDC_REG_DISPYEPOS2R, 336 + location->y_end / 2  - 1);
	}
	else if (lcdc_mode == DWFB_MODE_VIDEO_NTSC) {
		dw74fb_writel(dw74fb, LCDC_REG_DISPYEPOS1R, 20 + location->y_end / 2  - 1);
		dw74fb_writel(dw74fb, LCDC_REG_DISPYEPOS2R, 283 + location->y_end / 2  - 1);
	}
	else{
		dw74fb_writel(dw74fb, LCDC_REG_DISPYEPOS1R, location->y_end - 1);
	}
}

static void set_display_background(struct dw74fb *dw74fb, unsigned short background_color)
{
    printk("display========================background\n");
	dw74fb_writel(dw74fb,LCDC_REG_BACKCPR, background_color);
}

static void display_palette_set(struct dw74fb *dw74fb, const struct fb_cmap* pal){

	dw74fb_writel(dw74fb,LCDC_REG_PALR0 , get_rgb565(pal->red[0],pal->green[0], pal->blue[0]) );
	dw74fb_writel(dw74fb,LCDC_REG_PALR1 , get_rgb565(pal->red[1],pal->green[1], pal->blue[1]) );
	dw74fb_writel(dw74fb,LCDC_REG_PALR2 , get_rgb565(pal->red[2],pal->green[2], pal->blue[2]) );
	dw74fb_writel(dw74fb,LCDC_REG_PALR3 , get_rgb565(pal->red[3],pal->green[3], pal->blue[3]) );
}

#ifdef CONFIG_FB_DMW96_CURSOR
static void cursor_palette_set(struct dw74fb *dw74fb, const struct fb_cmap* pal){

	dw74fb_writel(dw74fb,LCDC_REG_CURPALR1 , get_rgb565(pal->red[0],pal->green[0], pal->blue[0]) );
	dw74fb_writel(dw74fb,LCDC_REG_CURPALR2 , get_rgb565(pal->red[1],pal->green[1], pal->blue[1]) );
	dw74fb_writel(dw74fb,LCDC_REG_CURPALR3 , get_rgb565(pal->red[2],pal->green[2], pal->blue[2]) );

	/*
	printk (KERN_ERR "%s(%d) 0x%x 0x%x 0x%x \n",__func__, __LINE__, pal->red[0] ,pal->green[0], pal->blue[0]);
	printk (KERN_ERR "%s(%d) 0x%x 0x%x 0x%x \n",__func__, __LINE__, pal->red[1] ,pal->green[1], pal->blue[1]);
	printk (KERN_ERR "%s(%d) 0x%x 0x%x 0x%x \n",__func__, __LINE__, pal->red[2], pal->green[2], pal->blue[2]);
	*/
}

static void lcdc_cursor_bitmap_set(struct dw74fb *dw74fb, const struct cursor_bitmap* bitmap)
{
	int i = 0;

	for (i = 0 ; i < LCDC_CURSOR_BITMAP_ARRAY_SIZE ; i++ ) {
		dw74fb_writel(dw74fb, LCDC_REG_CURBITMAP + (i * 4) , bitmap->bitmap_array[i] );
	}
	dw74fb_writel(dw74fb, LCDC_REG_CURCR1, bitmap->bitmap_size);
	/*
	for (i = 0 ; i < LCDC_CURSOR_BITMAP_ARRAY_SIZE ; i++ ) {
		printk (KERN_ERR "%s(%d) \n" ,__func__, __LINE__);
		printk (KERN_ERR "bitmap[%i] = 0x%x \n" , i , bitmap->bitmap_array[i] );
	}
	printk (KERN_ERR "%s(%d) bitmap size %s\n" ,__func__, __LINE__ , bitmap->bitmap_size ? "64x64" : "32x32");
	*/
}

static void lcdc_cursor_bitmap_get(struct dw74fb *dw74fb, struct cursor_bitmap* bitmap)
{
	int i = 0;
	for (i = 0 ; i < LCDC_CURSOR_BITMAP_ARRAY_SIZE ; i++ ) {
		bitmap->bitmap_array[i] = dw74fb_readl(dw74fb, LCDC_REG_CURBITMAP + (i * 4));
	}
	bitmap->bitmap_size = dw74fb_readl(dw74fb, LCDC_REG_CURCR1);
}

static void lcdc_cursor_plane_location_set(struct dw74fb *dw74fb, struct dwfb_location* location){

	unsigned short y_start = location->y_start & 0xFFFE;
	dw74fb_writel(dw74fb, LCDC_REG_CURXSPR, location->x_start);
	dw74fb_writel(dw74fb, LCDC_REG_CURYSPR, y_start);
	/*printk (KERN_ERR "%s(%d) x = %d y = %d\n",__func__, __LINE__,location->x_start , y_start);*/
}

static void lcdc_cursor_plane_location_get(struct dw74fb *dw74fb, struct dwfb_location* location)
{
	location->x_start = dw74fb_readl(dw74fb, LCDC_REG_CURXSPR);
	location->y_start = dw74fb_readl(dw74fb, LCDC_REG_CURYSPR);
}

static void lcdc_cursor_plane_set(struct dw74fb *dw74fb, const struct fb_info *cursor)
{
	dw74fb_writel(dw74fb, LCDC_REG_CURCR0, (cursor->var.nonstd)? 1 : 0 );
	/*printk (KERN_ERR "%s(%d) Cursor status = %s \n",__func__, __LINE__,(cursor->var.nonstd)? "On" : "Off" );*/
}

#endif

static void lcdc_set_panel_size(struct dw74fb *dw74fb, const struct fb_info *info)
{
	dw74fb_writel(dw74fb, LCDC_REG_INDXSR, info->var.xres - 1);
	dw74fb_writel(dw74fb, LCDC_REG_INDYSR, info->var.yres - 1);
}

static void dw74fb_hw_init(struct dw74fb *dw74fb)
{
	struct fb_info *info = dw74fb->dwfb_info.info;
	dma_addr_t addr = info->fix.smem_start;

#ifdef CONFIG_MACH_VERSATILE_BROADTILE
	addr -= 0x80000000;
#endif

	dw74fb_run_sequence(dw74fb, dw74fb_hw_init_seq);
	set_display_bpp(dw74fb, info);

	dw74fb_set_videomode(dw74fb, DWFB_MODE_LCD);
	dw74fb_hw_disable(dw74fb);

	/* Set bright ness to defaults */
	dw74fb_writel(dw74fb, LCDC_REG_RBCR,0x80 );
	dw74fb_writel(dw74fb, LCDC_REG_GBCR,0x80 );
	dw74fb_writel(dw74fb, LCDC_REG_BBCR,0x80 );
	/* input data transfer sizes */
	dw74fb_writel(dw74fb, LCDC_REG_INDXSR, panel->xres - 1);
	dw74fb_writel(dw74fb, LCDC_REG_INDYSR, panel->yres - 1);

	/* display position */
	dw74fb_writel(dw74fb, LCDC_REG_DISPXSPOSR,  0);
	dw74fb_writel(dw74fb, LCDC_REG_DISPXEPOSR,  panel->xres - 1);
	dw74fb_writel(dw74fb, LCDC_REG_DISPYSPOS1R, 0);
	dw74fb_writel(dw74fb, LCDC_REG_DISPYEPOS1R, panel->yres - 1);

	/* input buffer */
	dw74fb_writel(dw74fb, LCDC_REG_MSBAHBA0R, addr >> 16);
	dw74fb_writel(dw74fb, LCDC_REG_LSBAHBA0R, addr & 0xffff);

#ifdef CONFIG_FB_DW74_DOUBLEBUF
	addr += (panel->yres * info->fix.line_length);
	dw74fb_writel(dw74fb, LCDC_REG_MSBAHBA1R, addr >> 16);
	dw74fb_writel(dw74fb, LCDC_REG_LSBAHBA1R, addr & 0xffff);
#else
	dw74fb_writel(dw74fb, LCDC_REG_MSBAHBA1R, 0);
	dw74fb_writel(dw74fb, LCDC_REG_LSBAHBA1R, 0);
#endif

	if (panel->init_seq)
		dw74fb_run_sequence(dw74fb, panel->init_seq);
	dw74fb_hw_enable(dw74fb);

#ifdef FB_DW74_PANEL_ILI9325
	platform_device_register(&dw_ili9325_device);
#endif
}

static void dw74fb_hw_param_update(struct dw74fb *dw74fb)
{
	dw74fb_writel(dw74fb, LCDC_REG_PARUP, 1);
}

static void dw74fb_hw_param_update_sync(struct dw74fb *dw74fb)
{
	unsigned long val;
	dw74fb->is_update_sync = 1;

	/* enable frame display done interrupt (0x2) (clear it first) */
	dw74fb_writel(dw74fb, LCDC_REG_INTSR, 0x1);
	val = dw74fb_readl(dw74fb, LCDC_REG_INTER);
	dw74fb_writel(dw74fb, LCDC_REG_INTER, val | 0x1);

    dw74fb_writel(dw74fb, LCDC_REG_PARUP, 1);

	/* wait for this interrupt to be sure that the lcdc updated the registers */
	if (down_timeout(&dw74fb->parameter_update_sync, 4) != 0)
		printk(KERN_WARNING "[%s:%s] parameters update timeout\n", __FILE__, __func__);

	if (dw74fb_readl(dw74fb, LCDC_REG_PARUP) & 0x1)
		printk(KERN_WARNING "[%s:%s] lcdc parameters not updated yet!\n", __FILE__, __func__);
}

static int get_bpp(struct dw74fb *dw74fb)
{
	int bpp = 0;

	switch (dw74fb_readl(dw74fb, LCDC_REG_INTMR)) {
		case 0:
			bpp = 16;
			break;
		case 1:
			bpp = 24;
			break;
		/* YUV422 and 2bpp are not supported */
	}

	return bpp;
}

static void convert_16bpp_to_24bpp(unsigned char *dest, unsigned short *src, int num_pixels)
{
	int r;
	int g;
	int b;

	while (num_pixels--) {
		b = *src & 0x001f;
		g = (*src & 0x07e0) >> 5;
		r = (*src & 0xf800) >> 11;

		dest[0] = b << 3;
		dest[1] = g << 2;
		dest[2] = r << 3;

		src++;
		dest += 3;
	}
}

static void update_refresh_address(struct dw74fb *dw74fb, int index, dma_addr_t addr, int start_refresh_new_addr)
{
	struct fb_info *info = dw74fb->dwfb_info.info;

	if (index == 0) {
		dw74fb_writel(dw74fb, LCDC_REG_MSBAHBA0R, addr >> 16);
		dw74fb_writel(dw74fb, LCDC_REG_LSBAHBA0R, addr & 0xffff);
		dw74fb_writel(dw74fb, LCDC_REG_OFFAR0,    info->fix.line_length);
	} else {
		dw74fb_writel(dw74fb, LCDC_REG_MSBAHBA1R, addr >> 16);
		dw74fb_writel(dw74fb, LCDC_REG_LSBAHBA1R, addr & 0xffff);
		dw74fb_writel(dw74fb, LCDC_REG_OFFAR1,    info->fix.line_length);
	}

	if (start_refresh_new_addr) {
		dw74fb_writel(dw74fb, LCDC_REG_DISPIDXR, index);
		dw74fb_hw_param_update_sync(dw74fb);
	}
}

static int dw74fb_hw_try_resume(struct dw74fb *dw74fb)
{
	struct fb_info *info = dw74fb->dwfb_info.info;
	dma_addr_t addr1 = info->fix.smem_start + (panel->yres * info->fix.line_length);
	dma_addr_t old;
	int running = dw74fb_readl(dw74fb, LCDC_REG_LCDCCR) & 1;
	void *old_virt;
	int old_bpp = get_bpp(dw74fb);
	int new_bpp = panel->bits_per_pixel == 16 ? 16 : 24;

	if (!running)
		return 0;

	/* copy over the contents of the framebuffer to the new location */
	old = dw74fb_readl(dw74fb, LCDC_REG_LSBAHBA0R) |
	      dw74fb_readl(dw74fb, LCDC_REG_MSBAHBA0R) << 16;
	old_virt = ioremap(old, info->screen_size/2);
	if (!old_virt)
		return 0;
	if (old_bpp == new_bpp) {
		/* simple copy. u-boot has only one buffer, so we copy it to both our buffers */
		memcpy(info->screen_base, old_virt, info->screen_size/2);
#ifdef CONFIG_FB_DW74_DOUBLEBUF
		memcpy(info->screen_base + info->screen_size/2, old_virt, info->screen_size/2);
#endif
	} else	if (new_bpp == 24 && old_bpp == 16) {
		/* convert the image while copying */
		convert_16bpp_to_24bpp(info->screen_base, old_virt, panel->xres * panel->yres);
#ifdef CONFIG_FB_DW74_DOUBLEBUF
		convert_16bpp_to_24bpp(info->screen_base + info->screen_size/2, old_virt, panel->xres * panel->yres);
#endif
	} else {
		memset(info->screen_base, 0, info->screen_size);
	}

	iounmap(old_virt);

	/*
	 * the next code is ugly, but with a reason :-)
	 * PURAP mechanism is not working on OFFARX, MSBAHBAXR and LSBAHBXR registers!
	 * Thus, writing to any of these registers will cause a flickering on the screen (while refreshing from address X)
	 * Solution:
	 *		If we are refreshing from address 0:
	 *			Update address1, and switch to it
	 *			Then update address0, and switch back to it
	 *		If we are refreshing from address 1:
	 *			Update address0, and switch to it
	 *			Then update address1, and don't switch back to it!
	 *
	 * Note: rest of the code assumes we are on address0 at boot time!
	 */
	dw74fb_writel(dw74fb,LCDC_REG_INTMR, new_bpp == 24 ? 1 : 0);
	dw74fb_writel(dw74fb, LCDC_REG_INDTR, ((info->var.xres * info->var.bits_per_pixel ) / 32 ) - 1);

	if (dw74fb_readl(dw74fb, LCDC_REG_DISPIDXR) == 0) {
		update_refresh_address(dw74fb, 1, addr1, 1);
		update_refresh_address(dw74fb, 0, info->fix.smem_start, 1);
	}
	else {
		update_refresh_address(dw74fb, 0, info->fix.smem_start, 1);
		update_refresh_address(dw74fb, 1, addr1, 0);
	}

	return 1;
}

/*
 * FIXME: uglyness, quick integration into DW branch for DW74 chip
 */
#include <mach/hardware.h>
#ifdef CONFIG_FB_DMW96_OSDM
int get_osdm_parameters_from_lcdc(struct fb_info* display , void (*osdm_switch_complete_cb)(void))
{
	int res =0;
	int non_active_index;

	display->var.xres = g_dw74fb->dwfb_info.info->var.xres;
	display->var.yres = g_dw74fb->dwfb_info.info->var.yres;
	display->var.bits_per_pixel = g_dw74fb->dwfb_info.info->var.bits_per_pixel;
#ifdef CONFIG_FB_DMW96_OSDM
	display->var.reserved[0] = panel->osdm_config;
#endif

	if (g_dw74fb != NULL) {
		non_active_index = dw74fb_readl(g_dw74fb, LCDC_REG_DISPIDXR) == 1 ? 0 : 1;
		display->fix.smem_start = g_dw74fb->dwfb_info.info->fix.smem_start +
			(non_active_index * g_dw74fb->dwfb_info.info->var.yres * g_dw74fb->dwfb_info.info->fix.line_length);
		#ifdef CONFIG_MACH_VERSATILE_BROADTILE
		display->fix.smem_start = 0x60000000;
		#endif
	}
	else {
		printk("%s unable to hand display info to OSDM devide was not initiated\n", __func__);
		res = -ENOMEM;
	}

	if (osdm_switch_complete_cb)
		g_dw74fb->osdm_switch_complete_cb = osdm_switch_complete_cb;
	else
		res = -ENOMEM;

	return res;
}


int switch_osdm_and_lcdc_address(struct fb_info* display)
{
	unsigned long index_lcdc=0x0;
	unsigned int val=0;

	index_lcdc = dw74fb_readl(g_dw74fb, LCDC_REG_DISPIDXR);

	/*Switch the OSDM address to LCDC address*/
	display->fix.smem_start = g_dw74fb->dwfb_info.info->fix.smem_start + (index_lcdc * g_dw74fb->dwfb_info.info->fix.smem_len / 2);
	
	if (index_lcdc == 0x1)
		index_lcdc = 0x0;
	else
		index_lcdc = 0x1;

    g_dw74fb->is_switch_complete = 1;
	dw74fb_writel(g_dw74fb,LCDC_REG_DISPIDXR, index_lcdc );

    /*enable frame display start interrupt (0x1) (clear it first) */
	dw74fb_writel(g_dw74fb, LCDC_REG_INTSR, 0x1);
	val = dw74fb_readl(g_dw74fb, LCDC_REG_INTER);
	dw74fb_writel(g_dw74fb, LCDC_REG_INTER, val | 0x1);

	dw74fb_hw_param_update(g_dw74fb);

	return 0;
}


EXPORT_SYMBOL(get_osdm_parameters_from_lcdc);
EXPORT_SYMBOL(switch_osdm_and_lcdc_address);
#endif

/*
 * driver implementation
 */
#ifdef CONFIG_FB_DW74_LOW_POWER
static void preper_to_pan(struct fb_var_screeninfo *var, struct fb_info *info)
{
	struct dw74fb *dw74fb = info->par;
	struct dwfb_info *l_dwfb_info = NULL;

	if (dw74fb->dwfb_info.info == info) {
		l_dwfb_info = &(dw74fb->dwfb_info);
	}
	else if (dw74fb->dwfb_info_low_power.info == info){
		l_dwfb_info = &(dw74fb->dwfb_info_low_power);
	}
	else {
		printk(KERN_ERR "Fail to handle IOCT invalide frame buffer\n");
		return;
	}

	set_display_location(dw74fb, &(l_dwfb_info->location));
	set_display_background(dw74fb, l_dwfb_info->background_color);
	set_display_bpp(dw74fb, info);
}
#endif

/*
 * this function will only be called if CONFIG_FB_DW74_DOUBLEBUF is set
 */
static int
dw74fb_pan_display(struct fb_var_screeninfo *var, struct fb_info *info)
{
 printk("dw74fb_pan_display\n");

#ifdef CONFIG_FB_DMW96_OSDM
	/* we don't support panning when OSDM is used */
	return 0;
#else
	struct dw74fb *dw74fb = info->par;
	dma_addr_t addr;
	dma_addr_t offs = var->yoffset * info->fix.line_length;

	if ((lcdc_mode >= DWFB_MODE_480P60A) && (lcdc_mode < DWFB_MODE_480P60))
		offs = var->yoffset * ANDROID_XRES * (info->var.bits_per_pixel/8);

#ifdef CONFIG_FB_DW74_LOW_POWER
	preper_to_pan(var,info);
#endif

	if (panel->is_cpu_type)
		lcdc_own_hw();

	addr = info->fix.smem_start + offs;

	dw74fb_writel(dw74fb, LCDC_REG_MSBAHBA0R, addr >> 16);
	dw74fb_writel(dw74fb, LCDC_REG_LSBAHBA0R, addr & 0x0000ffff);

	/* Attention if not defined that we are working with v-sync which is connected to external interrupts
	then we need to update the parameters of the LCDC other wise we need will update them in the interrupt
	handler of the v-sync.
	*/
	if (!panel->is_cpu_type)
		dw74fb_hw_param_update_sync(dw74fb);
	else
		dw74fb_hw_param_update(dw74fb);

	return 0;
#endif
}

static u32 pseudo_palette[16];
#ifdef CONFIG_FB_DW74_LOW_POWER
static u32 pseudo_palette_low_power[16];
#endif

static int dw74fb_setcolreg(unsigned regno, unsigned red, unsigned green,
                          unsigned blue, unsigned transp, struct fb_info *info)
{
	u32 *palette = info->pseudo_palette;

	if (regno >= 16)
		return -EINVAL;

	/* only FB_VISUAL_TRUECOLOR supported */

	red >>= 16 - info->var.red.length;
	green >>= 16 - info->var.green.length;
	blue >>= 16 - info->var.blue.length;
	transp >>= 16 - info->var.transp.length;

	palette[regno] = (red << info->var.red.offset) |
	  (green << info->var.green.offset) |
	  (blue << info->var.blue.offset) |
	  (transp << info->var.transp.offset);

	return 0;
}

static int
dw74fb_check_var(struct fb_var_screeninfo *var, struct fb_info *info)
{
	/* For now, don't check anything... */
	return 0;
}

static int dw74fb_ioctl(struct fb_info *info, unsigned int cmd,unsigned long arg)
{
	void __user *argp = (void __user *)arg;
	struct dw74fb *dw74fb = info->par;
	struct dwfb_info *l_dwfb_info = NULL;
	unsigned int clock_rate = 0;
#ifdef CONFIG_FB_DMW96_CURSOR
	struct cursor_bitmap cursor_bit;
#endif

	if (dw74fb->dwfb_info.info == info) {
		l_dwfb_info = &(dw74fb->dwfb_info);
	}
#ifdef CONFIG_FB_DW74_LOW_POWER
	else if (dw74fb->dwfb_info_low_power.info == info){
		l_dwfb_info = &(dw74fb->dwfb_info_low_power);
	}
#endif
#ifdef CONFIG_FB_DMW96_CURSOR
	else if (dw74fb->dwfb_info_cursor.info == info){
		l_dwfb_info = &(dw74fb->dwfb_info_cursor);
	}
#endif
	else {
		printk(KERN_ERR "Fail to handle IOCT invalide frame buffer\n");
		return -1;
	}

	switch (cmd) {
		case FBIOSET_BACKGROUND:
			if (copy_from_user(&(l_dwfb_info->background_color), argp, sizeof(unsigned short)) )
				return -EFAULT;
			set_display_background(dw74fb,l_dwfb_info->background_color);
			break;

		case FBIOSET_DISPLAY_LOCATION:
			if (copy_from_user(&(l_dwfb_info->location), argp, sizeof(struct dwfb_location)) )
				return -EFAULT;
			if (l_dwfb_info->location.x_start >= l_dwfb_info->location.x_end ||
				l_dwfb_info->location.y_start >= l_dwfb_info->location.y_end ) {

				printk(KERN_ERR "Fail to handle IOCT invalide location\n");
				return -EFAULT;
			}
			set_display_location(dw74fb,&(l_dwfb_info->location));
			break;

		case FBIOSET_DISPLAY_CLOCK_RATE:
			if (copy_from_user(&(clock_rate), argp, sizeof(unsigned int)) )
				return -EFAULT;
			//printk(KERN_ERR "change clock rate to %d\n", clock_rate);
			clk_set_rate(dw74fb->clk, clk_round_rate(dw74fb->clk, clock_rate));
			break;

		case FBIOSET_DISPLAY_MODE:
			if (copy_from_user(&(lcdc_mode), argp, sizeof(unsigned int)) )
				return -EFAULT;
			dw74fb_set_videomode(dw74fb, lcdc_mode);
			break;

#ifdef CONFIG_FB_DMW96_CURSOR
		case FBIOSET_LCDC_CURSOR_BITMAP:
			if (copy_from_user(&cursor_bit, argp, sizeof(struct cursor_bitmap)) )
				return -EFAULT;
			lcdc_cursor_bitmap_set(dw74fb, &cursor_bit);
			break;
		case FBIOGET_LCDC_CURSOR_BITMAP:
			lcdc_cursor_bitmap_get(dw74fb, &cursor_bit);
			if (copy_to_user(argp, &cursor_bit , sizeof(struct cursor_bitmap)) )
				return -EFAULT;
			break;
		case FBIOSET_LCDC_CURSOR_LOCATION:
			if (copy_from_user(&(l_dwfb_info->location), argp, sizeof(struct dwfb_location)) )
				return -EFAULT;
			lcdc_cursor_plane_location_set(dw74fb,&(l_dwfb_info->location));
			break;
		case FBIOGET_LCDC_CURSOR_LOCATION:
			lcdc_cursor_plane_location_get(dw74fb,&(l_dwfb_info->location));
			if (copy_to_user(argp, &(l_dwfb_info->location) , sizeof(struct dwfb_location)) )
				return -EFAULT;
			break;
#endif
	}

	return 0;
}

static int dw74fb_cmap(struct fb_cmap *cmap, struct fb_info *info)
{
	int res = 0;
	struct dw74fb *dw74fb = info->par;

#ifdef CONFIG_FB_DMW96_CURSOR
	if (dw74fb->dwfb_info_cursor.info == info) {
		cursor_palette_set(dw74fb, cmap);
	}
#endif

	display_palette_set(dw74fb, cmap);
	return res;
}

int dw74fb_set_par(struct fb_info *info)
{
	int res = 0;
	struct dw74fb *dw74fb = info->par;

	if (lcdc_mode >= DWFB_MODE_480P60A)
		return res;

	#ifdef CONFIG_FB_DMW96_CURSOR
	if (dw74fb->dwfb_info_cursor.info != info)
	#endif
		res = set_display_bpp(dw74fb, info);

	#ifdef CONFIG_FB_DMW96_CURSOR
	if (dw74fb->dwfb_info_cursor.info == info) {
		lcdc_cursor_plane_set(dw74fb,info);
	}
	#endif

	if (dw74fb->dwfb_info.info == info) {
		lcdc_set_panel_size(dw74fb,info);
		if (lcdc_mode == DWFB_MODE_VIDEO_PAL) {

			dw74fb_writel(dw74fb, LCDC_REG_DISPIR,  0x0061);
			dw74fb_writel(dw74fb, LCDC_REG_VATR, (unsigned int) 624);
			dw74fb_writel(dw74fb, LCDC_REG_HSTR,(unsigned int) 279);

		} else if (lcdc_mode == DWFB_MODE_VIDEO_NTSC) {

			dw74fb_writel(dw74fb, LCDC_REG_DISPIR,  0x00e1);
			dw74fb_writel(dw74fb, LCDC_REG_VATR, (unsigned int) 524);
			dw74fb_writel(dw74fb, LCDC_REG_HSTR,(unsigned int) 267);
		} else if (lcdc_mode == DWFB_MODE_LCD) {

			dw74fb_run_sequence(dw74fb, dw74fb_hw_init_seq);
			if (panel->init_seq)
					dw74fb_run_sequence(dw74fb, panel->init_seq);
			res = set_display_bpp(dw74fb, info);
		}
	}

	return res;
}

static struct fb_ops dw74fb_ops = {
	.owner          = THIS_MODULE,
	.fb_pan_display = dw74fb_pan_display,
	.fb_check_var   = dw74fb_check_var,
	.fb_ioctl       = dw74fb_ioctl,
	.fb_fillrect    = cfb_fillrect,
	.fb_copyarea    = cfb_copyarea,
	.fb_imageblit   = cfb_imageblit,
	.fb_setcolreg   = dw74fb_setcolreg,
	.fb_setcmap     = dw74fb_cmap,
	.fb_set_par     = dw74fb_set_par,
};

static irqreturn_t dw74fb_irq(int irq, void *priv)
{
	struct dw74fb *dw74fb = priv;
	unsigned long val;
	int stat;

	/* read & clear active interrupts */
	stat = dw74fb_readl(dw74fb, LCDC_REG_INTSR);
	dw74fb_writel(dw74fb, LCDC_REG_INTSR, stat);

	if (panel->is_cpu_type) {
		if (stat & 0x2) {
			/*Frame display done*/
			lcdc_release_hw();
		}
	} else {
		if (stat & 0x1) {
			/* release parameters update semaphore and disable the interrupt */
			val = dw74fb_readl(dw74fb, LCDC_REG_INTER);
			dw74fb_writel(dw74fb, LCDC_REG_INTER, val & (~0x1));
#ifdef CONFIG_FB_DMW96_OSDM
			if (dw74fb->is_switch_complete && dw74fb->osdm_switch_complete_cb) {
				dw74fb->is_switch_complete = 0;
				dw74fb->osdm_switch_complete_cb();
			}
#endif
			if (dw74fb->is_update_sync) {
				dw74fb->is_update_sync = 0;
				up(&dw74fb->parameter_update_sync);
			}

			return IRQ_HANDLED;
		}
	}

	/* handle interrupts */
	if (stat & 0x4) {
		underruns++;
		return IRQ_HANDLED;
	}

	unhandled_ints++;

	return IRQ_HANDLED;
}

#ifdef CONFIG_FB_DW74_LOW_POWER
int release_low_power_fb(struct dw74fb *dw74fb)
{
	struct dw74fb *dw74fb = info->par;
	struct fb_info *info = dw74fb->dwfb_info_low_power.info;

	dma_free_coherent(NULL, (int)info->screen_size, info->screen_base, (dma_addr_t)info->fix.smem_start);
	fb_dealloc_cmap(&info->cmap);
	framebuffer_release(info);

	return 0;
}

int configure_low_power_fb(struct dw74fb *dw74fb)
{
	struct fb_info *info;
	int ret = 0;

	info = framebuffer_alloc(sizeof(u32) * 256, NULL);
	if (!info)
		goto err_free_dw74fb_low_power;

	info->pseudo_palette = pseudo_palette_low_power;
	info->par = dw74fb;
	info->flags = FBINFO_FLAG_DEFAULT;

	strcpy(info->fix.id, DW74FB_NAME_LOW_POWER);
	info->fix.type = FB_TYPE_PACKED_PIXELS;
	info->fix.visual = FB_VISUAL_MONO10;
	info->fix.accel = FB_ACCEL_NONE;
	info->fix.line_length = panel->xres / 4 ;
	info->var.xres = panel->xres;
	info->var.yres = panel->yres;
	info->var.xres_virtual = panel->xres;
	info->var.yres_virtual = panel->yres;
	info->var.xoffset = 0;
	info->var.yoffset = 0;
	info->var.bits_per_pixel = 2;
	info->var.red.offset = 0;
	info->var.red.length = 0;
	info->var.green.offset = 0;
	info->var.green.length = 0;
	info->var.blue.offset = 0;
	info->var.blue.length = 0;
	info->var.transp.offset = 0;
	info->var.transp.length = 0;
	info->var.vmode = FB_VMODE_NONINTERLACED;
	info->var.height = 0;
	info->var.width = 0;

	#ifdef CONFIG_FB_DW74_DOUBLEBUF
	info->fix.ypanstep = 1;
	info->var.yres_virtual = panel->yres * 2;
	#endif

	info->screen_size = info->fix.line_length * info->var.yres_virtual;
	info->fix.smem_len = info->screen_size;
	info->fbops = &dw74fb_ops;

	dw74fb->dwfb_info_low_power.location.x_start = 0;
	dw74fb->dwfb_info_low_power.location.x_end = panel->xres;
	dw74fb->dwfb_info_low_power.location.y_start = 0;
	dw74fb->dwfb_info_low_power.location.y_end = panel->yres;

	/*Set the backgrount of the display*/
	dw74fb->dwfb_info_low_power.background_color = 0;

	ret = fb_alloc_cmap(&info->cmap, 256, 0);
	if (ret < 0)
		goto err_free_info_low_power;

	/* attach info to driver data */
	dw74fb->dwfb_info_low_power.info = info;

	dw74fb->dwfb_info_low_power.vidmem = dma_alloc_coherent(NULL, info->screen_size,
										(dma_addr_t *)&info->fix.smem_start,
										GFP_DMA);

	if (!dw74fb->dwfb_info_low_power.vidmem) {
		//dev_err(dev, "could not allocate framebuffer (%lu bytes)\n",
		//			 info->screen_size);
		goto err_free_cmap_low_power;
	}

	info->screen_base = dw74fb->dwfb_info_low_power.vidmem;
	memset(info->screen_base, 0x00, info->screen_size);

	/* register framebuffer in userspace */
	ret = register_framebuffer(info);
	if (ret < 0)
		goto err_free_allocate_low_power;

	return 0;

err_free_allocate_low_power:
	dma_free_coherent(NULL, (int)info->screen_size, info->screen_base, (dma_addr_t)info->fix.smem_start);
err_free_cmap_low_power:
	fb_dealloc_cmap(&info->cmap);
err_free_info_low_power:
	framebuffer_release(info);
err_free_dw74fb_low_power:
	return -1;
}
#endif

#ifdef CONFIG_FB_DMW96_CURSOR
const char* lcdc_cursor_name = "dmw96_cursor";
static int configure_cursor_fb(struct dw74fb *dw74fb)
{
	struct fb_info *info;
	int ret = -ENOMEM;

	info = framebuffer_alloc(0 , NULL);
	if (!info)
		return -ENOMEM;

	info->par = dw74fb;
	info->var.nonstd = 0;

	info->screen_base = NULL;
	info->fbops = &dw74fb_ops;
	info->flags = FBINFO_FLAG_DEFAULT;

	strcpy(info->fix.id, lcdc_cursor_name);
	info->fix.type = FB_TYPE_PACKED_PIXELS;
	info->fix.visual = FB_VISUAL_TRUECOLOR;
	info->fix.accel = FB_ACCEL_NONE;
	info->fix.line_length = 0;
	info->fix.smem_start = 0;
	info->fix.smem_len = 0;
	info->screen_size = info->fix.smem_len;

	ret = fb_alloc_cmap(&info->cmap, CURSOR_PALETTE_SIZE , 0);
	if (ret < 0)
		goto err_free_info_cursor;

	/* register framebuffer in userspace */
	ret = register_framebuffer(info);
	if (ret < 0)
		goto err_free_cmap_cursor;

	/* attach info to driver data */
	dw74fb->dwfb_info_cursor.info = info;

	printk(KERN_INFO "fb%d: %s frame buffer device, %dx%d, %d bpp, %luk\n",
	       info->node,
	       info->fix.id,
	       info->var.xres,
	       info->var.yres,
	       info->var.bits_per_pixel,
	       info->screen_size >> 10);

	return 0;

err_free_cmap_cursor:
	fb_dealloc_cmap(&info->cmap);
err_free_info_cursor:
	framebuffer_release(info);

	return ret;
}
#endif /*CONFIG_FB_DMW96_CURSOR*/

static void dw74fb_suspend(struct dw74fb *dw74fb)
{
	if (panel->suspend)
		panel->suspend(panel);

	dw74fb_hw_disable(dw74fb);
	udelay(1000);
	clk_disable(dw74fb->clk);
}

static void dw74fb_resume(struct dw74fb *dw74fb)
{
	clk_enable(dw74fb->clk);
	dw74fb_hw_enable(dw74fb);

	if (panel->resume)
		panel->resume(panel);
}

#ifdef CONFIG_HAS_EARLYSUSPEND
static void dw74fb_early_suspend(struct early_suspend *es)
{
	struct dw74fb *dw74fb = container_of(es, struct dw74fb, early_suspend);
	dw74fb_suspend(dw74fb);
}

static void dw74fb_late_resume(struct early_suspend *es)
{
	struct dw74fb *dw74fb = container_of(es, struct dw74fb, early_suspend);
	dw74fb_resume(dw74fb);
}
#endif

#define gs_logo_physadd 0x58000000
#define GS_LOGO
static int __init dw74fb_probe(struct platform_device *pdev)
{
	struct device *dev = &pdev->dev;
	struct dw74fb *dw74fb;
	struct fb_info *info;
	struct resource *res;
    void *gs_logo_virtadd;
	int ret;
  
	printk("%s()\n", __func__);
    
	if (!dev->platform_data)
		return -EINVAL;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res) {
		dev_err(&pdev->dev, "need a memory resource\n");
		return -ENXIO;
	}

	if (!request_mem_region(res->start, res->end - res->start,
	                        "dw74fb regs")) {
		dev_err(dev, "failed to request memory region\n");
		return -EBUSY;
	}

	/* init driver data */
	ret = -ENOMEM;
	dw74fb = kzalloc(sizeof(*dw74fb), GFP_KERNEL);
	if (!dw74fb)
		goto err_rel_mem;

	sema_init(&dw74fb->parameter_update_sync, 0);
	panel = dev->platform_data;
	//dw74fb->panel = panel;

	dw74fb->irq = platform_get_irq(pdev, 0);
	if (dw74fb->irq < 0) {
		dev_err(&pdev->dev, "need an irq resource\n");
		ret = dw74fb->irq;
		goto err_free_dw74fb;
	}

	if (panel->init)
		panel->init(panel);

	dw74fb->clk = clk_get(&pdev->dev, "lcdc");
	if (IS_ERR(dw74fb->clk)) {
		dev_err(&pdev->dev, "could not get clock\n");
		ret = PTR_ERR(dw74fb->clk);
		goto err_free_dw74fb;
	}

	if (panel->clock_rate)
		clk_set_rate(dw74fb->clk, clk_round_rate(dw74fb->clk, panel->clock_rate));
	clk_enable(dw74fb->clk);

	/* init frame buffer */
	info = framebuffer_alloc(sizeof(u32) * 256, dev);
	if (!info)
		goto err_free_dw74fb;

	info->pseudo_palette = pseudo_palette;
	info->par = dw74fb;
	info->flags = FBINFO_FLAG_DEFAULT;

	strcpy(info->fix.id, DW74FB_NAME);
	info->fix.type = FB_TYPE_PACKED_PIXELS;
	info->fix.visual = FB_VISUAL_TRUECOLOR;
	info->fix.accel = FB_ACCEL_NONE;
	info->fix.smem_start = 0xe0000000;
	info->fix.smem_len = 0x04000000;

	info->var.xres = panel->xres;
	info->var.yres = panel->yres;
	info->var.xres_virtual = panel->xres;
	info->var.yres_virtual = panel->yres;
	info->var.xoffset = 0;
	info->var.yoffset = 0;

	/*
	 * if panel supports only 16bits, then frame buffer will hold rgb565
	 * if panel supports rgb666 or rgb888, then frame buffer will hold rgb888
	 * (in case of panel which supports rgb666, the LCDC will drop the 2
	 * least significat bits of each color component))
	 */
	if (bpp_to_info(&info->var, panel->bits_per_pixel) < 0) {
		printk("dw74fb: panel->bits_per_pixel is not supported!");
		return -EBUSY;
	}

	info->fix.line_length = panel->xres * info->var.bits_per_pixel / 8;

	info->var.transp.offset = 0;
	info->var.transp.length = 0;
	info->var.vmode = FB_VMODE_NONINTERLACED;
	info->var.width = panel->width_mm;
	info->var.height = panel->height_mm;

#ifdef CONFIG_FB_DW74_DOUBLEBUF
	info->fix.ypanstep = 1;
	info->var.yres_virtual = panel->yres * 2;
#endif

	info->screen_size = info->fix.line_length * info->var.yres_virtual;
	info->fix.smem_len = info->screen_size;
	info->fbops = &dw74fb_ops;

	ret = fb_alloc_cmap(&info->cmap, 256, 0);
	if (ret < 0)
		goto err_free_info;

	/*Set the location of the display in the screen*/
	dw74fb->dwfb_info.location.x_start = 0;
	dw74fb->dwfb_info.location.x_end = panel->xres;
	dw74fb->dwfb_info.location.y_start = 0;
	dw74fb->dwfb_info.location.y_end = panel->yres;

	/*Set the backgrount of the display*/
	dw74fb->dwfb_info.background_color = 0;

	/* attach info to driver data */
	dw74fb->dwfb_info.info = info;

	/* setup memory */
#ifndef CONFIG_MACH_VERSATILE_BROADTILE
	dw74fb->dwfb_info.vidmem = dma_alloc_coherent(&pdev->dev, 1920*1080*3*2, //FIXME info->screen_size,
	                                    (dma_addr_t *)&info->fix.smem_start,
	                                    GFP_DMA);
#else
	if (!request_mem_region(info->fix.smem_start, info->screen_size ,dw74fb_name)) {
		printk("cannot request mem region for %s \n",dw74fb_name);
		return -EBUSY;
	}
   
	dw74fb->dwfb_info.vidmem  = ioremap_nocache(info->fix.smem_start, info->screen_size );
	if (!dw74fb->dwfb_info.vidmem) {
		printk("cannot map sdram for %s\n",dw74fb_name);
		return -EBUSY;
	}
#endif

	if (!dw74fb->dwfb_info.vidmem) {
		dev_err(dev, "could not allocate framebuffer (%lu bytes)\n",
		             info->screen_size);
		goto err_free_cmap;
	}

	info->fix.smem_len = info->screen_size;
	info->screen_base = dw74fb->dwfb_info.vidmem;

	/* init hardware */
	ret = -ENOMEM;
	dw74fb->regs = ioremap(res->start, res->end - res->start);
	if (!dw74fb->regs) {
		dev_err(dev, "unable to map registers\n");
		ret = -ENOMEM;
		goto err_free_cmap;
	}

	/* disable update int, and clear all pending interrupts (may exist from u-boot) */
	dw74fb_writel(dw74fb, LCDC_REG_INTER, dw74fb_readl(dw74fb, LCDC_REG_INTER) & ~0x1);
	dw74fb_writel(dw74fb, LCDC_REG_INTSR, dw74fb_readl(dw74fb, LCDC_REG_INTSR));

	ret = request_irq(dw74fb->irq, dw74fb_irq, 0, dw74fb_name, dw74fb);
	if (ret < 0) {
		dev_err(dev, "request irq %d failed\n", dw74fb->irq);
		goto err_free_cmap;
	}
    
	if (!dw74fb_hw_try_resume(dw74fb)) {
     
		memset(info->screen_base, 0, info->screen_size);
       
		dw74fb_hw_init(dw74fb);
		dw74fb_hw_enable(dw74fb);
     
	}

	/* attach driver data to the device */
	dev_set_drvdata(dev, dw74fb);

	/* register framebuffer in userspace */
	ret = register_framebuffer(info);
	if (ret < 0)
		goto err_free_irq;
       
	printk(KERN_INFO "fb%d: %s frame buffer, %s (%dx%d, %d bpp, %luk)\n",
	       info->node,
	       info->fix.id,
	       panel->name,
	       info->var.xres,
	       info->var.yres,
	       info->var.bits_per_pixel,
	       info->screen_size >> 10);

	/*This ugly global pointer is used to access dma address from OSDM driver*/
	g_dw74fb = dw74fb;

#ifdef CONFIG_HAS_EARLYSUSPEND
	dw74fb->early_suspend.suspend = dw74fb_early_suspend;
	dw74fb->early_suspend.resume = dw74fb_late_resume;
	dw74fb->early_suspend.level = EARLY_SUSPEND_LEVEL_DISABLE_FB;

	register_early_suspend(&dw74fb->early_suspend);
#endif

	/*We need that semaphore in case we are in CPU type panel and we
	want to make sure that there will not be any updates from the external
	controller while the DMA of the external controller was NOT finished*/
	if (panel->is_cpu_type){
		sema_init(&(dw74fb->lcdc_hw_busy), 1);
		goto err_free_irq;
	}

#ifdef CONFIG_FB_DW74_LOW_POWER
	ret = configure_low_power_fb(dw74fb);
	if (ret < 0)
		goto err_free_irq;
#endif

#ifdef CONFIG_FB_DMW96_CURSOR
	ret = configure_cursor_fb(dw74fb);
	#ifdef CONFIG_FB_DW74_LOW_POWER
		if (ret < 0)
			release_low_power_fb(dw74fb);
	#endif
	if (ret < 0)
		goto err_free_irq;
#endif

#ifdef	CONFIG_BACKLIGHT_DMW_AMOLED
	dev_set_drvdata(&amoled_backlight_device.dev,(void*)dw74fb);
	ret = platform_device_register(&amoled_backlight_device);
	
	if (ret < 0){

		printk(KERN_ERR"amoled device registration failed!");
		goto err_free_irq;
	}
#endif

#if 0
       gs_logo_virtadd = ioremap((u32)gs_logo_physadd, 480*272*4 );
	if (!gs_logo_virtadd) {
		dev_err(dev,"cannot map sdram for gs_logo\n");
		return -EBUSY;
	}
    
    convert_16bpp_to_24bpp((unsigned char *)info->screen_base, 
        (unsigned short *)gs_logo_virtadd, 
        480*272);
 
    iounmap(gs_logo_virtadd);

#endif /*GS_LOGO*/

	return 0;

err_free_irq:
	free_irq(dw74fb->irq, dw74fb);

err_free_cmap:
	fb_dealloc_cmap(&info->cmap);

err_free_info:
	framebuffer_release(dw74fb->dwfb_info.info);

err_free_dw74fb:
	kfree(dw74fb);

err_rel_mem:
	release_mem_region(res->start, res->end - res->start);
	return ret;
}

static int __exit dw74fb_remove(struct platform_device *pdev)
{
	g_dw74fb = NULL;

#ifdef	CONFIG_BACKLIGHT_DMW_AMOLED
	platform_device_unregister(&amoled_backlight_device);
#endif
	
	/* TODO: free everything */
	return 0;
}

#if defined(CONFIG_PM) && !defined(CONFIG_HAS_EARLYSUSPEND)
static int dw74fb_dev_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct dw74fb *dw74fb = dev_get_drvdata(&pdev->dev);
	dw74fb_suspend(dw74fb);
	return 0;
}

static int dw74fb_dev_resume(struct platform_device *pdev)
{
	struct dw74fb *dw74fb = dev_get_drvdata(&pdev->dev);
	dw74fb_resume(dw74fb);
	return 0;
}
#else
#define dw74fb_dev_suspend NULL
#define dw74fb_dev_resume NULL
#endif

static struct platform_driver dw74fb_driver = {
	.remove = __exit_p(dw74fb_remove),
	.suspend = dw74fb_dev_suspend,
	.resume = dw74fb_dev_resume,
	.driver = {
		.name = dw74fb_name,
		.owner = THIS_MODULE,
	},
};

static int __init dw74fb_init(void)
{
	return platform_driver_probe(&dw74fb_driver, dw74fb_probe);
}

static void __exit dw74fb_exit(void)
{
	platform_driver_unregister(&dw74fb_driver);
}


module_init(dw74fb_init);
module_exit(dw74fb_exit);
