#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/platform_device.h>
#include <linux/fb.h>
#include <linux/init.h>
#include <linux/semaphore.h>
#include <linux/string.h>

#include <linux/sysfs.h>
#include <linux/device.h>
#include <linux/clk.h>

#include <video/dmw96osdm_w.h>

#include "dmw96osdm.h"

extern int get_plane_num(struct dmw96osdm* osdm, struct fb_info *info);
extern const char* get_plane_name(unsigned int plane);

static void osdm_clk_enable(struct device *dev)
{
	struct fb_info *info;
	struct dmw96osdm* osdm;

	if (!dev)
		return;
	info = dev_get_drvdata(dev);
	if (!info)
		return;
	osdm = FB2OOSDM(info);
	if (!osdm)
		return;
	if (!osdm->clk)
		return;

	clk_enable(osdm->clk);
}

static void osdm_clk_disable(struct device *dev)
{
	struct fb_info *info;
	struct dmw96osdm* osdm;

	if (!dev)
		return;
	info = dev_get_drvdata(dev);
	if (!info)
		return;
	osdm = FB2OOSDM(info);
	if (!osdm)
		return;
	if (!osdm->clk)
		return;

	clk_disable(osdm->clk);
}

static ssize_t get_display_pause(struct device *dev, struct device_attribute *attr, char *buf)
{
	unsigned int enb = 0;

	osdm_clk_enable(dev);
	dmw96osdm_pause_get(&enb);
	osdm_clk_disable(dev);

	PDEBUG("pause %s \n", (enb > 0) ? "Enabled" : "Disabled");

	return snprintf(buf, PAGE_SIZE, "%d\n", enb);
}

static ssize_t set_display_pause(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	unsigned int enb = simple_strtoul(buf, NULL, 0);
	enb = (enb > 0) ? 1 : 0;

	osdm_clk_enable(dev);
	dmw96osdm_pause_set(enb);
	osdm_clk_disable(dev);

	PDEBUG("pause %s \n", (enb > 0) ? "Enabled" : "Disabled");
    return count;
}


static ssize_t get_display_loop_mode(struct device *dev, struct device_attribute *attr, char *buf)
{
	unsigned int enb = 0;

	osdm_clk_enable(dev);
	dmw96osdm_loop_get(&enb);
	osdm_clk_disable(dev);

	PDEBUG("loop %s \n", (enb > 0) ? "Enabled" : "Disabled");
	return snprintf(buf, PAGE_SIZE, "%d\n", enb);
}

static ssize_t set_display_loop_mode(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	unsigned int enb = simple_strtoul(buf, NULL, 0);
	enb = (enb > 0) ? 1 : 0;

	osdm_clk_enable(dev);
	dmw96osdm_loop_set(enb);
	osdm_clk_disable(dev);

	PDEBUG("loop %s \n", (enb > 0) ? "Enabled" : "Disabled");
	return count;
}

static ssize_t get_display_config(struct device *dev, struct device_attribute *attr, char *buf)
{
	int x = 123;

	PDEBUG("%s(%d)\n",__FILE__,__LINE__);
	return snprintf(buf, PAGE_SIZE, "%d\n", x);
}

static ssize_t set_display_config(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	return count;
}

static ssize_t get_display_dithering(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int dith = osdm->last_planes[DISPLAY_PLANE].dithering;

	PDEBUG("Dithering %d \n",dith);
	return snprintf(buf, PAGE_SIZE, "%d\n", dith);
}

static ssize_t set_display_dithering(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int dith = simple_strtoul(buf, NULL, 0);

	osdm->planes[DISPLAY_PLANE].dithering = dith;
	PDEBUG("Dithering %d \n",dith);
	return count;
}

static ssize_t get_display_xres(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info display;
	memset(&display,0,sizeof(display));

	osdm_clk_enable(dev);
	dmw96osdm_display_dim_get(&display);
	osdm_clk_disable(dev);

	PDEBUG("%s(%d)\n",__FILE__,__LINE__);
	return snprintf(buf, PAGE_SIZE, "%d\n", display.var.xres);
}

static ssize_t set_display_xres(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	return -1;
}

static ssize_t get_display_yres(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info display;
	memset(&display,0,sizeof(display));

	osdm_clk_enable(dev);
	dmw96osdm_display_dim_get(&display);
	osdm_clk_disable(dev);

	PDEBUG("%s(%d)\n",__FILE__,__LINE__);
	return snprintf(buf, PAGE_SIZE, "%d\n", display.var.yres);
}

static ssize_t set_display_yres(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	return -1;
}

static ssize_t get_display_bg_enble(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int enb = osdm->last_planes[DISPLAY_PLANE].bg_enb;

	PDEBUG("background %s \n", (enb > 0) ? "Enabled" : "Disabled");
	return snprintf(buf, PAGE_SIZE, "%d\n", enb);
}

static ssize_t set_display_bg_enble(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int enb = simple_strtoul(buf, NULL, 0);

	enb = (enb > 0) ? 1 : 0;
	osdm->planes[DISPLAY_PLANE].bg_enb = enb;
	PDEBUG("background %s \n", (enb > 0) ? "Enabled" : "Disabled");
	return count;
}

static ssize_t get_display_bg_color(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
    osdm_rgb888 color = 0;

	color = osdm->last_planes[DISPLAY_PLANE].bg_color;
    PDEBUG("background red = 0x%4x green = 0x%4x blue = 0x%4x\n",get_red_from_rgb888(color) , get_green_from_rgb888(color) , 		get_blue_from_rgb888(color));
	
	return snprintf(buf, PAGE_SIZE, "0x%x\n", color);
}

static ssize_t set_display_bg_color(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	osdm_rgb888 color = simple_strtoul(buf, NULL, 0);;

	osdm->planes[DISPLAY_PLANE].bg_color = color;
    PDEBUG("background red = 0x%4x green = 0x%4x blue = 0x%4x\n",get_red_from_rgb888(color) , get_green_from_rgb888(color) , get_blue_from_rgb888(color));
	return count;
}

static ssize_t get_plane_enable(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	
	unsigned int enb = osdm->last_planes[plane_num].enb;
	PDEBUG("Plane %s is %s \n", get_plane_name(plane_num) ,(enb > 0) ? "Enabled" : "Disabled");

	return snprintf(buf, PAGE_SIZE, "%d\n", enb);
}

static ssize_t set_plane_enable(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int enb = simple_strtoul(buf, NULL, 0);
	unsigned int plane_num = get_plane_num(osdm,info);

	enb = (enb > 0) ? 1 : 0;
	osdm->planes[plane_num].enb = enb;

	PDEBUG("Plane %s is %s \n", get_plane_name(plane_num) ,(enb > 0) ? "Enabled" : "Disabled");
	return count;
}

static ssize_t get_plane_config(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	
	unsigned int config = osdm->last_planes[plane_num].config;
	PDEBUG("Plane %s config is  0x%x \n", get_plane_name(plane_num) ,config);
	return snprintf(buf, PAGE_SIZE, "0x%x\n", config);
}

static ssize_t set_plane_config(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int  config = simple_strtoul(buf, NULL, 0);
	unsigned int plane_num = get_plane_num(osdm,info);

    osdm->planes[plane_num].config = config;
	PDEBUG("Plane %s config  is 0x%x \n", get_plane_name(plane_num) ,config);
	return count;
}

static ssize_t get_chroma_key_enable(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	
	unsigned int chroma_enb = osdm->last_planes[plane_num].chroma_enb;
	PDEBUG("Plane %s chroma key is  %s \n", get_plane_name(plane_num) ,(chroma_enb > 0) ? "Enabled" : "Disabled");
	return snprintf(buf, PAGE_SIZE, "%d\n", chroma_enb);
}

static ssize_t set_chroma_key_enable(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int chroma_enb = simple_strtoul(buf, NULL, 0);
	unsigned int plane_num = get_plane_num(osdm,info);

	chroma_enb = (chroma_enb > 0) ? 1 : 0;
	osdm->planes[plane_num].chroma_enb = chroma_enb;
	PDEBUG("Plane %s chroma key is  %s \n", get_plane_name(plane_num) ,(chroma_enb > 0) ? "Enabled" : "Disabled");
	return count;
}


static ssize_t get_chroma_key(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	
	osdm_rgb888 chroma_key = osdm->last_planes[plane_num].chroma_key;
	PDEBUG("Plane %s chroma key is 0x%x \n", get_plane_name(plane_num) ,chroma_key );
	return snprintf(buf, PAGE_SIZE, "0x%x\n", chroma_key);
}

static ssize_t set_chroma_key(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	osdm_rgb888  chroma_key = simple_strtoul(buf, NULL, 0);
	unsigned int plane_num = get_plane_num(osdm,info);

    osdm->planes[plane_num].chroma_key = chroma_key;
	PDEBUG("Plane %s chroma key is 0x%x \n", get_plane_name(plane_num) ,chroma_key);
	return count;
}

static ssize_t get_pre_pix_enb(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);

	unsigned int pp_alpha_enb = osdm->last_planes[plane_num].per_pixel_alpha_enb;
	PDEBUG("Plane %s per pixel alpha is  %s \n", get_plane_name(plane_num) ,(pp_alpha_enb > 0) ? "Enabled" : "Disabled");
	return snprintf(buf, PAGE_SIZE, "%d\n", pp_alpha_enb);
}

static ssize_t set_pre_pix_enb(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int pp_alpha_enb = simple_strtoul(buf, NULL, 0);
	unsigned int plane_num = get_plane_num(osdm,info);

	pp_alpha_enb = (pp_alpha_enb > 0) ? 1 : 0;
	osdm->planes[plane_num].per_pixel_alpha_enb = pp_alpha_enb;
	PDEBUG("Plane %s per pixel alpha is  %s \n", get_plane_name(plane_num) ,(pp_alpha_enb > 0) ? "Enabled" : "Disabled");
	return count;
}

static ssize_t get_pre_plane_alpha(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);

	unsigned int pp_alpha = osdm->last_planes[plane_num].per_plane_alpha;
	PDEBUG("Plane %s per plane alpha is %d \n", get_plane_name(plane_num) ,pp_alpha );
	return snprintf(buf, PAGE_SIZE, "%d\n", pp_alpha);
}

static ssize_t set_pre_plane_alpha(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int pp_alpha = simple_strtoul(buf, NULL, 0);
	unsigned int plane_num = get_plane_num(osdm,info);

    osdm->planes[plane_num].per_plane_alpha = pp_alpha;
	PDEBUG("Plane %s per plane alpha is  %d \n", get_plane_name(plane_num) ,pp_alpha);
	return count;
}


static ssize_t get_xres(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	unsigned int xres = osdm->last_var[plane_num].xres;
	
	PDEBUG("Plane %s xres is  %d \n", get_plane_name(plane_num) ,xres);
	return snprintf(buf, PAGE_SIZE, "%d\n", xres);
}

static ssize_t set_xres(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	unsigned int xres = simple_strtoul(buf, NULL, 0);

	osdm->planes[plane_num].info->var.xres = xres;

	PDEBUG("Plane %s xres is  %d \n", get_plane_name(plane_num) ,xres);
	return count;
}

static ssize_t get_yres(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	unsigned int yres = osdm->last_var[plane_num].yres;
	
	PDEBUG("Plane %s yres is  %d \n", get_plane_name(plane_num) ,yres);
	return snprintf(buf, PAGE_SIZE, "%d\n", yres);
}

static ssize_t set_yres(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	unsigned int yres = simple_strtoul(buf, NULL, 0);

	osdm->planes[plane_num].info->var.yres = yres;

	PDEBUG("Plane %s yres is  %d \n", get_plane_name(plane_num) ,yres);
	return count;
}

static ssize_t get_xpos(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	unsigned int xpos = osdm->last_planes[plane_num].pos.xpos;
	PDEBUG("Plane %s xpos is %d \n", get_plane_name(plane_num) ,xpos );

	return snprintf(buf, PAGE_SIZE, "%d\n", xpos);
}

static ssize_t set_xpos(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	unsigned int xpos = simple_strtoul(buf, NULL, 0);

	osdm->planes[plane_num].pos.xpos = xpos;
	PDEBUG("Plane %s xops is  %d \n", get_plane_name(plane_num) ,xpos);
	return count;
}

static ssize_t get_ypos(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	unsigned int ypos = osdm->last_planes[plane_num].pos.ypos;
	PDEBUG("Plane %s ypos is %d \n", get_plane_name(plane_num) ,ypos );

	return snprintf(buf, PAGE_SIZE, "%d\n", ypos);
	
}

static ssize_t set_ypos(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	unsigned int ypos = simple_strtoul(buf, NULL, 0);

	osdm->planes[plane_num].pos.ypos = ypos;
	PDEBUG("Plane %s yops is  %d \n", get_plane_name(plane_num) ,ypos);
	return count;
}

static ssize_t get_brightness(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int brightness = 0;
	unsigned int plane_num = get_plane_num(osdm,info);
	if (plane_num == MAIN_VIDEO_PLANE || plane_num ==  SUB_VIDEO_PLANE )
		brightness = osdm->last_planes[plane_num].video_color_control.brightness;
    
	PDEBUG("Plane %s brightness is %d \n", get_plane_name(plane_num) ,brightness);
	
	return snprintf(buf, PAGE_SIZE, "%d\n", brightness);
}

static ssize_t set_brightness(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int brightness = simple_strtoul(buf, NULL, 0);
	unsigned int plane_num = get_plane_num(osdm,info);
	if (plane_num == MAIN_VIDEO_PLANE || plane_num ==  SUB_VIDEO_PLANE )
		osdm->planes[plane_num].video_color_control.brightness = brightness ;
    
	PDEBUG("Plane %s brightness is %d \n", get_plane_name(plane_num) ,brightness);

	return count;
}

static ssize_t get_contrast(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int contrast = 0;
	unsigned int plane_num = get_plane_num(osdm,info);
	if (plane_num == MAIN_VIDEO_PLANE || plane_num ==  SUB_VIDEO_PLANE )
		contrast = osdm->last_planes[plane_num].video_color_control.contrast;
    
	PDEBUG("Plane %s contrast is %d \n", get_plane_name(plane_num) ,contrast);
	
	return snprintf(buf, PAGE_SIZE, "%d\n", contrast);
}

static ssize_t set_contrast(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int contrast = simple_strtoul(buf, NULL, 0);
	unsigned int plane_num = get_plane_num(osdm,info);
	if (plane_num == MAIN_VIDEO_PLANE || plane_num ==  SUB_VIDEO_PLANE )
		osdm->planes[plane_num].video_color_control.contrast = contrast ;
    
	PDEBUG("Plane %s contrast is %d \n", get_plane_name(plane_num) ,contrast);

	return count;
}



static ssize_t get_saturation(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int saturation = 0;
	unsigned int plane_num = get_plane_num(osdm,info);
	if (plane_num == MAIN_VIDEO_PLANE || plane_num ==  SUB_VIDEO_PLANE )
		saturation = osdm->last_planes[plane_num].video_color_control.saturation;
    
	PDEBUG("Plane %s saturation is %d \n", get_plane_name(plane_num) ,saturation);
	
	return snprintf(buf, PAGE_SIZE, "%d\n", saturation);
}

static ssize_t set_saturation(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int saturation = simple_strtoul(buf, NULL, 0);
	unsigned int plane_num = get_plane_num(osdm,info);
	if (plane_num == MAIN_VIDEO_PLANE || plane_num ==  SUB_VIDEO_PLANE )
		osdm->planes[plane_num].video_color_control.saturation = saturation ;
    
	PDEBUG("Plane %s saturation is %d \n", get_plane_name(plane_num) ,saturation);

	return count;
}


static ssize_t get_hue(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int hue = 0;
	unsigned int plane_num = get_plane_num(osdm,info);
	if (plane_num == MAIN_VIDEO_PLANE || plane_num ==  SUB_VIDEO_PLANE )
		hue = osdm->last_planes[plane_num].video_color_control.hue;
    
	PDEBUG("Plane %s hue is %d \n", get_plane_name(plane_num) ,hue);
	
	return snprintf(buf, PAGE_SIZE, "%d\n", hue);
}

static ssize_t set_hue(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int hue = simple_strtoul(buf, NULL, 0);
	unsigned int plane_num = get_plane_num(osdm,info);
	if (plane_num == MAIN_VIDEO_PLANE || plane_num ==  SUB_VIDEO_PLANE )
		osdm->planes[plane_num].video_color_control.hue = hue ;
    
	PDEBUG("Plane %s hue is %d \n", get_plane_name(plane_num) ,hue);

	return count;
}

static ssize_t get_transp_xres(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	unsigned int xres = osdm->last_planes[plane_num].t_dim.xres ;

	PDEBUG("Plane %s transparent xres  is %d \n", get_plane_name(plane_num) ,xres);

	return snprintf(buf, PAGE_SIZE, "%d\n", xres);
}

static ssize_t set_transp_xres(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
    unsigned int plane_num = get_plane_num(osdm,info);
	unsigned int xres = simple_strtoul(buf, NULL, 0);

	osdm->planes[plane_num].t_dim.xres  = xres;
	PDEBUG("Plane %s transparent xres  is %d \n", get_plane_name(plane_num) ,xres);

	return count;
}

static ssize_t get_transp_yres(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
    unsigned int plane_num = get_plane_num(osdm,info);
	unsigned int yres = osdm->last_planes[plane_num].t_dim.yres ;

	PDEBUG("Plane %s transparent yres  is %d \n", get_plane_name(plane_num) ,yres);
	return snprintf(buf, PAGE_SIZE, "%d\n", yres);
}

static ssize_t set_transp_yres(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
    unsigned int plane_num = get_plane_num(osdm,info);
	unsigned int yres = simple_strtoul(buf, NULL, 0);

	osdm->planes[plane_num].t_dim.yres  = yres;
	PDEBUG("Plane %s transparent yres  is %d \n", get_plane_name(plane_num) ,yres);

	return count;
}

static ssize_t get_transp_xpos(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	unsigned int xpos = osdm->last_planes[plane_num].t_pos.xpos;
	PDEBUG("Plane %s transparent xpos is %d \n", get_plane_name(plane_num) ,xpos );

	return snprintf(buf, PAGE_SIZE, "%d\n", xpos);
}

static ssize_t set_transp_xpos(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	unsigned int xpos = simple_strtoul(buf, NULL, 0);

	osdm->planes[plane_num].t_pos.xpos = xpos;
	PDEBUG("Plane %s transparent xpos is  %d \n", get_plane_name(plane_num) ,xpos);
	return count;
}

static ssize_t get_transp_ypos(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	unsigned int ypos = osdm->last_planes[plane_num].t_pos.ypos;
	PDEBUG("Plane %s transparent ypos is %d \n", get_plane_name(plane_num) ,ypos );

	return snprintf(buf, PAGE_SIZE, "%d\n", ypos);
}

static ssize_t set_transp_ypos(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	unsigned int ypos = simple_strtoul(buf, NULL, 0);

	osdm->planes[plane_num].t_pos.ypos = ypos;
	PDEBUG("Plane %s transparent ypos is  %d \n", get_plane_name(plane_num) ,ypos);
	return count;
}

static ssize_t get_transp_pixel(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	osdm_abgr8888 t_pixel_color = osdm->last_planes[plane_num].t_pixel_color;

	PDEBUG("Plane %s transparent pixel is 0x%x \n", get_plane_name(plane_num) ,t_pixel_color );

	return snprintf(buf, PAGE_SIZE, "0x%x\n", t_pixel_color);
}

static ssize_t set_transp_pixel(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	osdm_abgr8888 t_pixel_color = simple_strtoul(buf, NULL, 0);
	osdm->planes[plane_num].t_pixel_color = t_pixel_color;

	PDEBUG("Plane %s transparent pixel is 0x%x \n", get_plane_name(plane_num) ,t_pixel_color );

	return count;
}

static ssize_t get_masking_left(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	unsigned int mask = osdm->last_planes[plane_num].mask.left;

	PDEBUG("Plane %s mask left is %d \n", get_plane_name(plane_num) ,mask );

	return snprintf(buf, PAGE_SIZE, "0x%x\n", mask);
}

static ssize_t set_masking_left(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	unsigned int mask = simple_strtoul(buf, NULL, 0);
	osdm->planes[plane_num].mask.left = mask;

	PDEBUG("Plane %s mask left is %d \n", get_plane_name(plane_num) ,mask );
	return count;
}

static ssize_t get_masking_right(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	unsigned int mask = osdm->last_planes[plane_num].mask.right;

	PDEBUG("Plane %s mask right is %d \n", get_plane_name(plane_num) ,mask );

	return snprintf(buf, PAGE_SIZE, "%d\n", mask);
}

static ssize_t set_masking_right(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	unsigned int mask = simple_strtoul(buf, NULL, 0);
	osdm->planes[plane_num].mask.right = mask;

	PDEBUG("Plane %s mask right is %d \n", get_plane_name(plane_num) ,mask );
	return count;
}

static ssize_t get_masking_top(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	unsigned int mask = osdm->last_planes[plane_num].mask.top;

	PDEBUG("Plane %s mask top is %d \n", get_plane_name(plane_num) ,mask );

	return snprintf(buf, PAGE_SIZE, "%d\n", mask);
}

static ssize_t set_masking_top(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	unsigned int mask = simple_strtoul(buf, NULL, 0);
	osdm->planes[plane_num].mask.top = mask;

	PDEBUG("Plane %s mask top is %d \n", get_plane_name(plane_num) ,mask );
	return count;
}

static ssize_t get_masking_bottom(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	unsigned int mask = osdm->last_planes[plane_num].mask.bottom;

	PDEBUG("Plane %s mask bottom is %d \n", get_plane_name(plane_num) ,mask );
	return snprintf(buf, PAGE_SIZE, "%d\n", mask);
}

static ssize_t set_masking_bottom(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int plane_num = get_plane_num(osdm,info);
	unsigned int mask = simple_strtoul(buf, NULL, 0);
	osdm->planes[plane_num].mask.bottom = mask;

	PDEBUG("Plane %s mask bottom is %d \n", get_plane_name(plane_num) ,mask );
	return count;

}

static ssize_t get_limit_luma_min(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int luma_min = 0;
    unsigned int plane_num = get_plane_num(osdm,info);
	if (plane_num == MAIN_VIDEO_PLANE || plane_num ==  SUB_VIDEO_PLANE )
		luma_min = osdm->last_planes[plane_num].video_limits.luma_min;
    
	PDEBUG("Plane %s luma min is %d \n", get_plane_name(plane_num) ,luma_min);
	
	return snprintf(buf, PAGE_SIZE, "%d\n", luma_min);
}

static ssize_t set_limit_luma_min(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int luma_min = simple_strtoul(buf, NULL, 0);
	unsigned int plane_num = get_plane_num(osdm,info);
	if (plane_num == MAIN_VIDEO_PLANE || plane_num ==  SUB_VIDEO_PLANE )
		osdm->planes[plane_num].video_limits.luma_min = luma_min ;
    
	PDEBUG("Plane %s luma min is %d \n", get_plane_name(plane_num) ,luma_min);

	return count;
}

static ssize_t get_limit_luma_max(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int luma_max = 0;
	unsigned int plane_num = get_plane_num(osdm,info);
	if (plane_num == MAIN_VIDEO_PLANE || plane_num ==  SUB_VIDEO_PLANE )
		luma_max = osdm->last_planes[plane_num].video_limits.luma_max;
    
	PDEBUG("Plane %s luma max is %d \n", get_plane_name(plane_num) ,luma_max);
	
	return snprintf(buf, PAGE_SIZE, "%d\n", luma_max);
}

static ssize_t set_limit_luma_max(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int luma_max = simple_strtoul(buf, NULL, 0);
	unsigned int plane_num = get_plane_num(osdm,info);
	if (plane_num == MAIN_VIDEO_PLANE || plane_num ==  SUB_VIDEO_PLANE )
		osdm->planes[plane_num].video_limits.luma_max = luma_max ;
    
	PDEBUG("Plane %s luma max is %d \n", get_plane_name(plane_num) ,luma_max);

	return count;
}

static ssize_t get_limit_chroma_min(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int chroma_min = 0;
	unsigned int plane_num = get_plane_num(osdm,info);
	if (plane_num == MAIN_VIDEO_PLANE || plane_num ==  SUB_VIDEO_PLANE )
		chroma_min = osdm->last_planes[plane_num].video_limits.chroma_min;
    
	PDEBUG("Plane %s chroma min is %d \n", get_plane_name(plane_num) ,chroma_min);
	
	return snprintf(buf, PAGE_SIZE, "%d\n", chroma_min);
}

static ssize_t set_limit_chroma_min(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int chroma_min = simple_strtoul(buf, NULL, 0);
	unsigned int plane_num = get_plane_num(osdm,info);
	if (plane_num == MAIN_VIDEO_PLANE || plane_num ==  SUB_VIDEO_PLANE )
		osdm->planes[plane_num].video_limits.chroma_min = chroma_min ;
    
	PDEBUG("Plane %s chroma min is %d \n", get_plane_name(plane_num) ,chroma_min);

	return count;
}

static ssize_t get_limit_chroma_max(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int chroma_max = 0;
	unsigned int plane_num = get_plane_num(osdm,info);
	if (plane_num == MAIN_VIDEO_PLANE || plane_num ==  SUB_VIDEO_PLANE )
		chroma_max = osdm->last_planes[plane_num].video_limits.chroma_max;
    
	PDEBUG("Plane %s chroma max is %d \n", get_plane_name(plane_num) ,chroma_max);
	
	return snprintf(buf, PAGE_SIZE, "%d\n", chroma_max);
}

static ssize_t set_limit_chroma_max(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int chroma_max = simple_strtoul(buf, NULL, 0);
	unsigned int plane_num = get_plane_num(osdm,info);
	if (plane_num == MAIN_VIDEO_PLANE || plane_num ==  SUB_VIDEO_PLANE )
		osdm->planes[plane_num].video_limits.chroma_max = chroma_max ;
    
	PDEBUG("Plane %s chroma max is %d \n", get_plane_name(plane_num) ,chroma_max);

	return count;
}

static ssize_t get_lut_base(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int lut_base = 0;
	unsigned int plane_num = get_plane_num(osdm,info);
	if (plane_num == MAIN_OSD_PLANE || plane_num ==  SUB_OSD_PLANE_1 || plane_num ==  SUB_OSD_PLANE_2 )
		lut_base = osdm->last_planes[plane_num].lut_base;
    
	PDEBUG("Plane %s lut_base is %d \n", get_plane_name(plane_num) ,lut_base);
	
	return snprintf(buf, PAGE_SIZE, "%d\n", lut_base);
}

static ssize_t set_lut_base(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int lut_base = simple_strtoul(buf, NULL, 0);
	unsigned int plane_num = get_plane_num(osdm,info);
	if (plane_num == MAIN_OSD_PLANE || plane_num ==  SUB_OSD_PLANE_1 || plane_num ==  SUB_OSD_PLANE_2 )
		osdm->planes[plane_num].lut_base = lut_base ;
    
	PDEBUG("Plane %s lut_base is %d \n", get_plane_name(plane_num) ,lut_base);

	return count;
}

static ssize_t get_dma0_byte_order(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int order = 0;
	unsigned int plane_num = get_plane_num(osdm,info);
    order = osdm->last_planes[plane_num].byte_order[0];
    
	PDEBUG("Channel 0  %s byte order is %d \n", get_plane_name(plane_num) ,order);
	
	return snprintf(buf, PAGE_SIZE, "%d\n", order);
}

static ssize_t set_dma0_byte_order(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int order = simple_strtoul(buf, NULL, 0);
	unsigned int plane_num = get_plane_num(osdm,info);
    osdm->planes[plane_num].byte_order[0] = order ;
    
	PDEBUG("Channel 0  %s byte order is %d \n", get_plane_name(plane_num) ,order);

	return count;
}

static ssize_t get_dma1_byte_order(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int order = 0;
	unsigned int plane_num = get_plane_num(osdm,info);
	if (plane_num == MAIN_VIDEO_PLANE || plane_num ==  SUB_VIDEO_PLANE )
		order = osdm->last_planes[plane_num].byte_order[1];
    
	PDEBUG("Channel 1  %s byte order is %d \n", get_plane_name(plane_num) ,order);
	
	return snprintf(buf, PAGE_SIZE, "%d\n", order);
}

static ssize_t set_dma1_byte_order(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int order = simple_strtoul(buf, NULL, 0);
	unsigned int plane_num = get_plane_num(osdm,info);
	if (plane_num == MAIN_VIDEO_PLANE || plane_num ==  SUB_VIDEO_PLANE )
		osdm->planes[plane_num].byte_order[1] = order ;
    
	PDEBUG("Channel 1  %s byte order is %d \n", get_plane_name(plane_num) ,order);

	return count;
}

static ssize_t get_dma2_byte_order(struct device *dev, struct device_attribute *attr, char *buf)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int order = 0;
	unsigned int plane_num = get_plane_num(osdm,info);
	if (plane_num == MAIN_VIDEO_PLANE || plane_num ==  SUB_VIDEO_PLANE )
		order = osdm->last_planes[plane_num].byte_order[2];
    
	PDEBUG("Channel 2  %s byte order is %d \n", get_plane_name(plane_num) ,order);
	
	return snprintf(buf, PAGE_SIZE, "%d\n", order);
}

static ssize_t set_dma2_byte_order(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	struct fb_info *info = dev_get_drvdata(dev);
	struct dmw96osdm* osdm = FB2OOSDM(info);
	unsigned int order = simple_strtoul(buf, NULL, 0);
	unsigned int plane_num = get_plane_num(osdm,info);
	if (plane_num == MAIN_VIDEO_PLANE || plane_num ==  SUB_VIDEO_PLANE )
		osdm->planes[plane_num].byte_order[2] = order ;
    
	PDEBUG("Channel 2  %s byte order is %d \n", get_plane_name(plane_num) ,order);

	return count;
}

static struct device_attribute dmw96osdm_attrs[] = {

__ATTR(display_pause, S_IRUGO | S_IWUSR, get_display_pause, set_display_pause),
__ATTR(display_loop_mode, S_IRUGO | S_IWUSR, get_display_loop_mode, set_display_loop_mode),
__ATTR(display_config, S_IRUGO | S_IWUSR, get_display_config, set_display_config),
__ATTR(display_dithering, S_IRUGO | S_IWUSR, get_display_dithering, set_display_dithering),
__ATTR(display_xres, S_IRUGO | S_IWUSR, get_display_xres, set_display_xres),
__ATTR(display_yres, S_IRUGO | S_IWUSR, get_display_yres, set_display_yres),
__ATTR(display_bg_enble, S_IRUGO | S_IWUSR, get_display_bg_enble, set_display_bg_enble),
__ATTR(display_bg_color, S_IRUGO | S_IWUSR, get_display_bg_color, set_display_bg_color),

/* Video / OSD specific */ 
__ATTR(enable, S_IRUGO | S_IWUSR, get_plane_enable, set_plane_enable),
__ATTR(config, S_IRUGO | S_IWUSR, get_plane_config, set_plane_config),
__ATTR(chroma_key_enable, S_IRUGO | S_IWUSR, get_chroma_key_enable, set_chroma_key_enable),
__ATTR(chroma_key, S_IRUGO | S_IWUSR, get_chroma_key, set_chroma_key),
__ATTR(per_pix_alpha_enb, S_IRUGO | S_IWUSR, get_pre_pix_enb, set_pre_pix_enb),
__ATTR(per_plane_alpha, S_IRUGO | S_IWUSR, get_pre_plane_alpha, set_pre_plane_alpha),
__ATTR(xdim, S_IRUGO | S_IWUSR, get_xres, set_xres),
__ATTR(ydim, S_IRUGO | S_IWUSR, get_yres, set_yres),
__ATTR(xpos, S_IRUGO | S_IWUSR, get_xpos, set_xpos),
__ATTR(ypos, S_IRUGO | S_IWUSR, get_ypos, set_ypos),
__ATTR(brightness, S_IRUGO | S_IWUSR, get_brightness, set_brightness),
__ATTR(contrast, S_IRUGO | S_IWUSR, get_contrast, set_contrast),
__ATTR(saturation, S_IRUGO | S_IWUSR, get_saturation, set_saturation),
__ATTR(hue, S_IRUGO | S_IWUSR, get_hue, set_hue),
__ATTR(transp_xres, S_IRUGO | S_IWUSR, get_transp_xres, set_transp_xres),
__ATTR(transp_yres, S_IRUGO | S_IWUSR, get_transp_yres, set_transp_yres),
__ATTR(transp_xpos, S_IRUGO | S_IWUSR, get_transp_xpos, set_transp_xpos),
__ATTR(transp_ypos, S_IRUGO | S_IWUSR, get_transp_ypos, set_transp_ypos),
__ATTR(transp_pixel, S_IRUGO | S_IWUSR, get_transp_pixel, set_transp_pixel),
__ATTR(masking_left, S_IRUGO | S_IWUSR, get_masking_left, set_masking_left),
__ATTR(masking_right, S_IRUGO | S_IWUSR, get_masking_right, set_masking_right),
__ATTR(masking_top, S_IRUGO | S_IWUSR, get_masking_top, set_masking_top),
__ATTR(masking_bottom, S_IRUGO | S_IWUSR, get_masking_bottom, set_masking_bottom),
__ATTR(limit_luma_min, S_IRUGO | S_IWUSR, get_limit_luma_min, set_limit_luma_min),
__ATTR(limit_luma_max, S_IRUGO | S_IWUSR, get_limit_luma_max, set_limit_luma_max),
__ATTR(limit_chroma_min, S_IRUGO | S_IWUSR, get_limit_chroma_min, set_limit_chroma_min),
__ATTR(limit_chroma_max, S_IRUGO | S_IWUSR, get_limit_chroma_max, set_limit_chroma_max),
__ATTR(lut_base, S_IRUGO | S_IWUSR, get_lut_base, set_lut_base),

/*DMA based settings*/
__ATTR(dma0_byte_order, S_IRUGO | S_IWUSR, get_dma0_byte_order, set_dma0_byte_order),
__ATTR(dma1_byte_order, S_IRUGO | S_IWUSR, get_dma1_byte_order, set_dma1_byte_order),
__ATTR(dma2_byte_order, S_IRUGO | S_IWUSR, get_dma2_byte_order, set_dma2_byte_order),


};

int create_osdm_sysfs(struct fb_info *fbi)
{
	int ret = 0;
	int i = 0;

	for (i = 0; i < ARRAY_SIZE(dmw96osdm_attrs); i++) {
		ret = device_create_file(fbi->dev, &dmw96osdm_attrs[i]);
		if (ret) {
			dev_err(fbi->dev, "failed to create sysfs file\n");
			return ret;
		}
	}

	return 0;
}

void remove_osdm_sysfs(struct fb_info *fbi)
{
	int i = 0;

    for (i = 0; i < ARRAY_SIZE(dmw96osdm_attrs); i++) {
		device_remove_file(fbi->dev, &dmw96osdm_attrs[i]);
	}
}
