#ifdef __KERNEL__
	#include <linux/module.h>
    #include <linux/kernel.h>
	#include <linux/io.h>
#else
	#include <stdio.h>
	#include <string.h>
#endif


#include "dmw96osdm.h"

#ifdef __KERNEL__
	void __iomem *osdm_regs;
#endif

unsigned short is_display_set = 0 ;
#define OFFSET_ARRAY_LEN				20


static const unsigned short osdm_offset_val[6][OFFSET_ARRAY_LEN]= {
{VIDEO0_CONFIG ,VIDEO0_POS, VIDEO0_DIM, VIDEO0_BCSH, VIDEO0_TPOS, VIDEO0_TDIM, VIDEO0_TPIXEL, VIDEO0_ALCK, VIDEO0_YUVLIM}, 
{VIDEO1_CONFIG ,VIDEO1_POS, VIDEO1_DIM, VIDEO1_BCSH, VIDEO1_TPOS, VIDEO1_TDIM, VIDEO1_TPIXEL, VIDEO1_ALCK, VIDEO1_YUVLIM},
{OSD0_CONFIG, OSD0_POS, OSD0_DIM, OSD0_TPOS, OSD0_TDIM, OSD0_TPIXEL, OSD0_ALCK},
{OSD1_CONFIG, OSD1_POS, OSD1_DIM, OSD1_TPOS, OSD1_TDIM, OSD1_TPIXEL, OSD1_ALCK},
{OSD2_CONFIG, OSD2_POS, OSD2_DIM, OSD2_TPOS, OSD2_TDIM, OSD2_TPIXEL, OSD2_ALCK}
};


static const unsigned short osdm_dma_val[10][10][OFFSET_ARRAY_LEN] = {

	{ {DMA1_CONFIG, DMA1_BADDR0, DMA1_BADDR1, DMA1_STRIDE, DMA1_BSIZE, DMA1_BCOUNT, DMA1_TPOS, DMA1_TDIM}, /* Main Video 0 */
	  {DMA2_CONFIG, DMA2_BADDR0, DMA2_BADDR1, DMA2_STRIDE, DMA2_BSIZE, DMA2_BCOUNT, DMA2_TPOS, DMA2_TDIM}, /* Main Video 1 */
	  {DMA3_CONFIG, DMA3_BADDR0, DMA3_BADDR1, DMA3_STRIDE, DMA3_BSIZE, DMA3_BCOUNT, DMA3_TPOS, DMA3_TDIM},}, /* Main Video 2 */

	{ {DMA4_CONFIG, DMA4_BADDR0, DMA4_BADDR1, DMA4_STRIDE, DMA4_BSIZE, DMA4_BCOUNT, DMA4_TPOS, DMA4_TDIM}, /* Sub Video 0 */
	  {DMA5_CONFIG, DMA5_BADDR0, DMA5_BADDR1, DMA5_STRIDE, DMA5_BSIZE, DMA5_BCOUNT, DMA5_TPOS, DMA5_TDIM}, /* Sub Video 1 */
	  {DMA6_CONFIG, DMA6_BADDR0, DMA6_BADDR1, DMA6_STRIDE, DMA6_BSIZE, DMA6_BCOUNT, DMA6_TPOS, DMA6_TDIM},}, /* Sub Video 2 */

	{ {DMA7_CONFIG, DMA7_BADDR0, DMA7_BADDR1, DMA7_STRIDE, DMA7_BSIZE, DMA7_BCOUNT, DMA7_TPOS, DMA7_TDIM},}, /* Main OSD */

	{ {DMA8_CONFIG, DMA8_BADDR0, DMA8_BADDR1, DMA8_STRIDE, DMA8_BSIZE, DMA8_BCOUNT, DMA8_TPOS, DMA8_TDIM},}, /* Sub OSD 1 */

	{ {DMA9_CONFIG, DMA9_BADDR0, DMA9_BADDR1, DMA9_STRIDE, DMA9_BSIZE, DMA9_BCOUNT, DMA9_TPOS, DMA9_TDIM},},  /* Sub OSD 2 */

	{ {DMA0_CONFIG, DMA0_BADDR0, DMA0_BADDR1, DMA0_STRIDE, DMA0_BSIZE, DMA0_BCOUNT, DMA0_TPOS, DMA0_TDIM},}, /* Display */

};

/************************************************************************
	dmw96osdm_osdm_get_bpp

	get the bit per pixel value from the enum

************************************************************************/
unsigned int dmw96osdm_osdm_get_bpp(unsigned int config){

	if (IS_VIDEO_VALID_CONFIG(config)) {

		switch (OSDM_PLANE_FORMAT(config)) {

			/*YUV 420*/
			case 0:
			case 1:
				return 12;
			/*YUV 422 or RGB565 or RGB4444 */
            case 2:
			case 4:
			case 5:
				return 16;
			/*YUV 444*/
			case 3:
				return 24;
			/*RGBA 32*/
			case 6:
				return 32;
			default:
				return 0;
		};
	}
	if (IS_OSD_VALID_CONFIG(config)) {

		switch (OSDM_PLANE_FORMAT(config)) {
			/*1 bpp palette*/
			case 0:
				return 1;
			/*2 bpp palette*/
			case 1:
				return 2;
			/*4 bpp palette*/
			case 2:
				return 4;
			/*8 bpp palette*/
			case 3:
				return 8;
			/*RGB565 or RGB4444*/
			case 4:
			case 5:
				return 16;
			/*RGBA32*/
			case 6:
				return 32;
			default:
				return 0;
		};
	}
	if (IS_DISPLAY_VALID_CONFIG(config)) {

		switch (OSDM_PLANE_FORMAT(config)) {
			case 0:
				/*RGB888*/
				return 24;
			case 1:
			case 3:
				/*RGB565 or YUV422*/
				return 16;
			default:
				return 0;
		};
	}
	else {
		return 0;
	}	
	return 0;
}

/*****************************************************************
	dmw96osdm_display_enb_set

    OSDM mixer enable: 0= disable OSDM mixer; 1= enable OSDM mixer operation.
	When this bit is cleared OSDM mixer returns to its initial (disabled) state;mixer output FIFO is cleared. 
	This bit does not affect video/OSD planes or DMA channels, which are controlled by separate configuration bits.

*****************************************************************/
int dmw96osdm_display_enb_set(unsigned int enb){

	unsigned int config = IOR32_OSDM(OSDM_CONFIG) & ~0x1;

	PDEBUG("%s \n", enb  > 0 ? "Enabled" : "Disabled");

	PDEBUG("config = 0x%x \n", ( config | (unsigned int)((enb > 0) ? 0x1 : 0x0)) );

	IOW32_OSDM(OSDM_CONFIG,  config | (unsigned int)((enb > 0) ? 0x1 : 0x0  ));

	return SUCCESS;

}


/*****************************************************************
	display_pause

    If PAUSE configuration bit is set to '1', DMAC completes any pending AHB bus transaction and transparent
	transfers but will not initiate any new AHB transactions or transparent transfers as along as PAUSE=1.
	After all pending transactions complete PAUSE interrupt status is set to '1'. 
	PAUSE interrupt status can only be cleared by setting PAUSE=0 in the configuration register.

*****************************************************************/
int dmw96osdm_pause_set(unsigned int enb){

	unsigned int config = IOR32_OSDM(OSDM_CONFIG) & (~0x2);

    IOW32_OSDM(OSDM_CONFIG,  config | (unsigned int)((enb > 0) ? 0x2 : 0x0  ));

	return SUCCESS;

}
/*****************************************************************

*****************************************************************/
int dmw96osdm_pause_get(unsigned int* enb){

	*enb = ((IOR32_OSDM(OSDM_CONFIG) & (0x2) ) >> 1);

	return SUCCESS;
}


/*****************************************************************
	display_loop

	If set, OSDM continuously renders frames as new plane input data buffers become available.
	Otherwise rendering stops after each frame, until DPINTR status bit is cleared.

*****************************************************************/
int dmw96osdm_loop_set(unsigned int enb){

	unsigned int config = IOR32_OSDM(OSDM_CONFIG) &  ~0x8;

	IOW32_OSDM(OSDM_CONFIG,  (config | (unsigned int)((enb > 0) ? 0x8 : 0x0  )) );

	return SUCCESS;

}
/*****************************************************************

*****************************************************************/
int dmw96osdm_loop_get(unsigned int* enb){

	*enb = ((IOR32_OSDM(OSDM_CONFIG) &  0x8) >> 3);

	return SUCCESS;
}
/*****************************************************************
	dmw96osdm_display__format_and_pkmod_set

	Display plane format: 0= RGB24 (888); 1= RGB16 (565); 2=reserved; 3=YUV422.
	Pixel packing mode: see the dmw96osdm_common 

*****************************************************************/
void dmw96osdm_display_format_and_pkmod_set(unsigned int config)
{
	unsigned int t_config = 0;
	t_config = IOR32_OSDM(OSDM_CONFIG) & ~0xf030;
	PDEBUG("config = 0x%x t_config = 0x%x\n", config, t_config);
	t_config |= (config) & ~0xf0000000;
	PDEBUG("t_config = 0x%x\n", t_config);

	IOW32_OSDM(OSDM_CONFIG,  t_config);
}

void dmw96osdm_display_format_and_pkmod_get(unsigned int *config)
{
	*config = IOR32_OSDM(OSDM_CONFIG);
	*config |= 0x40000000;
	*config &= 0x4000f030;
}

/***********************************************************************
	display_dithering

	Dithering gain:
	0= 0.00; 1= 0.25; 2= 0.50; 3= 0.75; 4= 1.00.
	If this field is set to 0, dithering disabled. 
	Normal gain value to activate dithering is 1.00 (field=4).
    
************************************************************************/
int dmw96osdm_display_dithering_set(unsigned int dithering)
{
	unsigned int config = 0 ;

	if (dithering != 0 && dithering != 1 && dithering != 2 && dithering != 3 && dithering != 4) {
		PDEBUG("Fail to set dithering invalid value 0x%x\n", dithering);
		return FAIL;
	}

	PDEBUG("dithering = %d\n" , dithering);

	config = IOR32_OSDM(OSDM_CONFIG) &  ~0x700;
	IOW32_OSDM(OSDM_CONFIG,  config | dithering << 8);
	return SUCCESS;
}

/*****************************************************************
	dmw96osdm_is_mixer_finished
******************************************************************/
unsigned int dmw96osdm_is_mixer_finished(void){

	return (IOR32_OSDM(INTR_STATUS) & 0x1);	
}

/*****************************************************************
	dmw96osdm_is_display_finished
******************************************************************/
unsigned int dmw96osdm_is_display_finished(void){

	//Check if mixer output dma channel and the display was finished
	return (IOR32_OSDM(INTR_CAUSE) & 0x11) == 0x11;
}

/*****************************************************************
	dmw96osdm_interrupt_enable

	Enable getting interrupts

	A set mask bit indicates the corresponding interrupt status is enabled and can cause OSDM interrupt signal assertion.

	0: Mask display plane interrupt.
	1: Mask pause interrupt.
	2 - 3: Reserved. Read as 0.
	4: Mask DMA channel 0 interrupt on first base address.
	5: Mask DMA channel interrupt on first base address.
	6: Mask DMA channel interrupt on first base address.
	7: Mask DMA channel interrupt on first base address.
	8: Mask DMA channel interrupt on first base address.
	9: Mask DMA channel interrupt on first base address.
	10: Mask DMA channel interrupt on first base address.
	11: Mask DMA channel interrupt on first base address.
	12: Mask DMA channel interrupt on first base address.
	13: Mask DMA channel interrupt on first base address.
	14 - 15: Reserved. Read as 0.
	16: Mask DMA channel interrupt on second base address.
	17: Mask DMA channel interrupt on second base address.
	18: Mask DMA channel interrupt on second base address.
	19: Mask DMA channel interrupt on second base address.
	20: Mask DMA channel interrupt on second base address.
	21: Mask DMA channel interrupt on second base address.
	22: Mask DMA channel interrupt on second base address.
	23: Mask DMA channel interrupt on second base address.
	24: Mask DMA channel interrupt on second base address.
	25: Mask DMA channel interrupt on second base address.
	26 - 31: Reserved. Read as 0.

*****************************************************************/
int dmw96osdm_interrupt_enable(unsigned int plane_type, unsigned int pause)
{
	unsigned int mask = 0;
	if (!IS_PLANE_TYPE_VALID(plane_type) && plane_type != DISPLAY_PLANE)
	{
		PDEBUG("Inalid plane type\n");
		return FAIL;
	}

    mask = IOR32_OSDM(INTR_MASK);

	PDEBUG("mask = 0x%x\n",mask);

	/*yet unmask only the first address*/
	switch (plane_type) {
		case DISPLAY_PLANE:
			mask |= (0x1 << 4) | 0x1;
			break;
		case MAIN_VIDEO_PLANE:
			mask |= 0x70;
			break;
		case SUB_VIDEO_PLANE:
			mask |= 0x380;
			break;
		case MAIN_OSD_PLANE:
			mask |= 0x800;
			break;
		case SUB_OSD_PLANE_1:
			mask |= 0x1000;
			break;
		case SUB_OSD_PLANE_2:
			mask |= 0x2000;
			break;
	}

	/*Enable pause*/
	(pause) ? (mask |= 0x2) : 0;

	PDEBUG("INTR_MASK = 0x%x\n",mask);
	IOW32_OSDM(INTR_MASK, mask);

	return SUCCESS;
}


/*****************************************************************
	dmw96osdm_interrupt_clear

	Clear interrupt

*****************************************************************/
void dmw96osdm_interrupt_clear(unsigned int cause)
{
	IOW32_OSDM(INTR_CLEAR, cause);
}


/*****************************************************************
	dmw96osdm_interrupt_clear

	Clear interrupt

*****************************************************************/
void dmw96osdm_interrupt_cause(unsigned int* cause)
{
	*cause = IOR32_OSDM(INTR_CAUSE);
}

/*****************************************************************
	dmw96osdm_interrupt_error

	handle error above getting interrupts

*****************************************************************/
void dmw96osdm_interrupt_error(unsigned int error)
{
	if (error & (1 << 4)) {
		PDEBUG("Error channel 0 first addres\n");
	}
	if (error & (1 << 5)) {
		PDEBUG("Error channel 1 first addres\n");
	}
	if (error & (1 << 6)) {
		PDEBUG("Error channel 2 first addres\n");
	}
	if (error & (1 << 7)) {
		PDEBUG("Error channel 3 first addres\n");
	}
	if (error & (1 << 8)) {
		PDEBUG("Error channel 4 first addres\n");
	}
	if (error & (1 << 9)) {
		PDEBUG("Error channel 5 first addres\n");
	}
	if (error & (1 << 10)) {
		PDEBUG("Error channel 6 first addres\n");
	}
	if (error & (1 << 11)) {
		PDEBUG("Error channel 7 first addres\n");
	}
	if (error & (1 << 12)) {
		PDEBUG("Error channel 8 first addres\n");
	}
	if (error & (1 << 13)) {
		PDEBUG("Error channel 9 first addres\n");
	}
	if (error & (1 << 16)) {
		PDEBUG("Error channel 0 second addres\n");
	}
	if (error & (1 << 17)) {
		PDEBUG("Error channel 1 second addres\n");
	}
	if (error & (1 << 18)) {
		PDEBUG("Error channel 2 second addres\n");
	}
	if (error & (1 << 19)) {
		PDEBUG("Error channel 3 second addres\n");
	}
	if (error & (1 << 20)) {
		PDEBUG("Error channel 4 second addres\n");
	}
	if (error & (1 << 21)) {
		PDEBUG("Error channel 5 second addres\n");
	}
	if (error & (1 << 22)) {
		PDEBUG("Error channel 6 second addres\n");
	}
	if (error & (1 << 23)) {
		PDEBUG("Error channel 7 second addres\n");
	}
	if (error & (1 << 24)) {
		PDEBUG("Error channel 8 second addres\n");
	}
	if (error & (1 << 25)) {
		PDEBUG("Error channel 9 second addres\n");
	}

}




/*****************************************************************
	dmw96osdm_interrupt_handle

	handle interrupt

*****************************************************************/
int dmw96osdm_interrupt_handle(void)
{
    unsigned int error = IOR32_OSDM(DMA_ERROR);
	dmw96osdm_interrupt_error(error);

    dmw96osdm_interrupt_clear(0xffffffff);
    return SUCCESS;
}

/***********************************************************************
	dmw96osdm_display_validate

	There is a limitation on image size of display plane as follows.

    
************************************************************************/
int dmw96osdm_display_validate(const struct fb_info* info, const unsigned int config){

	if(!IS_DISPLAY_VALID_CONFIG(config)) {

		PDEBUG("Invalid display config 0x%x\n",config);
		return FAIL;
	}

	if ( (info->var.xres > 0x7ff) || (info->var.yres > 0x7ff) ) {

		PDEBUG("Invalid valuse for Display. Width or hight excceed limits\n");
		return FAIL;
	}

	return SUCCESS;
}

/***********************************************************************
	dmw96osdm_reset_all_plane_settings

	reset all the plane position and dimention and set them to 0
    
************************************************************************/
void dmw96osdm_reset_all_plane_settings(void)
{
	unsigned int i = 0;

	/* Stop the mixing operation */
	dmw96osdm_display_enb_set(0);
	/* Dissable the dma for display */
	dmw96osdm_display_dma_enb_set(0);

	/* Passing all the DMA channels except the DISPLAY */
	for (i = 0 ; i < NUM_OF_IN_PLANES ; i++)
	{
		/* Dissable the each plane */
		dmw96osdm_plane_enb_set( 0 , i);

		/* Reset the dimention and the configuration of each plane at the mixer */
		IOW32_OSDM( osdm_offset_val[i][PLANEn_DIM] , 0x0 );
		IOW32_OSDM(osdm_offset_val[i][PLANEn_CONFIG], 0x0);

		/* This case is for Main and Sub video */
		if (i <= SUB_VIDEO_PLANE)
		{
			IOW32_OSDM( osdm_dma_val[i][2][DMAn_CONFIG] , 0x0 );
			IOW32_OSDM( osdm_dma_val[i][2][DMAn_STRIDE] , 0x0 );
			IOW32_OSDM( osdm_dma_val[i][2][DMAn_BSIZE]  , 0x0 );
			IOW32_OSDM( osdm_dma_val[i][2][DMAn_BCOUNT] , 0x0 );
	
			IOW32_OSDM( osdm_dma_val[i][1][DMAn_CONFIG] , 0x0 );
			IOW32_OSDM( osdm_dma_val[i][1][DMAn_STRIDE] , 0x0 );
			IOW32_OSDM( osdm_dma_val[i][1][DMAn_BSIZE]  , 0x0);
			IOW32_OSDM( osdm_dma_val[i][1][DMAn_BCOUNT] , 0x0 );
		}

		IOW32_OSDM( osdm_dma_val[i][0][DMAn_CONFIG] , 0x0 );
		IOW32_OSDM( osdm_dma_val[i][0][DMAn_STRIDE] , 0x0 );
		IOW32_OSDM( osdm_dma_val[i][0][DMAn_BSIZE]  , 0x0 );
		IOW32_OSDM( osdm_dma_val[i][0][DMAn_BCOUNT] , 0x0 );

	}
}


/************************************************************************
	dmw96osdm_display_dim_get

	Get the Display size

************************************************************************/
void dmw96osdm_display_dim_get(struct fb_info* info){

	unsigned int display_dim = IOR32_OSDM( DISPLAY_DIM );

	//Getting the width (X size) of the display
    info->var.xres = (0x7ff & display_dim) ;
	PDEBUG("OSDM display xsize = %d\n", info->var.xres );
	//Getting the high (Y size) of the display
	info->var.yres = 0x7ff & ( display_dim  >> 16 );
	PDEBUG("OSDM display ysize = %d\n", info->var.yres );
}

/***********************************************************************
	dmw96osdm_display_dim_set

	Set the dimentions of the Display plane.
************************************************************************/
void dmw96osdm_display_dim_set(const struct fb_info* info ){

	PDEBUG("OSDM display xres = %d\n", info->var.xres );
	PDEBUG("OSDM display yres = %d\n", info->var.yres );
	PDEBUG("DISPLAY_DIM = 0x%x\n", (info->var.yres) << 16 |  info->var.xres );
    IOW32_OSDM( DISPLAY_DIM , (info->var.yres) << 16 |  info->var.xres );
}

/***********************************************************************
	dmw96osdm_display_address_set

    Set the address of the display
    
************************************************************************/
void dmw96osdm_display_address_set(const struct fb_info* info){

	PDEBUG("DMA0_BADDR0 = 0x%x\n", (unsigned int)info->fix.smem_start );
	IOW32_OSDM( DMA0_BADDR0 , (unsigned int)info->fix.smem_start);
}


/***********************************************************************
	dmw96osdm_display_address_set

    Get the address of the display
    
************************************************************************/
void dmw96osdm_display_address_get(struct fb_info* info){

	info->fix.smem_start = IOR32_OSDM( DMA0_BADDR0);
}

void dmw96osdm_display_dma_enb_set(const unsigned int enb)
{
	unsigned int cfg = IOR32_OSDM( DMA0_CONFIG) & ~(0x1);
    IOW32_OSDM(DMA0_CONFIG, (enb > 0) ? (cfg | 0x1) : (cfg & ~0x1));
	PDEBUG("%s \n", enb  > 0 ? "Enabled" : "Disabled");
}

void dmw96osdm_display_dma_set(const struct fb_info* info , const unsigned int config)
{
	unsigned int display_bits = 0 ;
	unsigned int bpp = 0;

	bpp = dmw96osdm_osdm_get_bpp(config);
	display_bits = info->var.yres * info->var.xres * bpp;

	IOW32_OSDM( DMA0_STRIDE , ( (display_bits + 31)/ 32 ) * 4 );
	IOW32_OSDM( DMA0_BSIZE , ( (display_bits + 31)/ 32 ) );
	IOW32_OSDM( DMA0_BCOUNT , 1);
}

/***********************************************************************
	dmw96osdm_display_set

	Set the parameters of the Display plane.

	1) set the size of the display
	2) set the DMA channel for the display
	3) configure the type of the display
    
************************************************************************/
int dmw96osdm_display_set(const struct fb_info* info , const unsigned int config){

	int res = SUCCESS;

	res = dmw96osdm_display_validate(info , config);
	if (res == SUCCESS) {
		/* Reset all the other plane settings */
		//dmw96osdm_reset_all_plane_settings();

		dmw96osdm_display_format_and_pkmod_set(config);
	
		dmw96osdm_display_dim_set(info);

		dmw96osdm_interrupt_enable(DISPLAY_PLANE, 0);

		dmw96osdm_display_address_set(info);

		dmw96osdm_display_dma_set(info,config);

        PDEBUG("config = 0x%x, dim = 0x%x disp_addr = 0x%x\n",
			IOR32_OSDM(OSDM_CONFIG), IOR32_OSDM(DISPLAY_DIM), IOR32_OSDM(DMA0_BADDR0));
	}

	return res;
}

int dmw96osdm_display_get(struct fb_info *display)
{
	dmw96osdm_display_format_and_pkmod_get(&(display->var.reserved[0]));
	dmw96osdm_display_dim_get(display);
	dmw96osdm_display_address_get(display);
	return 0;
}

/************************************************************************
	dmw96osdm_display_background_set

	Setting the background color of the display 
		

************************************************************************/
int dmw96osdm_display_background_set(const osdm_rgb888 background){

	/*Read in to keep the enable status*/
	unsigned int bg_en =  IOR32_OSDM(DISPLAY_BGC) & 0x00000001;

    PDEBUG("Background color R=%d G=%d B=%d\n",get_red_from_rgb888(background),
														get_green_from_rgb888(background),
														get_blue_from_rgb888(background));

	IOW32_OSDM(DISPLAY_BGC, (background << 8) | bg_en );
		
	return SUCCESS;
}

/************************************************************************
	dmw96osdm_display_background_get

	Getting the background color
	
************************************************************************/
int dmw96osdm_display_background_get(osdm_rgb888* background){

	*background = 0xffffff & ( IOR32_OSDM(DISPLAY_BGC) >> 8);

	PDEBUG("Background color R=%d G=%d B=%d\n",get_red_from_rgb888(*background),
														get_green_from_rgb888(*background),
														get_blue_from_rgb888(*background));

	return SUCCESS;
}

/************************************************************************
	dmw96osdm_display_background_enable

	Setting whether the background color is enabled or dissabled
	
************************************************************************/
int dmw96osdm_display_background_enable(unsigned int is_enb){

	unsigned int bg = IOR32_OSDM(DISPLAY_BGC);

    IOW32_OSDM(DISPLAY_BGC, (is_enb > 0) ? (bg | 0x1) : (bg & ~0x1));
	PDEBUG("Background color is %s\n", (is_enb > 0) ? "enabled" : "disabled");

	return SUCCESS;
		 
}


/************************************************************************
	osdm_validate_position

	validate basics coordinates settings
	y/x within the display boundry

*************************************************************************/
int dmw96osdm_validate_pos_and_dim(const struct fb_info* info,const struct osdm_position* pos ,const unsigned int config)
{
	struct fb_info display;
	unsigned int cond =0;
	memset(&display,0,sizeof(display));
    
	if (pos->xpos > 0x7ff || pos->ypos > 0x7ff ) {
		PDEBUG("Position invalid values - excceed boundry\n");
		return FAIL;
	}
	if (info->var.xres > 0x7ff || info->var.yres > 0x7ff ) {
		PDEBUG("size invalid values - excceed boundry\n");
		return FAIL;
	}

	if (IS_VIDEO_420(config) && ( ((info->var.xres % 2) != 0) || ((info->var.yres % 2 ) != 0) ) ) {
		PDEBUG("dimentios invalid for 420 - high or width must be even!\n");
		return FAIL;
    }

	if (IS_VIDEO_422(config) && ((info->var.xres % 2) != 0) ) {
		PDEBUG("dimentios invalid for 422 - width must be even!\n");
		return FAIL;
	}

    dmw96osdm_display_dim_get(&display);
	if (info->var.rotate == 90 || info->var.rotate == 270)
		cond = ((pos->ypos + info->var.xres) > display.var.yres || (pos->xpos + info->var.yres) > display.var.xres);
	else
		cond = ((pos->xpos + info->var.xres) > display.var.xres || (pos->ypos + info->var.yres) > display.var.yres);
	
	if ( cond ) {

		PDEBUG("invalid position or dimention plane excceed display boundry\n");
		PDEBUG("display xres = %d\n", display.var.xres );
		PDEBUG("display yres = %d\n", display.var.yres );
		PDEBUG("Plane xres = %d\n", info->var.xres );
		PDEBUG("Plane yres = %d\n", info->var.yres );
		PDEBUG("Plane posx = %d\n", pos->xpos );
		PDEBUG("Plane posy = %d\n", pos->ypos );

		return FAIL;
	}
	
	return SUCCESS;
}

/************************************************************************
	dmw96osdm_plane_enable_get

	
*************************************************************************/
int dmw96osdm_plane_enable_get(const unsigned int osdm_type){

	return ( 0x1 & IOR32_OSDM( osdm_offset_val[osdm_type][PLANEn_CONFIG]));
}

/************************************************************************
	dmw96osdm_plane_enb_set

	
*************************************************************************/
void dmw96osdm_plane_enb_set(unsigned int enb , const unsigned int osdm_type){

	unsigned int plane_config = IOR32_OSDM(osdm_offset_val[osdm_type][PLANEn_CONFIG]);

	PDEBUG("%s osdm type %d \n", enb  > 0 ? "Enabled" : "Disabled" , osdm_type);

	IOW32_OSDM( osdm_offset_val[osdm_type][PLANEn_CONFIG] , ((enb > 0 ) ? (plane_config | (unsigned int)(0x1)) : (plane_config & (unsigned int)(~0x1)) ) );
	
}

/************************************************************************
	dmw96osdm_plane_validate_aligment

	Check if a given plane is outsize the display boundry and worn in case
	it is not address aligned. 
*************************************************************************/
int dmw96osdm_plane_validate_aligment(const struct fb_info *info){

	int res = SUCCESS;
    //check if the image reside in word aligned memory
	if ( (( (unsigned int)info->fix.smem_start % 4)) == 0 ){
		PDEBUG("OSD are validated successfuly \n");
	}
	else {
		PDEBUG("OSD error the osd image start address is not in word aligned memory\n");
		res = FAIL;
	}
	
	return res;
}

/************************************************************************
	dmw96osdm_print

	Print the osd  settings

************************************************************************/
void dmw96osdm_print(const struct fb_info *info){

	PDEBUG("xres = 0x%4x\n", info->var.xres);
	PDEBUG("yres = 0x%4x\n", info->var.yres);
    PDEBUG("bit per pixel = 0x%4x\n", info->var.bits_per_pixel);
	PDEBUG("image address = 0x%8p\n" , (void*)info->fix.smem_start);
}

void dmw96osdm_print_type(unsigned int osdm_type)
{
	const char* plane_name;

	switch (osdm_type) {

	case MAIN_VIDEO_PLANE:
		plane_name = "MAIN_VIDEO_PLANE";
		break;
	case SUB_VIDEO_PLANE:
		plane_name = "SUB_VIDEO_PLANE";
		break;
	case MAIN_OSD_PLANE:
		plane_name = "MAIN_OSD_PLANE";  
		break;
	case SUB_OSD_PLANE_1:
		plane_name = "SUB_OSD_PLANE_1";
		break;
	case SUB_OSD_PLANE_2:
		plane_name = "SUB_OSD_PLANE_2";
		break;
	case DISPLAY_PLANE:
		plane_name = "DISPLAY_PLANE";
		break;
	default:
		plane_name = "UNKNOWN";
	};

	PDEBUG("%s\n",plane_name);
}

/************************************************************************
	dmw96osdm_print_plane_org



************************************************************************/
void dmw96osdm_print_plane_org(unsigned int plane_org)
{
	const char* name;

	switch (plane_org) {

	case PLANE_ORG_SEMI_PLANAR:
		name = "PLANE_ORG_SEMI_PLANAR";
		break;
	case PLANE_ORG_PLANAR:
		name = "PLANE_ORG_PLANAR";
		break;
	case PLANE_ORG_PACKED:
		name = "PLANE_ORG_PACKED";  
		break;
	default:
		name = "Error";
	};

	PDEBUG("%s\n",name);
}

/************************************************************************
	dmw96osdm_print_plane_config



************************************************************************/
void dmw96osdm_print_plane_config(unsigned int config)
{
	const char* name;

	switch (config) {
	case OSDM_1_BPP_PALETTE:
		name = "OSDM_1_BPP_PALETTE";
		break;
	case OSDM_2_BPP_PALETTE:
		name = "OSDM_2_BPP_PALETTE";
		break;
	case OSDM_4_BPP_PALETTE:
		name = "OSDM_4_BPP_PALETTE";
		break;
	case OSDM_8_BPP_PALETTE:
		name = "OSDM_8_BPP_PALETTE";
		break;
	case OSDM_YUV420_MPEG2_PLANAR:
		name = "OSDM_YUV420_MPEG2_PLANAR";
		break;
	case OSDM_YUV420_MPEG2_SEMI_PLANAR_V1_U1_V0_U0:
		name = "OSDM_YUV420_MPEG2_SEMI_PLANAR_V1_U1_V0_U0";
		break;
	case OSDM_YUV420_MPEG2_SEMI_PLANAR_U1_V1_U0_V0:
		name = "OSDM_YUV420_MPEG2_SEMI_PLANAR_U1_V1_U0_V0";
		break;
	case OSDM_YUV420_MPEG1_JPEG_PLANAR:
		name = "OSDM_YUV420_MPEG1_JPEG_PLANAR";
		break;
	case OSDM_YUV420_MPEG1_JPEG_SEMI_PLANAR_V1_U1_V0_U0:
		name = "OSDM_YUV420_MPEG1_JPEG_SEMI_PLANAR_V1_U1_V0_U0";
		break;
	case OSDM_YUV420_MPEG1_JPEG_SEMI_PLANAR_U1_V1_U0_V0:
		name = "OSDM_YUV420_MPEG1_JPEG_SEMI_PLANAR_U1_V1_U0_V0";
		break;
	case OSDM_YUV422_PLANAR:
		name = "OSDM_YUV422_PLANAR";
		break;
	case OSDM_YUV422_SEMI_PLANAR_V1_U1_V0_U0:
		name = "OSDM_YUV422_SEMI_PLANAR_V1_U1_V0_U0";
		break;
	case OSDM_YUV422_SEMI_PLANAR_U1_V1_U0_V0:
		name = "OSDM_YUV422_SEMI_PLANAR_U1_V1_U0_V0";
		break;
	case OSDM_YUV422_Y1_V_Y0_U:
		name = "OSDM_YUV422_Y1_V_Y0_U";
		break;
	case OSDM_YUV422_Y1_U_Y0_V:
		name = "OSDM_YUV422_Y1_U_Y0_V";
		break;
	case OSDM_YUV422_V_Y1_U_Y0:
		name = "OSDM_YUV422_V_Y1_U_Y0";
		break;
	case OSDM_YUV422_U_Y1_V_Y0:
		name = "OSDM_YUV422_V_Y1_U_Y0";
		break;
	case OSDM_YUV422_Y1_Y0_V_U:
		name = "OSDM_YUV422_Y1_Y0_V_U";
		break;
	case OSDM_YUV422_Y1_Y0_U_V:
		name = "OSDM_YUV422_Y1_Y0_U_V";
		break;
	case OSDM_YUV422_V_U_Y1_Y0:
		name = "OSDM_YUV422_V_U_Y1_Y0";
		break;
	case OSDM_YUV422_U_V_Y1_Y0:
		name = "OSDM_YUV422_U_V_Y1_Y0";
		break;
	case OSDM_YUV444_PLANAR:
		name = "OSDM_YUV444_PLANAR";
		break;
	case OSDM_YUV444_SEMI_PLANAR_V1_U1_V0_U0:
		name = "OSDM_YUV444_SEMI_PLANAR_V1_U1_V0_U0";
		break;
	case OSDM_YUV444_SEMI_PLANAR_U1_V1_U0_V0:
		name = "OSDM_YUV444_SEMI_PLANAR_U1_V1_U0_V0";
		break;
	case OSDM_RGB32_8888_ARGB:
		name = "OSDM_RGB32_8888_ARGB";
		break;
	case OSDM_RGB32_8888_ARBG:
		name = "OSDM_RGB32_8888_ARBG";
		break;
	case OSDM_RGB32_8888_AGRB:
		name = "OSDM_RGB32_8888_AGRB";
		break;
	case OSDM_RGB32_8888_AGBR:
		name = "OSDM_RGB32_8888_AGBR";
		break;
	case OSDM_RGB32_8888_ABRG:
		name = "OSDM_RGB32_8888_ABRG";
		break;
	case OSDM_RGB32_8888_ABGR:
		name = "OSDM_RGB32_8888_ABGR";
		break;
	case OSDM_RGB32_8888_RGBA:
		name = "OSDM_RGB32_8888_RGBA";
		break;
	case OSDM_RGB32_8888_RBGA:
		name = "OSDM_RGB32_8888_RBGA";
		break;
	case OSDM_RGB32_8888_GRBA:
		name = "OSDM_RGB32_8888_GRBA";
		break;
	case OSDM_RGB32_8888_GBRA:
		name = "OSDM_RGB32_8888_GBRA";
		break;
	case OSDM_RGB32_8888_BRGA:
		name = "OSDM_RGB32_8888_BRGA";
		break;
	case OSDM_RGB32_8888_BGRA:
		name = "OSDM_RGB32_8888_BGRA";
		break;
	case OSDM_RGB16_4444_ARGB:
		name = "OSDM_RGB16_4444_ARGB";
		break;
	case OSDM_RGB16_4444_ARBG:
		name = "OSDM_RGB16_4444_ARBG";
		break;
	case OSDM_RGB16_4444_AGRB:
		name = "OSDM_RGB16_4444_AGRB";
		break;
	case OSDM_RGB16_4444_AGBR:
		name = "OSDM_RGB16_4444_AGBR";
		break;
	case OSDM_RGB16_4444_ABRG:
		name = "OSDM_RGB16_4444_ABRG";
		break;
	case OSDM_RGB16_4444_ABGR:
		name = "OSDM_RGB16_4444_ABGR";
		break;
	case OSDM_RGB16_4444_RGBA:
		name = "OSDM_RGB16_4444_RGBA";
		break;
	case OSDM_RGB16_4444_RBGA:
		name = "OSDM_RGB16_4444_RBGA";
		break;
	case OSDM_RGB16_4444_GRBA:
		name = "OSDM_RGB16_4444_GRBA";
		break;
	case OSDM_RGB16_4444_GBRA:
		name = "OSDM_RGB16_4444_GBRA";
		break;
	case OSDM_RGB16_4444_BRGA:
		name = "OSDM_RGB16_4444_BRGA";
		break;
	case OSDM_RGB16_4444_BGRA:
		name = "OSDM_RGB16_4444_BGRA";
		break;	
	case OSDM_RGB16_565_RGB:
		name = "OSDM_RGB16_565_RGB";
		break;
	case OSDM_RGB16_565_BGR:
		name = "OSDM_RGB16_565_BGR";
		break;
	default:
		name = "UNKNOWN";
	}

	PDEBUG("%s\n",name);
}

/************************************************************************
	dmw96osdm_calc_bits_per_line_for_y_rgb
    
************************************************************************/
static inline int dmw96osdm_calc_bits_per_line_for_y_rgb(const unsigned int config, const struct fb_info* info){

	 return (info->var.xres * (( OSDM_PLANE_ORG(config) == PLANE_ORG_PACKED ) ? dmw96osdm_osdm_get_bpp(config) : 8));
}


/************************************************************************
	dmw96osdm_calc_bits_per_line_for_uv
    
************************************************************************/
static inline int dmw96osdm_calc_bits_per_line_for_uv(const unsigned int config, const struct fb_info* info){

	return ((8 * info->var.xres * ((OSDM_PLANE_ORG(config) == PLANE_ORG_SEMI_PLANAR) ? 2 : 1)) / ((IS_VIDEO_420(config) || IS_VIDEO_422(config)) ? 2 : 1 ));
	
}



/************************************************************************
	dmw96osdm_plane_address_set

	TODO: Dudi treat the case when the image is not 32bits aligned

************************************************************************/
int dmw96osdm_plane_address_set(unsigned int xres, unsigned int yres, unsigned int address, const unsigned int config, const unsigned int osdm_type){

	unsigned int rgb_y_add = 0;
	unsigned int u_add = 0;
	unsigned int v_add = 0;

	rgb_y_add = DMW96OSDM_GET_Y_RGB_ADDRESS(xres, yres , config , address);
	u_add = DMW96OSDM_GET_U_UV_ADDRESS(xres, yres , config , address);
	v_add = DMW96OSDM_GET_V_ADDRESS(xres, yres , config , address);


	#ifdef CONFIG_MACH_VERSATILE_BROADTILE
		v_add = v_add - 0x80000000;
		u_add = u_add - 0x80000000;
		rgb_y_add = rgb_y_add - 0x80000000;

	#endif

	switch (OSDM_PLANE_ORG(config)){

	case PLANE_ORG_PLANAR:
		IOW32_OSDM( osdm_dma_val[osdm_type][2][DMAn_BADDR0] , v_add );
		PDEBUG( "DMAn_BADDR0[2] = 0x%x\n" , v_add );
	case PLANE_ORG_SEMI_PLANAR:
		IOW32_OSDM( osdm_dma_val[osdm_type][1][DMAn_BADDR0] , u_add );
		PDEBUG( "DMAn_BADDR0[1] = 0x%x\n" , u_add );
	case PLANE_ORG_PACKED:
		IOW32_OSDM( osdm_dma_val[osdm_type][0][DMAn_BADDR0] , rgb_y_add );
		PDEBUG( "DMAn_BADDR0[0] = 0x%x\n" , rgb_y_add );
		break;
	default:
		PDEBUG("plane orgainzation is not defined\n");
		return FAIL;
	}

	return SUCCESS;


}


/************************************************************************
	dmw96osdm_plane_dma_channel_byteorder_set

	TODO: Dudi treat the case when the image is not 32bits aligned

************************************************************************/
int dmw96osdm_plane_dma_channel_byteorder_set(const unsigned int byte_order, unsigned int channel, const unsigned int osdm_type)
{
	unsigned int t_config = 0;
    PDEBUG("%d \n", byte_order);

	if (byte_order > BYTE_ORDER_DCBA) {
		PDEBUG("Invalid osdm byte order\n");
		return FAIL;
	}

	if (channel > 2) {
		PDEBUG("Invalid channel number\n");
		return FAIL;
	}

	switch (channel){
		case 2:
			t_config = (IOR32_OSDM( osdm_dma_val[osdm_type][2][DMAn_CONFIG]) & ~0xf00) | (byte_order << 8);
			PDEBUG("0x%x \n", t_config);
			IOW32_OSDM( osdm_dma_val[osdm_type][2][DMAn_CONFIG] , t_config );
			break;
		case 1:
			t_config = (IOR32_OSDM( osdm_dma_val[osdm_type][1][DMAn_CONFIG]) & ~0xf00) | (byte_order << 8);
			PDEBUG("0x%x \n", t_config);
			IOW32_OSDM( osdm_dma_val[osdm_type][1][DMAn_CONFIG] , t_config );
			break;
		case 0:
			t_config = (IOR32_OSDM( osdm_dma_val[osdm_type][0][DMAn_CONFIG]) & ~0xf00) | (byte_order << 8);
			PDEBUG("0x%x \n", t_config);
			IOW32_OSDM( osdm_dma_val[osdm_type][0][DMAn_CONFIG] , t_config );
			break;
		default:
			PDEBUG("invalid channel\n");
			return FAIL;
	}

	return SUCCESS;
}



/************************************************************************
	dmw96osdm_plane_dma_byteorder_set

	TODO: Dudi treat the case when the image is not 32bits aligned

************************************************************************/
int dmw96osdm_plane_dma_byteorder_set(const unsigned int byte_order, const unsigned int config, const unsigned int osdm_type)
{
	unsigned int t_config = 0;
    PDEBUG("%d \n", byte_order);

	/* Check if plane configuration appropiate to plane type */
	if ( ( (!IS_PLANE_OSD(osdm_type) || !IS_OSD_VALID_CONFIG(config)) && (!IS_PLANE_VIDEO(osdm_type) || !IS_VIDEO_VALID_CONFIG(config)) ) ) {
		PDEBUG("Invalid osdm type or configuration\n");
		return FAIL;
	}

	if (byte_order > BYTE_ORDER_DCBA) {
		PDEBUG("Invalid osdm byte order\n");
		return FAIL;
	}

	switch (OSDM_PLANE_ORG(config)){
		case PLANE_ORG_PLANAR:
			t_config = (IOR32_OSDM( osdm_dma_val[osdm_type][2][DMAn_CONFIG]) & ~0xf00) | (byte_order << 8);
			PDEBUG("0x%x \n", t_config);
			IOW32_OSDM( osdm_dma_val[osdm_type][2][DMAn_CONFIG] , t_config );
		case PLANE_ORG_SEMI_PLANAR:
			t_config = (IOR32_OSDM( osdm_dma_val[osdm_type][1][DMAn_CONFIG]) & ~0xf00) | (byte_order << 8);
			PDEBUG("0x%x \n", t_config);
			IOW32_OSDM( osdm_dma_val[osdm_type][1][DMAn_CONFIG] , t_config );
		case PLANE_ORG_PACKED:
			t_config = (IOR32_OSDM( osdm_dma_val[osdm_type][0][DMAn_CONFIG]) & ~0xf00) | (byte_order << 8);
			PDEBUG("0x%x \n", t_config);
			IOW32_OSDM( osdm_dma_val[osdm_type][0][DMAn_CONFIG] , t_config );
			break;
		default:
			PDEBUG("plane orgainzation is not defined\n");
			return FAIL;
	}

	return SUCCESS;
}


/************************************************************************
	dmw96osdm_plane_dma_enb_reset


************************************************************************/
int dmw96osdm_plane_dma_enb_reset(const unsigned int enb, const unsigned int osdm_type)
{
	unsigned int is_enb = (enb > 0) ?  1 : 0;
	unsigned int t_config = 0;

	PDEBUG("%s \n", enb  > 0 ? "Enabled" : "Disabled");

	/* Check if plane configuration appropiate to plane type */
	if ( !IS_PLANE_OSD(osdm_type) && !IS_PLANE_VIDEO(osdm_type)  ) {
		PDEBUG("Invalid osdm type %d or configuration \n", osdm_type);
		return FAIL;
	}

	switch (osdm_type){

		case MAIN_VIDEO_PLANE:
		case SUB_VIDEO_PLANE:
			t_config = (IOR32_OSDM( osdm_dma_val[osdm_type][2][DMAn_CONFIG]) & ~0x1) | (is_enb);
			PDEBUG("0x%x \n", t_config);
			IOW32_OSDM( osdm_dma_val[osdm_type][2][DMAn_CONFIG] , t_config );
			t_config = (IOR32_OSDM( osdm_dma_val[osdm_type][1][DMAn_CONFIG]) & ~0x1) | (is_enb);
			PDEBUG("0x%x \n", t_config);
			IOW32_OSDM( osdm_dma_val[osdm_type][1][DMAn_CONFIG] , t_config );

		case MAIN_OSD_PLANE:
		case SUB_OSD_PLANE_1:
		case SUB_OSD_PLANE_2:
			t_config = (IOR32_OSDM( osdm_dma_val[osdm_type][0][DMAn_CONFIG]) & ~0x1) | (is_enb);
			PDEBUG("0x%x \n", t_config);
			IOW32_OSDM( osdm_dma_val[osdm_type][0][DMAn_CONFIG] , t_config );
			break;
		default:
			PDEBUG("plane orgainzation is not defined\n");
			return FAIL;
	}

	return SUCCESS;

}

/************************************************************************
	dmw96osdm_plane_dma_enb_set

	TODO: Dudi treat the case when the image is not 32bits aligned

************************************************************************/
int dmw96osdm_plane_dma_enb_set(const unsigned int enb, const unsigned int config, const unsigned int osdm_type)
{
	unsigned int t_config = 0;
    unsigned int is_enb = (enb > 0) ?  1 : 0;

	PDEBUG("osdm type %d is %s \n", osdm_type, enb  > 0 ? "Enabled" : "Disabled");

	/* Check if plane configuration appropiate to plane type */
	if ( ( (!IS_PLANE_OSD(osdm_type) || !IS_OSD_VALID_CONFIG(config)) && (!IS_PLANE_VIDEO(osdm_type) || !IS_VIDEO_VALID_CONFIG(config)) ) ) {
		PDEBUG("Invalid osdm type %d or configuration 0x%x\n", osdm_type, config);
		return FAIL;
	}

	switch (OSDM_PLANE_ORG(config)){

		case PLANE_ORG_PLANAR:
			t_config = (IOR32_OSDM( osdm_dma_val[osdm_type][2][DMAn_CONFIG]) & ~0x1) | (is_enb);
			PDEBUG("0x%x \n", t_config);
			IOW32_OSDM( osdm_dma_val[osdm_type][2][DMAn_CONFIG] , t_config );
		case PLANE_ORG_SEMI_PLANAR:
			t_config = (IOR32_OSDM( osdm_dma_val[osdm_type][1][DMAn_CONFIG]) & ~0x1) | (is_enb);
			PDEBUG("0x%x \n", t_config);
			IOW32_OSDM( osdm_dma_val[osdm_type][1][DMAn_CONFIG] , t_config );
		case PLANE_ORG_PACKED:
			t_config = (IOR32_OSDM( osdm_dma_val[osdm_type][0][DMAn_CONFIG]) & ~0x1) | (is_enb);
			PDEBUG("0x%x \n", t_config);
			IOW32_OSDM( osdm_dma_val[osdm_type][0][DMAn_CONFIG] , t_config );
			break;
		default:
			PDEBUG("plane orgainzation is not defined\n");
			return FAIL;
	}

	return SUCCESS;
}

void check_rotate_and_swap(unsigned int xres , unsigned int yres , unsigned int* _xres , unsigned int* _yres , unsigned int degree)
{

	if (degree == 90 || degree == 270)
	{	*_xres = yres;
		*_yres = xres;
	} else {
		*_xres = xres;
		*_yres = yres;
	}

}

/************************************************************************
	dmw96osdm_plane_dma_set

	TODO: Dudi treat the case when the image is not 32bits aligned

************************************************************************/
int dmw96osdm_plane_dma_set(const unsigned int config, const unsigned int osdm_type, struct fb_info* info){

    unsigned int uv_bcount = 0;
	unsigned int uv_stride = 0;
	unsigned int uv_bsize = 0;
	/*This applicable to RGB as well*/
	unsigned int rgb_y_bcount = 0;
	unsigned int rgb_y_stride = 0;
	unsigned int rgb_y_bsize = 0;

	unsigned int rgb_y_bits_per_line = 0;
	unsigned int uv_bits_per_line = 0;
	unsigned int yres_t = 0;

	int res = SUCCESS;
	struct fb_info temp_info;
	memcpy(&temp_info, info , sizeof(*info));

	/* Check if plane configuration appropiate to plane type */
	if ( ( (!IS_PLANE_OSD(osdm_type) || !IS_OSD_VALID_CONFIG(config)) && (!IS_PLANE_VIDEO(osdm_type) || !IS_VIDEO_VALID_CONFIG(config)) ) ) {
		PDEBUG("Invalid osdm type or configuration\n");
		return FAIL;
	}

	res = dmw96osdm_plane_validate_aligment(info);
	if (res != SUCCESS) {
		PDEBUG("Address is not alighed\n");
		return res;
	}

	PDEBUG("xres = %d yres = %d \n", temp_info.var.xres , temp_info.var.yres );

	if ( info->var.rotate == 90 || info->var.rotate == 270) 
			check_rotate_and_swap(temp_info.var.xres , temp_info.var.yres , &(temp_info.var.xres) , &(temp_info.var.yres) , info->var.rotate);

	PDEBUG("xres = %d yres = %d \n", temp_info.var.xres , temp_info.var.yres );

	/* Calculate the size of Luma (Y) or the RGB plane */
	rgb_y_bits_per_line = dmw96osdm_calc_bits_per_line_for_y_rgb(config,&temp_info);
    rgb_y_bsize = (rgb_y_bits_per_line % 32 == 0) ? (rgb_y_bits_per_line/32) : ((temp_info.var.yres * rgb_y_bits_per_line) % 32 == 0) ? ((temp_info.var.yres * rgb_y_bits_per_line)/32) : (((temp_info.var.yres * rgb_y_bits_per_line)/32) + 1);
	rgb_y_bcount = (rgb_y_bits_per_line % 32 == 0) ? temp_info.var.yres : 1;
	rgb_y_stride = rgb_y_bsize * 4; /* TODO DudiD why at the C-model the stride = 4 * rgb_y_bsize ? */

	/* Calculate the size of Chroma (CbCr) */
	uv_bits_per_line =  dmw96osdm_calc_bits_per_line_for_uv(config,&temp_info);
	yres_t =  temp_info.var.yres / ((IS_VIDEO_420(config)) ? 2 : 1 );
    uv_bsize = ( uv_bits_per_line % 32 == 0 ) ? (uv_bits_per_line/32) : ((yres_t * uv_bits_per_line) % 32 == 0) ? (yres_t * uv_bits_per_line)/32 : ((yres_t * uv_bits_per_line)/32 + 1);
	uv_bcount = ( uv_bits_per_line % 32 == 0 ) ? yres_t : 1 ;
	uv_stride = uv_bsize * 4;
    
    dmw96osdm_print_plane_org( OSDM_PLANE_ORG(config));
    
	switch (OSDM_PLANE_ORG(config)){
	
		case PLANE_ORG_PLANAR:
            IOW32_OSDM( osdm_dma_val[osdm_type][2][DMAn_STRIDE] , uv_stride );
			IOW32_OSDM( osdm_dma_val[osdm_type][2][DMAn_BSIZE]  , uv_bsize );
			IOW32_OSDM( osdm_dma_val[osdm_type][2][DMAn_BCOUNT] , uv_bcount );
		case PLANE_ORG_SEMI_PLANAR:
            IOW32_OSDM( osdm_dma_val[osdm_type][1][DMAn_STRIDE] , uv_stride );
			IOW32_OSDM( osdm_dma_val[osdm_type][1][DMAn_BSIZE]  , uv_bsize);
			IOW32_OSDM( osdm_dma_val[osdm_type][1][DMAn_BCOUNT] , uv_bcount );
		case PLANE_ORG_PACKED:
            IOW32_OSDM( osdm_dma_val[osdm_type][0][DMAn_STRIDE] , rgb_y_stride );
			IOW32_OSDM( osdm_dma_val[osdm_type][0][DMAn_BSIZE]  , rgb_y_bsize );
			IOW32_OSDM( osdm_dma_val[osdm_type][0][DMAn_BCOUNT] , rgb_y_bcount );
			break;
		default:
			PDEBUG("plane orgainzation is not defined\n");
			return FAIL;
	}

	return SUCCESS;

}

/************************************************************************
	dmw96osdm_plane_format_and_pkmode_set

    The config in this case come form the saved state
************************************************************************/
int dmw96osdm_plane_format_and_pkmode_set(const unsigned int config, const unsigned int osdm_type,struct fb_info* info)
{
	unsigned int t_config = 0;

	PDEBUG("config=0x%x osdm_type=0x%x\n" ,config, osdm_type);

	if ( ( (!IS_PLANE_OSD(osdm_type) || !IS_OSD_VALID_CONFIG(config)) && (!IS_PLANE_VIDEO(osdm_type) || !IS_VIDEO_VALID_CONFIG(config)) ) )
	{
		PDEBUG("Invalid osdm type or configuration\n");
		return FAIL;
	}

	dmw96osdm_print_plane_config(config);

	/*Remove prev setting of the packet and format for Video and OSD*/
	t_config = IOR32_OSDM(osdm_offset_val[osdm_type][PLANEn_CONFIG]) & ((IS_PLANE_VIDEO(config))? ~0xf370 : ~0xf070);
    /*If it is OSD then we remove the plane organization (Packed, Semi planar or Planar) bits (because it applicable to video only)
	 and remove the leading MSB 4 bits which determin the type of the plane*/
    t_config |= (config & (IS_PLANE_OSD(osdm_type)? ~0xf00 : ~0x0) & ~0xf0000000);
    PDEBUG("t_config 0x%x\n" , t_config);

	dmw96osdm_print_type(osdm_type);
	
	IOW32_OSDM(osdm_offset_val[osdm_type][PLANEn_CONFIG], t_config);

	info->var.bits_per_pixel = dmw96osdm_osdm_get_bpp(config);

	info->fix.line_length = info->var.xres * info->var.xres * info->var.bits_per_pixel  / 8;

	PDEBUG("bits_per_pixel = %d\n", info->var.bits_per_pixel);

	return SUCCESS;
}

int dmw96osdm_plane_chroma_enb(const int is_enb, const unsigned int osdm_type)
{
    unsigned int t_config = 0;

	if (!IS_PLANE_TYPE_VALID(osdm_type)){
		PDEBUG("Invalid osdm type\n");
		return FAIL;
	}

	PDEBUG("Chroma key is %s\n", is_enb > 0 ? "Enabled" : "Disabled");

	t_config = IOR32_OSDM(osdm_offset_val[osdm_type][PLANEn_CONFIG]);

	t_config = (t_config & ~(0x2)) | ((is_enb > 0)? 0x2 : 0x0) ; 

	IOW32_OSDM(osdm_offset_val[osdm_type][PLANEn_CONFIG], t_config);

	return SUCCESS;
	
}

int dmw96osdm_plane_per_pixel_alpha_enb(const int is_enb, const unsigned int osdm_type)
{
    unsigned int t_config = 0;

	if (!IS_PLANE_TYPE_VALID(osdm_type)){
		PDEBUG("Invalid osdm type\n");
		return FAIL;
	}

	PDEBUG("Per Pixel alpha is %s\n", is_enb > 0 ? "Enabled" : "Disabled");

	t_config = IOR32_OSDM(osdm_offset_val[osdm_type][PLANEn_CONFIG]);

	t_config = (t_config & ~(0x4)) | ((is_enb > 0)? 0x4 : 0x0) ; 

	IOW32_OSDM(osdm_offset_val[osdm_type][PLANEn_CONFIG], t_config);

	return SUCCESS;

}

int dmw96osdm_plane_masking_test(const struct osdm_mask mask, const unsigned int osdm_type)
{
	if (!IS_PLANE_TYPE_VALID(osdm_type)){
		PDEBUG("Invalid osdm type\n");
		return FAIL;
	}

	PDEBUG("mask.right = %d mask.left = %d mask.top = %d  mask.bottm = %d \n",mask.right ,mask.left ,mask.top , mask.bottom);

	if ( (mask.top > 31) || (mask.bottom > 31)) {
		PDEBUG("Invalid mask top or bottom values\n");
		return FAIL;
	}

	if ( ( IS_PLANE_OSD(osdm_type) && ( mask.left >  31 || mask.right > 31  ) ) ||
		 ( IS_PLANE_VIDEO(osdm_type) && ( mask.left >  3 || mask.right > 3 ) ) ) {
			PDEBUG("Invalid mask right or left values\n");
			return FAIL;
	}
	
	return SUCCESS;
}


int dmw96osdm_plane_masking_set(const struct osdm_mask mask, const unsigned int osdm_type)
{
	unsigned int config  =  0 ;

    if (!IS_PLANE_TYPE_VALID(osdm_type)){
		PDEBUG("Invalid osdm type\n");
		return FAIL;
	}

	PDEBUG("mask.right = %d mask.left = %d mask.top = %d  mask.bottm = %d \n",mask.right ,mask.left ,mask.top , mask.bottom);

	if ( (mask.top > 31) || (mask.bottom > 31)) {
		PDEBUG("Invalid mask top or bottom values\n");
		return FAIL;
	}

	if ( ( IS_PLANE_OSD(osdm_type) && ( mask.left >  31 || mask.right > 31  ) ) ||
		 ( IS_PLANE_VIDEO(osdm_type) && ( mask.left >  3 || mask.right > 3 ) ) ) {
			PDEBUG("Invalid mask right or left values\n");
			return FAIL;
	}


	config = IOR32_OSDM(osdm_offset_val[osdm_type][PLANEn_CONFIG]) & (0x0000ffff);

	config |= (unsigned int)( mask.left << 16) | (unsigned int)(mask.right << 21) | (unsigned int)( mask.top << 26) | (unsigned int) (mask.bottom  << 28);

    IOW32_OSDM( osdm_offset_val[osdm_type][PLANEn_CONFIG] , config);

	return SUCCESS;
}

int dmw96osdm_plane_dim_set(struct fb_info *info ,const unsigned int osdm_type) {

	unsigned int xres = 0 , yres = 0;

	if (!IS_PLANE_TYPE_VALID(osdm_type)){
		PDEBUG("Invalid osdm type\n");
		return FAIL;
	}

	PDEBUG("xres = %d yres= %d \n", info->var.xres , info->var.yres);

	if (info->var.rotate == 90 || info->var.rotate == 270){
		yres = info->var.xres;
		xres = info->var.yres;
	}else{
		xres = info->var.xres;
		yres = info->var.yres;
	}

	PDEBUG("xres = %d yres= %d \n", xres , yres);

	/* Set dimentions in the mixer */
	IOW32_OSDM( osdm_offset_val[osdm_type][PLANEn_DIM] , ((0x7ff & yres) << 16) | (0x7ff & xres) );

	return SUCCESS;
}


int dmw96osdm_plane_dim_get(struct fb_info *info ,const unsigned int osdm_type) {

	unsigned int dim = 0 ;
	if (!IS_PLANE_TYPE_VALID(osdm_type)){
		PDEBUG("Invalid osdm type\n");
		return FAIL;
	}
	dim = IOR32_OSDM( osdm_offset_val[osdm_type][PLANEn_DIM]);
    
	info->var.xres = dim & 0x7ff;
	info->var.yres = (dim >> 16 ) & 0x7ff;

	return SUCCESS;
}

/************************************************************************
	dmw96osdm_plane_pos_set

	set the position of the plane 

************************************************************************/
int dmw96osdm_plane_pos_set(const struct osdm_position* pos ,const unsigned int osdm_type)
{
    
	if (!IS_PLANE_TYPE_VALID(osdm_type)){
		PDEBUG("Invalid osdm type\n");
		return FAIL;
	}

	PDEBUG("xpos = %d ypos= %d \n", pos->xpos , pos->ypos );
    IOW32_OSDM( osdm_offset_val[osdm_type][PLANEn_POS] , (unsigned int )((pos->ypos) << 16 | (pos->xpos ) ));
    
	return SUCCESS;
}

/************************************************************************
	dmw96osdm_plane_pos_set

	set the position of the plane 

************************************************************************/
void dmw96osdm_plane_pos_get(struct osdm_position* pos ,const unsigned int osdm_type)
{
    unsigned int pos_tmp = IOR32_OSDM( osdm_offset_val[osdm_type][PLANEn_POS]); 

	pos->xpos = pos_tmp & 0x7ff;
	pos->ypos = (pos_tmp >> 16) & 0x7ff;
}

/************************************************************************
	dmw96osdm_plane_test

	Set the dimentions of the plane 

************************************************************************/
int dmw96osdm_plane_test(struct fb_info *info, struct osdm_position* pos ,const unsigned int config ,const unsigned int osdm_type) {

	int res = SUCCESS;
	
	res = dmw96osdm_validate_pos_and_dim(info,pos,config);
	if (SUCCEEDED(res)) {
		res = dmw96osdm_plane_validate_aligment(info);
	}
	
    return res;
}


/************************************************************************
	dmw96osdm_plane_set

	Set the dimentions of the plane 

************************************************************************/
int dmw96osdm_plane_set(struct fb_info *info, struct osdm_position* pos ,const unsigned int config ,const unsigned int osdm_type) {

	int res = SUCCESS;

	if (!IS_PLANE_TYPE_VALID(osdm_type)){
		PDEBUG("Invalid osdm type\n");
		res = FAIL;
	}else {

		res = dmw96osdm_validate_pos_and_dim(info,pos,config);
		if (SUCCEEDED(res)) {
			res = dmw96osdm_plane_pos_set(pos ,osdm_type);
			if (SUCCEEDED(res)) {
				res = dmw96osdm_plane_dim_set(info,osdm_type);
				if (SUCCEEDED(res)) {
					/* Set the format, packing, plane organization*/
					res = dmw96osdm_plane_format_and_pkmode_set(config,osdm_type,info);
					if (SUCCEEDED(res)) {
							/* configure the DMA */
							res = dmw96osdm_plane_dma_set(config,osdm_type,info);
					}
				}
			}
        }
	}
    return res;
}


/************************************************************************
	dmw96osdm_plane_alpha_set


*************************************************************************/
int dmw96osdm_plane_alpha_set(const unsigned char alpha , unsigned int osdm_type) {

	unsigned int alck = 0;

	PDEBUG("Alpha is %d\n" , alpha);

	if (IS_PLANE_OSD(osdm_type)){

		alck = IOR32_OSDM(	osdm_offset_val[osdm_type][OSDn_ALCK] );
		IOW32_OSDM(	osdm_offset_val[osdm_type][OSDn_ALCK] , (alck & ~(0xf) )| (unsigned int)alpha);

	} else if (IS_PLANE_VIDEO(osdm_type)){

		alck = IOR32_OSDM(	osdm_offset_val[osdm_type][VIDEOn_ALCK] );
		IOW32_OSDM(	osdm_offset_val[osdm_type][VIDEOn_ALCK] , (alck & ~(0xf) )| (unsigned int)alpha);

	} else {
		PDEBUG("Invalid osdm plane type");
		return FAIL;
	}

	PDEBUG("Alpha written is 0x%x\n" , (alck & ~(0xf) )| alpha);

    return SUCCESS;
}



/************************************************************************
	dmw96osdm_plane_alpha_get


*************************************************************************/
int dmw96osdm_plane_alpha_get(unsigned char* alpha , unsigned int osdm_type) {

	if (IS_PLANE_OSD(osdm_type)){

		 *alpha = (IOR32_OSDM(	osdm_offset_val[osdm_type][OSDn_ALCK])) & 0xf;

	} else if (IS_PLANE_VIDEO(osdm_type)){

		 *alpha = (IOR32_OSDM(	osdm_offset_val[osdm_type][VIDEOn_ALCK])) & 0xf;

	} else {
		PDEBUG("Invalid osdm plane type");
		return FAIL;
	}
    return SUCCESS;
}

/************************************************************************
	dmw96osdm_plane__chromakey_set


*************************************************************************/
int dmw96osdm_plane_chromakey_set(const osdm_rgb888 chroma , unsigned int osdm_type) {

	unsigned int alck = 0;

	PDEBUG("Chroma key is 0x%x\n", chroma);

	if (IS_PLANE_OSD(osdm_type)){

		alck = IOR32_OSDM(	osdm_offset_val[osdm_type][OSDn_ALCK] );
		IOW32_OSDM(	osdm_offset_val[osdm_type][OSDn_ALCK] , (alck & ~(0xffffff0) )| chroma << 8);

	} else if (IS_PLANE_VIDEO(osdm_type)){

		alck = IOR32_OSDM(	osdm_offset_val[osdm_type][VIDEOn_ALCK] );
		IOW32_OSDM(	osdm_offset_val[osdm_type][VIDEOn_ALCK] , (alck & ~(0xffffff0) )| chroma << 8);

	} else {
		PDEBUG("Invalid osdm plane type");
		return FAIL;
	}

	PDEBUG("Chroma written is 0x%x\n", (alck & ~(0xffffff0) )| chroma << 8 );

    return SUCCESS;
}



/************************************************************************
	dmw96osdm_plane_chromakey_get


*************************************************************************/
int dmw96osdm_plane_chromakey_get(osdm_rgb888* chroma , unsigned int osdm_type) {

	if (IS_PLANE_OSD(osdm_type)){

		 *chroma = ((IOR32_OSDM(osdm_offset_val[osdm_type][OSDn_ALCK])) & 0xffffff0) >> 8;

	} else if (IS_PLANE_VIDEO(osdm_type)){

		 *chroma = ((IOR32_OSDM(osdm_offset_val[osdm_type][VIDEOn_ALCK])) & 0xffffff0) >> 8;

	} else {
		PDEBUG("Invalid osdm plane type");
		return FAIL;
	}
    return SUCCESS;
}


/************************************************************************
	dmw96osdm_plane_video_limit_set

    
*************************************************************************/
int dmw96osdm_plane_video_limit_set(const struct osdm_video_limits* limit  , unsigned int osdm_type){

	if (IS_PLANE_VIDEO(osdm_type)){
		IOW32_OSDM(	osdm_offset_val[osdm_type][VIDEOn_YUVLIM],  (unsigned int)((unsigned int)limit->chroma_max << 24 | (unsigned int)limit->chroma_min << 16 | (unsigned int)limit->luma_max << 8 | (unsigned int)limit->luma_min ));
	} else{ 
		PDEBUG("Invalid osdm plane type");
		return FAIL;
	}

	return SUCCESS;
}

/************************************************************************
	dmw96osdm_plane_video_limit_get

    
*************************************************************************/
int dmw96osdm_plane_video_limit_get(struct osdm_video_limits* limit  , unsigned int osdm_type){

	unsigned int limit_tmp = 0;
	
	if (IS_PLANE_VIDEO(osdm_type)){

		limit_tmp = IOR32_OSDM(osdm_offset_val[osdm_type][VIDEOn_YUVLIM]);
		limit->luma_min = (unsigned char)(limit_tmp & 0xf);
		limit->luma_max = (unsigned char)(limit_tmp >> 8);
		limit->chroma_min = (unsigned char)(limit_tmp >> 16);
		limit->chroma_max = (unsigned char)(limit_tmp >> 24);
	} else{ 
		PDEBUG("Invalid osdm plane type");
		return FAIL;
	}

	return SUCCESS;
}

/************************************************************************
	dmw96osdm_plane_transparent_block_can_dma_save_bandwidth

    
*************************************************************************/
int dmw96osdm_plane_transparent_block_can_dma_save_bandwidth(const unsigned int config, const struct fb_info *plane_info)
{
	unsigned int uv_bits_per_line = 0;
	unsigned int rgb_y_bits_per_line = 0;

	rgb_y_bits_per_line = dmw96osdm_calc_bits_per_line_for_y_rgb(config,plane_info);
	if (( OSDM_PLANE_ORG(config) != PLANE_ORG_PACKED )) {
		uv_bits_per_line =  dmw96osdm_calc_bits_per_line_for_uv(config,plane_info);
	}

	/* Configure the DMA if the lines of the transparency block are aligned to 32b*/
	/* The driver dont save bandwidth in Planar and Semi planar when Luma is aligned to 32bit and the Chroma is not*/
	if ( ( OSDM_PLANE_ORG(config) == PLANE_ORG_PACKED  && rgb_y_bits_per_line % 32 == 0) || 
		 ( OSDM_PLANE_ORG(config) != PLANE_ORG_PACKED  && rgb_y_bits_per_line % 32 == 0 && uv_bits_per_line % 32 == 0 ) )
		return 1;
	else 
		return 0;
}
/************************************************************************
	dmw96osdm_plane_transparent_block_dim_get

    
*************************************************************************/
int dmw96osdm_plane_transparent_block_dim_get(struct fb_var_screeninfo *vinfo, unsigned int osdm_type)
{

  	unsigned int dim = 0;
	if (IS_PLANE_VIDEO(osdm_type)){
		dim = IOR32_OSDM(	osdm_offset_val[osdm_type][VIDEOn_TDIM]);
	} else if (IS_PLANE_OSD(osdm_type)) {
		dim = IOR32_OSDM(	osdm_offset_val[osdm_type][OSDn_TDIM]);
	} else {
		PDEBUG("Invalid osdm type\n");
		return FAIL;
	}

	vinfo->xres = dim & 0x7ff;
	vinfo->yres = (dim >> 16) & 0x7ff;

    return SUCCESS;

}
/************************************************************************
	dmw96osdm_plane_transparent_block_dim_dma_set

	HW constraint: we might no fully save bandwidth because the size and dimention of the transparency block might not be 
	aligned to 32 bits (DMA is always reading 32 bits words)

	SW: in order not to check positions when setting dimensions, we always decrease the transparency block size by one word (per line).
	In case that both position and size are aligned to 32 bits, we will read 1 extra word per each line (so we won't save all
	possible bandwidth). In other cases we are optimal... 
    
*************************************************************************/
int dmw96osdm_plane_transparent_block_dim_dma_set(const unsigned int config, const struct fb_var_screeninfo *vinfo, unsigned int osdm_type)
{
	unsigned int rgb_y_bits_per_line = 0;
	unsigned int rgb_y_wcount = 0;
	unsigned int rgb_y_bcount = 0;
	unsigned int uv_bits_per_line = 0;
	unsigned int uv_wcount = 0;
	unsigned int uv_bcount = 0;

	if ( ( (!IS_PLANE_OSD(osdm_type) || !IS_OSD_VALID_CONFIG(config)) && (!IS_PLANE_VIDEO(osdm_type) || !IS_VIDEO_VALID_CONFIG(config)) ) ) {
		PDEBUG("Invalid osdm type or configuration\n");
		return FAIL;
	}

	/* Calculate the size of Luma (Y) or the RGB plane */
	rgb_y_bits_per_line = vinfo->xres * (( OSDM_PLANE_ORG(config) == PLANE_ORG_PACKED ) ? dmw96osdm_osdm_get_bpp(config) : 8);
    rgb_y_wcount = (rgb_y_bits_per_line == 0) ? 0 : (rgb_y_bits_per_line/32 - 1);/*We always decrease by 1 because we don't want be depend on the transparency block position*/
	rgb_y_bcount = vinfo->yres;

	/* Calculate the size of Chroma (CbCr) */
	uv_bits_per_line =  8 * vinfo->xres * ((OSDM_PLANE_ORG(config) == PLANE_ORG_SEMI_PLANAR) ? 2 : 1) / ((IS_VIDEO_420(config) || IS_VIDEO_422(config)) ? 2 : 1 );
    uv_wcount = (uv_bits_per_line == 0) ? 0 : (uv_bits_per_line/32 - 1); /*We always decrease by 1 because we don't want be depend on the transparency block position */
	uv_bcount = vinfo->yres / ((IS_VIDEO_420(config)) ? 2 : 1 ) - 2 ;

	if (uv_bcount > 0x7ff || uv_wcount > 0x7ff || rgb_y_bcount > 0x7ff || rgb_y_wcount > 0x7ff ) {
	    PDEBUG("Invalid b/w count for transparency\n");
	    return FAIL;
	}
	switch (OSDM_PLANE_ORG(config)){
		
			case PLANE_ORG_PLANAR:
				IOW32_OSDM( osdm_dma_val[osdm_type][2][DMAn_TDIM] , (uv_bcount << 16) | uv_wcount );
			case PLANE_ORG_SEMI_PLANAR:                
				IOW32_OSDM( osdm_dma_val[osdm_type][1][DMAn_TDIM] , (uv_bcount << 16) | uv_wcount );
			case PLANE_ORG_PACKED:
				PDEBUG("0x%x\n" , (rgb_y_bcount << 16) | rgb_y_wcount );                     
				IOW32_OSDM( osdm_dma_val[osdm_type][0][DMAn_TDIM] , (rgb_y_bcount << 16) | rgb_y_wcount );
				break;
			default:
				PDEBUG("plane orgainzation is not defined\n");
				return FAIL;
	}

	return SUCCESS;
}


/************************************************************************
	dmw96osdm_plane_transparent_block_dim_test

    
*************************************************************************/
int dmw96osdm_plane_transparent_block_dim_test(const unsigned int config, const struct fb_var_screeninfo *vinfo, const struct fb_info *plane_info, unsigned int osdm_type)
{

	if (vinfo->xres > 0x7ff || vinfo->yres > 0x7ff ) {

		PDEBUG("Invalid dimentions, excceed plane limits\n");
		return FAIL;
	}

	return SUCCESS;
}

/************************************************************************
	dmw96osdm_plane_transparent_block_dim_set

    
*************************************************************************/
int dmw96osdm_plane_transparent_block_dim_set(const unsigned int config, const struct fb_var_screeninfo *vinfo, const struct fb_info *plane_info, unsigned int osdm_type)
{
    unsigned int dim_tmp = 0;
    
	if ( ( (!IS_PLANE_OSD(osdm_type) || !IS_OSD_VALID_CONFIG(config)) && (!IS_PLANE_VIDEO(osdm_type) || !IS_VIDEO_VALID_CONFIG(config)) ) ) {
		PDEBUG("Invalid osdm type or configuration\n");
		return FAIL;
	}

	if (vinfo->xres > 0x7ff || vinfo->yres > 0x7ff ) {

		PDEBUG("Invalid dimentions, excceed plane limits\n");
		return FAIL;
	}

	PDEBUG("xres = %d yres = %d \n", vinfo->xres, vinfo->yres);

	if (vinfo->xres == 0x0 && vinfo->yres == 0x0) {
		PDEBUG("Transparent block is not enabled due to 0 size \n");
		return SUCCESS;
	}

	dim_tmp = (unsigned int)(vinfo->yres << 16| vinfo->xres);

    /* Set the transparent at the Mixer*/
	if (IS_PLANE_VIDEO(osdm_type)){
		IOW32_OSDM(	osdm_offset_val[osdm_type][VIDEOn_TDIM], dim_tmp );
	} else if (IS_PLANE_OSD(osdm_type)) {
		IOW32_OSDM(	osdm_offset_val[osdm_type][OSDn_TDIM],  dim_tmp );
	}

    /* Configure the DMA if the lines of the transparency block are aligned to 32b*/
	if (dmw96osdm_plane_transparent_block_can_dma_save_bandwidth(config,plane_info))
        dmw96osdm_plane_transparent_block_dim_dma_set(config , vinfo, osdm_type);
    return SUCCESS;
}

int dmw96osdm_plane_transparent_block_pos_dma_set(const unsigned int config, const struct osdm_position* pos, unsigned int osdm_type)
{
	unsigned int rgb_y_bits_per_line = 0;
	unsigned int rgb_y_woffset = 0;
	unsigned int rgb_y_boffset = 0;

	unsigned int uv_bits_per_line = 0;
	unsigned int uv_woffset = 0;
	unsigned int uv_boffset = 0;

	if ( ( (!IS_PLANE_OSD(osdm_type) || !IS_OSD_VALID_CONFIG(config)) && (!IS_PLANE_VIDEO(osdm_type) || !IS_VIDEO_VALID_CONFIG(config)) ) ) {
		PDEBUG("Invalid osdm type or configuration\n");
		return FAIL;
	}

	/* Calculate the size of Luma (Y) or the RGB plane */
	rgb_y_bits_per_line = pos->xpos * (( OSDM_PLANE_ORG(config) == PLANE_ORG_PACKED ) ? dmw96osdm_osdm_get_bpp(config) : 8);
    rgb_y_woffset = (rgb_y_bits_per_line == 0) ?  0 : (rgb_y_bits_per_line/32 + 1);/*We inclease by 1 because we don't know the dimentions*/
	rgb_y_boffset = pos->ypos;

	/* Calculate the size of Chroma (CbCr) */
	uv_bits_per_line =  8 * pos->xpos * ((OSDM_PLANE_ORG(config) == PLANE_ORG_SEMI_PLANAR) ? 2 : 1) / ((IS_VIDEO_420(config) || IS_VIDEO_422(config)) ? 2 : 1 );
    uv_woffset = (uv_bits_per_line == 0) ?  0 : (uv_bits_per_line/32 + 1); /*We always inclease by 1 because we don't know the dimentions*/
    //uv_boffset = pos->ypos / ((IS_VIDEO_420(config)) ? 2 : 1 );
	uv_boffset = pos->ypos / ((IS_VIDEO_420(config)) ? 2 : 1 ) + 2 ;
	//uv_boffset = ( pos->ypos / ((IS_VIDEO_420(config)) ? 2 : 1 ) ) - ((OSDM_PLANE_ORG(config) == PLANE_ORG_SEMI_PLANAR) ? 1 : 0);

    if (uv_boffset > 0x7ff || uv_woffset > 0x7ff || rgb_y_boffset > 0x7ff || rgb_y_woffset > 0x7ff ) {
	PDEBUG("Invalid b/w offset for transparency\n");
	return FAIL;
    }

    switch (OSDM_PLANE_ORG(config)){

			case PLANE_ORG_PLANAR:
				IOW32_OSDM( osdm_dma_val[osdm_type][2][DMAn_TPOS] , (uv_boffset << 16) | uv_woffset );
			case PLANE_ORG_SEMI_PLANAR:                
				IOW32_OSDM( osdm_dma_val[osdm_type][1][DMAn_TPOS] , (uv_boffset << 16) | uv_woffset );
			case PLANE_ORG_PACKED:                     
				IOW32_OSDM( osdm_dma_val[osdm_type][0][DMAn_TPOS] , (rgb_y_boffset << 16) | rgb_y_woffset );
				break;
			default:
				PDEBUG("plane orgainzation is not defined\n");
				return FAIL;
	}

	return SUCCESS;
}


/************************************************************************
    dmw96osdm_plane_transparent_block_pos_test
    
*************************************************************************/
int dmw96osdm_plane_transparent_block_pos_test(const unsigned int config, const struct osdm_position* pos, const struct fb_info *plane_info, unsigned int osdm_type)
{
    if (pos->xpos > 0x7ff || pos->ypos > 0x7ff) {
		PDEBUG("Invalid position values excceed limits\n");
		return FAIL;
	}
	return SUCCESS;

}


/************************************************************************
    dmw96osdm_plane_transparent_block_pos_set
    
*************************************************************************/
int dmw96osdm_plane_transparent_block_pos_set(const unsigned int config, const struct osdm_position* pos, const struct fb_info *plane_info, unsigned int osdm_type)
{
	unsigned int pos_tmp = 0;
	
	if ( ( (!IS_PLANE_OSD(osdm_type) || !IS_OSD_VALID_CONFIG(config)) && (!IS_PLANE_VIDEO(osdm_type) || !IS_VIDEO_VALID_CONFIG(config)) ) ) {
		PDEBUG("Invalid osdm type or configuration\n");
		return FAIL;
	}

    if (pos->xpos > 0x7ff || pos->ypos > 0x7ff) {
		PDEBUG("Invalid position values excceed limits\n");
		return FAIL;
	}

	PDEBUG("xpos = %d ypos = %d \n", pos->xpos , pos->ypos);

	pos_tmp = (unsigned int)(pos->ypos << 16| pos->xpos);

    if (IS_PLANE_VIDEO(osdm_type)){
		IOW32_OSDM(	osdm_offset_val[osdm_type][VIDEOn_TPOS], pos_tmp );
	} else if (IS_PLANE_OSD(osdm_type)) {
		IOW32_OSDM(	osdm_offset_val[osdm_type][OSDn_TPOS],  pos_tmp );
	} else {
		PDEBUG("Invalid osdm type\n");
		return FAIL;
	}

	/* Configure the DMA if the pos of the transparency block are aligned to 32b*/
	if (dmw96osdm_plane_transparent_block_can_dma_save_bandwidth(config,plane_info))
		 dmw96osdm_plane_transparent_block_pos_dma_set(config , pos, osdm_type);

	return SUCCESS;
}

/************************************************************************
	dmw96osdm_plane_transparet_block_pos_get


*************************************************************************/
int dmw96osdm_plane_transparet_block_pos_get(struct osdm_position* pos, unsigned int osdm_type) {

	unsigned int pos_tmp = 0;
	if (IS_PLANE_VIDEO(osdm_type)){
		pos_tmp = IOR32_OSDM(	osdm_offset_val[osdm_type][VIDEOn_TPOS]);
	} else if (IS_PLANE_OSD(osdm_type)) {
		pos_tmp = IOR32_OSDM(	osdm_offset_val[osdm_type][OSDn_TPOS]);
	} else {
		PDEBUG("Invalid osdm type\n");
		return FAIL;
	}

	pos->xpos = pos_tmp & 0x7ff;
	pos->ypos = (pos_tmp >> 16) & 0x7ff;

    return SUCCESS;
}


/************************************************************************
	dmw96osdm_plane_transparent_pixel_set


*************************************************************************/
int dmw96osdm_plane_transparent_pixel_set(const osdm_abgr8888 abgr8888, unsigned int osdm_type) {

	if (IS_PLANE_OSD(osdm_type)){

		IOW32_OSDM(	osdm_offset_val[osdm_type][OSDn_TPIXEL] , abgr8888);

	} else if (IS_PLANE_VIDEO(osdm_type)){

		IOW32_OSDM(	osdm_offset_val[osdm_type][VIDEOn_TPIXEL] , abgr8888);

	} else {
		PDEBUG("Invalid osdm plane type");
		return FAIL;
	}

	PDEBUG("red=%d green=%d blue=%d alpha=%d \n" , get_red_from_abgr8888(abgr8888) , get_green_from_abgr8888(abgr8888) , get_blue_from_abgr8888(abgr8888) , get_alpha_from_abgr8888(abgr8888));

    return SUCCESS;
}



/************************************************************************
	dmw96osdm_plane_transparent_pixel_get


*************************************************************************/
int dmw96osdm_plane_transparent_pixel_get(osdm_abgr8888* abgr8888, unsigned int osdm_type) {

	if (IS_PLANE_OSD(osdm_type)){

		 *abgr8888 = IOR32_OSDM(	osdm_offset_val[osdm_type][OSDn_TPIXEL]);

	} else if (IS_PLANE_VIDEO(osdm_type)){

		 *abgr8888 = IOR32_OSDM(	osdm_offset_val[osdm_type][VIDEOn_TPIXEL]);

	} else {
		PDEBUG("Invalid osdm plane type");
		return FAIL;
	}
    return SUCCESS;
}


int dmw96osdm_plane_print_color_control(const struct osdm_video_color_control* color_control){

	PDEBUG("brightness = 0x%4x\n" , color_control->brightness);

	PDEBUG("contrast = 0x%4x\n" , color_control->contrast);

	PDEBUG("hue color blue = 0x%4x\n" , color_control->hue);

	PDEBUG("saturation color blue = 0x%4x\n" , color_control->saturation);
    return SUCCESS;
}


/************************************************************************
	dmw96osdm_color_video_control_set

	brightness: parameter range is -128 to +127
	contrast: values below 64 decrease contrast whereas values above 64 increase contrast.
	saturation: values below 64 decrease saturation whereas values above 64 increase saturation.
	hue degree: parameter range is -32 to +32 (signed 7b value); values outside this range are clipped.

************************************************************************/
int  dmw96osdm_color_video_control_set(const struct osdm_video_color_control* video_color_ctl, unsigned int osdm_type){

    /*Get Brightness Contrast Sturation Hue*/
	unsigned int bcsh = 0;

	if (IS_PLANE_VIDEO(osdm_type)) {
		//Brightness is in 2's complement
		bcsh = ( (video_color_ctl->hue & 0x7f) << 24) | ( (video_color_ctl->saturation & 0x7f) << 16) |
		( (video_color_ctl->contrast & 0x7f) << 8) | (video_color_ctl->brightness & 0xff);
        
		PDEBUG("brightness =%d contrast=%d saturation=%d hue=%d bcsh=0x%x\n",
			   video_color_ctl->brightness ,
			   video_color_ctl->contrast,
			   video_color_ctl->saturation,
			   video_color_ctl->hue,
			   bcsh);

		IOW32_OSDM(	osdm_offset_val[osdm_type][VIDEOn_BCSH] ,  bcsh);
		return SUCCESS;
	} else {
		PDEBUG("Fail to set video control invalid plane type\n");
		return FAIL;
	}
}

/************************************************************************
	dmw96osdm_color_video_control_get

************************************************************************/
int  dmw96osdm_color_video_control_get(struct osdm_video_color_control* video_color_ctl, unsigned int osdm_type){

	/*Get Brightness Contrast Sturation Hue*/
	unsigned int bcsh = IOR32_OSDM(	osdm_offset_val[osdm_type][VIDEOn_BCSH]);

	if (IS_PLANE_VIDEO(osdm_type)) {
		//Brightness is in 2's complement
		video_color_ctl->brightness = bcsh & 0xf;
		video_color_ctl->contrast = (bcsh & 0xf0) >> 8;
		video_color_ctl->saturation = (bcsh & 0xf00) >> 16;
		video_color_ctl->hue = (bcsh & 0xf000) >> 24;
		return SUCCESS;
	} else {
		PDEBUG("Fail to set video control invalid plane type\n");
		return FAIL;
	}
}


/************************************************************************
	dmw96osdm_osd_palette_set

	Settig the OSD palette this palette include alpha blending

************************************************************************/
int dmw96osdm_osd_palette_set(const struct fb_cmap* osd_pal, unsigned int lut_base , unsigned int osdm_type){

	int i = 0;

	unsigned lut_base_add = 0;
	unsigned int config = 0;
    
	if (lut_base > 0xf) {
		PDEBUG("Invalid LUT base\n");
		return FAIL;
	}

	if (IS_PLANE_OSD(osdm_type)){

		if (osdm_type == MAIN_OSD_PLANE) {
			lut_base_add = OSD0_LUT;
		} 
		else if (osdm_type == SUB_OSD_PLANE_1) {
			lut_base_add = OSD1_LUT;
		} 
		else if (osdm_type == SUB_OSD_PLANE_2) { 
			lut_base_add = OSD2_LUT;
		}

		PDEBUG("lut_base = %d \n" , lut_base);

		config = (IOR32_OSDM(osdm_offset_val[osdm_type][PLANEn_CONFIG]) & (~0xf00)) |  (lut_base << 8);
		IOW32_OSDM(osdm_offset_val[osdm_type][PLANEn_CONFIG], config);

		for ( i=0 ; ( (i < osd_pal->len) && (i < OSD_LUT_SIZE) ) ; i++) {
			//PDEBUG("pallete[%d] = 0x%x lut_base_add= 0x%x osdm_type = %d \n" , i , get_rgba8888( osd_pal->transp[i], osd_pal->red[i] ,osd_pal->green[i], osd_pal->blue[i]), lut_base_add + 4*i , osdm_type );
			IOW32_OSDM(lut_base_add + 4*i, get_rgba8888( osd_pal->transp[i], osd_pal->red[i] ,osd_pal->green[i], osd_pal->blue[i]) );
		}
	} else {
		PDEBUG("Invalid osd type\n");
		return FAIL;

	}

	return SUCCESS;
}


/************************************************************************
	dmw96osdm_osd_palette_get

	Get the palette of the OSDM
    
*************************************************************************/
int dmw96osdm_osd_palette_get(struct fb_cmap* osd_pal, unsigned int osdm_type){	

	int i = 0;
    osdm_rgba8888 color;
	unsigned int lut_base = 0;

	if (IS_PLANE_OSD(osdm_type)){

		if (osdm_type == MAIN_OSD_PLANE) {
			lut_base = OSD0_LUT;
		} 
		if (osdm_type == SUB_OSD_PLANE_1) {
			lut_base = OSD1_LUT;
		} 
		else { /*osdm_type == SUB_OSD_PLANE_2*/
			lut_base = OSD2_LUT;
		}

		for ( i=0 ; i < OSD_LUT_SIZE ; i++) {

			color = ( IOR32_OSDM(lut_base + 4*i) ) ;
			osd_pal->red[i] 	= get_red_from_rgba8888(color);
			osd_pal->green[i] 	= get_green_from_rgba8888(color);
			osd_pal->blue[i] 	= get_blue_from_rgba8888(color);
			osd_pal->transp[i] = get_alpha_from_rgba8888(color);

		}
	} else {
		PDEBUG("Invalid osd type\n");
		return FAIL;
	}

	return SUCCESS;
}


int dmw96osdm_get_regs(struct osdm_regs* pregs)
{
	unsigned int i = 0;

	if (!pregs)
		return -1;

	pregs->reg[i].offset = 	    OSDM_CONFIG;
	pregs->reg[i++].val = IOR32_OSDM(OSDM_CONFIG);
	
	pregs->reg[i].offset =		INTR_STATUS;
	pregs->reg[i++].val = IOR32_OSDM(INTR_STATUS);
	
	pregs->reg[i].offset =		INTR_MASK;
	pregs->reg[i++].val = IOR32_OSDM(INTR_MASK);
	
	pregs->reg[i].offset =		INTR_CAUSE;
	pregs->reg[i++].val = IOR32_OSDM(INTR_CAUSE);
	
	pregs->reg[i].offset =	INTR_CLEAR;
	pregs->reg[i++].val = IOR32_OSDM(INTR_CLEAR);
	
	pregs->reg[i].offset =		DMA_ERROR;
	pregs->reg[i++].val = IOR32_OSDM(DMA_ERROR);
	
	pregs->reg[i].offset =		DISPLAY_DIM;
	pregs->reg[i++].val = IOR32_OSDM(DISPLAY_DIM);
	
	pregs->reg[i].offset =		DISPLAY_BGC;
	pregs->reg[i++].val = IOR32_OSDM(DISPLAY_BGC);
	
	pregs->reg[i].offset =		VIDEO0_CONFIG;
	pregs->reg[i++].val = IOR32_OSDM(VIDEO0_CONFIG);
	
	pregs->reg[i].offset =		VIDEO0_POS;
	pregs->reg[i++].val = IOR32_OSDM(VIDEO0_POS);
	
	pregs->reg[i].offset =		VIDEO0_DIM;
	pregs->reg[i++].val = IOR32_OSDM(VIDEO0_DIM);
	
	pregs->reg[i].offset = 	    VIDEO0_BCSH;
	pregs->reg[i++].val = IOR32_OSDM(VIDEO0_BCSH);

	pregs->reg[i].offset =		VIDEO0_TPOS;
	pregs->reg[i++].val = IOR32_OSDM(VIDEO0_TPOS);

	pregs->reg[i].offset =		VIDEO0_TDIM;
	pregs->reg[i++].val = IOR32_OSDM(VIDEO0_TDIM);

	pregs->reg[i].offset =	VIDEO0_TPIXEL;
	pregs->reg[i++].val = IOR32_OSDM(VIDEO0_TPIXEL);

	pregs->reg[i].offset =	VIDEO0_ALCK;
	pregs->reg[i++].val = IOR32_OSDM(VIDEO0_ALCK);

	pregs->reg[i].offset =	VIDEO0_YUVLIM;
	pregs->reg[i++].val = IOR32_OSDM(VIDEO0_YUVLIM);
	
	pregs->reg[i].offset =	VIDEO1_CONFIG;
	pregs->reg[i++].val = IOR32_OSDM(VIDEO1_CONFIG);

	pregs->reg[i].offset =	VIDEO1_POS;
	pregs->reg[i++].val = IOR32_OSDM(VIDEO1_POS);

	pregs->reg[i].offset =	VIDEO1_DIM;
	pregs->reg[i++].val = IOR32_OSDM(VIDEO1_DIM);

	pregs->reg[i].offset =	VIDEO1_BCSH;
	pregs->reg[i++].val = IOR32_OSDM(VIDEO1_BCSH);

	pregs->reg[i].offset =	VIDEO1_TPOS;
	pregs->reg[i++].val = IOR32_OSDM(VIDEO1_TPOS);

	pregs->reg[i].offset =	VIDEO1_TDIM;
	pregs->reg[i++].val = IOR32_OSDM(VIDEO1_TDIM);

	pregs->reg[i].offset = VIDEO1_TPIXEL;
	pregs->reg[i++].val = IOR32_OSDM(VIDEO1_TPIXEL);

	pregs->reg[i].offset =	VIDEO1_ALCK;
	pregs->reg[i++].val = IOR32_OSDM(VIDEO1_ALCK);

	pregs->reg[i].offset =	VIDEO1_YUVLIM;
	pregs->reg[i++].val = IOR32_OSDM(VIDEO1_YUVLIM);

	pregs->reg[i].offset =	OSD0_CONFIG;
	pregs->reg[i++].val = IOR32_OSDM(OSD0_CONFIG);

	pregs->reg[i].offset =	OSD0_POS;
	pregs->reg[i++].val = IOR32_OSDM(OSD0_POS);

	pregs->reg[i].offset =	OSD0_DIM;
	pregs->reg[i++].val = IOR32_OSDM(OSD0_DIM);

	pregs->reg[i].offset =	OSD0_TPOS;
	pregs->reg[i++].val = IOR32_OSDM(OSD0_TPOS);

	pregs->reg[i].offset = 	OSD0_TDIM;
	pregs->reg[i++].val = IOR32_OSDM(OSD0_TDIM);

	pregs->reg[i].offset =	OSD0_TPIXEL;
	pregs->reg[i++].val = IOR32_OSDM(OSD0_TPIXEL);

	pregs->reg[i].offset =	OSD0_ALCK;
	pregs->reg[i++].val = IOR32_OSDM(OSD0_ALCK);

	pregs->reg[i].offset =	OSD1_CONFIG;
	pregs->reg[i++].val = IOR32_OSDM(OSD1_CONFIG);

	pregs->reg[i].offset =	OSD1_POS;
	pregs->reg[i++].val = IOR32_OSDM(OSD1_POS);

	pregs->reg[i].offset =	OSD1_DIM;
	pregs->reg[i++].val = IOR32_OSDM(OSD1_DIM);

	pregs->reg[i].offset =	OSD1_TPOS;
	pregs->reg[i++].val = IOR32_OSDM(OSD1_TPOS);

	pregs->reg[i].offset =	OSD1_TDIM;
	pregs->reg[i++].val = IOR32_OSDM(OSD1_TDIM);

	pregs->reg[i].offset =	OSD1_TPIXEL;
	pregs->reg[i++].val = IOR32_OSDM(OSD1_TPIXEL);

	pregs->reg[i].offset =	OSD1_ALCK;
	pregs->reg[i++].val = IOR32_OSDM(OSD1_ALCK);

	pregs->reg[i].offset =	OSD2_CONFIG;
	pregs->reg[i++].val = IOR32_OSDM(OSD2_CONFIG);

	pregs->reg[i].offset =	OSD2_POS;
	pregs->reg[i++].val = IOR32_OSDM(OSD2_POS);

	pregs->reg[i].offset =	OSD2_DIM;
	pregs->reg[i++].val = IOR32_OSDM(OSD2_DIM);

	pregs->reg[i].offset =	OSD2_TPOS;
	pregs->reg[i++].val = IOR32_OSDM(OSD2_TPOS);

	pregs->reg[i].offset =	OSD2_TDIM;
	pregs->reg[i++].val = IOR32_OSDM(OSD2_TDIM);

	pregs->reg[i].offset =	OSD2_TPIXEL;
	pregs->reg[i++].val = IOR32_OSDM(OSD2_TPIXEL);

	pregs->reg[i].offset =	OSD2_ALCK;
	pregs->reg[i++].val = IOR32_OSDM(OSD2_ALCK);

	pregs->reg[i].offset =	DMA0_CONFIG;
	pregs->reg[i++].val = IOR32_OSDM(DMA0_CONFIG);

	pregs->reg[i].offset =	DMA0_BADDR0;
	pregs->reg[i++].val = IOR32_OSDM(DMA0_BADDR0);

	pregs->reg[i].offset =	DMA0_BADDR1;
	pregs->reg[i++].val = IOR32_OSDM(DMA0_BADDR1);

	pregs->reg[i].offset =	DMA0_STRIDE;
	pregs->reg[i++].val = IOR32_OSDM(DMA0_STRIDE);

	pregs->reg[i].offset =	DMA0_BSIZE;
	pregs->reg[i++].val = IOR32_OSDM(DMA0_BSIZE);

	pregs->reg[i].offset =	DMA0_BCOUNT;
	pregs->reg[i++].val = IOR32_OSDM(DMA0_BCOUNT);

	pregs->reg[i].offset =	DMA0_TPOS;
	pregs->reg[i++].val = IOR32_OSDM(DMA0_TPOS);

	pregs->reg[i].offset =	DMA0_TDIM;
	pregs->reg[i++].val = IOR32_OSDM(DMA0_TDIM);

	pregs->reg[i].offset =	DMA1_CONFIG;
	pregs->reg[i++].val = IOR32_OSDM(DMA1_CONFIG);

	pregs->reg[i].offset =	DMA1_BADDR0;
	pregs->reg[i++].val = IOR32_OSDM(DMA1_BADDR0);

	pregs->reg[i].offset =	DMA1_BADDR1;
	pregs->reg[i++].val = IOR32_OSDM(DMA1_BADDR1);

	pregs->reg[i].offset =	DMA1_STRIDE;
	pregs->reg[i++].val = IOR32_OSDM(DMA1_STRIDE);

	pregs->reg[i].offset =	DMA1_BSIZE;
	pregs->reg[i++].val = IOR32_OSDM(DMA1_BSIZE);

	pregs->reg[i].offset =	DMA1_BCOUNT;
	pregs->reg[i++].val = IOR32_OSDM(DMA1_BCOUNT);

	pregs->reg[i].offset =	DMA1_TPOS;
	pregs->reg[i++].val = IOR32_OSDM(DMA1_TPOS);

	pregs->reg[i].offset =	DMA1_TDIM;
	pregs->reg[i++].val = IOR32_OSDM(DMA1_TDIM);

	pregs->reg[i].offset =	DMA2_CONFIG;
	pregs->reg[i++].val = IOR32_OSDM(DMA2_CONFIG);

	pregs->reg[i].offset =	DMA2_BADDR0;
	pregs->reg[i++].val = IOR32_OSDM(DMA2_BADDR0);

	pregs->reg[i].offset =	DMA2_BADDR1;
	pregs->reg[i++].val = IOR32_OSDM(DMA2_BADDR1);

	pregs->reg[i].offset =	DMA2_STRIDE;
	pregs->reg[i++].val = IOR32_OSDM(DMA2_STRIDE);

	pregs->reg[i].offset =	DMA2_BSIZE;
	pregs->reg[i++].val = IOR32_OSDM(DMA2_BSIZE);

	pregs->reg[i].offset =	DMA2_BCOUNT;
	pregs->reg[i++].val = IOR32_OSDM(DMA2_BCOUNT);

	pregs->reg[i].offset =	DMA2_TPOS;
	pregs->reg[i++].val = IOR32_OSDM(DMA2_TPOS);

	pregs->reg[i].offset =	DMA2_TDIM;
	pregs->reg[i++].val = IOR32_OSDM(DMA2_TDIM);

	pregs->reg[i].offset =	DMA3_CONFIG;
	pregs->reg[i++].val = IOR32_OSDM(DMA3_CONFIG);

	pregs->reg[i].offset =	DMA3_BADDR0;
	pregs->reg[i++].val = IOR32_OSDM(DMA3_BADDR0);

	pregs->reg[i].offset =	DMA3_BADDR1;
	pregs->reg[i++].val = IOR32_OSDM(DMA3_BADDR1);

	pregs->reg[i].offset =	DMA3_STRIDE;
	pregs->reg[i++].val = IOR32_OSDM(DMA3_STRIDE);

	pregs->reg[i].offset =	DMA3_BSIZE;
	pregs->reg[i++].val = IOR32_OSDM(DMA3_BSIZE);

	pregs->reg[i].offset =	DMA3_BCOUNT;
	pregs->reg[i++].val = IOR32_OSDM(DMA3_BCOUNT);

	pregs->reg[i].offset =	DMA3_TPOS;
	pregs->reg[i++].val = IOR32_OSDM(DMA3_TPOS);

	pregs->reg[i].offset =	DMA3_TDIM;
	pregs->reg[i++].val = IOR32_OSDM(DMA3_TDIM);
	
	pregs->reg[i].offset =	DMA4_CONFIG;
	pregs->reg[i++].val = IOR32_OSDM(DMA4_CONFIG);

	pregs->reg[i].offset =	DMA4_BADDR0;
	pregs->reg[i++].val = IOR32_OSDM(DMA4_BADDR0);

	pregs->reg[i].offset =	DMA4_BADDR1;
	pregs->reg[i++].val = IOR32_OSDM(DMA4_BADDR1);

	pregs->reg[i].offset =	DMA4_STRIDE;
	pregs->reg[i++].val = IOR32_OSDM(DMA4_STRIDE);

	pregs->reg[i].offset =	DMA4_BSIZE;
	pregs->reg[i++].val = IOR32_OSDM(DMA4_BSIZE);

	pregs->reg[i].offset =	DMA4_BCOUNT;
	pregs->reg[i++].val = IOR32_OSDM(DMA4_BCOUNT);

	pregs->reg[i].offset =	DMA4_TPOS;
	pregs->reg[i++].val = IOR32_OSDM(DMA4_TPOS);

	pregs->reg[i].offset =	DMA4_TDIM;
	pregs->reg[i++].val = IOR32_OSDM(DMA4_TDIM);

	pregs->reg[i].offset =	DMA5_CONFIG;
	pregs->reg[i++].val = IOR32_OSDM(DMA5_CONFIG);

	pregs->reg[i].offset =	DMA5_BADDR0;
	pregs->reg[i++].val = IOR32_OSDM(DMA5_BADDR0);

	pregs->reg[i].offset =	DMA5_BADDR1;
	pregs->reg[i++].val = IOR32_OSDM(DMA5_BADDR1);

	pregs->reg[i].offset =	DMA5_STRIDE;
	pregs->reg[i++].val = IOR32_OSDM(DMA5_STRIDE);

	pregs->reg[i].offset =	DMA5_BSIZE;
	pregs->reg[i++].val = IOR32_OSDM(DMA5_BSIZE);

	pregs->reg[i].offset =	DMA5_BCOUNT;
	pregs->reg[i++].val = IOR32_OSDM(DMA5_BCOUNT);

	pregs->reg[i].offset =	DMA5_TPOS;
	pregs->reg[i++].val = IOR32_OSDM(DMA5_TPOS);

	pregs->reg[i].offset =	DMA5_TDIM;
	pregs->reg[i++].val = IOR32_OSDM(DMA5_TDIM);

	pregs->reg[i].offset =	DMA6_CONFIG;
	pregs->reg[i++].val = IOR32_OSDM(DMA6_CONFIG);

	pregs->reg[i].offset =	DMA6_BADDR0;
	pregs->reg[i++].val = IOR32_OSDM(DMA6_BADDR0);

	pregs->reg[i].offset =	DMA6_BADDR1;
	pregs->reg[i++].val = IOR32_OSDM(DMA6_BADDR1);

	pregs->reg[i].offset =	DMA6_STRIDE;
	pregs->reg[i++].val = IOR32_OSDM(DMA6_STRIDE);

	pregs->reg[i].offset =	DMA6_BSIZE;
	pregs->reg[i++].val = IOR32_OSDM(DMA6_BSIZE);

	pregs->reg[i].offset =	DMA6_BCOUNT;
	pregs->reg[i++].val = IOR32_OSDM(DMA6_BCOUNT);

	pregs->reg[i].offset =	DMA6_TPOS;
	pregs->reg[i++].val = IOR32_OSDM(DMA6_TPOS);

	pregs->reg[i].offset =	DMA6_TDIM;
	pregs->reg[i++].val = IOR32_OSDM(DMA6_TDIM);

	pregs->reg[i].offset =	DMA7_CONFIG;
	pregs->reg[i++].val = IOR32_OSDM(DMA7_CONFIG);

	pregs->reg[i].offset =	DMA7_BADDR0;
	pregs->reg[i++].val = IOR32_OSDM(DMA7_BADDR0);

	pregs->reg[i].offset =	DMA7_BADDR1;
	pregs->reg[i++].val = IOR32_OSDM(DMA7_BADDR1);

	pregs->reg[i].offset =	DMA7_STRIDE;
	pregs->reg[i++].val = IOR32_OSDM(DMA7_STRIDE);

	pregs->reg[i].offset =	DMA7_BSIZE;
	pregs->reg[i++].val = IOR32_OSDM(DMA7_BSIZE);

	pregs->reg[i].offset =	DMA7_BCOUNT;
	pregs->reg[i++].val = IOR32_OSDM(DMA7_BCOUNT);

	pregs->reg[i].offset =	DMA7_TPOS;
	pregs->reg[i++].val = IOR32_OSDM(DMA7_TPOS);

	pregs->reg[i].offset =	DMA7_TDIM;
	pregs->reg[i++].val = IOR32_OSDM(DMA7_TDIM);

	pregs->reg[i].offset =	DMA8_CONFIG;
	pregs->reg[i++].val = IOR32_OSDM(DMA8_CONFIG);

	pregs->reg[i].offset =	DMA8_BADDR0;
	pregs->reg[i++].val = IOR32_OSDM(DMA8_BADDR0);

	pregs->reg[i].offset =	DMA8_BADDR1;
	pregs->reg[i++].val = IOR32_OSDM(DMA8_BADDR1);

	pregs->reg[i].offset =	DMA8_STRIDE;
	pregs->reg[i++].val = IOR32_OSDM(DMA8_STRIDE);

	pregs->reg[i].offset =	DMA8_BSIZE;
	pregs->reg[i++].val = IOR32_OSDM(DMA8_BSIZE);

	pregs->reg[i].offset =	DMA8_BCOUNT;
	pregs->reg[i++].val = IOR32_OSDM(DMA8_BCOUNT);

	pregs->reg[i].offset =	DMA8_TPOS;
	pregs->reg[i++].val = IOR32_OSDM(DMA8_TPOS);

	pregs->reg[i].offset =	DMA8_TDIM;
	pregs->reg[i++].val = IOR32_OSDM(DMA8_TDIM);

	pregs->reg[i].offset =	DMA9_CONFIG;
	pregs->reg[i++].val = IOR32_OSDM(DMA9_CONFIG);

	pregs->reg[i].offset =	DMA9_BADDR0;
	pregs->reg[i++].val = IOR32_OSDM(DMA9_BADDR0);

	pregs->reg[i].offset =	DMA9_BADDR1;
	pregs->reg[i++].val = IOR32_OSDM(DMA9_BADDR1);

	pregs->reg[i].offset =	DMA9_STRIDE;
	pregs->reg[i++].val = IOR32_OSDM(DMA9_STRIDE);

	pregs->reg[i].offset =	DMA9_BSIZE;
	pregs->reg[i++].val = IOR32_OSDM(DMA9_BSIZE);

	pregs->reg[i].offset =	DMA9_BCOUNT;
	pregs->reg[i++].val = IOR32_OSDM(DMA9_BCOUNT);

	pregs->reg[i].offset =	DMA9_TPOS;
	pregs->reg[i++].val = IOR32_OSDM(DMA9_TPOS);

   	pregs->reg[i].offset = 	DMA9_TDIM;
	pregs->reg[i++].val = IOR32_OSDM(DMA9_TDIM);

	pregs->reg[i].offset = 	OSD0_LUT;
	pregs->reg[i++].val = IOR32_OSDM(OSD0_LUT);

	pregs->reg[i].offset = 	OSD1_LUT;
	pregs->reg[i++].val = IOR32_OSDM(OSD1_LUT);

	pregs->reg[i].offset = 	OSD2_LUT;
	pregs->reg[i++].val = IOR32_OSDM(OSD2_LUT);

	pregs->len = i;

	return 0;
}


