#include <linux/kernel.h>
#include <linux/delay.h>
#include <linux/gpio.h>
#include <linux/platform_device.h>
#include <linux/spi/spi.h>
#include <video/dw74fb.h>
#include <video/amoled.h>
#include <linux/earlysuspend.h>

struct amoled {
	struct spi_device *spi;
	struct amoled_platform_data *pdata;

	struct early_suspend early_suspend;
};

struct amoled_commands {
	unsigned char command;
	unsigned char argument;
};

#define AMOLED_END_SEQ 0xff

static struct amoled_commands amoled_gamma_table_l_220[] = {
	{0x40,	0x00},
	{0x41,	0x3f},
	{0x42,	0x28},
	{0x43,	0x28},
	{0x44,	0x28},
	{0x45,	0x20},
	{0x46,	0x40},
	{0x50,	0x00},
	{0x51,	0x00},
	{0x52,	0x11},
	{0x53,	0x25},
	{0x54,	0x27},
	{0x55,	0x20},
	{0x56,	0x3f},
	{0x60,	0x00},
	{0x61,	0x3F},
	{0x62,	0x27},
	{0x63,	0x26},
	{0x64,	0x26},
	{0x65,	0x1c},
	{0x66,	0x56},
	{AMOLED_END_SEQ,0}
};

#if 0
static struct amoled_commands amoled_gamma_table_l_250[] = {
	{0x40,	0x00},
	{0x41,	0x3f},
	{0x42,	0x2a},
	{0x43,	0x27},
	{0x44,	0x27},
	{0x45,	0x1f},
	{0x46,	0x44},
	{0x50,	0x00},
	{0x51,	0x00},
	{0x52,	0x17},
	{0x53,	0x24},
	{0x54,	0x26},
	{0x55,	0x1F},
	{0x56,	0x43},
	{0x60,	0x00},
	{0x61,	0x3F},
	{0x62,	0x2A},
	{0x63,	0x25},
	{0x64,	0x24},
	{0x65,	0x1B},
	{0x66,	0x5C},
	{AMOLED_END_SEQ,0}
};


static struct amoled_commands amoled_gamma_table_l_100[] = {
	{0x40,	0x00},
	{0x41,	0x3f},
	{0x42,	0x30},
	{0x43,	0x2A},
	{0x44,	0x2B},
	{0x45,	0x24},
	{0x46,	0x2F},
	{0x50,	0x00},
	{0x51,	0x00},
	{0x52,	0x00},
	{0x53,	0x25},
	{0x54,	0x29},
	{0x55,	0x24},
	{0x56,	0x2E},
	{0x60,	0x00},
	{0x61,	0x3F},
	{0x62,	0x2F},
	{0x63,	0x29},
	{0x64,	0x29},
	{0x65,	0x21},
	{0x66,	0x3F},
	{AMOLED_END_SEQ,0}
};

static struct amoled_commands amoled_gamma_table_l_160[] = {
	{0x40,	0x00},
	{0x41,	0x3f},
	{0x42,	0x2B},
	{0x43,	0x29},
	{0x44,	0x28},
	{0x45,	0x23},
	{0x46,	0x38},
	{0x50,	0x00},
	{0x51,	0x00},
	{0x52,	0x0B},
	{0x53,	0x25},
	{0x54,	0x27},
	{0x55,	0x23},
	{0x56,	0x37},
	{0x60,	0x00},
	{0x61,	0x3F},
	{0x62,	0x29},
	{0x63,	0x28},
	{0x64,	0x25},
	{0x65,	0x20},
	{0x66,	0x4B},
	{AMOLED_END_SEQ,0}
};
#endif

static void
amoled_write_spi_command(struct amoled *amoled, unsigned char spi_command,
                         unsigned char spi_arg)
{
	unsigned char spi_cmd[2] = { spi_command, spi_arg };
	spi_write(amoled->spi, spi_cmd, 2);
}

#define AMOLED_COMMAND_ID  0x70
#define AMOLED_ARGUMET_ID  0x72

static void
amoled_write_command(struct amoled *amoled, unsigned char command,
                     unsigned char* arguments , unsigned int num_of_args)
{
	unsigned int i = 0;

	amoled_write_spi_command(amoled, AMOLED_COMMAND_ID, command);

	for (i=0 ; i < num_of_args ; i++ )
		amoled_write_spi_command(amoled,
		                         AMOLED_ARGUMET_ID, arguments[i]);
}


static void
amoled_write_simple_command(struct amoled *amoled, unsigned char command,
                            unsigned char argument)
{
	amoled_write_command(amoled, command, &argument , 1);
}


static void
amoled_run_sequence(struct amoled *amoled, struct amoled_commands *seq)
{
	while (seq->command != AMOLED_END_SEQ) {
		amoled_write_simple_command(amoled,
		                            seq->command, seq->argument);
		seq++;
	}
}

void amoled_pentile_key(struct amoled *amoled)
{
	unsigned char args[] = {0xd0,0xe8};
	amoled_write_command(amoled, 0xef, args, 2);
}

static void dmw96_amoled_init(struct amoled *amoled)
{

	amoled_write_simple_command(amoled, 0x31, 0x08 ); // SCTE set
	amoled_write_simple_command(amoled, 0x32, 0x14); // SCWE set
	amoled_write_simple_command(amoled, 0x30, 0x02); // Gateless signal
	amoled_write_simple_command(amoled, 0x27, 0x01); // Gateless signal

	//Display condition set
	amoled_write_simple_command(amoled, 0x12, 0x8); // VBP set
	amoled_write_simple_command(amoled, 0x13, 0x8); // VFP set
	amoled_write_simple_command(amoled, 0x15, 0x0); // Display con
	amoled_write_simple_command(amoled, 0x16, 0x0); // Color depth set

	amoled_pentile_key(amoled);// Pentile key

	amoled_write_simple_command(amoled, 0x39, 0x44); // Gamma set select

	//Normal gamma table
	amoled_run_sequence(amoled, amoled_gamma_table_l_220);

	// Analog power condition set
	amoled_write_simple_command(amoled, 0x17, 0x22); // VBP set
	amoled_write_simple_command(amoled, 0x18, 0x33); // VFP set
	amoled_write_simple_command(amoled, 0x19, 0x03); // Display con
	amoled_write_simple_command(amoled, 0x1A, 0x01); // Color depth set
	amoled_write_simple_command(amoled, 0x22, 0xA4); //
	amoled_write_simple_command(amoled, 0x23, 0x00); // Gamma set select
	amoled_write_simple_command(amoled, 0x26, 0xA0); // Gamma set select

	//STB off
	amoled_write_simple_command(amoled, 0x1D, 0xA0);
	msleep(100);

	//Display ON
	amoled_write_simple_command(amoled, 0x14, 0x03);
}

static void dmw96_amoled_reset(struct amoled *amoled)
{
	unsigned gpio = amoled->pdata->reset;

	gpio_direction_output(gpio, 1);
	udelay(100);
	gpio_set_value(gpio, 0);
	udelay(100);
	gpio_set_value(gpio, 1);
	udelay(100);
}

#ifdef CONFIG_HAS_EARLYSUSPEND
static void dmw96_amoled_early_suspend(struct early_suspend *es)
{
	struct amoled *amoled = container_of(es, struct amoled, early_suspend);

	amoled_write_simple_command(amoled, 0x14, 0x00);
	msleep(50);
	amoled_write_simple_command(amoled, 0x1D, 0xA1);
	msleep(200);

	gpio_set_value(amoled->pdata->reset, 0);
}

static void dmw96_amoled_late_resume(struct early_suspend *es)
{
	struct amoled *amoled = container_of(es, struct amoled, early_suspend);

	dmw96_amoled_reset(amoled);
	dmw96_amoled_init(amoled);

	msleep(20);
	amoled_write_simple_command(amoled, 0x1D,0xA0);
	msleep(250);
	amoled_write_simple_command(amoled, 0x14,0x03);
}
#endif

#if defined(CONFIG_PM) && !defined(CONFIG_HAS_EARLYSUSPEND)
static int dmw96_amoled_suspend(struct spi_device *spi, pm_message_t mesg)
{
	struct amoled *amoled = spi_get_drvdata(spi);

	amoled_write_simple_command(amoled, 0x14, 0x00);
	amoled_write_simple_command(amoled, 0x1D, 0xA1);

	gpio_set_value(amoled->pdata->reset, 0);
	return 0;
}

static int dmw96_amoled_resume(struct spi_device *spi)
{
	struct amoled *amoled = spi_get_drvdata(spi);

	dmw96_amoled_reset(amoled);
	dmw96_amoled_init(amoled);

	amoled_write_simple_command(amoled, 0x1D,0xA0);
	msleep(100);
	amoled_write_simple_command(amoled, 0x14,0x03);

	return 0;
}
#else
#define dmw96_amoled_suspend NULL
#define dmw96_amoled_resume  NULL
#endif

static int dmw96_amoled_probe(struct spi_device *spi)
{
	struct amoled_platform_data *pdata = spi->dev.platform_data;
	struct amoled *amoled;
	int initialized;
	int ret;

	if (!pdata) {
		dev_err(&spi->dev, "no platform data provided\n");
		return -ENODEV;
	}

	amoled = kzalloc(sizeof(*amoled), GFP_KERNEL);
	if (!amoled)
		return -ENOMEM;

	amoled->pdata = pdata;
	spi_set_drvdata(spi, amoled);

#ifdef CONFIG_HAS_EARLYSUSPEND
	amoled->early_suspend.suspend = dmw96_amoled_early_suspend;
	amoled->early_suspend.resume = dmw96_amoled_late_resume;
	amoled->early_suspend.level = EARLY_SUSPEND_LEVEL_BLANK_SCREEN;

	register_early_suspend(&amoled->early_suspend);
	dev_info(&spi->dev, "registered for early suspend\n");
#endif

	/*
	 * setup spi parameters; this makes sure that parameters we request
	 * are acceptable by the spi driver
	 */
	spi->mode = SPI_MODE_1; /* clk active high */
	spi->bits_per_word = 8;

	ret = spi_setup(spi);
	if (ret < 0) {
		dev_err(&spi->dev, "spi_setup() failed\n");
		goto err_amoled;
	}

	amoled->spi = spi;

	ret = gpio_request(amoled->pdata->reset, "amoled reset");
	if (ret < 0) {
		dev_err(&spi->dev, "failed requesting reset gpio\n");
		goto err_amoled;
	}

	/*
	 * Find out if we need to initialize the controller or if this already
	 * has been done earlier on (by e.g. the bootloader). Because the
	 * controller is write-only, we check if the reset gpio is configured
	 * for output. If so, it is assumed that the controller is initialized.
	 */
	ret = gpio_get_direction(amoled->pdata->reset);
	initialized = (ret == 0);

	if (!initialized) {
		dmw96_amoled_reset(amoled);
		dmw96_amoled_init(amoled);
	}

	dev_info(&spi->dev, "initialized\n");
	return 0;

err_amoled:
	kfree(amoled);
	return ret;
}

static int __devexit dmw96_amoled_remove(struct spi_device *spi)
{
	struct amoled *amoled = spi_get_drvdata(spi);
	kfree(amoled);

	return 0;
}

static struct spi_driver dmw96_amoled_spi_driver = {
	.driver = {
		.name  = "dmw96_amoled_spi",
		.bus   = &spi_bus_type,
		.owner = THIS_MODULE,
	},
	.probe	= dmw96_amoled_probe,
	.remove = __exit_p(dmw96_amoled_remove),
	.suspend = dmw96_amoled_suspend,
	.resume = dmw96_amoled_resume,
};

static int __init dmw96_amoled_spi_layer_init(void)
{
	return spi_register_driver(&dmw96_amoled_spi_driver);
}

module_init(dmw96_amoled_spi_layer_init);
