/*
 * Backlight code for TPS61160/TPS61161
 *
 * Copyright 2012 Yangao Zhang <ygzhang@grandstream.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/gpio.h>
#include <linux/fb.h>
#include <linux/backlight.h>
#include <linux/earlysuspend.h>
#include <linux/semaphore.h>
#include <linux/delay.h> 

struct bl_data
{
	struct backlight_device *bl;
	struct early_suspend early_suspend_backlight;
	unsigned int gpio;
	struct timer_list timer;
};
#ifdef CONFIG_HAS_EARLYSUSPEND
static int setup_early_suspend(struct bl_data *data);
#endif

void tps6116x_pwm(unsigned char data, unsigned int gpio, int delay)
{
	 u32 i, bit;

	 for (i = 8; i > 0; i -- )
	 {
		 bit = (data & (1 << (i - 1))) ? 1 : 0;
		 if(bit)
		 {
			 gpio_direction_output(gpio, 0);
			 udelay(delay);
			 gpio_direction_output(gpio, 1);
			 udelay(delay*3);
		 }
		 else
		 {
			 gpio_direction_output(gpio, 0);
			 udelay(delay*3);
			 gpio_direction_output(gpio, 1);
			 udelay(delay);
		 }
	 }
}

int tps6116x_set_bl(unsigned int level, unsigned int gpio, int delay)
{
        gpio_set_enable(gpio, 1);
        gpio_direction_output(gpio, 0);
        mdelay(3);
        gpio_direction_output(gpio, 1);
        udelay(200);
        gpio_direction_output(gpio, 0);
        udelay(270);
        gpio_direction_output(gpio, 1);
        udelay(4);

        /*start send addr*/
        tps6116x_pwm(0x72, gpio, delay);
        gpio_direction_output(gpio, 0);
        udelay(4);
        gpio_direction_output(gpio, 1);
        udelay(4);
        /*send value level*/
        tps6116x_pwm((level&0xff)>>3, gpio, delay);
        gpio_direction_output(gpio, 0);
        udelay(4);
        gpio_direction_output(gpio, 1);

        return 0;
}
 
static int tps6116x_bl_update(struct bl_data *bl_data)
{
	struct backlight_properties *props = &bl_data->bl->props;
	unsigned int brightness = props->brightness;

	if(props->power)
	{
		tps6116x_set_bl(brightness, bl_data->gpio, 5);
	}
	else
	{
		tps6116x_set_bl(0, bl_data->gpio, 5);
	}
	return 0;
}

void tps6116x_on_timer(unsigned long data)
{
	tps6116x_bl_update((struct bl_data *)data);
	return;
}

static int tps6116x_bl_update_status(struct backlight_device *dev)
{
	struct bl_data *bl_data = (struct bl_data *)dev_get_drvdata(&dev->dev);
	struct backlight_properties *props = &bl_data->bl->props;
	unsigned long expires = jiffies+HZ/100;

	if(props->power)
	{
		expires = jiffies+HZ/5;
	}	
	mod_timer(&bl_data->timer, expires);

	return 0;
}
 
static int tps6116x_bl_get_brightness(struct backlight_device *dev)
{
	struct backlight_properties *props = &dev->props;

	return props->brightness;
}
 
static struct backlight_ops tps6116x_bl_ops = {
	.get_brightness	 = tps6116x_bl_get_brightness,
	.update_status 	 = tps6116x_bl_update_status,
};
 
static int __devinit tps6116x_bl_probe(struct platform_device *pdev)
{
	struct backlight_properties props;
	int ret = 0;
	struct bl_data *data = kzalloc(sizeof(struct bl_data), GFP_KERNEL);

	if (!data)
		return -ENOMEM;

	data->gpio = (unsigned int )(pdev->dev.platform_data);
	ret = gpio_request(data->gpio, "LCD BL");
	if(ret)
	{
		goto err_bl_register;
	}

	memset(&props, 0, sizeof(struct backlight_properties));
	props.type = BACKLIGHT_RAW;
	props.max_brightness = 255;
	data->bl = backlight_device_register("tps6116x-bl", &pdev->dev ,data, &tps6116x_bl_ops,&props);
	if (IS_ERR(data->bl)) {
		ret = PTR_ERR(data->bl);
		goto err_bl_register;
	}

	data->bl->props.brightness = 0xff;
	data->bl->props.max_brightness = 255;
	data->bl->props.power = 1;

	init_timer(&data->timer);
	data->timer.function = tps6116x_on_timer;
	data->timer.data = (unsigned long)data;
	//tps6116x_bl_update(data);

#ifdef CONFIG_HAS_EARLYSUSPEND
	setup_early_suspend(data);
#endif

	return 0;

err_bl_register:
	kfree(data);
	return ret;
}
 
static int __devexit tps6116x_bl_remove(struct platform_device *pdev)
{
	struct bl_data *data = dev_get_drvdata(&pdev->dev);

#ifdef CONFIG_HAS_EARLYSUSPEND
	unregister_early_suspend(&data->early_suspend_backlight);
#endif

	backlight_device_unregister(data->bl);
	gpio_free(data->gpio);
	data->bl = NULL;

	kfree(data);

	return 0;
}

#ifdef CONFIG_HAS_EARLYSUSPEND
static void tps6116x_bl_early_suspend(struct early_suspend *es)
{
	struct bl_data *data = container_of(es, struct bl_data, early_suspend_backlight);// dev_get_drvdata(&pdev->dev);

	// Turn the backlight off
	data->bl->props.power = 0;
	tps6116x_bl_update_status(data->bl);
}

static void tps6116x_bl_late_resume(struct early_suspend *es)
{
	struct bl_data *data = container_of(es, struct bl_data, early_suspend_backlight);

	// Turn the backlight on
	data->bl->props.power = 1;
	tps6116x_bl_update_status(data->bl);
}
#else
#define tps6116x_bl_early_suspend NULL
#define tps6116x_bl_late_resume NULL
#endif 

static struct platform_driver tps6116x_bl_driver = {
	.probe = tps6116x_bl_probe,
	.remove = __devexit_p(tps6116x_bl_remove),
	.driver = {
		.name = "tps6116x-bl",
		.owner = THIS_MODULE,
	},
};

#ifdef CONFIG_HAS_EARLYSUSPEND
static int setup_early_suspend(struct bl_data *data)
{
	data->early_suspend_backlight.suspend = tps6116x_bl_early_suspend;
	data->early_suspend_backlight.resume = tps6116x_bl_late_resume;
	data->early_suspend_backlight.level = EARLY_SUSPEND_LEVEL_BLANK_SCREEN;

	register_early_suspend(&data->early_suspend_backlight);

	return 0;
}
#endif

static int __init tps6116x_bl_init(void)
{
	return platform_driver_register(&tps6116x_bl_driver);
}

static void __exit tps6116x_bl_exit(void)
{
	platform_driver_unregister(&tps6116x_bl_driver);
}

module_init(tps6116x_bl_init);
module_exit(tps6116x_bl_exit);

MODULE_AUTHOR("Yangao Zhang");
MODULE_LICENSE("GPL v2");
MODULE_DESCRIPTION("LCD/Backlight TPS61160/1");

