/*
 *  LCD / Backlight control code for DMW96
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/gpio.h>
#include <linux/fb.h>
#include <linux/backlight.h>
#include <video/dw_bl.h>
#include <linux/earlysuspend.h>
#include <linux/semaphore.h>
#include <video/dw_bl.h>
#include <video/dw74fb.h>

#ifdef CONFIG_HAS_EARLYSUSPEND
static int setup_early_suspend(struct dw_bl_data *data);
#endif

static int dmw_amoled_bl_update_status(struct backlight_device *dev)
{
	struct backlight_properties *props = &dev->props;
	unsigned int brightness = props->brightness;
	struct dw74fb* pdw74fb;

	pdw74fb = (struct dw74fb*) dev_get_drvdata(dev->dev.parent);

	if (pdw74fb != NULL) {
		/*printk("%s(%d) brightness = %d reg_value = 0x%lx\n",__func__,__LINE__ , brightness , dw74fb_readl(g_dw74fb , LCDC_REG_RBCR));*/
		dw74fb_writel(pdw74fb, LCDC_REG_RBCR, brightness);
		dw74fb_writel(pdw74fb, LCDC_REG_GBCR, brightness);
		dw74fb_writel(pdw74fb, LCDC_REG_BBCR, brightness);
		dw74fb_writel(pdw74fb, LCDC_REG_PARUP, 1);
	}
	
	return 0;
}

static int dmw_amoled_bl_get_brightness(struct backlight_device *dev)
{
	struct backlight_properties *props = &dev->props;

	return props->brightness;
}

static struct backlight_ops dmw_amoled_bl_ops = {
	.get_brightness		= dmw_amoled_bl_get_brightness,
	.update_status		= dmw_amoled_bl_update_status,
};

static int __devinit dmw_amoled_bl_probe(struct platform_device *pdev)
{
	struct backlight_properties props;
	int ret = 0;
	struct dw_bl_data *data = kzalloc(sizeof(struct dw_bl_data), GFP_KERNEL);

	if (!data)
		return -ENOMEM;

	memset(&props, 0, sizeof(struct backlight_properties));
	props.type = BACKLIGHT_RAW;
	props.max_brightness = 255;
	data->bl = backlight_device_register("dmw-amoled-bl", &pdev->dev ,data, &dmw_amoled_bl_ops,&props);
	if (IS_ERR(data->bl)) {
		ret = PTR_ERR(data->bl);
		goto err_bl_register;
	}

	data->bl->props.brightness = 0x80;
	data->bl->props.max_brightness = 255;
	data->bl->props.power = 1;

	backlight_update_status(data->bl);

#ifdef CONFIG_HAS_EARLYSUSPEND
	setup_early_suspend(data);
#endif

	return 0;

err_bl_register:
	kfree(data);
	return ret;
}

static int __devexit dmw_amoled_bl_remove(struct platform_device *pdev)
{
	struct dw_bl_data *data = dev_get_drvdata(&pdev->dev);

#ifdef CONFIG_HAS_EARLYSUSPEND
	unregister_early_suspend(&data->early_suspend_backlight);
#endif

	backlight_device_unregister(data->bl);
	data->bl = NULL;

	kfree(data);

	return 0;
}

#ifdef CONFIG_HAS_EARLYSUSPEND
static void dmw_amoled_bl_early_suspend(struct early_suspend *es)
{
	struct dw_bl_data *data = container_of(es, struct dw_bl_data, early_suspend_backlight);// dev_get_drvdata(&pdev->dev);

	// Turn the backlight off
	data->bl->props.power = 0;
	dmw_amoled_bl_update_status(data->bl);
}

static void dmw_amoled_bl_late_resume(struct early_suspend *es)
{
	struct dw_bl_data *data = container_of(es, struct dw_bl_data, early_suspend_backlight);

	// Turn the backlight on
	data->bl->props.power = 1;
	dmw_amoled_bl_update_status(data->bl);
}
#else
#define dmw_amoled_bl_early_suspend NULL
#define dmw_amoled_bl_late_resume NULL
#endif


static struct platform_driver dmw_amoled_bl_driver = {
	.probe = dmw_amoled_bl_probe,
	.remove = __devexit_p(dmw_amoled_bl_remove),
	.driver = {
		.name = "dmw-amoled-bl",
		.owner = THIS_MODULE,
	},
};

#ifdef CONFIG_HAS_EARLYSUSPEND
static int setup_early_suspend(struct dw_bl_data *data)
{
	data->early_suspend_backlight.suspend = dmw_amoled_bl_early_suspend;
	data->early_suspend_backlight.resume = dmw_amoled_bl_late_resume;
	data->early_suspend_backlight.level = EARLY_SUSPEND_LEVEL_BLANK_SCREEN;

	register_early_suspend(&data->early_suspend_backlight);

	return 0;
}
#endif

static int __init dmw_amoled_bl_init(void)
{
	return platform_driver_register(&dmw_amoled_bl_driver);
}

static void __exit dmw_amoled_bl_exit(void)
{
	platform_driver_unregister(&dmw_amoled_bl_driver);
}

module_init(dmw_amoled_bl_init);
module_exit(dmw_amoled_bl_exit);

MODULE_AUTHOR("DSP Group");
MODULE_LICENSE("GPL v2");
MODULE_DESCRIPTION("LCD/Backlight DMW AMOLED");
