/*
 * linux/drivers/video/adv7525_hdmi.c
 * I2C driver for ADV7525 (Analog Devices) HDMI controller
 *
 * Copyright (C) 2011 DSPG Technologies GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/i2c.h>
#include <linux/gpio.h>
#include <linux/interrupt.h>
#include <linux/switch.h>
#include <linux/gpio.h>
#include <linux/mutex.h>
#include <linux/slab.h>
#include <linux/earlysuspend.h>
#include <linux/i2c/adv7525_hdmi.h>

#define ADV7525_REG_CHIP_REVISION                   0x00ff00
#define ADV7525_REG_SPDIF_ENABLE                    0x0b8007
#define ADV7525_REG_INPUT_VIDEO_FORMAT              0x150e01
#define ADV7525_REG_ASPECT_RATIO                    0x170201
#define ADV7525_REG_POWER_DOWN                      0x414006
#define ADV7525_REG_RX_SENSE_STATE                  0x422005
#define ADV7525_REG_RX_SENSE_INTERRUPT_ENABLE       0x944006
#define ADV7525_REG_HPD_INTERRUPT_ENABLE            0x948007
#define ADV7525_REG_RX_SENSE_INTERRUPT              0x964006
#define ADV7525_REG_HPD_INTERRUPT                   0x968007
#define ADV7525_REG_RX_SENSE_POWER_DOWN             0xa14006
#define ADV7525_REG_HDMI_MODE                       0xaf0201
#define ADV7525_REG_CLOCK_DELAY                     0xbae005
#define ADV7525_REG_VIDEO_INPUT_AND_CLOCK_GATING    0xd60100
#define ADV7525_REG_CEC_POWER_DOWN                  0xe20100
#define ADV7525_REG_HDCP_CONTROLLER_POWER           0xe40201
#define ADV7525_REG_V1P2_ENABLE                     0xe48007
#define ADV7525_REG_RX_SENSE_MONITOR_OSC_POWER_DOWN 0xe60100
#define ADV7525_REG_CEC_CLOCK_INPUT_ENABLE          0xe62005

struct adv7525_dev {
	struct i2c_client *client;
	struct switch_dev hdmi_sdev;
	struct work_struct sense_work;
	struct adv7525_platform_data *pdata;
	struct early_suspend early_suspend;
	struct attribute_group *attr_group;
	struct mutex mutex;
	int mode;
	int reg;
	int irq;
};

enum adv7525_mode {
	ADV7525_MODE_OFF,
	ADV7525_MODE_MONITOR,
	ADV7525_MODE_ON,
};

static unsigned char identity_matrix[24] = {
	0xA8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 8, 0, 0, 0
};

static int
adv7525_read(struct adv7525_dev *adv7525, int reg)
{
	return i2c_smbus_read_byte_data(adv7525->client, reg);
}

static int
adv7525_write(struct adv7525_dev *adv7525, int reg, uint8_t val)
{
	return i2c_smbus_write_byte_data(adv7525->client, reg, val);
}

static int
adv7525_write_data(struct adv7525_dev *adv7525, int reg, unsigned char *data, int size)
{
	return i2c_smbus_write_block_data(adv7525->client, reg, size, data);
}

static int
adv7525_read_reg(struct adv7525_dev *adv7525, int reg)
{
	char tmp, bit, mask;

	tmp = adv7525_read(adv7525, reg>>16);
	mask = (reg>>8) & 0xff;
	bit = reg & 0xff;

	return (tmp & mask) >> bit;
}

static void
adv7525_write_reg(struct adv7525_dev *adv7525, int reg, int val)
{
	char tmp, bit, mask;

	tmp = adv7525_read(adv7525, reg>>16);
	mask = (reg>>8) & 0xff;
	bit = reg & 0xff;

	tmp = (tmp & ~mask) | (char)((val << bit) & mask);

	adv7525_write(adv7525, reg>>16, tmp);
}

static void
adv7525_check_state(struct adv7525_dev *adv7525)
{
	if (!adv7525_read_reg(adv7525, ADV7525_REG_RX_SENSE_POWER_DOWN)) {
		if (adv7525_read_reg(adv7525, ADV7525_REG_RX_SENSE_STATE))
			switch_set_state(&adv7525->hdmi_sdev, 1);
		else
			switch_set_state(&adv7525->hdmi_sdev, 0);
	}
}

static void
adv7525_sense_work(struct work_struct *data)
{
	struct adv7525_dev *adv7525 =
		container_of(data, struct adv7525_dev, sense_work);
	int val;

	mutex_lock(&adv7525->mutex);

	val = adv7525_read_reg(adv7525, ADV7525_REG_RX_SENSE_INTERRUPT);
	if (val) {
		adv7525_check_state(adv7525);
		adv7525_write_reg(adv7525, ADV7525_REG_RX_SENSE_INTERRUPT, 1);
	}

	mutex_unlock(&adv7525->mutex);
}

static irqreturn_t
adv7525_isr(int irq, void *dev_id)
{
	struct adv7525_dev *adv7525 = (struct adv7525_dev *)dev_id;

	schedule_work(&adv7525->sense_work);

	return IRQ_HANDLED;
}

static int
adv7525_init_controller(struct adv7525_dev *adv7525)
{
	/* power up */
	adv7525_write_reg(adv7525, ADV7525_REG_POWER_DOWN, 0);
	if (adv7525_read_reg(adv7525, ADV7525_REG_POWER_DOWN))
		return -ENODEV;

	/* fixed registers */
	adv7525_write_reg(adv7525, 0x980f00, 3);
	adv7525_write_reg(adv7525, 0x9cff00, 0x38);
	adv7525_write_reg(adv7525, 0x9d0300, 1);
	adv7525_write_reg(adv7525, 0xa2ff00, 0xa0);
	adv7525_write_reg(adv7525, 0xa3ff00, 0xa0);
	adv7525_write_reg(adv7525, 0xdeff00, 0x82);
	adv7525_write_reg(adv7525, 0xe44006, 1);
	adv7525_write_reg(adv7525, 0xe40402, 1);
	adv7525_write_reg(adv7525, 0xe58007, 1);
	adv7525_write_reg(adv7525, 0xe60201, 0);
	adv7525_write_reg(adv7525, 0xeb0201, 1);

	/* no delay for input video clk capture */
	adv7525_write_reg(adv7525, ADV7525_REG_CLOCK_DELAY, 3);
	/* disable HDCP */
	adv7525_write_reg(adv7525, ADV7525_REG_HDCP_CONTROLLER_POWER, 0);
	/* select HDMI */
	adv7525_write_reg(adv7525, ADV7525_REG_HDMI_MODE, 1);

	adv7525_write_data(adv7525, 0x18, identity_matrix, 24);

	/* input video format = 24bit RGB */
	adv7525_write_reg(adv7525, ADV7525_REG_INPUT_VIDEO_FORMAT, 0);

	/* aspect ratio of input video = 16x9 */
	adv7525_write_reg(adv7525, ADV7525_REG_ASPECT_RATIO, 1);

	/* disable SPDIF */
	adv7525_write_reg(adv7525, ADV7525_REG_SPDIF_ENABLE, 0);

	/* CEC power down */
	adv7525_write_reg(adv7525, ADV7525_REG_CEC_POWER_DOWN, 1);

	/* disable CEC clock */
	adv7525_write_reg(adv7525, ADV7525_REG_CEC_CLOCK_INPUT_ENABLE, 0);

	/* DVDD1P2 is 1.2V */
	adv7525_write_reg(adv7525, ADV7525_REG_V1P2_ENABLE, 1);

	/* clear RX sense interrupt */
	adv7525_write_reg(adv7525, ADV7525_REG_RX_SENSE_INTERRUPT, 1);

	/* disable RX sense interrupt */
	adv7525_write_reg(adv7525, ADV7525_REG_RX_SENSE_INTERRUPT_ENABLE, 0);

	/* clear HPD interrupt */
	adv7525_write_reg(adv7525, ADV7525_REG_HPD_INTERRUPT, 1);

	/* disable HPD interrupt */
	adv7525_write_reg(adv7525, ADV7525_REG_HPD_INTERRUPT_ENABLE, 0);

	return 0;
}

static void
adv7525_power_control(struct adv7525_dev *adv7525, int mode)
{
	mutex_lock(&adv7525->mutex);

	if (adv7525->pdata->power_gpio && mode)
		gpio_set_value(adv7525->pdata->power_gpio, 1);

	adv7525_write_reg(adv7525, ADV7525_REG_VIDEO_INPUT_AND_CLOCK_GATING, !mode);

	adv7525_write_reg(adv7525, ADV7525_REG_POWER_DOWN, !(mode==ADV7525_MODE_ON));

	adv7525_write_reg(adv7525, ADV7525_REG_RX_SENSE_POWER_DOWN, !mode);

	adv7525_write_reg(adv7525, ADV7525_REG_RX_SENSE_MONITOR_OSC_POWER_DOWN, !mode);

	adv7525_write_reg(adv7525, ADV7525_REG_RX_SENSE_INTERRUPT, 1);

	adv7525_write_reg(adv7525, ADV7525_REG_RX_SENSE_INTERRUPT_ENABLE, !!mode);

	if (mode)
		adv7525_write_reg(adv7525, ADV7525_REG_HDMI_MODE, 1);

	if (adv7525->pdata->power_gpio && !mode)
		gpio_set_value(adv7525->pdata->power_gpio, 0);

	adv7525_check_state(adv7525);

	mutex_unlock(&adv7525->mutex);
}

static ssize_t
adv7525_sysfs_show_mode(struct device *dev, struct device_attribute *attr,
                        char *buf)
{
	struct switch_dev *sdev = (struct switch_dev *)dev_get_drvdata(dev);
	struct adv7525_dev *adv7525 =
		container_of(sdev, struct adv7525_dev, hdmi_sdev);
	int ret = 0;

	mutex_lock(&adv7525->mutex);

	switch (adv7525->mode) {
	case ADV7525_MODE_OFF:
		ret = scnprintf(buf, PAGE_SIZE, "off\n");
		break;
	case ADV7525_MODE_MONITOR:
		ret = scnprintf(buf, PAGE_SIZE, "monitor\n");
		break;
	case ADV7525_MODE_ON:
		ret = scnprintf(buf, PAGE_SIZE, "on\n");
		break;
	default:
		ret = scnprintf(buf, PAGE_SIZE, "invalid\n");
		break;
	}

	mutex_unlock(&adv7525->mutex);

	return ret;
}

static ssize_t
adv7525_sysfs_store_mode(struct device *dev, struct device_attribute *attr,
                         const char *buf, size_t count)
{
	struct switch_dev *sdev = (struct switch_dev *)dev_get_drvdata(dev);
	struct adv7525_dev *adv7525 =
		container_of(sdev, struct adv7525_dev, hdmi_sdev);
	int ret = count;

	if (!strncmp(buf, "on", 2)) {
		adv7525_power_control(adv7525, ADV7525_MODE_ON);
		adv7525->mode = ADV7525_MODE_ON;
	} else if (!strncmp(buf, "monitor", 7)) {
		adv7525_power_control(adv7525, ADV7525_MODE_MONITOR);
		adv7525->mode = ADV7525_MODE_MONITOR;
	} else if (!strncmp(buf, "off", 3)) {
		adv7525_power_control(adv7525, ADV7525_MODE_OFF);
		adv7525->mode = ADV7525_MODE_OFF;
	} else {
		ret = -EINVAL;
	}

	return ret;
}

static DEVICE_ATTR(mode,       S_IWUSR | S_IRUGO, adv7525_sysfs_show_mode, adv7525_sysfs_store_mode);

static struct attribute *adv7525_attrs[] = {
	&dev_attr_mode.attr,
	NULL
};

static struct attribute_group adv7525_attr_group = {
	.attrs = adv7525_attrs,
};

#ifdef CONFIG_HAS_EARLYSUSPEND
static void
adv7525_early_suspend(struct early_suspend *es)
{
	struct adv7525_dev *adv7525 =
		container_of(es, struct adv7525_dev, early_suspend);

	if (!adv7525->mode)
		return;

	adv7525_power_control(adv7525, ADV7525_MODE_OFF);
}

static void
adv7525_late_resume(struct early_suspend *es)
{
	struct adv7525_dev *adv7525 =
		container_of(es, struct adv7525_dev, early_suspend);

	if (adv7525->mode)
		adv7525_power_control(adv7525, adv7525->mode);
}
#endif

static int
adv7525_probe(struct i2c_client *client,
              const struct i2c_device_id *dev_id)
{
	struct adv7525_dev *adv7525;
	struct adv7525_platform_data *pdata = client->dev.platform_data;
	int err = 0;

	if (!pdata) {
		dev_err(&client->dev, "missing platform data.\n");
		return -EINVAL;
	}

	if (!i2c_check_functionality(client->adapter,
	                             I2C_FUNC_SMBUS_BYTE |
	                             I2C_FUNC_SMBUS_BYTE_DATA |
	                             I2C_FUNC_SMBUS_WRITE_BLOCK_DATA)) {
		dev_err(&client->dev, "insufficient I2C functionality\n");
		goto exit;
	}

	adv7525 = kzalloc(sizeof(struct adv7525_dev), GFP_KERNEL);
	if (!adv7525) {
		err = -ENOMEM;
		dev_err(&client->dev, "out of memory\n");
		goto exit;
	}

	adv7525->pdata = pdata;
	adv7525->client = client;
	i2c_set_clientdata(client, adv7525);

	if (adv7525->pdata->power_gpio) {
		err = gpio_request(pdata->power_gpio, "adv7525");
		if (err) {
			dev_err(&client->dev, "gpio_request\n");
			goto exit_free;
		}

		gpio_direction_output(pdata->power_gpio, 1);
		gpio_set_enable(pdata->power_gpio, 1);
		if (err) {
			dev_err(&client->dev, "gpio_set_enable\n");
			goto exit_gpio;
		}
	}

	adv7525->hdmi_sdev.name = "hdmi";
	err = switch_dev_register(&adv7525->hdmi_sdev);
	if (unlikely(err)) {
		dev_err(&client->dev, "unable to register to switch framework.\n");
		goto exit_gpio;
	}

	mutex_init(&adv7525->mutex);
	INIT_WORK(&adv7525->sense_work, adv7525_sense_work);

#ifdef CONFIG_HAS_EARLYSUSPEND
	adv7525->early_suspend.suspend = adv7525_early_suspend;
	adv7525->early_suspend.resume = adv7525_late_resume;
	adv7525->early_suspend.level = EARLY_SUSPEND_LEVEL_DISABLE_FB;

	register_early_suspend(&adv7525->early_suspend);
#endif

	adv7525->attr_group = &adv7525_attr_group;
	err = sysfs_create_group(&adv7525->hdmi_sdev.dev->kobj, adv7525->attr_group);
	if (err) {
		dev_err(&client->dev, "unable to register sysfs attributes.\n");
		goto exit_early;
	}

	err = adv7525_init_controller(adv7525);
	if (err) {
		dev_err(&client->dev, "unable to power up controller.\n");
		goto exit_sysfs;
	}

	adv7525->irq = client->irq;
	err = request_irq(adv7525->irq, adv7525_isr,
	                  IRQF_TRIGGER_RISING | IRQF_TRIGGER_FALLING,
	                 client->name, adv7525);
	if (err) {
		dev_err(&client->dev, "unable to request interrupt.\n");
		goto exit_sysfs;
	}

	adv7525_power_control(adv7525, ADV7525_MODE_OFF);
	adv7525->mode = ADV7525_MODE_OFF;

	dev_info(&client->dev, "adv7525 registered, chip revision 0x%x\n",
	         adv7525_read_reg(adv7525, ADV7525_REG_CHIP_REVISION));

	return 0;

exit_sysfs:
	sysfs_remove_group(&client->dev.kobj, adv7525->attr_group);
exit_early:
	switch_dev_unregister(&adv7525->hdmi_sdev);
	unregister_early_suspend(&adv7525->early_suspend);
exit_gpio:
	if (adv7525->pdata->power_gpio)
		gpio_free(pdata->power_gpio);
exit_free:
	kfree(adv7525);
exit:
	return err;
}

static int __devexit
adv7525_remove(struct i2c_client *client)
{
	struct adv7525_platform_data *pdata = client->dev.platform_data;
	struct adv7525_dev *adv7525 = i2c_get_clientdata(client);

	free_irq(adv7525->irq, adv7525);
	sysfs_remove_group(&client->dev.kobj, adv7525->attr_group);
	switch_dev_unregister(&adv7525->hdmi_sdev);
	unregister_early_suspend(&adv7525->early_suspend);
	if (adv7525->pdata->power_gpio)
		gpio_free(pdata->power_gpio);
	kfree(adv7525);

	return 0;
}

static const struct i2c_device_id adv7525_id[] = {
	{ "adv7525", 0 },
	{}
};

static struct i2c_driver adv7525_i2c_driver = {
	.driver = {
		.name	= "adv7525",
	},
	.probe    = adv7525_probe,
	.remove   = __devexit_p(adv7525_remove),
	.id_table = adv7525_id,
};

static int __init
adv7525_init(void)
{
	return i2c_add_driver(&adv7525_i2c_driver);
}

static void __exit
adv7525_exit(void)
{
	return i2c_del_driver(&adv7525_i2c_driver);
}

MODULE_DEVICE_TABLE(i2c, adv7525_id);

MODULE_DESCRIPTION("ADV7525 driver");
MODULE_AUTHOR("DSP Group");
MODULE_LICENSE("GPL");

module_init(adv7525_init);
module_exit(adv7525_exit);
