/* ==========================================================================
 * $File: //dwh/usb_iip/dev/software/otg/linux/drivers/dwc_otg_pcd.c $
 * $Revision: 1.1 $
 * $Date: 2012/04/09 18:07:08 $
 * $Change: 1636033 $
 *
 * Synopsys HS OTG Linux Software Driver and documentation (hereinafter,
 * "Software") is an Unsupported proprietary work of Synopsys, Inc. unless
 * otherwise expressly agreed to in writing between Synopsys and you.
 *
 * The Software IS NOT an item of Licensed Software or Licensed Product under
 * any End User Software License Agreement or Agreement for Licensed Product
 * with Synopsys or any supplement thereto. You are permitted to use and
 * redistribute this Software in source and binary forms, with or without
 * modification, provided that redistributions of source code must retain this
 * notice. You may not view, use, disclose, copy or distribute this file or
 * any information contained herein except pursuant to this license grant from
 * Synopsys. If you do not agree with this notice, including the disclaimer
 * below, then you are not authorized to use the Software.
 *
 * THIS SOFTWARE IS BEING DISTRIBUTED BY SYNOPSYS SOLELY ON AN "AS IS" BASIS
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE HEREBY DISCLAIMED. IN NO EVENT SHALL SYNOPSYS BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 * ========================================================================== */
#ifndef DWC_HOST_ONLY

/** @file
 * This file implements PCD Core. All code in this file is portable and doesn't
 * use any OS specific functions.
 * PCD Core provides Interface, defined in <code><dwc_otg_pcd_if.h></code>
 * header file, which can be used to implement OS specific PCD interface.
 *
 * An important function of the PCD is managing interrupts generated
 * by the DWC_otg controller. The implementation of the DWC_otg device
 * mode interrupt service routines is in dwc_otg_pcd_intr.c.
 *
 * @todo Add Device Mode test modes (Test J mode, Test K mode, etc).
 * @todo Does it work when the request size is greater than DEPTSIZ
 * transfer size
 *
 */

#include "dwc_otg_pcd.h"

#ifdef DWC_UTE_CFI
#include "dwc_otg_cfi.h"

extern int init_cfi(dwc_otg_pcd_t *pcd, cfiobject_t * cfiobj);
#endif

int check_ep_for_in_token(dwc_otg_core_if_t * core_if, int epnum);

int check_next_ep_token(dwc_otg_pcd_ep_t * ep)
{
	extern int get_ep_of_last_in_token(dwc_otg_core_if_t * core_if);
	dwc_otg_core_if_t *core_if = GET_CORE_IF(ep->pcd);
#if 0
	int eptoken;
#endif
	int ret = 1;

	if ((core_if->en_multiple_tx_fifo == 0) && (ep->dwc_ep.type != DWC_OTG_EP_TYPE_ISOC)) {
#if 0
		eptoken = get_ep_of_last_in_token(core_if);
		if (ep->dwc_ep.num != eptoken) {
			DWC_DEBUGPL(DBG_PCD, "Last NP SHARED FIFO IN token is EP%d, requeueing transfer for EP%d\n",
					eptoken, ep->dwc_ep.num);
#else
		if (!check_ep_for_in_token(core_if, ep->dwc_ep.num)) {
#endif
#if 0
#if 1
			DWC_DEBUGPL(DBG_PCD, "No token for EP%d in NP SHARED FIFO IN, requeueing transfer\n", ep->dwc_ep.num);
			ep->queue_sof = 1;
			DWC_TASK_SCHEDULE(ep->pcd->start_xfer_tasklet);
#else
                        DWC_DEBUGPL(DBG_PCD, "No token for EP%d in NP SHARED FIFO IN, nuking transfer\n", ep->dwc_ep.num);
			dwc_otg_request_nuke(ep);
#endif
#endif
			ret = 0;
		}
	}

	return ret;
}

/**
 * Choose endpoint from ep arrays using usb_ep structure.
 */
static dwc_otg_pcd_ep_t *get_ep_from_handle(dwc_otg_pcd_t * pcd, void *handle)
{
	int i;
	if (pcd->ep0.priv == handle) {
		return &pcd->ep0;
	}
	for (i = 0; i < MAX_EPS_CHANNELS - 1; i++) {
		if (pcd->in_ep[i].priv == handle)
			return &pcd->in_ep[i];
		if (pcd->out_ep[i].priv == handle)
			return &pcd->out_ep[i];
	}

	return NULL;
}

/**
 * This function completes a request.  It call's the request call back.
 */
void dwc_otg_request_done(dwc_otg_pcd_ep_t * ep, dwc_otg_pcd_request_t * req,
			  int32_t status)
{
	unsigned stopped = ep->stopped;

	DWC_DEBUGPL(DBG_PCDV, "%s EP%d-%s (%p,%p,0x%x) %d req_pend:%d buf:%p len:%d (%d)\n", __func__,
		ep->dwc_ep.num, (ep->dwc_ep.is_in ? "IN" : "OUT"),
		ep, req, status, stopped, ep->pcd->request_pending,
		ep->dwc_ep.start_xfer_buff, ep->dwc_ep.xfer_len, req->actual);
	if (status == 0) {
		DWC_DEBUGPL(DBG_PCDV, "%s EP%d-%s\n", __func__, ep->dwc_ep.num, (ep->dwc_ep.is_in ? "IN" : "OUT"));
		dump_msg(DBG_PCDV, ep->dwc_ep.start_xfer_buff, req->actual);
	}

	/* don't modify queue heads during completion callback */
	ep->stopped = 1;

        DWC_DEBUGPL(DBG_PCDV, " complete(%p,%p,%p,0x%x,0x%x)\n",
		ep->pcd, ep->priv, req->priv, status, req->actual);
	ep->pcd->fops->complete(ep->pcd, ep->priv, req->priv, status, req->actual);

	if (ep->pcd->request_pending > 0) {
		--ep->pcd->request_pending;
	}

	ep->stopped = stopped;
	DWC_FREE(req);
}

/**
 * This function terminates all the requsts in the EP request queue.
 */
void dwc_otg_request_nuke(dwc_otg_pcd_ep_t * ep)
{
	dwc_otg_pcd_request_t *req;
	dwc_otg_pcd_t *pcd;

	DWC_DEBUGPL(DBG_PCDV, "%s\n", __func__);

	pcd = ep->pcd;
	ep->stopped = 1;

	while (!DWC_CIRCLEQ_EMPTY(&ep->queue)) {
		req = DWC_CIRCLEQ_FIRST(&ep->queue);
		DWC_CIRCLEQ_REMOVE_INIT(&ep->queue, req, queue_entry);
		dwc_otg_request_done(ep, req, -DWC_E_SHUTDOWN);
	}
}

void dwc_otg_pcd_start(dwc_otg_pcd_t * pcd,
		       const struct dwc_otg_pcd_function_ops *fops)
{
	pcd->fops = fops;
}

/**
 * PCD Callback function for initializing the PCD when switching to
 * device mode.
 *
 * @param p void pointer to the <code>dwc_otg_pcd_t</code>
 */
static int32_t dwc_otg_pcd_start_cb(void *p)
{
	dwc_otg_pcd_t *pcd = (dwc_otg_pcd_t *) p;
	dwc_otg_core_if_t *core_if = GET_CORE_IF(pcd);

	/*
	 * Initialized the Core for Device mode.
	 */
	if (dwc_otg_is_device_mode(core_if)) {
		dwc_otg_core_dev_init(core_if);
		//dwc_otg_pcd_soft_disconnect(pcd);
	}
	return 1;
}

/** CFI-specific buffer allocation function for EP */
#ifdef DWC_UTE_CFI
uint8_t *cfiw_ep_alloc_buffer(dwc_otg_pcd_t * pcd, void *pep, dwc_dma_t * addr,
			      size_t buflen, int flags)
{
	dwc_otg_pcd_ep_t *ep;
	ep = get_ep_from_handle(pcd, pep);
	if (!ep) {
		DWC_WARN("bad ep\n");
		return -DWC_E_INVALID;
	}

	return pcd->cfi->ops.ep_alloc_buf(pcd->cfi, pcd, ep, addr, buflen,
					  flags);
}
#else
uint8_t *cfiw_ep_alloc_buffer(dwc_otg_pcd_t * pcd, void *pep, dwc_dma_t * addr,
			      size_t buflen, int flags);
#endif

/**
 * PCD Callback function for notifying the PCD when resuming from
 * suspend.
 *
 * @param p void pointer to the <code>dwc_otg_pcd_t</code>
 */
static int32_t dwc_otg_pcd_resume_cb(void *p)
{
	dwc_otg_pcd_t *pcd = (dwc_otg_pcd_t *) p;

	if (pcd->fops->resume) {
		pcd->fops->resume(pcd);
	}

	/* Stop the SRP timeout timer. */
	if ((GET_CORE_IF(pcd)->core_params->phy_type != DWC_PHY_TYPE_PARAM_FS)
	    || (!GET_CORE_IF(pcd)->core_params->i2c_enable)) {
		if (GET_CORE_IF(pcd)->srp_timer_started) {
			GET_CORE_IF(pcd)->srp_timer_started = 0;
			DWC_TIMER_CANCEL(GET_CORE_IF(pcd)->srp_timer);
		}
	}
	return 1;
}

/**
 * PCD Callback function for notifying the PCD device is suspended.
 *
 * @param p void pointer to the <code>dwc_otg_pcd_t</code>
 */
static int32_t dwc_otg_pcd_suspend_cb(void *p)
{
	dwc_otg_pcd_t *pcd = (dwc_otg_pcd_t *) p;

	if (pcd->fops->suspend) {
		pcd->fops->suspend(pcd);
	}

	return 1;
}

#ifdef DEBUG_DWC_OTG_PCD_DMA_AHB_ERROR
static void start_ep0_out_dma_tasklet_func(void *data)
{
	dwc_otg_pcd_t *pcd = (dwc_otg_pcd_t *) data;
	dwc_otg_core_if_t *core_if = GET_CORE_IF(pcd);
	dwc_otg_dev_if_t *dev_if = core_if->dev_if;
	depctl_data_t doepctl = {.d32 = 0 };
	dwc_otg_dev_dma_desc_t *dma_desc;
	
	DWC_OTG_PCD_DUMP_DMA_EP0_OUT_REGS(pcd);
	doepctl.b.epena = 1;
	doepctl.b.cnak = 1;
	DWC_WRITE_REG32(&dev_if->out_ep_regs[0]->doepctl, doepctl.d32);
	DWC_OTG_PCD_DUMP_DMA_EP0_OUT_REGS(pcd);
}
#endif

#ifdef DWC_PCD_STOP_TASKLET
static void dwc_otg_pcd_stop_tasklet_func(void *data)
{
	extern void dwc_otg_pcd_stop(dwc_otg_pcd_t * _pcd);
	dwc_otg_pcd_t *pcd = (dwc_otg_pcd_t *) data;

	DWC_OTG_DRIVER_LOCK(pcd->core_if);
	dwc_otg_pcd_stop(pcd);
	DWC_OTG_DRIVER_UNLOCK(pcd->core_if);
}
#endif

/**
 * PCD Callback function for stopping the PCD when switching to Host
 * mode.
 *
 * @param p void pointer to the <code>dwc_otg_pcd_t</code>
 */
static int32_t dwc_otg_pcd_stop_cb(void *p)
{
	extern void dwc_otg_pcd_stop(dwc_otg_pcd_t * _pcd);
	dwc_otg_pcd_t *pcd = (dwc_otg_pcd_t *) p;

#ifdef DWC_PCD_STOP_TASKLET
	DWC_TASK_SCHEDULE(pcd->pcd_stop_tasklet);
#else
	dwc_otg_pcd_stop(pcd);
#endif
	return 1;
}

/**
 * This function allocates a DMA Descriptor chain for the Endpoint
 * buffer to be used for a transfer to/from the specified endpoint.
 */
dwc_otg_dev_dma_desc_t *dwc_otg_ep_alloc_desc_chain(
	dwc_otg_pcd_t *pcd,
	dwc_dma_t * dma_desc_addr,
	uint32_t count
)
{
	dwc_otg_dev_dma_desc_t *dma_desc;

	DWC_OTG_PCD_DUMP_DMA_EP0_OUT_REGS(pcd);

	dma_desc =  __DWC_DMA_ALLOC_ATOMIC(dwc_otg_get_device(pcd->otg_dev),
		count * sizeof(dwc_otg_dev_dma_desc_t),
		dma_desc_addr
	);

	DWC_DEBUGPL(DBG_PCD, "%s dma_desc_addr:%p count:%d return %p\n", __func__,
			dma_desc_addr, count, dma_desc
		   );

	DWC_OTG_PCD_DUMP_DMA_EP0_OUT_REGS(pcd);

	return dma_desc;
}

/**
 * This function frees a DMA Descriptor chain that was allocated by ep_alloc_desc.
 */
void dwc_otg_ep_free_desc_chain(
        dwc_otg_pcd_t *pcd,
	dwc_otg_dev_dma_desc_t * desc_addr,
	uint32_t dma_desc_addr,
	uint32_t count
)
{
        DWC_DEBUGPL(DBG_PCD, "%s desc_addr:%p dma_desc_addr:0x%08x count:%d\n", __func__, desc_addr, dma_desc_addr, count);
	__DWC_DMA_FREE(dwc_otg_get_device(pcd->otg_dev),
		     count * sizeof(dwc_otg_dev_dma_desc_t), desc_addr,
		     dma_desc_addr);
}

#ifdef DWC_EN_ISOC

/**
 * This function initializes a descriptor chain for Isochronous transfer
 *
 * @param core_if Programming view of DWC_otg controller.
 * @param dwc_ep The EP to start the transfer on.
 *
 */
void dwc_otg_iso_ep_start_ddma_transfer(
	dwc_otg_pcd_t * pcd,
	dwc_otg_core_if_t * core_if,
	dwc_ep_t * dwc_ep)
{

	dsts_data_t dsts = {.d32 = 0 };
	depctl_data_t depctl = {.d32 = 0 };
	volatile uint32_t *addr;
	int i, j;
	uint32_t len;

	if (dwc_ep->is_in)
		dwc_ep->desc_cnt = dwc_ep->buf_proc_intrvl / dwc_ep->bInterval;
	else
		dwc_ep->desc_cnt =
		    dwc_ep->buf_proc_intrvl * dwc_ep->pkt_per_frm /
		    dwc_ep->bInterval;

	/** Allocate descriptors for double buffering */
	dwc_ep->iso_desc_addr =
	    dwc_otg_ep_alloc_desc_chain(pcd,
			&dwc_ep->iso_dma_desc_addr,
			dwc_ep->desc_cnt * 2);
	if (dwc_ep->desc_addr) {
		DWC_WARN("%s, can't allocate DMA descriptor chain\n", __func__);
		return;
	}

	dsts.d32 = DWC_READ_REG32(&core_if->dev_if->dev_global_regs->dsts);

	/** ISO OUT EP */
	if (dwc_ep->is_in == 0) {
		dev_dma_desc_sts_t sts = {.d32 = 0 };
		dwc_otg_dev_dma_desc_t *dma_desc = dwc_ep->iso_desc_addr;
		dma_addr_t dma_ad;
		uint32_t data_per_desc;
		dwc_otg_dev_out_ep_regs_t *out_regs =
		    core_if->dev_if->out_ep_regs[dwc_ep->num];
		int offset;

		addr = &core_if->dev_if->out_ep_regs[dwc_ep->num]->doepctl;
		dma_ad = (dma_addr_t) DWC_READ_REG32(&(out_regs->doepdma));

		/** Buffer 0 descriptors setup */
		dma_ad = dwc_ep->dma_addr0;

		sts.b_iso_out.bs = BS_HOST_BUSY;
		sts.b_iso_out.rxsts = 0;
		sts.b_iso_out.l = 0;
		sts.b_iso_out.sp = 0;
		sts.b_iso_out.ioc = 0;
		sts.b_iso_out.pid = 0;
		sts.b_iso_out.framenum = 0;
		sts.b_iso_out.bs = BS_HOST_READY;

		offset = 0;
		for (i = 0; i < dwc_ep->desc_cnt - dwc_ep->pkt_per_frm;
		     i += dwc_ep->pkt_per_frm) {

			for (j = 0; j < dwc_ep->pkt_per_frm; ++j) {
				uint32_t len = (j + 1) * dwc_ep->maxpacket;
				if (len > dwc_ep->data_per_frame)
					data_per_desc =
					    dwc_ep->data_per_frame -
					    j * dwc_ep->maxpacket;
				else
					data_per_desc = dwc_ep->maxpacket;
				len = data_per_desc % 4;
				if (len)
					data_per_desc += 4 - len;

				sts.b_iso_out.rxbytes = data_per_desc;
				dma_desc->buf = dma_ad;
				dma_desc->status.d32 = sts.d32;

				offset += data_per_desc;
				dma_desc++;
				dma_ad += data_per_desc;
			}
		}

		for (j = 0; j < dwc_ep->pkt_per_frm - 1; ++j) {
			uint32_t len = (j + 1) * dwc_ep->maxpacket;
			if (len > dwc_ep->data_per_frame)
				data_per_desc =
				    dwc_ep->data_per_frame -
				    j * dwc_ep->maxpacket;
			else
				data_per_desc = dwc_ep->maxpacket;
			len = data_per_desc % 4;
			if (len)
				data_per_desc += 4 - len;
			sts.b_iso_out.rxbytes = data_per_desc;
			dma_desc->buf = dma_ad;
			dma_desc->status.d32 = sts.d32;

			offset += data_per_desc;
			dma_desc++;
			dma_ad += data_per_desc;
		}

		sts.b_iso_out.ioc = 1;
		len = (j + 1) * dwc_ep->maxpacket;
		if (len > dwc_ep->data_per_frame)
			data_per_desc =
			    dwc_ep->data_per_frame - j * dwc_ep->maxpacket;
		else
			data_per_desc = dwc_ep->maxpacket;
		len = data_per_desc % 4;
		if (len)
			data_per_desc += 4 - len;
		sts.b_iso_out.rxbytes = data_per_desc;

		dma_desc->buf = dma_ad;
		dma_desc->status.d32 = sts.d32;
		dma_desc++;

		/** Buffer 1 descriptors setup */
		sts.b_iso_out.ioc = 0;
		dma_ad = dwc_ep->dma_addr1;

		offset = 0;
		for (i = 0; i < dwc_ep->desc_cnt - dwc_ep->pkt_per_frm;
		     i += dwc_ep->pkt_per_frm) {
			for (j = 0; j < dwc_ep->pkt_per_frm; ++j) {
				uint32_t len = (j + 1) * dwc_ep->maxpacket;
				if (len > dwc_ep->data_per_frame)
					data_per_desc =
					    dwc_ep->data_per_frame -
					    j * dwc_ep->maxpacket;
				else
					data_per_desc = dwc_ep->maxpacket;
				len = data_per_desc % 4;
				if (len)
					data_per_desc += 4 - len;

				data_per_desc =
				    sts.b_iso_out.rxbytes = data_per_desc;
				dma_desc->buf = dma_ad;
				dma_desc->status.d32 = sts.d32;

				offset += data_per_desc;
				dma_desc++;
				dma_ad += data_per_desc;
			}
		}
		for (j = 0; j < dwc_ep->pkt_per_frm - 1; ++j) {
			data_per_desc =
			    ((j + 1) * dwc_ep->maxpacket >
			     dwc_ep->data_per_frame) ? dwc_ep->data_per_frame -
			    j * dwc_ep->maxpacket : dwc_ep->maxpacket;
			data_per_desc +=
			    (data_per_desc % 4) ? (4 - data_per_desc % 4) : 0;
			sts.b_iso_out.rxbytes = data_per_desc;
			dma_desc->buf = dma_ad;
			dma_desc->status.d32 = sts.d32;

			offset += data_per_desc;
			dma_desc++;
			dma_ad += data_per_desc;
		}

		sts.b_iso_out.ioc = 1;
		sts.b_iso_out.l = 1;
		data_per_desc =
		    ((j + 1) * dwc_ep->maxpacket >
		     dwc_ep->data_per_frame) ? dwc_ep->data_per_frame -
		    j * dwc_ep->maxpacket : dwc_ep->maxpacket;
		data_per_desc +=
		    (data_per_desc % 4) ? (4 - data_per_desc % 4) : 0;
		sts.b_iso_out.rxbytes = data_per_desc;

		dma_desc->buf = dma_ad;
		dma_desc->status.d32 = sts.d32;

		dwc_ep->next_frame = 0;

		/** Write dma_ad into DOEPDMA register */
		DWC_WRITE_REG32(&(out_regs->doepdma),
				(uint32_t) dwc_ep->iso_dma_desc_addr);

	}
	/** ISO IN EP */
	else {
		dev_dma_desc_sts_t sts = {.d32 = 0 };
		dwc_otg_dev_dma_desc_t *dma_desc = dwc_ep->iso_desc_addr;
		dma_addr_t dma_ad;
		dwc_otg_dev_in_ep_regs_t *in_regs =
		    core_if->dev_if->in_ep_regs[dwc_ep->num];
		unsigned int frmnumber;
		fifosize_data_t txfifosize, rxfifosize;

		txfifosize.d32 =
		    DWC_READ_REG32(&core_if->dev_if->
				   in_ep_regs[dwc_ep->num]->dtxfsts);
		rxfifosize.d32 =
		    DWC_READ_REG32(&core_if->core_global_regs->grxfsiz);

		addr = &core_if->dev_if->in_ep_regs[dwc_ep->num]->diepctl;

		dma_ad = dwc_ep->dma_addr0;

		dsts.d32 =
		    DWC_READ_REG32(&core_if->dev_if->dev_global_regs->dsts);

		sts.b_iso_in.bs = BS_HOST_BUSY;
		sts.b_iso_in.txsts = 0;
		sts.b_iso_in.sp =
		    (dwc_ep->data_per_frame % dwc_ep->maxpacket) ? 1 : 0;
		sts.b_iso_in.ioc = 0;
		sts.b_iso_in.pid = dwc_ep->pkt_per_frm;
		sts.b_iso_in.bs = BS_HOST_READY;

		frmnumber = dwc_ep->next_frame;

		sts.b_iso_in.framenum = frmnumber;
		sts.b_iso_in.txbytes = dwc_ep->data_per_frame;
		sts.b_iso_in.l = 0;

		/** Buffer 0 descriptors setup */
		for (i = 0; i < dwc_ep->desc_cnt - 1; i++) {
			dma_desc->buf = dma_ad;
			dma_desc->status.d32 = sts.d32;
			dma_desc++;

			dma_ad += dwc_ep->data_per_frame;
			sts.b_iso_in.framenum += dwc_ep->bInterval;
		}

		sts.b_iso_in.ioc = 1;
		dma_desc->buf = dma_ad;
		dma_desc->status.d32 = sts.d32;
		++dma_desc;

		/** Buffer 1 descriptors setup */
		sts.b_iso_in.ioc = 0;
		dma_ad = dwc_ep->dma_addr1;

		for (i = 0; i < dwc_ep->desc_cnt - dwc_ep->pkt_per_frm;
		     i += dwc_ep->pkt_per_frm) {
			dma_desc->buf = dma_ad;
			dma_desc->status.d32 = sts.d32;
			dma_desc++;

			dma_ad += dwc_ep->data_per_frame;
			sts.b_iso_in.framenum += dwc_ep->bInterval;

			sts.b_iso_in.ioc = 0;
		}
		sts.b_iso_in.ioc = 1;
		sts.b_iso_in.l = 1;

		dma_desc->buf = dma_ad;
		dma_desc->status.d32 = sts.d32;

		dwc_ep->next_frame = sts.b_iso_in.framenum + dwc_ep->bInterval;

		/** Write dma_ad into diepdma register */
		DWC_WRITE_REG32(&(in_regs->diepdma),
				(uint32_t) dwc_ep->iso_dma_desc_addr);
	}
	/** Enable endpoint, clear nak  */
	depctl.d32 = 0;
	depctl.b.epena = 1;
	depctl.b.usbactep = 1;
	depctl.b.cnak = 1;

	DWC_MODIFY_REG32(addr, depctl.d32, depctl.d32);
	depctl.d32 = DWC_READ_REG32(addr);
}

/**
 * This function initializes a descriptor chain for Isochronous transfer
 *
 * @param core_if Programming view of DWC_otg controller.
 * @param ep The EP to start the transfer on.
 *
 */
void dwc_otg_iso_ep_start_buf_transfer(dwc_otg_core_if_t * core_if,
				       dwc_ep_t * ep)
{
	depctl_data_t depctl = {.d32 = 0 };
	volatile uint32_t *addr;

	if (ep->is_in) {
		addr = &core_if->dev_if->in_ep_regs[ep->num]->diepctl;
	} else {
		addr = &core_if->dev_if->out_ep_regs[ep->num]->doepctl;
	}

	if (core_if->dma_enable == 0 || core_if->dma_desc_enable != 0) {
		return;
	} else {
		deptsiz_data_t deptsiz = {.d32 = 0 };

		ep->xfer_len =
		    ep->data_per_frame * ep->buf_proc_intrvl / ep->bInterval;
		ep->pkt_cnt =
		    (ep->xfer_len - 1 + ep->maxpacket) / ep->maxpacket;
		ep->xfer_count = 0;
		ep->xfer_buff =
		    (ep->proc_buf_num) ? ep->xfer_buff1 : ep->xfer_buff0;
		ep->dma_addr =
		    (ep->proc_buf_num) ? ep->dma_addr1 : ep->dma_addr0;

		if (ep->is_in) {
			/* Program the transfer size and packet count
			 *      as follows: xfersize = N * maxpacket +
			 *      short_packet pktcnt = N + (short_packet
			 *      exist ? 1 : 0) 
			 */
			deptsiz.b.mc = ep->pkt_per_frm;
			deptsiz.b.xfersize = ep->xfer_len;
			deptsiz.b.pktcnt =
			    (ep->xfer_len - 1 + ep->maxpacket) / ep->maxpacket;
			DWC_WRITE_REG32(&core_if->dev_if->
					in_ep_regs[ep->num]->dieptsiz,
					deptsiz.d32);

			/* Write the DMA register */
			DWC_WRITE_REG32(&
					(core_if->dev_if->
					 in_ep_regs[ep->num]->diepdma),
					(uint32_t) ep->dma_addr);

		} else {
			deptsiz.b.pktcnt =
			    (ep->xfer_len + (ep->maxpacket - 1)) /
			    ep->maxpacket;
			deptsiz.b.xfersize = deptsiz.b.pktcnt * ep->maxpacket;

			DWC_WRITE_REG32(&core_if->dev_if->
					out_ep_regs[ep->num]->doeptsiz,
					deptsiz.d32);

			/* Write the DMA register */
			DWC_WRITE_REG32(&
					(core_if->dev_if->
					 out_ep_regs[ep->num]->doepdma),
					(uint32_t) ep->dma_addr);

		}
		/** Enable endpoint, clear nak  */
		depctl.d32 = 0;
		depctl.b.epena = 1;
		depctl.b.cnak = 1;

		DWC_MODIFY_REG32(addr, depctl.d32, depctl.d32);
	}
}

/**
 * This function does the setup for a data transfer for an EP and
 * starts the transfer. For an IN transfer, the packets will be
 * loaded into the appropriate Tx FIFO in the ISR. For OUT transfers,
 * the packets are unloaded from the Rx FIFO in the ISR.
 *
 * @param core_if Programming view of DWC_otg controller.
 * @param ep The EP to start the transfer on.
 */

static void dwc_otg_iso_ep_start_transfer(
	dwc_otg_pcd_t * pcd,
	dwc_otg_core_if_t * core_if,
	dwc_ep_t * ep)
{
	if (core_if->dma_enable) {
		if (core_if->dma_desc_enable) {
			if (ep->is_in) {
				ep->desc_cnt = ep->pkt_cnt / ep->pkt_per_frm;
			} else {
				ep->desc_cnt = ep->pkt_cnt;
			}
			dwc_otg_iso_ep_start_ddma_transfer(pcd, core_if, ep);
		} else {
			if (core_if->pti_enh_enable) {
				dwc_otg_iso_ep_start_buf_transfer(core_if, ep);
			} else {
				ep->cur_pkt_addr =
				    (ep->proc_buf_num) ? ep->
				    xfer_buff1 : ep->xfer_buff0;
				ep->cur_pkt_dma_addr =
				    (ep->proc_buf_num) ? ep->
				    dma_addr1 : ep->dma_addr0;
				dwc_otg_iso_ep_start_frm_transfer(core_if, ep);
			}
		}
	} else {
		ep->cur_pkt_addr =
		    (ep->proc_buf_num) ? ep->xfer_buff1 : ep->xfer_buff0;
		ep->cur_pkt_dma_addr =
		    (ep->proc_buf_num) ? ep->dma_addr1 : ep->dma_addr0;
		dwc_otg_iso_ep_start_frm_transfer(core_if, ep);
	}
}

/**
 * This function stops transfer for an EP and
 * resets the ep's variables. 
 *
 * @param core_if Programming view of DWC_otg controller.
 * @param ep The EP to start the transfer on.
 */

void dwc_otg_iso_ep_stop_transfer(
	dwc_otg_pcd_t *pcd,
	dwc_otg_core_if_t * core_if,
	dwc_ep_t * ep
)
{
	depctl_data_t depctl = {.d32 = 0 };
	volatile uint32_t *addr;

	if (ep->is_in == 1) {
		addr = &core_if->dev_if->in_ep_regs[ep->num]->diepctl;
	} else {
		addr = &core_if->dev_if->out_ep_regs[ep->num]->doepctl;
	}

	/* disable the ep */
	depctl.d32 = DWC_READ_REG32(addr);

	depctl.b.epdis = 1;
	depctl.b.snak = 1;

	DWC_WRITE_REG32(addr, depctl.d32);

	if (core_if->dma_desc_enable &&
	    ep->iso_desc_addr && ep->iso_dma_desc_addr) {
		dwc_otg_ep_free_desc_chain(pcd,
					   ep->iso_desc_addr,
					   ep->iso_dma_desc_addr,
					   ep->desc_cnt * 2);
	}

	/* reset varibales */
	ep->dma_addr0 = 0;
	ep->dma_addr1 = 0;
	ep->xfer_buff0 = 0;
	ep->xfer_buff1 = 0;
	ep->data_per_frame = 0;
	ep->data_pattern_frame = 0;
	ep->sync_frame = 0;
	ep->buf_proc_intrvl = 0;
	ep->bInterval = 0;
	ep->proc_buf_num = 0;
	ep->pkt_per_frm = 0;
	ep->pkt_per_frm = 0;
	ep->desc_cnt = 0;
	ep->iso_desc_addr = 0;
	ep->iso_dma_desc_addr = 0;
}

int dwc_otg_pcd_iso_ep_start(dwc_otg_pcd_t * pcd, void *ep_handle,
			     uint8_t * buf0, uint8_t * buf1, dwc_dma_t dma0,
			     dwc_dma_t dma1, int sync_frame, int dp_frame,
			     int data_per_frame, int start_frame,
			     int buf_proc_intrvl, void *req_handle,
			     int atomic_alloc)
{
	dwc_otg_pcd_ep_t *ep;
	dwc_ep_t *dwc_ep;
	int32_t frm_data;
	dsts_data_t dsts;
	dwc_otg_core_if_t *core_if;

	ep = get_ep_from_handle(pcd, ep_handle);

	if (!ep || !ep->desc || ep->dwc_ep.num == 0) {
		DWC_WARN("bad ep\n");
		return -DWC_E_INVALID;
	}

	core_if = GET_CORE_IF(pcd);
	dwc_ep = &ep->dwc_ep;

	if (ep->iso_req_handle) {
		DWC_WARN("ISO request in progress\n");
	}

	dwc_ep->dma_addr0 = dma0;
	dwc_ep->dma_addr1 = dma1;

	dwc_ep->xfer_buff0 = buf0;
	dwc_ep->xfer_buff1 = buf1;

	dwc_ep->data_per_frame = data_per_frame;

	/** @todo - pattern data support is to be implemented in the future */
	dwc_ep->data_pattern_frame = dp_frame;
	dwc_ep->sync_frame = sync_frame;

	dwc_ep->buf_proc_intrvl = buf_proc_intrvl;

	dwc_ep->bInterval = 1 << (ep->desc->bInterval - 1);

	dwc_ep->proc_buf_num = 0;

	dwc_ep->pkt_per_frm = 0;
	frm_data = ep->dwc_ep.data_per_frame;
	while (frm_data > 0) {
		dwc_ep->pkt_per_frm++;
		frm_data -= ep->dwc_ep.maxpacket;
	}

	dsts.d32 = DWC_READ_REG32(&core_if->dev_if->dev_global_regs->dsts);

	if (start_frame == -1) {
		dwc_ep->next_frame = dsts.b.soffn + 1;
		if (dwc_ep->bInterval != 1) {
			dwc_ep->next_frame =
			    dwc_ep->next_frame + (dwc_ep->bInterval - 1 -
						  dwc_ep->next_frame %
						  dwc_ep->bInterval);
		}
	} else {
		dwc_ep->next_frame = start_frame;
	}

	if (!core_if->pti_enh_enable) {
		dwc_ep->pkt_cnt =
		    dwc_ep->buf_proc_intrvl * dwc_ep->pkt_per_frm /
		    dwc_ep->bInterval;
	} else {
		dwc_ep->pkt_cnt =
		    (dwc_ep->data_per_frame *
		     (dwc_ep->buf_proc_intrvl / dwc_ep->bInterval)
		     - 1 + dwc_ep->maxpacket) / dwc_ep->maxpacket;
	}

	if (core_if->dma_desc_enable) {
		dwc_ep->desc_cnt =
		    dwc_ep->buf_proc_intrvl * dwc_ep->pkt_per_frm /
		    dwc_ep->bInterval;
	}

	if (atomic_alloc) {
		dwc_ep->pkt_info =
		    DWC_ALLOC_ATOMIC(sizeof(iso_pkt_info_t) * dwc_ep->pkt_cnt);
	} else {
		dwc_ep->pkt_info =
		    DWC_ALLOC(sizeof(iso_pkt_info_t) * dwc_ep->pkt_cnt);
	}
	if (!dwc_ep->pkt_info) {
		return -DWC_E_NO_MEMORY;
	}
	if (core_if->pti_enh_enable) {
		dwc_memset(dwc_ep->pkt_info, 0,
			   sizeof(iso_pkt_info_t) * dwc_ep->pkt_cnt);
	}

	dwc_ep->cur_pkt = 0;
	ep->iso_req_handle = req_handle;

	dwc_otg_iso_ep_start_transfer(pcd, core_if, dwc_ep);
	return 0;
}

int dwc_otg_pcd_iso_ep_stop(dwc_otg_pcd_t * pcd, void *ep_handle,
			    void *req_handle)
{
	dwc_otg_pcd_ep_t *ep;
	dwc_ep_t *dwc_ep;

	ep = get_ep_from_handle(pcd, ep_handle);
	if (!ep || !ep->desc || ep->dwc_ep.num == 0) {
		DWC_WARN("bad ep\n");
		return -DWC_E_INVALID;
	}
	dwc_ep = &ep->dwc_ep;

	dwc_otg_iso_ep_stop_transfer(pcd, GET_CORE_IF(pcd), dwc_ep);

	DWC_FREE(dwc_ep->pkt_info);

	if (ep->iso_req_handle != req_handle) {
		return -DWC_E_INVALID;
	}
	ep->iso_req_handle = 0;

	return 0;
}

/**
 * This function is used for perodical data exchnage between PCD and gadget drivers.
 * for Isochronous EPs
 *
 *	- Every time a sync period completes this function is called to
 *	  perform data exchange between PCD and gadget
 */
void dwc_otg_iso_buffer_done(dwc_otg_pcd_t * pcd, dwc_otg_pcd_ep_t * ep,
			     void *req_handle)
{
	int i;
	dwc_ep_t *dwc_ep;
	dwc_otg_pcd_t *pcd;

	dwc_ep = &ep->dwc_ep;
	pcd = ep->pcd;

	pcd->fops->isoc_complete(pcd, ep->priv, ep->iso_req_handle,
				 dwc_ep->proc_buf_num ^ 0x1);

	for (i = 0; i < dwc_ep->pkt_cnt; ++i) {
		dwc_ep->pkt_info[i].status = 0;
		dwc_ep->pkt_info[i].offset = 0;
		dwc_ep->pkt_info[i].length = 0;
	}
}

int dwc_otg_pcd_get_iso_packet_count(dwc_otg_pcd_t * pcd, void *ep_handle,
				     void *iso_req_handle)
{
	dwc_otg_pcd_ep_t *ep;
	dwc_ep_t *dwc_ep;

	ep = get_ep_from_handle(pcd, ep_handle);
	if (!ep->desc || ep->dwc_ep.num == 0) {
		DWC_WARN("bad ep\n");
		return -DWC_E_INVALID;
	}
	dwc_ep = &ep->dwc_ep;

	return dwc_ep->pkt_cnt;
}

void dwc_otg_pcd_get_iso_packet_params(dwc_otg_pcd_t * pcd, void *ep_handle,
				       void *iso_req_handle, int packet,
				       int *status, int *actual, int *offset)
{
	dwc_otg_pcd_ep_t *ep;
	dwc_ep_t *dwc_ep;

	ep = get_ep_from_handle(pcd, ep_handle);
	if (!ep)
		DWC_WARN("bad ep\n");

	dwc_ep = &ep->dwc_ep;

	*status = dwc_ep->pkt_info[packet].status;
	*actual = dwc_ep->pkt_info[packet].length;
	*offset = dwc_ep->pkt_info[packet].offset;
}

#endif /* DWC_EN_ISOC */

static void dwc_otg_pcd_init_ep(dwc_otg_pcd_t * pcd, dwc_otg_pcd_ep_t * pcd_ep,
				uint32_t is_in, uint32_t ep_num)
{
	/* Init EP structure */
	pcd_ep->desc = 0;
	pcd_ep->pcd = pcd;
	pcd_ep->stopped = 1;
	pcd_ep->queue_sof = 0;

	/* Init DWC ep structure */
	pcd_ep->dwc_ep.is_in = is_in;
	pcd_ep->dwc_ep.num = ep_num;
	pcd_ep->dwc_ep.active = 0;
	pcd_ep->dwc_ep.tx_fifo_num = 0;
	/* Control until ep is actvated */
	pcd_ep->dwc_ep.type = DWC_OTG_EP_TYPE_CONTROL;
	pcd_ep->dwc_ep.maxpacket = MAX_PACKET_SIZE;
	pcd_ep->dwc_ep.dma_addr = 0;
	pcd_ep->dwc_ep.start_xfer_buff = 0;
	pcd_ep->dwc_ep.xfer_buff = 0;
	pcd_ep->dwc_ep.xfer_len = 0;
	pcd_ep->dwc_ep.xfer_count = 0;
	pcd_ep->dwc_ep.sent_zlp = 0;
	pcd_ep->dwc_ep.total_len = 0;
	pcd_ep->dwc_ep.desc_addr = 0;
	pcd_ep->dwc_ep.dma_desc_addr = 0;
	DWC_CIRCLEQ_INIT(&pcd_ep->queue);
}

/**
 * Initialize ep's
 */
static void dwc_otg_pcd_reinit(dwc_otg_pcd_t * pcd)
{
	int i;
	uint32_t hwcfg1;
	dwc_otg_pcd_ep_t *ep;
	int in_ep_cntr, out_ep_cntr;
	uint32_t num_in_eps = (GET_CORE_IF(pcd))->dev_if->num_in_eps;
	uint32_t num_out_eps = (GET_CORE_IF(pcd))->dev_if->num_out_eps;

	/**
	 * Initialize the EP0 structure.
	 */
	ep = &pcd->ep0;
	dwc_otg_pcd_init_ep(pcd, ep, 0, 0);

	in_ep_cntr = 0;
	hwcfg1 = (GET_CORE_IF(pcd))->hwcfg1.d32 >> 3;
	for (i = 1; in_ep_cntr < num_in_eps; i++) {
		if ((hwcfg1 & 0x1) == 0) {
			dwc_otg_pcd_ep_t *ep = &pcd->in_ep[in_ep_cntr];
			in_ep_cntr++;
			/**
			 * @todo NGS: Add direction to EP, based on contents
			 * of HWCFG1.  Need a copy of HWCFG1 in pcd structure?
			 * sprintf(";r
			 */
			dwc_otg_pcd_init_ep(pcd, ep, 1 /* IN */ , i);

			DWC_CIRCLEQ_INIT(&ep->queue);
		}
		hwcfg1 >>= 2;
	}

	out_ep_cntr = 0;
	hwcfg1 = (GET_CORE_IF(pcd))->hwcfg1.d32 >> 2;
	for (i = 1; out_ep_cntr < num_out_eps; i++) {
		if ((hwcfg1 & 0x1) == 0) {
			dwc_otg_pcd_ep_t *ep = &pcd->out_ep[out_ep_cntr];
			out_ep_cntr++;
			/**
			 * @todo NGS: Add direction to EP, based on contents
			 * of HWCFG1.  Need a copy of HWCFG1 in pcd structure?
			 * sprintf(";r
			 */
			dwc_otg_pcd_init_ep(pcd, ep, 0 /* OUT */ , i);
			DWC_CIRCLEQ_INIT(&ep->queue);
		}
		hwcfg1 >>= 2;
	}

	pcd->ep0state = EP0_DISCONNECT;
	pcd->ep0.dwc_ep.maxpacket = MAX_EP0_SIZE;
	pcd->ep0.dwc_ep.type = DWC_OTG_EP_TYPE_CONTROL;
}

/**
 * This function is called when the SRP timer expires. The SRP should
 * complete within 6 seconds.
 */
static void srp_timeout(void *ptr)
{
	gotgctl_data_t gotgctl;
	dwc_otg_core_if_t *core_if = (dwc_otg_core_if_t *) ptr;
	volatile uint32_t *addr = &core_if->core_global_regs->gotgctl;

	gotgctl.d32 = DWC_READ_REG32(addr);

	core_if->srp_timer_started = 0;

	if ((core_if->core_params->phy_type == DWC_PHY_TYPE_PARAM_FS) &&
	    (core_if->core_params->i2c_enable)) {
		DWC_PRINTF("SRP Timeout\n");

		if ((core_if->srp_success) && (gotgctl.b.bsesvld)) {
			if (core_if->pcd_cb && core_if->pcd_cb->resume_wakeup) {
				core_if->pcd_cb->resume_wakeup(core_if->pcd_cb->p);
			}

			/* Clear Session Request */
			gotgctl.d32 = 0;
			gotgctl.b.sesreq = 1;
			DWC_MODIFY_REG32(&core_if->core_global_regs->gotgctl,
					 gotgctl.d32, 0);

			core_if->srp_success = 0;
		} else {
			__DWC_ERROR("Device not connected/responding\n");
			gotgctl.b.sesreq = 0;
			DWC_WRITE_REG32(addr, gotgctl.d32);
		}
	} else if (gotgctl.b.sesreq) {
		DWC_PRINTF("SRP Timeout\n");

		__DWC_ERROR("Device not connected/responding\n");
		gotgctl.b.sesreq = 0;
		DWC_WRITE_REG32(addr, gotgctl.d32);
	} else {
		DWC_PRINTF(" SRP GOTGCTL=%0x\n", gotgctl.d32);
	}

	if (core_if->adp_enable) {
		if (gotgctl.b.bsesvld == 0) {
			gpwrdn_data_t gpwrdn = {.d32 = 0 };

			/* Power off the core */
			if (core_if->power_down == 2) {
				gpwrdn.b.pwrdnswtch = 1;
				DWC_MODIFY_REG32(&core_if->core_global_regs->
						 gpwrdn, gpwrdn.d32, 0);
			}

			gpwrdn.d32 = 0;
			gpwrdn.b.pmuactv = 1;
			DWC_MODIFY_REG32(&core_if->core_global_regs->gpwrdn, 0, gpwrdn.d32);
						
			dwc_otg_adp_probe_start(core_if);
		} else {
			core_if->op_state = B_PERIPHERAL;
			dwc_otg_core_init(core_if);
			dwc_otg_enable_global_interrupts(core_if);
			cil_pcd_start(core_if);
		}
	}
}

/* stephen.smith@intelligraphics.com */
int check_ep_for_in_token(dwc_otg_core_if_t * core_if, int epnum)
{
	dwc_otg_device_global_regs_t *dev_global_regs =
		core_if->dev_if->dev_global_regs;
	const uint32_t TOKEN_Q_DEPTH = core_if->hwcfg2.b.dev_token_q_depth;

	/*
	 * Number of Token Queue Registers
	 * the first register only contains 6 tokens
	 *  so we add 2 to the depth to correctly calculate the offset
	 */
	const int DTKNQ_REG_CNT = (TOKEN_Q_DEPTH + 2 + 7) / 8;

	dtknq1_data_t dtknqr1;
	uint32_t in_tkn_epnums[4];
	int cnt;
	int i;
	int token;
	int ret;
	volatile uint32_t *addr = &dev_global_regs->dtknqr1;

	ret = 0;

	/* Read the DTKNQ Registers */
	for (i = 0; i < DTKNQ_REG_CNT; i++) {
		in_tkn_epnums[i] = DWC_READ_REG32(addr);
		DWC_DEBUGPL(DBG_PCDV, "DTKNQR%d=0x%08x\n", i + 1, in_tkn_epnums[i]);
		if (addr == &dev_global_regs->dvbusdis) {
			addr = &dev_global_regs->dtknqr3_dthrctl;
		} else {
			++addr;
		}
	}

	dtknqr1.d32 = in_tkn_epnums[0];
	if (!dtknqr1.b.wrap_bit)
		cnt = dtknqr1.b.intknwptr;
	else
		cnt = TOKEN_Q_DEPTH;

	DWC_DEBUGPL(DBG_PCDV,"dtknqr1:0x%08x wrap_bit:%d intknwptr:%d TOKEN_Q_DEPTH:%d cnt:%d\n",
			dtknqr1.d32, dtknqr1.b.wrap_bit, dtknqr1.b.intknwptr, TOKEN_Q_DEPTH, cnt);

	for (i = 2; i < (cnt + 2); i++) {
		token = (in_tkn_epnums[i >> 3] >> ((i & 0x7)) * 4) & 0x0f;
		DWC_DEBUGPL(DBG_PCDV,"i:%d index:%d shift:%d token:%d epnum:%d\n", i, i >> 3, (i % 0x7) * 4, token, epnum);
		if (epnum == token) {
			ret = 1;
			break;
		}
	}

	return ret;
}

void dwc_otg_pcd_flush_np_tx_fifo(dwc_otg_core_if_t *core_if)
{
        dwc_otg_core_global_regs_t *global_regs = core_if->core_global_regs;
        volatile grstctl_t grstctl;
#if 0
	volatile gintmsk_data_t gintsts;
	volatile dctl_data_t dctl;
	volatile depctl_data_t depctl;
#endif
        int i;

#if 0
	/*
	** Check that GINTSTS.GINNakEff=0. If this bit is cleared then set DCTL.SGNPInNak=1.
	**  NAK Effective interrupt = H indicating that the core is not reading from the FIFO.
	*/
        gintsts.d32 = DWC_READ_REG32(&global_regs->gintsts);
	if (!gintsts.b.ginnakeff) {
		dctl.d32 = 0;
		dctl.b.sgnpinnak = 1;
		DWC_MODIFY_REG32(&core_if->dev_if->dev_global_regs->dctl, 0, dctl.d32);
	}

        /*
        ** Wait for GINTSTS.GINNakEff=1, which indicates the NAK setting has taken effect to all IN
        **  endpoints.
        */
        gintsts.d32 = DWC_READ_REG32(&global_regs->gintsts);
	i = 0;
        while (!gintsts.b.ginnakeff) {
                if (i++ > 1000) {
                        DWC_WARN("%s() timeout waiting on GINTSTS.GINNakEff (0x%08x)!", __func__, gintsts.d32);
                        break;
                }
                dwc_udelay(1);
                gintsts.d32 = DWC_READ_REG32(&global_regs->gintsts);
	}

        /* disable the endpoint */
        depctl.d32 = DWC_READ_REG32(&core_if->dev_if->in_ep_regs[epnum]->diepctl);
        depctl.b.epdis = 1;
        DWC_WRITE_REG32(&core_if->dev_if->in_ep_regs[epnum]->diepctl, depctl.d32);
        depctl.d32 = DWC_READ_REG32(&core_if->dev_if->in_ep_regs[epnum]->diepctl);
        i = 0;
        while (depctl.b.epdis) {
                if (i++ > 1000) {
                        DWC_WARN("%s() timeout waiting on DIEPCTL0.EPDis (0x%08x)!", __func__, depctl.d32);
                        break;
                }
                dwc_udelay(1);
                depctl.d32 = DWC_READ_REG32(&core_if->dev_if->in_ep_regs[epnum]->diepctl);
        }
#endif

	/*
	** Poll GRSTCTL.AHBIdle until it is 1.
	**  HBIdle = H indicates that the core is not writing anything to the FIFO.
	*/
        grstctl.d32 = DWC_READ_REG32(&global_regs->grstctl);
	i = 0;
        while (!grstctl.b.ahbidle) {
                if (i++ > 1000) {
                        DWC_WARN("%s() timeout waiting on GRSTCTL.AHBIdle (0x%08x)!", __func__, grstctl.d32);
                        break;
                }
                dwc_udelay(1);
                grstctl.d32 = DWC_READ_REG32(&global_regs->grstctl);
        }

	/*
	** Check that GRSTCTL.TxFFlsh =0. If it is 0, then write the TxFIFO number you want to flush to
	**  GRSTCTL.TxFNum.
	*/
	grstctl.d32 = DWC_READ_REG32(&global_regs->grstctl);
	i = 0;
	while (grstctl.b.txfflsh) {
		if (i++ > 1000) {
			DWC_WARN("%s() timeout waiting on initial clear of GRSTCTL.TxFFlsh (0x%08x)!", __func__, grstctl.d32);
			break;
		}
		dwc_udelay(1);
		grstctl.d32 = DWC_READ_REG32(&global_regs->grstctl);
	}

	/*  Set GRSTCTL.TxFFlsh=1and wait for it to clear. */
        grstctl.d32 = DWC_READ_REG32(&global_regs->grstctl);
        grstctl.b.txfnum = 0;
	grstctl.b.txfflsh = 1;
	grstctl.b.intknqflsh = 1;
	DWC_WRITE_REG32(&global_regs->grstctl, grstctl.d32);

        grstctl.d32 = DWC_READ_REG32(&global_regs->grstctl);
	i = 0;
        while (grstctl.b.txfflsh) {
                if (i++ > 1000) {
                        DWC_WARN("%s() timeout waiting on GRSTCTL.TxFFlsh to finish (0x%08x)!", __func__, grstctl.d32);
                        break;
                }
		udelay(1);
		grstctl.d32 = DWC_READ_REG32(&global_regs->grstctl);
	}

        /* flush the learning queue */
        grstctl.d32 = DWC_READ_REG32(&global_regs->grstctl);
        grstctl.b.intknqflsh = 1;
//        DWC_WRITE_REG32(&global_regs->grstctl, grstctl.d32);

#if 0
        /* enable the endpoint */
        depctl.d32 = DWC_READ_REG32(&core_if->dev_if->in_ep_regs[epnum]->diepctl);
        depctl.b.epena = 1;
        depctl.d32 = DWC_WRITE_REG32(&core_if->dev_if->in_ep_regs[epnum]->diepctl, depctl.d32);

	/* Set the DCTL.GCNPInNak bit.*/
        dctl.d32 = 0;
        dctl.b.cgnpinnak = 1;
        DWC_MODIFY_REG32(&core_if->dev_if->dev_global_regs->dctl, 0, dctl.d32);
#endif
}

extern void start_next_request(dwc_otg_pcd_ep_t * ep);
void clear_ep_mismatch(dwc_otg_pcd_t * pcd, uint32_t epnum)
{
	dwc_otg_core_if_t *core_if = GET_CORE_IF(pcd);
        dwc_otg_core_global_regs_t *global_regs = core_if->core_global_regs;
	dwc_otg_pcd_ep_t *ep;
        volatile gintmsk_data_t gintsts;
	volatile depctl_data_t depctl;
        dctl_data_t dctl;
#if 0
        grstctl_t resetctl;
#endif
	int i;
	int iTimeout;

        DWC_DEBUGPL(DBG_PCDV, "***** %s epnum:%d *****\n", __func__, epnum);

        /* Sets a global non-periodic IN NAK handshake */
        dctl.d32 = 0;
        dctl.b.sgnpinnak = 1;
        DWC_MODIFY_REG32(&core_if->dev_if->dev_global_regs->dctl, 0, dctl.d32);

        /*
        ** Wait for GINTSTS.GINNakEff=1, which indicates the NAK setting has taken effect to all IN
        **  endpoints.
        */
        gintsts.d32 = DWC_READ_REG32(&global_regs->gintsts);
        i = 0;
        while (!gintsts.b.ginnakeff) {
                if (i++ > 1000) {
                        DWC_WARN("%s() timeout waiting on GINTSTS.GINNakEff (0x%08x)!", __func__, gintsts.d32);
                        break;
                }
                dwc_udelay(1);
                gintsts.d32 = DWC_READ_REG32(&global_regs->gintsts);
        }

	/* Disables In endpoints */
	for (i = 0; i < core_if->dev_if->num_in_eps; i++) {
		if (i == 0)
			ep = &pcd->ep0;
		else
			ep = &pcd->in_ep[i - 1];

		if ((ep->dwc_ep.is_in) && (ep->dwc_ep.type != DWC_OTG_EP_TYPE_ISOC)) {
			depctl.d32 = DWC_READ_REG32(&core_if->dev_if->in_ep_regs[i]->diepctl);
			if (depctl.b.epena) {
				depctl.b.epdis = 1;
				DWC_WRITE_REG32(&core_if->dev_if->in_ep_regs[i]->diepctl, depctl.d32);
				depctl.d32 = DWC_READ_REG32(&core_if->dev_if->in_ep_regs[i]->diepctl);
				iTimeout = 0;
				while (depctl.b.epdis) {
					if (iTimeout++ > 100000) {
						DWC_WARN("%s() timeout waiting on DIEPCTL%d.EPDis (0x%08x)!", __func__, i, depctl.d32);
						break;
					}
					dwc_udelay(1);
					depctl.d32 = DWC_READ_REG32(&core_if->dev_if->in_ep_regs[i]->diepctl);
				}
			}
		}
	}

	/* Flushes the FIFO */
	dwc_otg_pcd_flush_np_tx_fifo(core_if);

        /* Clears the global non-periodic IN NAK handshake */
        dctl.d32 = 0;
        dctl.b.cgnpinnak = 1;
        DWC_MODIFY_REG32(&core_if->dev_if->dev_global_regs->dctl, 0, dctl.d32);

	/* requeue starting with the EP that caused the mismatch */
        if (epnum == 0)
                ep = &pcd->ep0;
        else
                ep = &pcd->in_ep[epnum - 1];
	if ((ep->dwc_ep.is_in) && (ep->dwc_ep.type != DWC_OTG_EP_TYPE_ISOC)) {
		if (DWC_CIRCLEQ_EMPTY(&ep->queue))
			DWC_WARN("%s DWC_CIRCLEQ_EMPTY() is empty!?!?\n", __func__);
#if 1
		if (epnum == 0)
			dwc_otg_ep0_start_transfer(GET_CORE_IF(pcd), &ep->dwc_ep);
		else
			dwc_otg_ep_start_transfer(core_if, &ep->dwc_ep);
#else
		ep->queue_sof = 1;
		DWC_TASK_SCHEDULE(ep->pcd->start_xfer_tasklet);
#endif
	}

	for (i = 0; i < core_if->dev_if->num_in_eps; i++) {
		if (i != epnum) {
			if (i == 0)
				ep = &pcd->ep0;
			else
				ep = &pcd->in_ep[i - 1];

			if ((ep->dwc_ep.is_in) && (ep->dwc_ep.type != DWC_OTG_EP_TYPE_ISOC)) {
				if (!DWC_CIRCLEQ_EMPTY(&ep->queue)) {
#if 1
					if (i == 0)
						dwc_otg_ep0_start_transfer(GET_CORE_IF(pcd), &ep->dwc_ep);
					else
						dwc_otg_ep_start_transfer(core_if, &ep->dwc_ep);
#else
					ep->queue_sof = 1;
					DWC_TASK_SCHEDULE(ep->pcd->start_xfer_tasklet);
#endif
//					if (check_ep_for_in_token(core_if, i))
//						dwc_otg_ep_start_transfer(core_if, &ep->dwc_ep);
//						start_next_request(&pcd->in_ep[i]);
//					else
//						dwc_otg_request_nuke(ep);
				}
			}
		}
	}

}

/**
 * Tasklet
 *
 */

/* stephen.smith@intelligraphics.com */
static void clear_ep_mismatch_tasklet_func(void *data)
{
	dwc_otg_pcd_t *pcd = (dwc_otg_pcd_t *) data;
        dwc_otg_core_if_t *core_if = GET_CORE_IF(pcd);
	gintmsk_data_t intr_mask;
        diepmsk_data_t diepmsk;

	DWC_DEBUGPL(DBG_PCDV, "%s ep:%d\n", __func__, pcd->ep_mismatch_ep_num);

	clear_ep_mismatch(pcd, pcd->ep_mismatch_ep_num);

	/* enable the EP Mismatch Interrupt now that we've handled it */
	intr_mask.d32 = 0;
	intr_mask.b.epmismatch = 1;
	DWC_MODIFY_REG32(&GET_CORE_IF(pcd)->core_global_regs->gintmsk, 0, intr_mask.d32);

        diepmsk.d32 = 0;
        diepmsk.b.intknepmis = 1;
        if (core_if->multiproc_int_enable) {
                DWC_MODIFY_REG32(&core_if->dev_if->dev_global_regs->diepeachintmsk[pcd->ep_mismatch_ep_num], 0, diepmsk.d32);
#ifdef DEBUG
                diepmsk.d32 = DWC_READ_REG32(&core_if->dev_if->dev_global_regs->diepeachintmsk[pcd->ep_mismatch_ep_num]);
#endif
        } else {
                DWC_MODIFY_REG32(&core_if->dev_if->dev_global_regs->diepmsk, 0, diepmsk.d32);
#ifdef DEBUG
                diepmsk.d32 = DWC_READ_REG32(&core_if->dev_if->dev_global_regs->diepmsk);
#endif
        }
#ifdef DEBUG
         DWC_DEBUGPL(DBG_ANY,"diepmsk[%d]:0x%08x", pcd->ep_mismatch_ep_num, diepmsk.d32);
#endif


}

extern void start_next_request(dwc_otg_pcd_ep_t * ep);

static void start_xfer_tasklet_func(void *data)
{
	dwc_otg_pcd_t *pcd = (dwc_otg_pcd_t *) data;
	dwc_otg_core_if_t *core_if = GET_CORE_IF(pcd);

	int i;
	depctl_data_t diepctl;

	DWC_DEBUGPL(DBG_PCDV, "Start xfer tasklet\n");

	diepctl.d32 = DWC_READ_REG32(&core_if->dev_if->in_ep_regs[0]->diepctl);

	if (pcd->ep0.queue_sof) {
		pcd->ep0.queue_sof = 0;
		start_next_request(&pcd->ep0);
		// break;
	}

	for (i = 0; i < core_if->dev_if->num_in_eps; i++) {
		depctl_data_t diepctl;
		diepctl.d32 =
		    DWC_READ_REG32(&core_if->dev_if->in_ep_regs[i]->diepctl);

		if (pcd->in_ep[i].queue_sof) {
			pcd->in_ep[i].queue_sof = 0;
			start_next_request(&pcd->in_ep[i]);
			// break;
		}
	}

	return;
}

#ifdef DWC_OTG_COMPLETE_TASKLET
static void complete_tasklet_func(void *data)
{
	dwc_otg_pcd_t *pcd = (dwc_otg_pcd_t *) data;
	dwc_list_link_t *tmp;
	struct pcd_complete_context *pcomplete_ctx;

	DWC_OTG_DRIVER_LOCK(pcd->core_if);

	DWC_DEBUGPL(DBG_PCDV, "%s\n", __func__);
	while (!DWC_LIST_EMPTY(&pcd->complete_list_entry)) {
		tmp = DWC_LIST_FIRST(&pcd->complete_list_entry);
		DWC_LIST_REMOVE(tmp);

		pcomplete_ctx = DWC_LIST_ENTRY(tmp, struct pcd_complete_context, lh);
		DWC_OTG_DRIVER_UNLOCK(pcd->core_if);
		pcomplete_ctx->req->complete(pcomplete_ctx->ep_handle, pcomplete_ctx->req);
		DWC_OTG_DRIVER_LOCK(pcd->core_if);
		DWC_FREE(pcomplete_ctx);

	}
	DWC_OTG_DRIVER_UNLOCK(pcd->core_if);
}
#endif // #ifdef DWC_OTG_COMPLETE_TASKLET

#ifdef DWC_DO_GADGET_SETUP_TASKLET
static void do_gadget_setup_tasklet_func(void *data)
{
	extern void call_gadget_setup(dwc_otg_pcd_t * pcd, usb_device_request_t * ctrl);
	dwc_otg_pcd_t *pcd = (dwc_otg_pcd_t *) data;
	dwc_list_link_t *tmp;
	struct pcd_do_gadget_setup_context *p_ctx;

	DWC_OTG_DRIVER_LOCK(pcd->core_if);

	DWC_DEBUGPL(DBG_PCDV, "%s\n", __func__);
	while (!DWC_LIST_EMPTY(&pcd->do_gadget_setup_list_entry)) {
		tmp = DWC_LIST_FIRST(&pcd->do_gadget_setup_list_entry);
		DWC_LIST_REMOVE(tmp);

		p_ctx = DWC_LIST_ENTRY(tmp, struct pcd_do_gadget_setup_context, lh);
		call_gadget_setup(pcd, &p_ctx->ctrl);
		DWC_FREE(p_ctx);
	}
	DWC_OTG_DRIVER_UNLOCK(pcd->core_if);
}
#endif

static void free_dma_desc_tasklet_func(void *data)
{
	dwc_otg_pcd_t *pcd = (dwc_otg_pcd_t *) data;
	dwc_list_link_t *tmp;
	struct pcd_free_dma_desc_context *pdma_desc_ctx;

	DWC_DEBUGPL(DBG_PCDV, "%s\n", __func__);

	while (!DWC_LIST_EMPTY(&pcd->free_dma_desc_list_entry)) {
		tmp = DWC_LIST_FIRST(&pcd->free_dma_desc_list_entry);
		DWC_LIST_REMOVE(tmp);

		pdma_desc_ctx = DWC_LIST_ENTRY(tmp, struct pcd_free_dma_desc_context, lh);
		DWC_DEBUGPL(DBG_PCD, "%s desc_addr:%p dma_desc_addr:0x%08x\n", __func__,
			pdma_desc_ctx->desc_addr, pdma_desc_ctx->dma_desc_addr);
		dwc_otg_ep_free_desc_chain(pcd, pdma_desc_ctx->desc_addr, pdma_desc_ctx->dma_desc_addr, MAX_DMA_DESC_CNT);
		DWC_FREE(pdma_desc_ctx);

	}
}

static void reset_timeout(void *ptr)
{
	dwc_otg_pcd_t *pcd = (dwc_otg_pcd_t *) ptr;
	dwc_otg_core_if_t *core_if = pcd->core_if;
	dwc_otg_core_global_regs_t *global_regs = core_if->core_global_regs;
	gintmsk_data_t reg, orig;
	gintsts_data_t sts;
	hprt0_data_t status;

	/*
	 * Disable mode mismatch interrupt because this is exactly what we're
	 * about to do.
	 */
	orig.d32 = reg.d32 = DWC_READ_REG32(&global_regs->gintmsk);
	reg.b.modemismatch = 0;
	DWC_WRITE_REG32(&global_regs->gintmsk, reg.d32);

	/* Look at D+ and D- */
	status.d32 = DWC_READ_REG32(core_if->host_if->hprt0);
	if (status.b.prtlnsts == 3) {
		pcd->charger_state = DWC_CHRG_DEDICATED;
	} else {
		pcd->charger_state = DWC_CHRG_ATTACH;
	}

	/* Re-enable mode mismtach int */
	sts.d32 = 0;
	sts.b.modemismatch = 1;
	DWC_WRITE_REG32(&global_regs->gintsts, sts.d32);
	DWC_WRITE_REG32(&global_regs->gintmsk, orig.d32);

	if (pcd->fops->charger_changed)
		pcd->fops->charger_changed(pcd);
}

/**
 * This function initialized the PCD portion of the driver.
 *
 */
dwc_otg_pcd_t *dwc_otg_pcd_init(
	dwc_otg_device_t * otg_dev,
	dwc_otg_core_if_t * core_if
)
{
	dwc_otg_pcd_t *pcd = NULL;
	dwc_otg_dev_if_t *dev_if;

	DWC_DEBUGPL(DBG_PCDV, "%s\n", __func__);

	/*
	 * Allocate PCD structure
	 */
	pcd = DWC_ALLOC(sizeof(dwc_otg_pcd_t));

	if (pcd == NULL) {
		return NULL;
	}

        pcd->otg_dev = otg_dev;

	pcd->core_if = core_if;

	dev_if = core_if->dev_if;

	if (core_if->hwcfg4.b.ded_fifo_en) {
		DWC_PRINTF("Dedicated Tx FIFOs mode\n");
	} else {
		DWC_PRINTF("Shared Tx FIFO mode\n");
	}

	/*
	 * Initialized the Core for Device mode.
	 */
	if (dwc_otg_is_device_mode(core_if)) {
		dwc_otg_core_dev_init(core_if);
		dwc_otg_pcd_soft_disconnect(pcd);
	}

	/*
	 * Register the PCD Callbacks.
	 */
	memset(&pcd->pcd_callbacks, 0, sizeof(pcd->pcd_callbacks));
	pcd->pcd_callbacks.start = dwc_otg_pcd_start_cb;
	pcd->pcd_callbacks.stop = dwc_otg_pcd_stop_cb;
	pcd->pcd_callbacks.suspend = dwc_otg_pcd_suspend_cb;
	pcd->pcd_callbacks.resume_wakeup = dwc_otg_pcd_resume_cb;
	dwc_otg_cil_register_pcd_callbacks(core_if, &pcd->pcd_callbacks, pcd);

	/*
	 * Initialize the DMA buffer for SETUP packets
	 */
	if (GET_CORE_IF(pcd)->dma_enable) {
		pcd->setup_pkt =
                    __DWC_DMA_ALLOC(dwc_otg_get_device(otg_dev),
				((sizeof(*pcd->setup_pkt) + sizeof(uint32_t)) * MAX_SUP_CNT) + sizeof(uint32_t),
				&pcd->setup_pkt_dma_handle);
		if (pcd->setup_pkt == NULL) {
			DWC_FREE(pcd);
			return NULL;
		}

		pcd->status_buf =
                    __DWC_DMA_ALLOC(dwc_otg_get_device(otg_dev),
				sizeof(uint16_t),
				&pcd->status_buf_dma_handle);
		if (pcd->status_buf == NULL) {
			__DWC_DMA_FREE(dwc_otg_get_device(pcd->otg_dev),
				     ((sizeof(*pcd->setup_pkt) + sizeof(uint32_t)) * MAX_SUP_CNT) + sizeof(uint32_t),
				     pcd->setup_pkt, pcd->setup_pkt_dma_handle);
			DWC_FREE(pcd);
			return NULL;
		}

		if (GET_CORE_IF(pcd)->dma_desc_enable) {
			dev_if->setup_desc_addr[0] =
			    dwc_otg_ep_alloc_desc_chain(pcd,
					&dev_if->dma_setup_desc_addr[0],
					1);
			dev_if->setup_desc_addr[1] =
			    dwc_otg_ep_alloc_desc_chain(pcd,
					&dev_if->dma_setup_desc_addr[1],
					1);
			dev_if->in_desc_addr =
			    dwc_otg_ep_alloc_desc_chain(pcd,
					&dev_if->dma_in_desc_addr,
					1);
			dev_if->out_desc_addr =
			    dwc_otg_ep_alloc_desc_chain(pcd,
					&dev_if->dma_out_desc_addr,
					1);

			if (dev_if->setup_desc_addr[0] == 0
			    || dev_if->setup_desc_addr[1] == 0
			    || dev_if->in_desc_addr == 0
			    || dev_if->out_desc_addr == 0) {

				if (dev_if->out_desc_addr)
					dwc_otg_ep_free_desc_chain(pcd,
								   dev_if->
								   out_desc_addr,
								   dev_if->
								   dma_out_desc_addr,
								   1);
				if (dev_if->in_desc_addr)
					dwc_otg_ep_free_desc_chain(pcd,
								   dev_if->
								   in_desc_addr,
								   dev_if->
								   dma_in_desc_addr,
								   1);
				if (dev_if->setup_desc_addr[1])
					dwc_otg_ep_free_desc_chain(pcd,
								   dev_if->
								   setup_desc_addr
								   [1],
								   dev_if->
								   dma_setup_desc_addr
								   [1], 1);
				if (dev_if->setup_desc_addr[0])
					dwc_otg_ep_free_desc_chain(pcd,
								   dev_if->
								   setup_desc_addr
								   [0],
								   dev_if->
								   dma_setup_desc_addr
								   [0], 1);

				__DWC_DMA_FREE(dwc_otg_get_device(pcd->otg_dev),
					     ((sizeof(*pcd->setup_pkt) + sizeof(uint32_t)) * MAX_SUP_CNT) + sizeof(uint32_t),
					     pcd->setup_pkt,
					     pcd->setup_pkt_dma_handle);
				__DWC_DMA_FREE(dwc_otg_get_device(pcd->otg_dev),
					     sizeof(*pcd->status_buf),
					     pcd->status_buf,
					     pcd->status_buf_dma_handle);

				DWC_FREE(pcd);

				return NULL;
			}
		}
	} else {
		pcd->setup_pkt = DWC_ALLOC(((sizeof(*pcd->setup_pkt) + sizeof(uint32_t)) * MAX_SUP_CNT) + sizeof(uint32_t));
		if (pcd->setup_pkt == NULL) {
			DWC_FREE(pcd);
			return NULL;
		}

		pcd->status_buf = DWC_ALLOC(sizeof(uint16_t));
		if (pcd->status_buf == NULL) {
			DWC_FREE(pcd->setup_pkt);
			DWC_FREE(pcd);
			return NULL;
		}
	}

	dwc_otg_pcd_reinit(pcd);

	/* Allocate the cfi object for the PCD */
#ifdef DWC_UTE_CFI
	pcd->cfi = DWC_ALLOC(sizeof(cfiobject_t));
	if (NULL == pcd->cfi)
		goto fail;
	if (init_cfi(pcd, pcd->cfi)) {
		CFI_INFO("%s: Failed to init the CFI object\n", __func__);
		goto fail;
	}
#endif

	/* Initialize tasklets */
	pcd->start_xfer_tasklet = DWC_TASK_ALLOC("xfer_tasklet",
			start_xfer_tasklet_func, pcd);
	pcd->test_mode_tasklet = DWC_TASK_ALLOC("test_mode_tasklet",
			do_test_mode, pcd);
	pcd->clear_ep_mismatch_tasklet = DWC_TASK_ALLOC("clear_ep_mismatch_tasklet",
			clear_ep_mismatch_tasklet_func, pcd);
#ifdef DWC_PCD_STOP_TASKLET
	pcd->pcd_stop_tasklet = DWC_TASK_ALLOC("pcd_stop_tasklet",
			dwc_otg_pcd_stop_tasklet_func, pcd);
#endif

#ifdef DWC_DO_GADGET_SETUP_TASKLET
        DWC_LIST_INIT(&pcd->do_gadget_setup_list_entry);
        pcd->do_gadget_setup_tasklet = DWC_TASK_ALLOC(
                "do_gadget_setup_tasklet",
                do_gadget_setup_tasklet_func,
                pcd
        );
#endif

#ifdef DEBUG_DWC_OTG_PCD_DMA_AHB_ERROR
	pcd->start_ep0_out_dma_tasklet = DWC_TASK_ALLOC("start_ep0_out_dma_tasklet",
			start_ep0_out_dma_tasklet_func, pcd);
#endif

	DWC_LIST_INIT(&pcd->free_dma_desc_list_entry);
	pcd->free_dma_desc_tasklet = DWC_TASK_ALLOC(
		"free_dma_desc_tasklet",
		free_dma_desc_tasklet_func,
		pcd
	);

#ifdef DWC_OTG_COMPLETE_TASKLET
	DWC_LIST_INIT(&pcd->complete_list_entry);
	pcd->complete_tasklet = DWC_TASK_ALLOC(
		"complete_tasklet",
		complete_tasklet_func,
		pcd
	);
#endif // #ifdef DWC_OTG_COMPLETE_TASKLET

	/* Initialize SRP timer */
	core_if->srp_timer = DWC_TIMER_ALLOC("SRP TIMER", srp_timeout, core_if);

	core_if->reset_timer = DWC_TIMER_ALLOC("RESET TIMER", reset_timeout, pcd);

	return pcd;
#ifdef DWC_UTE_CFI
fail:
#endif
	if (pcd->setup_pkt)
		DWC_FREE(pcd->setup_pkt);
	if (pcd->status_buf)
		DWC_FREE(pcd->status_buf);
#ifdef DWC_UTE_CFI
	if (pcd->cfi)
		DWC_FREE(pcd->cfi);
#endif
	if (pcd)
		DWC_FREE(pcd);
	return NULL;

}

/**
 * Remove PCD specific data
 */
void dwc_otg_pcd_remove(dwc_otg_pcd_t * pcd)
{
	dwc_otg_dev_if_t *dev_if = GET_CORE_IF(pcd)->dev_if;

	if (GET_CORE_IF(pcd)->dma_enable) {
		__DWC_DMA_FREE(dwc_otg_get_device(pcd->otg_dev),
			     ((sizeof(*pcd->setup_pkt) + sizeof(uint32_t)) * MAX_SUP_CNT) + sizeof(uint32_t),
			     pcd->setup_pkt,
			     pcd->setup_pkt_dma_handle);
		__DWC_DMA_FREE(dwc_otg_get_device(pcd->otg_dev),
			     sizeof(uint16_t), pcd->status_buf,
			     pcd->status_buf_dma_handle);
		if (GET_CORE_IF(pcd)->dma_desc_enable) {
			dwc_otg_ep_free_desc_chain(pcd,
						   dev_if->setup_desc_addr[0],
						   dev_if->dma_setup_desc_addr
						   [0], 1);
			dwc_otg_ep_free_desc_chain(pcd,
						   dev_if->setup_desc_addr[1],
						   dev_if->dma_setup_desc_addr
						   [1], 1);
			dwc_otg_ep_free_desc_chain(pcd,
						   dev_if->in_desc_addr,
						   dev_if->dma_in_desc_addr, 1);
			dwc_otg_ep_free_desc_chain(pcd,
						   dev_if->out_desc_addr,
						   dev_if->dma_out_desc_addr,
						   1);
		}
	} else {
		DWC_FREE(pcd->setup_pkt);
		DWC_FREE(pcd->status_buf);
	}

#ifdef DEBUG_DWC_OTG_PCD_DMA_AHB_ERROR
	DWC_TASK_FREE(pcd->start_ep0_out_dma_tasklet);
#endif

#ifdef DWC_OTG_COMPLETE_TASKLET
	complete_tasklet_func(pcd);
        DWC_TASK_FREE(pcd->complete_tasklet);
#endif // #ifdef DWC_OTG_COMPLETE_TASKLET

	free_dma_desc_tasklet_func(pcd);
	DWC_TASK_FREE(pcd->free_dma_desc_tasklet);

#ifdef DWC_DO_GADGET_SETUP_TASKLET
	do_gadget_setup_tasklet_func(pcd);
	DWC_TASK_FREE(pcd->do_gadget_setup_tasklet);
#endif

#ifdef DWC_PCD_STOP_TASKLET
	DWC_TASK_FREE(pcd->pcd_stop_tasklet);
#endif

	DWC_TASK_FREE(pcd->clear_ep_mismatch_tasklet);
	DWC_TASK_FREE(pcd->start_xfer_tasklet);
	DWC_TASK_FREE(pcd->test_mode_tasklet);

/* Release the CFI object's dynamic memory */
#ifdef DWC_UTE_CFI
	if (pcd->cfi->ops.release) {
		pcd->cfi->ops.release(pcd->cfi);
	}
#endif

	DWC_FREE(pcd);
}

/**
 * Returns whether registered pcd is dual speed or not
 */
uint32_t dwc_otg_pcd_is_dualspeed(dwc_otg_pcd_t * pcd)
{
	dwc_otg_core_if_t *core_if = GET_CORE_IF(pcd);

	if ((core_if->core_params->speed == DWC_SPEED_PARAM_FULL) ||
	    ((core_if->hwcfg2.b.hs_phy_type == 2) &&
	     (core_if->hwcfg2.b.fs_phy_type == 1) &&
	     (core_if->core_params->ulpi_fs_ls))) {
		return 0;
	}

	return 1;
}

/**
 * Returns whether registered pcd is OTG capable or not
 */
uint32_t dwc_otg_pcd_is_otg(dwc_otg_pcd_t * pcd)
{
	dwc_otg_core_if_t *core_if = GET_CORE_IF(pcd);
	gusbcfg_data_t usbcfg = {.d32 = 0 };

	usbcfg.d32 = DWC_READ_REG32(&core_if->core_global_regs->gusbcfg);
	if (!usbcfg.b.srpcap || !usbcfg.b.hnpcap) {
		return 0;
	}

	return 1;
}

/* stephen.smith@intelligraphics.com */
#if 0
/**
 * This function assigns periodic Tx FIFO to an periodic EP
 * in shared Tx FIFO mode
 */
static uint32_t assign_tx_fifo(dwc_otg_core_if_t * core_if)
{
	uint32_t TxMsk = 1;
	int i;

	for (i = 0; i < core_if->hwcfg4.b.num_in_eps; ++i) {
		if ((TxMsk & core_if->tx_msk) == 0) {
			core_if->tx_msk |= TxMsk;
			return i + 1;
		}
		TxMsk <<= 1;
	}
	return 0;
}
#endif

/**
 * This function assigns periodic Tx FIFO to an periodic EP
 * in shared Tx FIFO mode
 */
static uint32_t assign_perio_tx_fifo(dwc_otg_core_if_t * core_if)
{
	uint32_t PerTxMsk = 1;
	int i;

	for (i = 0; i < core_if->hwcfg4.b.num_dev_perio_in_ep; ++i) {
		if ((PerTxMsk & core_if->p_tx_msk) == 0) {
			core_if->p_tx_msk |= PerTxMsk;
			return i + 1;
		}
		PerTxMsk <<= 1;
	}
	return 0;
}

/**
 * This function releases periodic Tx FIFO
 * in shared Tx FIFO mode
 */
static void release_perio_tx_fifo(dwc_otg_core_if_t * core_if,
				  uint32_t fifo_num)
{
	core_if->p_tx_msk =
	    (core_if->p_tx_msk & (1 << (fifo_num - 1))) ^ core_if->p_tx_msk;
}

/**
 * This function releases periodic Tx FIFO
 * in shared Tx FIFO mode
 */
static void release_tx_fifo(dwc_otg_core_if_t * core_if, uint32_t fifo_num)
{
	core_if->tx_msk =
	    (core_if->tx_msk & (1 << (fifo_num - 1))) ^ core_if->tx_msk;
}

/**
 * This function is being called from gadget 
 * to enable PCD endpoint.
 */
int dwc_otg_pcd_ep_enable(dwc_otg_pcd_t * pcd,
			  const uint8_t * ep_desc, void *usb_ep)
{
	int num, dir;
	dwc_otg_pcd_ep_t *ep = NULL;
	const usb_endpoint_descriptor_t *desc;
#if 0
	fifosize_data_t dptxfsiz = {.d32 = 0 };
	gdfifocfg_data_t gdfifocfg = {.d32 = 0 };
	gdfifocfg_data_t gdfifocfgbase = {.d32 = 0 };
#endif
	int retval = 0;
	int i, epcount;

	desc = (const usb_endpoint_descriptor_t *)ep_desc;

	if (!desc) {
		pcd->ep0.priv = usb_ep;
		ep = &pcd->ep0;
		retval = -DWC_E_INVALID;
		goto out;
	}

	num = UE_GET_ADDR(desc->bEndpointAddress);
	dir = UE_GET_DIR(desc->bEndpointAddress);

	DWC_DEBUGPL(DBG_PCD, "%s: EP%d-%s\n", __func__, num, dir == UE_DIR_IN ? "IN" : "OUT");

	DWC_OTG_PCD_DUMP_DMA_EP0_OUT_REGS(pcd);

	if (!desc->wMaxPacketSize) {
		DWC_WARN("bad maxpacketsize\n");
		retval = -DWC_E_INVALID;
		goto out;
	}

	if (dir == UE_DIR_IN) {
		epcount = pcd->core_if->dev_if->num_in_eps;
		for (i = 0; i < epcount; i++) {
			if (num == pcd->in_ep[i].dwc_ep.num) {
				ep = &pcd->in_ep[i];
				break;
			}
		}
	} else {
		epcount = pcd->core_if->dev_if->num_out_eps;
		for (i = 0; i < epcount; i++) {
			if (num == pcd->out_ep[i].dwc_ep.num) {
				ep = &pcd->out_ep[i];
				break;
			}
		}
	}

	if (!ep) {
		DWC_WARN("bad address\n");
		retval = -DWC_E_INVALID;
		goto out;
	}

	DWC_DEBUGPL(DBG_PCD, "%s %d 0x%02x (%p)\n", __func__, ep->dwc_ep.num, desc->bEndpointAddress, usb_ep);

	ep->desc = desc;
	ep->priv = usb_ep;

	/*
	 * Activate the EP
	 */
	ep->stopped = 0;

	ep->dwc_ep.is_in = (dir == UE_DIR_IN);
	ep->dwc_ep.maxpacket = UGETW(desc->wMaxPacketSize);

	ep->dwc_ep.type = desc->bmAttributes & UE_XFERTYPE;

	if (ep->dwc_ep.is_in) {
		if (!GET_CORE_IF(pcd)->en_multiple_tx_fifo) {
			ep->dwc_ep.tx_fifo_num = 0;

			if ((ep->dwc_ep.type == UE_ISOCHRONOUS)) {
				/*
				 * if ISOC EP then assign a Periodic Tx FIFO.
				 */
				ep->dwc_ep.tx_fifo_num =
				    assign_perio_tx_fifo(GET_CORE_IF(pcd));
			}
		} else {
			/*
			 * if Dedicated FIFOs mode is on then assign a Tx FIFO.
			 */

/* stephen.smith@intelligraphics.com
   I believe this should be the tx_fifo_num, however throught out the code there
   are mixed references, so I have fixed the code to always assure that the tx_fifo_num
   is the same as the endpoint
*/
#if 1
                        ep->dwc_ep.tx_fifo_num = num;
#else
			ep->dwc_ep.tx_fifo_num =
				assign_tx_fifo(GET_CORE_IF(pcd));
#endif
		}

		DWC_OTG_PCD_DUMP_DMA_EP0_OUT_REGS(pcd);

/* stephen.smith@intelligraphics.com
   This breaks dedicated fifos in slave and dma mode.  Other comments exist in the code about this.
   Don't know if this is part of dynamic fifo implementation?
   DWC Datasheet says:
      "This register is available to use only when OTG_EN_DED_TX_FIFO is set to 1 using coreConsultant."
*/
#if 0
		/* Calculating EP info controller base address */
		if (ep->dwc_ep.tx_fifo_num) {
			gdfifocfg.d32 =
				DWC_READ_REG32(&GET_CORE_IF(pcd)->core_global_regs->
						gdfifocfg);
			DWC_DEBUGPL(DBG_PCD, "%s gdfifocfg:0x%08x\n", __func__, gdfifocfg.d32);

			gdfifocfgbase.d32 = gdfifocfg.d32 >> 16;
			dptxfsiz.d32 =
				(DWC_READ_REG32
				 (&GET_CORE_IF(pcd)->
				  core_global_regs->dtxfsiz[ep->dwc_ep.
				  tx_fifo_num]) >> 16);
			gdfifocfg.b.epinfobase =
				gdfifocfgbase.d32 + dptxfsiz.d32;
			DWC_WRITE_REG32(&GET_CORE_IF(pcd)->core_global_regs->
					gdfifocfg, gdfifocfg.d32);
			DWC_DEBUGPL(DBG_PCD, "%s gdfifocfg:0x%08x dptxfsiz:0x%08x gdfifocfg:0x%08x\n", __func__,
					gdfifocfg.d32, dptxfsiz.d32, gdfifocfg.d32);
		}
#endif

	}

	DWC_OTG_PCD_DUMP_DMA_EP0_OUT_REGS(pcd);

	/* Set initial data PID. */
	if (ep->dwc_ep.type == UE_BULK) {
		ep->dwc_ep.data_pid_start = 0;
	}

	/* Alloc DMA Descriptors */
	if (GET_CORE_IF(pcd)->dma_desc_enable) {
#ifndef DWC_UTE_PER_IO
		if (ep->dwc_ep.type != UE_ISOCHRONOUS) {
#endif
			dwc_otg_dev_dma_desc_t *pdesc_addr;

			pdesc_addr = dwc_otg_ep_alloc_desc_chain(
					pcd,
					&ep->dwc_ep.dma_desc_addr,
					MAX_DMA_DESC_CNT
					);

			ep->dwc_ep.desc_addr = pdesc_addr;
			if (!ep->dwc_ep.desc_addr) {
				DWC_WARN("%s, can't allocate DMA descriptor\n",
					 __func__);
				retval = -DWC_E_SHUTDOWN;
				goto out;
			}

#ifndef DWC_UTE_PER_IO
		}
#endif
	}

	DWC_DEBUGPL(DBG_PCD, "Activate %s: type=%d, mps=%d desc=%p dma desc_addr:%p\n",
		    (ep->dwc_ep.is_in ? "IN" : "OUT"),
		    ep->dwc_ep.type, ep->dwc_ep.maxpacket, ep->desc, ep->dwc_ep.desc_addr);
#ifdef DWC_UTE_PER_IO
	ep->dwc_ep.xiso_bInterval = 1 << (ep->desc->bInterval - 1);
#endif

	dwc_otg_ep_activate(GET_CORE_IF(pcd), &ep->dwc_ep);

#ifdef DWC_UTE_CFI
	if (pcd->cfi->ops.ep_enable) {
		pcd->cfi->ops.ep_enable(pcd->cfi, pcd, ep);
	}
#endif

out:
	return retval;
}

/**
 * This function is being called from gadget 
 * to disable PCD endpoint.
 */
int dwc_otg_pcd_ep_disable(dwc_otg_pcd_t * pcd, void *ep_handle)
{
	dwc_otg_pcd_ep_t *ep;
#if 0
	dwc_otg_dev_dma_desc_t *desc_addr;
	dwc_dma_t dma_desc_addr;
#endif

/* stephen.smith@intelligraphics.com */
#if 0
	gdfifocfg_data_t gdfifocfgbase = {.d32 = 0 };
	gdfifocfg_data_t gdfifocfg = {.d32 = 0 };
	fifosize_data_t dptxfsiz = {.d32 = 0 };
#endif

	ep = get_ep_from_handle(pcd, ep_handle);

	if (!ep || !ep->desc) {
		DWC_ERROR("bad ep address (ep_handle:%p)\n", ep_handle);
		return -DWC_E_INVALID;
	}

	DWC_DEBUGPL(DBG_PCD, "%s: EP%d-%s (%p)\n", __func__,
		ep->dwc_ep.num, ep->dwc_ep.is_in ? "IN" : "OUT", ep_handle);

	dwc_otg_request_nuke(ep);

	dwc_otg_ep_deactivate(GET_CORE_IF(pcd), &ep->dwc_ep);
	ep->desc = NULL;
	ep->stopped = 1;

/* stephen.smith@intelligraphics.com */
#if 0
	gdfifocfg.d32 =
	    DWC_READ_REG32(&GET_CORE_IF(pcd)->core_global_regs->gdfifocfg);
	gdfifocfgbase.d32 = gdfifocfg.d32 >> 16;
#endif

	if (ep->dwc_ep.is_in) {
		dwc_otg_flush_tx_fifo(GET_CORE_IF(pcd), ep->dwc_ep.tx_fifo_num);
		release_perio_tx_fifo(GET_CORE_IF(pcd), ep->dwc_ep.tx_fifo_num);
		release_tx_fifo(GET_CORE_IF(pcd), ep->dwc_ep.tx_fifo_num);

/* stephen.smith@intelligraphics.com */
#if 0
		/* Decreasing EPinfo Base Addr */
		dptxfsiz.d32 =
		    (DWC_READ_REG32
		     (&GET_CORE_IF(pcd)->
		      core_global_regs->dtxfsiz[ep->dwc_ep.tx_fifo_num]) >> 16);
		gdfifocfg.b.epinfobase = gdfifocfgbase.d32 - dptxfsiz.d32;
		DWC_WRITE_REG32(&GET_CORE_IF(pcd)->core_global_regs->gdfifocfg,
				gdfifocfg.d32);
#endif
	}

	/* Free DMA Descriptors */
	if (GET_CORE_IF(pcd)->dma_desc_enable) {
		if (ep->dwc_ep.type != UE_ISOCHRONOUS) {
#if 1
/* stephen.smith@intelligraphics.com
   schedule a deferred task to free the DMA Descriptors since this
   can be called in INTR context
*/
			struct pcd_free_dma_desc_context *pdma_ctx;

			pdma_ctx = DWC_ALLOC_ATOMIC(sizeof(*pdma_ctx));
			if (pdma_ctx) {
				DWC_LIST_INIT(&pdma_ctx->lh);
				pdma_ctx->desc_addr = ep->dwc_ep.desc_addr;
				pdma_ctx->dma_desc_addr = ep->dwc_ep.dma_desc_addr;
				ep->dwc_ep.desc_addr = 0;
				ep->dwc_ep.dma_desc_addr = 0;
				DWC_LIST_INSERT_TAIL(&pcd->free_dma_desc_list_entry, &pdma_ctx->lh);
				DWC_DEBUGPL(DBG_PCD, "%s desc_addr:%p dma_desc_addr:0x%08x\n", __func__,
					pdma_ctx->desc_addr, pdma_ctx->dma_desc_addr);
			} else {
				DWC_ERROR("Out of memory, cannot free DMA Descriptor!");
			}

			DWC_DEBUGPL(DBG_PCD, "%s Scheduling DWC TASK free_dma_desc_tasklet\n", __func__);
			DWC_TASK_SCHEDULE(ep->pcd->free_dma_desc_tasklet);
#else
			desc_addr = ep->dwc_ep.desc_addr;
			dma_desc_addr = ep->dwc_ep.dma_desc_addr;

			/* Cannot call dma_free_coherent() with IRQs disabled */
			DWC_OTG_DRIVER_UNLOCK(pcd->core_if);
			dwc_otg_ep_free_desc_chain(pcd,
					desc_addr, dma_desc_addr,
					MAX_DMA_DESC_CNT);
			DWC_OTG_DRIVER_LOCK(pcd->core_if);

			goto out_unlocked;
#endif
		}
	}

#if 0
out_unlocked:
#endif
	DWC_DEBUGPL(DBG_PCD, "%d %s disabled\n", ep->dwc_ep.num,
		    ep->dwc_ep.is_in ? "IN" : "OUT");
	return 0;

}

/******************************************************************************/
#ifdef DWC_UTE_PER_IO

/**
 * Free the request and its extended parts
 *
 */
void dwc_pcd_xiso_ereq_free(dwc_otg_pcd_ep_t * ep, dwc_otg_pcd_request_t * req)
{
	DWC_FREE(req->ext_req.per_io_frame_descs);
	DWC_FREE(req);
}

/**
 * Start the next request in the endpoint's queue.
 *
 */
int dwc_otg_pcd_xiso_start_next_request(dwc_otg_pcd_t * pcd,
					dwc_otg_pcd_ep_t * ep)
{
	int i;
	dwc_otg_pcd_request_t *req = NULL;
	dwc_ep_t *dwcep = NULL;
	struct dwc_iso_xreq_port *ereq = NULL;
	struct dwc_iso_pkt_desc_port *ddesc_iso;
	uint16_t nat;
	depctl_data_t diepctl;

	dwcep = &ep->dwc_ep;

	if (dwcep->xiso_active_xfers > 0) {
#if 0				//Disable this to decrease s/w overhead that is crucial for Isoc transfers
		DWC_WARN("There are currently active transfers for EP%d \
				(active=%d; queued=%d)", dwcep->num, dwcep->xiso_active_xfers, 
				dwcep->xiso_queued_xfers);
#endif
		return 0;
	}

	nat = UGETW(ep->desc->wMaxPacketSize);
	nat = (nat >> 11) & 0x03;

	if (!DWC_CIRCLEQ_EMPTY(&ep->queue)) {
		req = DWC_CIRCLEQ_FIRST(&ep->queue);
		ereq = &req->ext_req;
		ep->stopped = 0;

		/* Get the frame number */
		if (dwcep->xiso_frame_num == 0xFFFFFFFF) {
			dwcep->xiso_frame_num =
			    dwc_otg_get_frame_number(GET_CORE_IF(pcd));
			DWC_DEBUG("FRM_NUM=%d", dwcep->xiso_frame_num);
		}

		ddesc_iso = ereq->per_io_frame_descs;

		if (dwcep->is_in) {
			/* Setup DMA Descriptor chain for IN Isoc request */
			for (i = 0; i < ereq->pio_pkt_count; i++) {
				//if ((i % (nat + 1)) == 0)
				dwcep->xiso_frame_num = (dwcep->xiso_bInterval +
										dwcep->xiso_frame_num) & 0x3FFF;
				dwcep->desc_addr[i].status.b_iso_in.bs = BS_HOST_BUSY;
				dwcep->desc_addr[i].buf =
				    req->dma + ddesc_iso[i].offset;
				dwcep->desc_addr[i].status.b_iso_in.txbytes =
				    ddesc_iso[i].length;
				dwcep->desc_addr[i].status.b_iso_in.framenum =
				    dwcep->xiso_frame_num;
				dwcep->desc_addr[i].status.b_iso_in.txsts = 0;
				dwcep->desc_addr[i].status.b_iso_in.sp =
				    (ddesc_iso[i].length %
				     dwcep->maxpacket) ? 1 : 0;
				dwcep->desc_addr[i].status.b_iso_in.ioc = 0;
				dwcep->desc_addr[i].status.b_iso_in.pid = nat + 1;
				dwcep->desc_addr[i].status.b_iso_in.l = 0;
				dwcep->desc_addr[i].status.b_iso_in.bs =
					BS_HOST_READY;

				/* Process the last descriptor */
				if (i == ereq->pio_pkt_count - 1) {
					dwcep->desc_addr[i].status.b_iso_in.ioc = 1;
					dwcep->desc_addr[i].status.b_iso_in.l = 1;
				}
			}

			/* Setup and start the transfer for this endpoint */
			dwcep->xiso_active_xfers++;
			DWC_WRITE_REG32(&GET_CORE_IF(pcd)->dev_if->
					in_ep_regs[dwcep->num]->diepdma,
					dwcep->dma_desc_addr);
			diepctl.d32 = 0;
			diepctl.b.epena = 1;
			diepctl.b.cnak = 1;
			DWC_MODIFY_REG32(&GET_CORE_IF(pcd)->dev_if->
					 in_ep_regs[dwcep->num]->diepctl, 0,
					 diepctl.d32);
		} else {
			/* Setup DMA Descriptor chain for OUT Isoc request */
			for (i = 0; i < ereq->pio_pkt_count; i++) {
				//if ((i % (nat + 1)) == 0)
				dwcep->xiso_frame_num = (dwcep->xiso_bInterval + 
										dwcep->xiso_frame_num) & 0x3FFF;
				dwcep->desc_addr[i].status.b_iso_out.bs = BS_HOST_BUSY;
				dwcep->desc_addr[i].buf =
				    req->dma + ddesc_iso[i].offset;
				dwcep->desc_addr[i].status.b_iso_out.rxbytes =
				    ddesc_iso[i].length;
				dwcep->desc_addr[i].status.b_iso_out.framenum =
				    dwcep->xiso_frame_num;
				dwcep->desc_addr[i].status.b_iso_out.rxsts = 0;
				dwcep->desc_addr[i].status.b_iso_out.sp =
				    (ddesc_iso[i].length %
				     dwcep->maxpacket) ? 1 : 0;
				dwcep->desc_addr[i].status.b_iso_out.ioc = 0;
				dwcep->desc_addr[i].status.b_iso_out.pid = nat + 1;
				dwcep->desc_addr[i].status.b_iso_out.l = 0;
				dwcep->desc_addr[i].status.b_iso_out.bs = BS_HOST_READY;
				
				/* Process the last descriptor */
				if (i == ereq->pio_pkt_count - 1) {
					dwcep->desc_addr[i].status.b_iso_out.ioc = 1;
					dwcep->desc_addr[i].status.b_iso_out.l = 1;
				}			
			}
			
			/* Setup and start the transfer for this endpoint */
			dwcep->xiso_active_xfers++;
			DWC_WRITE_REG32(&GET_CORE_IF(pcd)->dev_if->
					out_ep_regs[dwcep->num]->doepdma,
					dwcep->dma_desc_addr);
			diepctl.d32 = 0;
			diepctl.b.epena = 1;
			diepctl.b.cnak = 1;
			DWC_MODIFY_REG32(&GET_CORE_IF(pcd)->dev_if->
					 out_ep_regs[dwcep->num]->doepctl, 0,
					 diepctl.d32);
		}

	} else {
		ep->stopped = 1;
	}

	return 0;
}

/**
 *	- Remove the request from the queue
 */
void complete_xiso_ep(dwc_otg_pcd_ep_t * ep)
{
	dwc_otg_pcd_request_t *req = NULL;
	struct dwc_iso_xreq_port *ereq = NULL;
	struct dwc_iso_pkt_desc_port *ddesc_iso = NULL;
	dwc_ep_t *dwcep = NULL;
	int i;
	dwc_otg_pcd_t * pcd;

	//DWC_DEBUG();
	dwcep = &ep->dwc_ep;
	pcd = ep->pcd;

	/* Get the first pending request from the queue */
	if (!DWC_CIRCLEQ_EMPTY(&ep->queue)) {
		req = DWC_CIRCLEQ_FIRST(&ep->queue);
		if (!req) {
			DWC_PRINTF("%s:0x%p, req = NULL!\n", __func__, ep);
			return;
		}
		dwcep->xiso_active_xfers--;
		dwcep->xiso_queued_xfers--;
		/* Remove this request from the queue */
		DWC_CIRCLEQ_REMOVE_INIT(&ep->queue, req, queue_entry);
	} else {
		DWC_PRINTF("%s:0x%p, ep->queue empty!\n", __func__, ep);
		return;
	}

	ep->stopped = 1;
	ereq = &req->ext_req;
	ddesc_iso = ereq->per_io_frame_descs;

	if (dwcep->xiso_active_xfers < 0) {
		DWC_WARN("EP#%d (xiso_active_xfers=%d)", dwcep->num,
			 dwcep->xiso_active_xfers);
	}

	/* Fill the Isoc descs of portable extended req from dma descriptors */
	for (i = 0; i < ereq->pio_pkt_count; i++) {
		if (dwcep->is_in) {	/* IN endpoints */
			ddesc_iso[i].actual_length = ddesc_iso[i].length -
			    dwcep->desc_addr[i].status.b_iso_in.txbytes;
			ddesc_iso[i].status =
			    dwcep->desc_addr[i].status.b_iso_in.txsts;
		} else {	/* OUT endpoints */
			ddesc_iso[i].actual_length = ddesc_iso[i].length -
			    dwcep->desc_addr[i].status.b_iso_out.rxbytes;
			ddesc_iso[i].status =
			    dwcep->desc_addr[i].status.b_iso_out.rxsts;
		}
	}

	/* Call the completion function in the non-portable logic */
	ep->pcd->fops->xisoc_complete(ep->pcd, ep->priv, req->priv, 0,
				      &req->ext_req);

	/* Free the request - specific freeing needed for extended request object */
	dwc_pcd_xiso_ereq_free(ep, req);

	/* Start the next request */
	dwc_otg_pcd_xiso_start_next_request(ep->pcd, ep);

	return;
}

/**
 * Create and initialize the Isoc pkt descriptors of the extended request.
 *
 */
static int dwc_otg_pcd_xiso_create_pkt_descs(dwc_otg_pcd_request_t * req,
					     void *ereq_nonport,
					     int atomic_alloc)
{
	struct dwc_iso_xreq_port *ereq = NULL;
	struct dwc_iso_xreq_port *req_mapped = NULL;
	struct dwc_iso_pkt_desc_port *ipds = NULL;	/* To be created in this function */
	uint32_t pkt_count;
	int i;

	ereq = &req->ext_req;
	req_mapped = (struct dwc_iso_xreq_port *)ereq_nonport;
	pkt_count = req_mapped->pio_pkt_count;

	/* Create the isoc descs */
	if (atomic_alloc) {
		ipds = DWC_ALLOC_ATOMIC(sizeof(*ipds) * pkt_count);
	} else {
		ipds = DWC_ALLOC(sizeof(*ipds) * pkt_count);
	}

	if (!ipds) {
		DWC_ERROR("Failed to allocate isoc descriptors");
		return -DWC_E_NO_MEMORY;
	}

	/* Initialize the extended request fields */
	ereq->per_io_frame_descs = ipds;
	ereq->error_count = 0;
	ereq->pio_alloc_pkt_count = pkt_count;
	ereq->pio_pkt_count = pkt_count;
	ereq->tr_sub_flags = req_mapped->tr_sub_flags;

	/* Init the Isoc descriptors */
	for (i = 0; i < pkt_count; i++) {
		ipds[i].length = req_mapped->per_io_frame_descs[i].length;
		ipds[i].offset = req_mapped->per_io_frame_descs[i].offset;
		ipds[i].status = req_mapped->per_io_frame_descs[i].status;	/* 0 */
		ipds[i].actual_length =
		    req_mapped->per_io_frame_descs[i].actual_length;
	}

	return 0;
}

static void prn_ext_request(struct dwc_iso_xreq_port *ereq)
{
	struct dwc_iso_pkt_desc_port *xfd = NULL;
	int i;

	DWC_DEBUG("per_io_frame_descs=%p", ereq->per_io_frame_descs);
	DWC_DEBUG("tr_sub_flags=%d", ereq->tr_sub_flags);
	DWC_DEBUG("error_count=%d", ereq->error_count);
	DWC_DEBUG("pio_alloc_pkt_count=%d", ereq->pio_alloc_pkt_count);
	DWC_DEBUG("pio_pkt_count=%d", ereq->pio_pkt_count);
	DWC_DEBUG("res=%d", ereq->res);

	for (i = 0; i < ereq->pio_pkt_count; i++) {
		xfd = &ereq->per_io_frame_descs[0];
		DWC_DEBUG("FD #%d", i);

		DWC_DEBUG("xfd->actual_length=%d", xfd->actual_length);
		DWC_DEBUG("xfd->length=%d", xfd->length);
		DWC_DEBUG("xfd->offset=%d", xfd->offset);
		DWC_DEBUG("xfd->status=%d", xfd->status);
	}
}

/**
 *
 */
int dwc_otg_pcd_xiso_ep_queue(dwc_otg_pcd_t * pcd, void *ep_handle,
			      uint8_t * buf, dwc_dma_t dma_buf, uint32_t buflen,
			      int zero, void *req_handle, int atomic_alloc,
			      void *ereq_nonport)
{
	dwc_otg_pcd_request_t *req = NULL;
	dwc_otg_pcd_ep_t *ep;
	int res;

	ep = get_ep_from_handle(pcd, ep_handle);
	if (!ep) {
		DWC_WARN("bad ep\n");
		return -DWC_E_INVALID;
	}

	/* We support this extension only for DDMA mode */
	if (ep->dwc_ep.type == DWC_OTG_EP_TYPE_ISOC)
		if (!GET_CORE_IF(pcd)->dma_desc_enable)
			return -DWC_E_INVALID;

	/* Create a dwc_otg_pcd_request_t object */
	if (atomic_alloc) {
		req = DWC_ALLOC_ATOMIC(sizeof(*req));
	} else {
		req = DWC_ALLOC(sizeof(*req));
	}

	if (!req) {
		return -DWC_E_NO_MEMORY;
	}

	/* Create the Isoc descs for this request which shall be the exact match
	 * of the structure sent to us from the non-portable logic */
	res =
	    dwc_otg_pcd_xiso_create_pkt_descs(req, ereq_nonport, atomic_alloc);
	if (res) {
		DWC_WARN("Failed to init the Isoc descriptors");
		DWC_FREE(req);
		return res;
	}

	DWC_CIRCLEQ_INIT_ENTRY(req, queue_entry);
	req->buf = buf;
	req->dma = dma_buf;
	req->length = buflen;
	req->sent_zlp = zero;
	req->priv = req_handle;

	ep->dwc_ep.dma_addr = dma_buf;
	ep->dwc_ep.start_xfer_buff = buf;
	ep->dwc_ep.xfer_buff = buf;
	ep->dwc_ep.xfer_len = 0;
	ep->dwc_ep.xfer_count = 0;
	ep->dwc_ep.sent_zlp = 0;
	ep->dwc_ep.total_len = buflen;

	/* Add this request to the tail */
	DWC_CIRCLEQ_INSERT_TAIL(&ep->queue, req, queue_entry);
	ep->dwc_ep.xiso_queued_xfers++;

//DWC_DEBUG("CP_0");
//DWC_DEBUG("req->ext_req.tr_sub_flags=%d", req->ext_req.tr_sub_flags);
//prn_ext_request((struct dwc_iso_xreq_port *) ereq_nonport);
//prn_ext_request(&req->ext_req);

	/* If the req->status == ASAP  then check if there is any active transfer
	 * for this endpoint. If no active transfers, then get the first entry
	 * from the queue and start that transfer
	 */
	if (req->ext_req.tr_sub_flags == DWC_EREQ_TF_ASAP) {
		res = dwc_otg_pcd_xiso_start_next_request(pcd, ep);
		if (res) {
			DWC_WARN("Failed to start the next Isoc transfer");
			DWC_FREE(req);
			return res;
		}
	}

	return 0;
}

#endif
/* END ifdef DWC_UTE_PER_IO ***************************************************/
int dwc_otg_pcd_ep_queue(dwc_otg_pcd_t * pcd, void *ep_handle,
			 uint8_t * buf, dwc_dma_t dma_buf, uint32_t buflen,
			 int zero, void *req_handle, int atomic_alloc)
{
	dwc_otg_pcd_request_t *req;
	dwc_otg_pcd_ep_t *ep;
	uint32_t max_transfer;
	int ret = 0;

	ep = get_ep_from_handle(pcd, ep_handle);
	if (!ep || (!ep->desc && ep->dwc_ep.num != 0)) {
		DWC_WARN("bad ep\n");
		return -DWC_E_INVALID;
	}

	if (atomic_alloc) {
		req = DWC_ALLOC_ATOMIC(sizeof(*req));
	} else {
		DWC_OTG_DRIVER_UNLOCK(pcd->core_if);
		req = DWC_ALLOC(sizeof(*req));
		DWC_OTG_DRIVER_LOCK(pcd->core_if);
	}

	if (!req) {
		return -DWC_E_NO_MEMORY;
	}
	DWC_CIRCLEQ_INIT_ENTRY(req, queue_entry);
	if (!GET_CORE_IF(pcd)->core_params->opt) {
		if (ep->dwc_ep.num != 0) {
			DWC_ERROR("queue req %p, len %d buf %p\n",
				  req_handle, buflen, buf);
		}
	}

	req->buf = buf;
	req->dma = dma_buf;
	req->length = buflen;
	req->sent_zlp = zero;
	req->priv = req_handle;

	DWC_DEBUGPL(DBG_PCD, "%s ep%d: buf:0x%08x dma_buf:0x%08x len:%d zero:%d\n", __func__,
			ep->dwc_ep.num, (uint32_t) buf, (uint32_t) dma_buf, buflen, zero);

#if 0
	/*
	 * For EP0 IN without premature status, zlp is required?
	 */
	if (ep->dwc_ep.num == 0 && ep->dwc_ep.is_in) {
		DWC_DEBUGPL(DBG_PCDV, "%d-OUT ZLP\n", ep->dwc_ep.num);
		//_req->zero = 1;
	}
#endif

	/* Start the transfer */
	if (DWC_CIRCLEQ_EMPTY(&ep->queue) && !ep->stopped) {
		/* EP0 Transfer? */
		if (ep->dwc_ep.num == 0) {
			switch (pcd->ep0state) {
			case EP0_IN_DATA_PHASE:
				DWC_DEBUGPL(DBG_PCD,
					    "%s ep0: EP0_IN_DATA_PHASE\n",
					    __func__);
				break;

			case EP0_OUT_DATA_PHASE:
				DWC_DEBUGPL(DBG_PCD, 
					    "%s ep0: EP0_OUT_DATA_PHASE (request_config:%d)\n",
					    __func__, pcd->request_config);
				if (pcd->request_config) {
					/* Complete STATUS PHASE */
					ep->dwc_ep.is_in = 1;
					pcd->ep0state = EP0_IN_STATUS_PHASE;
				}
				break;

			case EP0_IN_STATUS_PHASE:
				DWC_DEBUGPL(DBG_PCD,
					    "%s ep0: EP0_IN_STATUS_PHASE\n",
					    __func__);
				break;

			default:
				DWC_DEBUGPL(DBG_ANY, "ep0: odd state %d\n",
					    pcd->ep0state);
				return -DWC_E_SHUTDOWN;
			}

			ep->dwc_ep.dma_addr = dma_buf;
			ep->dwc_ep.start_xfer_buff = buf;
			ep->dwc_ep.xfer_buff = buf;
			ep->dwc_ep.xfer_len = buflen;
			ep->dwc_ep.xfer_count = 0;
			ep->dwc_ep.sent_zlp = 0;
			ep->dwc_ep.total_len = ep->dwc_ep.xfer_len;

			if (zero) {
				if ((ep->dwc_ep.xfer_len %
				     ep->dwc_ep.maxpacket == 0)
				    && (ep->dwc_ep.xfer_len != 0)) {
					ep->dwc_ep.sent_zlp = 1;
				}

			}

			dwc_otg_ep0_start_transfer(GET_CORE_IF(pcd),
						   &ep->dwc_ep);
		}		// non-ep0 endpoints
		else {

#ifdef DWC_UTE_CFI
			if (ep->dwc_ep.buff_mode != BM_STANDARD) {
				/* store the request length */
				ep->dwc_ep.cfi_req_len = buflen;
				pcd->cfi->ops.build_descriptors(pcd->cfi, pcd,
								ep, req);
			} else {
#endif
				max_transfer =
				    GET_CORE_IF(ep->pcd)->
				    core_params->max_transfer_size;

				/* Setup and start the Transfer */
				ep->dwc_ep.dma_addr = dma_buf;
				ep->dwc_ep.start_xfer_buff = buf;
				ep->dwc_ep.xfer_buff = buf;
				ep->dwc_ep.xfer_len = 0;
				ep->dwc_ep.xfer_count = 0;
				ep->dwc_ep.sent_zlp = 0;
				ep->dwc_ep.total_len = buflen;

				ep->dwc_ep.maxxfer = max_transfer;
				if (GET_CORE_IF(pcd)->dma_desc_enable) {
					uint32_t out_max_xfer =
					    DDMA_MAX_TRANSFER_SIZE -
					    (DDMA_MAX_TRANSFER_SIZE % 4);
					if (ep->dwc_ep.is_in) {
						if (ep->dwc_ep.maxxfer >
						    DDMA_MAX_TRANSFER_SIZE) {
							ep->dwc_ep.maxxfer =
							    DDMA_MAX_TRANSFER_SIZE;
						}
					} else {
						if (ep->dwc_ep.maxxfer >
						    out_max_xfer) {
							ep->dwc_ep.maxxfer =
							    out_max_xfer;
						}
					}
				}
				if (ep->dwc_ep.maxxfer < ep->dwc_ep.total_len) {
					ep->dwc_ep.maxxfer -=
					    (ep->dwc_ep.maxxfer %
					     ep->dwc_ep.maxpacket);
				}

				if (zero) {
					if ((ep->dwc_ep.total_len %
					     ep->dwc_ep.maxpacket == 0)
					    && (ep->dwc_ep.total_len != 0)) {
						ep->dwc_ep.sent_zlp = 1;
					}
				}
#ifdef DWC_UTE_CFI
			}
#endif
			if ((GET_CORE_IF(pcd)->en_multiple_tx_fifo == 0) && (ep->dwc_ep.type != DWC_OTG_EP_TYPE_ISOC)) {
				if (check_next_ep_token(ep))
					dwc_otg_ep_start_transfer(GET_CORE_IF(pcd),
								  &ep->dwc_ep);
				else ret = -DWC_E_AGAIN;
			}
			else
				dwc_otg_ep_start_transfer(GET_CORE_IF(pcd),
						&ep->dwc_ep);
		}
	}

	if ((ret == 0) && (req != 0)) {
		++pcd->request_pending;
		DWC_CIRCLEQ_INSERT_TAIL(&ep->queue, req, queue_entry);
		if (ep->dwc_ep.is_in && ep->stopped && !(GET_CORE_IF(pcd)->dma_enable)) {
			/** @todo NGS Create a function for this. */
			diepmsk_data_t diepmsk = {.d32 = 0 };
			diepmsk.b.intktxfemp = 1;
			if (GET_CORE_IF(pcd)->multiproc_int_enable) {
				DWC_MODIFY_REG32(&GET_CORE_IF(pcd)->dev_if->
						 dev_global_regs->
						 diepeachintmsk[ep->dwc_ep.num],
						 0, diepmsk.d32);
			} else {
				DWC_MODIFY_REG32(&GET_CORE_IF(pcd)->dev_if->
						 dev_global_regs->diepmsk, 0,
						 diepmsk.d32);
			}

		}
	}

	DWC_DEBUGPL(DBG_PCD, "%s: ep%d %s req_pend:%d\n", __func__,
		ep->dwc_ep.num, ep->dwc_ep.is_in ? "IN" : "OUT", pcd->request_pending);

	return ret;
}

int dwc_otg_pcd_ep_dequeue(dwc_otg_pcd_t * pcd, void *ep_handle,
			   void *req_handle)
{
	dwc_otg_pcd_request_t *req;
	dwc_otg_pcd_ep_t *ep;

	DWC_DEBUGPL(DBG_PCDV, "%s\n", __func__);

	ep = get_ep_from_handle(pcd, ep_handle);
	if (!ep || (!ep->desc && ep->dwc_ep.num != 0)) {
		DWC_WARN("bad argument\n");
		return -DWC_E_INVALID;
	}

	/* make sure it's actually queued on this endpoint */
	DWC_CIRCLEQ_FOREACH(req, &ep->queue, queue_entry) {
		if (req->priv == (void *)req_handle) {
			break;
		}
	}

	if (req->priv != (void *)req_handle) {
		return -DWC_E_INVALID;
	}

	if (!DWC_CIRCLEQ_EMPTY_ENTRY(req, queue_entry)) {
		DWC_CIRCLEQ_REMOVE_INIT(&ep->queue, req, queue_entry);
		dwc_otg_request_done(ep, req, -DWC_E_RESTART);
	} else {
		req = NULL;
	}

	return req ? 0 : -DWC_E_SHUTDOWN;
}

int dwc_otg_pcd_ep_halt(dwc_otg_pcd_t * pcd, void *ep_handle, int value)
{
	dwc_otg_pcd_ep_t *ep;
	int retval = 0;

	DWC_DEBUGPL(DBG_PCD, "%s\n", __func__);

	ep = get_ep_from_handle(pcd, ep_handle);

	if (!ep || (!ep->desc && ep != &pcd->ep0) ||
	    (ep->desc && (ep->desc->bmAttributes == UE_ISOCHRONOUS))) {
		DWC_WARN("%s, bad ep\n", __func__);
		return -DWC_E_INVALID;
	}

	if (!DWC_CIRCLEQ_EMPTY(&ep->queue)) {
		DWC_WARN("%d %s XFer In process\n", ep->dwc_ep.num,
			 ep->dwc_ep.is_in ? "IN" : "OUT");
		retval = -DWC_E_AGAIN;
	} else if (value == 0) {
		dwc_otg_ep_clear_stall(GET_CORE_IF(pcd), &ep->dwc_ep);
	} else if (value == 1) {
		if (ep->dwc_ep.is_in == 1 && GET_CORE_IF(pcd)->dma_desc_enable) {
			dtxfsts_data_t txstatus;
			fifosize_data_t txfifosize;

			txfifosize.d32 = DWC_READ_REG32(
				&GET_CORE_IF(pcd)->core_global_regs->dtxfsiz[ep->dwc_ep.tx_fifo_num]);
			txstatus.d32 = DWC_READ_REG32(
				&GET_CORE_IF(pcd)->dev_if->in_ep_regs[ep->dwc_ep.num]->dtxfsts);
			if (txstatus.b.txfspcavail < txfifosize.b.depth) {
/* stephen.smith@intelligraphics.com
   TODO: condition never clears up.  Need to look into
     1. why there is data pending
     2. can I flush?
*/
#if 0
				DWC_WARN("Data In Tx Fifo (txfspcavail:%d depth:%d)\n",
						txstatus.b.txfspcavail,
						txfifosize.b.depth
					);
				retval = -DWC_E_AGAIN;
			} else {
#else
			}
#endif
				if (ep->dwc_ep.num == 0) {
					pcd->ep0state = EP0_STALL;
				}

				ep->stopped = 1;
				dwc_otg_ep_set_stall(GET_CORE_IF(pcd), &ep->dwc_ep);
#if 0
			}
#endif
		} else {
			if (ep->dwc_ep.num == 0) {
				pcd->ep0state = EP0_STALL;
			}

			ep->stopped = 1;

			dwc_otg_ep_set_stall(GET_CORE_IF(pcd), &ep->dwc_ep);
		}
	} else if (value == 2) {
		ep->dwc_ep.stall_clear_flag = 0;
	} else if (value == 3) {
		ep->dwc_ep.stall_clear_flag = 1;
	}

	return retval;
}

/**
 * This function initiates remote wakeup of the host from suspend state.
 */
void dwc_otg_pcd_rem_wkup_from_suspend(dwc_otg_pcd_t * pcd, int set)
{
	dctl_data_t dctl = { 0 };
	dwc_otg_core_if_t *core_if = GET_CORE_IF(pcd);
	dsts_data_t dsts;

	dsts.d32 = DWC_READ_REG32(&core_if->dev_if->dev_global_regs->dsts);
	if (!dsts.b.suspsts) {
		DWC_WARN("Remote wakeup while is not in suspend state\n");
	}
	/* Check if DEVICE_REMOTE_WAKEUP feature enabled */
	if (pcd->remote_wakeup_enable) {
		if (set) {

			if (core_if->adp_enable) {
				gpwrdn_data_t gpwrdn;

				dwc_otg_adp_probe_stop(core_if);

				/* Mask SRP detected interrupt from Power Down Logic */
				gpwrdn.d32 = 0;
				gpwrdn.b.srp_det_msk = 1;
				DWC_MODIFY_REG32(&core_if->core_global_regs->
						 gpwrdn, gpwrdn.d32, 0);

				/* Disable Power Down Logic */
				gpwrdn.d32 = 0;
				gpwrdn.b.pmuactv = 1;
				DWC_MODIFY_REG32(&core_if->core_global_regs->
						 gpwrdn, gpwrdn.d32, 0);

				/*
				 * Initialize the Core for Device mode.
				 */
				core_if->op_state = B_PERIPHERAL;
				dwc_otg_core_init(core_if);
				dwc_otg_enable_global_interrupts(core_if);
				cil_pcd_start(core_if);

				dwc_otg_initiate_srp(core_if);
			}

			dctl.b.rmtwkupsig = 1;
			DWC_MODIFY_REG32(&core_if->dev_if->
					 dev_global_regs->dctl, 0, dctl.d32);
			DWC_DEBUGPL(DBG_PCD, "Set Remote Wakeup\n");

			dwc_mdelay(2);
			DWC_MODIFY_REG32(&core_if->dev_if->
					 dev_global_regs->dctl, dctl.d32, 0);
			DWC_DEBUGPL(DBG_PCD, "Clear Remote Wakeup\n");
		}
	} else {
		DWC_DEBUGPL(DBG_PCD, "Remote Wakeup is disabled\n");
	}
}

#ifdef CONFIG_USB_DWC_OTG_LPM
/**
 * This function initiates remote wakeup of the host from L1 sleep state.
 */
void dwc_otg_pcd_rem_wkup_from_sleep(dwc_otg_pcd_t * pcd, int set)
{
	glpmcfg_data_t lpmcfg;
	dwc_otg_core_if_t *core_if = GET_CORE_IF(pcd);

	lpmcfg.d32 = DWC_READ_REG32(&core_if->core_global_regs->glpmcfg);

	/* Check if we are in L1 state */
	if (!lpmcfg.b.prt_sleep_sts) {
		DWC_DEBUGPL(DBG_PCD, "Device is not in sleep state\n");
		return;
	}

	/* Check if host allows remote wakeup */
	if (!lpmcfg.b.rem_wkup_en) {
		DWC_DEBUGPL(DBG_PCD, "Host does not allow remote wakeup\n");
		return;
	}

	/* Check if Resume OK */
	if (!lpmcfg.b.sleep_state_resumeok) {
		DWC_DEBUGPL(DBG_PCD, "Sleep state resume is not OK\n");
		return;
	}

	lpmcfg.d32 = DWC_READ_REG32(&core_if->core_global_regs->glpmcfg);
	lpmcfg.b.en_utmi_sleep = 0;
	lpmcfg.b.hird_thres &= (~(1 << 4));
	DWC_WRITE_REG32(&core_if->core_global_regs->glpmcfg, lpmcfg.d32);

	if (set) {
		dctl_data_t dctl = {.d32 = 0 };
		dctl.b.rmtwkupsig = 1;
		/* Set RmtWkUpSig bit to start remote wakup signaling.
		 * Hardware will automatically clear this bit.
		 */
		DWC_MODIFY_REG32(&core_if->dev_if->dev_global_regs->dctl,
				 0, dctl.d32);
		DWC_DEBUGPL(DBG_PCD, "Set Remote Wakeup\n");
	}

}
#endif

/**
 * Performs remote wakeup.
 */
void dwc_otg_pcd_remote_wakeup(dwc_otg_pcd_t * pcd, int set)
{
	dwc_otg_core_if_t *core_if = GET_CORE_IF(pcd);

	if (dwc_otg_is_device_mode(core_if)) {
#ifdef CONFIG_USB_DWC_OTG_LPM
		if (core_if->lx_state == DWC_OTG_L1) {
			dwc_otg_pcd_rem_wkup_from_sleep(pcd, set);
		} else {
#endif
			dwc_otg_pcd_rem_wkup_from_suspend(pcd, set);
#ifdef CONFIG_USB_DWC_OTG_LPM
		}
#endif
	}
	return;
}

void dwc_otg_pcd_disconnect_us(dwc_otg_pcd_t * pcd, int no_of_usecs)
{
	dwc_otg_core_if_t *core_if = GET_CORE_IF(pcd);
	dctl_data_t dctl = { 0 };

	if (dwc_otg_is_device_mode(core_if)) {
		dctl.b.sftdiscon = 1;
		DWC_DEBUGPL(DBG_PCD, "Soft disconnect for %d useconds\n", no_of_usecs);
		DWC_MODIFY_REG32(&core_if->dev_if->dev_global_regs->dctl, 0, dctl.d32);
		dwc_udelay(no_of_usecs);
		DWC_MODIFY_REG32(&core_if->dev_if->dev_global_regs->dctl, dctl.d32, 0);

	} else{
		DWC_ERROR("NOT SUPPORTED IN HOST MODE\n");
	}
}

void dwc_otg_pcd_soft_disconnect(dwc_otg_pcd_t * pcd)
{
	dwc_otg_core_if_t *core_if = GET_CORE_IF(pcd);
	dctl_data_t dctl = { 0 };

	if (dwc_otg_is_device_mode(core_if)) {
		dctl.d32 = DWC_READ_REG32(&core_if->dev_if->dev_global_regs->dctl);
		dctl.b.sftdiscon = 1;
		DWC_WRITE_REG32(&core_if->dev_if->dev_global_regs->dctl, dctl.d32);
	} else {
		DWC_ERROR("NOT SUPPORTED IN HOST MODE\n");
	}
}

void dwc_otg_pcd_soft_connect(dwc_otg_pcd_t * pcd)
{
	dwc_otg_core_if_t *core_if = GET_CORE_IF(pcd);
	dctl_data_t dctl;

	if (dwc_otg_is_device_mode(core_if)) {
		dctl.d32 = DWC_READ_REG32(&core_if->dev_if->dev_global_regs->dctl);
		dctl.b.sftdiscon = 0;
		DWC_WRITE_REG32(&core_if->dev_if->dev_global_regs->dctl, dctl.d32);
	} else{
		DWC_ERROR("NOT SUPPORTED IN HOST MODE\n");
	}
}

int dwc_otg_pcd_wakeup(dwc_otg_pcd_t * pcd)
{
	dsts_data_t dsts;
	gotgctl_data_t gotgctl;

	/*
	 * This function starts the Protocol if no session is in progress. If
	 * a session is already in progress, but the device is suspended,
	 * remote wakeup signaling is started.
	 */

	/* Check if valid session */
	gotgctl.d32 =
	    DWC_READ_REG32(&(GET_CORE_IF(pcd)->core_global_regs->gotgctl));
	if (gotgctl.b.bsesvld) {
		/* Check if suspend state */
		dsts.d32 =
		    DWC_READ_REG32(&
				   (GET_CORE_IF(pcd)->dev_if->
				    dev_global_regs->dsts));
		if (dsts.b.suspsts) {
			dwc_otg_pcd_remote_wakeup(pcd, 1);
		}
	} else {
		dwc_otg_pcd_initiate_srp(pcd);
	}

	return 0;

}

int dwc_otg_pcd_ep_set_stall(dwc_otg_pcd_t * pcd, void *ep_handle)
{
	dwc_otg_pcd_ep_t *ep;

	DWC_DEBUGPL(DBG_PCD, "%s\n", __func__);

	ep = get_ep_from_handle(pcd, ep_handle);
	if (!ep) {
		DWC_WARN("%s, bad ep\n", __func__);
		return -DWC_E_INVALID;
	}

	dwc_otg_ep_set_stall(GET_CORE_IF(pcd), &ep->dwc_ep);
	return 0;
}

int dwc_otg_pcd_ep_clear_stall(dwc_otg_pcd_t * pcd, void *ep_handle)
{
	dwc_otg_pcd_ep_t *ep;

	DWC_DEBUGPL(DBG_PCD, "%s\n", __func__);

	ep = get_ep_from_handle(pcd, ep_handle);
	if (!ep) {
		DWC_WARN("%s, bad ep\n", __func__);
		return -DWC_E_INVALID;
	}

	dwc_otg_ep_clear_stall(GET_CORE_IF(pcd), &ep->dwc_ep);
	return 0;
}

/**
 * Start the SRP timer to detect when the SRP does not complete within
 * 6 seconds.
 *
 * @param pcd the pcd structure.
 */
void dwc_otg_pcd_initiate_srp(dwc_otg_pcd_t * pcd)
{
	dwc_otg_initiate_srp(GET_CORE_IF(pcd));
}

int dwc_otg_pcd_get_frame_number(dwc_otg_pcd_t * pcd)
{
	return dwc_otg_get_frame_number(GET_CORE_IF(pcd));
}

int dwc_otg_pcd_is_lpm_enabled(dwc_otg_pcd_t * pcd)
{
	return GET_CORE_IF(pcd)->core_params->lpm_enable;
}

uint32_t get_b_hnp_enable(dwc_otg_pcd_t * pcd)
{
	return pcd->b_hnp_enable;
}

uint32_t get_a_hnp_support(dwc_otg_pcd_t * pcd)
{
	return pcd->a_hnp_support;
}

uint32_t get_a_alt_hnp_support(dwc_otg_pcd_t * pcd)
{
	return pcd->a_alt_hnp_support;
}

int dwc_otg_pcd_get_rmwkup_enable(dwc_otg_pcd_t * pcd)
{
	return pcd->remote_wakeup_enable;
}

#endif /* DWC_HOST_ONLY */
