/*
 * drivers/staging/gs/provision.c
 *
 * Driver for the provison.
 *
 * Copyright (C) 2012, 2013 GrandStream Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/delay.h>
#include <linux/rslib.h>
#include <linux/moduleparam.h>
#include <linux/platform_device.h>
#include <linux/dma-mapping.h>
#include <linux/types.h>
#include <linux/clk.h>
#include <linux/interrupt.h>
#include <linux/semaphore.h>
#include <linux/io.h>
#include <linux/slab.h>
#include <linux/ccu.h>
#include <linux/completion.h>
#include <linux/uaccess.h>
#include <linux/proc_fs.h>
#include <linux/vmalloc.h>
#include <linux/seq_file.h>

#include "provision.h"
#include "mmc_rw.h"

#define PROV_ID_SECTORS (32UL)
#define PROV_CURRENT_START_SEC	(34305UL)
#define PROV_LASTUPDATE_START_SEC	(PROV_CURRENT_START_SEC+PROV_ID_SECTORS)
#define IMGHDR_START_SECTOR	(PROV_LASTUPDATE_START_SEC+PROV_ID_SECTORS)
#define IMGHDR_SECTORS_OLD		(512UL-(PROV_ID_SECTORS*2))
#define IMGHDR_SECTORS		(IMGHDR_SECTORS_OLD+(2*(IMGHDR_SECTORS_OLD/8)))

int
provision_proc_read_status (
    char            *page,
    char            **start,
    off_t           off,
    int             count,
    int             *eof,
    void            *data)
{
	int len = 0;
	unsigned int i;

	if(len<=count)
		len += sprintf(page + len, "%-16s: %lu\n", "Start Sector", IMGHDR_START_SECTOR);
	if(len<=count)
		len += sprintf(page + len, "%-16s: %lu\n\n", "Sectors", IMGHDR_SECTORS);

	for(i=0;i<ARRAY_SIZE(provs);i++)
	{
		if(len<=count)
		{
			len += sprintf(page + len, "%-16s: %lu\n", provs[i].name, provs[i].start[0]);;
		}
	}

	if(len<=count)
		len += sprintf(page + len, "\n%-16s: %lu\n", "Current", PROV_CURRENT_START_SEC);
	if(len<=count)
		len += sprintf(page + len, "%-16s: %lu\n", "Lastupdate", PROV_LASTUPDATE_START_SEC);

	return len;
}

int
provision_proc_read_mmc (
    char            *page,
    char            **start,
    off_t           off,
    int             count,
    int             *eof,
    void            *data)
{
	int len;
	int rval;
	prov_t *prov = (prov_t *)data;
	ImageHeader *hdr;
	sector_t start_sector;
	sector_t i;

	hdr = (ImageHeader *)vmalloc(PAGE_SIZE);
	if(!hdr)	return (-ENOMEM);

	len = sizeof(ImageHeader);
	memset(hdr, 0, sizeof(ImageHeader));
	start_sector = prov->start[0];
	for(i=0; i<IMGHDR_SECTORS/ARRAY_SIZE(provs); i++)
	{
		rval = emmc_read(start_sector+i, 1, (char *)hdr, 512);
		if(0==rval) break;
	}	

	//if (hdr->signature != GS_PROD_SIGNATURE || rval != 0)
	//	memset(hdr, 0x0, sizeof(seg_header_t));

	memcpy(page, hdr, sizeof(ImageHeader));
#if 0
	if (copy_to_user(page, hdr, sizeof(seg_header_t)))
	{
		vfree(hdr);
		return (-EFAULT);
	}
#endif

	if (len <= off+count) *eof = 1;
	*start = page + off;
	len -= off;
	if (len>count) len = count;
	if (len<0) len = 0;

	vfree(hdr);
	return (len);
}

int
provision_proc_write_mmc (
    struct file     *file,
    const char      *buffer,
    unsigned long   count,
    void            *data)
{
	int rval = 0;
	prov_t *prov = (prov_t *)data;
	ImageHeader *hdr;
	sector_t off;
	sector_t start;

	if (count > sizeof(ImageHeader)) return (-EINVAL);

	hdr = (ImageHeader *)vmalloc(PAGE_SIZE);
	if(!hdr)	return (-ENOMEM);

	memset(hdr, 0, sizeof(ImageHeader));
	if (copy_from_user(hdr, buffer, count))
	{
		vfree(hdr);
		return (-EFAULT);
	}

	start= prov->start[0];
	for(off=0; off<IMGHDR_SECTORS/ARRAY_SIZE(provs); off++)
	{
		rval = emmc_write(start+off, 1, (char *)hdr, 512);
		if(0==rval ) break;
	}

	vfree(hdr);

	return (count);
}

int
provision_proc_read_current (
    char            *page,
    char            **start,
    off_t           off,
    int             count,
    int             *eof,
    void            *data)
{
	int len;
	int rval;

	ImageHeader *hdr;
	sector_t start_sector;
	sector_t i;

	hdr = (ImageHeader *)vmalloc(PAGE_SIZE);
	if(!hdr)	return (-ENOMEM);

	len = sizeof(ImageHeader);
	memset(hdr, 0, sizeof(ImageHeader));
	start_sector= PROV_CURRENT_START_SEC;
	for(i=0; i<PROV_ID_SECTORS; i++)
	{
		rval = emmc_read(start_sector+i, 1, (char *)hdr, 512);
		if(0==rval) break;
	}	

	memcpy(page, hdr, sizeof(ImageHeader));

	if (len <= off+count) *eof = 1;
	*start = page + off;
	len -= off;
	if (len>count) len = count;
	if (len<0) len = 0;

	vfree(hdr);
	return (len);
}

int
provision_proc_write_current (
    struct file     *file,
    const char      *buffer,
    unsigned long   count,
    void            *data)
{
	int rval = 0;
	ImageHeader *hdr;
	sector_t off;
	sector_t start;

	if (count > sizeof(ImageHeader)) return (-EINVAL);

	hdr = (ImageHeader *)vmalloc(PAGE_SIZE);
	if(!hdr)	return (-ENOMEM);

	memset(hdr, 0, sizeof(ImageHeader));
	if (copy_from_user(hdr, buffer, count))
	{
		vfree(hdr);
		return (-EFAULT);
	}

	start= PROV_CURRENT_START_SEC;
	for(off=0; off<PROV_ID_SECTORS; off++)
	{
		rval = emmc_write(start+off, 1, (char *)hdr, 512);
		if(0==rval ) break;
	}

	vfree(hdr);

	return (count);
}


int
provision_proc_read_lastupdate (
    char            *page,
    char            **start,
    off_t           off,
    int             count,
    int             *eof,
    void            *data)
{
	int len;
	int rval;

	ImageHeader *hdr;
	sector_t start_sector;
	sector_t i;

	hdr = (ImageHeader *)vmalloc(PAGE_SIZE);
	if(!hdr)	return (-ENOMEM);

	len = sizeof(ImageHeader);
	memset(hdr, 0, sizeof(ImageHeader));
	start_sector= PROV_LASTUPDATE_START_SEC;
	for(i=0; i<PROV_ID_SECTORS; i++)
	{
		rval = emmc_read(start_sector+i, 1, (char *)hdr, 512);
		if(0==rval) break;
	}	

	memcpy(page, hdr, sizeof(ImageHeader));

	if (len <= off+count) *eof = 1;
	*start = page + off;
	len -= off;
	if (len>count) len = count;
	if (len<0) len = 0;

	vfree(hdr);
	return (len);
}

int
provision_proc_write_lastupdate (
    struct file     *file,
    const char      *buffer,
    unsigned long   count,
    void            *data)
{
	int rval = 0;
	ImageHeader *hdr;
	sector_t off;
	sector_t start;

	if (count > sizeof(ImageHeader)) return (-EINVAL);

	hdr = (ImageHeader *)vmalloc(PAGE_SIZE);
	if(!hdr)	return (-ENOMEM);

	memset(hdr, 0, sizeof(ImageHeader));
	if (copy_from_user(hdr, buffer, count))
	{
		vfree(hdr);
		return (-EFAULT);
	}

	start= PROV_LASTUPDATE_START_SEC;
	for(off=0; off<PROV_ID_SECTORS; off++)
	{
		rval = emmc_write(start+off, 1, (char *)hdr, 512);
		if(0==rval ) break;
	}

	vfree(hdr);

	return (count);
}

static int prov_partitions_show(struct seq_file *f, void *v)
{
	int i = *(loff_t *) v;

	if (i < PROV_COUNT) {
		if (i == 0)
			seq_puts(f, "ID procfname devfname:\n");
		seq_printf(f, "%4d %11s %s\n", provs[i].id, provs[i].name, provs[i].devfname);
	}

	return 0;
}

static void *prov_partitions_start(struct seq_file *f, loff_t *pos)
{
	if (*pos < PROV_COUNT)
		return pos;
	return NULL;
}

static void *prov_partitions_next(struct seq_file *f, void *v, loff_t *pos)
{
	(*pos)++;
	if (*pos >= PROV_COUNT)
		return NULL;
	return pos;
}

static void prov_partitions_stop(struct seq_file *f, void *v)
{
	/* Nothing to do */
}

static const struct seq_operations prov_partitions_ops = {
	.start = prov_partitions_start,
	.next  = prov_partitions_next,
	.stop  = prov_partitions_stop,
	.show  = prov_partitions_show
};

static int prov_partitions_open(struct inode *inode, struct file *filp)
{
        return seq_open(filp, &prov_partitions_ops);
}

static const struct file_operations proc_prov_partitions_operations = {
        .open           = prov_partitions_open,
        .read           = seq_read,
        .llseek         = seq_lseek,
        .release        = seq_release,
};

int provision_install_proc_entry_mmc (void)
{
	struct proc_dir_entry *provision_root_dir;
	struct proc_dir_entry   *ent;
	int i;

	for(i=0;i<ARRAY_SIZE(provs);i++)
	{
		provs[i].start[0] = IMGHDR_START_SECTOR+i*(IMGHDR_SECTORS/ARRAY_SIZE(provs));
	}

	provision_root_dir = proc_mkdir( "provision", NULL);
	if (provision_root_dir != NULL)
	{
		struct proc_dir_entry	*dent;

		ent = create_proc_entry("status", S_IFREG|S_IRUGO, provision_root_dir);
		ent->read_proc	= provision_proc_read_status;
		
		dent = proc_mkdir("partition", provision_root_dir);

		for (i = 0; i < ARRAY_SIZE(provs); i ++) 
		{
			ent = create_proc_entry(provs[i].name, S_IFREG|S_IRUGO|S_IWUSR, dent);
			if (!ent) {
				continue;
			}
			ent->read_proc	= provision_proc_read_mmc;
			ent->write_proc  = provision_proc_write_mmc;
			ent->data		= (void *)&provs[i];
		}
		ent = create_proc_entry("current", S_IFREG|S_IRUGO, provision_root_dir);
		if(ent)
		{
			ent->read_proc	= provision_proc_read_current;
			ent->write_proc  = provision_proc_write_current;
		}
		ent = create_proc_entry("lastupdate", S_IFREG|S_IRUGO, provision_root_dir);
		if(ent)
		{
			ent->read_proc	= provision_proc_read_lastupdate;
			ent->write_proc  = provision_proc_write_lastupdate;
		}
		ent = proc_create("partitions", S_IFREG|S_IRUGO, provision_root_dir, &proc_prov_partitions_operations);
	}

	return 0;
}

