/*
 * drivers/staging/gs/provision.c
 *
 * Driver for the provison.
 *
 * Copyright (C) 2012, 2013 GrandStream Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/delay.h>
#include <linux/rslib.h>
#include <linux/moduleparam.h>
#include <linux/platform_device.h>
#include <linux/dma-mapping.h>
#include <linux/types.h>
#include <linux/clk.h>
#include <linux/interrupt.h>
#include <linux/semaphore.h>
#include <linux/io.h>
#include <linux/slab.h>
#include <linux/ccu.h>
#include <linux/completion.h>
#include <linux/uaccess.h>
#include <linux/proc_fs.h>
#include <linux/vmalloc.h>

#include <gs_def.h>
#include <mach/gxp22xx.h>
#include "flash_rw.h"

#define ERASE_SIZE 0x20000
#define WRITE_SIZE 0x800

static prov_t provs[]=
{
	{SEG_UBOOT_STR, },
	{SEG_KERNEL_STR, },
	{SEG_SYSTEM_STR },
	{SEG_RECOVERY_STR },
};

/*
 * proc routine for mtd partitions
 */
static int
provision_proc_read_flash (
    char            *page,
    char            **start,
    off_t           off,
    int             count,
    int             *eof,
    void            *data)
{
	int len;
	prov_t *prov = (prov_t *)data;
	int ret = 0;
	int addr;
	seg_header_t *hdr = (seg_header_t *)page;
	int i;

	len = sizeof(seg_header_t);
	for(i=0; i<ARRAY_SIZE(prov->start); i++)
	{	
		addr = prov->start[i];
		if(!gxv31xx_internal_mtd_checkbad(addr))
		{
			break;
		}
		/* try the backup block */
		addr += ERASE_SIZE;
	}

	if(ARRAY_SIZE(prov->start)==i)
	{
		return (-ENOSPC);
	}
	ret = gxv31xx_internal_mtd_read(addr, len, (char *)hdr);

	if (hdr->signature != GS_PROD_SIGNATURE || ret != 0)
		memset(hdr, 0x0, len);

	//SXIE: do we need blank hw_model info to use same image for all the models?
	//p->reserved[0] = 0x0;

	if (len <= off+count) *eof = 1;
	*start = page + off;
	len -= off;
	if (len>count) len = count;
	if (len<0) len = 0;

	return (len);
}

static int
provision_proc_write_flash (
    struct file     *file,
    const char      *buffer,
    unsigned long   count,
    void            *data)
{
	prov_t *prov = (prov_t *)data;
	int len;
	int addr;
	seg_header_t *hdr;
	int i;

	if (count > WRITE_SIZE)	return (-EINVAL);

	len = sizeof(seg_header_t);
	for(i=0; i<ARRAY_SIZE(prov->start); i++)
	{	
		addr = prov->start[i];
		if(!gxv31xx_internal_mtd_checkbad(addr))
		{
			break;
		}
		/* try the backup block */
		addr += ERASE_SIZE;
	}

	if(ARRAY_SIZE(prov->start)==i)
	{
		return (-ENOSPC);
	}

	hdr = (seg_header_t *)vmalloc(WRITE_SIZE);
	if(!hdr)	return (-ENOMEM);

	memset(hdr, 0, WRITE_SIZE);
	if (copy_from_user(hdr, buffer, count))
	{
		vfree(hdr);
		return (-EFAULT);
	}

	/* SXIE: not needed if header is not using the entire block */
	gxv31xx_internal_mtd_erase((u_int32_t)addr, ERASE_SIZE);

	/* Write Flash */
	gxv31xx_internal_mtd_write((u_int32_t)addr, WRITE_SIZE, (char *)hdr);

	vfree(hdr);
	return (count);
}

int provision_install_proc_entry_flash (void)
{
	struct proc_dir_entry *provision_root_dir;
	struct proc_dir_entry   *ent;
	int i;

	for(i=0;i<ARRAY_SIZE(provs);i++)
	{
		provs[i].start[0] = MTD_OFFSET_SEGHDR0+i*SZ_128K;
		provs[i].start[1] = MTD_OFFSET_SEGHDR1+i*SZ_128K;
		provs[i].start[2] = MTD_OFFSET_SEGHDR2+i*SZ_128K;
		provs[i].start[3] = MTD_OFFSET_SEGHDR3+i*SZ_128K;
	}

	provision_root_dir = proc_mkdir( "provision", NULL);
	if (provision_root_dir != NULL)
	{
		struct proc_dir_entry	*dent;

		dent = proc_mkdir("partition", provision_root_dir);

		for (i = 0; i < ARRAY_SIZE(provs); i ++) 
		{
			ent = create_proc_entry(provs[i].name, S_IFREG|S_IRUGO, dent);
			if (!ent) {
				continue;
			}
			ent->read_proc	= provision_proc_read_flash;
			ent->write_proc  = provision_proc_write_flash;
			ent->data		= (void *)&provs[i];
		}
	}

	return 0;
}

