/*
 * drivers/staging/gs/provision.c
 *
 * Driver for the provison.
 *
 * Copyright (C) 2012, 2013 GrandStream Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/delay.h>
#include <linux/rslib.h>
#include <linux/moduleparam.h>
#include <linux/platform_device.h>
#include <linux/dma-mapping.h>
#include <linux/types.h>
#include <linux/clk.h>
#include <linux/interrupt.h>
#include <linux/semaphore.h>
#include <linux/io.h>
#include <linux/slab.h>
#include <linux/ccu.h>
#include <linux/completion.h>
#include <linux/uaccess.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/vmalloc.h>

#include "provision.h"
#include "gs_misc.h"

prov_t provs[PROV_COUNT]=
{
	{IMGHDR_BOOT_STR, {} , IMGHDR_BOOT_ID, "mmcblk0p1"},
	{IMGHDR_RCVR_STR, {} , IMGHDR_RCVR_ID, "mmcblk0p2" },
	{IMGHDR_CORE_STR, {}, IMGHDR_CORE_ID, "mmcblk0p3" },
	{IMGHDR_CORE_STR2, {}, IMGHDR_CORE_ID, "mmcblk0p5" },
	{IMGHDR_BASE_STR, {}, IMGHDR_BASE_ID, "mmcblk0p6" },
	{IMGHDR_BASE_STR2, {}, IMGHDR_BASE_ID, "mmcblk0p7"},
	{IMGHDR_PROG_STR, {}, IMGHDR_PROG_ID, "mmcblk0p8" },
	{IMGHDR_PROG_STR2, {}, IMGHDR_PROG_ID, "mmcblk0p9"},
	{IMGHDR_LANG_STR, {}, IMGHDR_LANG_ID, "mmcblk0p10"},
	{IMGHDR_LANG_STR2, {}, IMGHDR_LANG_ID, "mmcblk0p11"},
};

extern int provision_install_proc_entry_flash (void);
extern int provision_install_proc_entry_mmc (void);

static int
provision_proc_read_dummy(
    char            *page,
    char            **start,
    off_t           off,
    int             count,
    int             *eof,
    void            *data)
{
	return count;
}

static int
provision_proc_write_dummy (
    struct file     *file,
    const char      *buffer,
    unsigned long   count,
    void            *data)
{
	return count;	
}

static int prov_partitions_show(struct seq_file *f, void *v)
{
	int i = *(loff_t *) v;

	if (i < PROV_COUNT) {
		if (i == 0)
			seq_puts(f, "ID procfname devfname:\n");
		seq_printf(f, "%4d %11s %s\n", provs[i].id, provs[i].name, provs[i].devfname);
	}

	return 0;
}

static void *prov_partitions_start(struct seq_file *f, loff_t *pos)
{
	if (*pos < PROV_COUNT)
		return pos;
	return NULL;
}

static void *prov_partitions_next(struct seq_file *f, void *v, loff_t *pos)
{
	(*pos)++;
	if (*pos >= PROV_COUNT)
		return NULL;
	return pos;
}

static void prov_partitions_stop(struct seq_file *f, void *v)
{
	/* Nothing to do */
}

static const struct seq_operations prov_partitions_ops = {
	.start = prov_partitions_start,
	.next  = prov_partitions_next,
	.stop  = prov_partitions_stop,
	.show  = prov_partitions_show
};

static int prov_partitions_open(struct inode *inode, struct file *filp)
{
        return seq_open(filp, &prov_partitions_ops);
}

static const struct file_operations proc_prov_partitions_operations = {
        .open           = prov_partitions_open,
        .read           = seq_read,
        .llseek         = seq_lseek,
        .release        = seq_release,
};

static int provision_install_proc_entry_dummy (void)
{
	struct proc_dir_entry *provision_root_dir;
	struct proc_dir_entry   *ent;
	int i;

	provision_root_dir = proc_mkdir( "provision", NULL);
	if (provision_root_dir != NULL)
	{
		struct proc_dir_entry	*dent;

		dent = proc_mkdir("partition", provision_root_dir);

		for (i = 0; i < ARRAY_SIZE(provs); i ++) 
		{
			ent = create_proc_entry(provs[i].name, S_IFREG|S_IRUGO, dent);
			if (!ent) {
				continue;
			}
			ent->read_proc	= provision_proc_read_dummy;
			ent->write_proc  = provision_proc_write_dummy;
			ent->data		= (void *)&provs[i];
		}
		ent = create_proc_entry("current", S_IFREG|S_IRUGO, provision_root_dir);
		if(ent)
		{
			ent->read_proc	= provision_proc_read_dummy;
			ent->write_proc  = provision_proc_write_dummy;
		}
		ent = create_proc_entry("lastupdate", S_IFREG|S_IRUGO, provision_root_dir);
		if(ent)
		{
			ent->read_proc	= provision_proc_read_dummy;
			ent->write_proc  = provision_proc_write_dummy;
		}
		ent = proc_create("partitions", S_IFREG|S_IRUGO, provision_root_dir, &proc_prov_partitions_operations);
	}

	return 0;
}


static int provision_install_proc_entry (void)
{
	int rval = 0;
	int storage = gs_get_storage();

	switch(storage)
	{
	case GS_STORAGE_FLASH:
#ifdef CONFIG_GS_STORAGE_FLASH
		printk("provision flash storage\n");
		rval = provision_install_proc_entry_flash();
#endif
		break;
	case GS_STORAGE_EMMC:
#ifdef CONFIG_GS_STORAGE_EMMC
		printk("provision emmc storage\n");
		rval = provision_install_proc_entry_mmc();
#endif
		break;
	default:
		printk("provision dummy storage\n");
		rval = provision_install_proc_entry_dummy();
		break;
	}
	return rval;
}

static int __init
provision_init(void)
{
	provision_install_proc_entry();
	return 0;
}

static void __exit
provision_cleanup(void)
{
	remove_proc_entry("provision", NULL);
	return;
}

module_init(provision_init);
module_exit(provision_cleanup);

MODULE_AUTHOR("ygzhang@grandstream.com");
MODULE_DESCRIPTION("GS provision driver");
MODULE_LICENSE("GPL");
