/*
 * NVRAM variable manipulation (common)
 *
 * Copyright 2004, Broadcom Corporation
 * All Rights Reserved.
 * 
 * THIS SOFTWARE IS OFFERED "AS IS", AND BROADCOM GRANTS NO WARRANTIES OF ANY
 * KIND, EXPRESS OR IMPLIED, BY STATUTE, COMMUNICATION OR OTHERWISE. BROADCOM
 * SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A SPECIFIC PURPOSE OR NONINFRINGEMENT CONCERNING THIS SOFTWARE.
 *
 */

#include <linux/types.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/ctype.h>

#include "nvram.h"
#include "../mmc_rw.h"

#define NVRAM_MMCBLK_START	(32769)
#define NVRAM_MMCBLK_SECTORS	(1536)

static sector_t start_sector = NVRAM_MMCBLK_START;
static sector_t total_sectors = NVRAM_MMCBLK_SECTORS;
static sector_t current_sector = NVRAM_MMCBLK_START;
static unsigned long current_write_counter = 0;

int nvram_mmc_write_zbuf(char *zbuf, u_int32_t len)
{
	int rval;
	struct nvram_header *flash_header = (struct nvram_header *) zbuf;
	sector_t unit_sectors;

	if (len != NVRAM_SPACE) //just sanity check
		printk("!!!! nvram_flash_read wrong size, 0x%x\n", len);

	current_write_counter++;
	flash_header->write_counter = current_write_counter;
	
	unit_sectors = NVRAM_SPACE>>9;
	current_sector += unit_sectors;
	if(current_sector+unit_sectors>start_sector+total_sectors)
	{
		current_sector = start_sector;
	}
	rval = emmc_write(current_sector, unit_sectors, zbuf, len);
	if(rval<0)
	{
		printk("nvram part[%llu] write err %d\n", (current_sector-start_sector)/unit_sectors, rval);
	}
	else
	{
		printk("nvram part[%llu] write success\n", (current_sector-start_sector)/unit_sectors);
	}
	return rval;
}

#ifdef DEBUG_SHOW_HDR
static int show_header(struct nvram_header *flash_header)
{
	printk("magic: %.4s\n", (char *)&flash_header->magic);
	printk("len: %d\n", flash_header->len);
	printk("space_used: %d\n", flash_header->space_used);
	printk("len_compressed: %d\n", flash_header->len_compressed);
	printk("chksum_protected: 0x%04x\n", flash_header->chksum_protected);
	printk("chksum: 0x%04x\n", flash_header->chksum);
	printk("write_counter: %lu\n", flash_header->write_counter);
	printk("reserved: %02x%02x%02x%02x\n", flash_header->reserved[0], flash_header->reserved[1],
		flash_header->reserved[2], flash_header->reserved[3]);
	return 0;
}
#endif

int nvram_mmc_read_zbuf(char *zbuf, u_int32_t len)
{
	int rval;
	struct nvram_header *flash_header = (struct nvram_header *) zbuf;
	sector_t off;
	sector_t unit_sectors = NVRAM_SPACE>>9;

	if (len != NVRAM_SPACE) //just sanity check
		printk("!!!! nvram_flash_read wrong size, 0x%x\n", len);

	current_write_counter = 0;
	printk("scan nvram data blocks\n");
	for(off=0; (off+unit_sectors)<=total_sectors; off+=unit_sectors)
	{
		flash_header->magic = 0;
		rval = emmc_read(start_sector+off, unit_sectors, zbuf, len);
		if(rval<0)
		{
			printk("part [%llu] read err %d\n", off/unit_sectors, rval);
			continue;
		}
		if (flash_header->magic != NVRAM_MAGIC || verify_zbuf_chksum(zbuf))
		{
			/* nvram data invalid */
			continue;
		}
#ifdef DEBUG_SHOW_HDR
		show_header(flash_header);
#endif
		if(current_write_counter<=flash_header->write_counter)
		{
			current_write_counter = flash_header->write_counter;
			current_sector = start_sector+off;
		}
	}

	rval = emmc_read(current_sector, unit_sectors, zbuf, len);
	if(rval<0)
	{
		return -1;
	}
	printk("active nvram part[%llu], write=%ld\n", (current_sector-start_sector)/unit_sectors, flash_header->write_counter);
	return 0;
}

int proc_nvram_mmc_get_status(char *buf, char **start, off_t offset,
				     int count, int *eof, void *data)
{
    int len = 0;

    if (len <= count)
        len += sprintf(buf + len, "%-35s: %s\n", "Storage", "eMMC"); 
    if (len <= count)
        len += sprintf(buf + len, "%-35s: %llu\n", "Total Sectors", total_sectors); 
    if (len <= count)
        len += sprintf(buf + len, "%-35s: %llu\n\n", "Total Parts Valid", total_sectors/(NVRAM_SPACE>>9));
    if (len <= count)
        len += sprintf(buf + len, "%-35s: %u\n\n", "Part Sectors", (NVRAM_SPACE>>9));

    if (len <= count)
        len += sprintf(buf + len, "%-35s: %llu\n", "Active Part", (current_sector-start_sector)/(NVRAM_SPACE>>9)); 
    if (len <= count)
        len += sprintf(buf + len, "%-35s: %lu\n", "Write Counter", current_write_counter); 

    if (len <= offset + count) *eof = 1;
    *start = buf + offset;
    len -= offset;
    if (len > count) len = count;
    if (len < 0) len = 0;

    return (len);
}

