/*
 * NVRAM variable manipulation (Linux kernel half)
 *
 * Copyright 2004, Broadcom Corporation
 * All Rights Reserved.
 * 
 * THIS SOFTWARE IS OFFERED "AS IS", AND BROADCOM GRANTS NO WARRANTIES OF ANY
 * KIND, EXPRESS OR IMPLIED, BY STATUTE, COMMUNICATION OR OTHERWISE. BROADCOM
 * SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A SPECIFIC PURPOSE OR NONINFRINGEMENT CONCERNING THIS SOFTWARE.
 *
 * $Id: nvram_linux.c,v 1.1 2012/04/09 18:06:51 pradev Exp $
 *
 * SXIE - added ioctl to erase all tuples
 * SCF - rewrote _nvram_read and nvram_commit from scratch to support zlib compression
 *     - modified _nvram_free to correctly track nvram_size
 *     - modified _nvram_realloc to allow hash table to grow dynamically to any size up
       -          to max avail kmalloc'able memory
 */
#include <linux/init.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/fs.h>       /* everything... */
#include <linux/proc_fs.h>
#include <linux/device.h>
#include <linux/cdev.h>     /* Used for struct cdev */
#include <linux/string.h>
#include <linux/interrupt.h>
#include <linux/spinlock.h>
#include <linux/slab.h>
#include <linux/bootmem.h>
#include <linux/fs.h>
#include <linux/zlib.h>
#include <linux/vmalloc.h>
#include <linux/mm.h>
//#include <asm/addrspace.h>
#include <asm/io.h>
#include <asm/uaccess.h>

#include "nvram.h"
#include <mach/gxp22xx.h>
#include "nvram_flash.h"
#include "nvram_mmc.h"

/* Defines */
#define DEVNO_NVRAM_READ_SAFE       1
#define DEVNO_NVRAM_READ_UNSAFE     0

/* In BSS to minimize text size and page aligned so it can be mmap()-ed */
static char nvram_buf[NVRAM_BUFFER_SPACE] __attribute__((aligned(PAGE_SIZE)));

extern char * _nvram_get(const char *name);
extern int _nvram_set(const char *name, const char *value, int set_dirty_flag);
extern int _nvram_unset(const char *name);
extern int _nvram_getall(char *buf, int count);
extern int _nvram_commit(struct nvram_header *header);
extern int _nvram_init(void);
extern int _nvram_erase_hushtable(void);
extern int _nvram_purge_unlocked(void);
extern void _nvram_exit(void);
extern int _nvram_setacl(const char *name);
extern int _nvram_savetemp(struct nvram_header *header);
extern int _nvram_loadtemp(struct nvram_header *header);

/* Globals */
static spinlock_t nvram_lock = __ARCH_SPIN_LOCK_UNLOCKED;
static struct semaphore nvram_sem;
static unsigned long nvram_offset = 0;

unsigned int  nvram_dirty = 0;

static dev_t dev = MKDEV(NVRAM_DEV_MAJOR, NVRAM_DEV_MINOR);
static struct cdev cdev;
static struct class *nvram_class = NULL;

static int nvram_major = -1;

static int nvram_space_used = 0;

static unsigned short calc_zbuf_chksum(int size, char *p)
{
    unsigned short sum = 0;
    unsigned short checksum = 0;
    int i;

    size = ((size + 1)>>1) << 1;
    for (i = 0; i < size; i+=2)
    {
        sum += (p[i] << 8 | p[i+1]);    // fact config use be for checksum calc
    }

    checksum = 0xffff - sum + 0x1;

    return (checksum >> 8 | checksum << 8); // be

}
static unsigned short apply_zbuf_chksum(struct nvram_header *header, char *zbuf)
{
    header->chksum = calc_zbuf_chksum(header->len_compressed, zbuf);
    return 0;
}
unsigned short verify_zbuf_chksum(char *zbuf_with_header)
{
    struct nvram_header *header = (struct nvram_header*)zbuf_with_header;
    return (header->chksum == calc_zbuf_chksum(header->len_compressed, zbuf_with_header + NVRAM_HEADER_SIZE)? 0 : 1);
}

/* return NULL pointer on failure */
struct nvram_tuple *
_nvram_realloc(struct nvram_tuple *t, const char *name, const char *value, int *value_changed)
{
    if ((nvram_offset + strlen(value) + 1) > NVRAM_BUFFER_SPACE)
    {
        /* We are run out of buffer space */
        return NULL;
    }

	if (t == NULL) {
        int size_to_allocate = sizeof(struct nvram_tuple) + strlen(name) + 1;

		if (!(t = kmalloc(size_to_allocate, GFP_KERNEL)))
			return NULL;

        /* Copy name */ /* name is kmalloc-ed */
		t->name = (char *) &t[1];
		strcpy(t->name, name);

        t->flag = 0;
		t->value = NULL;
        if (value_changed != NULL)
            *value_changed = 1;
	}

        /* Copy value */
	if (!t->value || strlen(t->value) != strlen(value) || strcmp(t->value, value)) {
		if ( t->value && ( strlen(value) <= strlen(t->value) )) {
			strcpy(t->value, value);
		}
		else {
            /* SXIE: we will get unused fragment here in the size not enough case!!! */
			t->value = &nvram_buf[nvram_offset];
			strcpy(t->value, value);
			nvram_offset += strlen(value) + 1;
		}

        if (value_changed != NULL)
            *value_changed = 1;
	}

	return t;
}

void
_nvram_free(struct nvram_tuple *t)
{
	if (!t)
		nvram_offset = 0;	
	else 
		kfree(t);
}

int
nvram_set(const char *name, const char *value)
{
	unsigned long flags;
	int ret;
	struct nvram_header *header;

	spin_lock_irqsave(&nvram_lock, flags);
	if ((ret = _nvram_set(name, value, 1)) == -ENOMEM) {
		/* Consolidate space and try again */
		if ((header = kmalloc(NVRAM_BUFFER_SPACE, GFP_ATOMIC))) {
			if (_nvram_commit(header) == 0)
				ret = _nvram_set(name, value, 1);
			kfree(header);
		}
	}
	spin_unlock_irqrestore(&nvram_lock, flags);

	return ret;
}

char *
real_nvram_get(const char *name)
{
	unsigned long flags;
	char *value;

	spin_lock_irqsave(&nvram_lock, flags);
	value = _nvram_get(name);
	spin_unlock_irqrestore(&nvram_lock, flags);

	return value;
}

char *
nvram_get(const char *name)
{
	if (nvram_major >= 0)
		return real_nvram_get(name);
	else {
		//this should not happen
		printk("NO NVRAM_EARLY_ACCESS DEFINED");
		return (char *)0;
	}
}

int
nvram_unset(const char *name)
{
	unsigned long flags;
	int ret;

	spin_lock_irqsave(&nvram_lock, flags);
	ret = _nvram_unset(name);
	spin_unlock_irqrestore(&nvram_lock, flags);

	return ret;
}

int
nvram_setacl(const char *name)
{
	unsigned long flags;
	int ret;

	spin_lock_irqsave(&nvram_lock, flags);
	ret = _nvram_setacl(name);
	spin_unlock_irqrestore(&nvram_lock, flags);

	return ret;
}

static int _nvram_write_zbuf(char *zbuf, u_int32_t len)
{
	int rval = 0;
	int storage = gs_get_storage();

	switch(storage)
	{
	case GS_STORAGE_FLASH:
#ifdef CONFIG_GS_STORAGE_FLASH
		rval = nvram_flash_write_zbuf(zbuf, len);
#endif
		break;
	case GS_STORAGE_EMMC:
#ifdef CONFIG_GS_STORAGE_EMMC
		rval = nvram_mmc_write_zbuf(zbuf, len);
#endif
		break;
	default:
		break;
	}
	return rval;
}


static int _nvram_read_zbuf(char *zbuf, u_int32_t len)
{
	int rval = 0;
	int storage = gs_get_storage();

	switch(storage)
	{
	case GS_STORAGE_FLASH:
#ifdef CONFIG_GS_STORAGE_FLASH
		rval = nvram_flash_read_zbuf(zbuf, len);
#endif
		break;
	case GS_STORAGE_EMMC:
#ifdef CONFIG_GS_STORAGE_EMMC
		rval = nvram_mmc_read_zbuf(zbuf, len);
#endif
		break;
	default:
		break;
	}
	return rval;
}

/* read all of nvram data from flash into buffer */
int
_nvram_read(char *buf)
{
	int ret = 0;
	char *zbuf = NULL;
	z_stream zlib_stream;
    struct nvram_header *flash_header;
	int header_offset = NVRAM_HEADER_SIZE;
    //u_int32_t nvram_flash_addr;

	do {
		if (!(zbuf = vmalloc(NVRAM_SPACE))) {
			ret = -ENOMEM;
			break;
		}
		
        memset(zbuf, 0x0, NVRAM_SPACE);

	_nvram_read_zbuf(zbuf, NVRAM_SPACE);

        /* Check magic number */
        flash_header = (struct nvram_header *) zbuf;
        {
            if (flash_header->magic != NVRAM_MAGIC)
            {
                printk("Invalid nvram data\n");
                break;
            }
        }

        /* verify chksum */
        if (((struct nvram_header *)zbuf)->chksum_protected)
        {
            if (verify_zbuf_chksum(zbuf))
            {
                printk("nvram_init: invalid chksum, corrupt nvram detected.\n");
                break;
            }
        }
        else
        {
            /* keep it back compatible */
            flash_header->len_compressed = NVRAM_SPACE - NVRAM_HEADER_SIZE;
        }
		
        nvram_space_used = flash_header->space_used;
        /* copy header */
		memcpy(buf, zbuf, NVRAM_HEADER_SIZE);

		zlib_stream.workspace = vmalloc(zlib_inflate_workspacesize());
		if (!(zlib_stream.workspace)) {
			ret = -ENOMEM;
			break;
		}
		
		zlib_stream.next_in = (unsigned char*)zbuf+NVRAM_HEADER_SIZE;
		zlib_stream.avail_in = ((struct nvram_header *)zbuf)->len_compressed; //NVRAM_SPACE-NVRAM_HEADER_SIZE;
		zlib_stream.next_out = (unsigned char*)buf + header_offset; //p_inflate_stream->data;
		zlib_stream.avail_out = NVRAM_BUFFER_SPACE - header_offset;

		zlib_inflateInit(&zlib_stream);
		if (zlib_inflateReset(&zlib_stream)!=Z_OK) { //necessary???
			ret = -ENODATA;
			break;
		}
		
		/* decompress zbuf into NVRAM_SPACE size chunks stored in linked list */
        ret = zlib_inflate(&zlib_stream, Z_SYNC_FLUSH);
        printk("Inflate: %d, total_out=%ld\n", ret, zlib_stream.total_out);
        
        zlib_inflateEnd(&zlib_stream);
        if (zlib_stream.workspace) vfree(zlib_stream.workspace);

		if (ret!=Z_STREAM_END) {
			printk("nvram_init: corrupt nvram detected when decompress\n");
            memset(buf + header_offset, 0x0, NVRAM_BUFFER_SPACE - header_offset);
			break; /* premature inflate loop termination */
		}
        printk("nvram: %d -> %d (%d%% -> %d%%)\n", 
                    flash_header->len_compressed,
                    flash_header->len,
                    (flash_header->len_compressed * 100) / NVRAM_SPACE,
                    (flash_header->len * 100) / NVRAM_BUFFER_SPACE
                    );

		ret = 0;
	
	} while (0);
	
	if (zbuf) vfree(zbuf);
	
	return ret;	
}

int
nvram_renew(void)
{
	struct nvram_header *header = NULL;

    /* this is specially for refresh nvram variables after resume */
	if ((header = kmalloc(NVRAM_BUFFER_SPACE, GFP_ATOMIC))) {
        /* save memory saved variables */
        _nvram_savetemp(header);
    }

    /* reload flash block for flash saved settings */
    _nvram_init();

    if (header)
    {
        _nvram_loadtemp(header);
	    kfree(header);
    }
    return 0;
}

int
nvram_commit(void)
{
	char *buf;
	char *zbuf;
	int ret;
	struct nvram_header *header;
	unsigned long flags;
	u_int32_t offset;
	z_stream zlib_stream;

	if (in_interrupt()) {
		//printk("nvram_commit: not committing in interrupt\n");
		return -EINVAL;
	}

	/* buffer for uncompressed data */
	if (!(buf = vmalloc(NVRAM_BUFFER_SPACE))) {
		return -ENOMEM;
	}
	
	/* buffer for compressed data */
	if (!(zbuf = vmalloc(NVRAM_SPACE-NVRAM_HEADER_SIZE))) {
		vfree(buf);
		return -ENOMEM;
	}
	memset(zbuf, 0, NVRAM_SPACE-NVRAM_HEADER_SIZE);

	down(&nvram_sem);
		
	offset = 0;
	header = (struct nvram_header *)buf;
	
	/* Regenerate NVRAM */
	spin_lock_irqsave(&nvram_lock, flags);
	ret = _nvram_commit(header); /* create copy of nvram data in header */
	spin_unlock_irqrestore(&nvram_lock, flags);
	
	nvram_space_used = header->space_used = (100*(header->len))/(NVRAM_BUFFER_SPACE-NVRAM_HEADER_SIZE);

	if (ret || !nvram_dirty) {
		//printk("nvram_commit: commit abort, ret=%x, dirty=%x\n", ret, nvram_dirty);
		up(&nvram_sem);
		vfree(buf);
		vfree(zbuf);
		return 0;
	}

	/* zlib compress buf */
    zlib_stream.workspace = vmalloc(zlib_deflate_workspacesize(MAX_WBITS, DEF_MEM_LEVEL));
    if (!zlib_stream.workspace) {
        ret = -ENOMEM;
        goto done;
    }

    zlib_stream.next_in = (unsigned char *)buf+NVRAM_HEADER_SIZE;
    zlib_stream.avail_in = header->len;
    zlib_stream.next_out = (unsigned char *)zbuf + NVRAM_HEADER_SIZE;
    zlib_stream.avail_out = NVRAM_SPACE-NVRAM_HEADER_SIZE;
	zlib_deflateInit(&zlib_stream, 9);
    zlib_deflateReset(&zlib_stream);

	if (zlib_deflate(&zlib_stream, Z_FINISH) != Z_STREAM_END) {
		ret = -EFBIG;
		goto done;
	}

	header->magic = NVRAM_MAGIC;	
    /* SXIE: we record the bigger one between compressed and uncompressed */
	if (header->space_used < (100*(zlib_stream.total_out))/(NVRAM_SPACE-NVRAM_HEADER_SIZE))
	    nvram_space_used = header->space_used = (100*(zlib_stream.total_out))/(NVRAM_SPACE-NVRAM_HEADER_SIZE);
    /* len_compressed is needed for calculate checksum below */
    header->len_compressed = zlib_stream.total_out;
	
    apply_zbuf_chksum((struct nvram_header*)buf,zbuf + NVRAM_HEADER_SIZE);
    header->chksum_protected = 1;

    /* add header to zbuf */
    memcpy(zbuf, header, NVRAM_HEADER_SIZE);

    _nvram_write_zbuf(zbuf, NVRAM_SPACE);

	ret = header->len; //success

	nvram_dirty = 0;
	
done:
	up(&nvram_sem);
	zlib_deflateEnd(&zlib_stream);
	if (zlib_stream.workspace) vfree(zlib_stream.workspace);
	vfree(zbuf);
	vfree(buf);
	return ret;
}

int
nvram_getall(char *buf, int count)
{
	unsigned long flags;
	int ret;

	spin_lock_irqsave(&nvram_lock, flags);
	ret = _nvram_getall(buf, count);
	spin_unlock_irqrestore(&nvram_lock, flags);

	return ret;
}

int
nvram_eraseall(void)
{
	unsigned long flags;
	int ret;

	spin_lock_irqsave(&nvram_lock, flags);
	ret = _nvram_purge_unlocked();
	nvram_dirty = 1;
	spin_unlock_irqrestore(&nvram_lock, flags);

	return ret;
}

EXPORT_SYMBOL(nvram_get);
EXPORT_SYMBOL(nvram_getall);
EXPORT_SYMBOL(nvram_eraseall);
EXPORT_SYMBOL(nvram_set);
EXPORT_SYMBOL(nvram_unset);
EXPORT_SYMBOL(nvram_commit);
EXPORT_SYMBOL(nvram_renew);

/* User mode interface below */
//=============================================================================
static ssize_t
dev_nvram_read_safe(
    struct file * file,
    char * buf,
    size_t count,
    loff_t * ppos
)
//=============================================================================
{
    char tmp[2050], *name = tmp, *value;
    ssize_t ret;
    unsigned long off;
    char *saved_name_p;
    int len;
    
    if (count > sizeof(tmp))
    {
        if (!(name = vmalloc(count)))
            return -ENOMEM;
    }

    saved_name_p = name;

    if (copy_from_user(name, buf, count))
    {
        ret = -EFAULT;
        goto done;
    }

    /* sanity check input */
    if ( strnlen( name, count ) == count ) 
    {
        ret = -EINVAL;
        goto done;
    }

    if (*name == '\0')
    {
        /* Get all variables */
        ret = nvram_getall(name, count);
        if (ret == 0)
        {
            if (copy_to_user(buf, name, count))
            {
                ret = -EFAULT;
                goto done;
            }
            ret = count;
        }
    } 
    else 
    {
        //mutex_lock( &nvram_mutex );
        if (!(value = nvram_get(name))) 
        {
            ret = 0;
            goto unlock_done;
        }

        /* Provide the offset into mmap() space */
        off = (unsigned long) value - (unsigned long) nvram_buf;

        len = strnlen( &nvram_buf[ off ], count );
        if ( len == count )
        {
            ret = -ENOSPC;
            goto unlock_done;
        }

        if ( copy_to_user( buf, &nvram_buf[ off ], len+1 ) != 0 )
        {
            ret = -EFAULT;
            goto unlock_done;
        }
        
        ret = len+1;
    }

unlock_done:
    //mutex_unlock( &nvram_mutex );
done:
    if (saved_name_p != tmp)
        vfree(saved_name_p);

    return ret;
}

static ssize_t
dev_nvram_read_unsafe(
    struct file * file,
    char * buf,
    size_t count,
    loff_t * ppos
)
{
	char tmp[100], *name = tmp, *value;
	ssize_t ret;
	unsigned long off;
	char *saved_name_p;

	if (count > sizeof(tmp))
	{
		if (!(name = vmalloc(count)))
			return -ENOMEM;
	}

	saved_name_p = name;

	if (copy_from_user(name, buf, count))
	{
		ret = -EFAULT;
		goto done;
	}


	/* SCF - check for NULL termination prior to calling strXXX functions */
	if (name[count-1]!='\0') {
		ret = -EINVAL;
		goto done;
	}

	if (*name == '\0')
	{
		/* Get all variables */
		ret = nvram_getall(name, count);
		if (ret == 0)
		{
			if (copy_to_user(buf, name, count))
			{
				ret = -EFAULT;
				goto done;
			}
			ret = count;
		}
	} 
	else 
	{
		if (!(value = nvram_get(name))) 
		{
			ret = 0;
			goto done;
		}

		/* Provide the offset into mmap() space */
		off = (unsigned long) value - (unsigned long) nvram_buf;
	
		if (put_user(off, (unsigned long *) buf)) 
		{
			ret = -EFAULT;
			goto done;
		}
		
		ret = sizeof(unsigned long);
	}

	//SXIE: flush_cache_all();	
 
done:
	if (saved_name_p != tmp)
		vfree(saved_name_p);

	return ret;
}

static ssize_t
dev_nvram_read(struct file *file, char *buf, size_t count, loff_t *ppos)
{
    int devno;
    
    devno = iminor( file->f_dentry->d_inode );
    
    switch ( devno )
    {
        case DEVNO_NVRAM_READ_UNSAFE:
            return dev_nvram_read_unsafe( file, buf, count, ppos );
        case DEVNO_NVRAM_READ_SAFE:
            return dev_nvram_read_safe( file, buf, count, ppos );
    }
    
    return -ENODEV;
}

static ssize_t
dev_nvram_write(struct file *file, const char *buf, size_t count, loff_t *ppos)
{
	char tmp[100], *name = tmp, *value;
	ssize_t ret;

	if (count > sizeof(tmp)) {
		if (!(name = kmalloc(count, GFP_KERNEL)))
			return -ENOMEM;
	}

	if (copy_from_user(name, buf, count)) {
		ret = -EFAULT;
		goto done;
	}

	/* SCF - check for NULL termination, otherwise assume malformed input and bail */
	if (name[count-1]!='\0') {
        printk("Name/Value is invalid\n");
		ret = -EINVAL;
		goto done;
	}

	value = name;
	name = strsep(&value, "=");
    if (name[0] == MARK_L1LOCKED || (name[0] == MARK_L2LOCKED))
    {
        if (value)
        {
            //printk("set value with locked api\n");
            ret = nvram_set(name, value) ? : count;
        }
        else 
        {
            //printk("set locked level\n");
            ret = nvram_setacl(name) ? : count;
        }
    }
    else
    {
        if (value)
        {
            ret = nvram_set(name, value) ? : count;
        }
        else 
        {
            ret = nvram_unset(name) ? : count;
        }
    }
		
 done:
	if (name != tmp)
		kfree(name);

	return ret;
}	

int nvram_reload(void)
{
	//_nvram_init(); by dign, need add lock?
    return 0;
}

static int
dev_nvram_ioctl(struct inode *inode, struct file *file, unsigned int cmd, unsigned long arg)
{
	if (cmd == NVRAM_MAGIC)
    {
		return nvram_commit();
    }
    else if (cmd == NVRAM_SPACE_FREE)
	{
        return nvram_space_used;
	}
    else if (cmd == NVRAM_ERASEALL)
    {
		return nvram_eraseall();
    }
	else if (cmd == NVRAM_RELOAD)
	{
		printk("nvram reflash\n");
		return nvram_reload();
	}
	else
		return -EINVAL;
}

static long
dev_nvram_unlocked_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	return dev_nvram_ioctl(NULL, file, cmd, arg);
}

static int
dev_nvram_mmap(struct file *file, struct vm_area_struct *vma)
{
	unsigned long offset = page_to_pfn( virt_to_page( nvram_buf ) );

    //vma->vm_flags |= (VM_RESERVED | VM_LOCKED | VM_SHM);

	if (remap_pfn_range(vma, vma->vm_start, offset, vma->vm_end - vma->vm_start,
			     vma->vm_page_prot))
		return -EAGAIN;

	return 0;
}

static int
dev_nvram_open(struct inode *inode, struct file * file)
{
	//MOD_INC_USE_COUNT;
	return 0;
}

static int
dev_nvram_release(struct inode *inode, struct file * file)
{
	//MOD_DEC_USE_COUNT;
	return 0;
}

static int proc_nvram_get_status(char *buf, char **start, off_t offset,
				     int count, int *eof, void *data)
{
	int rval = 0;
	int storage = gs_get_storage();

	switch(storage)
	{
	case GS_STORAGE_FLASH:
#ifdef CONFIG_GS_STORAGE_FLASH
		rval = proc_nvram_flash_get_status(buf, start, offset, count, eof, data);
#endif
		break;
	case GS_STORAGE_EMMC:
#ifdef CONFIG_GS_STORAGE_EMMC
		rval = proc_nvram_mmc_get_status(buf, start, offset, count, eof, data);
#endif
	default:
		break;
	}
	return rval;
}

static struct file_operations dev_nvram_fops = {
	owner:		THIS_MODULE,
	open:		dev_nvram_open,
	release:	dev_nvram_release,
	read:		dev_nvram_read,
	write:		dev_nvram_write,
	unlocked_ioctl:		dev_nvram_unlocked_ioctl,
	mmap:		dev_nvram_mmap,
};

static void
dev_nvram_exit(void)
{
	int order = 0;
	struct page  *end;

	device_destroy(nvram_class, MKDEV(NVRAM_DEV_MAJOR, 0));

    /* destroy simple class */
    class_destroy(nvram_class);

    cdev_del(&cdev);

    /* unregistering the driver from the kernel */
    unregister_chrdev(MAJOR(dev), "nvram");

    while ((PAGE_SIZE << order) < NVRAM_SPACE)
		order++;
	end = virt_to_page(nvram_buf + (PAGE_SIZE << order) - 1);
	
	_nvram_exit();
}

static int __init
dev_nvram_init(void)
{
	int order = 0, ret = 0;
	struct page *end;
    int result;

    /* initialize cdev with file operations */
    cdev_init(&cdev, &dev_nvram_fops);

    cdev.owner = THIS_MODULE;
    cdev.ops = &dev_nvram_fops;

    /* add cdev to the kernel */
    result = cdev_add(&cdev, dev, 1);

    if (result) {
        goto err;
    }

	/* Register char device */
	if ((nvram_major = register_chrdev(NVRAM_DEV_MAJOR, "nvram", &dev_nvram_fops)) < 0) {
		ret = nvram_major;
		goto err;
	}

    nvram_class = class_create(THIS_MODULE, "nvram");
    if (!nvram_class) {
        unregister_chrdev_region(dev, 1);
        unregister_chrdev(MAJOR(dev), "nvram");
        cdev_del(&cdev);
        goto err;
    }

	device_create(nvram_class, NULL, MKDEV(NVRAM_DEV_MAJOR, DEVNO_NVRAM_READ_UNSAFE), NULL, "nvram");

	device_create(nvram_class, NULL, MKDEV(NVRAM_DEV_MAJOR, DEVNO_NVRAM_READ_SAFE), NULL, "nvram_safe");
	/* Allocate and reserve memory to mmap() */
	while ((PAGE_SIZE << order) < NVRAM_BUFFER_SPACE)
		order++;
	end = virt_to_page(nvram_buf + (PAGE_SIZE << order) - 1);

	/* Initialize hash table lock */
	spin_lock_init(&nvram_lock);

	/* Initialize commit semaphore */
	sema_init(&nvram_sem, 1);

	/* Initialize hash table */
	_nvram_init();

	create_proc_read_entry("nvram_status", 0, NULL,
				       proc_nvram_get_status, NULL);
	printk(KERN_NOTICE "NVRAM Driver Initialized\n");

	return 0;

 err:
	printk(KERN_NOTICE "NVRAM Driver failed to initialize\n");
	dev_nvram_exit();
	return ret;
}


module_init(dev_nvram_init);
module_exit(dev_nvram_exit);
