/*
 * NVRAM variable manipulation (common)
 *
 * Copyright 2004, Broadcom Corporation
 * All Rights Reserved.
 * 
 * THIS SOFTWARE IS OFFERED "AS IS", AND BROADCOM GRANTS NO WARRANTIES OF ANY
 * KIND, EXPRESS OR IMPLIED, BY STATUTE, COMMUNICATION OR OTHERWISE. BROADCOM
 * SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A SPECIFIC PURPOSE OR NONINFRINGEMENT CONCERNING THIS SOFTWARE.
 *
 */

#include <linux/types.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/ctype.h>
#include <linux/vmalloc.h>
#include <linux/bio.h>
#include <linux/blkdev.h>
#include <linux/scatterlist.h>

#include "nvram.h"
#include <mach/gxp22xx.h>
#include "../flash_rw.h"

static unsigned int current_block_index = 0;
static unsigned int current_part_index = 0;
static unsigned long current_write_counter = 0;
static unsigned int total_valid_parts = 0;

struct nvram_flash_block nvram_flash_blocks[] =
{
    { MTD_OFFSET_NVRAM, 8 },
    { MTD_OFFSET_NVRAM+0x20000, 8 },
    { MTD_OFFSET_NVRAM+0x40000, 8 },
};
#define NVRAM_FLASH_BLOCK_COUNT ((sizeof(nvram_flash_blocks))/(sizeof(struct nvram_flash_block)))

#define NVRAM_MINIMUM_GOOD_BLOCKS   2   // for reduncy

static int 
nvram_flash_is_all_bad(void)
{
    int ii;
    int good_block = 0;

    /* let's make sure we have two blocks for redundency, so recover when we have less than 2 good blocks */
    for (ii = 0; ii < NVRAM_FLASH_BLOCK_COUNT; ii ++)
    {
        if (gxv31xx_internal_mtd_checkbad(nvram_flash_blocks[ii].start_addr))
            continue;
        good_block ++;
    }

    if (good_block < NVRAM_MINIMUM_GOOD_BLOCKS)
        return 1;

    return 0;
}

/*
 * This read will be called only once when nvram init. 
 * Normal read error does not really indicate block is bad, so we will not mark block bad here
 */
int nvram_flash_read_zbuf(char *zbuf, u_int32_t len)
{
    struct nvram_header *flash_header = (struct nvram_header *) zbuf;
    int block_read_err;
    int ii, jj;

    if (len != NVRAM_SPACE) //just sanity check
        printk("!!!! nvram_flash_read wrong size, 0x%x\n", len);

    current_write_counter = 0;
    printk("scan nvram data blocks\n");
    /* scan all blocks to locate current using block and part */
    for (ii = 0; ii < NVRAM_FLASH_BLOCK_COUNT; ii ++)
    {
        /* we scan part even bad blocks, to maximum extension to get setting back */
        if (!gxv31xx_internal_mtd_checkbad(nvram_flash_blocks[ii].start_addr))
        {
            /* increase good parts for writing */
            total_valid_parts += nvram_flash_blocks[ii].part_count;
        }

        block_read_err = 0; 

        for (jj = 0; jj < nvram_flash_blocks[ii].part_count; jj ++)
        {
            if (gxv31xx_internal_mtd_read(nvram_flash_blocks[ii].start_addr + jj * NVRAM_SPACE, NVRAM_SPACE, zbuf))
            {
                block_read_err ++; 
                printk("part [%d:%d] ecc failed\n", ii, jj);
            }
            else
            {
                if (flash_header->magic != NVRAM_MAGIC || verify_zbuf_chksum(zbuf))
                {
                    /* nvram data invalid */
                    continue;
                }
                else
                {
                    if (current_write_counter <= flash_header->write_counter) /* use = here is for reading old version  nvram data */
                    {
                        current_write_counter = flash_header->write_counter;
                        current_block_index = ii;
                        current_part_index = jj;
                    }
                }
            }
        }
    }
    printk("total valid parts=%d\n", total_valid_parts);
    
    if (gxv31xx_internal_mtd_read(nvram_flash_blocks[current_block_index].start_addr + current_part_index * NVRAM_SPACE, NVRAM_SPACE, zbuf))
    {
        /* Will it fail in second read? */
        return -1;
    }
    printk("active nvram part[%d:%d], write=%ld\n", current_block_index, current_part_index, flash_header->write_counter);
    return 0;
}

static int 
nvram_flash_get_part_for_write(int *block_index, int *part_index)
{
    int ii;

    *part_index += 1;
    if (*part_index == nvram_flash_blocks[*block_index].part_count)
    {
        *block_index += 1;
        *part_index = 0;
    }

    for (ii = 0; ii < NVRAM_FLASH_BLOCK_COUNT; ii ++)
    {
        if (*block_index == NVRAM_FLASH_BLOCK_COUNT)
            *block_index = 0;
        if (gxv31xx_internal_mtd_checkbad(nvram_flash_blocks[*block_index].start_addr))
        {
            /* try next block, first part */
            *block_index += 1;
            *part_index = 0;
            continue;
        }
        break;
    }
    if (ii == NVRAM_FLASH_BLOCK_COUNT)
    {
        return -1;
    }
    return 0;
}

/*
 * We only erase block once but will write multiple times. After each write we will read it out to check 
 * whether the part is able be read out correctly. If we found one write bad we will write it again to next
 * part.
 * Note we do not mark block bad any more unless all parts are not able to write
 */
int nvram_flash_write_zbuf(char *zbuf, u_int32_t len)
{
    int ii;
    int block_index_write = current_block_index;
    int part_index_write = current_part_index;
    char *tmpbuf = NULL;
    int ret = 0;
    int write_fail = 0;

    if (len != NVRAM_SPACE) //just sanity check
        printk("!!!! nvram_flash_write wrong size, 0x%x\n", len);

	current_write_counter++;

    if (nvram_flash_is_all_bad())
    {
        /* all blocks are marked as bad, try to recovery non factory bad blocks */
        printk("No enough nvram flash blocks, try to reuse bad blocks\n");
        for (ii = 0; ii < NVRAM_FLASH_BLOCK_COUNT; ii ++)
        {
            if (gxv31xx_internal_mtd_recoverbad(nvram_flash_blocks[ii].start_addr) == 0)
            {
                /* recover succeeded */
                total_valid_parts += nvram_flash_blocks[ii].part_count;
            }
        }
    }

    tmpbuf = vmalloc(NVRAM_SPACE);

    for (ii = 0; ii < total_valid_parts; ii ++)
    {
        /* we write to next part, even the current part is empty case (no nvram data at all). */
        if (nvram_flash_get_part_for_write(&block_index_write, &part_index_write))
        {
            /* no blocks found for writing */
            goto all_bad;
        }
        if ( part_index_write == 0)
        {
            gxv31xx_internal_mtd_erase(nvram_flash_blocks[block_index_write].start_addr, 
                    NVRAM_SPACE * nvram_flash_blocks[block_index_write].part_count);
            write_fail = 0; /* reset write failure count */
        }

        /* Write Flash */
        gxv31xx_internal_mtd_write(nvram_flash_blocks[block_index_write].start_addr + NVRAM_SPACE * part_index_write, NVRAM_SPACE, zbuf);
        /* Check with reading back */
        {    
            if (tmpbuf)
            {
                if (!gxv31xx_internal_mtd_read(nvram_flash_blocks[block_index_write].start_addr + part_index_write * NVRAM_SPACE, 
                            NVRAM_SPACE, tmpbuf)) /* Does it have value to go ahead to verify checksum here? */
                {
                    /* read back success */
                    break;
                }
            }
            else    //be lazy, if vmalloc failed we just skip doing read back check
                break;
            write_fail ++;

            if (write_fail == nvram_flash_blocks[block_index_write].part_count)
            {
                /* we failed to write all parts in this block, mark the block as bad block */
                gxv31xx_internal_mtd_markbad(nvram_flash_blocks[block_index_write].start_addr);
                total_valid_parts -= nvram_flash_blocks[block_index_write].part_count;
                printk("block[%d] has no good parts any more, marked bad for retirement\n", block_index_write);
            }
            printk("nvram part[%d:%d] failure, rewrite performed\n", block_index_write, part_index_write);
        }
    }
    if (ii == NVRAM_FLASH_BLOCK_COUNT)
    {
        /* no good blocks any more */
        goto all_bad;
    }
    else
    {
        /* update to new current block/part index */
        current_block_index = block_index_write;
        current_part_index = part_index_write;
        ret = 0;
    }

    goto exit;

all_bad:
    printk("no good parts found for nvram settings\n");
    ret = -1;

exit:
    if (tmpbuf) vfree(tmpbuf);
    return ret;
}

int proc_nvram_flash_get_status(char *buf, char **start, off_t offset,
				     int count, int *eof, void *data)
{
    int len = 0;
    int ii;

    if (len <= count)
        len += sprintf(buf + len, "%-35s: %s\n", "Storage", "flash");
    if (len <= count)
        len += sprintf(buf + len, "%-35s: %u\n", "Total Blocks", NVRAM_FLASH_BLOCK_COUNT); 
    if (len <= count)
        len += sprintf(buf + len, "%-35s: %u\n\n", "Total Parts Valid", total_valid_parts); 
    for (ii = 0; ii < NVRAM_FLASH_BLOCK_COUNT; ii ++)
    {
        if (len <= count)
            len += sprintf(buf + len, "%-35s: %d, 0x%x, %d %s\n", "Block", ii, nvram_flash_blocks[ii].start_addr, 
                    nvram_flash_blocks[ii].part_count, 
                    gxv31xx_internal_mtd_checkbad(nvram_flash_blocks[ii].start_addr)? "[bad block]":""
                    ); 
    }

    if (len <= count)
        len += sprintf(buf + len, "\n%-35s: %u\n", "Active Block", current_block_index); 
    if (len <= count)
        len += sprintf(buf + len, "%-35s: %u\n", "Active Part", current_part_index); 
    if (len <= count)
        len += sprintf(buf + len, "%-35s: %lu\n", "Write Counter", current_write_counter); 

    //if (len <= count)
    //    len += sprintf(buf + len, "%-35s: %lu\n", "nvram_offset", nvram_offset); 

    if (len <= offset + count) *eof = 1;
    *start = buf + offset;
    len -= offset;
    if (len > count) len = count;
    if (len < 0) len = 0;

    return (len);
}

