/*
 * NVRAM variable manipulation
 *
 * Copyright 2005, Broadcom Corporation
 * All Rights Reserved.
 * 
 * THIS SOFTWARE IS OFFERED "AS IS", AND BROADCOM GRANTS NO WARRANTIES OF ANY
 * KIND, EXPRESS OR IMPLIED, BY STATUTE, COMMUNICATION OR OTHERWISE. BROADCOM
 * SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A SPECIFIC PURPOSE OR NONINFRINGEMENT CONCERNING THIS SOFTWARE.
 *
 * $Id: nvram.h,v 1.1 2012/04/09 18:06:51 pradev Exp $
 *
 * SCF - ADM5120 port - 6/30/2006
 *     - removed broadcom specific portions and OS indepent layer
 *     - removed MTD dependence, ported to raw flash access routines
 *     - added zlib support and checksumming for nvram file
 *                     
 */

#ifndef __NVRAM_H__
#define __NVRAM_H__

#define NVRAM_DEV_MAJOR 241
#define NVRAM_DEV_MINOR 0

#define MARK_TEMP         1
#define MARK_L1LOCKED     2
#define MARK_L2LOCKED     3
#define MARK_PASSWD       4     // SXIE: 4 can be applied on 2 or 3 so we can allow passwd fields to be l1 or l2 locked.

#define FLAG_TEMP       0x1
#define FLAG_L1LOCKED   0x2
#define FLAG_L2LOCKED   0x4
#define FLAG_PASSWD     0x8

#define PASSWD_TYPE_SUPPORT     1

#ifndef _LANGUAGE_ASSEMBLY

#include <linux/types.h>

struct nvram_header {
	int magic;
	int len;
	int space_used;
    int len_compressed;
    unsigned short chksum_protected;    /* SXIE: this can be used for back compatiblity */
    unsigned short chksum;
    unsigned long  write_counter;
	char reserved[4];
};
#define NVRAM_HEADER_SIZE  sizeof(struct nvram_header)

struct nvram_flash_block {
    u_int32_t start_addr;
    u_int32_t part_count;
};

struct nvram_tuple {
	char *name;
	char *value;
	int flag;
	struct nvram_tuple *next;
};

#define IS_TEMP(t) (t->flag & FLAG_TEMP)
#define IS_PASSWD(t) (t->flag & FLAG_PASSWD)
#define IS_L1LOCKED(t) (t->flag & FLAG_L1LOCKED)
#define IS_L2LOCKED(t) (t->flag & FLAG_L2LOCKED)
#define IS_LOCKED(t) (IS_L1LOCKED(t) || IS_L2LOCKED(t))

#define SET_TEMP(t) do { t->flag |= FLAG_TEMP; } while (0)
#define SET_PASSWD(t) do { t->flag |= FLAG_PASSWD; } while (0)
#define SET_L1LOCKED(t) do { t->flag |= FLAG_L1LOCKED; } while (0)
#define SET_L2LOCKED(t) do { t->flag |= FLAG_L2LOCKED; } while (0)

struct nvram_inflate_stream {
	char *data;
	struct nvram_inflate_stream *next;
};

/*
 * Initialize NVRAM access. May be unnecessary or undefined on certain
 * platforms.
 */
extern int nvram_init(void *sbh);

/*
 * Disable NVRAM access. May be unnecessary or undefined on certain
 * platforms.
 */
extern void nvram_exit(void);

/*
 * Get the value of an NVRAM variable. The pointer returned may be
 * invalid after a set.
 * @param	name	name of variable to get
 * @return	value of variable or NULL if undefined
 */
extern char * nvram_get(const char *name);

/* 
 * Get the value of an NVRAM variable.
 * @param	name	name of variable to get
 * @return	value of variable or NUL if undefined
 */
#define nvram_safe_get(name) (nvram_get(name) ? : "")

/*
 * Match an NVRAM variable.
 * @param	name	name of variable to match
 * @param	match	value to compare against value of variable
 * @return	TRUE if variable is defined and its value is string equal
 *		to match or FALSE otherwise
 */
static __inline__ int
nvram_match(char *name, char *match) {
	const char *value = nvram_get(name);
	return (value && !strcmp(value, match));
}

/*
 * Inversely match an NVRAM variable.
 * @param	name	name of variable to match
 * @param	match	value to compare against value of variable
 * @return	TRUE if variable is defined and its value is not string
 *		equal to invmatch or FALSE otherwise
 */
static __inline__ int
nvram_invmatch(char *name, char *invmatch) {
	const char *value = nvram_get(name);
	return (value && strcmp(value, invmatch));
}

/*
 * Set the value of an NVRAM variable. The name and value strings are
 * copied into private storage. Pointers to previously set values
 * may become invalid. The new value may be immediately
 * retrieved but will not be permanently stored until a commit.
 * @param	name	name of variable to set
 * @param	value	value of variable
 * @return	0 on success and errno on failure
 */
extern int nvram_set(const char *name, const char *value);

/*
 * Unset an NVRAM variable. Pointers to previously set values
 * remain valid until a set.
 * @param	name	name of variable to unset
 * @return	0 on success and errno on failure
 * NOTE: use nvram_commit to commit this change to flash.
 */
extern int nvram_unset(const char *name);

/*
 * Commit NVRAM variables to permanent storage. All pointers to values
 * may be invalid after a commit.
 * NVRAM values are undefined after a commit.
 * @return	0 on success and errno on failure
 */
extern int nvram_commit(void);

/*
 * Get all NVRAM variables (format name=value\0 ... \0\0).
 * @param	buf	buffer to store variables
 * @param	count	size of buffer in bytes
 * @return	0 on success and errno on failure
 */
extern int nvram_getall(char *buf, int count);

/*
 * erase all NVRAM variables except locked ones
 */
extern int nvram_eraseall(void);

#endif /* _LANGUAGE_ASSEMBLY */

#define FLASH_NVRAM_OFFSET  0x80000     /* Block 4, will use 5,6 for backup blocks */

#define NVRAM_MAGIC		    0x48534C46	/* 'FLSH' */
#define NVRAM_SPACE_FREE	0x00000001
#define NVRAM_ERASEALL	    0x00000010
#define NVRAM_RELOAD        0x00000100
#define NVRAM_VERSION		1
#define NVRAM_SPACE		    0x4000             /* 16K                   */
#define NVRAM_ERASESIZE	    0x20000            /* 128K for 1 Nand block */
#define NVRAM_BUFFER_SPACE	(2*NVRAM_SPACE)
#define NVRAM_REDUNDANT     3                  /* we reserved 3 blocks for nvram setting, if one goes bad, use next */

unsigned short verify_zbuf_chksum(char *zbuf_with_header);

#endif /* _nvram_h_ */
