/*
 * NVRAM variable manipulation (common)
 *
 * Copyright 2004, Broadcom Corporation
 * All Rights Reserved.
 * 
 * THIS SOFTWARE IS OFFERED "AS IS", AND BROADCOM GRANTS NO WARRANTIES OF ANY
 * KIND, EXPRESS OR IMPLIED, BY STATUTE, COMMUNICATION OR OTHERWISE. BROADCOM
 * SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A SPECIFIC PURPOSE OR NONINFRINGEMENT CONCERNING THIS SOFTWARE.
 *
 * $Id: nvram.c,v 1.1 2012/04/09 18:06:51 pradev Exp $
 */

#include <linux/types.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/ctype.h>
#include <linux/zlib.h>
#include <linux/vmalloc.h>

#include "nvram.h"


#define ARRAYSIZE(a)            (sizeof(a)/sizeof(a[0]))

extern struct nvram_tuple * _nvram_realloc(struct nvram_tuple *t, const char *name, const char *value, int *value_changed);
extern void _nvram_free(struct nvram_tuple *t);
extern int _nvram_read(char *buf);

char * _nvram_get(const char *name);
int _nvram_set(const char *name, const char *value, int set_dirty_flag);
int _nvram_setacl(const char *name);
int _nvram_unset(const char *name);
int _nvram_tag(const char *name);
int _nvram_getall(char *buf, int count);
int _nvram_commit(struct nvram_header *header);
int _nvram_init(void);
int _nvram_erase_hushtable(void);
void _nvram_exit(void);

static struct nvram_tuple * nvram_hash[257];
static struct nvram_tuple * nvram_dead;

extern unsigned int nvram_dirty;
extern unsigned long nvram_size;

/* Free all tuples. Should be locked. */
int  
_nvram_erase_hushtable(void)
{
	uint i;
	struct nvram_tuple *t, *next;

	/* Free hash table */
	for (i = 0; i < ARRAYSIZE(nvram_hash); i++) {
		for (t = nvram_hash[i]; t; t = next) {
			next = t->next;
			_nvram_free(t);
		}
		nvram_hash[i] = NULL;
	}

	/* Free dead table */
	for (t = nvram_dead; t; t = next) {
		next = t->next;
		_nvram_free(t);
	}
	nvram_dead = NULL;

	/* Indicate to per-port code that all tuples have been freed */
	_nvram_free(NULL);
    return 0;
}

/* Purge all unlocked tuples. Should be locked. */
int  
_nvram_purge_unlocked(void)
{
	uint i;
	struct nvram_tuple *t, *next, **prev;

	/* Free hash table */
	for (i = 0; i < ARRAYSIZE(nvram_hash); i++) {
		for (t = nvram_hash[i], prev = &nvram_hash[i]; t; t = next) {
			next = t->next;
	        /* Move unlocked tuple to the dead table */
            if (!IS_LOCKED(t)) 
            {
                //printk("purge %s\n", t->name);
                *prev = t->next;
                t->next = nvram_dead;
                nvram_dead = t;
            }
            else
            {
                //printk("reserve %s\n", t->name);
                prev = &t->next;
            }
		}
	}

    return 0;
}


/* String hash */
static __inline__ uint
hash(const char *s)
{
	uint hash = 0;

	while (*s)
		hash = 31 * hash + *s++;

	return hash;
}

/* Free and recreate the hash table. Should be locked. */
static int 
nvram_rehash(struct nvram_header *header)
{
	char *name, *value, *end, *eq;

	/* (Re)initialize hash table, free all the nodes */
	_nvram_erase_hushtable();

	/* Parse and set "name=value\0 ... \0\0" */
	name = (char *) &header[1];
	end = (char *) header + NVRAM_BUFFER_SPACE - 2;
	end[0] = end[1] = '\0';
	for (; *name; name = value + strlen(value) + 1) {
		if (!(eq = strchr(name, '=')))
			break;
		*eq = '\0';
		value = eq + 1;
        //printk("%s=%s\n", name, value);
		_nvram_set(name, value, 0); /* recreate node for it, and do not touch the dirty flag in this case */
        if (name[0] == MARK_L1LOCKED || name[0] == MARK_L2LOCKED) /* _nvram_set will not set the lock bit */
        {
            _nvram_setacl(name);
        }
		*eq = '='; /* restore it back */
	}

	return 0;
}

/* Get the value of an NVRAM variable. Should be locked. */
char * 
_nvram_get(const char *name)
{
	uint i;
	struct nvram_tuple *t;
	char *value;
	int offset = 0;
    int istmp = 0;
    int aclset = 0;

	if (!name)
		return NULL;

    //printk("[0]=0x%x\n", (int)name[0]);
	if (name[0]==':' || name[0] == MARK_TEMP)
    {
		offset = 1;
        istmp = 1;
    }
    else if (name[0] == MARK_L1LOCKED || name[0] == MARK_L2LOCKED)
    {
        //printk("locked value get\n");
        offset = 1;
        aclset = 1;
    }

	/* Hash the name */
	i = hash(name + offset) % ARRAYSIZE(nvram_hash);

	/* Find the associated tuple in the hash table */
	for (t = nvram_hash[i]; t && strcmp(t->name, name + offset); t = t->next);

    if (t)
    { 
        if (!IS_L2LOCKED(t) || (name[0] == MARK_L2LOCKED))
	        value = t->value;
        else
        {
            //printk("acl check failed\n");
            value = NULL;
        }
    }
    else
    {
        //printk("node not found\n");
        value = NULL;
    }

	return value;
}

/* Set the value of an NVRAM variable. Should be locked. */
/* SXIE: If the name start with ":", then it is temp variable, we allow to overwrite a perm to temp one, or the other way */
int 
_nvram_set(const char *name, const char *value, int set_dirty_flag)
{
	uint i;
	struct nvram_tuple *t, *u, **prev;
	int offset=0;
    int istmp = 0;
    int aclset = 0;
    int value_changed = 0;

	if (name[0]==':' || name[0] == MARK_TEMP)
    {
		offset = 1;
        istmp = 1;
    }
	else
    {
        if (name[0] == MARK_L1LOCKED || name[0] == MARK_L2LOCKED)
        {
            //printk("priviliged set\n");
		    offset = 1;
            aclset = 1;
        }
    }
	
	/* Hash the name */
	i = hash(name+offset) % ARRAYSIZE(nvram_hash);

	/* Find the associated tuple in the hash table */
	for (prev = &nvram_hash[i], t = *prev; t && strcmp(t->name, name+offset); prev = &t->next, t = *prev);

    /* check acl lock level */
    if (t != NULL && IS_LOCKED(t))
    {
        //printk("node is under acl\n");
        if (aclset == 0 || (name[0] == MARK_L1LOCKED && IS_L2LOCKED(t)))
            return -EACCES;
    }
    //if (!t) printk("node not found\n");

	/* (Re)allocate tuple, if t is NULL, we will allocate new, otherwise try to reuse it if possible */
	if (!(u = _nvram_realloc(t, name+offset, value, &value_changed)))
		return -ENOMEM;

    /* if node reused, the flag is not touched. for new node, set to temp */
    /* SXIE: note!!! LOCKED flag shall not set so only these locked nodes will keep locked */
	if (istmp)
        SET_TEMP(u);
    else if (set_dirty_flag && value_changed)
    {
        //printk("set dirty for %s set/change to %s\n",name, value);
        nvram_dirty = 1;
    }

    if (set_dirty_flag && value_changed == 0)
    {
        return -EEXIST;
    }

	/* Value reallocated */
	if (t && t == u) /* SXIE: same node used */
		return 0;

	/* Move old tuple to the dead table */
	if (t) /* SXIE: this is not going to happen */
    {
        printk("move tuple to dead table\n");
		*prev = t->next;
		t->next = nvram_dead;
		nvram_dead = t;
	}

	/* Add new tuple to the hash table */
	u->next = nvram_hash[i];
	nvram_hash[i] = u;

	return 0;
}

int 
_nvram_setacl(const char *name)
{
	uint i;
	struct nvram_tuple *t, *u, **prev;
	int offset = 0;

	if (!name)
		return 0;
	
	if ( name[0] == MARK_L1LOCKED ||name[0] == MARK_L2LOCKED )
    {
		offset = 1;
    }

	/* Hash the name */
	i = hash(name + offset) % ARRAYSIZE(nvram_hash);

	/* Find the associated tuple in the hash table */
	for (prev = &nvram_hash[i], t = *prev; t && strcmp(t->name, name + offset); prev = &t->next, t = *prev);

	/* if node found, tag it */
	if (t && !IS_LOCKED(t)) 
    {
        if (name[0] == MARK_L1LOCKED)
        {
            //printk("set l1locked\n");
            SET_L1LOCKED(t);
        }
        else if (name[0] == MARK_L2LOCKED)
        {
            //printk("set l2locked\n");
            SET_L2LOCKED(t);
        }
	}
    else if (t == NULL)
    {
        /* create a empty node for the name */
        /* SXIE: looks we do not set dirty flag for this, so not monitor the value change flag */
	    if (!(u = _nvram_realloc(NULL, name+offset, "", NULL)))
		    return -ENOMEM;

        if (name[0] == MARK_L1LOCKED)
        {
            //printk("set l1locked\n");
            SET_L1LOCKED(u);
        }
        else if (name[0] == MARK_L2LOCKED)
        {
            //printk("set l2locked\n");
            SET_L2LOCKED(u);
        }

	    /* Add new tuple to the hash table */
	    u->next = nvram_hash[i];
	    nvram_hash[i] = u;
    }

	return 0;

}

/* Unset the value of an NVRAM variable. Should be locked. */
/* SXIE: If the name start with ":", then it is temp variable, we allow to overwrite a perm to temp one, or the other way */
int 
_nvram_unset(const char *name)
{
	uint i;
	struct nvram_tuple *t, **prev;
	int offset = 0;

	if (!name)
		return 0;

	if (name[0]==':' || name[0] == MARK_TEMP)
		offset = 1;
		
	/* Hash the name */
	i = hash(name + offset) % ARRAYSIZE(nvram_hash);

	/* Find the associated tuple in the hash table */
	for (prev = &nvram_hash[i], t = *prev; t && strcmp(t->name, name + offset); prev = &t->next, t = *prev);

	/* if node found, Move it to the dead table */
	if (t) {
        /* check acl lock level */
        if (IS_LOCKED(t))
        {
            return -EACCES;
        }

		if ( !IS_TEMP(t) )
        {
            //printk("set dirty for %s unset\n",name);
	        nvram_dirty = 1;
        }
		*prev = t->next;
		t->next = nvram_dead;
		nvram_dead = t;
	}
    else
        return -EEXIST;

	return 0;
}

/* Get all NVRAM variables. Should be locked. */
int 
_nvram_getall(char *buf, int count)
{
	uint i;
	struct nvram_tuple *t;
	int len = 0;

	memset(buf, 0, count);

	/* Write name=value\0 ... \0\0 */
	for (i = 0; i < ARRAYSIZE(nvram_hash); i++) {
		for (t = nvram_hash[i]; t; t = t->next) {
			if ((count - len) > (strlen(t->name) + 1 + strlen(t->value) + 1)) {
                if ( !IS_L2LOCKED(t))
                {
                    if ( !IS_TEMP(t) )
                        len += sprintf(buf + len, "%s=%s", t->name, t->value) + 1;
                    else
                        len += sprintf(buf + len, ":%s=%s", t->name, t->value) + 1;
                }
			}
			else
				break;
		}
	}

	return 0;
}

int
_nvram_savetemp(struct nvram_header *temp_table)
{
	int i;
	struct nvram_tuple *t;

	char *ptr_temp, *end_temp;
	char *fmt_str;

	fmt_str = "%c%s=%s";

	/* Regenerate header */
	temp_table->magic = NVRAM_MAGIC;

	/* Clear data area */
	ptr_temp = (char *) temp_table + sizeof(struct nvram_header);
	memset(ptr_temp, 0, NVRAM_BUFFER_SPACE - sizeof(struct nvram_header));

	/* Leave space for a double NUL at the end (account for in nvram_size) */
	end_temp = (char *) temp_table + NVRAM_BUFFER_SPACE - 2;

	/* Write out all tuples */
	for (i = 0; i < ARRAYSIZE(nvram_hash); i++) {
		for (t = nvram_hash[i]; t; t = t->next) {
            if (IS_TEMP(t))
            {
			    ptr_temp += sprintf(ptr_temp, fmt_str ,MARK_TEMP, t->name, t->value) + 1;
            }
		}
	}

	/* End with a double NUL */
	ptr_temp += 2;

	/* Set new length */
	temp_table->len = ptr_temp - (char *)temp_table;
	
	return 0;
}

int
_nvram_loadtemp(struct nvram_header *header)
{
	char *name, *value, *end, *eq;

	/* Parse and set "name=value\0 ... \0\0" */
	name = (char *) &header[1];
	end = (char *) header + NVRAM_BUFFER_SPACE - 2;
	end[0] = end[1] = '\0';
	for (; *name; name = value + strlen(value) + 1) {
		if (!(eq = strchr(name, '=')))
			break;
		*eq = '\0';
		value = eq + 1;
        //printk("%s=%s\n", name, value);
		_nvram_set(name, value, 0); /* recreate node for it, and do not touch the dirty flag in this case */
		*eq = '='; /* restore it back */
	}

	return 0;
}

/* Regenerate NVRAM. Should be locked. */
int
_nvram_commit(struct nvram_header *header)
{
	char *ptr, *end;
	int i, ret;
	struct nvram_tuple *t;

	char *ptr_full, *end_full;
	struct nvram_header *full_table;
	char *fmt_str[2];

	fmt_str[0] = "%s=%s";
	fmt_str[1] = "%c%s=%s";

	/* Allocate space for full table */
	if (!(full_table = vmalloc(NVRAM_BUFFER_SPACE)))
		return -ENOMEM;
	
	/* Regenerate header */
	header->magic = NVRAM_MAGIC;
	full_table->magic = NVRAM_MAGIC;

	/* Clear data area */
	ptr = (char *) header + sizeof(struct nvram_header);
	ptr_full = (char *) full_table + sizeof(struct nvram_header);
	memset(ptr, 0, NVRAM_BUFFER_SPACE - sizeof(struct nvram_header));
	memset(ptr_full, 0, NVRAM_BUFFER_SPACE - sizeof(struct nvram_header));

	/* Leave space for a double NUL at the end (account for in nvram_size) */
	end = (char *) header + NVRAM_BUFFER_SPACE - 2;
	end_full = (char *) full_table + NVRAM_BUFFER_SPACE - 2;

	/* Write out all tuples */
	for (i = 0; i < ARRAYSIZE(nvram_hash); i++) {
		for (t = nvram_hash[i]; t; t = t->next) {
		    if ( !IS_TEMP(t) )
            {
				if ((ptr + strlen(t->name) + 1 + strlen(t->value) + 1) > end)
					break;
				ptr += sprintf(ptr, "%s=%s", t->name, t->value) + 1;
			}
			if ((ptr_full + strlen(t->name) + 1 + strlen(t->value) + 1) > end_full)
				break;
            if (IS_TEMP(t))
            {
			    ptr_full += sprintf(ptr_full, fmt_str[1] ,MARK_TEMP, t->name, t->value) + 1;
            }
            else if (IS_L1LOCKED(t))
            {
			    ptr_full += sprintf(ptr_full, fmt_str[1] ,MARK_L1LOCKED,t->name, t->value) + 1;
            }
            else if (IS_L2LOCKED(t))
            {
			    ptr_full += sprintf(ptr_full, fmt_str[1] ,MARK_L2LOCKED,t->name, t->value) + 1;
            }
            else
			    ptr_full += sprintf(ptr_full, fmt_str[0] ,t->name, t->value) + 1;
		}
	}

	/* End with a double NUL */
	ptr += 2;
	ptr_full += 2;

	/* Set new length */
	header->len = ptr - (char *) header;
	full_table->len = ptr_full - (char *)full_table;
	
	/* Reinitialize hash table */
	ret = nvram_rehash(full_table);
	vfree(full_table);

	return ret;
}

/* Initialize hash table. Should be locked. */
int 
_nvram_init(void)
{
	struct nvram_header *header=NULL;
	int ret;

	if (!(header = vmalloc(NVRAM_BUFFER_SPACE))) {
		return -ENOMEM;
	}
	
	ret = _nvram_read((char*)header);

	if (ret<0) { /* -EIO, -ENOMEM, etc... */
		if (header) vfree(header);
		return ret;
	}

	if (header->magic == NVRAM_MAGIC)
		nvram_rehash(header);

	if (header) vfree(header);

	return ret;
}

/* Free hash table. Should be locked. */
void 
_nvram_exit(void)
{
	_nvram_erase_hushtable();
}
