/*
 * kernel mode mmc rw
 *
 * Copyright 2012, GrandStream Corporation
 * All Rights Reserved.
 * 
 * THIS SOFTWARE IS OFFERED "AS IS", AND BROADCOM GRANTS NO WARRANTIES OF ANY
 * KIND, EXPRESS OR IMPLIED, BY STATUTE, COMMUNICATION OR OTHERWISE. BROADCOM
 * SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A SPECIFIC PURPOSE OR NONINFRINGEMENT CONCERNING THIS SOFTWARE.
 *
 */

#include <linux/types.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/ctype.h>
#include <linux/vmalloc.h>
#include <linux/bio.h>
#include <linux/blkdev.h>
#include <linux/scatterlist.h>

struct bio_batch
{
	atomic_t 		done;
	unsigned long 		flags;
	struct completion 	*wait;
};

static void bio_batch_end_io(struct bio *bio, int err)
{
	struct bio_batch *bb = bio->bi_private;

	if (err) {
		if (err == -EOPNOTSUPP)
			set_bit(BIO_EOPNOTSUPP, &bb->flags);
		else
			clear_bit(BIO_UPTODATE, &bb->flags);
	}
	if (bb)
		if (atomic_dec_and_test(&bb->done))
			complete(bb->wait);
	bio_put(bio);
}

static int mmcblk_rw(int rw, sector_t start, sector_t nr_sects, char *buf, unsigned len)
{
	int ret = -ENODEV;
	struct bio *bio = NULL;
	struct bio_batch bb;
	unsigned int sz;
	DECLARE_COMPLETION_ONSTACK(wait);
	struct page *page;
	struct block_device *bdev = NULL;

	bdev = blkdev_get_by_dev(MKDEV(MMC_BLOCK_MAJOR, 0), FMODE_READ|FMODE_WRITE, NULL);
	if(IS_ERR(bdev))
	{
		printk("get blkdev err\n");
		return ret;
	}

	atomic_set(&bb.done, 1);
	bb.flags = 1 << BIO_UPTODATE;
	bb.wait = &wait;

submit:
	ret = 0;

	while (nr_sects != 0) {
		bio = bio_alloc(0, min(nr_sects, (sector_t)BIO_MAX_PAGES));
		if (!bio) {
			ret = -ENOMEM;
			goto out;
		}

		bio->bi_sector = start;
		bio->bi_bdev   = bdev;
		bio->bi_end_io = bio_batch_end_io;
		bio->bi_private = &bb;

		while (nr_sects != 0) {
			sz = min((sector_t) PAGE_SIZE >> 9 , nr_sects);
			if (sz == 0)
				/* bio has maximum size possible */
				break;
			page = vmalloc_to_page(buf);
			ret = bio_add_page(bio, page, sz << 9, 0);
			nr_sects -= ret >> 9;
			buf += PAGE_SIZE;
			if (ret < (sz << 9))
				break;
		}
	}
	ret = 0;
	atomic_inc(&bb.done);
	submit_bio(rw, bio);

	/* Wait for bios in-flight */
	if (!atomic_dec_and_test(&bb.done))
		wait_for_completion(&wait);

	if (!test_bit(BIO_UPTODATE, &bb.flags))
		/* One of bios in the batch was completed with error.*/
		ret = -EIO;

	if (ret)
		goto out;

	if (test_bit(BIO_EOPNOTSUPP, &bb.flags)) {
		ret = -EOPNOTSUPP;
		goto out;
	}

	if (nr_sects != 0)
		goto submit;

out:
	blkdev_put(bdev, FMODE_READ|FMODE_WRITE);
	return ret;
}

int emmc_zeroout(sector_t start, sector_t nr_sects)
{
	int rval;
	struct block_device *bdev = NULL;
	
	bdev = blkdev_get_by_dev(MKDEV(MMC_BLOCK_MAJOR, 0), FMODE_READ|FMODE_WRITE, NULL);
	if(IS_ERR(bdev))
	{
		printk("get blkdev err\n");
		return -ENODEV;
	}

	rval = blkdev_issue_zeroout(bdev, start, nr_sects, 0);
	blkdev_put(bdev, FMODE_READ|FMODE_WRITE);
	return rval;
}

int emmc_write(sector_t start, sector_t nr_sects, char *buf, unsigned int len)
{
	return mmcblk_rw(WRITE_FLUSH, start, nr_sects, buf, len);
}

int emmc_read(sector_t start, sector_t nr_sects, char *buf, unsigned int len)
{
	int rval;
	struct block_device *bdev = NULL;

get_again:	
	bdev = blkdev_get_by_dev(MKDEV(MMC_BLOCK_MAJOR, 0), FMODE_READ|FMODE_WRITE, NULL);
	if(IS_ERR(bdev))
	{
		goto get_again;
	}

	rval = mmcblk_rw(READ, start, nr_sects, buf, len);
	blkdev_put(bdev, FMODE_READ|FMODE_WRITE);
	return rval;
}

