/*
 * drivers/staging/gs/fact_cfg.c
 *
 * Driver for the provison.
 *
 * Copyright (C) 2012, 2013 GrandStream Inc.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
 * MA 02110-1301, USA.
 */

#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/delay.h>
#include <linux/rslib.h>
#include <linux/moduleparam.h>
#include <linux/platform_device.h>
#include <linux/dma-mapping.h>
#include <linux/types.h>
#include <linux/clk.h>
#include <linux/interrupt.h>
#include <linux/semaphore.h>
#include <linux/io.h>
#include <linux/slab.h>
#include <linux/ccu.h>
#include <linux/completion.h>
#include <linux/uaccess.h>
#include <linux/proc_fs.h>
#include <linux/vmalloc.h>

#include <linux/seq_file.h>
#include "gs_def.h"
#include "gs_misc.h"
#include <mach/gxp22xx.h>
#include "flash_rw.h"
#include "mmc_rw.h"

static char *factParaBuf;
static int fact_data_size;
static int fact_buf;

#define EMMC_FACTORY_START	(22529)
#define EMMC_RWU_SECTORS		(256)

extern int gs_set_ihwi(unsigned short hw_model, unsigned int hw_rev);

static unsigned short hwf_num_fxo;
static unsigned short hwf_num_fxs;
static unsigned short hwf_num_bri;
static unsigned short hwf_num_eth;
static unsigned short hwf_allow_nat;
static unsigned short hwf_svip_type;

typedef struct
{
	const char *pname;
	void *data;
	int (*proc_read)(char *page, char **start, off_t off, int count, int *eof, void *data);
}ipentry_t;

static int short_hwf_proc_read (char *page, char **start, off_t off, int count, int *eof, void *data)
{
	int len = 0;
	char *p = page;

	p += sprintf(p, "%d\n", *((unsigned short *)data));
    
	len = (p - page);
	if (len <= off+count) *eof = 1;
	*start = page + off;
	len -= off;
	if (len>count) len = count;
	if (len<0) len = 0;

	return (len);
}

static ipentry_t ipitems[] =
{
	{"num_fxo", &hwf_num_fxo, short_hwf_proc_read},
	{"num_fxs", &hwf_num_fxs, short_hwf_proc_read},
	{"num_bri", &hwf_num_bri, short_hwf_proc_read},
	{"num_eth", &hwf_num_eth, short_hwf_proc_read},
	{"allow_nat", &hwf_allow_nat, short_hwf_proc_read},
	{"svip_type", &hwf_svip_type, short_hwf_proc_read},
};

int hwf_pentrys_create(struct proc_dir_entry *dent)
{
	int i;
	struct proc_dir_entry *ent;

	for(i=0; i<sizeof(ipitems)/sizeof(ipitems[0]); i++)
	{
		ent = create_proc_entry(ipitems[i].pname, S_IFREG|S_IRUGO, dent);
		if (ent) 
		{
			ent->read_proc = ipitems[i].proc_read;
			ent->data = ipitems[i].data;
		}
	}
	return 0;
}

void set_hwf_default(unsigned short hw_model, unsigned int hw_rev)
{
	printk("set default hw feature\n");
	hwf_num_fxs = 2;
	switch(hw_model)
	{
	case IHWI_MODEL_GXE5116:
		hwf_num_fxo = 16;
		hwf_svip_type = 16;
		hwf_allow_nat = 0;
		hwf_num_eth = 1;
		break;
	case IHWI_MODEL_GXE5108:
		hwf_num_fxo = 8;
		hwf_svip_type = 16;
		hwf_allow_nat = 0;
		hwf_num_eth = 1;
		break;
	case IHWI_MODEL_GXE5104:
		hwf_num_fxo = 4;
		hwf_svip_type = 4;
		hwf_allow_nat = 0;
		switch(hw_rev)
		{
		case IHWI_REV_V12A:
		case IHWI_REV_V14A:
			hwf_num_eth = 1;
			break;
		case IHWI_REV_V11A:
		default:
			hwf_num_eth = 2;
			break;
		}
		break;
	case IHWI_MODEL_GXE5102:
		hwf_num_fxo = 2;
		hwf_svip_type = 4;
		hwf_allow_nat = 1;
		hwf_num_eth = 2;
		break;
	case IHWI_MODEL_UCM6202:
		hwf_num_fxo = 0;
		hwf_num_bri = 2;
		hwf_svip_type = 4;
		hwf_allow_nat = 0;
		hwf_num_eth = 2;
		break;
	case IHWI_MODEL_UCM6201:
		hwf_num_fxo = 0;
		hwf_num_bri = 1;
		hwf_svip_type = 4;
		hwf_allow_nat = 1;
		hwf_num_eth = 2;
		break;
	default:
		;
	}

	return;
}

void factparam_hwf_parse(GS_FACTORYPARA* factPara)
{
	int length = 0;
	int count = 0;
	unsigned short	*featurePtr = NULL;
	unsigned short id;
	unsigned short	*end = NULL;

	if(!factPara) return;

	length = factPara->length[1] << 16 | factPara->length[0];
	if (length <= sizeof (GS_FACTORYPARA)) return;
	if (length > GS_FACTORY_CFG_SIZE) return;

	count = factPara->hw_feature_table_word[2];
	featurePtr = (unsigned short*)factPara->hw_feature_table_word+3;
	end = featurePtr+factPara->hw_feature_table_word[0];

	while (count && featurePtr<end)
	{
		id = *featurePtr;
		switch(id)
		{
		case HW_FEATURE_NUM_FXO:
			hwf_num_fxo = *(featurePtr+2);
			printk("num_fxo=%d\n", hwf_num_fxo);
			break;
		case HW_FEATURE_NUM_FXS:
			hwf_num_fxs = *(featurePtr+2);
			printk("num_fxs=%d\n", hwf_num_fxs);
			break;
		case HW_FEATURE_NUM_BRI:
			hwf_num_bri = *(featurePtr+2);
			printk("num_bri=%d\n", hwf_num_bri);
			break;
		case HW_FEATURE_NUM_ETH:
			hwf_num_eth = *(featurePtr+2);
			printk("num_eth=%d\n", hwf_num_eth);
			break;
		case HW_FEATURE_ALLOW_NAT:
			hwf_allow_nat = *(featurePtr+2);
			printk("allow_nat=%d\n", hwf_allow_nat);
			break;
		case HW_FEATURE_SVIP_TYPE:
			hwf_svip_type = *(featurePtr+2);
			printk("svip_type=%d\n", hwf_svip_type);
			break;
		default:
			break;
		}
		featurePtr += *(featurePtr+1) + 2;
		count--;
	}

	return;
}

void factparam2ihwi(GS_FACTORYPARA* factPara)
{
	unsigned short hw_model;
	unsigned int hw_rev;

	if(factPara->hw_model)
	{
		hw_model = factPara->hw_model;
		hw_rev = (factPara->RevID[0][0])|((factPara->RevID[0][1])<<16);
		factparam_hwf_parse(factPara);
		printk("use hw_model %x in factory cfg\n", hw_model);
	}
	else
	{
		printk("hw_model by PN <%.11s>\n", (char *)(factPara->PartNumber));
		hw_rev = factPara->PartNumber[4] | (factPara->PartNumber[5]<<16);
		if(!compare_pn_str((char *)factPara->PartNumber, HW_UCM6201_PNSTR))
		{
			hw_model = IHWI_MODEL_UCM6201;
		}
		if(!compare_pn_str((char *)factPara->PartNumber, HW_UCM6202_PNSTR))
		{
			hw_model = IHWI_MODEL_UCM6202;
		}
		else if(!compare_pn_str((char *)factPara->PartNumber, HW_GXE5116_PNSTR))
		{
			hw_model = IHWI_MODEL_GXE5116;
		}
		else if(!compare_pn_str((char *)factPara->PartNumber, HW_GXE5108_PNSTR))
		{
			hw_model = IHWI_MODEL_GXE5108;
		}
		else if(!compare_pn_str((char *)factPara->PartNumber, HW_GXE5104_PNSTR))
		{
			hw_model = IHWI_MODEL_GXE5104;
		}
		else if(!compare_pn_str((char *)factPara->PartNumber, HW_GXE5102_PNSTR))
		{
			hw_model = IHWI_MODEL_GXE5102;
		}
		else
		{
			hw_model = IHWI_MODEL_GXE5116;
			hw_rev = IHWI_REV_V14A;
			printk("unsupport PartNumber default to <%s>\n", HW_GXE5116_PNSTR);
		}
		set_hwf_default(hw_model, hw_rev);
	}

	gs_set_ihwi(hw_model, hw_rev);
	return;
}

int factcfg_read(void)
{
	int rval = 0;
	int storage = gs_get_storage();
	sector_t start = EMMC_FACTORY_START;
	sector_t total = (GS_FACTORY_CFG_SIZE)>>9;
	sector_t off;
	sector_t nsecs;

	factParaBuf = vmalloc(GS_FACTORY_CFG_SIZE);
	if(!factParaBuf)
	{
		printk("factcfg read out of memory\n");
		return -ENOMEM;
	}
	
	switch(storage)
	{
	case GS_STORAGE_FLASH:
#ifdef CONFIG_GS_STORAGE_FLASH
		rval = gxv31xx_internal_mtd_read(GS_FACTORY_CFG_OFFSET, GS_FACTORY_CFG_SIZE, factParaBuf);
#endif
		break;
	case GS_STORAGE_EMMC:
#ifdef CONFIG_GS_STORAGE_EMMC
		factParaBuf[0] = 0x55;
		factParaBuf[1] = 0xaa;
		for(off=0;off<total;)
		{
			nsecs = total-off;
			if(nsecs>EMMC_RWU_SECTORS) nsecs = EMMC_RWU_SECTORS;
			rval = emmc_read(start+off, nsecs, factParaBuf+(off<<9), nsecs<<9);
			if(rval<0)
			{
				printk("factcfg emmc read err %d", rval);
			}
			off += nsecs;
		}
#endif
		break;
	default:
		break;
	}
	return rval;
}

int factcfg_write(void)
{
	int rval = 0;
	int storage = gs_get_storage();
	sector_t start = EMMC_FACTORY_START;
	sector_t total = (GS_FACTORY_CFG_SIZE)>>9;
	sector_t off;
	sector_t nsecs;

	switch(storage)
	{
	case GS_STORAGE_FLASH:
#ifdef CONFIG_GS_STORAGE_FLASH
		gxv31xx_internal_mtd_erase(GS_FACTORY_CFG_OFFSET, GS_FACTORY_CFG_SIZE);
		rval = gxv31xx_internal_mtd_write(GS_FACTORY_CFG_OFFSET, GS_FACTORY_CFG_SIZE, factParaBuf);
#endif
		break;
	case GS_STORAGE_EMMC:
#ifdef CONFIG_GS_STORAGE_EMMC
		for(off=0;off<total;)
		{
			nsecs = total-off;
			if(nsecs>EMMC_RWU_SECTORS) nsecs = EMMC_RWU_SECTORS;
			rval = emmc_write(start+off, nsecs, factParaBuf+(off<<9), nsecs<<9);
			if(rval<0)
			{
				printk("factcfg emmc write err %d", rval);
			}
			off += nsecs;
		}
#endif
		break;
	default:
		break;
	}
	return rval;
}


static unsigned short checksum(GS_FACTORYPARA *factPara)
{
    unsigned char  *p = (unsigned char *)factPara;
    unsigned short sum = 0;
    unsigned short checksum = 0;
    unsigned length = 0;
    int i;

    // the length field is designed to be able expand the structure but since we need copy
    // the whole structure, it makes nothing except has chance to crash when length is invalid
    length = factPara->length[1]<<16|factPara->length[0]; // le
    
    if (length > GS_FACTORY_CFG_SIZE)
        length = GS_FACTORY_CFG_SIZE;

    for (i = 0; i < length; i+=2)   // first 4 byts are 
    {
        if ( i == offsetof(GS_FACTORYPARA,checksum))
        {
            continue;   // skip checksum itself
        }
        sum += ((p[i] << 8) | p[i+1]);    // fact config use be for checksum calc
    }
    checksum = 0xffff - sum + 0x1;

    return ((checksum >> 8) | (checksum << 8)); // be
}

static int read_fact_para(const unsigned int id, GS_FACTORYPARA *buf_ptr, char *data)
{
	int length = 0;
	GS_FACTORYPARA *factPara = buf_ptr;

	if(!buf_ptr) return -1;

	length = factPara->length[1] << 16 | factPara->length[0];
	if (length <= sizeof (GS_FACTORYPARA))
		return 0;

	if (factPara->checksum == checksum(factPara))
	{
		int count = 0;
		unsigned short  *featurePtr = NULL;
		unsigned short data_len=0;

		count = factPara->hw_feature_table_word[2];
		featurePtr = (unsigned short*)factPara->hw_feature_table_word+3;

		while (count)
		{
			if (id == *featurePtr)
			{
				data_len = (*(featurePtr+1))<<1;
				memcpy((char*)data, (char*)(featurePtr+2), data_len);
				return data_len;
			}
			else
			{
				featurePtr += *(featurePtr+1) + 2;
				count--;
			}
		}
	}
	return 0;
}

static void *gs_seq_start(struct seq_file *s, loff_t *pos)
{
	if (*pos >= fact_data_size)
		return NULL;
	else
		return (void *)(fact_buf + (int)(*pos));

}

/*************************************************************************
*          Use seq_file to handle who's data big then 1 page             *
**************************************************************************/
static void *gs_seq_next(struct seq_file *s, void *v, loff_t *pos)
{
	(*pos)++;
	return gs_seq_start(s, pos);
}

static void gs_seq_stop(struct seq_file *s, void *v)
{
	return;
}

static int gs_seq_show(struct seq_file *s, void *v)
{
	char *spos = (char *) v;
	seq_printf(s, "%c", *spos);
	return 0;
}

static struct seq_operations gs_seq_ops={
	.start = gs_seq_start,
	.next = gs_seq_next,
	.stop = gs_seq_stop,
	.show = gs_seq_show
};


static int data_proc_open(struct inode *inode, struct file *file)
{
	GS_FACTORYPARA *factPara = (GS_FACTORYPARA *)factParaBuf;
	if (checksum(factPara) != factPara->checksum)
		return -EFAULT;

	fact_data_size = factPara->length[1] << 16 | factPara->length[0];
	fact_buf = (int)factPara;
	return seq_open(file, &gs_seq_ops);
}

static struct file_operations fact_data_proc_ops={
	.owner = THIS_MODULE,
	.open = data_proc_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = seq_release
};

/*************************************************************************
* Read functions of factory data for /proc file system 
 ************************************************************************/
static int board_factcfg_hwmodel_proc_read (char *page, char **start, off_t off, int count, int *eof, void *data)
{
    int len = 0;
	switch((int)data)
	{
		case 0:
        len = read_fact_para(HW_FEATURE_MODELALIAS, (GS_FACTORYPARA*)factParaBuf, page);
		break;
		case 1:
        len = read_fact_para(HW_FEATURE_MODELBASE, (GS_FACTORYPARA*)factParaBuf, page);
		break;
		case 2:
        len = read_fact_para(HW_FEATURE_MODELSTR, (GS_FACTORYPARA*)factParaBuf, page);
		break;
	}

	if (len <= off+count) 
		*eof = 1;
    *start = page + off;
    len -= off;
    if (len>count) 
		len = count;
    if (len<0) 
		len = 0;

    return (len);
}


static int board_factcfg_hwrev_proc_read (char *page, char **start, off_t off, int count, int *eof, void *data)
{
    int len = 0;
    GS_FACTORYPARA* factPara = (GS_FACTORYPARA*) factParaBuf;
    char *hwrev = (char *)page;
    char revStr[128];

    if (factPara->checksum == checksum(factPara))
    {
        {
            char *pn = (char *)factPara->PartNumber;
            sprintf(revStr, "V%c.%c%c", pn[8], pn[9], pn[10]);
        }
        len = strlen(revStr);
        memcpy(hwrev, revStr, len);
    }

    if (len <= off+count) 
		*eof = 1;
    *start = page + off;
    len -= off;
    if (len>count) 
		len = count;
    if (len<0) 
		len = 0;

    return (len);
}

static int board_factcfg_dev_mac_proc_read (char *page, char **start, off_t off, int count, int *eof, void *data)
{
    GS_FACTORYPARA* factPara = (GS_FACTORYPARA*) factParaBuf;
    int len = 0;
    char *default_mac = "00:0B:82:FF:FF:FF";
    char *dev_mac = (char *)page;
    char revStr[128];

    if (factPara->checksum == checksum(factPara))
    {
        sprintf(revStr, "%02X:%02X:%02X:%02X:%02X:%02X",
                factPara->mac[0] & 0xFF, (factPara->mac[0] >> 8) & 0xFF,
                factPara->mac[1] & 0xFF, (factPara->mac[1] >> 8) & 0xFF,
                factPara->mac[2] & 0xFF, (factPara->mac[2] >> 8) & 0xFF);
        len = strlen(revStr);
        memcpy(dev_mac, revStr, len);
    }
    else
    {
        len = strlen(default_mac);
        memcpy(dev_mac, default_mac, len);
    }
    
    if (len <= off+count) 
		*eof = 1;
    *start = page + off;
    len -= off;
    if (len>count) 
		len = count;
    if (len<0) 
		len = 0;

    return (len);
}



static int board_factcfg_dev_mac_proc_write(struct file *file, const char __user *buffer, unsigned long count, void *data)
{
	GS_FACTORYPARA* factPara = (GS_FACTORYPARA*) factParaBuf;
    	unsigned int mac[6];
	unsigned char mac_str[64] = {0};
	
 
	if (copy_from_user(mac_str, buffer, count)) 
		return (-EFAULT);
	
	sscanf(mac_str, "%02x:%02x:%02x:%02x:%02x:%02x", &mac[0], &mac[1], &mac[2], &mac[3], &mac[4], &mac[5]);
	if (factPara->checksum != checksum(factPara))
		return (-EINVAL);
	
	factPara->mac[0] = (0xff&mac[0]) | (0xff00&(mac[1] << 8));
	factPara->mac[1] = (0xff&mac[2]) | (0xff00&(mac[3] << 8));
	factPara->mac[2] = (0xff&mac[4]) | (0xff00&(mac[5] << 8));
		 
	factPara->checksum = checksum(factPara);

	factcfg_write();
	return (count);
}

int gxp22xx_getmac_factory(char *mac)
{
	char *default_mac = "00:0B:82:FF:FF:FF";
	int i;
	unsigned int m[6];
	GS_FACTORYPARA* factPara = (GS_FACTORYPARA*) factParaBuf;
	if (factPara->checksum == checksum(factPara))
	{
		unsigned char *mc;
		mc = (unsigned char *)(factPara->mac);
		for(i = 0; i < 6; i++)
		{
			if(*mc != 0xff && *mc != 0 )
			{
				break;
			}
			mc++;
		}
		if(i != 6)
		{
			memcpy(mac, factPara->mac, sizeof(factPara->mac));
			return 0;
		}
	}
	sscanf(default_mac, "%02x:%02x:%02x:%02x:%02x:%02x", &m[0], &m[1], &m[2], &m[3], &m[4], &m[5]);
	*mac++ = m[0];
	*mac++ = m[1];
	*mac++ = m[2];
	*mac++ = m[3];
	*mac++ = m[4];
	*mac = m[5];
	return 0;
}

EXPORT_SYMBOL(gxp22xx_getmac_factory);

int gxp22xx_setmac_factory(const char *mac)
{
	GS_FACTORYPARA* factPara = (GS_FACTORYPARA*) factParaBuf;
	if (factPara->checksum != checksum(factPara))
	{
		printk("factory checksum is error\n");
		return -1;
	}
	if(memcmp((char *)factPara->mac, mac, 6) != 0 )
	{
		printk("write mac to flash\n");
		factPara->mac[0] = mac[0] | (mac[1] << 8);
		factPara->mac[1] = mac[2] | (mac[3] << 8);
		factPara->mac[2] = mac[4] | (mac[5] << 8);
		 
		factPara->checksum = checksum(factPara);
	}
	return 0;
}

EXPORT_SYMBOL(gxp22xx_setmac_factory);

static int board_factcfg_pn_proc_read (char *page, char **start, off_t off, int count, int *eof, void *data)
{
    GS_FACTORYPARA* factPara = (GS_FACTORYPARA*) factParaBuf;
    int len = 0;
    char *dev_pn = (char *)page;

    if (factPara->checksum == checksum(factPara))
    {
        char *pn = (char *)factPara->PartNumber;
        len = strlen(pn);
        memcpy(dev_pn, pn, len);
    }
    
    if (len <= off+count) 
		*eof = 1;
    *start = page + off;
    len -= off;
    if (len>count) 
		len = count;
    if (len<0) 
		len = 0;

    return (len);
}

static int board_factcfg_oemid_proc_read (char *page, char **start, off_t off, int count, int *eof, void *data)
{
    int len = 0;
	char buf;

	len = read_fact_para(HW_FEATURE_OEM_ID, (GS_FACTORYPARA*)factParaBuf, &buf);
    sprintf(page, "%d", buf);

    if (len <= off+count) 
		*eof = 1;
    *start = page + off;
    len -= off;
    if (len>count) 
		len = count;
    if (len<0) 
		len = 0;

    return (len);
}


static int board_factcfg_fwkey_proc_read (char *page, char **start, off_t off, int count, int *eof, void *data)
{
    int len = 0;
    char buf;

	len = read_fact_para(HW_FEATURE_FWKEY, (GS_FACTORYPARA*)factParaBuf, &buf);
	sprintf(page, "%d", buf);

    if (len <= off+count) 
		*eof = 1;
    *start = page + off;
    len -= off;
    if (len>count) 
		len = count;
    if (len<0) 
		len = 0;

    return (len);
}

static int board_factcfg_vendor_proc_read (char *page, char **start, off_t off, int count, int *eof, void *data)
{
    int len = 0;

	if ((int)data == 0)
		len = read_fact_para(HW_FEATURE_VENDOR, (GS_FACTORYPARA*)factParaBuf, page);
	else
		len = read_fact_para(HW_FEATURE_VENDOR_FULLNAME, (GS_FACTORYPARA*)factParaBuf, page);

	if (len <= off+count) 
		*eof = 1;
	*start = page + off;
	len -= off;
	if (len>count) 
		len = count;
	if (len<0) 
		len = 0;

	return (len);
}

static int board_factcfg_wan_proc_read (char *page, char **start, off_t off, int count, int *eof, void *data)
{
    int len = 0;
	char buf;

	len = read_fact_para(HW_FEATURE_HAS_WAN, (GS_FACTORYPARA*)factParaBuf, &buf);
	sprintf(page, "%d", buf);

	if (len <= off+count) 
		*eof = 1;
	*start = page + off;
	len -= off;
	if (len>count) 
		len = count;
	if (len<0) 
		len = 0;

	return (len);
}

static int
gxv_board_timecheck_proc_read (
    char            *page,
    char            **start,
    off_t           off,
    int             count,
    int             *eof,
    void            *data)
{
    /* simple printk */
    printk("--------------->uptime\n");
    return 0;
}
static int sec_feature_proc_open(struct inode *inode, struct file *file)
{
	unsigned short  *featurePtr, idx;
	void *data = PDE(inode)->data;
	unsigned int delta, fact_len;
	GS_FACTORYPARA *factPara;

	if (factParaBuf != NULL) {
		vfree(factParaBuf);
		factParaBuf=NULL;
	}

	/* refresh factParaBuf */
	if (factcfg_read()) {
		pr_err("factcfg_read failed!\n");
		return 0;
	}
	factPara = (GS_FACTORYPARA*)factParaBuf;

	fact_data_size = 0;
	featurePtr = (unsigned short*)factPara->hw_feature_table_word + 3;
	fact_len = factPara->length[1] << 16 | factPara->length[0];

	while (HW_SECURITY_TABLE_MAGIC != *(featurePtr))
	{
		delta = (unsigned int)((char*)featurePtr - (char*)factPara);
		if((delta >= fact_len) || (delta >= GS_FACTORY_CFG_SIZE)) {
			pr_info("No security table!\n");
			goto no_sec_table;
		}
		featurePtr += *(featurePtr + 1) + 2;
	}
	idx = *(featurePtr + 2);
	featurePtr += 3;
	while (idx) {
		if(((void*)featurePtr - (void*)factPara) > GS_FACTORY_CFG_SIZE)
			return 0;
		if ((int)data == *featurePtr) {
			fact_data_size = *(featurePtr + 1);
			fact_buf = (int)(featurePtr + 2);
			break;
		} else {
			featurePtr += ((*(featurePtr + 1) + 1) >> 1) + 2;
			idx--;
		}
	}

no_sec_table:
	return seq_open(file, &gs_seq_ops);
}

static int sec_feature_proc_write(struct file *file, const char __user *buffer,
		size_t count, loff_t *pos)
{
	GS_FACTORYPARA* factPara = (GS_FACTORYPARA*) factParaBuf;
	unsigned short *featurePtr, *sec_ptr, idx;
	unsigned short sec_table_size = 0, sec_table_entries = 0;
	unsigned char *existing_features;
	unsigned int  factpara_len = 0;
	int count_feature = 0;
	void *data = PDE(file->f_path.dentry->d_inode)->data;
	if (factPara->checksum != checksum(factPara))
		return -EINVAL;

	count_feature = factPara->hw_feature_table_word[2];
	featurePtr = (unsigned short*)factPara->hw_feature_table_word + 3;
	factpara_len = sizeof(GS_FACTORYPARA) + 3*sizeof(unsigned short);

	while (HW_SECURITY_TABLE_MAGIC != *(featurePtr))
	{
		factpara_len += (*(featurePtr+1) << 1) + 2*sizeof(unsigned short);
		featurePtr += *(featurePtr + 1) + 2;
		count_feature--;
		if (count_feature == 0) {
			*featurePtr = HW_SECURITY_TABLE_MAGIC;
			*(featurePtr + 2) = 0;
			factPara->hw_feature_table_word[2] += 1;
			break;
		}
	}
	idx = *(featurePtr + 2);

	existing_features = vmalloc(PAGE_SIZE * 4);

	if (existing_features == NULL)
		return -ENOMEM;
	/* save existing entries */
	if (idx > 0) {
		sec_ptr = featurePtr + 3;
		while (idx) {
			if ((int)data != *sec_ptr) {
				int ft_len = *(sec_ptr + 1) + 2 * sizeof(unsigned short);
				if (ft_len + sec_table_size > PAGE_SIZE * 4)
				{
					break;
				}
				memcpy(existing_features + sec_table_size, (void *)sec_ptr, ft_len);
				sec_table_size += ((ft_len + 1) >> 1) * 2;
				sec_table_entries++;
				sec_ptr += ((*(sec_ptr + 1) + 1) >> 1) + 2;
				idx--;
			}
			else
			{	/* we already have one in flash, exit */
				goto finish;
			}
		}
	}
	/* copy new entry */
	if (sec_table_size + count > PAGE_SIZE * 4)
	{
		return -EINVAL;
	}
	else {
		sec_ptr = (unsigned short *)(existing_features + sec_table_size);
		*sec_ptr = (int)data;
		*(sec_ptr + 1) = count;
		if (copy_from_user((void *)(sec_ptr + 2), buffer, count))
		{
			return -EFAULT;
		}
		/* 2 bytes align */
		sec_table_size += ((count + 1) >> 1) * 2 + 2 * sizeof(unsigned short);
		sec_table_entries++;
	}

	/* fill table */
	memcpy((void *)(featurePtr + 3), existing_features, sec_table_size);
	sec_table_size += 3 * sizeof(unsigned short);
	*(featurePtr + 1) = sec_table_size >> 1;
	*(featurePtr + 2) = sec_table_entries;
	factpara_len += sec_table_size;
	factPara->length[0] = (unsigned short)factpara_len;
	factPara->length[1] = (unsigned short)(factpara_len >> 16);
	factPara->checksum = checksum(factPara);
	factcfg_write();
finish:
	vfree(existing_features);
	return (count);
}

static struct file_operations sec_proc_ops = {
	.owner = THIS_MODULE,
	.open = sec_feature_proc_open,
	.read = seq_read,
	.write = sec_feature_proc_write,
	.llseek = seq_lseek,
	.release = seq_release
};

/*************************************************************************
 *Install the /proc entry for factory data
 ************************************************************************/
static int install_proc_entry (void)
{
	struct proc_dir_entry *gxvboard_root_dir;
	struct proc_dir_entry *ent ,*sent;
	struct proc_dir_entry *dent = NULL;

	gxvboard_root_dir = proc_mkdir("gxvboard", NULL);
	if(gxvboard_root_dir)
	{		
		dent = proc_mkdir("dev_info", gxvboard_root_dir);
	}

	if (dent != NULL)
	{
	
	ent = create_proc_entry("uptime", S_IFREG|S_IRUGO, gxvboard_root_dir);
	if (ent) {
		ent->read_proc	= gxv_board_timecheck_proc_read;
		ent->data		= NULL;
	}

        ent = create_proc_entry("dev_data", S_IFREG|S_IRUGO, dent);
        if (ent)
        {
		ent->proc_fops = &fact_data_proc_ops;
		ent->data = NULL;
        }

        ent = create_proc_entry("dev_alias", S_IFREG|S_IRUGO, dent);
        if (ent) 
		{
            ent->read_proc  = board_factcfg_hwmodel_proc_read;
            ent->data       = (void *)0;
        }
        ent = create_proc_entry("dev_base", S_IFREG|S_IRUGO, dent);
        if (ent) 
		{
            ent->read_proc  = board_factcfg_hwmodel_proc_read;
            ent->data       = (void *)1;
        }
        ent = create_proc_entry("dev_id", S_IFREG|S_IRUGO, dent);
        if (ent) 
		{
            ent->read_proc  = board_factcfg_hwmodel_proc_read;
            ent->data       = (void *)2;
        }

        ent = create_proc_entry("dev_rev", S_IFREG|S_IRUGO, dent);
        if (ent) 
		{
            ent->read_proc  = board_factcfg_hwrev_proc_read;
            ent->data       = NULL;
        }

        ent = create_proc_entry("fwkey", S_IFREG|S_IRUGO, dent);
        if (ent) 
		{
            ent->read_proc  = board_factcfg_fwkey_proc_read;
            ent->data       = NULL;
        }

        ent = create_proc_entry("vendor", S_IFREG|S_IRUGO, dent);
        if (ent) 
		{
            ent->read_proc  = board_factcfg_vendor_proc_read;
            ent->data       = (void *)0;
        }
        ent = create_proc_entry("vendor_fullname", S_IFREG|S_IRUGO, dent);
        if (ent) 
		{
            ent->read_proc  = board_factcfg_vendor_proc_read;
            ent->data       = (void *)1;
        }

        ent = create_proc_entry("oem_id", S_IFREG|S_IRUGO, dent);
        if (ent) 
		{
            ent->read_proc  = board_factcfg_oemid_proc_read;
            ent->data       = NULL;
        }

        ent = create_proc_entry("dev_mac", S_IFREG|S_IRUGO, dent);
        if (ent) 
		{
            ent->read_proc  = board_factcfg_dev_mac_proc_read;
			ent->write_proc = board_factcfg_dev_mac_proc_write;
            ent->data       = NULL;
        }

        ent = create_proc_entry("PN", S_IFREG|S_IRUGO, dent);
        if (ent) 
	{
            ent->read_proc  = board_factcfg_pn_proc_read;
            ent->data       = NULL;
        }
        
        ent = create_proc_entry("have_WAN", S_IFREG|S_IRUGO, dent);
        if (ent) 
        {
            ent->read_proc  = board_factcfg_wan_proc_read;
            ent->data       = NULL;
        }
		sent = proc_mkdir("security", dent);
		if (sent) {
			ent = create_proc_entry("private_key", S_IFREG|S_IRUGO|S_IWUGO, sent);
			if (ent) {
				ent->proc_fops = &sec_proc_ops;
				ent->data = (void *)0;
			}
			ent = create_proc_entry("certificate", S_IFREG|S_IRUGO|S_IWUGO, sent);
			if (ent) {
				ent->proc_fops = &sec_proc_ops;
				ent->data = (void *)1;
			}
			ent = create_proc_entry("checksum", S_IFREG|S_IRUGO|S_IWUGO, sent);
			if (ent) {
				ent->proc_fops = &sec_proc_ops;
				ent->data = (void *)2;
			}
		}
		hwf_pentrys_create(dent);
	}
	else
	{
		printk("Create proc dir error!\n");
	}
	
	return 0;
}

static int __init
factcfg_init(void)
{
	int rval;
	GS_FACTORYPARA* factPara;

	rval = factcfg_read();
	if (rval)
	{
		printk("Load factory data failed!\n");
	}
	else
	{
		 factPara = (GS_FACTORYPARA*) factParaBuf;
		if (factPara->checksum == checksum(factPara))
		{
			factparam2ihwi(factPara);
			printk("factory data loaded.\n");
		}
		else
		{
			printk("factory data invalid checksum.\n");
		}
	}

	if(-ENOMEM!=rval) install_proc_entry();
	return 0;
}

static void __exit
factcfg_cleanup(void)
{
	if(factParaBuf)
	{
		vfree(factParaBuf);
		factParaBuf = NULL;
	}
	return;
}

module_init(factcfg_init);
module_exit(factcfg_cleanup);

MODULE_AUTHOR("ygzhang@grandstream.com");
MODULE_DESCRIPTION("GS factory cfg driver");
MODULE_LICENSE("GPL");
