#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/delay.h>
#include <linux/rslib.h>
#include <linux/moduleparam.h>
#include <linux/platform_device.h>
#include <linux/dma-mapping.h>
#include <linux/types.h>
#include <linux/clk.h>
#include <linux/interrupt.h>
#include <linux/semaphore.h>
#include <linux/io.h>
#include <linux/slab.h>
#include <linux/ccu.h>
#include <linux/completion.h>
#include <linux/uaccess.h>
#include <linux/proc_fs.h>
#include <linux/vmalloc.h>

#include <mach/gxp22xx.h>
#include "mmc_rw.h"

/* from 34817 reserve 512 sectors for extra image header,
 such as 'lang' image
 */
#define BCD_SECTOR_OFFSET (35329)
#define BCD_SECTOR_NUM (64)
#define aligned_sector(x) (x?((x-1)>>9) +1:0)

struct bootloader_message {
    char command[32];
    char status[32];
    char recovery[1024];
};

int
bcd_proc_read_mmc (
    char            *page,
    char            **start,
    off_t           off,
    int             count,
    int             *eof,
    void            *data)
{
	int len;
	int rval;
	struct bootloader_message *pmsg;

	int sec_n= aligned_sector(sizeof(struct bootloader_message));
	if(sec_n > BCD_SECTOR_NUM)
	{
		return EINVAL;
	}

	pmsg = (struct bootloader_message *)vmalloc(sec_n*512);
	if(!pmsg)	return (-ENOMEM);
	
	memset(pmsg, 0, sec_n*512);

	rval = emmc_read(BCD_SECTOR_OFFSET, sec_n, (char *)pmsg, sec_n*512);
	if(0 < rval)
	{
		vfree(pmsg);
		return EIO;
	}
	

	memcpy(page, pmsg, sizeof(struct bootloader_message));
	len = sizeof(struct bootloader_message);
	if (len <= off+count) *eof = 1;
	*start = page + off;
	len -= off;
	if (len>count) len = count;
	if (len<0) len = 0;

	vfree(pmsg);
	return (len);
}

int
bcd_proc_write_mmc (
    struct file     *file,
    const char      *buffer,
    unsigned long   count,
    void            *data)
{
	int rval = 0;
	struct bootloader_message *pmsg;
	int sec_n= aligned_sector(sizeof(struct bootloader_message));

	if (count > sizeof(struct bootloader_message)) return (-EINVAL);

	pmsg = (struct bootloader_message *)vmalloc(sec_n*512);
	if(!pmsg)	return (-ENOMEM);

	memset(pmsg, 0, sec_n*512);
	if (copy_from_user(pmsg, buffer, count))
	{
		vfree(pmsg);
		return (-EFAULT);
	}

	rval = emmc_write(BCD_SECTOR_OFFSET, sec_n, (char *)pmsg, sec_n*512);
	if(rval < 0)
	{
		vfree(pmsg);
		return -EIO;
	}

	vfree(pmsg);

	return (count);
}


int create_bcd_proc_emmc(void)
{
	struct proc_dir_entry	*ent;

	ent = create_proc_entry("bootloader_bcd", S_IFREG|S_IRUGO, NULL);
	if(!ent)
	{
		printk("can not create proc node\n");
		return -1;
	}
	ent->read_proc	= bcd_proc_read_mmc;
	ent->write_proc = bcd_proc_write_mmc;
	return 0;
}

static int __init bcd_init(void)
{
		int storage = gs_get_storage();
		int ret = 0;
		
		switch(storage)
		{

			case GS_STORAGE_EMMC:
#ifdef CONFIG_GS_STORAGE_EMMC
				printk("provision emmc storage\n");
				ret = create_bcd_proc_emmc();
#endif
				break;
			case GS_STORAGE_FLASH:
			default:
				printk("not support BCD in the type\n");
				ret = -1;
				break;
		}

	return ret;
}

static void __exit bcd_exit(void)
{
	
}

module_init(bcd_init);
module_exit(bcd_exit);

