#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/spi/spi.h>
#include <linux/spi/spi_dmw96.h>
#include <mach/gpio.h>
#include <linux/gpio_keys.h>
#include <linux/input.h>
#include <sound/si3050.h>
#include "gpio-setup.h"
#include "gs_misc.h"

static __initdata struct gpio_setup_t gpio_setup_default[] = {
	GPIO_SETUP(C, 0,  GPIO_OUT_HIGH,  GPIO_PULL_DISABLE ), /* WAN_LED */
	GPIO_SETUP(C, 1,  GPIO_OUT_HIGH,  GPIO_PULL_DISABLE ), /* FXS_LED0 */
	GPIO_SETUP(C, 2,  GPIO_OUT_HIGH,  GPIO_PULL_DISABLE ), /* FXS_LED1 */
	GPIO_SETUP(C, 3,  GPIO_OUT_HIGH,  GPIO_PULL_DISABLE ), /* BRI_LED1 */
	GPIO_SETUP(C, 4,  GPIO_OUT_LOW,  GPIO_PULL_DISABLE ), /* BRI0_NTPOW */
	GPIO_SETUP(C, 5,  GPIO_OUT_LOW,  GPIO_PULL_DISABLE ), /* BRI1_NTPOW */
	GPIO_SETUP(C, 10, GPIO_OUT_LOW,  GPIO_PULL_DISABLE ), /* BRI1_MODE */

	GPIO_SETUP(C, 15, GPIO_OUT_HIGH,  GPIO_PULL_DISABLE ), /* DCDC_ON */
	GPIO_SETUP(C, 16, GPIO_OUT_HIGH,  GPIO_PULL_DISABLE ), /* WAN_LED_RJ45 */
	GPIO_SETUP(C, 17, GPIO_OUT_HIGH,  GPIO_PULL_DISABLE ), /* LAN_LED */

	GPIO_SETUP(C, 18, GPIO_OUT_HIGH,  GPIO_PULL_DISABLE ), /* BRI_LED0 */
	GPIO_SETUP(C, 19, GPIO_OUT_HIGH,  GPIO_PULL_DISABLE ), /* LAN_LED_RJ45 */
	GPIO_SETUP(C, 20, GPIO_IN,        GPIO_PULL_DISABLE ), /* POE_/DC_DET*/
    GPIO_SETUP(C, 21, GPIO_OUT_HIGH,  GPIO_PULL_DISABLE ), /* FAN_MODE*/
	GPIO_SETUP(C, 22, GPIO_OUT_LOW,   GPIO_PULL_DISABLE ), 	/* CD_LCD */
	GPIO_SETUP(C, 23, GPIO_OUT_HIGH,  GPIO_PULL_DISABLE ), 	/* /RST_LCD */
	GPIO_SETUP(C, 27, GPIO_OUT_HIGH,  GPIO_PULL_DISABLE ), /* /ISDN_RST */

	GPIO_SETUP(D, 0,  GPIO_OUT_HIGH,  GPIO_PULL_DISABLE ), /* SD_LED */
	GPIO_SETUP(D, 1,  GPIO_OUT_HIGH,  GPIO_PULL_DISABLE ), /* USB_LED */
	GPIO_SETUP(D, 2,  GPIO_OUT_LOW,   GPIO_PULL_DISABLE ), /* BRI1_TERM */
	
	GPIO_SETUP(F, 2,  GPIO_IN,        GPIO_PULL_DISABLE ), /* KEY_DOWN */
	GPIO_SETUP(F, 3,  GPIO_IN,        GPIO_PULL_DISABLE ), /* KEY_OK */
	GPIO_SETUP(F, 4,  GPIO_OUT_LOW,   GPIO_PULL_DISABLE ), /* BRI0_TERM */
	GPIO_SETUP(F, 5,  GPIO_OUT_LOW,   GPIO_PULL_DISABLE ), /* BRI0_MODE */
	GPIO_SETUP(F, 6,  GPIO_OUT_HIGH,  GPIO_PULL_DISABLE ), /* LED_CTRL */
	GPIO_SETUP(F, 7,  GPIO_OUT_LOW,   GPIO_PULL_DISABLE ), /* /USB1_EN */

	GPIO_SETUP(F, 16, GPIO_OUT_HIGH,  GPIO_PULL_DISABLE ), /* /RST_SVIP */
	GPIO_SETUP(F, 17, GPIO_IN,        GPIO_PULL_DISABLE ), /* USB1_FAULT */
	GPIO_SETUP(F, 18, GPIO_OUT_HIGH,  GPIO_PULL_DISABLE ), /* /RST_FXS */
	GPIO_SETUP(F, 19, GPIO_IN,        GPIO_PULL_DISABLE ), /* FAN_CONT */

	GPIO_SETUP(F, 21, GPIO_IN,        GPIO_PULL_DISABLE ), /* FAN_DET */
	GPIO_SETUP(F, 22, GPIO_IN,        GPIO_PULL_DISABLE ), /* /ETH_INT */
	GPIO_SETUP(F, 23, GPIO_IN,        GPIO_PULL_DISABLE ), /* /FXS_INT */
	GPIO_SETUP(F, 24, GPIO_IN,        GPIO_PULL_DISABLE ), /* /ISDN_INT */
	GPIO_SETUP(F, 25, GPIO_IN,        GPIO_PULL_DISABLE ), /* /SVIPA_INT */
	GPIO_SETUP(F, 26, GPIO_IN,        GPIO_PULL_DISABLE ), /* /SD_CD */
	GPIO_SETUP(F, 27, GPIO_IN,        GPIO_PULL_DISABLE ), /* /RTC_INT */
	GPIO_SETUP(F, 28, GPIO_OUT_HIGH,  GPIO_PULL_DISABLE ), /* ISDN_SPI2_CS*/
	GPIO_SETUP(F, 29, GPIO_OUT_HIGH,  GPIO_PULL_DISABLE ), /* SPI2_/CS_FXS */
	GPIO_SETUP(F, 30, GPIO_IN,        GPIO_PULL_DISABLE ), /* /SD_WP */
	GPIO_SETUP(F, 31, GPIO_IN,        GPIO_PULL_DISABLE ), /* /LOW_OUT */
};

static struct gsboard_gpio board_gpio[] = {
    {
        .gpio_name = "watchdog",
        .gpio_pin  = GPIO_PORTF(1),
        .gpio_dir  = 1,
        .skip_init = 1,
        .init_val  = 1,
        .pwm_value = 0
    },
    {
        .gpio_name = "svipA_reset",
        .gpio_pin  = GPIO_PORTF(16),
        .gpio_dir  = 1,
        .skip_init = 0,
        .init_val  = 1,
        .pwm_value = 0
    },
    {
        .gpio_name = "lcd_backlight",
        .gpio_pin  = GPIO_PORTF(6),
        .gpio_dir  = 1,
        .skip_init = 0,
        .init_val  = 1,
        .pwm_value = 0
    },
    {
        .gpio_name = "nUSB1_enable",
        .gpio_pin  = GPIO_PORTF(7),
        .gpio_dir  = 1,
        .skip_init = 0,
        .init_val  = 0,
        .pwm_value = 0
    },

	/* BRI LEDs */
	{ "bri_led0",  GPIO_PORTC(18), 1, 0, 1, 0 },
	{ "bri_led1",  GPIO_PORTC(3), 1, 0, 1, 0 },

    /* BRI mode*/
	{"bri0_term",   GPIO_PORTF(4),  1, 0, 0, 0 },
	{"bri0_mode",   GPIO_PORTF(5),  1, 0, 0, 0 },
	{"bri0_ntpow",  GPIO_PORTC(4),  1, 0, 0, 0 },
	{"bri1_term",   GPIO_PORTD(2),  1, 0, 0, 0 },
	{"bri1_mode",   GPIO_PORTC(10), 1, 0, 0, 0 },
	{"bri1_ntpow",  GPIO_PORTC(5),  1, 0, 0, 0 },
			
	/* DCDC ON*/
	{"dcdc_on",	    GPIO_PORTC(15),	1, 0, 0, 0 },

    /* FXS LEDs*/
    { "fxs_led0", GPIO_PORTC(1),  1, 0, 1, 0 },
    { "fxs_led1", GPIO_PORTC(2),  1, 0, 1, 0 },

    { "led_sd",   GPIO_PORTD(0),  1, 0, 0, 0 },
    { "led_usb",  GPIO_PORTD(1),  1, 0, 0, 0 },
    { "led_lan_rj45",  GPIO_PORTC(19), 1, 0, 0, 0 },
    { "led_wan_rj45",  GPIO_PORTC(16), 1, 0, 0, 0 },
    { "led_lan",  GPIO_PORTC(17), 1, 0, 0, 0 },
    { "led_wan",  GPIO_PORTC(0), 1, 0, 0, 0 },
    { "fan_cont", GPIO_PORTC(21), 1, 0, 0, 0 },

    /* Dummy GPIO dump*/
    { "gpio_dump", 0, 1, 0, 1, 0 },
};

/*
********************************************************
* platform devices
********************************************************
*/
static int xhfc_leds0[] = {
	GPIO_PORTC(18),
	GPIO_PORTC(3),
};
static int xhfc_gpio[] = {
	GPIO_PORTF(4),    //bri0_term
	GPIO_PORTF(5),    //bri0_mod
	GPIO_PORTC(4),    //bri0_ntpow
	GPIO_PORTD(2),    //bri1_term
	GPIO_PORTC(10),   //bri1_mod
	GPIO_PORTC(5),    //bri1_ntpow
	GPIO_PORTC(15),   //dcdc_on
};

static int si32260_leds[] = {
	GPIO_PORTC(1),
	GPIO_PORTC(2),
};

static struct xhfc_platform_data xhfc_pdata = {
	.reset 	= GPIO_PORTC(27),
	.chips	= 1,
	.led_count	= 2,
	.led_gpio	= xhfc_leds0,
	.isdn_gpio	= xhfc_gpio,
	.irqno		= DMW_IID_EXTERNAL_REQUEST_8,
};

static struct si3050_platform_data si32260_pdata = {
    .reset 	= GPIO_PORTF(18),
	.chips	= 1,
	.led_count	= 2,
	.led_gpio	= si32260_leds,
	.irqno		= DMW_IID_EXTERNAL_REQUEST_7,
};

static struct gs_lcd_platform_data lcd209B_pdata = {
    .reset = GPIO_PORTC(23),
    .lcd_cd = GPIO_PORTC(22),
};

#define SPI_SPEED	( 5 * 1000 * 1000 )
static struct spi_board_info spi_devs[] = {
	{
		.modalias	= "dmw96_xhfc_spi",
		.chip_select	= 0,
		.max_speed_hz	= SPI_SPEED,
		.bus_num	= 1,
		.platform_data	= &xhfc_pdata,
	},
	{
		.modalias	= "dmw96_si3226x_spi",
		.chip_select	= 4,
		.max_speed_hz	= SPI_SPEED,
		.bus_num	= 1,
		.platform_data	= &si32260_pdata,
	},
    {  
        .modalias   = "uc1701fb",
        .chip_select = 0,
        .irq = -1,
        .max_speed_hz   = SPI_SPEED,
        .bus_num    = 0,
        .platform_data  = &lcd209B_pdata,
    }, 

};

static void
spi2_set_cs(struct dmw96_spi_info *ds_info, unsigned int chip_select, int val, int pol)
{
	int cs = 0;
//printk("SPI CS called -- chip_select=%d val=%d pol=%d\n", chip_select, val, pol);
	switch (chip_select)
	{
		default:
		case 0:
			cs = GPIO_PORTF(28);
			break;
		case 4:
			cs = GPIO_PORTF(29);
			break;
	}

	gpio_set_value(cs , !val);

//	gpio_set_value(GPIO_PORTG(19), !val);
	return;
}

static uint32_t spi2_get_cs_mask( unsigned int chip_select )
{
    switch ( chip_select )
    {
        default:
        case 0:
            return ( 1 << 28 );
        case 4:
            return ( 1 << 29 );
    }
}

static char spi2_get_cs_bank( unsigned int chip_select )
{
    return 'F';
}

static struct dmw96_spi_info dmw96_spi2_info = {
	.bus_num    = 1,
	.set_cs     = spi2_set_cs,
    .get_cs_bank = spi2_get_cs_bank,
    .get_cs_mask = spi2_get_cs_mask
};

static struct resource dmw96_spi2_resources[] = {
	{
		.start  = DMW_SPI2_BASE,
		.end    = DMW_SPI2_BASE + 0x200 - 1,
		.flags  = IORESOURCE_MEM,
	},
	{
		.start  = DMW_IID_SPI2,
		.end    = DMW_IID_SPI2,
		.flags  = IORESOURCE_IRQ,
	},
	{
		.start  = 16, /* SPI2 RX */
		.end    = 16,
		.flags  = IORESOURCE_DMA,
	},
	{
		.start  = 17, /* SPI2 TX */
		.end    = 17,
		.flags  = IORESOURCE_DMA,
	},
};

static struct platform_device spi2_device = {
	.name = "dmw96-spi",
	.id = 1,
	.num_resources = ARRAY_SIZE(dmw96_spi2_resources),
	.resource = dmw96_spi2_resources,
	.dev = {
		.platform_data = &dmw96_spi2_info
	},
};

static struct gpio_keys_button gpio_buttons[] = {
	{
		.code = KEY_F1,
		.gpio = GPIO_PORTF(0),
		.active_low = 1,
		.desc = "/MR",
		.type = EV_KEY,
		.debounce_interval = 100,
	},
	{
		.code = KEY_DOWN,
		.gpio = GPIO_PORTF(2),
		.active_low = 0,
		.desc = "KEY_DOWN",
		.type = EV_KEY,
		.debounce_interval = 100,
	},
	{
		.code = KEY_ENTER,
		.gpio = GPIO_PORTF(3),
		.active_low = 0,
		.desc = "KEY_OK",
		.type = EV_KEY,
		.debounce_interval = 100,
	},
};

static struct gpio_keys_platform_data gpio_keys_pdata = {
	.buttons = gpio_buttons,
	.nbuttons = ARRAY_SIZE(gpio_buttons),
	.poll_interval = 50,
};

static struct platform_device gpio_keys_device = {
	.name = "gpio-keys-polled",
	.id = 1,
	.dev = {
		.platform_data = &gpio_keys_pdata,
	},
};

static struct gpio_keys_button gpio_fan_event[] = {
	{
		.code = KEY_STOP,
		.gpio = GPIO_PORTF(21),
		.active_low = 1,
		.desc = "FAN_STOP",
		.type = EV_KEY,
		.debounce_interval = 100,
	},
	{
		.code = KEY_AGAIN,
		.gpio = GPIO_PORTF(19),
		.active_low = 0,
		.desc = "FAN_HIGH_TEMP",
		.type = EV_KEY,
		.debounce_interval = 100,
	},
};

static struct gpio_keys_platform_data gpio_fan_event_pdata = {
	.buttons = gpio_fan_event,
	.nbuttons = ARRAY_SIZE(gpio_fan_event),
	.poll_interval = 50,
};

static struct platform_device gpio_fan_device = {
	.name = "gpio-keys",
	.id = 1,
	.dev = {
		.platform_data = &gpio_fan_event_pdata,
	},
};

static struct blkdev_led blkdev_leds[LEDID_MAX] = {
	{
		.id = LEDID_SD,
		.gpio = GPIO_PORTD(0),
	},
	{
		.id = LEDID_USB,
		.gpio = GPIO_PORTD(1),
	},
};

static struct blkdev_led_platform_data blkdev_leds_pdata = {
	.count = ARRAY_SIZE(blkdev_leds),
	.blkdev_leds = blkdev_leds,
};

static struct platform_device blkdev_leds_device = {
	.name = "blkdev-leds",
	.id = 1,
	.dev = {
		.platform_data = &blkdev_leds_pdata,
	},
};

static struct platform_device *platform_devs[] __initdata = {
	&spi2_device,
	&gpio_keys_device,
	&gpio_fan_device,
	&blkdev_leds_device,
};

int __init board_ucm6202_init(unsigned int hwrev)
{
	/* Setup GPIOs */
	dmw_gpio_setup(gpio_setup_default, ARRAY_SIZE(gpio_setup_default));
	spi_register_board_info(spi_devs, ARRAY_SIZE(spi_devs));
	return platform_add_devices(platform_devs, ARRAY_SIZE(platform_devs));
}

void *__init ucm6202_get_board_gpio(unsigned int hwrev, int *count)
{
	if(count) *count = ARRAY_SIZE(board_gpio);
	return board_gpio;
}

