/*
 * kernel mode block device LED
 *
 * Copyright 2012, GrandStream Corporation
 * All Rights Reserved.
 * 
 * THIS SOFTWARE IS OFFERED "AS IS", AND BROADCOM GRANTS NO WARRANTIES OF ANY
 * KIND, EXPRESS OR IMPLIED, BY STATUTE, COMMUNICATION OR OTHERWISE. BROADCOM
 * SPECIFICALLY DISCLAIMS ANY IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
 * FOR A SPECIFIC PURPOSE OR NONINFRINGEMENT CONCERNING THIS SOFTWARE.
 *
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/gpio.h>
#include <linux/major.h>
#include <linux/kdev_t.h>
#include <linux/genhd.h>
#include <linux/kthread.h>
#include <linux/ctype.h>
#include <linux/platform_device.h>
#include "gs_misc.h"

typedef struct
{
	LED_ID id;/* LED ID */
	unsigned int gpio;/*LED GPIO pin*/
	int oval;
	int oval_last;
	unsigned long io;
	struct vfsmount *mnt;
	int low_active;
}blkdev_led_t;

static blkdev_led_t blkdev_leds[LEDID_MAX];
static struct timer_list timer;
static unsigned int led_tv = 20;

module_param(led_tv, uint, 0600);

static int is_sd(const char *name)
{
	const char *p;

	while(1)
	{
		p = strchr(name, '/');
		if(!p) break;
		name = p+1;
	}
	if(strncasecmp(name, "mmcblk", 6)) return 0;
	if('\0'==name[6] || '0'==name[6]) return 0;
	
	return 1;
}

static int is_usb(const char *name)
{
	const char *p;

	while(1)
	{
		p = strchr(name, '/');
		if(!p) break;
		name = p+1;
	}
	if(strncasecmp(name, "sd", 2)) return 0;
	if(!isalpha(name[2])) return 0;
	
	return 1;
}

static LED_ID devname2ledID(const char *name)
{
	if(is_sd(name))
	{
		return LEDID_SD;
	}
	else if(is_usb(name))
	{
		return LEDID_USB;
	}
	else
	{		
		return LEDID_MAX;
	}
}

static LED_ID disk2ledID(struct gendisk *disk)
{
	switch(disk->major)
	{
	case MMC_BLOCK_MAJOR:
		if(disk->first_minor<CONFIG_MMC_BLOCK_MINORS)
		{
			return LEDID_MAX;
		}
		else return LEDID_SD;
	case SCSI_DISK0_MAJOR:
		return LEDID_USB;
	default:		
		return LEDID_MAX;
	}
}

void blkdev_led_mount(struct vfsmount *mnt, const char *devname)
{
	LED_ID id;

	id = devname2ledID(devname);
	if(id<LEDID_MAX)
	{
		blkdev_leds[id].mnt = mnt;
	}

	return;
}
EXPORT_SYMBOL(blkdev_led_mount);

void blkdev_led_umount(struct vfsmount *mnt)
{
	unsigned int i;

	for(i=0; i<LEDID_MAX; i++)
	{
		if(blkdev_leds[i].mnt == mnt)
		{
			blkdev_leds[i].mnt = NULL;
			blkdev_leds[i].io = 0;
			return;
		}
	}

	return;
}
EXPORT_SYMBOL(blkdev_led_umount);

void blkdev_led_io(struct gendisk *disk)
{
	LED_ID id;

	id = disk2ledID(disk);
	if(id<LEDID_MAX)
	{
		/* set io */
		blkdev_leds[id].io = 1;
	}

	return;
}
EXPORT_SYMBOL(blkdev_led_io);

static void blkdev_led_output(void)
{
	blkdev_led_t *bdl;
	unsigned int i;

	for(i=0; i<LEDID_MAX; i++)
	{
		bdl = blkdev_leds+i;
		if(bdl->io)
		{
			/* toggle oval, clear io */
			bdl->oval = !bdl->oval;
			bdl->io = 0;
		}
		else
		{
			bdl->oval = bdl->mnt?1:0;
		}

		if(bdl->oval != bdl->oval_last)
		{
			/* update gpio output */
			if(bdl->low_active) gpio_set_value(bdl->gpio, !bdl->oval);
			else gpio_set_value(bdl->gpio, bdl->oval);
			bdl->oval_last = bdl->oval;
		}
	}

	return;
}

static void blkdev_led_timer(unsigned long data)
{
	blkdev_led_output();
	mod_timer(&timer, jiffies+msecs_to_jiffies(led_tv));
	return;
}

static int blkdev_led_add(LED_ID id, int gpio)
{
	if(id<LEDID_MAX)
	{
		blkdev_leds[id].id = id;
		blkdev_leds[id].gpio = gpio;		
	}
	return 0;
}

static int __devinit blkdev_leds_probe(struct platform_device *pdev)
{
	struct blkdev_led_platform_data *pdata = pdev->dev.platform_data;
	struct blkdev_led *bdl;
	int i;

	for (i = 0; i < pdata->count; i++) {
		bdl = pdata->blkdev_leds + i;
		blkdev_led_add(bdl->id, bdl->gpio);	
	}
	return 0;
}

static int __devexit blkdev_leds_remove(struct platform_device *pdev)
{
	return 0;
}

static struct platform_driver blkdev_leds_device_driver = {
        .probe          = blkdev_leds_probe,
        .remove         = __devexit_p(blkdev_leds_remove),
        .driver         = {
                .name   = "blkdev-leds",
                .owner  = THIS_MODULE,
        }
};

static int __init blkdev_led_init(void)
{
	int rval = 0;
	
	rval = platform_driver_register(&blkdev_leds_device_driver);
	if(!rval)
	{
		/* setup timer */
		setup_timer(&timer, blkdev_led_timer, 0);
		mod_timer(&timer, jiffies+msecs_to_jiffies(led_tv));
	}
	else
	{
		printk(KERN_ERR"register blkdev LED driver err\n");
	}

	return rval;
}

static void __exit blkdev_led_exit(void)
{
	/* delete timer */
	del_timer(&timer);

	return;
}
module_init(blkdev_led_init);
module_exit(blkdev_led_exit);

MODULE_AUTHOR("ygzhang@grandstream.com");
MODULE_DESCRIPTION("Block device LED");
MODULE_LICENSE("GPL");
