/*
 * Encoder device driver (kernel module)
 *
 * Copyright (C) 2011  On2 Technologies Finland Oy.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 --------------------------------------------------------------------------------
 --
 --
 --  Version control information, please leave untouched.
 --
 --  $RCSfile: hx280enc.c,v $
 --  $Date: 2012/04/09 18:06:49 $
 --  $Revision: 1.1 $
 --
 ------------------------------------------------------------------------------*/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/mm.h>
#include <linux/slab.h>
#include <linux/fs.h>
#include <linux/errno.h>
#include <linux/moduleparam.h>
#include <linux/interrupt.h>
#include <linux/clk.h>
#include <linux/platform_device.h>
#include <asm/io.h>
#include <linux/pci.h>
#include <asm/uaccess.h>
#include <linux/ioport.h>
#include <asm/irq.h>
#include <linux/version.h>
#include <linux/hx280enc.h>
#include <linux/pm_runtime.h>

/* module description */
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Hantro Products Oy");
MODULE_DESCRIPTION("Hantro 6280/7280/8270 Encoder driver");

#define ENC_HW_ID1                  0x62800000ul
#define ENC_HW_ID2                  0x72800000ul
#define ENC_HW_ID3                  0x82700000ul

/* and this is our MAJOR; use 0 for dynamic allocation (recommended) */
static int hx280enc_major = 0;


struct hx280enc_private_data {
	int holds_enc_lock;
};

/* here's all the must remember stuff */
typedef struct {
	char *buffer;
	unsigned int buffsize;
	unsigned long iobaseaddr;
	unsigned int iosize;
	volatile u8 *hwregs;
	unsigned int irq;

	struct semaphore enc_sem;
	struct semaphore enc_ready_sem;

	struct clk *clk;
	struct device *dev;
} hx280enc_t;

/* dynamic allocation? */
static hx280enc_t hx280enc_data;

#ifdef HX280ENC_DEBUG
static void dump_regs(unsigned long data)
{
	hx280enc_t *dev = (hx280enc_t *) data;
	int i;

	PDEBUG("Reg Dump Start\n");
	for (i = 0; i < dev->iosize; i += 4) {
		PDEBUG("\toffset %02X = %08X\n", i, readl(dev->hwregs + i));
	}
	PDEBUG("Reg Dump End\n");
}
#endif

/* VM operations */
static int hx280enc_vm_fault(struct vm_area_struct *vma, struct vm_fault *vmf)
{
	PDEBUG("hx280enc_vm_fault: problem with mem access\n");
	return VM_FAULT_SIGBUS;   /* send a SIGBUS */
}

static void hx280enc_vm_open(struct vm_area_struct *vma)
{
	PDEBUG("hx280enc_vm_open:\n");
}

static void hx280enc_vm_close(struct vm_area_struct *vma)
{
	PDEBUG("hx280enc_vm_close:\n");
}

static struct vm_operations_struct hx280enc_vm_ops = {
	open:hx280enc_vm_open,
	close:hx280enc_vm_close,
	fault:hx280enc_vm_fault,
};

/* the device's mmap method. The VFS has kindly prepared the process's
 * vm_area_struct for us, so we examine this to see what was requested.
 */
static int hx280enc_mmap(struct file *filp, struct vm_area_struct *vma)
{
	int result = -EINVAL;

	result = -EINVAL;

	vma->vm_ops = &hx280enc_vm_ops;

	return result;
}

static long
hx280enc_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	int err = 0;

	struct hx280enc_private_data *pdata = (struct hx280enc_private_data*)filp->private_data;

	PDEBUG("ioctl cmd 0x%08ux\n", cmd);
	/*
	 * extract the type and number bitfields, and don't encode
	 * wrong cmds: return ENOTTY (inappropriate ioctl) before access_ok()
	 */
	if(_IOC_TYPE(cmd) != HX280ENC_IOC_MAGIC)
		return -ENOTTY;
	if(_IOC_NR(cmd) > HX280ENC_IOC_MAXNR)
		return -ENOTTY;

	/*
	 * the direction is a bitmask, and VERIFY_WRITE catches R/W
	 * transfers. `Type' is user-oriented, while
	 * access_ok is kernel-oriented, so the concept of "read" and
	 * "write" is reversed
	 */
	if(_IOC_DIR(cmd) & _IOC_READ)
		err = !access_ok(VERIFY_WRITE, (void *) arg, _IOC_SIZE(cmd));
	else if(_IOC_DIR(cmd) & _IOC_WRITE)
		err = !access_ok(VERIFY_READ, (void *) arg, _IOC_SIZE(cmd));
	if(err)
		return -EFAULT;

	switch (cmd) {
	case HX280ENC_IOC_ENC_LOCK:
		pdata->holds_enc_lock = 1;
		err = down_interruptible(&hx280enc_data.enc_sem);
		if (err) {
			printk(KERN_ERR "HX280ENC_IOC_ENC_LOCK: error in down_interruptible\n");
			return err;
		}
		/* unset enc_ready_sem (keep synchronization for lock->enable->wait calls) */
		sema_init(&hx280enc_data.enc_ready_sem, 0);
		break;
	case HX280ENC_IOC_ENC_UNLOCK:
		up(&hx280enc_data.enc_sem);
		pdata->holds_enc_lock = 0;
		break;
	case HX280ENC_IOC_WAIT_ENC:
		err = down_interruptible(&hx280enc_data.enc_ready_sem);
		return err;
		break;
	case HX280ENC_IOCGHWOFFSET:
		__put_user(hx280enc_data.iobaseaddr, (unsigned long *) arg);
		break;

	case HX280ENC_IOCGHWIOSIZE:
		__put_user(hx280enc_data.iosize, (unsigned int *) arg);
		break;
	}
	return 0;
}

static int hx280enc_open(struct inode *inode, struct file *filp)
{
	int result = 0;

	struct hx280enc_private_data *pdata = (struct hx280enc_private_data*)kmalloc(sizeof(struct hx280enc_private_data), GFP_KERNEL);
	if (pdata == NULL) {
		printk(KERN_ERR "error: cannot allocate pdata\n");
		return -1;
	}

	pdata->holds_enc_lock = 0;
	filp->private_data = (void*)pdata;

	pm_runtime_get_sync(hx280enc_data.dev);

	PDEBUG("dev opened\n");
	return result;
}

static int hx280enc_release(struct inode *inode, struct file *filp)
{
#ifdef HX280ENC_DEBUG
	//    hx280enc_t *dev = (hx280enc_t *) filp->private_data;

	//    dump_regs((unsigned long) dev); /* dump the regs */
#endif

	struct hx280enc_private_data *pdata = (struct hx280enc_private_data*)filp->private_data;

	if (pdata->holds_enc_lock) {
		// raise dec semaphore
		up(&hx280enc_data.enc_sem);
	}

	kfree(filp->private_data);

	pm_runtime_put(hx280enc_data.dev);

	PDEBUG("dev closed\n");
	return 0;
}

/* VFS methods */
static struct file_operations hx280enc_fops = {
	mmap:		hx280enc_mmap,
	open:		hx280enc_open,
	release:	hx280enc_release,
	unlocked_ioctl:	hx280enc_ioctl,
};

static irqreturn_t hx280enc_isr(int irq, void *dev_id)
{
	hx280enc_t *dev = (hx280enc_t *) dev_id;
	u32 irq_status;

	irq_status = readl(dev->hwregs + 0x04);

	if(irq_status & 0x01) {

		writel(irq_status & (~0x01), dev->hwregs + 0x04);   /* clear enc IRQ */

		up(&hx280enc_data.enc_ready_sem);

		PDEBUG("IRQ handled!\n");
		return IRQ_HANDLED;
	} else {
		PDEBUG("IRQ received, but NOT handled!\n");
		return IRQ_NONE;
	}

}

static void ResetAsic(hx280enc_t * dev)
{
	int i;

	writel(0, dev->hwregs + 0x38);

	for(i = 4; i < dev->iosize; i += 4) {
		writel(0, dev->hwregs + i);
	}
}

static int hx280enc_suspend(struct device *dev)
{
	writel(0, hx280enc_data.hwregs + 0x38);  /* disable HW */
	writel(0, hx280enc_data.hwregs + 0x04); /* clear enc IRQ */

	clk_disable(hx280enc_data.clk);
	return 0;
}

static int hx280enc_resume(struct device *dev)
{
	clk_enable(hx280enc_data.clk);
	ResetAsic(&hx280enc_data);  /* reset hardware */

	return 0;
}

static int __init hx280enc_probe(struct platform_device *pdev)
{
	struct resource *res;
	struct clk *clk;
	int result = -EBUSY;
	int irq;
	unsigned long hwid;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res)
		return -EINVAL;

	irq = platform_get_irq(pdev, 0);

	clk = clk_get(&pdev->dev, NULL);
	if (IS_ERR(clk)) {
		dev_err(&pdev->dev, "no clock for device\n");
		return -ENOENT;
	}

	if (!request_mem_region(res->start, res->end - res->start + 1, "hx280enc")) {
		dev_err(&pdev->dev, "failed to reserve HW regs\n");
		goto err_put_clk;
	}

	hx280enc_data.iobaseaddr = res->start;
	hx280enc_data.iosize = res->end - res->start + 1;
	hx280enc_data.irq = irq;
	hx280enc_data.dev = &pdev->dev;
	hx280enc_data.clk = clk;
	hx280enc_data.hwregs =
		(volatile u8 *) ioremap_nocache(hx280enc_data.iobaseaddr,
				hx280enc_data.iosize);

	if (hx280enc_data.hwregs == NULL) {
		dev_err(&pdev->dev, "failed to ioremap HW regs\n");
		result = -ENOMEM;
		goto err_release_mem;
	}

	/*
	 * Default state is that the hardware is powered off. Wake it up...
	 */
	pm_runtime_enable(&pdev->dev);
	pm_runtime_resume(&pdev->dev);
#ifndef CONFIG_PM_RUNTIME
	hx280enc_resume(&pdev->dev);
#endif

	/* check for encoder HW ID */
	hwid = readl(hx280enc_data.hwregs) & 0xFFFF0000ul;
	if (hwid != ENC_HW_ID1 && hwid != ENC_HW_ID2 && hwid != ENC_HW_ID3) {
		dev_err(&pdev->dev, "HW not found at 0x%08lx\n",
			hx280enc_data.iobaseaddr);
#ifdef HX280ENC_DEBUG
		dump_regs((unsigned long) &hx280enc_data);
#endif
		result = -ENODEV;
		goto err_unmap;
	}

	dev_info(&pdev->dev, "HW at base <0x%08lx> with ID <0x%08lx>\n",
		 hx280enc_data.iobaseaddr, hwid);

	/* get the IRQ line */
	if (irq >= 0) {
		result = request_irq(irq, hx280enc_isr, IRQF_SHARED, "hx280enc",
				     (void *) &hx280enc_data);
		if (result)
			goto err_unmap;
	} else {
		dev_info(&pdev->dev, "IRQ not in use!\n");
	}

	/* initialize lock and ready semaphores */
	sema_init(&hx280enc_data.enc_sem, 1);
	sema_init(&hx280enc_data.enc_ready_sem, 0);

	result = register_chrdev(hx280enc_major, "hx280enc", &hx280enc_fops);
	if (result < 0) {
		dev_err(&pdev->dev, "unable to get major <%d>\n", hx280enc_major);
		goto err_release_irq;
	} else if (result != 0) {   /* this is for dynamic major */
		hx280enc_major = result;
	}

	dev_info(&pdev->dev, "driver loaded. Major <%d>\n", hx280enc_major);
	return 0;

err_release_irq:
	if (irq >= 0)
		free_irq(irq, (void *)&hx280enc_data);
err_unmap:
	hx280enc_suspend(&pdev->dev);
	pm_runtime_disable(&pdev->dev);
	iounmap(hx280enc_data.hwregs);
err_release_mem:
	release_mem_region(hx280enc_data.iobaseaddr, hx280enc_data.iosize);
err_put_clk:
	clk_put(clk);
	return result;
}

static int __exit hx280enc_remove(struct platform_device *pdev)
{
	unregister_chrdev(hx280enc_major, "hx280enc");
	if (hx280enc_data.irq >= 0)
		free_irq(hx280enc_data.irq, (void *)&hx280enc_data);
	pm_runtime_disable(&pdev->dev);
	iounmap(hx280enc_data.hwregs);
	release_mem_region(hx280enc_data.iobaseaddr, hx280enc_data.iosize);
	clk_put(hx280enc_data.clk);

	return 0;
}

static struct dev_pm_ops hx280enc_pm_ops = {
	.runtime_suspend = hx280enc_suspend,
	.runtime_resume = hx280enc_resume,
};

static struct platform_driver hx280enc_driver = {
	.driver = {
		.name = "hx280enc",
		.owner = THIS_MODULE,
		.pm = &hx280enc_pm_ops,
	},
	.remove = __exit_p(hx280enc_remove),
};

static int __init hx280enc_init(void)
{
	return platform_driver_probe(&hx280enc_driver, hx280enc_probe);
}

static void __exit hx280enc_cleanup(void)
{
	platform_driver_unregister(&hx280enc_driver);
}

module_init(hx280enc_init);
module_exit(hx280enc_cleanup);
