/*
 * Decoder device driver (kernel module)
 *
 * Copyright (C) 2009  Hantro Products Oy.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 --------------------------------------------------------------------------------
 --
 --  Version control information, please leave untouched.
 --
 --  $RCSfile: hx170dec.c,v $
 --  $Date: 2012/04/09 18:06:49 $
 --  $Revision: 1.1 $
 --
 ------------------------------------------------------------------------------*/

#include <linux/fs.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/clk.h>
#include <linux/platform_device.h>
#include <asm/uaccess.h>
#include <asm/io.h>
#include <mach/hardware.h>
#include <linux/interrupt.h>
#include <linux/version.h>
#include <linux/slab.h>
#include <linux/hx170dec.h>
#include <linux/pm_runtime.h>

/* module description */
MODULE_LICENSE("GPL");
MODULE_AUTHOR("Hantro Products Oy");
MODULE_DESCRIPTION("driver module for 8170/81990 Hantro decoder/pp");

/* Decoder interrupt register */
#define X170_INTERRUPT_REGISTER_DEC     (1*4)
#define X170_INTERRUPT_REGISTER_PP      (60*4)

#define HX_DEC_INTERRUPT_BIT        0x100
#define HX_PP_INTERRUPT_BIT         0x100

static const int DecHwId[] = { 0x8190, 0x8170, 0x9170, 0x9190, 0x6731 };

struct hx170dec_private_data {
	int holds_pp_lock;
	int holds_dec_lock;
};

/* and this is our MAJOR; use 0 for dynamic allocation (recommended)*/
static int hx170dec_major = 0;

/* here's all the must remember stuff */
typedef struct
{
	char *buffer;
	unsigned long iobaseaddr;
	unsigned int iosize;
	volatile u8 *hwregs;
	int irq;

	struct semaphore pp_sem;
	struct semaphore dec_sem;
	struct semaphore pp_ready_sem;
	struct semaphore dec_ready_sem;

	struct clk *clk;
	struct device *dev;
} hx170dec_t;

static hx170dec_t hx170dec_data;    /* dynamic allocation? */

#ifdef HW_PERFORMANCE
static struct timeval end_time;
#endif

static int CheckHwId(hx170dec_t * dev)
{
	long int hwid;

	size_t numHw = sizeof(DecHwId) / sizeof(*DecHwId);

	hwid = readl(dev->hwregs);
	printk(KERN_INFO "hx170dec: HW ID=0x%08lx\n", hwid);

	hwid = (hwid >> 16) & 0xFFFF;   /* product version only */

	while(numHw--)
	{
		if(hwid == DecHwId[numHw])
		{
			printk(KERN_INFO "hx170dec: Compatible HW found at 0x%08lx\n",
					dev->iobaseaddr);
			return 1;
		}
	}

	printk(KERN_INFO "hx170dec: No Compatible HW found at 0x%08lx\n",
			dev->iobaseaddr);
	return 0;
}

/*------------------------------------------------------------------------------
  Function name   : hx170dec_isr
Description     : interrupt handler

Return type     : irqreturn_t
------------------------------------------------------------------------------*/
static irqreturn_t hx170dec_isr(int irq, void *dev_id)
{
	unsigned int handled = 0;

	hx170dec_t *dev = (hx170dec_t *) dev_id;
	u32 irq_status_dec;
	u32 irq_status_pp;

	handled = 0;

	/* interrupt status register read */
	irq_status_dec = readl(dev->hwregs + X170_INTERRUPT_REGISTER_DEC);
	irq_status_pp = readl(dev->hwregs + X170_INTERRUPT_REGISTER_PP);

	if((irq_status_dec & HX_DEC_INTERRUPT_BIT) ||
			(irq_status_pp & HX_PP_INTERRUPT_BIT))
	{

		if(irq_status_dec & HX_DEC_INTERRUPT_BIT)
		{
#ifdef HW_PERFORMANCE
			do_gettimeofday(&end_time);
#endif
			/* clear dec IRQ */
			writel(irq_status_dec & (~HX_DEC_INTERRUPT_BIT),
					dev->hwregs + X170_INTERRUPT_REGISTER_DEC);

			up(&hx170dec_data.dec_ready_sem);

			PDEBUG("decoder IRQ received!\n");
		}

		if(irq_status_pp & HX_PP_INTERRUPT_BIT)
		{
#ifdef HW_PERFORMANCE
			do_gettimeofday(&end_time);
#endif
			/* clear pp IRQ */
			writel(irq_status_pp & (~HX_PP_INTERRUPT_BIT),
					dev->hwregs + X170_INTERRUPT_REGISTER_PP);

			up(&hx170dec_data.pp_ready_sem);

			PDEBUG("pp IRQ received!\n");
		}

		handled = 1;
	}
	else
	{
		PDEBUG("IRQ received, but not x170's!\n");
	}

	return IRQ_RETVAL(handled);
}

/*------------------------------------------------------------------------------
  Function name   : ResetAsic
Description     : reset asic

Return type     :
------------------------------------------------------------------------------*/

static void ResetAsic(hx170dec_t * dev)
{
	int i;

	writel(0, dev->hwregs + 0x04);

	for(i = 4; i < dev->iosize; i += 4)
	{
		writel(0, dev->hwregs + i);
	}
}

/*------------------------------------------------------------------------------
  Function name   : hx170dec_ioctl
Description     : communication method to/from the user space

Return type     : int
------------------------------------------------------------------------------*/

static long
hx170dec_ioctl(struct file *filp, unsigned int cmd, unsigned long arg)
{
	int err = 0;
	struct hx170dec_private_data *pdata = (struct hx170dec_private_data*)filp->private_data;

#ifdef HW_PERFORMANCE
	struct timeval *end_time_arg;
#endif

	PDEBUG("ioctl cmd 0x%08ux\n", cmd);
	/*
	 * extract the type and number bitfields, and don't decode
	 * wrong cmds: return ENOTTY (inappropriate ioctl) before access_ok()
	 */
	if(_IOC_TYPE(cmd) != HX170DEC_IOC_MAGIC)
		return -ENOTTY;
	if(_IOC_NR(cmd) > HX170DEC_IOC_MAXNR)
		return -ENOTTY;

	/*
	 * the direction is a bitmask, and VERIFY_WRITE catches R/W
	 * transfers. `Type' is user-oriented, while
	 * access_ok is kernel-oriented, so the concept of "read" and
	 * "write" is reversed
	 */
	if(_IOC_DIR(cmd) & _IOC_READ)
		err = !access_ok(VERIFY_WRITE, (void *) arg, _IOC_SIZE(cmd));
	else if(_IOC_DIR(cmd) & _IOC_WRITE)
		err = !access_ok(VERIFY_READ, (void *) arg, _IOC_SIZE(cmd));
	if(err)
		return -EFAULT;

	switch (cmd)
	{
		case HX170DEC_IOC_PP_LOCK:
			pdata->holds_pp_lock = 1;
			err = down_interruptible(&hx170dec_data.pp_sem);
			if (err) {
				printk(KERN_ERR "HX170DEC_IOC_PP_LOCK: error in down_interruptible\n");
				return err;
			}
			/* unset pp_ready_sem (keep synchronization for lock->enable->wait calls) */
			sema_init(&hx170dec_data.pp_ready_sem, 0);
			return 0;
		case HX170DEC_IOC_PP_UNLOCK:
			up(&hx170dec_data.pp_sem);
			pdata->holds_pp_lock = 0;
			return 0;
		case HX170DEC_IOC_DEC_LOCK:
			pdata->holds_dec_lock = 1;
			err = down_interruptible(&hx170dec_data.dec_sem);
			if (err) {
				printk(KERN_ERR "HX170DEC_IOC_DEC_LOCK: error in down_interruptible\n");
				return err;
			}
			/* unset dec_ready_sem (keep synchronization for lock->enable->wait calls) */
			sema_init(&hx170dec_data.dec_ready_sem, 0);
			return 0;
		case HX170DEC_IOC_DEC_UNLOCK:
			up(&hx170dec_data.dec_sem);
			pdata->holds_dec_lock = 0;
			return 0;
		case HX170DEC_IOC_INIT_LOCKS:
			return 0;
		case HX170DEC_IOC_WAIT_DEC:
			err = down_interruptible(&hx170dec_data.dec_ready_sem);
			return err;
		case HX170DEC_IOC_WAIT_PP:
			err = down_interruptible(&hx170dec_data.pp_ready_sem);
			return err;
		case HX170DEC_IOC_CLI:
			disable_irq(hx170dec_data.irq);
			break;

		case HX170DEC_IOC_STI:
			enable_irq(hx170dec_data.irq);
			break;
		case HX170DEC_IOCGHWOFFSET:
			__put_user(hx170dec_data.iobaseaddr, (unsigned long *) arg);
			break;
		case HX170DEC_IOCGHWIOSIZE:
			__put_user(hx170dec_data.iosize, (unsigned int *) arg);
			break;
		case HX170DEC_PP_INSTANCE:
			break;

#ifdef HW_PERFORMANCE
		case HX170DEC_HW_PERFORMANCE:
			end_time_arg = (struct timeval *) arg;
			end_time_arg->tv_sec = end_time.tv_sec;
			end_time_arg->tv_usec = end_time.tv_usec;
			break;
#endif
	}
	return 0;
}

/*------------------------------------------------------------------------------
  Function name   : hx170dec_open
Description     : open method

Return type     : int
------------------------------------------------------------------------------*/

static int hx170dec_open(struct inode *inode, struct file *filp)
{
	struct hx170dec_private_data *pdata = (struct hx170dec_private_data*)kmalloc(sizeof(struct hx170dec_private_data), GFP_KERNEL);
	if (pdata==NULL) {
		printk(KERN_ERR "error: cannot allocate pdata\n");
		return -1;
	}

	pm_runtime_get_sync(hx170dec_data.dev);

	pdata->holds_dec_lock = 0;
	pdata->holds_pp_lock = 0;
	filp->private_data = (void*)pdata;

	PDEBUG("dev opened\n");
	return 0;
}


/*------------------------------------------------------------------------------
  Function name   : hx170dec_release
Description     : Release driver

Return type     : int
------------------------------------------------------------------------------*/

static int hx170dec_release(struct inode *inode, struct file *filp)
{

	/* hx170dec_t *dev = &hx170dec_data; */

	struct hx170dec_private_data *pdata = (struct hx170dec_private_data*)filp->private_data;

	if (pdata->holds_dec_lock) {
		// raise dec semaphore
		up(&hx170dec_data.dec_sem);
	}

	if (pdata->holds_pp_lock) {
		// raise pp semaphore
		up(&hx170dec_data.pp_sem);
	}

	kfree(filp->private_data);

	pm_runtime_put(hx170dec_data.dev);

	PDEBUG("dev closed\n");
	return 0;
}

/* VFS methods */
static struct file_operations hx170dec_fops = {
	open:hx170dec_open,
	release:hx170dec_release,
	unlocked_ioctl:hx170dec_ioctl,
};


static int hx170dec_suspend(struct device *dev)
{
	clk_disable(hx170dec_data.clk);
	return 0;
}

static int hx170dec_resume(struct device *dev)
{
	clk_enable(hx170dec_data.clk);
	ResetAsic(&hx170dec_data);  /* reset hardware */

	return 0;
}

static int __init hx170dec_probe(struct platform_device *pdev)
{
	struct resource *res;
	struct clk *clk;
	int result = -EBUSY;
	int irq;

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!res)
		return -EINVAL;

	irq = platform_get_irq(pdev, 0);

	dev_info(&pdev->dev, "dec/pp kernel module. %s \n", "$Revision: 1.1 $");
	dev_info(&pdev->dev, "supports 8170 and 8190 hardware \n");
	dev_info(&pdev->dev, "base_port=0x%08x irq=%i\n", res->start, irq);

	clk = clk_get(&pdev->dev, NULL);
	if (IS_ERR(clk)) {
		dev_err(&pdev->dev, "no clock for device\n");
		return -ENOENT;
	}

	if (!request_mem_region(res->start, res->end - res->start + 1, "hx170dec")) {
		dev_err(&pdev->dev, "failed to reserve HW regs\n");
		goto err_put_clk;
	}

	hx170dec_data.iobaseaddr = res->start;
	hx170dec_data.iosize = res->end - res->start + 1;
	hx170dec_data.irq = irq;
	hx170dec_data.clk = clk;
	hx170dec_data.dev = &pdev->dev;
	hx170dec_data.hwregs =
		(volatile u8 *) ioremap_nocache(hx170dec_data.iobaseaddr,
				hx170dec_data.iosize);

	if (hx170dec_data.hwregs == NULL) {
		dev_err(&pdev->dev, "failed to ioremap HW regs\n");
		result = -ENOMEM;
		goto err_release_mem;
	}

	/*
	 * Default state is that the hardware is powered off. Wake it up...
	 */
	pm_runtime_enable(&pdev->dev);
	pm_runtime_resume(&pdev->dev);
#ifndef CONFIG_PM_RUNTIME
	hx170dec_resume(&pdev->dev);
#endif

	/* check for correct HW */
	if (!CheckHwId(&hx170dec_data)) {
		dev_err(&pdev->dev, "hardware not found\n");
		result = -ENODEV;
		goto err_unmap;
	}

	/* get the IRQ line */
	if (irq >= 0) {
		result = request_irq(irq, hx170dec_isr, IRQF_SHARED, "hx170dec",
				     (void *) &hx170dec_data);
		if (result != 0)
			goto err_unmap;
	} else {
		dev_info(&pdev->dev, "IRQ not in use!\n");
	}

	/* initialize pp and dec semaphores */
	sema_init(&hx170dec_data.pp_sem, 1);
	sema_init(&hx170dec_data.dec_sem, 1);

	/* initialize hw ready semaphores */
	sema_init(&hx170dec_data.dec_ready_sem, 0);
	sema_init(&hx170dec_data.pp_ready_sem, 0);

	result = register_chrdev(hx170dec_major, "hx170dec", &hx170dec_fops);
	if (result < 0) {
		dev_err(&pdev->dev, "unable to get major %d\n", hx170dec_major);
		goto err_release_irq;
	} else if (result != 0) {    /* this is for dynamic major */
		hx170dec_major = result;
	}

	dev_info(&pdev->dev, "driver loaded. Major = %d\n", hx170dec_major);
	return 0;

err_release_irq:
	if (irq >= 0)
		free_irq(irq, (void *)&hx170dec_data);
err_unmap:
	hx170dec_suspend(&pdev->dev);
	pm_runtime_disable(&pdev->dev);
	iounmap(hx170dec_data.hwregs);
err_release_mem:
	release_mem_region(hx170dec_data.iobaseaddr, hx170dec_data.iosize);
err_put_clk:
	clk_put(clk);
	return result;
}

static int __exit hx170dec_remove(struct platform_device *pdev)
{
	unregister_chrdev(hx170dec_major, "hx170dec");
	if (hx170dec_data.irq >= 0)
		free_irq(hx170dec_data.irq, (void *)&hx170dec_data);
	pm_runtime_disable(&pdev->dev);
	iounmap(hx170dec_data.hwregs);
	release_mem_region(hx170dec_data.iobaseaddr, hx170dec_data.iosize);
	clk_put(hx170dec_data.clk);

	return 0;
}

static struct dev_pm_ops hx170dec_pm_ops = {
	.runtime_suspend = hx170dec_suspend,
	.runtime_resume = hx170dec_resume,
};

static struct platform_driver hx170dec_driver = {
	.driver = {
		.name = "hx170dec",
		.owner = THIS_MODULE,
		.pm = &hx170dec_pm_ops,
	},
	.remove = __exit_p(hx170dec_remove),
};

static int __init hx170dec_init(void)
{
	return platform_driver_probe(&hx170dec_driver, hx170dec_probe);
}

static void __exit hx170dec_cleanup(void)
{
	platform_driver_unregister(&hx170dec_driver);
}

module_init(hx170dec_init);
module_exit(hx170dec_cleanup);
