/*
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

//#define DEBUG
//#define VERBOSE_DEBUG

#include <linux/moduleparam.h>
#include <linux/module.h>
#include <linux/cdev.h>
#include <linux/platform_device.h>
#include <linux/gpio.h>
#include <linux/sched.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/irq.h>
#include <linux/uaccess.h>
#include <linux/switch.h>
#include <linux/proc_fs.h>
#include <linux/seq_file.h>
#include <linux/err.h>

#include <asm/io.h>
#include <mach/platform.h>
#include <mach/headsetdet.h>
#include <mach/cradledet.h>

/*****************************************************************************/

#ifdef VERBOSE_DEBUG

  #ifndef DEBUG
  #define DEBUG
  #endif
  
  #define vdbg_prt(fmt, ...) \
	printk("gpio-hsdet: %s():" fmt "\n", __func__, ##__VA_ARGS__)
#else
  #define vdbg_prt(fmt, ...)
#endif


#ifdef DEBUG
  #define dbg_prt(fmt, ...) \
	printk("gpio-hsdet: %s():" fmt "\n", __func__, ##__VA_ARGS__)
#else
  #define dbg_prt(fmt, ...)	
#endif

/*****************************************************************************/

#define DEVICE_NAME  "gpio-hsdet"

/*****************************************************************************/

static int dw_headset_get_curr_state(struct headset_dev *hdst_dev);


#ifndef CONFIG_HSDET_POLLING

/* Interrupt-mode code: */
/* -------------------  */

static void dw_change_int_polarity(struct headset_dev *hdst_dev)
{
	int hsplugged = hdst_dev->headset_det_state;
	int irq = hdst_dev->dp52_interrupt;

	/* Note: 'hsplugged' is:
	 *  0 -> Headset is Unplugged  --> conf detection of 'plug'-event
	 *  1 -> Headset is Plugged-in --> conf detection of 'unplug'-event
	 */
	irq_set_irq_type(irq,(hsplugged) ? IRQF_TRIGGER_LOW:IRQF_TRIGGER_HIGH);
}

/* If IRQF_ONESHOT used, this interrupt is disabled upon occurance of interrupt,
 and later this interrupt is enabled back upon completion of the handler */

/* This handler is the threaded irq handler */
static irqreturn_t dw_headsetdet_irq_handler(int irq, void *data)
{
	struct headset_dev *hdst_dev = data;

	/* Check the headset state */
	dw_headset_get_curr_state(hdst_dev);

	/* Change interrupt polarity -
	 * to detect the Inverse(!) plug/unplug action
	 */
	dw_change_int_polarity(hdst_dev);

	return IRQ_HANDLED;
}

#else

/* Polling-mode code: */
/* -----------------  */

/* when working in polling mode, we use this thread */
static int dw_headset_poll_thread(void *data)
{
	struct headset_dev *hdst_dev = data;

	dbg_prt();

	if (!data) {
		vdbg_prt("%s\n", "No data. Aborting!");
		return 0;
	}

	/* Remove any userspace resources (mem maps and signal handlers)
	 * and re-parent to init() task
	 */
	daemonize("dw_headset_poll_thread");
	allow_signal(SIGTERM);

	while (1) {
		/* Check whether headset state has changed */
		dw_headset_get_curr_state(hdst_dev);
		/* Sleep for awhile (130 mSec) */
		msleep_interruptible(hdst_dev->poll_period);
	}

	return 0;
}
#endif /* CONFIG_HSDET_POLLING */


static int dw_headset_debounce_state(struct headset_dev *hdst_dev)
{
	int headset_val1, headset_val2;
	int dbnc_times;
	int gpio;

	/* Note: plugging in/out the headset jack is a lengthy
	 *	 mechanical action. Debounce must be adequate !
	 */
	gpio = hdst_dev->conf->headsetdetpin;

	dbnc_times = hdst_dev->debounce_times;
	do {
		/* Read physical HeadSet state (at dp52 reg 0x44) */
		headset_val1 = gpio_get_value(gpio);

		/* Wait, for Debounce */
		msleep_interruptible(hdst_dev->debounce_period);

		/* Read physical HeadSet state again */
		headset_val2 = gpio_get_value(gpio);

		dbnc_times--;

	} while ((headset_val2 != headset_val1) && (dbnc_times > 0));

	if (headset_val2 != headset_val1) {
		/* Restore former steady-state */
		headset_val2 = hdst_dev->headset_det_state;
	}

	return headset_val2;
}

static int dw_headset_get_curr_state(struct headset_dev *hdst_dev)
{
	int headset_val;

	headset_val = dw_headset_debounce_state(hdst_dev);

	/* Compare with former states */
	if (headset_val != hdst_dev->headset_det_state) {
		hdst_dev->headset_det_state = headset_val;
		vdbg_prt("%s %i", "Change detected. New state:", headset_val);

		/* Update the sysfs headset state with that of headset */
		switch_set_state(&hdst_dev->headset_sdev,
					hdst_dev->headset_det_state);

		/* Notify the user-space */
		kobject_uevent(&hdst_dev->g_dev->kobj, KOBJ_CHANGE);

		/* State change occured - return 1 */
		return 1;
	}

	/* No state change - return 0 */
	return 0;
}

static void dw_enable_headset_detection(struct headset_dev *hdst_dev)
{
	int gpio;

	dbg_prt();

	gpio = hdst_dev->conf->headsetdetpin;

	gpio_set_enable(gpio,1);

	/* Init headset State according to the current physical state
	 * of Headset plug - Plugged-In or Unpluuged
	 */
	hdst_dev->headset_det_state = dw_headset_debounce_state(hdst_dev);

	vdbg_prt("%s %s", "Initial Headset state: ",
		 (hdst_dev->headset_det_state) ? "Plugged-In" : "Unplugged");

}

static void dw_headsetdet_configure(struct platform_device *pdev)
{
	struct headset_dev *hdst_dev = platform_get_drvdata(pdev);

	dbg_prt();

	/* Enable & Init Headset Detection */
	dw_enable_headset_detection(hdst_dev);
	
	/* Notify userspace about the state */
	switch_set_state(&hdst_dev->headset_sdev, hdst_dev->headset_det_state);

#ifdef CONFIG_HSDET_POLLING
	/* Note: Polling mode used */
	/* Launch AFE Detection thread */
	kernel_thread(dw_headset_poll_thread, hdst_dev, CLONE_KERNEL);
#endif
}

static int dw_headsetdet_dev_uevent(struct device *dev,
				    struct kobj_uevent_env *env)
{
	struct headset_dev *hdst_dev = dev_get_drvdata(dev);

	dbg_prt();

	add_uevent_var(env, "headset_Status=%d", hdst_dev->headset_det_state);
	hdst_dev->g_dev = dev;

	return 0;
}

static ssize_t dw_headsetdet_switch_print_state(struct switch_dev *sdev, char *buf)
{
	const char *state = NULL;
	struct headset_dev *hdst_dev =
			container_of(sdev, struct headset_dev, headset_sdev);

	dbg_prt();

	state = switch_get_state(sdev) ?
			hdst_dev->state_on : hdst_dev->state_off;
	if (state) {
		return sprintf(buf, "%s\n", state);
	}

	return -1;
}

/* Notify the user space, in case of a headset-state change */
static ssize_t dw_headset_notify_state_change(struct device *dev,
				 struct device_attribute *attr,
				 char *buf)
{
	struct headset_dev *hdst_dev = dev_get_drvdata(dev);

	dbg_prt();

	kobject_uevent(&dev->kobj, KOBJ_CHANGE);
	return sprintf(buf, "%d\n", hdst_dev->headset_det_state);
}

static struct device_attribute dw_headsetdet_dev_attrs[] = {
    __ATTR(hs_detect, 0644, dw_headset_notify_state_change, NULL),
    __ATTR_NULL,
};

/*****************************************************************************/

static int __devinit dw_headsetdet_probe(struct platform_device *pdev)
{
	int err = 0;
	struct gpio_switch_platform_data *pswitchdata;
	struct headset_dev *hdst_dev = NULL;

#ifndef CONFIG_HSDET_POLLING
	unsigned long irqpol;
	int irq = platform_get_irq(pdev, 0);
#endif
	dbg_prt();
#ifndef CONFIG_DMW_BOARD_GXP22XX
	/* Check whether headset-detection feature is enabled */
	if (!dmw_board_has_feature("hsdet")) {
		dev_err(&pdev->dev, "headset-detection is Disabled!\n");
		return 0;
	}
#endif

	/* Allocate memory for headset struct */
	hdst_dev = kzalloc(sizeof(struct headset_dev), GFP_KERNEL);
	if (!hdst_dev) {
		dev_err(&pdev->dev, "can't allocate private memory!\n");
		return -ENOMEM;
	}

	/* Keep pointer to This platform device (pdev)
	 * on our private-data structure
	 */
	hdst_dev->pdev = pdev;
	hdst_dev->conf=pdev->dev.platform_data;

	/* Keep pointer to our private-data structure on pdev->dev:
	 * "pdev->dev.p->driver_data = hdst_dev;"
	 */
	platform_set_drvdata(pdev, hdst_dev);

	/* Keep pointer to our private-data structure
	 * on our private-data structure's device structure
	 * "hdst_dev->dev.p->driver_data = hdst_dev;"
	 */
	dev_set_drvdata(&hdst_dev->dev, hdst_dev);

	/* Set time periods for:
	 * - polling thread (if used)
	 * - debouncing coefficients
	 */
	hdst_dev->poll_period = HSDET_POLL_PERIOD;
	hdst_dev->debounce_period = HSDET_DEBOUNCE_PERIOD;
	hdst_dev->debounce_times = HSDET_DEBOUNCE_TIMES;

	/* Create sysfs entry */
	hdst_dev->headset_class = class_create(THIS_MODULE, "headset");
	hdst_dev->headset_class->dev_attrs = dw_headsetdet_dev_attrs;
	hdst_dev->headset_class->dev_uevent = dw_headsetdet_dev_uevent;

	/* Keep poniter to our headset class
	 * on our private-data structure's device structure
	 */
	hdst_dev->dev.class = hdst_dev->headset_class;
	hdst_dev->dev.parent = &pdev->dev;
	dev_set_name(&hdst_dev->dev, "status");

	/* Register headset-detection device (as Platform device) */
	err = device_register(&hdst_dev->dev);
	if (err) {
		dev_err(&pdev->dev, "headset-det dev register failed!\n");
	   goto L_ERR_REG;
	}


	/* Update switch data */
	pswitchdata = &hdst_dev->conf->switch_platform_data;
	hdst_dev->headset_sdev.name = pswitchdata->name;
	hdst_dev->name_on = pswitchdata->name_on;
	hdst_dev->name_off = pswitchdata->name_off;
	hdst_dev->state_on = pswitchdata->state_on;
	hdst_dev->state_off = pswitchdata->state_off;
	hdst_dev->headset_sdev.print_state = dw_headsetdet_switch_print_state;

	/* Register switch device */
	err = switch_dev_register(&hdst_dev->headset_sdev);
	if (err < 0) {
		dev_err(&pdev->dev, "switch dev register failed!\n");
		goto L_ERR_SW;
	}

	/* Initialize the sysfs headset state with that of headset */
	switch_set_state(&hdst_dev->headset_sdev, hdst_dev->headset_det_state);

	/* Enable Power-Management wakeup capability */
	device_init_wakeup(&pdev->dev, 1);

	/* Enable headset detection in hw layer */
	dw_headsetdet_configure(hdst_dev->pdev);

#ifndef CONFIG_HSDET_POLLING

	hdst_dev->dp52_interrupt = irq;
	/* Note: headset state is 
	 *  0 -> Headset is Unplugged  --> conf detection of 'plug'-event
	 *  1 -> Headset is Plugged-in --> conf detection of 'unplug'-event
	 */
	irqpol = (hdst_dev->headset_det_state) ? IRQF_TRIGGER_LOW: IRQF_TRIGGER_HIGH;

	/* Note: hdst_dev will be passed to the irq Thread,
	 *	 dw_headsetdet_irq_handler(), as the second parameter.
	 */

	err = request_threaded_irq(irq, NULL, dw_headsetdet_irq_handler,
	                            irqpol | IRQF_ONESHOT , "gpio-hsdet", hdst_dev);
	if (err < 0) {
		dev_err(&pdev->dev, "Unable to request IRQ: %i\n", irq);
		goto L_ERR_IRQ;
	}

#endif /* ! CONFIG_HSDET_POLLING */

	return 0;

L_ERR_SW:

#ifndef CONFIG_HSDET_POLLING
	free_irq(irq, hdst_dev);


L_ERR_IRQ:
#endif

	device_unregister(&hdst_dev->dev);

L_ERR_REG:
	kfree(hdst_dev);

	return err;
}

static int __devexit dw_headsetdet_remove(struct platform_device *pdev)
{
	struct headset_dev *hdst_dev = platform_get_drvdata(pdev);
	int irq = platform_get_irq(pdev, 0);

	dbg_prt();

	if (hdst_dev) {
		free_irq(irq, hdst_dev);
		switch_dev_unregister(&hdst_dev->headset_sdev);
		kfree(hdst_dev);
	}

	return 0;
}

static struct platform_driver dw_headset_driver =
{
	.probe	= dw_headsetdet_probe,
	.remove	= __devexit_p(dw_headsetdet_remove),
	.driver	=
	{
		.name	= "gpio-hsdet",
		.owner	= THIS_MODULE,
	},
};

/**************************************************************************************************/
/* Module's Init & Exit */
/* -------------------- */

/* Init function */
static int __init dw_headset_init(void)
{
	dbg_prt();

	return platform_driver_register(&dw_headset_driver);
}

/* Exit function */
static void __exit dw_headset_exit(void)
{
	dbg_prt();

	return platform_driver_unregister(&dw_headset_driver);
}

module_init(dw_headset_init);
module_exit(dw_headset_exit);

MODULE_AUTHOR("Murali T.D. Mohan");
MODULE_DESCRIPTION("DMW96 Headset Detection Driver");
MODULE_LICENSE("GPL");
/**************************************************************************************************/

