/*
 *  LCD / Backlight control code for DMW96
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/mm.h>
#include <linux/gpio.h>
#include <linux/fb.h>
#include <linux/backlight.h>
#include <video/dw_bl.h>

static void dw_bl_set_backlight(struct dw_bl_data *data, int brightness)
{
    
    /* SetBacklight */
	brightness = ((brightness != 0) ? 1 : 0);
	gpio_set_value(data->dw_bl_props.backlight_gpio, brightness);
	//printk("%s(%d) \n", __func__ , __LINE__);

}

static int dw_bl_update_status(struct backlight_device *dev)
{
	struct backlight_properties *props = &dev->props;
	struct dw_bl_data *data = dev_get_drvdata(&dev->dev);
	int power = max(props->power, props->fb_blank);
	int brightness = props->brightness;

	if (power)
		brightness = 0;

	dw_bl_set_backlight(data, brightness);

	return 0;
}

static int dw_bl_get_brightness(struct backlight_device *dev)
{
	struct backlight_properties *props = &dev->props;

	return props->brightness;
}

static struct backlight_ops dw_bl_ops = {
	.get_brightness		= dw_bl_get_brightness,
	.update_status		= dw_bl_update_status,
};

static int __devinit dw_bl_probe(struct platform_device *pdev)
{
	#ifdef CONFIG_MACH_VERSATILE_BROADTILE
		void __iomem *gpio;
		unsigned long val;
	#endif
	struct backlight_properties props;
    struct device *dev = &pdev->dev;
	int ret = 0;
	struct dw_bl_data *data = kzalloc(sizeof(struct dw_bl_data), GFP_KERNEL);
    if (!data)
		return -ENOMEM;

    memcpy((void*)&data->dw_bl_props, (void*)dev->platform_data, sizeof(struct dw_bl_props));

    #ifdef CONFIG_MACH_VERSATILE_BROADTILE
		gpio = ioremap(0x85000000, SZ_4K);
	
		/* direction (output) */
		val = readl(gpio + 0x1c);
		val &= ~(1 << data->dw_bl_props.backlight_gpio);
		writel(val, gpio + 0x1c);
	
		/* value */
		val = readl(gpio + 0x18);
		val |= (1 << data->dw_bl_props.backlight_gpio);
		writel(val, gpio + 0x18);
	
		/* enable */
		val = readl(gpio + 0x20);
		val |= (1 << data->dw_bl_props.backlight_gpio);
		writel(val, gpio + 0x20);
	
		iounmap(gpio);
	#else

		ret = gpio_request(data->dw_bl_props.backlight_gpio , (char*)data->dw_bl_props.name);
		if (ret) {
			dev_dbg(&data->bl->dev, "Unable to request gpio!\n");
			goto err_gpio_bl;
		}
	
		//gpio_set_enable(data->dw_bl_props.backlight_gpio, 1); 
	
		ret = gpio_direction_output(data->dw_bl_props.backlight_gpio, 1);
		if (ret)
			goto err_gpio_dir;
	#endif

    data->bl = backlight_device_register("dw-bl", &pdev->dev ,data, &dw_bl_ops,&props);
	if (IS_ERR(data->bl)) {
		ret = PTR_ERR(data->bl);
		goto err_gpio_dir;
	}

	data->bl->props.brightness = 1;
	data->bl->props.max_brightness = 1;
	data->bl->props.power = 0;

	backlight_update_status(data->bl);

	return 0;


err_gpio_dir:
#ifndef CONFIG_MACH_VERSATILE_BROADTILE
	gpio_free(data->dw_bl_props.backlight_gpio);
err_gpio_bl:
#endif
	kfree(data);

	return ret;
}

static int __devexit dw_bl_remove(struct platform_device *pdev)
{
	struct dw_bl_data *data = dev_get_drvdata(&pdev->dev);

	backlight_device_unregister(data->bl);
	data->bl = NULL;

	gpio_free(data->dw_bl_props.backlight_gpio);

	kfree(data);

	return 0;
}

#ifdef CONFIG_PM
static int dw_bl_suspend(struct platform_device *pdev, pm_message_t state)
{
	struct dw_bl_data *data = dev_get_drvdata(&pdev->dev);

	dw_bl_set_backlight(data, 0);

	return 0;
}

static int dw_bl_resume(struct platform_device *pdev)
{
	
	struct dw_bl_data *data = dev_get_drvdata(&pdev->dev);

	backlight_update_status(data->bl);
	return 0;
}
#else
#define dw_bl_suspend NULL
#define dw_bl_resume NULL
#endif


static struct platform_driver dw_bl_driver = {
	.probe = dw_bl_probe,
	.remove = __exit_p(dw_bl_remove),
	.suspend = dw_bl_suspend,
	.resume = dw_bl_resume,
	.driver = {
		.name = "dw-bl",
		.owner = THIS_MODULE,
	},
};

static int __init dw_bl_init(void)
{
    return platform_driver_register(&dw_bl_driver);
}

static void __exit dw_bl_exit(void)
{
	platform_driver_unregister(&dw_bl_driver);
}

module_init(dw_bl_init);
module_exit(dw_bl_exit);

MODULE_AUTHOR("DSP Group");
MODULE_LICENSE("GPL v2");
MODULE_DESCRIPTION("LCD/Backlight DMW96");
