/*
 *  LCD / Backlight control code for DMW96
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/platform_device.h>
#include <linux/gpio.h>
#include <linux/fb.h>
#include <linux/backlight.h>
#include <video/dw_bl.h>
#include <linux/dp52.h>
#include <linux/earlysuspend.h>

//#define BL_DEBUG
#ifdef BL_DEBUG
  static int print_counter = 0;
#endif

static int setup_early_suspend(struct dw_bl_data *data);

struct dc2gain_regs {
	u16 dc2gaincnt;
	u16 dc2gainbp1;
	u16 dc2gainbp2;
};

static int dp_version;

#ifdef BL_DEBUG
static void print_regs(void)
{
	printk("Backlight[%d]: DP52_CMU_CCR0 = 0x%04x\n", print_counter, dp52_direct_read(DP52_CMU_CCR0));
	printk("Backlight[%d]: DP52_CMU_CCR1 = 0x%04x\n", print_counter, dp52_direct_read(DP52_CMU_CCR1));
	printk("Backlight[%d]: DP52_CMU_CCR2 = 0x%04x\n", print_counter, dp52_direct_read(DP52_CMU_CCR2));
	printk("Backlight[%d]: DP52_CMU_PLLCR = 0x%04x\n", print_counter, dp52_direct_read(DP52_CMU_PLLCR));
	printk("Backlight[%d]: DP52_CMU_CSTSR = 0x%04x\n", print_counter, dp52_direct_read(DP52_CMU_CSTSR));
	printk("Backlight[%d]: DP52_CMU_SYSCFG = 0x%04x\n", print_counter, dp52_direct_read(DP52_CMU_SYSCFG));
	printk("Backlight[%d]: DP52_PWM2_CFG1 = 0x%04x\n", print_counter, dp52_direct_read(DP52_PWM2_CFG1));
	printk("Backlight[%d]: DP52_PWM2_CFG2 = 0x%04x\n", print_counter, dp52_direct_read(DP52_PWM2_CFG2));

	print_counter++;
}
#endif

struct vcd_gain {
	unsigned int gain;
	unsigned int regval;
};

/*Note we multiply the amplifier gain by 100 to avoid fructures*/
static struct vcd_gain vcd_gain_arr[] = {
		{ 100,	0x1},
		{ 125,	0x2},
		{ 150,	0x3},
		{ 200,	0x4},
		{ 250,	0x5},
		{ 300,	0x6},                      
		{ 400,	0x7},
		{ 500,	0x8},
		{ 600,	0x9},
		{ 800,	0xa},
		{ 1000,	0xb},
		{ 1300,	0xc},
		{ 1700,	0xd},
		{ 2200,	0xe},
		{ 2900,	0xf},
		{ 3800,	0x10},
		{ 5000,	0x11},
		{ 6600,	0x12},
		{ 8600,	0x13},
		{ 10100,0x14}
};

static unsigned int find_bigger_or_equal_gain(unsigned int gain)
{
	unsigned int i = 0 ;
	gain *= 100;
    
	for (i = 0 ; i < ( sizeof(vcd_gain_arr) / sizeof(vcd_gain_arr[0]) ) ; i++)
		if (vcd_gain_arr[i].gain >= gain)
			return i;
	
	return i-1;
}

/**********************************************************************
	According to SPEC the Attenuation multiplier is
    0.75 + (i+1)/256
	where i is VDCxxATT

	Note:
	We multiply by 256 to avoid using fructurs hence:
	192 + (i+1) = val*100*256 / gain
	i = val*100*256 / gain - 192 - 1

**********************************************************************/
static unsigned int find_atten(unsigned int val, unsigned int gain){
	if (gain ==0) {
		return 0x3f;
	} else
		return ((val * 100 * 256) / gain) - 192 - 1;
}

/**********************************************************************
	In order to calculate the registers we 

	I_led = 1000 / (gain * attenuation) 

	or (in case on gain is bigger the 120) 

	I_led = 1000 / (gain' * attenuation) 

	Where:
	gain - VDCxxGAIN field in register DC2GAINCNT
	attenuation - 0.75 + (i+1)/256;  where i is VDCxxATT field int register DC2GAINCNT
	gain' is composition of DC2GAINBP1 and DC2GAINBP2 registers in some fixed values

    Note:
	When gain' registers are in use the gain registers will be zero.

**********************************************************************/
static int dc2gaincnt_calc(const unsigned int val,struct dc2gain_regs *comp_regs) 
{
    unsigned int after_gain = 0;
	unsigned int attn_val = 0;
	unsigned int after_gain_i = 0;

	comp_regs->dc2gainbp1 = 0x0;
	comp_regs->dc2gainbp2 = 0x0;
	
	if ( val > 120) { 
			comp_regs->dc2gainbp1 = 0x93ff;
			comp_regs->dc2gainbp2 = 0xf800;
			after_gain = 5300;
	} else if ( val > 101 ) {
			comp_regs->dc2gainbp1 = 0x83ff;
			comp_regs->dc2gainbp2 = 0xb801;
			after_gain = 3700;
	} else {
		after_gain_i = find_bigger_or_equal_gain(val);
        after_gain = vcd_gain_arr[after_gain_i].gain; 
	}

	/* Set atttentuation to decrease after_gain to the appropriate val */
	attn_val = find_atten(val , after_gain);

    comp_regs->dc2gaincnt = ((attn_val & 0x3f) << 5) | (vcd_gain_arr[after_gain_i].regval & 0x1f);

    return 0;
}

#if (CONFIG_DP52_BL_MAX_MA > 1000) 
	#error "DP Backlight current MA value is out of range"
#endif

static void dp52_bl_get_gaincnt_val(struct dc2gain_regs *comp_regs,unsigned int val,unsigned int max_val)
{
	unsigned int mA = 0;
	unsigned int gain_attn = 0;

	if (max_val == 0 ){
		printk("Invalid backlight max value\n");
		return;
	}

    mA = (CONFIG_DP52_BL_MAX_MA*val)/max_val;

	gain_attn = 1000 / ( (mA != 0 ) ? mA : 1000);

	dc2gaincnt_calc(gain_attn , comp_regs );

}

static void dp52_bl_set_backlight_closedloop(struct dw_bl_data *data, int brightness)
{
	struct dc2gain_regs dc2_regs;
	memset(&dc2_regs ,0, sizeof(struct dc2gain_regs) );

    /* SetBacklight */
	if (brightness <= 0x0){
		brightness = 1;
	}
	if (brightness > 0x20) {
		brightness = 0x20;
	}

	dp52_direct_write(DP52_PWM2_CFG1, 0x441f);		// Set duty-cycle to 75% and enable DCIN2 feedback (new DP)

	// Set brightness by limiting the current flow from pwm2
	dp52_bl_get_gaincnt_val(&dc2_regs, brightness, data->bl->props.max_brightness);

	/* this is a hack to eliminate the flicker of the backlight when using gainbp
	 * 1. shutdown backlight, 2. set registers, 3. start backlight */
	if (dc2_regs.dc2gainbp1 & 0x8000) {
		// shutdown backlight
		dp52_direct_write(DP52_PWM2_CFG1, 0x4400);
	}

	dp52_direct_write(DP52_AUX_DC2GAINCNT1, dc2_regs.dc2gaincnt);
	dp52_direct_write(DP52_AUX_DC2GAINBP1, dc2_regs.dc2gainbp1);
	dp52_direct_write(DP52_AUX_DC2GAINBP2, dc2_regs.dc2gainbp2);

	if (dc2_regs.dc2gainbp1 & 0x8000) {
		// start backlight again
		dp52_direct_write(DP52_PWM2_CFG1, 0x441f);
	}
}

static void dp52_bl_set_backlight_openloop(struct dw_bl_data *data, int brightness)
{
    /* SetBacklight */
	if (brightness <= 0x0){
		brightness = 1;
	}
	if (brightness > 0x20) {
		brightness = 0x20;
	}

	// Set brightness by changing the duty-cycle of the pwm
	dp52_direct_write(DP52_PWM2_CFG1, brightness);					// Set duty-cycle to brightness (old DP)
}

static int dp52_bl_update_status(struct backlight_device *dev)
{
	struct backlight_properties *props = &dev->props;
	struct dw_bl_data *data = dev_get_drvdata(&dev->dev);
	int power = props->power;
	int brightness = props->brightness;

	/* Test DP version and set the right modulo */
	u16 DP52_MODULO_32 = 0;							// Default (old DP)
	if (dp_version > 0) DP52_MODULO_32 = 0x800;			// DP-TO2 (new DP)
	
	if (power) {
		//this means that the power of the LCDC full on
		/* Enable PWM2 */
		dp52_direct_write(DP52_PWM2_CFG2, 0x2000 | DP52_MODULO_32);

	}else {
		//this means that the power of the LCD off
		/* Disable PWM2 */
		dp52_direct_write(DP52_PWM2_CFG2, 0x0);
	}

	if (dp_version > 0) dp52_bl_set_backlight_closedloop(data, brightness);
	else dp52_bl_set_backlight_openloop(data, brightness);

	return 0;
}

static int dp52_bl_get_brightness(struct backlight_device *dev)
{
	struct backlight_properties *props = &dev->props;

	return props->brightness;
}

static struct backlight_ops dp52_bl_ops = {
	.get_brightness		= dp52_bl_get_brightness,
	.update_status		= dp52_bl_update_status,
};


static int __devinit dp52_dp_bl_probe(struct platform_device *pdev)
{
	struct backlight_properties props;

    struct device *dev = &pdev->dev;
	int ret = 0;

	struct dw_bl_data *data = kzalloc(sizeof(struct dw_bl_data), GFP_KERNEL);
    if (!data)
		return -ENOMEM;

    memcpy((void*)&data->dw_bl_props, (void*)dev->platform_data, sizeof(struct dw_bl_props));

	memset(&props, 0, sizeof(struct backlight_properties));
	/*
	Brightness resulution is dicteted by the DP52 PWM_MOD (register PWM CFG2 bit 11-10)
	since we set the Modulo to be 00 (32) we set the resulution to be in 32 steps
    */

	props.max_brightness = 32;
	props.max_brightness = 32;
	props.power = 1;

    data->bl = backlight_device_register("dp52-bl", &pdev->dev ,data, &dp52_bl_ops, &props);
	if (IS_ERR(data->bl)) {
		ret = PTR_ERR(data->bl);
		goto err_bl_register;
	}

	/*
	Brightness resulution is dicteted by the DP52 PWM_MOD (register PWM CFG2 bit 11-10)
	since we set the Modulo to be 00 (32) we set the resulution to be in 32 steps
    */
/*	data->bl->props.brightness = 32;
	data->bl->props.max_brightness = 32;
	data->bl->props.power = 1;
*/
	dp_version = dp52_get_chip_version();

	dp52_set_bits(DP52_CMU_CCR2,(1<<9));		// Enable 4Mhz as pwm2 source

	if (dp_version > 0)  // Only for the new DP
	{
		dp52_set_bits(DP52_AUX_EN, 0x40);	// Enable DCIN2
		dp52_direct_write(0x5b, 0x1);				// Enable close loop switch
	}
	else {
		dp52_clr_bits(DP52_AUX_EN, 0x40);
	}

	backlight_update_status(data->bl);

	if (dp_version > 0) printk("dp52_bl: using closed-loop (DP ES%d)\n", dp_version);
	else printk("dp52_bl: using open-loop (DP ES%d)\n", dp_version);

#ifdef CONFIG_HAS_EARLYSUSPEND
	setup_early_suspend(data);
#endif

	return 0;

err_bl_register:
	kfree(data);
	return ret;
}

static int __devexit dp52_dp_bl_remove(struct platform_device *pdev)
{
	struct dw_bl_data *data = dev_get_drvdata(&pdev->dev);


#ifdef CONFIG_HAS_EARLYSUSPEND
	unregister_early_suspend(&data->early_suspend_backlight);
#endif

	backlight_device_unregister(data->bl);
	data->bl = NULL;

	kfree(data);

	return 0;
}

#ifdef CONFIG_HAS_EARLYSUSPEND
static void dp52_dp_bl_early_suspend(struct early_suspend *es)
{
	struct dw_bl_data *data = container_of(es, struct dw_bl_data, early_suspend_backlight);// dev_get_drvdata(&pdev->dev);

	// Turn the backlight off
	data->bl->props.power = 0;
	dp52_bl_update_status(data->bl);
}

static void dp52_dp_bl_late_resume(struct early_suspend *es)
{
	struct dw_bl_data *data = container_of(es, struct dw_bl_data, early_suspend_backlight);

	// Turn the backlight on
	data->bl->props.power = 1;
	dp52_bl_update_status(data->bl);
}
#else
#define dp52_dp_bl_early_suspend NULL
#define dp52_dp_bl_late_resume NULL
#endif


static struct platform_driver dp52_dp_bl_driver = {
	.probe = dp52_dp_bl_probe,
	.remove = __devexit_p(dp52_dp_bl_remove),
	.driver = {
		.name = "dp52-backlight",
		.owner = THIS_MODULE,
	},
};

#ifdef CONFIG_HAS_EARLYSUSPEND
static int setup_early_suspend(struct dw_bl_data *data)
{
	data->early_suspend_backlight.suspend = dp52_dp_bl_early_suspend;
	data->early_suspend_backlight.resume = dp52_dp_bl_late_resume;
	data->early_suspend_backlight.level = EARLY_SUSPEND_LEVEL_BLANK_SCREEN;

	register_early_suspend(&data->early_suspend_backlight);

	return 0;
}
#endif

static int __init dp52_dp_bl_init(void)
{
	return platform_driver_register(&dp52_dp_bl_driver);
}

static void __exit dp52_dp_bl_exit(void)
{
	platform_driver_unregister(&dp52_dp_bl_driver);
}

module_init(dp52_dp_bl_init);
module_exit(dp52_dp_bl_exit);

MODULE_AUTHOR("DSP Group");
MODULE_LICENSE("GPL v2");
MODULE_DESCRIPTION("LCD/Backlight DP52");
