/*
 * spi_mini.c - Mini-SPI
 *
 * Copyright (C) 2005 Grandstream Networks, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/kernel.h>
#include <linux/device.h>
#include <linux/init.h>
#include <linux/cache.h>
#include <linux/mutex.h>
#include <linux/of_device.h>
#include <linux/mod_devicetable.h>
#include <linux/spi/spi.h>
#include <linux/io.h>
#include <linux/of_spi.h>
#include <linux/pm_runtime.h>
#include <linux/spi/spi_dmw96.h>
#include <linux/spi/mini.h>

DEFINE_MUTEX( mini_spi_mutex );
struct mini_spi_dat mini_spi_dat;

void mini_spi_setup( struct spi_device* spi )
{   
    static int spi_started = 0;
    struct dmw96_spi_info* hw = ((struct dmw96_spi*)spi_master_get_devdata( spi->master ))->pdata;
    int i;

    if ( spi_started )
        return;
    spi_started = 1;

    memset( &mini_spi_dat, 0, sizeof ( mini_spi_dat ) );

    mini_spi_dat.iomem = ioremap_nocache( SPI2_BASE, 0x1000 );
    if ( mini_spi_dat.iomem == NULL )
    {   
        printk("mini_spi_setup: error mapping registers\n");
        spi_started = 0;
        return;
    }
    mini_spi_dat.gpio_iomem = ioremap_nocache( GPIO_BASE, 0x1000 );
    if ( mini_spi_dat.gpio_iomem == NULL )
    {   
        printk("mini_spi_setup: error mapping gpio registers\n");
        spi_started = 0;
        iounmap( mini_spi_dat.iomem );
        return;
    }

    spin_lock_init( &mini_spi_dat.lock );
    mutex_init( &mini_spi_mutex );

    mini_spi_dat.regs.spi_cfg_reg = (uint32_t*)(mini_spi_dat.iomem + 0x0000);
    mini_spi_dat.regs.spi_rate_cntl = (uint32_t*)(mini_spi_dat.iomem + 0x0004);
    mini_spi_dat.regs.spi_int_en = (uint32_t*)(mini_spi_dat.iomem + 0x0008);
    mini_spi_dat.regs.spi_int_st = (uint32_t*)(mini_spi_dat.iomem + 0x000C);
    mini_spi_dat.regs.spi_int_clr = (uint32_t*)(mini_spi_dat.iomem + 0x0010);
    mini_spi_dat.regs.spi_cause_reg = (uint32_t*)(mini_spi_dat.iomem + 0x00014);
    mini_spi_dat.regs.spi_tx_dat = (uint32_t*)(mini_spi_dat.iomem + 0x0018);
    mini_spi_dat.regs.spi_rx_dat = (uint32_t*)(mini_spi_dat.iomem + 0x001C);
    mini_spi_dat.regs.spi_del_val = (uint32_t*)(mini_spi_dat.iomem + 0x0020);
    mini_spi_dat.regs.spi_rx_buff = (uint32_t*)(mini_spi_dat.iomem + 0x003C);

    mini_spi_dat.gpio.port_a_set = (uint32_t*)(mini_spi_dat.gpio_iomem + 0x004);
    mini_spi_dat.gpio.port_a_clr = (uint32_t*)(mini_spi_dat.gpio_iomem + 0x008);
    mini_spi_dat.gpio.port_f_set = (uint32_t*)(mini_spi_dat.gpio_iomem + 0x1E4);
    mini_spi_dat.gpio.port_f_clr = (uint32_t*)(mini_spi_dat.gpio_iomem + 0x1E8);
    mini_spi_dat.gpio.port_g_set = (uint32_t*)(mini_spi_dat.gpio_iomem + 0x244);
    mini_spi_dat.gpio.port_g_clr = (uint32_t*)(mini_spi_dat.gpio_iomem + 0x248);

    for ( i = 0; i < 5; i++ )
    {   
        switch ( hw->get_cs_bank( i ) )
        {   
            default:
            case 'A':
                mini_spi_dat.chip_select.cs_set[i] = mini_spi_dat.gpio.port_a_set;
                mini_spi_dat.chip_select.cs_clr[i] = mini_spi_dat.gpio.port_a_clr;
                break;
            case 'F':
                mini_spi_dat.chip_select.cs_set[i] = mini_spi_dat.gpio.port_f_set;
                mini_spi_dat.chip_select.cs_clr[i] = mini_spi_dat.gpio.port_f_clr;
                break;
            case 'G':
                mini_spi_dat.chip_select.cs_set[i] = mini_spi_dat.gpio.port_g_set;
                mini_spi_dat.chip_select.cs_clr[i] = mini_spi_dat.gpio.port_g_clr;
                break;
        }
        mini_spi_dat.chip_select.mask[i] = hw->get_cs_mask( i );
    }

    printk("mini_spi_setup\n");
}

EXPORT_SYMBOL_GPL( mini_spi_setup );
EXPORT_SYMBOL_GPL( mini_spi_mutex );
EXPORT_SYMBOL_GPL( mini_spi_dat );

