/*
 * spi_dmw96.c - SPI master driver for the DMW96
 *
 * Copyright (C) 2010 DSPG Technologies GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <linux/init.h>
#include <linux/spinlock.h>
#include <linux/workqueue.h>
#include <linux/interrupt.h>
#include <linux/io.h>
#include <linux/errno.h>
#include <linux/err.h>
#include <linux/clk.h>
#include <linux/delay.h>
#include <linux/platform_device.h>
#include <linux/spi/spi.h>
#include <linux/spi/spi_bitbang.h>
#include <linux/dmw96dma.h>
#include <linux/spi/spi_dmw96.h>

#include <linux/mm.h>
#include <asm/memory.h>

#include <mach/hardware.h>
#include <mach/platform.h>

#define DRIVER_NAME "dmw96-spi"

#define DMW_SPI_CFG_REG		0x00
#define DMW_SPI_RATE_CNTL	0x04
#define DMW_SPI_INT_EN		0x08
#define DMW_SPI_INT_STAT	0x0C
#define DMW_SPI_INT_CLR		0x10
#define DMW_SPI_INT_CAUSE	0x14
#define DMW_SPI_TX_DAT		0x18
#define DMW_SPI_RX_DAT		0x1C
#define DMW_SPI_DEL_VAL		0x20
#define DMW_SPI_RX_BUFF		0x3C
#define DMW_SPI_DBG		0x40

#define DMW_SPI_FIFO_SIZE	16

#define DMW_SPI_NUM_CHIPSELECT	5

#define INT_RX_FIFO_WM		(1<<0)
#define INT_RX_TO		(1<<1)
#define INT_RX_FIFO_UR		(1<<2)
#define INT_TX_FIFO_WM		(1<<8)
#define INT_TX_FIFO_OV		(1<<9)
#define INT_ALL_INTS		(INT_RX_FIFO_WM | INT_RX_TO | \
				 INT_RX_FIFO_UR | INT_TX_FIFO_WM | \
				 INT_TX_FIFO_OV)
#define INT_NO_INTS		0

#define INT_FULL_FIFO_WM	(15<<12)

#define SPI_IDLE		(1<<15)
#define SPI_DONE		(1<<10)

#define CFG_EN			(1<<31)
#define CFG_FIFOS_FLUSH		((1<<17) | (1<<18))
#define CFG_TX_DMA_EN		(1<<16)
#define CFG_RX_DMA_EN		(1<<15)
#define CFG_RX_IGNORE		(1<<8)
#define CFG_DIRECT_CS		(1<<7)
#define CFG_SWCS		(1<<6)
#define CFG_CS1_HOLD		(1<<4)
#define CFG_CS1_DIS		(1<<3)
#define CFG_CS0_HOLD		(1<<2)
#define CFG_CS0_DIS		(1<<1)

#if 0
struct dmw96_spi {
	/* bitbang has to be first */
	struct spi_bitbang	 bitbang;
	struct completion	 done;

	void __iomem		*regs;
	int			 irq;
	int			 remaining_bytes;

	/* data buffers */
	const unsigned char	*tx_ptr;
	unsigned char		*rx_ptr;

	int			 chip_select;

	struct clk		*clk;
	struct resource		*ioarea;
	struct spi_master	*master;

	struct device		*dev;
    
	struct dmw96_spi_info	*pdata;

#ifdef CONFIG_DMW96_GDMAC
	/* DMA */
	struct dmw96dma_list	*dma_rx_list;
	struct dmw96dma_list	*dma_tx_list;
	unsigned int             dma_rx_channel;
	unsigned int             dma_tx_channel;
#endif
	struct timer_list timer;
};
#endif

static unsigned int dma = 1;
module_param(dma, uint, 0);
MODULE_PARM_DESC(dma, "Support for DMA mode (default: 1)");

static void dmw96_spi_debug(struct dmw96_spi *hw);

#ifdef CONFIG_DMW96_GDMAC
static void
dmw96_spi_dma_tx_callback(int status, void *context)
{
	struct dmw96_spi *hw = (struct dmw96_spi *)context;

	if (!status)
		hw->remaining_bytes = 0;

	if (!hw->dma_rx_list)
		complete(&hw->done);
}

static void
dmw96_spi_dma_rx_callback(int status, void *context)
{
	struct dmw96_spi *hw = (struct dmw96_spi *)context;

	if (!status)
		hw->remaining_bytes = 0;

	complete(&hw->done);
}
#endif

static inline struct dmw96_spi *
to_hw(struct spi_device *sdev)
{
	return spi_master_get_devdata(sdev->master);
}

static void
dmw96_spi_set_cs(struct dmw96_spi *hw, unsigned int chip_select, int val, int pol)
{
	unsigned int ctrl;
	unsigned int bit = CFG_CS0_HOLD;

	clk_enable(hw->clk);

	ctrl = readl(hw->regs + DMW_SPI_CFG_REG);

	if (hw->chip_select != 0)
		bit = CFG_CS1_HOLD;

	ctrl |= CFG_DIRECT_CS;
	if ((pol && (val == BITBANG_CS_ACTIVE)) ||
	   (!pol && (val == BITBANG_CS_INACTIVE))) {
		ctrl |=  CFG_SWCS;
		//ctrl &= ~bit; /* don't negate chip select at end of transaction */
	} else {
		ctrl &= ~CFG_SWCS;
		//ctrl |=  bit; /* negate chip select at end of transaction */
	}

	writel(ctrl, hw->regs + DMW_SPI_CFG_REG);

	clk_disable(hw->clk);
}

static void
dmw96_spi_chipsel(struct spi_device *spi, int value)
{
	struct dmw96_spi *hw = to_hw(spi);
	unsigned int cspol = spi->mode & SPI_CS_HIGH ? 1 : 0;
	unsigned int chip_select = spi->chip_select;

	if (hw->pdata->set_cs)
		hw->pdata->set_cs(hw->pdata, chip_select, value, cspol);
	else
		dmw96_spi_set_cs(hw, chip_select, value, cspol);
}

static int
dmw96_spi_setupxfer(struct spi_device *spi, struct spi_transfer *t)
{
	struct dmw96_spi *hw = to_hw(spi);
	unsigned int bpw;
	unsigned long clk_rate = clk_get_rate(hw->clk);
	unsigned int hz;
	unsigned int div;
	unsigned int div_code;
	unsigned int ctrl = 0;
	unsigned msb_first = ~(spi->mode & SPI_LSB_FIRST);
	unsigned int clk_polarity =
			 spi->mode & SPI_CPOL ? 1 : 0; /* 1 if active high */

	bpw = t ? t->bits_per_word : 0;
	if (!bpw)
		bpw = spi->bits_per_word;
	hz  = t ? t->speed_hz : 0;
	if (!hz)
		hz = spi->max_speed_hz;

	dev_dbg(&spi->dev, "hz = %u, clockrate = %lu, clk_polarity = %s\n",
		hz, clk_rate, clk_polarity ? "active high" : "active low");

	if (!hz) {
		dev_err(&spi->dev, "invalid speed value (%dHz)\n", hz);
		return -EINVAL;
	}

	if (hz > spi->max_speed_hz) {
		dev_err(&spi->dev, "speed values > %dHz not supported\n",
			spi->max_speed_hz);
		return -EINVAL;
	}

	if (bpw != 8) {
		dev_err(&spi->dev, "invalid bits-per-word (%d)\n", bpw);
		return -EINVAL;
	}

	div = clk_rate / hz;
	if (div < 2)
		div = 2;

	div_code = 0;
	while ((div > 2) && (div_code < 10)) {
		div_code++;
		div = (div+1)/2; /* round up */
	}

	dev_dbg(&spi->dev, "setting pre-scaler to %d, code %u " \
		"(hz %u, clk %lu)\n", div, div_code, hz, clk_rate);

	clk_enable(hw->clk);

	writel(div_code, hw->regs + DMW_SPI_RATE_CNTL);

	writel(0x100, hw->regs + DMW_SPI_DEL_VAL);

	hw->chip_select = spi->chip_select;

	/* important: keep cs at the same configuration, since setup_transfer might be called */
	/* between 2 consecutive transfers on the same device (for example: spi_write_then_read) */
	ctrl = readl(hw->regs + DMW_SPI_CFG_REG) & CFG_SWCS;
	ctrl |= CFG_FIFOS_FLUSH |
	       ((msb_first & 0x1) << 5) | (clk_polarity & 0x1) | CFG_DIRECT_CS;
	if (hw->chip_select == 0)
		ctrl |= CFG_CS1_DIS;
	else
		ctrl |= CFG_CS0_DIS;
	writel(ctrl, hw->regs + DMW_SPI_CFG_REG);

	ctrl &= ~CFG_FIFOS_FLUSH;
	writel(ctrl, hw->regs + DMW_SPI_CFG_REG);

	spin_lock(&hw->bitbang.lock);
	if (!hw->bitbang.busy) {
		hw->bitbang.chipselect(spi, BITBANG_CS_INACTIVE);
		/* need to ndelay for 0.5 clocktick ? */
	}

	spin_unlock(&hw->bitbang.lock);

	clk_disable(hw->clk);

	return 0;
}

static int
dmw96_spi_setup(struct spi_device *spi)
{
	int ret;

	if (!spi->bits_per_word)
		spi->bits_per_word = 8;

	ret = dmw96_spi_setupxfer(spi, NULL);
	if (ret < 0) {
		dev_err(&spi->dev, "setupxfer returned %d\n", ret);
		return ret;
	}

	dev_dbg(&spi->dev, "%s: mode %d, %u bpw, %d hz\n",
		__func__, spi->mode, spi->bits_per_word,
		spi->max_speed_hz);

	return 0;
}

static void
dmw96_spi_enable(struct dmw96_spi *hw)
{
	unsigned long val;

	while (!(readl(hw->regs + DMW_SPI_INT_STAT) & SPI_DONE))
		;

	val = readl(hw->regs + DMW_SPI_CFG_REG);
	val |= CFG_EN;
	writel(val, hw->regs + DMW_SPI_CFG_REG);
}

#ifdef CONFIG_DMW96_GDMAC
static void
dmw96_spi_dma(struct dmw96_spi *hw, int flags)
{
	unsigned long val;

	val = readl(hw->regs + DMW_SPI_CFG_REG);
	if (val & CFG_EN) {
		while (!(readl(hw->regs + DMW_SPI_INT_STAT) & SPI_DONE))
			;

		val &= ~CFG_EN;
		writel(val, hw->regs + DMW_SPI_CFG_REG);
	}

	val &= ~(CFG_TX_DMA_EN | CFG_RX_DMA_EN);
	val |= flags;
	if ((flags & (CFG_RX_DMA_EN|CFG_TX_DMA_EN)) == CFG_TX_DMA_EN)
		val |=  CFG_RX_IGNORE;
	else
		val &= ~CFG_RX_IGNORE;
	if (flags)
		val |=  CFG_FIFOS_FLUSH;
	writel(val, hw->regs + DMW_SPI_CFG_REG);

	val &= ~CFG_FIFOS_FLUSH;
	if (flags)
		val |= CFG_EN;
	writel(val, hw->regs + DMW_SPI_CFG_REG);
}

static void
dmw96_spi_watermarks(struct dmw96_spi *hw, int tx, int rx)
{
	writel((((tx-1) & 0xF) << 12) | (((rx-1) & 0xF) << 4),
	       hw->regs + DMW_SPI_INT_EN);
}
#endif

static void
dmw96_spi_disable_and_clear_ints(struct dmw96_spi *hw)
{
	writel(INT_NO_INTS,  hw->regs + DMW_SPI_INT_EN);
	writel(INT_ALL_INTS, hw->regs + DMW_SPI_INT_CLR);
}

static void
dmw96_spi_enable_rx_wm_int(struct dmw96_spi *hw, int count)
{
	writel(INT_RX_FIFO_WM | (((count-1) & 0xf)<<4),
	       hw->regs + DMW_SPI_INT_EN);
}

static int
dmw96_spi_rx_not_empty(struct dmw96_spi *hw)
{
	return !!(readl(hw->regs + DMW_SPI_INT_STAT) & INT_RX_FIFO_WM);
}

static int
dmw96_spi_tx_not_full(struct dmw96_spi *hw)
{
	return !!(readl(hw->regs + DMW_SPI_INT_STAT) & INT_TX_FIFO_WM);
}

static int
dmw96_spi_fill_tx_fifo(struct dmw96_spi *hw)
{
	int written = 0;
	unsigned char dat = 0;

	while (dmw96_spi_tx_not_full(hw) && (hw->remaining_bytes > 0) &&
	       (written < DMW_SPI_FIFO_SIZE)) {
		if (hw->tx_ptr)
			dat = *hw->tx_ptr++;
		writel(dat, hw->regs + DMW_SPI_TX_DAT);
		hw->remaining_bytes--;
		written++;
	}

	return written;
}

static int
dmw96_spi_txrx(struct spi_device *spi, struct spi_transfer *t)
{
	struct dmw96_spi *hw = to_hw(spi);
#ifdef CONFIG_DMW96_GDMAC
	int use_dma = 0;
#endif

	hw->chip_select = spi->chip_select;
	hw->tx_ptr = t->tx_buf;
	hw->rx_ptr = t->rx_buf;
	hw->remaining_bytes = t->len;
	INIT_COMPLETION(hw->done);

	dev_dbg(&spi->dev, "transfer %d bytes (%s %s)\n", t->len,
	        t->tx_buf ? "tx" : "  ", t->rx_buf ? "rx" : "  ");

	clk_enable(hw->clk);
	dmw96_spi_disable_and_clear_ints(hw);

#ifdef CONFIG_DMW96_GDMAC
	use_dma = dma &&
		!(hw->remaining_bytes % DMW_SPI_FIFO_SIZE) &&
		(!hw->rx_ptr || virt_addr_valid(hw->rx_ptr)) &&
		(!hw->tx_ptr || virt_addr_valid(hw->tx_ptr));
	if (use_dma) {
		if (hw->rx_ptr) {
			hw->dma_rx_list = dmw96dma_alloc_io_list(0, hw->dma_rx_channel,
			                                         dmw96_spi_dma_rx_callback,
			                                         (void *)hw);
			if (!hw->dma_rx_list) {
				dev_err(&spi->dev, "Out of memory for DMA\n");
				clk_disable(hw->clk);
				return t->len;
			}

			dev_dbg(&spi->dev, "setting up RX DMA transfer for %d bytes\n",
			        t->len);

			dmw96dma_add_io_transfer_buf(hw->dma_rx_list,
			                             (void *)hw->rx_ptr,
			                             t->len, NULL,
			                             (void *)hw);
		}

		hw->dma_tx_list = dmw96dma_alloc_io_list(0, hw->dma_tx_channel,
		                                         dmw96_spi_dma_tx_callback,
		                                         (void *)hw);
		if (!hw->dma_tx_list) {
			dev_err(&spi->dev, "Out of memory for DMA\n");
			if (hw->dma_rx_list) {
				dmw96dma_free_list(hw->dma_rx_list);
				hw->dma_rx_list = NULL;
			}
			clk_disable(hw->clk);
			return t->len;
		}

		dev_dbg(&spi->dev, "setting up TX DMA transfer for %d bytes\n",
		        t->len);

		dmw96dma_add_io_transfer_buf(hw->dma_tx_list,
		                             hw->tx_ptr ? (void *)hw->tx_ptr :
		                                          (void *)hw->rx_ptr,
		                             t->len, NULL,
		                             (void *)hw);

		dmw96_spi_watermarks(hw, DMW_SPI_FIFO_SIZE, DMW_SPI_FIFO_SIZE);

		if (hw->rx_ptr)
			dmw96dma_submit(hw->dma_rx_list);
		dmw96dma_submit(hw->dma_tx_list);

		if (hw->rx_ptr)
			dmw96_spi_dma(hw, CFG_RX_DMA_EN | CFG_TX_DMA_EN);
		else
			dmw96_spi_dma(hw, CFG_TX_DMA_EN);
	} else
#endif
	{
		int count;

		dmw96_spi_enable(hw);

		count = dmw96_spi_fill_tx_fifo(hw);

		dmw96_spi_enable_rx_wm_int(hw, count);
	}

	mod_timer(&hw->timer, jiffies+HZ);
	wait_for_completion(&hw->done);

#ifdef CONFIG_DMW96_GDMAC
	if (use_dma) {
		dmw96_spi_dma(hw, 0);

		if (hw->dma_rx_list) {
			dmw96dma_free_list(hw->dma_rx_list);
			hw->dma_rx_list = NULL;
		}

		if (hw->dma_tx_list) {
			dmw96dma_free_list(hw->dma_tx_list);
			hw->dma_tx_list = NULL;
		}
	}
#endif

	dmw96_spi_disable_and_clear_ints(hw);
	clk_disable(hw->clk);

	return t->len - hw->remaining_bytes;
}

static void
dmw96_spi_debug(struct dmw96_spi *hw)
{
	int val;

	printk(KERN_INFO "dmw96_spi: CFG_REG   = 0x%x\n", readl(hw->regs + DMW_SPI_CFG_REG));
	printk(KERN_INFO "dmw96_spi: RATE_CNTL = 0x%x\n", readl(hw->regs + DMW_SPI_RATE_CNTL));
	printk(KERN_INFO "dmw96_spi: DEL_VAL   = 0x%x\n", readl(hw->regs + DMW_SPI_DEL_VAL));
	printk(KERN_INFO "dmw96_spi: INT_EN =    0x%x\n", readl(hw->regs + DMW_SPI_INT_EN));
	printk(KERN_INFO "dmw96_spi: INT_STAT =  0x%x\n", readl(hw->regs + DMW_SPI_INT_STAT));
	printk(KERN_INFO "dmw96_spi: INT_CLR =   0x%x\n", readl(hw->regs + DMW_SPI_INT_CLR));
	printk(KERN_INFO "dmw96_spi: INT_CAUSE = 0x%x\n", readl(hw->regs + DMW_SPI_INT_CAUSE));

	val = readl(hw->regs + DMW_SPI_DBG);

	printk(KERN_INFO "dmw96_spi: TX_BUSY:       %d\n", (val & (0x1<<31))>>31);
	printk(KERN_INFO "dmw96_spi: SPI_FSM_CUR:   %d\n", (val & (0x3<<18))>>18);
	printk(KERN_INFO "dmw96_spi: SPI_FSM_NXT:   %d\n", (val & (0x3<<16))>>16);
	printk(KERN_INFO "dmw96_spi: TX_FIFO_LVL:   %d\n", (val & (0x1f<<9))>>9);
	printk(KERN_INFO "dmw96_spi: TX_FIFO_EMPTY: %d\n", (val & (0x1<<8))>>8);
	printk(KERN_INFO "dmw96_spi: TX_FIFO_FULL:  %d\n", (val & (0x1<<7))>>7);
	printk(KERN_INFO "dmw96_spi: RX_FIFO_LVL:   %d\n", (val & (0x1f<<2))>>2);
	printk(KERN_INFO "dmw96_spi: RX_FIFO_EMPTY: %d\n", (val & (0x1<<1))>>1);
	printk(KERN_INFO "dmw96_spi: RX_FIFO_FULL:  %d\n", val & 0x1);
}

void dmw96_spi_irq_timeout(unsigned long data)
{
	struct dmw96_spi *hw = (struct dmw96_spi *)data;

	printk(KERN_ERR "dmw96_spi: irq timeout\n");
	dmw96_spi_disable_and_clear_ints(hw);
	clk_disable(hw->clk);
	udelay(5);
	clk_enable(hw->clk);
	dmw96_spi_disable_and_clear_ints(hw);
	complete(&hw->done);

	return;
}

irqreturn_t
dmw96_spi_irq(int irq, void *dev)
{
	struct dmw96_spi *hw = dev;
	int cause;
	unsigned char dat;

	del_timer(&hw->timer);
	cause = readl(hw->regs + DMW_SPI_INT_CAUSE);
	dmw96_spi_disable_and_clear_ints(hw);

	if (cause & INT_RX_FIFO_WM) { /* A transmit has completed */
		while (dmw96_spi_rx_not_empty(hw)) {
			/* Read out all the data from the Rx FIFO */
			dat = readl(hw->regs + DMW_SPI_RX_DAT);
			if (hw->rx_ptr)
				*hw->rx_ptr++ = dat;
		}

		if (hw->remaining_bytes > 0) {
			int count = dmw96_spi_fill_tx_fifo(hw);

			dmw96_spi_enable_rx_wm_int(hw, count);
		} else {
			/* No more data to send.
			 * Indicate the transfer is completed.
			 */

			complete(&hw->done);
		}
	} else {
		printk(KERN_ERR "dmw96_spi: unhandled or ghost interrupt (0x%x)\n", cause);
		dmw96_spi_debug(hw);
	}

	return IRQ_HANDLED;
}

static int
dmw96_spi_probe(struct platform_device *pdev)
{
	struct dmw96_spi *hw;
	struct spi_master *master;
	struct resource *res;
	int err = 0;

	master = spi_alloc_master(&pdev->dev, sizeof(struct dmw96_spi));
	if (master == NULL) {
		dev_err(&pdev->dev, "No memory for spi_master\n");
		err = -ENOMEM;
		goto err_nomem;
	}

	master->num_chipselect = DMW_SPI_NUM_CHIPSELECT;

	hw = spi_master_get_devdata(master);
	memset(hw, 0, sizeof(struct dmw96_spi));

	hw->master = spi_master_get(master);
	hw->pdata = pdev->dev.platform_data;
	hw->dev = &pdev->dev;

	if (!hw->pdata) {
		dev_err(&pdev->dev, "No valid platform data supplied\n");
		err = -ENOENT;
		goto err_no_pdata;
	}

	master->bus_num = hw->pdata->bus_num;

	platform_set_drvdata(pdev, hw);
	init_completion(&hw->done);
	init_timer(&hw->timer);
	hw->timer.function = dmw96_spi_irq_timeout;
	hw->timer.data = (unsigned long)hw;

	/* setup the state for the bitbang driver */
	hw->bitbang.master         = hw->master;
	hw->bitbang.setup_transfer = dmw96_spi_setupxfer;
	hw->bitbang.chipselect     = dmw96_spi_chipsel;
	hw->bitbang.txrx_bufs      = dmw96_spi_txrx;
	hw->bitbang.master->setup  = dmw96_spi_setup;

	dev_dbg(hw->dev, "bitbang at %p\n", &hw->bitbang);

	/* find and map our resources */
#ifdef CONFIG_DMW96_GDMAC
	res = platform_get_resource(pdev, IORESOURCE_DMA, 0);
	if (res == NULL) {
		dev_err(&pdev->dev, "Cannot get RX DMA channel\n");
		err = -ENOENT;
		goto err_no_iores;
	} else
		hw->dma_rx_channel = res->start;

	res = platform_get_resource(pdev, IORESOURCE_DMA, 1);
	if (res == NULL) {
		dev_err(&pdev->dev, "Cannot get TX DMA channel\n");
		err = -ENOENT;
		goto err_no_iores;
	} else
		hw->dma_tx_channel = res->start;
#endif

	res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (res == NULL) {
		dev_err(&pdev->dev, "Cannot get IORESOURCE_MEM\n");
		err = -ENOENT;
		goto err_no_iores;
	}

	hw->ioarea = request_mem_region(res->start, (res->end - res->start)+1,
	                                pdev->name);

	if (hw->ioarea == NULL) {
		dev_err(&pdev->dev, "Cannot reserve region\n");
		err = -ENXIO;
		goto err_no_iores;
	}

	hw->regs = ioremap_nocache(res->start, (res->end - res->start)+1);
	if (hw->regs == NULL) {
		dev_err(&pdev->dev, "Cannot map IO\n");
		err = -ENXIO;
		goto err_no_iomap;
	}

	hw->irq = platform_get_irq(pdev, 0);
	if (hw->irq < 0) {
		dev_err(&pdev->dev, "No IRQ specified\n");
		err = -ENOENT;
		goto err_no_iomap;
	}

	hw->clk = clk_get(&pdev->dev, "spi");
	if (IS_ERR(hw->clk)) {
		dev_err(&pdev->dev, "No clock for device\n");
		err = PTR_ERR(hw->clk);
		goto err_no_clk;
	}

	clk_enable(hw->clk);

	dmw96_spi_disable_and_clear_ints(hw);

	err = request_irq(hw->irq, dmw96_spi_irq, IRQF_SHARED | IRQF_DISABLED, pdev->name, hw);
	if (err) {
		dev_err(&pdev->dev, "Cannot claim IRQ\n");
		goto err_no_irq;
	}

	/* register our spi controller */
	err = spi_bitbang_start(&hw->bitbang);
	if (err) {
		dev_err(&pdev->dev, "Failed to register SPI master\n");
		goto err_register;
	}

	clk_disable(hw->clk);

	return 0;

 err_register:
	free_irq(hw->irq, hw);

 err_no_irq:
	clk_disable(hw->clk);
	clk_put(hw->clk);

 err_no_clk:
	iounmap(hw->regs);

 err_no_iomap:
	release_resource(hw->ioarea);
	kfree(hw->ioarea);

 err_no_iores:
 err_no_pdata:
	spi_master_put(hw->master);

 err_nomem:
	return err;
}

static int
dmw96_spi_remove(struct platform_device *dev)
{
	struct dmw96_spi *hw = platform_get_drvdata(dev);
	unsigned long val;

	platform_set_drvdata(dev, NULL);

	spi_unregister_master(hw->master);

	val = readl(hw->regs + DMW_SPI_CFG_REG);
	val &= ~(CFG_EN);
	writel(val, hw->regs + DMW_SPI_CFG_REG);

	clk_disable(hw->clk);
	clk_put(hw->clk);

	free_irq(hw->irq, hw);
	iounmap(hw->regs);

	release_resource(hw->ioarea);
	kfree(hw->ioarea);

	spi_master_put(hw->master);

	return 0;
}

static struct platform_driver dmw96_spidrv = {
	.probe		= dmw96_spi_probe,
	.remove		= dmw96_spi_remove,

	.driver		= {
		.name	= DRIVER_NAME,
		.owner	= THIS_MODULE,
	},
};

static int __init
dmw96_spi_init(void)
{
	return platform_driver_register(&dmw96_spidrv);
}

static void __exit
dmw96_spi_exit(void)
{
	platform_driver_unregister(&dmw96_spidrv);
}

subsys_initcall(dmw96_spi_init);
module_exit(dmw96_spi_exit);

MODULE_DESCRIPTION("DMW96 SPI Driver");
MODULE_AUTHOR("Andreas Weissel <andreas.weissel@dpsg.com>");
MODULE_LICENSE("GPL");
