/*
 * dp52-pmu.c - DP52 PMU driver
 *
 * (C) Copyright 2011, DSP Group
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/err.h>
#include <linux/gpio.h>
#include <linux/pm.h>
#include <linux/mfd/dp52/core.h>
#include <linux/regulator/driver.h>

#define PMU_ENABLE_SP3		(1 <<  0)
#define PMU_ENABLE_SP2		(1 <<  1)
#define PMU_ENABLE_DP_IO	(1 <<  2)
#define PMU_ENABLE_RF		(1 <<  3)
#define PMU_ENABLE_RF_HILO	(1 <<  4)
#define PMU_ENABLE_IO		(1 <<  5)
#define PMU_ENABLE_SP1		(1 <<  6)
#define PMU_ENABLE_MEM		(1 <<  7)
#define PMU_ENABLE_CORE		(1 <<  8)
#define PMU_ENABLE_CORE_HILO	(1 <<  9)
#define PMU_ENABLE_DP_CORE	(1 << 10)

struct dp52_pmu_bank {
	uint16_t enable;
	union {
		uint16_t reg;
		struct {
			uint16_t dcdc_rf     : 4;
			uint16_t ldo_mem     : 5;
			uint16_t dcdc_core   : 4;
			uint16_t ldo_dp_core : 3;
		} fields;
	} lvl1;
	union {
		uint16_t reg;
		struct {
			uint16_t ldo_sp3     : 5;
			uint16_t ldo_sp2     : 5;
			uint16_t ldo_sp1     : 4;
		} fields;
	} lvl2;
	union {
		uint16_t reg;
		struct {
			uint16_t ldo_dp_io   : 4;
			uint16_t ldo_io      : 4;
		} fields;
	} lvl3;
	uint16_t crdc_cfg1;
	uint16_t crdc_cfg2;
};

enum dp52_pmu_banks {
	PMU_HWBANK0,
	PMU_HWBANK1,
	PMU_HWBANK2,
	PMU_HWBANK3,
	PMU_SWBANK,
	PMU_NUM_BANKS,
};

enum {
	DP52_RID_SPARE2,
	DP52_RID_SPARE3,
};

struct dp52_regulator_info {
	struct regulator_desc desc;
	struct regulator_dev *rdev;
	struct dp52 *dp52;

	int min_uV;
	int max_uV;
	int step_uV;

	int enable_bit;
	int level_bit;
	int level_msk;
};

static void dp52_pmu_init_hwbanks(struct dp52 *dp52)
{
	int i;

	for (i = 0; i < 4; i++) {
		dp52_write(dp52, DP52_PMU_RFDC_CFG1_SET(i), 0x45a);
		dp52_write(dp52, DP52_PMU_RFDC_CFG2_SET(i), 0x16a);
		dp52_write(dp52, DP52_PMU_CRDC_CFG1_SET(i), 0x45a);
		dp52_write(dp52, DP52_PMU_CRDC_CFG2_SET(i), 0x16a);
	}
}

static int
dp52_pmu_read_bank(struct dp52 *dp52, enum dp52_pmu_banks nr,
                   struct dp52_pmu_bank *bank)
{
	if (nr < 0 || nr >= PMU_NUM_BANKS || !bank)
		return -EINVAL;

	/* handle the software bank seperately (direct access) */
	if (nr == PMU_SWBANK) {
		bank->enable    = dp52_read(dp52, DP52_PMU_EN_SW);
		bank->lvl1.reg  = dp52_read(dp52, DP52_PMU_LVL1_SW);
		bank->lvl2.reg  = dp52_read(dp52, DP52_PMU_LVL2_SW);
		bank->lvl3.reg  = dp52_read(dp52, DP52_PMU_LVL3_SW);
		bank->crdc_cfg1 = dp52_read(dp52, DP52_PMU_CRDC_CFG1_SW);
		bank->crdc_cfg2 = dp52_read(dp52, DP52_PMU_CRDC_CFG2_SW);
		return 0;
	}

	bank->enable    = dp52_read(dp52, DP52_PMU_EN_SET(nr));
	bank->lvl1.reg  = dp52_read(dp52, DP52_PMU_LVL1_SET(nr));
	bank->lvl2.reg  = dp52_read(dp52, DP52_PMU_LVL2_SET(nr));
	bank->lvl3.reg  = dp52_read(dp52, DP52_PMU_LVL3_SET(nr));
	bank->crdc_cfg1 = dp52_read(dp52, DP52_PMU_CRDC_CFG1_SET(nr));
	bank->crdc_cfg2 = dp52_read(dp52, DP52_PMU_CRDC_CFG2_SET(nr));

	return 0;
}

static int
dp52_pmu_write_bank(struct dp52 *dp52, enum dp52_pmu_banks nr,
                    struct dp52_pmu_bank *bank)
{
	if (nr < 0 || nr >= PMU_NUM_BANKS || !bank)
		return -EINVAL;

	/* handle the software bank seperately (direct access) */
	if (nr == PMU_SWBANK) {
		dp52_write(dp52, DP52_PMU_EN_SW,        bank->enable);
		dp52_write(dp52, DP52_PMU_LVL1_SW,      bank->lvl1.reg);
		dp52_write(dp52, DP52_PMU_LVL2_SW,      bank->lvl2.reg);
		dp52_write(dp52, DP52_PMU_LVL3_SW,      bank->lvl3.reg);
		dp52_write(dp52, DP52_PMU_CRDC_CFG1_SW, bank->crdc_cfg1);
		dp52_write(dp52, DP52_PMU_CRDC_CFG2_SW, bank->crdc_cfg2);
		return 0;
	}

	dp52_write(dp52, DP52_PMU_EN_SET(nr), bank->enable);
	dp52_write(dp52, DP52_PMU_LVL1_SET(nr), bank->lvl1.reg);
	dp52_write(dp52, DP52_PMU_LVL2_SET(nr), bank->lvl2.reg);
	dp52_write(dp52, DP52_PMU_LVL3_SET(nr), bank->lvl3.reg);
	dp52_write(dp52, DP52_PMU_CRDC_CFG1_SET(nr), bank->crdc_cfg1);
	dp52_write(dp52, DP52_PMU_CRDC_CFG2_SET(nr), bank->crdc_cfg2);

	return 0;
}

/*
 * regulator implementation
 */

static int dp52_pmu_swcontrol;

static int dp52_reg_enable(struct regulator_dev *rdev)
{
	struct dp52_regulator_info *info = rdev_get_drvdata(rdev);
	uint16_t reg;

	if (dp52_pmu_swcontrol) {
		reg = dp52_read(info->dp52, DP52_PMU_EN_SW);
		reg |= (1 << info->enable_bit);
		dp52_write(info->dp52, DP52_PMU_EN_SW, reg);
		return 0;
	}

	/* hardware controls the regulators -> update both bank0 and bank1 */
	reg = dp52_read(info->dp52, DP52_PMU_EN_SET(0));
	reg |= (1 << info->enable_bit);
	dp52_write(info->dp52, DP52_PMU_EN_SET(0), reg);

	reg = dp52_read(info->dp52, DP52_PMU_EN_SET(1));
	reg |= (1 << info->enable_bit);
	dp52_write(info->dp52, DP52_PMU_EN_SET(1), reg);

	return 0;
}

static int dp52_reg_disable(struct regulator_dev *rdev)
{
	struct dp52_regulator_info *info = rdev_get_drvdata(rdev);
	uint16_t reg;

	if (dp52_pmu_swcontrol) {
		reg = dp52_read(info->dp52, DP52_PMU_EN_SW);
		reg &= ~(1 << info->enable_bit);
		dp52_write(info->dp52, DP52_PMU_EN_SW, reg);
		return 0;
	}

	/* hardware controls the regulators -> update both bank0 and bank1 */
	reg = dp52_read(info->dp52, DP52_PMU_EN_SET(0));
	reg &= ~(1 << info->enable_bit);
	dp52_write(info->dp52, DP52_PMU_EN_SET(0), reg);

	reg = dp52_read(info->dp52, DP52_PMU_EN_SET(1));
	reg &= ~(1 << info->enable_bit);
	dp52_write(info->dp52, DP52_PMU_EN_SET(1), reg);

	return 0;
}

static int dp52_reg_is_enabled(struct regulator_dev *rdev)
{
	struct dp52_regulator_info *info = rdev_get_drvdata(rdev);
	uint16_t reg;

	if (dp52_pmu_swcontrol)
		reg = dp52_read(info->dp52, DP52_PMU_EN_SW);
	else
		reg = dp52_read(info->dp52, DP52_PMU_EN_SET(0));

	return reg & (1 << info->enable_bit);
}

static int
dp52_reg_set_voltage(struct regulator_dev *rdev, int min_uV, int max_uV,
                     unsigned *selector)
{
	struct dp52_regulator_info *info = rdev_get_drvdata(rdev);
	int steps = rdev->desc->n_voltages;
	int off = info->level_bit / 16;
	int bit = info->level_bit & 15;
	uint16_t reg;
	unsigned vsel;

	if (min_uV < info->min_uV || max_uV > info->max_uV)
		return -EINVAL;

	for (vsel = 0; vsel < steps; vsel++) {
		int uV = info->min_uV + info->step_uV * vsel;

		if (min_uV <= uV && uV <= max_uV)
			break;
	}

	*selector = vsel;

	if (dp52_pmu_swcontrol) {
		reg = dp52_read(info->dp52, DP52_PMU_LVL1_SW + off);
		reg &= ~(info->level_msk << bit);
		reg |= vsel << bit;
		dp52_write(info->dp52, DP52_PMU_LVL1_SW + off, reg);
		return 0;
	}

	/* hardware controls the regulators -> update both bank0 and bank1 */
	reg = dp52_read(info->dp52, DP52_PMU_LVL1_SET(0) + off);
	reg &= ~(info->level_msk << bit);
	reg |= vsel << bit;
	dp52_write(info->dp52, DP52_PMU_LVL1_SET(0) + off, reg);

	reg = dp52_read(info->dp52, DP52_PMU_LVL1_SET(1) + off);
	reg &= ~(info->level_msk << bit);
	reg |= vsel << bit;
	dp52_write(info->dp52, DP52_PMU_LVL1_SET(1) + off, reg);

	return 0;
}

static int
dp52_reg_get_voltage(struct regulator_dev *rdev)
{
	struct dp52_regulator_info *info = rdev_get_drvdata(rdev);
	int off = info->level_bit / 16;
	int bit = info->level_bit & 15;
	uint16_t reg;
	int vsel;

	if (dp52_pmu_swcontrol)
		reg = dp52_read(info->dp52, DP52_PMU_LVL1_SW + off);
	else
		reg = dp52_read(info->dp52, DP52_PMU_LVL1_SET(0) + off);

	vsel = (reg >> bit) & info->level_msk;
	return info->min_uV + info->step_uV * vsel;
}

static int
dp52_reg_list_voltage(struct regulator_dev *rdev, unsigned selector)
{
	struct dp52_regulator_info *info = rdev_get_drvdata(rdev);
	return info->min_uV + info->step_uV * selector;
}

static struct regulator_ops dp52_regulator_ops = {
	.enable = dp52_reg_enable,
	.disable = dp52_reg_disable,
	.is_enabled = dp52_reg_is_enabled,
	.set_voltage = dp52_reg_set_voltage,
	.get_voltage = dp52_reg_get_voltage,
	.list_voltage = dp52_reg_list_voltage,
};

#define DP52_REGULATOR(_name, min, max, step, en, lvl, bits)              \
{                                                                         \
	.desc = {                                                         \
		.name = #_name,                                           \
		.ops = &dp52_regulator_ops,                               \
		.type = REGULATOR_VOLTAGE,                                \
		.id = DP52_RID_##_name,                                   \
		.n_voltages = ((max) - (min)) / (step) + 1,               \
		.owner = THIS_MODULE,                                     \
	},                                                                \
	.min_uV = (min) * 1000,                                           \
	.max_uV = (max) * 1000,                                           \
	.step_uV = (step) * 1000,                                         \
	.enable_bit = (en),                                               \
	.level_bit = (lvl),                                               \
	.level_msk = (1 << (bits)) - 1,                                   \
}

static struct dp52_regulator_info dp52_regulator_info[] = {
	DP52_REGULATOR(SPARE2, 1500, 3600, 100, 1, 21, 5),
	DP52_REGULATOR(SPARE3, 1500, 3600, 100, 0, 16, 5),
};

static int
dp52_pmu_add_regulator(struct device *dev, int id,
                       struct regulator_init_data *init)
{
	struct dp52_regulator_info *info = &dp52_regulator_info[id];
	int ret = 0;

	info->dp52 = dev_get_drvdata(dev);
	info->rdev = regulator_register(&info->desc, dev, init, info);
	if (IS_ERR(info->rdev)) {
		dev_err(dev, "cannot register regulator\n");
		ret = PTR_ERR(info->rdev);
	}

	return ret;
}

static struct dp52 *dp52_pm_dev = NULL;

static void dp52_pm_power_off(void)
{
	for (;;) {
		/* restore default configuration */
		dp52_write(dp52_pm_dev, DP52_PMU_CTRL, 0x1);
		dp52_write(dp52_pm_dev, DP52_PMU_EN_SW, 0x7a8);
		/* power-off sequence */
		dp52_write(dp52_pm_dev, DP52_PMU_OFF, 0xaaaa);
		dp52_write(dp52_pm_dev, DP52_PMU_OFF, 0x5555);
		dp52_write(dp52_pm_dev, DP52_PMU_OFF, 0xaaaa);
	}
}

int dp52_pmu_probe(struct device *dev, struct dp52_platform_data *pdata)
{
	struct dp52 *dp52 = dev_get_drvdata(dev);
	struct dp52_pmu_bank bank;
	uint16_t reg;

	if (!pdata->spare2 || !pdata->spare3)
		return -EINVAL;

	/* if we do not use the HILO pins, directly register the regulators */
	if (!pdata->hilo_lowpower) {
		dp52_pmu_swcontrol = 1;
		goto add_regulators;
	}

	/*
	 * because of a bug in the hardware we have to initialize the hardware
	 * banks with their reset values
	 */
	dp52_pmu_init_hwbanks(dp52);

	/*
	 * by default, the dp52 pmu is controlled by the software bank. we
	 * assume that the bootloader has already configured it with all the
	 * needed voltage levels.
	 */
	dp52_pmu_read_bank(dp52, PMU_SWBANK, &bank);

	/*
	 * we setup two bank configurations in order to facilitate the
	 * hardware support (HILO lines):
	 *
	 * bank0 - full power: rf dcdc used, core dcdc in high frequency
	 * bank1 - low power:  rf ldo used, core dcdc in low frequency
	 *
	 * the mapping between HILO signals and banks is provided via the
	 * platform data.
	 */
	bank.enable |= PMU_ENABLE_RF_HILO;
	dp52_pmu_write_bank(dp52, PMU_HWBANK0, &bank);

	bank.enable &= ~PMU_ENABLE_RF_HILO;
	dp52_pmu_write_bank(dp52, PMU_HWBANK1, &bank);

	reg = dp52_read(dp52, DP52_PMU_CTRL);
	reg &= ~0x07fe; /* clear out hilo mappings */

	/*
	 * map each hilo configurations to either bank0 or bank1, depending
	 * on whether it has been set as lowpower or not
	 */
	reg |= ((!!(pdata->hilo_lowpower & DP52_HILO_00)) << 1);
	reg |= ((!!(pdata->hilo_lowpower & DP52_HILO_01)) << 3);
	reg |= ((!!(pdata->hilo_lowpower & DP52_HILO_10)) << 5);
	reg |= ((!!(pdata->hilo_lowpower & DP52_HILO_11)) << 7);
	dp52_write(dp52, DP52_PMU_CTRL, reg);

	/* hardware controls regulators */
	dp52_clr_bits(dp52, DP52_PMU_CTRL, 1);

add_regulators:
	dp52_pmu_add_regulator(dev, DP52_RID_SPARE2, pdata->spare2);
	dp52_pmu_add_regulator(dev, DP52_RID_SPARE3, pdata->spare3);

	dp52_pm_dev = dp52;
	pm_power_off = dp52_pm_power_off;

	dev_info(dev, "pmu driver initialized\n");

	return 0;
}

void dp52_pmu_remove(struct device *dev)
{
	int i;

	dp52_pm_dev = NULL;
	pm_power_off = NULL;

	for (i=0; i < ARRAY_SIZE(dp52_regulator_info); i++)
		regulator_unregister(dp52_regulator_info[i].rdev);
}

