/*
 * drivers/media/video/ov2643.c
 *
 * ov2643 sensor driver
 *
 *
 * Copyright (C) 2011 DSPG.
 *
 * Leverage ov2643.c
 *
 * This file is licensed under the terms of the GNU General Public License
 * version 2. This program is licensed "as is" without any warranty of any
 * kind, whether express or implied.
 */

#include <linux/io.h>
#include <linux/i2c.h>
#include <linux/delay.h>
#include <media/v4l2-int-device.h>
#include <media/ov2643.h>
#include <linux/dma-mapping.h> /*For page align*/
#include "ov2643_regs.h"
#include <media/dmw96ciu.h>
#include <mach/hardware.h>

#define OV2643_DRIVER_NAME  "ov2643"
#define MOD_NAME "OV2643: "

#define I2C_M_WR 0

#define OV2643_USE_XCLKA       0
#define OV2643_USE_XCLKB       1

#define OV2643_CSI2_VIRTUAL_ID 0x1

/* FPS Capabilities */
#define OV2643_MIN_FPS                 3
#define OV2643_DEF_FPS                 15
#define OV2643_MAX_FPS                 30

#define OV2643_MIN_BRIGHT              0
#define OV2643_MAX_BRIGHT              6
#define OV2643_DEF_BRIGHT              0
#define OV2643_BRIGHT_STEP             1

#define OV2643_DEF_CONTRAST            0
#define OV2643_MIN_CONTRAST            0
#define OV2643_MAX_CONTRAST            6
#define OV2643_CONTRAST_STEP           1

#define OV2643_DEF_COLOR               0
#define OV2643_MIN_COLOR               0
#define OV2643_MAX_COLOR               2
#define OV2643_COLOR_STEP              1

#define OV2643_DEF_FLIP               	0

#define OV2643_DEF_MIRRO               	0

#define SENSOR_DETECTED                1
#define SENSOR_NOT_DETECTED    0

/* NOTE: Set this as 0 for enabling SoC mode */
#define OV2643_RAW_MODE        0

/* XCLK Frequency in Hz*/
#define OV2643_XCLK_MIN                24000000
#define OV2643_XCLK_MAX                24000000


/* High byte of product ID */
#define OV2643_PIDH_MAGIC      0x26
/* Low byte of product ID  */
#define OV2643_PIDL_MAGIC1     0x43


#define OV2643_REG_TERM 0xFFFF /* terminating list entry for reg */
#define OV2643_VAL_TERM 0xFF   /* terminating list entry for val */

struct ov2643_reg {
       unsigned int reg;
       unsigned char val;
};


enum image_size_ov {
	QCIF,
	CIF,
	SVGA,
	HD720P,
	UXGA,
};	


enum pixel_format_ov {
	YUV,
	RGB565,
	RGB555,
	RAW10
};

enum ov2643_frameintervals_type {
	OV26_15_FPS,
	OV26_30_FPS,
	};

#define OV_NUM_IMAGE_SIZES             2
#define OV_NUM_PIXEL_FORMATS           4

struct capture_size_ov {
       unsigned long width;
       unsigned long height;
};

struct reg_value {
      u16 u16RegAddr;
      u8 u8Val;
      u8 u8Mask;
      u32 u32Delay_ms;
};

enum ov2643_mode {
    OV2643_MODE_MIN = 0,
	OV2643_MODE_QCIF_200_150 = 0,
	OV2643_MODE_CIF_400_300 = 1,
    OV2643_MODE_SVGA_800_600 = 2,
    OV2643_MODE_720P_1280_720 = 3,
    OV2643_MODE_UXGA_1600_1200 = 4,
    OV2643_MODE_MAX = 4
};

const struct v4l2_fract ov2643_frameintervals[] = {
		{ .numerator = 1, .denominator = 15 },
		{ .numerator = 1, .denominator = 30 },
};

/* Array of image sizes supported by OV2643.  These must be ordered from
 * smallest image size to largest.
 */
const static struct capture_size_ov ov2643_sizes[] = {
		/* QCIF */
		{200, 150},
		/* CIF */
		{400, 300},
       /* SVGA */
       { 800, 600 },
       /* 720P */
       { 1280, 720 },
       /* UXGA */
       { 1600, 1200 },
};

const char* frame_rate_tostring(unsigned int frame_rate)
{
	switch (frame_rate)
	{
		case OV26_15_FPS:
			return "OV26_15_FPS";
		case OV26_30_FPS:
			return "OV26_30_FPS";
		default:
			return "INVALID";
	}
}


const char* mode_tostring(unsigned int mode)
{
	switch (mode)
	{
		case OV2643_MODE_QCIF_200_150:
			return "OV2643_MODE_QCIF_200_150";
		case OV2643_MODE_CIF_400_300:
			return "OV2643_MODE_CIF_400_300";
		case OV2643_MODE_SVGA_800_600:
			return "OV2643_MODE_SVGA_800_600";
		case OV2643_MODE_720P_1280_720:
			return "OV2643_MODE_720P_1280_720";
		case OV2643_MODE_UXGA_1600_1200:
			return "OV2643_MODE_UXGA_1600_1200";
		default:
			return "INVALID";
	}
}

/* List of image formats supported by OV2643 sensor */
const static struct v4l2_fmtdesc ov2643_formats[] = {
#if OV2643_RAW_MODE
       {
               .description    = "RAW10",
               .pixelformat    = V4L2_PIX_FMT_SGRBG10,
       },
#else
	   {
               /* Note:  V4L2 defines FMT_NV16 as:
                *	Two planes -- one Y, one Cr + Cb interleaved
                *   y0y1y2y3 u0v0u1v1
                */
		       .description    = "Y/CbCr, 4:2:2",
               .pixelformat    = V4L2_PIX_FMT_NV16,
	   },
	   {
               /* Note:  V4L2 defines FMT_NV61 as:
                *	Two planes -- one Y, one Cr + Cb interleaved
                *   y0y1y2y3 v0u0v1u1
                */
		       .description    = "Y/CrCb, 4:2:2",
               .pixelformat    = V4L2_PIX_FMT_NV61,
	   },

	   {
               /* Note:  V4L2 defines FMT_NV12 as:
                *	Two planes -- one Y, one Cr + Cb interleaved
                *   y0y1y2y3 u0v0
                */
		       .description    = "Y/CbCr, 4:2:0",
               .pixelformat    = V4L2_PIX_FMT_NV12,
	   },
	   {
               /* Note:  V4L2 defines FMT_NV12 as:
                *	Two planes -- one Y, one Cr + Cb interleaved
                *   y0y1y2y3 v0u0
                */
		       .description    = "Y/CrCb, 4:2:0",
               .pixelformat    = V4L2_PIX_FMT_NV21,
	   },
       {
               /* Note:  V4L2 defines RGB565 as:
                *
                *      Byte 0                    Byte 1
                *      g2 g1 g0 r4 r3 r2 r1 r0   b4 b3 b2 b1 b0 g5 g4 g3
                *
                * We interpret RGB565 as:
                *
                *      Byte 0                    Byte 1
                *      g2 g1 g0 b4 b3 b2 b1 b0   r4 r3 r2 r1 r0 g5 g4 g3
                */
               .description    = "RGB565, le",
               .pixelformat    = V4L2_PIX_FMT_RGB565,
       },
       {
               /* Note:  V4L2 defines RGB565X as:
                *
                *      Byte 0                    Byte 1
                *      b4 b3 b2 b1 b0 g5 g4 g3   g2 g1 g0 r4 r3 r2 r1 r0
                *
                * We interpret RGB565X as:
                *
                *      Byte 0                    Byte 1
                *      r4 r3 r2 r1 r0 g5 g4 g3   g2 g1 g0 b4 b3 b2 b1 b0
                */
               .description    = "RGB565, be",
               .pixelformat    = V4L2_PIX_FMT_RGB565X,
       },
       {
               .description    = "YUYV (YUV 4:2:2), packed",
               .pixelformat    = V4L2_PIX_FMT_YUYV,
       },
       {
               .description    = "UYVY, packed",
               .pixelformat    = V4L2_PIX_FMT_UYVY,
       },
       {
               /* Note:  V4L2 defines RGB555 as:
                *
                *      Byte 0                    Byte 1
                *      g2 g1 g0 r4 r3 r2 r1 r0   x  b4 b3 b2 b1 b0 g4 g3
                *
                * We interpret RGB555 as:
                *
                *      Byte 0                    Byte 1
                *      g2 g1 g0 b4 b3 b2 b1 b0   x  r4 r3 r2 r1 r0 g4 g3
                */
               .description    = "RGB555, le",
               .pixelformat    = V4L2_PIX_FMT_RGB555,
       },
       {
               /* Note:  V4L2 defines RGB555X as:
                *
                *      Byte 0                    Byte 1
                *      x  b4 b3 b2 b1 b0 g4 g3   g2 g1 g0 r4 r3 r2 r1 r0
                *
                * We interpret RGB555X as:
                *
                *      Byte 0                    Byte 1
                *      x  r4 r3 r2 r1 r0 g4 g3   g2 g1 g0 b4 b3 b2 b1 b0
                */
               .description    = "RGB555, be",
               .pixelformat    = V4L2_PIX_FMT_RGB555X,
       },

#endif
};

#define NUM_CAPTURE_FORMATS (sizeof(ov2643_formats) / sizeof(ov2643_formats[0]))
#define NUM_DISCRETE_SIZES (sizeof(ov2643_sizes) / sizeof(ov2643_sizes[0]))
#define NUM_TIMEFRAME_VALS (sizeof(ov2643_frameintervals) / sizeof(ov2643_frameintervals[0])) 

struct ov2643_mode_info {
      enum ov2643_mode mode;
      u32 width;
      u32 height;
      struct reg_value *init_data_ptr;
      u32 init_data_size;
};

static struct reg_value ov2643_setting_15fps_QCIF_200_150[] = {
{0xc3, 0x1f},{0xc4, 0xff},{0x3d, 0x48},{0xdd, 0xa5},{0x0e, 0x10},{0x10, 0x0a},{0x11, 0x00},
{0x0f, 0x14},{0x0e, 0x10},{0x20, 0x01},{0x21, 0x98},{0x22, 0x00},{0x23, 0x06},{0x24, 0x32},{0x26, 0x25},
{0x27, 0x84},{0x29, 0x05},{0x2a, 0xdc},{0x2b, 0x03},{0x2c, 0x20},{0x1d, 0x04},{0x25, 0x04},{0x27, 0x84},
{0x28, 0x40},{0x12, 0x0b},{0x39, 0xd0},{0xcd, 0x13},{0x13, 0xff},{0x14, 0xa7},{0x15, 0x43},{0x3c, 0xa4},
{0x18, 0x60},{0x19, 0x50},{0x1a, 0xe2},{0x37, 0xe8},{0x16, 0x90},{0x43, 0x00},{0x40, 0xfb},{0xa9, 0x44},
{0x2f, 0xec},{0x35, 0x10},{0x36, 0x10},{0x0c, 0x00},{0x0d, 0x20},{0xd0, 0x93},{0xdc, 0x2b},{0xd9, 0x41},
{0xd3, 0x02},{0x3d, 0x08},{0x0c, 0x00},{0x18, 0x24},{0x19, 0x1c},{0x1a, 0x40},{0x65, 0x08},{0x66, 0x15},
{0x67, 0x37},{0x68, 0x56},{0x69, 0x69},{0x6a, 0x7a},{0x6b, 0x85},{0x6c, 0x90},{0x6d, 0x98},{0x6e, 0xad},
{0x6f, 0xba},{0x70, 0xc8},{0x71, 0xda},{0x72, 0xe7},{0x73, 0xf3},{0x74, 0x11},{0xab, 0x2e},{0xac, 0x48},
{0xad, 0x09},{0xae, 0x1e},{0xaf, 0xa1},{0xb0, 0xc0},{0xb1, 0xa3},{0xb2, 0xa6},{0xb3, 0x03},{0xb4, 0x98},
{0x4c, 0x03},{0x4d, 0x30},{0x4e, 0x02},{0x4f, 0x5c},{0x50, 0x56},{0x51, 0x00},{0x52, 0x66},{0x53, 0x03},
{0x54, 0x30},{0x55, 0x02},{0x56, 0x5c},{0x57, 0x40},{0x58, 0x00},{0x59, 0x66},{0x5a, 0x03},{0x5b, 0x20},
{0x5c, 0x02},{0x5d, 0x5c},{0x5e, 0x3a},{0x5f, 0x00},{0x60, 0x66},{0x41, 0x1f},{0xb5, 0x00},{0xb6, 0x02},
{0xb9, 0x40},{0xba, 0x28},{0xbf, 0x0c},{0xc0, 0x3e},{0xa3, 0x0a},{0xa4, 0x0f},{0xa5, 0x04},{0xa6, 0x12},
{0x9f, 0x0a},{0xa0, 0x0f},{0xa7, 0x0a},{0xa8, 0x0f},{0xa1, 0x10},{0xa2, 0x04},{0xa9, 0x04},{0xaa, 0xa6},
{0x75, 0x6a},{0x76, 0x11},{0x77, 0x92},{0x78, 0x21},{0x79, 0xe1},{0x7a, 0x02},{0x7c, 0x05},{0x7d, 0x08},
{0x7e, 0x08},{0x7f, 0x7c},{0x80, 0x58},{0x81, 0x2a},{0x82, 0xc5},{0x83, 0x46},{0x84, 0x3a},{0x85, 0x54},
{0x86, 0x44},{0x87, 0xf8},{0x88, 0x08},{0x89, 0x70},{0x8a, 0xf0},{0x8b, 0xf0},

{0x12, 0x0b},{0x0e, 0xb8},{0x0f, 0x14},{0x10, 0x0a},{0x11, 0x01},{0x29, 0x04},{0x2a, 0xce},{0x2b, 0x02},
{0x2c, 0x8a},{0x2d, 0x00},{0x2e, 0x00},
};

static struct reg_value ov2643_setting_15fps_CIF_400_300[] = {
{0xc3, 0x1f},{0xc4, 0xff},{0x3d, 0x48},{0xdd, 0xa5},{0x0e, 0x10},{0x10, 0x0a},{0x11, 0x00},
{0x0f, 0x14},{0x0e, 0x10},{0x20, 0x01},{0x21, 0x98},{0x22, 0x00},{0x23, 0x06},{0x24, 0x32},{0x26, 0x25},
{0x27, 0x84},{0x29, 0x05},{0x2a, 0xdc},{0x2b, 0x03},{0x2c, 0x20},{0x1d, 0x04},{0x25, 0x04},{0x27, 0x84},
{0x28, 0x40},{0x12, 0x0a},{0x39, 0xd0},{0xcd, 0x13},{0x13, 0xff},{0x14, 0xa7},{0x15, 0x43},{0x3c, 0xa4},
{0x18, 0x60},{0x19, 0x50},{0x1a, 0xe2},{0x37, 0xe8},{0x16, 0x90},{0x43, 0x00},{0x40, 0xfb},{0xa9, 0x44},
{0x2f, 0xec},{0x35, 0x10},{0x36, 0x10},{0x0c, 0x00},{0x0d, 0x20},{0xd0, 0x93},{0xdc, 0x2b},{0xd9, 0x41},
{0xd3, 0x02},{0x3d, 0x08},{0x0c, 0x00},{0x18, 0x24},{0x19, 0x1c},{0x1a, 0x40},{0x65, 0x08},{0x66, 0x15},
{0x67, 0x37},{0x68, 0x56},{0x69, 0x69},{0x6a, 0x7a},{0x6b, 0x85},{0x6c, 0x90},{0x6d, 0x98},{0x6e, 0xad},
{0x6f, 0xba},{0x70, 0xc8},{0x71, 0xda},{0x72, 0xe7},{0x73, 0xf3},{0x74, 0x11},{0xab, 0x2e},{0xac, 0x48},
{0xad, 0x09},{0xae, 0x1e},{0xaf, 0xa1},{0xb0, 0xc0},{0xb1, 0xa3},{0xb2, 0xa6},{0xb3, 0x03},{0xb4, 0x98},
{0x4c, 0x03},{0x4d, 0x30},{0x4e, 0x02},{0x4f, 0x5c},{0x50, 0x56},{0x51, 0x00},{0x52, 0x66},{0x53, 0x03},
{0x54, 0x30},{0x55, 0x02},{0x56, 0x5c},{0x57, 0x40},{0x58, 0x00},{0x59, 0x66},{0x5a, 0x03},{0x5b, 0x20},
{0x5c, 0x02},{0x5d, 0x5c},{0x5e, 0x3a},{0x5f, 0x00},{0x60, 0x66},{0x41, 0x1f},{0xb5, 0x00},{0xb6, 0x02},
{0xb9, 0x40},{0xba, 0x28},{0xbf, 0x0c},{0xc0, 0x3e},{0xa3, 0x0a},{0xa4, 0x0f},{0xa5, 0x04},{0xa6, 0x12},
{0x9f, 0x0a},{0xa0, 0x0f},{0xa7, 0x0a},{0xa8, 0x0f},{0xa1, 0x10},{0xa2, 0x04},{0xa9, 0x04},{0xaa, 0xa6},
{0x75, 0x6a},{0x76, 0x11},{0x77, 0x92},{0x78, 0x21},{0x79, 0xe1},{0x7a, 0x02},{0x7c, 0x05},{0x7d, 0x08},
{0x7e, 0x08},{0x7f, 0x7c},{0x80, 0x58},{0x81, 0x2a},{0x82, 0xc5},{0x83, 0x46},{0x84, 0x3a},{0x85, 0x54},
{0x86, 0x44},{0x87, 0xf8},{0x88, 0x08},{0x89, 0x70},{0x8a, 0xf0},{0x8b, 0xf0},

{0x12, 0x09},{0x0e, 0xb8},{0x0f, 0x14},{0x10, 0x0a},{0x11, 0x01},{0x29, 0x04},{0x2a, 0xce},{0x2b, 0x02},
{0x2c, 0x8a},{0x2d, 0x00},{0x2e, 0x00},
};

static struct reg_value ov2643_setting_15fps_SVGA_800_600[] = {
 {0x12 , 0x80},{0xc3 , 0x1f},{0xc4 , 0xff},{0x3d , 0x48},{0xdd , 0xa5},{0x0e , 0x10},{0x10 , 0x0a},{0x11 , 0x00},
 {0x0f , 0x14},{0x0e , 0x10},{0x20 , 0x01},{0x21 , 0x98},{0x22 , 0x00},{0x23 , 0x06},{0x24 , 0x32},{0x26 , 0x25},
 {0x27 , 0x84},{0x29 , 0x05},{0x2a , 0xdc},{0x2b , 0x03},{0x2c , 0x20},{0x1d , 0x04},{0x25 , 0x04},{0x27 , 0x84},
 {0x28 , 0x40},{0x12 , 0x09},{0x39 , 0xd0},{0xcd , 0x13},{0x13 , 0xff},{0x14 , 0xa7},{0x15 , 0x43},{0x3c , 0xa4},
 {0x18 , 0x60},{0x19 , 0x50},{0x1a , 0xe2},{0x37 , 0xe8},{0x16 , 0x90},{0x43 , 0x00},{0x40 , 0xfb},{0xa9 , 0x44},
 {0x2f , 0xec},{0x35 , 0x10},{0x36 , 0x10},{0x0c , 0x00},{0x0d , 0x20},{0xd0 , 0x93},{0xdc , 0x2b},{0xd9 , 0x41},
 {0xd3 , 0x02},{0x3d , 0x08},{0x0c , 0x00},{0x18 , 0x24},{0x19 , 0x1c},{0x1a , 0x40},{0x65 , 0x08},{0x66 , 0x15},
 {0x67 , 0x37},{0x68 , 0x56},{0x69 , 0x69},{0x6a , 0x7a},{0x6b , 0x85},{0x6c , 0x90},{0x6d , 0x98},{0x6e , 0xad},
 {0x6f , 0xba},{0x70 , 0xc8},{0x71 , 0xda},{0x72 , 0xe7},{0x73 , 0xf3},{0x74 , 0x11},{0xab , 0x2e},{0xac , 0x48},
 {0xad , 0x09},{0xae , 0x1e},{0xaf , 0xa1},{0xb0 , 0xc0},{0xb1 , 0xa3},{0xb2 , 0xa6},{0xb3 , 0x03},{0xb4 , 0x98},
 {0x4c , 0x03},{0x4d , 0x30},{0x4e , 0x02},{0x4f , 0x5c},{0x50 , 0x56},{0x51 , 0x00},{0x52 , 0x66},{0x53 , 0x03},
 {0x54 , 0x30},{0x55 , 0x02},{0x56 , 0x5c},{0x57 , 0x40},{0x58 , 0x00},{0x59 , 0x66},{0x5a , 0x03},{0x5b , 0x20},
 {0x5c , 0x02},{0x5d , 0x5c},{0x5e , 0x3a},{0x5f , 0x00},{0x60 , 0x66},{0x41 , 0x1f},{0xb5 , 0x00},{0xb6 , 0x02},
 {0xb9 , 0x40},{0xba , 0x28},{0xbf , 0x0c},{0xc0 , 0x3e},{0xa3 , 0x0a},{0xa4 , 0x0f},{0xa5 , 0x04},{0xa6 , 0x12},
 {0x9f , 0x0a},{0xa0 , 0x0f},{0xa7 , 0x0a},{0xa8 , 0x0f},{0xa1 , 0x10},{0xa2 , 0x04},{0xa9 , 0x04},{0xaa , 0xa6},
 {0x75 , 0x6a},{0x76 , 0x11},{0x77 , 0x92},{0x78 , 0x21},{0x79 , 0xe1},{0x7a , 0x02},{0x7c , 0x05},{0x7d , 0x08},
 {0x7e , 0x08},{0x7f , 0x7c},{0x80 , 0x58},{0x81 , 0x2a},{0x82 , 0xc5},{0x83 , 0x46},{0x84 , 0x3a},{0x85 , 0x54},
 {0x86 , 0x44},{0x87 , 0xf8},{0x88 , 0x08},{0x89 , 0x70},{0x8a , 0xf0},{0x8b , 0xf0},

 {0x12, 0x09},{0x0e, 0xb8},{0x0f, 0x14},{0x10, 0x0a},{0x11, 0x01},{0x29, 0x04},{0x2a, 0xce},{0x2b, 0x02},{0x2c, 0x8a},
 {0x2d, 0x00},{0x2e, 0x00},
			 
};           

static struct reg_value ov2643_setting_15fps_720P_1280_720[] = {
{0x12 , 0x80},{0xc3 , 0x1f},{0xc4 , 0xff},{0x3d , 0x48},{0xdd , 0xa5},{0x0e , 0x10},{0x10 , 0x0a},{0x11 , 0x00},
{0x0f , 0x24},{0x0e , 0x10},{0x20 , 0x01},{0x21 , 0x25},{0x22 , 0x00},{0x23 , 0x0c},{0x24 , 0x50},{0x26 , 0x2d},
{0x27 , 0x04},{0x29 , 0x06},{0x2a , 0x40},{0x2b , 0x02},{0x2c , 0xee},{0x1d , 0x04},{0x25 , 0x04},{0x27 , 0x04},
{0x28 , 0x40},{0x12 , 0x48},{0x39 , 0x10},{0xcd , 0x12},{0x13 , 0xff},{0x14 , 0xa7},{0x15 , 0x42},{0x3c , 0xa4},
{0x18 , 0x60},{0x19 , 0x50},{0x1a , 0xe2},{0x37 , 0xe8},{0x16 , 0x90},{0x43 , 0x00},{0x40 , 0xfb},{0xa9 , 0x44},
{0x2f , 0xec},{0x35 , 0x10},{0x36 , 0x10},{0x0c , 0x00},{0x0d , 0x20},{0xd0 , 0x93},{0xdc , 0x2b},{0xd9 , 0x41},
{0xd3 , 0x02},{0x3d , 0x08},{0x0c , 0x00},{0x18 , 0x2c},{0x19 , 0x24},{0x1a , 0x71},{0x9b , 0x69},{0x9c , 0x7d},
{0x9d , 0x7d},{0x9e , 0x69},{0x35 , 0x04},{0x36 , 0x04},{0x65 , 0x12},{0x66 , 0x20},{0x67 , 0x39},{0x68 , 0x4e},
{0x69 , 0x62},{0x6a , 0x74},{0x6b , 0x85},{0x6c , 0x92},{0x6d , 0x9e},{0x6e , 0xb2},{0x6f , 0xc0},{0x70 , 0xcc},
{0x71 , 0xe0},{0x72 , 0xee},{0x73 , 0xf6},{0x74 , 0x11},{0xab , 0x20},{0xac , 0x5b},{0xad , 0x05},{0xae , 0x1b},
{0xaf , 0x76},{0xb0 , 0x90},{0xb1 , 0x90},{0xb2 , 0x8c},{0xb3 , 0x04},{0xb4 , 0x98},{0x4c , 0x03},{0x4d , 0x30},
{0x4e , 0x02},{0x4f , 0x5c},{0x50 , 0x56},{0x51 , 0x00},{0x52 , 0x66},{0x53 , 0x03},{0x54 , 0x30},{0x55 , 0x02},
{0x56 , 0x5c},{0x57 , 0x40},{0x58 , 0x00},{0x59 , 0x66},{0x5a , 0x03},{0x5b , 0x20},{0x5c , 0x02},{0x5d , 0x5c},
{0x5e , 0x3a},{0x5f , 0x00},{0x60 , 0x66},{0x41 , 0x1f},{0xb5 , 0x01},{0xb6 , 0x02},{0xb9 , 0x40},{0xba , 0x28},
{0xbf , 0x0c},{0xc0 , 0x3e},{0xa3 , 0x0a},{0xa4 , 0x0f},{0xa5 , 0x09},{0xa6 , 0x16},{0x9f , 0x0a},{0xa0 , 0x0f},
{0xa7 , 0x0a},{0xa8 , 0x0f},{0xa1 , 0x10},{0xa2 , 0x04},{0xa9 , 0x04},{0xaa , 0xa6},{0x75 , 0x6a},{0x76 , 0x11},
{0x77 , 0x92},{0x78 , 0x21},{0x79 , 0xe1},{0x7a , 0x02},{0x7c , 0x05},{0x7d , 0x08},{0x7e , 0x08},{0x7f , 0x7c},
{0x80 , 0x58},{0x81 , 0x2a},{0x82 , 0xc5},{0x83 , 0x46},{0x84 , 0x3a},{0x85 , 0x54},{0x86 , 0x44},{0x87 , 0xf8},
{0x88 , 0x08},{0x89 , 0x70},{0x8a , 0xf0},{0x8b , 0xf0},
};

static struct reg_value ov2643_setting_15fps_UXGA_1600_1200[] = {

{0x12  , 0x80},{0xc3  , 0x1f},{0xc4  , 0xff},{0x3d  , 0x48},{0xdd  , 0xa5},{0x0e  , 0x10},{0x10  , 0x0a},
{0x11  , 0x00},{0x0f  , 0x14},{0x0e  , 0x10},{0x21  , 0x25},{0x23  , 0x0c},{0x12  , 0x08},{0x39  , 0x10},
{0xcd  , 0x12},{0x13  , 0xff},{0x14  , 0xa7},{0x15  , 0x43},{0x3c  , 0xa4},{0x18  , 0x60},{0x19  , 0x50},
{0x1a  , 0xe2},{0x37  , 0xe8},{0x16  , 0x90},{0x43  , 0x00},{0x40  , 0xfb},{0xa9  , 0x44},{0x2f  , 0xec},
{0x35  , 0x10},{0x36  , 0x10},{0x0c  , 0x00},{0x0d  , 0x20},{0xd0  , 0x93},{0xdc  , 0x2b},{0xd9  , 0x41},
{0xd3  , 0x02},{0x3d  , 0x08},{0x0c  , 0x00},{0x18  , 0x24},{0x19  , 0x1c},{0x1a  , 0x40},{0x65  , 0x08},
{0x66  , 0x15},{0x67  , 0x37},{0x68  , 0x56},{0x69  , 0x69},{0x6a  , 0x7a},{0x6b  , 0x85},{0x6c  , 0x90},
{0x6d  , 0x98},{0x6e  , 0xad},{0x6f  , 0xba},{0x70  , 0xc8},{0x71  , 0xda},{0x72  , 0xe7},{0x73  , 0xf3},
{0x74  , 0x11},{0xab  , 0x2e},{0xac  , 0x48},{0xad  , 0x09},{0xae  , 0x1e},{0xaf  , 0xa1},{0xb0  , 0xc0},
{0xb1  , 0xa3},{0xb2  , 0xa6},{0xb3  , 0x03},{0xb4  , 0x98},{0x4c  , 0x03},{0x4d  , 0x30},{0x4e  , 0x02},
{0x4f  , 0x5c},{0x50  , 0x56},{0x51  , 0x00},{0x52  , 0x66},{0x53  , 0x03},{0x54  , 0x30},{0x55  , 0x02},
{0x56  , 0x5c},{0x57  , 0x40},{0x58  , 0x00},{0x59  , 0x66},{0x5a  , 0x03},{0x5b  , 0x20},{0x5c  , 0x02},
{0x5d  , 0x5c},{0x5e  , 0x3a},{0x5f  , 0x00},{0x60  , 0x66},{0x41  , 0x1f},{0xb5  , 0x00},{0xb6  , 0x02},
{0xb9  , 0x40},{0xba  , 0x28},{0xbf  , 0x0c},{0xc0  , 0x3e},{0xa3  , 0x0a},{0xa4  , 0x0f},{0xa5  , 0x04},
{0xa6  , 0x12},{0x9f  , 0x0a},{0xa0  , 0x0f},{0xa7  , 0x0a},{0xa8  , 0x0f},{0xa1  , 0x10},{0xa2  , 0x04},
{0xa9  , 0x04},{0xaa  , 0xa6},{0x75  , 0x6a},{0x76  , 0x11},{0x77  , 0x92},{0x78  , 0x21},{0x79  , 0xe1},
{0x7a  , 0x02},{0x7c  , 0x05},{0x7d  , 0x08},{0x7e  , 0x08},{0x7f  , 0x7c},{0x80  , 0x58},{0x81  , 0x2a},
{0x82  , 0xc5},{0x83  , 0x46},{0x84  , 0x3a},{0x85  , 0x54},{0x86  , 0x44},{0x87  , 0xf8},{0x88  , 0x08},
{0x89  , 0x70},{0x8a  , 0xf0},{0x8b  , 0xf0},

};

static struct reg_value ov2643_setting_30fps_QCIF_200_150[] = {
{0xc3, 0x1f},{0xc4, 0xff},{0x3d, 0x48},{0xdd, 0xa5},{0x0e, 0x10},{0x10, 0x0a},{0x11, 0x00},
{0x0f, 0x14},{0x0e, 0x10},{0x20, 0x01},{0x21, 0x98},{0x22, 0x00},{0x23, 0x06},{0x24, 0x32},{0x26, 0x25},
{0x27, 0x84},{0x29, 0x05},{0x2a, 0xdc},{0x2b, 0x03},{0x2c, 0x20},{0x1d, 0x04},{0x25, 0x04},{0x27, 0x84},
{0x28, 0x40},{0x12, 0x0b},{0x39, 0xd0},{0xcd, 0x13},{0x13, 0xff},{0x14, 0xa7},{0x15, 0x43},{0x3c, 0xa4},
{0x18, 0x60},{0x19, 0x50},{0x1a, 0xe2},{0x37, 0xe8},{0x16, 0x90},{0x43, 0x00},{0x40, 0xfb},{0xa9, 0x44},
{0x2f, 0xec},{0x35, 0x10},{0x36, 0x10},{0x0c, 0x00},{0x0d, 0x20},{0xd0, 0x93},{0xdc, 0x2b},{0xd9, 0x41},
{0xd3, 0x02},{0x3d, 0x08},{0x0c, 0x00},{0x18, 0x24},{0x19, 0x1c},{0x1a, 0x40},{0x65, 0x08},{0x66, 0x15},
{0x67, 0x37},{0x68, 0x56},{0x69, 0x69},{0x6a, 0x7a},{0x6b, 0x85},{0x6c, 0x90},{0x6d, 0x98},{0x6e, 0xad},
{0x6f, 0xba},{0x70, 0xc8},{0x71, 0xda},{0x72, 0xe7},{0x73, 0xf3},{0x74, 0x11},{0xab, 0x2e},{0xac, 0x48},
{0xad, 0x09},{0xae, 0x1e},{0xaf, 0xa1},{0xb0, 0xc0},{0xb1, 0xa3},{0xb2, 0xa6},{0xb3, 0x03},{0xb4, 0x98},
{0x4c, 0x03},{0x4d, 0x30},{0x4e, 0x02},{0x4f, 0x5c},{0x50, 0x56},{0x51, 0x00},{0x52, 0x66},{0x53, 0x03},
{0x54, 0x30},{0x55, 0x02},{0x56, 0x5c},{0x57, 0x40},{0x58, 0x00},{0x59, 0x66},{0x5a, 0x03},{0x5b, 0x20},
{0x5c, 0x02},{0x5d, 0x5c},{0x5e, 0x3a},{0x5f, 0x00},{0x60, 0x66},{0x41, 0x1f},{0xb5, 0x00},{0xb6, 0x02},
{0xb9, 0x40},{0xba, 0x28},{0xbf, 0x0c},{0xc0, 0x3e},{0xa3, 0x0a},{0xa4, 0x0f},{0xa5, 0x04},{0xa6, 0x12},
{0x9f, 0x0a},{0xa0, 0x0f},{0xa7, 0x0a},{0xa8, 0x0f},{0xa1, 0x10},{0xa2, 0x04},{0xa9, 0x04},{0xaa, 0xa6},
{0x75, 0x6a},{0x76, 0x11},{0x77, 0x92},{0x78, 0x21},{0x79, 0xe1},{0x7a, 0x02},{0x7c, 0x05},{0x7d, 0x08},
{0x7e, 0x08},{0x7f, 0x7c},{0x80, 0x58},{0x81, 0x2a},{0x82, 0xc5},{0x83, 0x46},{0x84, 0x3a},{0x85, 0x54},
{0x86, 0x44},{0x87, 0xf8},{0x88, 0x08},{0x89, 0x70},{0x8a, 0xf0},{0x8b, 0xf0},
};

static struct reg_value ov2643_setting_30fps_CIF_400_300[] = {
{0xc3, 0x1f},{0xc4, 0xff},{0x3d, 0x48},{0xdd, 0xa5},{0x0e, 0x10},{0x10, 0x0a},{0x11, 0x00},
{0x0f, 0x14},{0x0e, 0x10},{0x20, 0x01},{0x21, 0x98},{0x22, 0x00},{0x23, 0x06},{0x24, 0x32},{0x26, 0x25},
{0x27, 0x84},{0x29, 0x05},{0x2a, 0xdc},{0x2b, 0x03},{0x2c, 0x20},{0x1d, 0x04},{0x25, 0x04},{0x27, 0x84},
{0x28, 0x40},{0x12, 0x0a},{0x39, 0xd0},{0xcd, 0x13},{0x13, 0xff},{0x14, 0xa7},{0x15, 0x43},{0x3c, 0xa4},
{0x18, 0x60},{0x19, 0x50},{0x1a, 0xe2},{0x37, 0xe8},{0x16, 0x90},{0x43, 0x00},{0x40, 0xfb},{0xa9, 0x44},
{0x2f, 0xec},{0x35, 0x10},{0x36, 0x10},{0x0c, 0x00},{0x0d, 0x20},{0xd0, 0x93},{0xdc, 0x2b},{0xd9, 0x41},
{0xd3, 0x02},{0x3d, 0x08},{0x0c, 0x00},{0x18, 0x24},{0x19, 0x1c},{0x1a, 0x40},{0x65, 0x08},{0x66, 0x15},
{0x67, 0x37},{0x68, 0x56},{0x69, 0x69},{0x6a, 0x7a},{0x6b, 0x85},{0x6c, 0x90},{0x6d, 0x98},{0x6e, 0xad},
{0x6f, 0xba},{0x70, 0xc8},{0x71, 0xda},{0x72, 0xe7},{0x73, 0xf3},{0x74, 0x11},{0xab, 0x2e},{0xac, 0x48},
{0xad, 0x09},{0xae, 0x1e},{0xaf, 0xa1},{0xb0, 0xc0},{0xb1, 0xa3},{0xb2, 0xa6},{0xb3, 0x03},{0xb4, 0x98},
{0x4c, 0x03},{0x4d, 0x30},{0x4e, 0x02},{0x4f, 0x5c},{0x50, 0x56},{0x51, 0x00},{0x52, 0x66},{0x53, 0x03},
{0x54, 0x30},{0x55, 0x02},{0x56, 0x5c},{0x57, 0x40},{0x58, 0x00},{0x59, 0x66},{0x5a, 0x03},{0x5b, 0x20},
{0x5c, 0x02},{0x5d, 0x5c},{0x5e, 0x3a},{0x5f, 0x00},{0x60, 0x66},{0x41, 0x1f},{0xb5, 0x00},{0xb6, 0x02},
{0xb9, 0x40},{0xba, 0x28},{0xbf, 0x0c},{0xc0, 0x3e},{0xa3, 0x0a},{0xa4, 0x0f},{0xa5, 0x04},{0xa6, 0x12},
{0x9f, 0x0a},{0xa0, 0x0f},{0xa7, 0x0a},{0xa8, 0x0f},{0xa1, 0x10},{0xa2, 0x04},{0xa9, 0x04},{0xaa, 0xa6},
{0x75, 0x6a},{0x76, 0x11},{0x77, 0x92},{0x78, 0x21},{0x79, 0xe1},{0x7a, 0x02},{0x7c, 0x05},{0x7d, 0x08},
{0x7e, 0x08},{0x7f, 0x7c},{0x80, 0x58},{0x81, 0x2a},{0x82, 0xc5},{0x83, 0x46},{0x84, 0x3a},{0x85, 0x54},
{0x86, 0x44},{0x87, 0xf8},{0x88, 0x08},{0x89, 0x70},{0x8a, 0xf0},{0x8b, 0xf0},
};

static struct reg_value ov2643_setting_30fps_SVGA_800_600[] = {

{0x12  , 0x80},{0xc3  , 0x1f},{0xc4  , 0xff},{0x3d  , 0x48},{0xdd  , 0xa5},{0x0e  , 0x10},{0x10  , 0x0a},
{0x11  , 0x00},{0x0f  , 0x14},{0x0e  , 0x10},{0x20  , 0x01},{0x21  , 0x98},{0x22  , 0x00},{0x23  , 0x06},
{0x24  , 0x32},{0x26  , 0x25},{0x27  , 0x84},{0x29  , 0x05},{0x2a  , 0xdc},{0x2b  , 0x03},{0x2c  , 0x20},
{0x1d  , 0x04},{0x25  , 0x04},{0x27  , 0x84},{0x28  , 0x40},{0x12  , 0x09},{0x39  , 0xd0},{0xcd  , 0x13},
{0x13  , 0xff},{0x14  , 0xa7},{0x15  , 0x43},{0x3c  , 0xa4},{0x18  , 0x60},{0x19  , 0x50},{0x1a  , 0xe2},
{0x37  , 0xe8},{0x16  , 0x90},{0x43  , 0x00},{0x40  , 0xfb},{0xa9  , 0x44},{0x2f  , 0xec},{0x35  , 0x10},
{0x36  , 0x10},{0x0c  , 0x00},{0x0d  , 0x20},{0xd0  , 0x93},{0xdc  , 0x2b},{0xd9  , 0x41},{0xd3  , 0x02},
{0x3d  , 0x08},{0x0c  , 0x00},{0x18  , 0x24},{0x19  , 0x1c},{0x1a  , 0x40},{0x65  , 0x08},{0x66  , 0x15},
{0x67  , 0x37},{0x68  , 0x56},{0x69  , 0x69},{0x6a  , 0x7a},{0x6b  , 0x85},{0x6c  , 0x90},{0x6d  , 0x98},
{0x6e  , 0xad},{0x6f  , 0xba},{0x70  , 0xc8},{0x71  , 0xda},{0x72  , 0xe7},{0x73  , 0xf3},{0x74  , 0x11},
{0xab  , 0x2e},{0xac  , 0x48},{0xad  , 0x09},{0xae  , 0x1e},{0xaf  , 0xa1},{0xb0  , 0xc0},{0xb1  , 0xa3},
{0xb2  , 0xa6},{0xb3  , 0x03},{0xb4  , 0x98},{0x4c  , 0x03},{0x4d  , 0x30},{0x4e  , 0x02},{0x4f  , 0x5c},
{0x50  , 0x56},{0x51  , 0x00},{0x52  , 0x66},{0x53  , 0x03},{0x54  , 0x30},{0x55  , 0x02},{0x56  , 0x5c},
{0x57  , 0x40},{0x58  , 0x00},{0x59  , 0x66},{0x5a  , 0x03},{0x5b  , 0x20},{0x5c  , 0x02},{0x5d  , 0x5c},
{0x5e  , 0x3a},{0x5f  , 0x00},{0x60  , 0x66},{0x41  , 0x1f},{0xb5  , 0x00},{0xb6  , 0x02},{0xb9  , 0x40},
{0xba  , 0x28},{0xbf  , 0x0c},{0xc0  , 0x3e},{0xa3  , 0x0a},{0xa4  , 0x0f},{0xa5  , 0x04},{0xa6  , 0x12},
{0x9f  , 0x0a},{0xa0  , 0x0f},{0xa7  , 0x0a},{0xa8  , 0x0f},{0xa1  , 0x10},{0xa2  , 0x04},{0xa9  , 0x04},
{0xaa  , 0xa6},{0x75  , 0x6a},{0x76  , 0x11},{0x77  , 0x92},{0x78  , 0x21},{0x79  , 0xe1},{0x7a  , 0x02},
{0x7c  , 0x05},{0x7d  , 0x08},{0x7e  , 0x08},{0x7f  , 0x7c},{0x80  , 0x58},{0x81  , 0x2a},{0x82  , 0xc5},
{0x83  , 0x46},{0x84  , 0x3a},{0x85  , 0x54},{0x86  , 0x44},{0x87  , 0xf8},{0x88  , 0x08},{0x89  , 0x70},
{0x8a  , 0xf0},{0x8b  , 0xf0},

{0x12, 0x09},{0x0e, 0xb8},{0x0f, 0x14},{0x10, 0x0a},{0x11, 0x00},{0x29, 0x04},{0x2a, 0xce},{0x2b, 0x02},
{0x2c, 0x8a},{0x2d, 0x00},{0x2e, 0x00},


};

static struct reg_value ov2643_setting_30fps_720P_1280_720[] = {

{0x12   , 0x80},{0xc3   , 0x1f},{0xc4   , 0xff},{0x3d   , 0x48},{0xdd   , 0xa5},{0x0e   , 0x10},{0x10   , 0x0a},
{0x11   , 0x00},{0x0f   , 0x14},{0x0e   , 0x10},{0x20   , 0x01},{0x21   , 0x25},{0x22   , 0x00},{0x23   , 0x0c},
{0x24   , 0x50},{0x26   , 0x2d},{0x27   , 0x04},{0x29   , 0x06},{0x2a   , 0x40},{0x2b   , 0x02},{0x2c   , 0xee},
{0x1d   , 0x04},{0x25   , 0x04},{0x27   , 0x04},{0x28   , 0x40},{0x12   , 0x48},{0x39   , 0x10},{0xcd   , 0x12},
{0x13   , 0xff},{0x14   , 0xa7},{0x15   , 0x43},{0x3c   , 0xa4},{0x18   , 0x60},{0x19   , 0x50},{0x1a   , 0xe2},
{0x37   , 0xe8},{0x16   , 0x90},{0x43   , 0x00},{0x40   , 0xfb},{0xa9   , 0x44},{0x2f   , 0xec},{0x35   , 0x10},
{0x36   , 0x10},{0x0c   , 0x00},{0x0d   , 0x20},{0xd0   , 0x93},{0xdc   , 0x2b},{0xd9   , 0x41},{0xd3   , 0x02},
{0x3d   , 0x08},{0x0c   , 0x00},{0x18   , 0x24},{0x19   , 0x1c},{0x1a   , 0x40},{0x65   , 0x08},{0x66   , 0x15},
{0x67   , 0x37},{0x68   , 0x56},{0x69   , 0x69},{0x6a   , 0x7a},{0x6b   , 0x85},{0x6c   , 0x90},{0x6d   , 0x98},
{0x6e   , 0xad},{0x6f   , 0xba},{0x70   , 0xc8},{0x71   , 0xda},{0x72   , 0xe7},{0x73   , 0xf3},{0x74   , 0x11},
{0xab   , 0x2e},{0xac   , 0x48},{0xad   , 0x09},{0xae   , 0x1e},{0xaf   , 0xa1},{0xb0   , 0xc0},{0xb1   , 0xa3},
{0xb2   , 0xa6},{0xb3   , 0x03},{0xb4   , 0x98},{0x4c   , 0x03},{0x4d   , 0x30},{0x4e   , 0x02},{0x4f   , 0x5c},
{0x50   , 0x56},{0x51   , 0x00},{0x52   , 0x66},{0x53   , 0x03},{0x54   , 0x30},{0x55   , 0x02},{0x56   , 0x5c},
{0x57   , 0x40},{0x58   , 0x00},{0x59   , 0x66},{0x5a   , 0x03},{0x5b   , 0x20},{0x5c   , 0x02},{0x5d   , 0x5c},
{0x5e   , 0x3a},{0x5f   , 0x00},{0x60   , 0x66},{0x41   , 0x1f},{0xb5   , 0x00},{0xb6   , 0x02},{0xb9   , 0x40},
{0xba   , 0x28},{0xbf   , 0x0c},{0xc0   , 0x3e},{0xa3   , 0x0a},{0xa4   , 0x0f},{0xa5   , 0x04},{0xa6   , 0x12},
{0x9f   , 0x0a},{0xa0   , 0x0f},{0xa7   , 0x0a},{0xa8   , 0x0f},{0xa1   , 0x10},{0xa2   , 0x04},{0xa9   , 0x04},
{0xaa   , 0xa6},{0x75   , 0x6a},{0x76   , 0x11},{0x77   , 0x92},{0x78   , 0x21},{0x79   , 0xe1},{0x7a   , 0x02},
{0x7c   , 0x05},{0x7d   , 0x08},{0x7e   , 0x08},{0x7f   , 0x7c},{0x80   , 0x58},{0x81   , 0x2a},{0x82   , 0xc5},
{0x83   , 0x46},{0x84   , 0x3a},{0x85   , 0x54},{0x86   , 0x44},{0x87   , 0xf8},{0x88   , 0x08},{0x89   , 0x70},
{0x8a   , 0xf0},{0x8b   , 0xf0},

};

static struct ov2643_mode_info ov2643_mode_info_data[2][NUM_DISCRETE_SIZES] = {
      {
           {OV2643_MODE_QCIF_200_150,   200,  150,
            ov2643_setting_15fps_QCIF_200_150,
            ARRAY_SIZE(ov2643_setting_15fps_QCIF_200_150)},
			
            {OV2643_MODE_CIF_400_300,   400,  300,
            ov2643_setting_15fps_CIF_400_300,
            ARRAY_SIZE(ov2643_setting_15fps_CIF_400_300)},

			{OV2643_MODE_SVGA_800_600,    800,  600,
            ov2643_setting_15fps_SVGA_800_600,
            ARRAY_SIZE(ov2643_setting_15fps_SVGA_800_600)},

            {OV2643_MODE_720P_1280_720,   1280, 720,
            ov2643_setting_15fps_720P_1280_720,
            ARRAY_SIZE(ov2643_setting_15fps_720P_1280_720)},

            {OV2643_MODE_UXGA_1600_1200,   1600, 1200,
            ov2643_setting_15fps_UXGA_1600_1200,
            ARRAY_SIZE(ov2643_setting_15fps_UXGA_1600_1200)},
      },
      {
			{OV2643_MODE_QCIF_200_150,   200,  150,
            ov2643_setting_30fps_QCIF_200_150,
            ARRAY_SIZE(ov2643_setting_30fps_QCIF_200_150)},
			
            {OV2643_MODE_CIF_400_300,   400,  300,
            ov2643_setting_30fps_CIF_400_300,
            ARRAY_SIZE(ov2643_setting_30fps_CIF_400_300)},

			{OV2643_MODE_SVGA_800_600,    800,  600,
            ov2643_setting_30fps_SVGA_800_600,
            ARRAY_SIZE(ov2643_setting_30fps_SVGA_800_600)},

            {OV2643_MODE_720P_1280_720,   1280, 720,
            ov2643_setting_30fps_720P_1280_720,
            ARRAY_SIZE(ov2643_setting_30fps_720P_1280_720)},
			/*Not supported*/
            {OV2643_MODE_UXGA_1600_1200,   0, 0,
            NULL,
            0},
      },
};


const static struct ov2643_reg ov2643_common[OV_NUM_IMAGE_SIZES][100] = {
    /* 4:3 Default settings */
	{
		
	},

    /* 16:9 Default settings */
	{

		
	},
	   
};

const static struct ov2643_reg ov2643_standby_on[] = {

	{OV2643_REG_TERM , OV2643_VAL_TERM},
};

const static struct ov2643_reg ov2643_standby_off[] = {

	{OV2643_REG_TERM, OV2643_VAL_TERM},
};

/**
 * struct ov2643_sensor - main structure for storage of sensor information
 * @pdata: access functions and data for platform level information
 * @v4l2_int_device: V4L2 device structure structure
 * @i2c_client: iic client device structure
 * @pix: V4L2 pixel format information structure
 * @timeperframe: time per frame expressed as V4L fraction
 * @isize: base image size
 * @ver: ov2643 chip version
 * @width: configured width
 * @height: configuredheight
 * @vsize: vertical size for the image
 * @hsize: horizontal size for the image
 * @crop_rect: crop rectangle specifying the left,top and width and height
 */
struct ov2643_sensor {
       const struct ov2643_platform_data *pdata;
       struct v4l2_int_device *v4l2_int_device;
       struct i2c_client *i2c_client;
       struct v4l2_pix_format pix;
       struct v4l2_fract timeperframe;
       int isize;
       int ver;
       int fps;
       unsigned long width;
       unsigned long height;
       unsigned long vsize;
       unsigned long hsize;
       struct v4l2_rect crop_rect;
	   unsigned int vflip;
	   unsigned int hflip;
       int state;
};

static struct ov2643_sensor ov2643;
static struct i2c_driver ov2643sensor_i2c_driver;
static unsigned long xclk_current = OV2643_XCLK_MIN;

/* Brightness Settings - 7 levels */
const static struct ov2643_reg brightness[7][5] = {
       {
               {OV2643_SDE_CTRL, 0x04},
               {OV2643_SGNSET, 0x09},
               {OV2643_YBRIGHT, 0x30},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       {
               {OV2643_SDE_CTRL, 0x04},
               {OV2643_SGNSET, 0x09},
               {OV2643_YBRIGHT, 0x20},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       {
               {OV2643_SDE_CTRL, 0x04},
               {OV2643_SGNSET, 0x09},
               {OV2643_YBRIGHT, 0x10},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       {
               {OV2643_SDE_CTRL, 0x04},
               {OV2643_SGNSET, 0x01},
               {OV2643_YBRIGHT, 0x00},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       {
               {OV2643_SDE_CTRL, 0x04},
               {OV2643_SGNSET, 0x01},
               {OV2643_YBRIGHT, 0x10},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       {
               {OV2643_SDE_CTRL, 0x04},
               {OV2643_SGNSET, 0x01},
               {OV2643_YBRIGHT, 0x20},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       {
               {OV2643_SDE_CTRL, 0x04},
               {OV2643_SGNSET, 0x01},
               {OV2643_YBRIGHT, 0x30},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
};

/* Contrast Settings - 7 levels */
const static struct ov2643_reg contrast[7][5] = {
       {
               {OV2643_SDE_CTRL, 0x04},
               {OV2643_YOFFSET, 0x14},
               {OV2643_YGAIN, 0x14},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       {
               {OV2643_SDE_CTRL, 0x04},
               {OV2643_YOFFSET, 0x18},
               {OV2643_YGAIN, 0x18},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       {
               {OV2643_SDE_CTRL, 0x04},
               {OV2643_YOFFSET, 0x1c},
               {OV2643_YGAIN, 0x1c},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       {
               {OV2643_SDE_CTRL, 0x04},
               {OV2643_YOFFSET, 0x20},
               {OV2643_YGAIN, 0x20},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       {
               {OV2643_SDE_CTRL, 0x04},
               {OV2643_YOFFSET, 0x24},
               {OV2643_YGAIN, 0x24},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       {
               {OV2643_SDE_CTRL, 0x04},
               {OV2643_YOFFSET, 0x28},
               {OV2643_YGAIN, 0x28},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       {
               {OV2643_SDE_CTRL, 0x04},
               {OV2643_YOFFSET, 0x2c},
               {OV2643_YGAIN, 0x2c},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
};

/* Color Settings - 3 colors */
const static struct ov2643_reg colors[3][5] = {
       {
               {OV2643_SDE_CTRL, 0x00},
               {OV2643_UREG, 0x80},
               {OV2643_VREG, 0x80},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       {
               {OV2643_SDE_CTRL, 0x18},
               {OV2643_UREG, 0x40},
               {OV2643_VREG, 0xa6},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       {
               {OV2643_SDE_CTRL, 0x18},
               {OV2643_UREG, 0x80},
               {OV2643_VREG, 0x80},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
};

/* Average Based Algorithm - Based on target Luminance */
const static struct ov2643_reg exposure_avg[11][5] = {
       /* -1.7EV */
       {
               {OV2643_HISTO7, 0x00},
               {OV2643_WPT_HISH, 0x10},
               {OV2643_BPT_HISL, 0x08},
               {OV2643_VPT, 0x21},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       /* -1.3EV */
       {
               {OV2643_HISTO7, 0x00},
               {OV2643_WPT_HISH, 0x18},
               {OV2643_BPT_HISL, 0x10},
               {OV2643_VPT, 0x31},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       /* -1.0EV */
       {
               {OV2643_HISTO7, 0x00},
               {OV2643_WPT_HISH, 0x20},
               {OV2643_BPT_HISL, 0x18},
               {OV2643_VPT, 0x41},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       /* -0.7EV */
       {
               {OV2643_HISTO7, 0x00},
               {OV2643_WPT_HISH, 0x28},
               {OV2643_BPT_HISL, 0x20},
               {OV2643_VPT, 0x51},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       /* -0.3EV */
       {
               {OV2643_HISTO7, 0x00},
               {OV2643_WPT_HISH, 0x30},
               {OV2643_BPT_HISL, 0x28},
               {OV2643_VPT, 0x61},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       /* default */
       {
               {OV2643_HISTO7, 0x00},
               {OV2643_WPT_HISH, 0x38},
               {OV2643_BPT_HISL, 0x30},
               {OV2643_VPT, 0x61},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       /* 0.3EV */
       {
               {OV2643_HISTO7, 0x00},
               {OV2643_WPT_HISH, 0x40},
               {OV2643_BPT_HISL, 0x38},
               {OV2643_VPT, 0x71},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       /* 0.7EV */
       {
               {OV2643_HISTO7, 0x00},
               {OV2643_WPT_HISH, 0x48},
               {OV2643_BPT_HISL, 0x40},
               {OV2643_VPT, 0x81},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       /* 1.0EV */
       {
               {OV2643_HISTO7, 0x00},
               {OV2643_WPT_HISH, 0x50},
               {OV2643_BPT_HISL, 0x48},
               {OV2643_VPT, 0x91},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       /* 1.3EV */
       {
               {OV2643_HISTO7, 0x00},
               {OV2643_WPT_HISH, 0x58},
               {OV2643_BPT_HISL, 0x50},
               {OV2643_VPT, 0x91},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       /* 1.7EV */
       {
               {OV2643_HISTO7, 0x00},
               {OV2643_WPT_HISH, 0x60},
               {OV2643_BPT_HISL, 0x58},
               {OV2643_VPT, 0xa1},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
};

/* Histogram Based Algorithm - Based on histogram and probability */
const static struct ov2643_reg exposure_hist[11][5] = {
       /* -1.7EV */
       {
               {OV2643_HISTO7, 0x80},
               {OV2643_WPT_HISH, 0x58},
               {OV2643_BPT_HISL, 0x38},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       /* -1.3EV */
       {
               {OV2643_HISTO7, 0x80},
               {OV2643_WPT_HISH, 0x60},
               {OV2643_BPT_HISL, 0x40},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       /* -1.0EV */
       {
               {OV2643_HISTO7, 0x80},
               {OV2643_WPT_HISH, 0x68},
               {OV2643_BPT_HISL, 0x48},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       /* -0.7EV */
       {
               {OV2643_HISTO7, 0x80},
               {OV2643_WPT_HISH, 0x70},
               {OV2643_BPT_HISL, 0x50},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       /* -0.3EV */
       {
               {OV2643_HISTO7, 0x80},
               {OV2643_WPT_HISH, 0x78},
               {OV2643_BPT_HISL, 0x58},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       /* default */
       {
               {OV2643_HISTO7, 0x80},
               {OV2643_WPT_HISH, 0x80},
               {OV2643_BPT_HISL, 0x60},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       /* 0.3EV */
       {
               {OV2643_HISTO7, 0x80},
               {OV2643_WPT_HISH, 0x88},
               {OV2643_BPT_HISL, 0x68},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       /* 0.7EV */
       {
               {OV2643_HISTO7, 0x80},
               {OV2643_WPT_HISH, 0x90},
               {OV2643_BPT_HISL, 0x70},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       /* 1.0EV */
       {
               {OV2643_HISTO7, 0x80},
               {OV2643_WPT_HISH, 0x98},
               {OV2643_BPT_HISL, 0x78},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       /* 1.3EV */
       {
               {OV2643_HISTO7, 0x80},
               {OV2643_WPT_HISH, 0xa0},
               {OV2643_BPT_HISL, 0x80},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
       /* 1.7EV */
       {
               {OV2643_HISTO7, 0x80},
               {OV2643_WPT_HISH, 0xa8},
               {OV2643_BPT_HISL, 0x88},
               {OV2643_REG_TERM, OV2643_VAL_TERM}
       },
};


/*
 * struct vcontrol - Video controls
 * @v4l2_queryctrl: V4L2 VIDIOC_QUERYCTRL ioctl structure
 * @current_value: current value of this control
 */
static struct vcontrol {
       struct v4l2_queryctrl qc;
       int current_value;
} video_control[] = {
       {
               {
               .id = V4L2_CID_BRIGHTNESS,
               .type = V4L2_CTRL_TYPE_INTEGER,
               .name = "Brightness",
               .minimum = OV2643_MIN_BRIGHT,
               .maximum = OV2643_MAX_BRIGHT,
               .step = OV2643_BRIGHT_STEP,
               .default_value = OV2643_DEF_BRIGHT,
               },
       .current_value = OV2643_DEF_BRIGHT,
       },
       {
               {
               .id = V4L2_CID_CONTRAST,
               .type = V4L2_CTRL_TYPE_INTEGER,
               .name = "Contrast",
               .minimum = OV2643_MIN_CONTRAST,
               .maximum = OV2643_MAX_CONTRAST,
               .step = OV2643_CONTRAST_STEP,
               .default_value = OV2643_DEF_CONTRAST,
               },
       .current_value = OV2643_DEF_CONTRAST,
       },
       {
               {
               .id = V4L2_CID_PRIVATE_BASE,
               .type = V4L2_CTRL_TYPE_INTEGER,
               .name = "Color Effects",
               .minimum = OV2643_MIN_COLOR,
               .maximum = OV2643_MAX_COLOR,
               .step = OV2643_COLOR_STEP,
               .default_value = OV2643_DEF_COLOR,
               },
       .current_value = OV2643_DEF_COLOR,
       },
	   {
               {
               .id = V4L2_CID_VFLIP,
               .type = V4L2_CTRL_TYPE_BOOLEAN,
               .name = "Vertical Flip",
               .default_value = OV2643_DEF_FLIP,
               },
       .current_value = OV2643_DEF_FLIP,
       },
	   {
               {
               .id = V4L2_CID_HFLIP,
               .type = V4L2_CTRL_TYPE_BOOLEAN,
               .name = "Horizontal Flip (Mirror)",
               .default_value = OV2643_DEF_FLIP,
               },
       .current_value = OV2643_DEF_FLIP,
       }
};

/*
 * find_vctrl - Finds the requested ID in the video control structure array
 * @id: ID of control to search the video control array.
 *
 * Returns the index of the requested ID from the control structure array
 */
static int find_vctrl(int id)
{
       int i = 0;

       if (id < V4L2_CID_BASE)
               return -EDOM;

       for (i = (ARRAY_SIZE(video_control) - 1); i >= 0; i--)
               if (video_control[i].qc.id == id)
                       break;
       if (i < 0)
               i = -EINVAL;
       return i;
}

/*
 * Read a value from a register in ov2643 sensor device.
 * The value is returned in 'val'.
 * Returns zero if successful, or non-zero otherwise.
 */
static int ov2643_read_reg(struct i2c_client *client, u16 data_length, u16 reg,
                                                               u32 *val)
{
       int err = 0;
       struct i2c_msg msg[2];
       unsigned char data[4];


	  if (!client || !client->adapter) {
		    PDEBUG("Invalid input\n");
		   return -ENODEV;
	   }
      
       msg[0].addr = client->addr >> 1;
       msg[0].flags = I2C_M_WR;
	   msg[0].len = 1;
       msg[0].buf = data;

       msg[1].addr = client->addr >> 1;
       msg[1].flags = I2C_M_RD;
       msg[1].len = data_length;
       msg[1].buf = data;

	   data[0] = (u8) (reg & 0xff);
	   
       err = i2c_transfer(client->adapter, msg, 2);
       if (err >= 0) {
               *val = 0;
               /* High byte comes first */
               if (data_length == 1)
                       *val = data[0];
               else if (data_length == 2)
                       *val = data[1] + (data[0] << 8);
               else
                       *val = data[3] + (data[2] << 8) +
                               (data[1] << 16) + (data[0] << 24);
               return 0;
       }
       dev_err(&client->dev, "read from offset 0x%x error %d\n", reg, err);
       return err;
}

/* Write a value to a register in ov2643 sensor device.
 * @client: i2c driver client structure.
 * @reg: Address of the register to read value from.
 * @val: Value to be written to a specific register.
 * Returns zero if successful, or non-zero otherwise.
 */
static int ov2643_write_reg(struct i2c_client *client, u16 reg, u8 val)
{
       int err = 0;
       struct i2c_msg msg[1];
       unsigned char data[3];
       int retries = 0;

       if (!client || !client->adapter) {
		    PDEBUG("Invalid input\n");
		   return -ENODEV;
	   }
       PDEBUG("__78 %x %x\n",reg,val);
retry:
       msg->addr = client->addr >> 1;
       msg->flags = I2C_M_WR;
	   msg->len = 2;
       msg->buf = data;

	   data[0] = (u8) (reg & 0xff);
       data[1] = val;

       err = i2c_transfer(client->adapter, msg, 1);
       udelay(50);

       if (err >= 0)
               return 0;

       if (retries <= 5) {
               dev_dbg(&client->dev, "Retrying I2C... %d", retries);
               retries++;
               set_current_state(TASK_UNINTERRUPTIBLE);
               schedule_timeout(msecs_to_jiffies(20));
               goto retry;
       }

       return err;
}

/*
 * Initialize a list of ov2643 registers.
 * The list of registers is terminated by the pair of values
 * {OV2643_REG_TERM, OV2643_VAL_TERM}.
 * @client: i2c driver client structure.
 * @reglist[]: List of address of the registers to write data.
 * Returns zero if successful, or non-zero otherwise.
 */
static int ov2643_write_regs(struct i2c_client *client,
                                       const struct ov2643_reg reglist[])
{
       int err = 0;
       const struct ov2643_reg *next;

	    if (!client || !reglist) {
		   PDEBUG("Invalid input\n");
		   return -1;
	   }

		next = reglist;

       while (next && !((next->reg == OV2643_REG_TERM)
               && (next->val == OV2643_VAL_TERM))) {
               err = ov2643_write_reg(client, next->reg, next->val);
               udelay(100);
               if (err)
                       return err;
               next++;
       }
       return 0;
}

static int ov2643_init_mode(struct i2c_client *c, enum ov2643_frameintervals_type frame_rate,
                      enum image_size_ov size)

{
      struct reg_value *pModeSetting = NULL;
      s32 i = 0;
      s32 iModeSettingArySize = 0;
      register u32 Delay_ms = 0;
      register u16 RegAddr = 0;
      register u8 Mask = 0;
      register u8 Val = 0;
      u32 RegVal = 0;
      int retval = 0;

      
      if (size > OV2643_MODE_MAX || size < OV2643_MODE_MIN || c == NULL) {
            pr_err("Wrong ov2643 size detected! or invalid client\n");
            return -1;
      }

	  PDEBUG("frame_rate=%s size=%s\n",frame_rate_tostring(frame_rate), mode_tostring(size));
	  pModeSetting = ov2643_mode_info_data[frame_rate][size].init_data_ptr;
      iModeSettingArySize = ov2643_mode_info_data[frame_rate][size].init_data_size;

      for (i = 0; i < iModeSettingArySize; ++i, ++pModeSetting) {
            Delay_ms = pModeSetting->u32Delay_ms;
            RegAddr = pModeSetting->u16RegAddr;
            Val = pModeSetting->u8Val;
            Mask = pModeSetting->u8Mask;

            if (Mask) {
                 
                  retval =  ov2643_read_reg(c, 1, RegAddr, &RegVal);
                  if (retval < 0)
                        goto ovinit_err;

                  RegVal &= ~(u8)Mask;
                  Val &= Mask;
                  Val |= RegVal;
            }

            retval = ov2643_write_reg(c, RegAddr, Val);
            if (retval < 0)
                  goto ovinit_err;

            if (Delay_ms)
                  msleep(Delay_ms);
      }
   
ovinit_err:
      return retval;
}

static void ov2643_set_fmt(struct i2c_client *client, unsigned int pixelformat)
{
	u32 regval = 0;
	if (!client)
		return;
	ov2643_read_reg(client,1, 0x0d, &regval);
	/*Bits 5-6 configure the YCbCr composition*/
	regval &= ~(0x3 << 5);

	switch (pixelformat) {
		case V4L2_PIX_FMT_NV12:
		case V4L2_PIX_FMT_NV16:	
			regval |= (0x1<<5);
			break;
		case V4L2_PIX_FMT_NV21:
		case V4L2_PIX_FMT_NV61:
			regval |= (0x3<<5); 
	}
	
	ov2643_write_reg(client, 0x0d, regval);
}

static void ov2643_set_flip(struct i2c_client *client)
{
	u32 regval = 0;
	PDEBUG("Hflip is %s  Vflip is %s\n", ((ov2643.hflip == 1) ? "On" : "Off") , ((ov2643.vflip == 1) ? "On" : "Off") );
    PDEBUG("%s\n",__func__);

	ov2643_read_reg(client,1, 0x12, &regval);

	if (ov2643.hflip)
		ov2643_write_reg(client, 0x12, regval | (1<<5));
	else
		ov2643_write_reg(client, 0x12, regval & ~(1<<5));

	if (ov2643.vflip)
		ov2643_write_reg(client, 0x12, regval | (1<<4));
	else
		ov2643_write_reg(client, 0x12, regval & ~(1<<4));

}



static int ov2643_set_v_flip(struct i2c_client *client , unsigned int val)
{
	
	if (!client) {
		   PDEBUG("Invalid input\n");
		   return -1;
	}
    PDEBUG("%s\n",__func__);
	ov2643.vflip = ((val > 0) ? 1 : 0);

	ov2643_set_flip(client);

    return 0;
}

static int ov2643_set_h_flip(struct i2c_client *client , unsigned int val)
{
	if (!client) {
		   PDEBUG("Invalid input\n");
		   return -1;
	}
    PDEBUG("%s\n",__func__);
	ov2643.hflip = ((val > 0) ? 1 : 0);

	ov2643_set_flip(client);

    return 0;
}

/* Find the best match for a requested image capture size.  The best match
 * is chosen as the nearest match that has the same number or fewer pixels
 * as the requested size, or the smallest image size if the requested size
 * has fewer pixels than the smallest image.
 */
static int
ov2643_find_size(unsigned int width, unsigned int height , enum image_size_ov* size)
{
       if ((width == ov2643_sizes[QCIF].width) && (height == ov2643_sizes[QCIF].height))
		   *size = QCIF;
       else if ((width == ov2643_sizes[CIF].width) && (height == ov2643_sizes[CIF].height))
		   *size = CIF;
	   else if ((width == ov2643_sizes[SVGA].width) && (height == ov2643_sizes[SVGA].height))
		   *size = SVGA;
	   else if ((width == ov2643_sizes[HD720P].width) && (height == ov2643_sizes[HD720P].height))
		   *size = HD720P;
	   else if ((width == ov2643_sizes[UXGA].width) && (height == ov2643_sizes[UXGA].height))
		   *size = UXGA;
       else
           return -1;

       return 0;
}

static int switch_fps(struct v4l2_fract *a , struct i2c_client *c , enum ov2643_frameintervals_type* fps)
{
    unsigned int i = 0;
    if(!c || !a) {
        PDEBUG("Invalid input\n");
        return -1;
    }

	PDEBUG("Try to set FPS to denominator=%d numerator=%d \n",a->denominator ,a->numerator );

    for (i = 0 ; i < NUM_TIMEFRAME_VALS ; i++) {

		PDEBUG("Check FPS to denominator=%d numerator=%d \n", ov2643_frameintervals[i].denominator , ov2643_frameintervals[i].numerator);

        if ( (a->denominator == ov2643_frameintervals[i].denominator)  &&  (a->numerator == ov2643_frameintervals[i].numerator) ) {
			
			switch (i) {
				case OV26_15_FPS:
					*fps = OV26_15_FPS;
					PDEBUG("setting 15 FPS\n");
					break;
				case OV26_30_FPS:
					*fps = OV26_30_FPS;
					PDEBUG("setting 30 FPS\n");
					break;
				default:
					PDEBUG("Non existed set to 15 FPS as default\n");
					*fps = OV26_15_FPS;
					

			}

			break;
		}
	}

	if (i == NUM_TIMEFRAME_VALS) {
        PDEBUG("Invalid frame interval\n");
        return -1;
    }

	return 0;
}

static int ov2643_configure(struct v4l2_int_device *s)
{
       struct ov2643_sensor *sensor = NULL;
       struct v4l2_pix_format *pix = NULL;
       struct i2c_client *client = NULL;
       enum image_size_ov isize = QCIF;
	   enum ov2643_frameintervals_type frame_rate = OV26_15_FPS;
	   int err = 0;

	   if (s && s->priv) {
			sensor = s->priv;
			if (sensor->i2c_client){
				pix = &sensor->pix;
				client = sensor->i2c_client;
			}
	   }
	   if (!sensor || !pix || !client) {
		   PDEBUG("Invalid input\n");
		   return -1;
	   }
       
	   /* Set receivers virtual channel before sensor setup starts.
        * Only set the sensors virtual channel after all other setup
        * for the sensor is complete.
        */
	   err = ov2643_find_size(pix->width, pix->height, &isize);
       if (err)
           return err;

	   switch_fps(&sensor->timeperframe , client , &frame_rate);

	   ov2643_init_mode(client , frame_rate , isize);

		ov2643_set_fmt(client, sensor->pix.pixelformat);

	   //ov2643_set_flip(client);

       sensor->isize = isize;

       /* Store image size */
	   sensor->isize = isize;
       sensor->width = pix->width;
       sensor->height = pix->height;

       sensor->crop_rect.left = 0;
       sensor->crop_rect.width = pix->width;
       sensor->crop_rect.top = 0;
       sensor->crop_rect.height = pix->height;

       return err;
}

/* Detect if an ov2643 is present, returns a negative error number if no
 * device is detected, or pidl as version number if a device is detected.
 */
static int ov2643_detect(struct i2c_client *client)
{
       u32 pidh = 0, pidl = 0;
       PDEBUG("%s\n",__func__);
       if (!client) {
		   PDEBUG("Invalid input\n");
		   return -ENODEV;
	   }
       if (ov2643_read_reg(client, 1, OV2643_PIDH, &pidh))
               return -ENODEV;

       if (ov2643_read_reg(client, 1, OV2643_PIDL, &pidl))
               return -ENODEV;

       if ((pidh == OV2643_PIDH_MAGIC) &&  (pidl == OV2643_PIDL_MAGIC1) ) {
               dev_dbg(&client->dev, "Detect success (%02X,%02X)\n", pidh,
                                                                       pidl);
               return pidl;
       }

       return -ENODEV;
}

/* To get the cropping capabilities of ov2643 sensor
 * Returns zero if successful, or non-zero otherwise.
 */
static int ioctl_cropcap(struct v4l2_int_device *s, struct v4l2_cropcap *cropcap)
{
       struct ov2643_sensor *sensor;

	   if(!s || !cropcap) {
			  PDEBUG("Invalid input\n");
			   return -1;
	   }

	   sensor = s->priv;

	/*****************remove when sensor crop is set*************************/
	return -EINVAL;
	/*****************remove when sensor crop is set*************************/

       cropcap->bounds.top = 0;
       cropcap->bounds.left = 0;
       cropcap->bounds.width = sensor->width;
       cropcap->bounds.height = sensor->height;
       cropcap->defrect = cropcap->bounds;
       cropcap->pixelaspect.numerator = 1;
       cropcap->pixelaspect.denominator = 1;

	   PDEBUG("cropcap->bounds.top = %d\n cropcap->bounds.left = %d\n cropcap->bounds.width = %d\n cropcap->bounds.height = %d\n"
			"cropcap->defrect.top = %d\n cropcap->defrect.left = %d\n cropcap->defrect.width = %d\n cropcap->defrect.height = %d \n "
			"cropcap->pixelaspect.numerator = %d\n cropcap->pixelaspect.denominator = %d\n",
			cropcap->bounds.top, cropcap->bounds.left, cropcap->bounds.width, cropcap->bounds.height, 
			cropcap->defrect.top, cropcap->defrect.left, cropcap->defrect.width, cropcap->defrect.height, 
			cropcap->pixelaspect.numerator,cropcap->pixelaspect.denominator);

       return 0;
}

/* To get the current crop window for of ov2643 sensor
 * Returns zero if successful, or non-zero otherwise.
 */
static int ioctl_g_crop(struct v4l2_int_device *s, struct  v4l2_crop *crop)
{
       struct ov2643_sensor *sensor;

	   if(!s || !crop) {
			  PDEBUG("Invalid input\n");
			   return -1;
	   }

	   sensor = s->priv;

       crop->c = sensor->crop_rect;
       return 0;
}

/* To set the crop window for of ov2643 sensor
 * Returns zero if successful, or non-zero otherwise.
 */
static int ioctl_s_crop(struct v4l2_int_device *s, struct  v4l2_crop *crop)
{
		struct ov2643_sensor *sensor;
       /* FIXME: Temporary workaround for avoiding Zoom setting */
       /* struct i2c_client *client = sensor->i2c_client; */
       struct v4l2_rect *cur_rect;
       unsigned long *cur_width, *cur_height;
       int hstart, vstart, hsize, vsize, hsize_l, vsize_l, hsize_h, vsize_h;
       int hratio, vratio, zoomfactor, err = 0;
       PDEBUG("%s\n",__func__);
	   if(!s || !crop) {
			  PDEBUG("Invalid input\n");
			   return -1;
	   }

	   sensor = s->priv;
	/*****************remove when sensor crop is set*************************/
	return -EINVAL;
	/*****************remove when sensor crop is set*************************/

       cur_rect = &sensor->crop_rect;
       cur_width = &sensor->width;
       cur_height = &sensor->height;

       if ((crop->c.left == cur_rect->left) &&
           (crop->c.width == cur_rect->width) &&
           (crop->c.top == cur_rect->top) &&
           (crop->c.height == cur_rect->height))
               return 0;

       /* out of range? then return the current crop rectangle */
       if ((crop->c.left + crop->c.width) > sensor->width ||
           (crop->c.top + crop->c.height) > sensor->height) {
               crop->c = *cur_rect;
               return 0;
       }

       if (sensor->isize == UXGA)
               zoomfactor = 1;
       else
               zoomfactor = 2;

       hratio = (sensor->hsize * 1000) / sensor->width;
       vratio = (sensor->vsize * 1000) / sensor->height;
       hstart = (((crop->c.left * hratio + 500) / 1000) * zoomfactor) + 0x11d;
       vstart = (((crop->c.top * vratio + 500) / 1000) + 0x0a);
       hsize  = (crop->c.width * hratio + 500) / 1000;
       vsize  = (crop->c.height * vratio + 500) / 1000;

       if (vsize & 1)
               vsize--;
       if (hsize & 1)
               hsize--;

       /* Adjusting numbers to set register correctly */
       hsize_l = (0xFF & hsize);
       hsize_h = (0xF00 & hsize) >> 8;
       vsize_l = (0xFF & vsize);
       vsize_h = (0x700 & vsize) >> 8;

       if ((sensor->height > vsize) || (sensor->width > hsize))
               return -EINVAL;

       hsize = hsize * zoomfactor;
/*
       err = ov2643_write_reg(client, OV2643_DSP_CTRL_2, 0xEF);
       err = ov2643_write_reg(client, OV2643_SIZE_IN_MISC, (vsize_h |
                                                               hsize_h));
       err = ov2643_write_reg(client, OV2643_HSIZE_IN_L, hsize_l);
       err = ov2643_write_reg(client, OV2643_VSIZE_IN_L, vsize_l);
       err = ov2643_write_reg(client, OV2643_HS_H, (hstart >> 8) & 0xFF);
       err = ov2643_write_reg(client, OV2643_HS_L, hstart & 0xFF);
       err = ov2643_write_reg(client, OV2643_VS_H, (vstart >> 8) & 0xFF);
       err = ov2643_write_reg(client, OV2643_VS_L, vstart & 0xFF);
       err = ov2643_write_reg(client, OV2643_HW_H, ((hsize) >> 8) & 0xFF);
       err = ov2643_write_reg(client, OV2643_HW_L, hsize & 0xFF);
       err = ov2643_write_reg(client, OV2643_VH_H, ((vsize) >> 8) & 0xFF);
       err = ov2643_write_reg(client, OV2643_VH_L, vsize & 0xFF);
*/
       if (err)
               return err;

       /* save back */
       *cur_rect = crop->c;

       /* Setting crop too fast can cause frame out-of-sync. */

       set_current_state(TASK_UNINTERRUPTIBLE);
       schedule_timeout(msecs_to_jiffies(20));
       return 0;
}


/*
 * ioctl_queryctrl - V4L2 sensor interface handler for VIDIOC_QUERYCTRL ioctl
 * @s: pointer to standard V4L2 device structure
 * @qc: standard V4L2 VIDIOC_QUERYCTRL ioctl structure
 *
 * If the requested control is supported, returns the control information
 * from the video_control[] array.  Otherwise, returns -EINVAL if the
 * control is not supported.
 */
static int ioctl_queryctrl(struct v4l2_int_device *s,
                                               struct v4l2_queryctrl *qc)
{
       int i;

	   if(!s || !qc) {
			  PDEBUG("Invalid input\n");
			   return -1;
	   }

       i = find_vctrl(qc->id);
       if (i == -EINVAL)
               qc->flags = V4L2_CTRL_FLAG_DISABLED;

       if (i < 0)
               return -EINVAL;

       *qc = video_control[i].qc;
       return 0;
}

/*
 * ioctl_g_ctrl - V4L2 sensor interface handler for VIDIOC_G_CTRL ioctl
 * @s: pointer to standard V4L2 device structure
 * @vc: standard V4L2 VIDIOC_G_CTRL ioctl structure
 *
 * If the requested control is supported, returns the control's current
 * value from the video_control[] array.  Otherwise, returns -EINVAL
 * if the control is not supported.
 */

static int ioctl_g_ctrl(struct v4l2_int_device *s,
                            struct v4l2_control *vc)
{
       struct vcontrol *lvc;
       int i = 0;

	   if(!s || !vc) {
			  PDEBUG("Invalid input\n");
			   return -1;
	   }

       i = find_vctrl(vc->id);
       if (i < 0)
               return -EINVAL;
       lvc = &video_control[i];

       switch (vc->id) {
       case V4L2_CID_BRIGHTNESS:
               vc->value = lvc->current_value;
               break;
       case V4L2_CID_CONTRAST:
               vc->value = lvc->current_value;
               break;
       case V4L2_CID_PRIVATE_BASE:
               vc->value = lvc->current_value;
               break;
	   case V4L2_CID_VFLIP:
			   vc->value = lvc->current_value;
			   break;
	   case V4L2_CID_HFLIP:
			   vc->value = lvc->current_value;
			   break;
       }
       return 0;
}

/*
 * ioctl_s_ctrl - V4L2 sensor interface handler for VIDIOC_S_CTRL ioctl
 * @s: pointer to standard V4L2 device structure
 * @vc: standard V4L2 VIDIOC_S_CTRL ioctl structure
 *
 * If the requested control is supported, sets the control's current
 * value in HW (and updates the video_control[] array).  Otherwise,
 * returns -EINVAL if the control is not supported.
 */
static int ioctl_s_ctrl(struct v4l2_int_device *s,
                            struct v4l2_control *vc)
{
    
       int retval = -EINVAL;
       int i = 0;
       struct ov2643_sensor *sensor;
       struct i2c_client *client;
       struct vcontrol *lvc;
       PDEBUG("%s\n",__func__);
	   if(!s || !vc) {
			  PDEBUG("Invalid input\n");
			   return -1;
	   }

       sensor = s->priv;
       client = sensor->i2c_client;
       

       i = find_vctrl(vc->id);
       if (i < 0)
               return -EINVAL;

       lvc = &video_control[i];

       switch (vc->id) {
       case V4L2_CID_BRIGHTNESS:
               if (vc->value >= 0 && vc->value <= 6) {
                       retval = ov2643_write_regs(client,
                                                       brightness[vc->value]);
               } else {
                       dev_err(&client->dev, "BRIGHTNESS LEVEL NOT SUPPORTED\n");
                       return -EINVAL;
               }
               break;
       case V4L2_CID_CONTRAST:
               if (vc->value >= 0 && vc->value <= 6)
                       retval = ov2643_write_regs(client, contrast[vc->value]);
               else {
                       dev_err(&client->dev, "CONTRAST LEVEL NOT SUPPORTED\n");
                       return -EINVAL;
               }
               break;
       case V4L2_CID_PRIVATE_BASE:
               if (vc->value >= 0 && vc->value <= 2)
                       retval = ov2643_write_regs(client, colors[vc->value]);
               else {
                       dev_err(&client->dev, "COLOR LEVEL NOT SUPPORTED\n");
                       return -EINVAL;
               }
               break;
       case V4L2_CID_VFLIP:
               if (vc->value >= 0){
				   
 					unsigned int val = 0;
						val = (vc->value > 0) ? 1 : 0;
                        retval = ov2643_set_v_flip(client , val);
			   }
               else {
                       dev_err(&client->dev, "VERTIVCAL FLIP INVALID VALUE\n");
                       return -EINVAL;
               }
               break;
       case V4L2_CID_HFLIP:
               if (vc->value >= 0){
 					unsigned int val = 0;
						val = (vc->value > 0) ? 1 : 0;
                        retval = ov2643_set_h_flip(client ,val);
			   }
               else {
                       dev_err(&client->dev, "HORIZONTAL FLIP INVALID VALUE\n");
                       return -EINVAL;
               }
               break;
       }
       if (!retval)
               lvc->current_value = vc->value;
       return retval;
}

/*
 * ioctl_enum_fmt_cap - Implement the CAPTURE buffer VIDIOC_ENUM_FMT ioctl
 * @s: pointer to standard V4L2 device structure
 * @fmt: standard V4L2 VIDIOC_ENUM_FMT ioctl structure
 *
 * Implement the VIDIOC_ENUM_FMT ioctl for the CAPTURE buffer type.
 */
 static int ioctl_enum_fmt_cap(struct v4l2_int_device *s,
                                  struct v4l2_fmtdesc *fmt)
{
       int index = 0;
       enum v4l2_buf_type type;

	   if(!s || !fmt) {
			  PDEBUG("Invalid input\n");
			   return -1;
	   }

	   index = fmt->index;
       type = fmt->type;

       memset(fmt, 0, sizeof(*fmt));
       fmt->index = index;
       fmt->type = type;

       switch (fmt->type) {
       case V4L2_BUF_TYPE_VIDEO_CAPTURE:
               if (index >= NUM_CAPTURE_FORMATS)
                       return -EINVAL;
       break;
       default:
               return -EINVAL;
       }

       fmt->flags = ov2643_formats[index].flags;
       strlcpy(fmt->description, ov2643_formats[index].description,
                                       sizeof(fmt->description));
       fmt->pixelformat = ov2643_formats[index].pixelformat;

       return 0;
}


/*
 * ioctl_try_fmt_cap - Implement the CAPTURE buffer VIDIOC_TRY_FMT ioctl
 * @s: pointer to standard V4L2 device structure
 * @f: pointer to standard V4L2 VIDIOC_TRY_FMT ioctl structure
 *
 * Implement the VIDIOC_TRY_FMT ioctl for the CAPTURE buffer type.  This
 * ioctl is used to negotiate the image capture size and pixel format
 * without actually making it take effect.
 */

static int ioctl_try_fmt_cap(struct v4l2_int_device *s,
                            struct v4l2_format *f)
{
       int ifmt = 0;
       struct v4l2_pix_format *pix;

	   if(!s || !f) {
			  PDEBUG("Invalid input\n");
			   return -1;
	   }

	   pix = &f->fmt.pix;

       if (pix->width > ov2643_sizes[UXGA].width)
               pix->width = ov2643_sizes[UXGA].width;
       if (pix->height > ov2643_sizes[UXGA].height)
               pix->height = ov2643_sizes[UXGA].height;

       
       for (ifmt = 0; ifmt < NUM_CAPTURE_FORMATS; ifmt++) {
               if (pix->pixelformat == ov2643_formats[ifmt].pixelformat)
                       break;
       }
       if (ifmt == NUM_CAPTURE_FORMATS)
               ifmt = 0;
       pix->pixelformat = ov2643_formats[ifmt].pixelformat;
       pix->field = V4L2_FIELD_NONE;
       pix->bytesperline = bytes_per_line(pix->width, pix->pixelformat ); 
       pix->sizeimage = image_size(pix->width, pix->height, pix->pixelformat);
       pix->priv = 0;
       switch (pix->pixelformat) {
	   case V4L2_PIX_FMT_NV16:
	   case V4L2_PIX_FMT_NV61:
	   case V4L2_PIX_FMT_NV12:
	   case V4L2_PIX_FMT_NV21:
       case V4L2_PIX_FMT_YUYV:
       case V4L2_PIX_FMT_UYVY:
       default:
               pix->colorspace = V4L2_COLORSPACE_JPEG;
               break;
       case V4L2_PIX_FMT_SGRBG10:
       case V4L2_PIX_FMT_RGB565:
       case V4L2_PIX_FMT_RGB565X:
       case V4L2_PIX_FMT_RGB555:
       case V4L2_PIX_FMT_RGB555X:
               pix->colorspace = V4L2_COLORSPACE_SRGB;
               break;
       }
       return 0;
}


/*
 * ioctl_s_fmt_cap - V4L2 sensor interface handler for VIDIOC_S_FMT ioctl
 * @s: pointer to standard V4L2 device structure
 * @f: pointer to standard V4L2 VIDIOC_S_FMT ioctl structure
 *
 * If the requested format is supported, configures the HW to use that
 * format, returns error code if format not supported or HW can't be
 * correctly configured.
 */
 static int ioctl_s_fmt_cap(struct v4l2_int_device *s,
                               struct v4l2_format *f)
{
       struct ov2643_sensor *sensor;
       struct v4l2_pix_format *pix;
       int rval = 0;

	   if(!s || !f) {
			  PDEBUG("Invalid input\n");
			   return -1;
	   }

	   sensor = s->priv;
       pix = &f->fmt.pix;

       rval = ioctl_try_fmt_cap(s, f);
       if (rval)
               return rval;

       sensor->pix = *pix;

       return 0;
}

/*
 * ioctl_g_fmt_cap - V4L2 sensor interface handler for ioctl_g_fmt_cap
 * @s: pointer to standard V4L2 device structure
 * @f: pointer to standard V4L2 v4l2_format structure
 *
 * Returns the sensor's current pixel format in the v4l2_format
 * parameter.
 */
static int ioctl_g_fmt_cap(struct v4l2_int_device *s,
                               struct v4l2_format *f)
{
	struct ov2643_sensor *sensor;
    
	 if(!s || !f) {
			  PDEBUG("Invalid input\n");
			   return -1;
	 }

       sensor = s->priv;
       f->fmt.pix = sensor->pix;

       return 0;
}

/*
 * ioctl_g_parm - V4L2 sensor interface handler for VIDIOC_G_PARM ioctl
 * @s: pointer to standard V4L2 device structure
 * @a: pointer to standard V4L2 VIDIOC_G_PARM ioctl structure
 *
 * Returns the sensor's video CAPTURE parameters.
 */
static int ioctl_g_parm(struct v4l2_int_device *s,
                            struct v4l2_streamparm *a)
{
       struct ov2643_sensor *sensor;
       struct v4l2_captureparm *cparm;

	   if(!s || !a) {
			  PDEBUG("Invalid input\n");
			   return -1;
	   }

	   sensor = s->priv;
       cparm = &a->parm.capture;

       if (a->type != V4L2_BUF_TYPE_VIDEO_CAPTURE)
               return -EINVAL;

       memset(a, 0, sizeof(*a));
       a->type = V4L2_BUF_TYPE_VIDEO_CAPTURE;

       cparm->capability = V4L2_CAP_TIMEPERFRAME;
       cparm->timeperframe = sensor->timeperframe;

       return 0;
}

/*
 * ioctl_s_parm - V4L2 sensor interface handler for VIDIOC_S_PARM ioctl
 * @s: pointer to standard V4L2 device structure
 * @a: pointer to standard V4L2 VIDIOC_S_PARM ioctl structure
 *
 * Configures the sensor to use the input parameters, if possible.  If
 * not possible, reverts to the old parameters and returns the
 * appropriate error code.
 */
static int ioctl_s_parm(struct v4l2_int_device *s,
                            struct v4l2_streamparm *a)
{
       int rval = 0;
       struct ov2643_sensor *sensor;
       struct v4l2_fract *timeperframe;
       struct v4l2_fract timeperframe_old;
       int desired_fps = 0;
       struct i2c_client *c = NULL;
	   enum ov2643_frameintervals_type fps= 0;
       PDEBUG("%s\n",__func__);
       if(!s || !a) {
          PDEBUG("Invalid input\n");
          return -1;
	   }

       sensor = s->priv;
       c = sensor->i2c_client;
       if(!c) {
          PDEBUG("Invalid client\n");
          return -1;
	   }
       
       timeperframe = &a->parm.capture.timeperframe;
       timeperframe_old = sensor->timeperframe;
       sensor->timeperframe = *timeperframe;

       desired_fps = timeperframe->denominator / timeperframe->numerator;
	   PDEBUG("desired_fps = %d denominator=%d numerator=%d \n",desired_fps , timeperframe->denominator , timeperframe->numerator );
       if ((desired_fps < OV2643_DEF_FPS) || (desired_fps > OV2643_MAX_FPS))     
           return (-EINVAL);

       rval = switch_fps(timeperframe ,c, &fps);

       if (rval)
               sensor->timeperframe = timeperframe_old;
       else
               *timeperframe = sensor->timeperframe;

       return rval;
}

/*
 * ioctl_g_priv - V4L2 sensor interface handler for vidioc_int_g_priv_num
 * @s: pointer to standard V4L2 device structure
 * @p: void pointer to hold sensor's private data address
 *
 * Returns device's (sensor's) private data area address in p parameter
 * We need the configure since we close the clk when going to Standby mode.
 */
static int ioctl_g_priv(struct v4l2_int_device *s, void *p)
{
	struct ov2643_sensor *sensor;
	if(!s || !p) {
		PDEBUG("Invalid input\n");
		return -1;
	}
	sensor = s->priv;
	return sensor->pdata->priv_data_set(p);
}

/*
 * reset_ov2643_to_default  
 * @s: pointer to standard V4L2 device structure
 * Sets reset the device to default
 */
static void reset_ov2643_to_default(struct v4l2_int_device *s)
{
	struct ov2643_sensor *sensor = s->priv;
	struct i2c_client *c = sensor->i2c_client;

	 	/* Reset */
	ov2643_write_reg(c, OV2643_SYS, 0x80);
	mdelay(10);
	//ov2643_write_regs (c, ov2643_common[0]);
	PDEBUG("SENSOR RESET\n");
}


/*
 * ioctl_s_power - V4L2 sensor interface handler for vidioc_int_s_power_num
 * @s: pointer to standard V4L2 device structure
 * @on: power state to which device is to be set
 *
 * Sets devices power state to requrested state, if possible.
 */
 static int ioctl_s_power(struct v4l2_int_device *s, enum v4l2_power on)
{
       struct ov2643_sensor *sensor = s->priv;
       struct i2c_client *c = sensor->i2c_client;
	   static enum v4l2_power prev_on = V4L2_POWER_OFF;
   
       int rval = 0;

	   if (on == V4L2_POWER_ON)
		   PDEBUG("ov2643 = ON prev = %s \n", (prev_on == 0) ? "OFF" : (prev_on == 1) ? "ON" : "STANDBY" );
	   else if (on == V4L2_POWER_OFF)
		   PDEBUG("ov2643 = OFF prev = %s \n", (prev_on == 0) ? "OFF" : (prev_on == 1) ? "ON" : "STANDBY" );
	   else if (on == V4L2_POWER_STANDBY)
		   PDEBUG("ov2643 = STANDBY prev = %s \n", (prev_on == 0) ? "OFF" : (prev_on == 1) ? "ON" : "STANDBY" );

	   if ((prev_on == V4L2_POWER_ON) && (on == V4L2_POWER_STANDBY)){
		   ov2643_write_regs(c,ov2643_standby_on);
	   }
	   
       rval = sensor->pdata->power_set(on);

	   if ((prev_on == V4L2_POWER_STANDBY) && (on == V4L2_POWER_ON)){
		   ov2643_write_regs(c,ov2643_standby_off);
	   }
	   
       if (rval < 0) {
               dev_err(&c->dev, "Unable to set the power state: "
                       OV2643_DRIVER_NAME " sensor\n");

			   sensor->pdata->set_xclk(0);
               return rval;
       }
	   
	   if (on == V4L2_POWER_ON)
               sensor->pdata->set_xclk(xclk_current);
       else
               sensor->pdata->set_xclk(0);
	   
	   if ( (prev_on == V4L2_POWER_OFF && on == V4L2_POWER_ON)  && (sensor->state == SENSOR_DETECTED) )
			reset_ov2643_to_default(s);

       if ( (prev_on == V4L2_POWER_STANDBY || prev_on == V4L2_POWER_ON) && (on == V4L2_POWER_ON) && (sensor->state == SENSOR_DETECTED))
           ov2643_configure(s);
            
       if ((on == V4L2_POWER_ON) && (sensor->state == SENSOR_NOT_DETECTED)) {
		   rval = ov2643_detect(c);
		   if (rval < 0) {
				   dev_err(&c->dev, "Unable to detect "
								   OV2643_DRIVER_NAME " sensor\n");
				   sensor->state = SENSOR_NOT_DETECTED;
				   return rval;
		   }
		   sensor->state = SENSOR_DETECTED;
		   sensor->ver = rval;
		   pr_info(OV2643_DRIVER_NAME " Chip version 0x%02x detected\n",
														   sensor->ver);
		   reset_ov2643_to_default(s);
       }
	   prev_on = on;
       return 0;
}

/*
 * ioctl_init - V4L2 sensor interface handler for VIDIOC_INT_INIT
 * @s: pointer to standard V4L2 device structure
 *
 * Initialize the sensor device (call ov2643_configure())
 */
static int ioctl_init(struct v4l2_int_device *s)
{
       return 0;
}

/**
 * ioctl_dev_exit - V4L2 sensor interface handler for vidioc_int_dev_exit_num
 * @s: pointer to standard V4L2 device structure
 *
 * Delinitialise the dev. at slave detach.  The complement of ioctl_dev_init.
 */
static int ioctl_dev_exit(struct v4l2_int_device *s)
{
       return 0;
}

/**
 * ioctl_dev_init - V4L2 sensor interface handler for vidioc_int_dev_init_num
 * @s: pointer to standard V4L2 device structure
 *
 * Initialise the device when slave attaches to the master.  Returns 0 if
 * ov2643 device could be found, otherwise returns appropriate error.
 */
static int ioctl_dev_init(struct v4l2_int_device *s)
{
       return 0;
}

/**
 * ioctl_enum_framesizes - V4L2 sensor if handler for vidioc_int_enum_framesizes
 * @s: pointer to standard V4L2 device structure
 * @frms: pointer to standard V4L2 framesizes enumeration structure
 *
 * Returns possible framesizes depending on choosen pixel format
 **/
static int ioctl_enum_framesizes(struct v4l2_int_device *s,
                                       struct v4l2_frmsizeenum *frms)
{
       int ifmt = 0;
	   if(!s || !frms) {
			  PDEBUG("Invalid input\n");
			  return -1;
	   }

       for (ifmt = 0; ifmt < NUM_CAPTURE_FORMATS; ifmt++) {
               if (frms->pixel_format == ov2643_formats[ifmt].pixelformat)
                       break;
       }

	   /* Is requested pixelformat not found on sensor? */
       if (ifmt == NUM_CAPTURE_FORMATS){
		   PDEBUG("sensor is not supporting the format\n");
		   return -EINVAL;
	   }
	   
	   if (frms->index >= NUM_DISCRETE_SIZES){
		  PDEBUG("Invalid frame size or not supported by the sensor frms->index=0x%x NUM_DISCRETE_SIZES=%d\n",frms->index,NUM_DISCRETE_SIZES);
          return -EINVAL;
	   }

       frms->type = V4L2_FRMSIZE_TYPE_DISCRETE;
       frms->discrete.width = ov2643_sizes[frms->index].width;
       frms->discrete.height = ov2643_sizes[frms->index].height;

       return 0;
}

static int ioctl_enum_frameintervals(struct v4l2_int_device *s,
                                       struct v4l2_frmivalenum *frmi)
{
       int ifmt = 0;

	   if(!s || !frmi) {
			  PDEBUG("Invalid input\n");
			  return -1;
	   }

       for (ifmt = 0; ifmt < NUM_CAPTURE_FORMATS; ifmt++) {
               if (frmi->pixel_format == ov2643_formats[ifmt].pixelformat)
                       break;
       }
       /* Is requested pixelformat not found on sensor? */
       if (ifmt == NUM_CAPTURE_FORMATS)
               return -EINVAL;

	   if ( frmi->width > ov2643_sizes[UXGA].width || frmi->height > ov2643_sizes[UXGA].height)
		   return -EINVAL;

	   if (frmi->width > ov2643_sizes[HD720P].width || frmi->height > ov2643_sizes[HD720P].height) {
               if (frmi->index > OV26_15_FPS)
                       return -EINVAL;
       }

	   if (frmi->index >= NUM_TIMEFRAME_VALS || frmi->index > OV26_30_FPS)
           return -EINVAL;

	   frmi->type = V4L2_FRMIVAL_TYPE_DISCRETE;
       frmi->discrete.numerator = ov2643_frameintervals[frmi->index].numerator;
       frmi->discrete.denominator = ov2643_frameintervals[frmi->index].denominator;

       return 0;
}


static int ioctl_s_reg(struct v4l2_int_device *s, const struct v4l2_dbg_register* reg)
{
	struct ov2643_sensor *sensor;
    struct i2c_client *c;
	int ret;

    if(!s || !reg || !s->priv || ! (((struct ov2643_sensor *)s->priv)->i2c_client)) 
        return -1;

    sensor = s->priv;
    c = sensor->i2c_client;

	PDEBUG("reg=%llx val=%llx\n", reg->reg , reg->val);
    
    ret = ov2643_write_reg(c,reg->reg,reg->val);

	return ret;
}

static int ioctl_g_reg(struct v4l2_int_device *s, struct v4l2_dbg_register* reg)
{
	struct ov2643_sensor *sensor;
    struct i2c_client *c;
	int ret;

    if(!s || !reg || !s->priv || ! (((struct ov2643_sensor *)s->priv)->i2c_client)) 
        return -1;

    sensor = s->priv;
    c = sensor->i2c_client;

	PDEBUG("reg=%llx val=%llx\n", reg->reg , reg->val);
    
    ret = ov2643_read_reg(c,1,reg->reg,(unsigned int*) &(reg->val));

	return ret;
}

static struct v4l2_int_ioctl_desc ov2643_ioctl_desc[] = {
       {vidioc_int_enum_framesizes_num,
         (v4l2_int_ioctl_func *)ioctl_enum_framesizes},
       {vidioc_int_enum_frameintervals_num,
         (v4l2_int_ioctl_func *)ioctl_enum_frameintervals},
       {vidioc_int_dev_init_num,
         (v4l2_int_ioctl_func *)ioctl_dev_init},
       {vidioc_int_dev_exit_num,
         (v4l2_int_ioctl_func *)ioctl_dev_exit},
       {vidioc_int_s_power_num,
         (v4l2_int_ioctl_func *)ioctl_s_power},
       {vidioc_int_g_priv_num,
         (v4l2_int_ioctl_func *)ioctl_g_priv},
       {vidioc_int_init_num,
         (v4l2_int_ioctl_func *)ioctl_init},
       {vidioc_int_enum_fmt_cap_num,
         (v4l2_int_ioctl_func *)ioctl_enum_fmt_cap},
       {vidioc_int_try_fmt_cap_num,
         (v4l2_int_ioctl_func *)ioctl_try_fmt_cap},
       {vidioc_int_g_fmt_cap_num,
         (v4l2_int_ioctl_func *)ioctl_g_fmt_cap},
       {vidioc_int_s_fmt_cap_num,
         (v4l2_int_ioctl_func *)ioctl_s_fmt_cap},
       {vidioc_int_g_parm_num,
         (v4l2_int_ioctl_func *)ioctl_g_parm},
       {vidioc_int_s_parm_num,
         (v4l2_int_ioctl_func *)ioctl_s_parm},
       {vidioc_int_queryctrl_num,
         (v4l2_int_ioctl_func *)ioctl_queryctrl},
       {vidioc_int_g_ctrl_num,
         (v4l2_int_ioctl_func *)ioctl_g_ctrl},
       {vidioc_int_s_ctrl_num,
         (v4l2_int_ioctl_func *)ioctl_s_ctrl},
         { vidioc_int_g_crop_num,
         (v4l2_int_ioctl_func *)ioctl_g_crop},
       {vidioc_int_s_crop_num,
         (v4l2_int_ioctl_func *)ioctl_s_crop},
       {vidioc_int_cropcap_num,
         (v4l2_int_ioctl_func *)ioctl_cropcap},
	   {vidioc_int_s_reg_num,
         (v4l2_int_ioctl_func *)ioctl_s_reg},
	   {vidioc_int_g_reg_num,
         (v4l2_int_ioctl_func *)ioctl_g_reg},
};




static struct v4l2_int_slave ov2643_slave = {
       .ioctls         = ov2643_ioctl_desc,
       .num_ioctls     = ARRAY_SIZE(ov2643_ioctl_desc),
};

static struct v4l2_int_device ov2643_int_device = {
       .module = THIS_MODULE,
       .name   = OV2643_DRIVER_NAME,
       .priv   = &ov2643,
       .type   = v4l2_int_type_slave,
       .u      = {
               .slave = &ov2643_slave,
       },
};

/*
 * ov2643_probe - sensor driver i2c probe handler
 * @client: i2c driver client device structure
 *
 * Register sensor as an i2c client device and V4L2
 * device.
 */
static int
ov2643_probe(struct i2c_client *client, const struct i2c_device_id *id)
{
       struct ov2643_sensor *sensor = &ov2643;
       int err = 0;

	   if(!client || !id) {
		   PDEBUG("Invalid input\n");
		   return -1;
	   }

      if (i2c_get_clientdata(client))
               return -EBUSY;

       sensor->pdata = client->dev.platform_data;

       if (!sensor->pdata) {
               dev_err(&client->dev, "No platform data?\n");
               return -ENODEV;
       }

       sensor->v4l2_int_device = &ov2643_int_device;
       sensor->i2c_client = client;

       i2c_set_clientdata(client, sensor);

       /* Make the default capture format QXGA 4:2:0 Semi Planar */
       sensor->pix.width = ov2643_sizes[CIF].width;
	   sensor->pix.height = ov2643_sizes[CIF].height;
       sensor->pix.pixelformat = V4L2_PIX_FMT_NV12;

	   sensor->crop_rect.height = ov2643_sizes[CIF].height;
	   sensor->crop_rect.width = ov2643_sizes[CIF].width;
	   sensor->crop_rect.left = 0;
	   sensor->crop_rect.top = 0;

	   sensor->timeperframe.denominator = 15;
	   sensor->timeperframe.numerator = 1;
	   sensor->vflip = 0;
	   sensor->hflip = 0;
		

       err = v4l2_int_device_register(sensor->v4l2_int_device);
       if (err){
           i2c_set_clientdata(client, NULL);
		   return -1;
	   }

	   PDEBUG("\n\n\n P r o b i n g	OV2643	s e n s o r	 s u c c e s s   ! ! !\n\n\n");
       return 0;
}

/*
 * ov2643_remove - sensor driver i2c remove handler
 * @client: i2c driver client device structure
 *
 * Unregister sensor as an i2c client device and V4L2
 * device. Complement of ov2643_probe().
 */
static int __exit
ov2643_remove(struct i2c_client *client)
{
       struct ov2643_sensor *sensor = i2c_get_clientdata(client);

       if (!client->adapter)
               return -ENODEV; /* our client isn't attached */

       v4l2_int_device_unregister(sensor->v4l2_int_device);
       i2c_set_clientdata(client, NULL);

       return 0;
}

static const struct i2c_device_id ov2643_id[] = {
       { OV2643_DRIVER_NAME, 0 },
       { },
};
MODULE_DEVICE_TABLE(i2c, ov2643_id);

static struct i2c_driver ov2643sensor_i2c_driver = {
       .driver = {
               .name   = OV2643_DRIVER_NAME,
               .owner = THIS_MODULE,
       },
       .probe  = ov2643_probe,
       .remove = __exit_p(ov2643_remove),
       .id_table = ov2643_id,
};

static struct ov2643_sensor ov2643 = {
       .timeperframe = {
               .numerator = 1,
               .denominator = 15,
       },
       .state = SENSOR_NOT_DETECTED,
};

/*
 * ov2643sensor_init - sensor driver module_init handler
 *
 * Registers driver as an i2c client driver.  Returns 0 on success,
 * error code otherwise.
 */
static int __init ov2643sensor_init(void)
{
       int err = 0;

       err = i2c_add_driver(&ov2643sensor_i2c_driver);
       if (err) {
               printk(KERN_ERR "Failed to register" OV2643_DRIVER_NAME ".\n");
               return err;
       }
       return 0;
}
module_init(ov2643sensor_init);

/*
 * ov2643sensor_cleanup - sensor driver module_exit handler
 *
 * Unregisters/deletes driver as an i2c client driver.
 * Complement of ov2643sensor_init.
 */
static void __exit ov2643sensor_cleanup(void)
{
       i2c_del_driver(&ov2643sensor_i2c_driver);
}
module_exit(ov2643sensor_cleanup);

MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("OV2643 camera sensor driver");


