/**
 * dp52-onkey.c - DP52 On/Off button driver
 *
 * Copyright (C) 2011 DSP Group
 *
 * This file is subject to the terms and conditions of the GNU General
 * Public License. See the file "COPYING" in the main directory of this
 * archive for more details.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/kernel.h>
#include <linux/errno.h>
#include <linux/input.h>
#include <linux/interrupt.h>
#include <linux/platform_device.h>
#include <linux/workqueue.h>
#include <linux/mfd/dp52/core.h>

struct dp52_onkey {
	struct dp52 *dp52;
	struct input_dev *dev;
	struct delayed_work work;
	int irq;
};

/*
 * The DP gives us an interrupt when the On/Off button is pressed but we then
 * need to poll to see when the button is released.
 */
static void dp52_poll_onkey(struct work_struct *work)
{
	struct dp52_onkey *dp52_onkey = container_of(work, struct dp52_onkey, work.work);
	int pressed;

	pressed = !!(dp52_read(dp52_onkey->dp52, DP52_PMU_RAWSTAT) & 0x0001);
	input_report_key(dp52_onkey->dev, KEY_POWER, pressed);
	input_sync(dp52_onkey->dev);

	if (pressed)
		schedule_delayed_work(&dp52_onkey->work, HZ/10);
}

static irqreturn_t dp52_onkey_irq(int irq, void *data)
{
	struct dp52_onkey *dp52_onkey = data;
	int pending;

	pending = dp52_read(dp52_onkey->dp52, DP52_PMU_STAT);

	if (pending & 0x0001) {
		schedule_delayed_work(&dp52_onkey->work, 0);
		dp52_write(dp52_onkey->dp52, DP52_PMU_STAT, ~0x0001);
		return IRQ_HANDLED;
	}

	return IRQ_NONE;
}

static int __devinit dp52_onkey_probe(struct platform_device *pdev)
{
	struct dp52 *dp52 = dev_get_drvdata(pdev->dev.parent);
	struct dp52_onkey *dp52_onkey;
	int irq = platform_get_irq(pdev, 0);
	int ret;

	dp52_onkey = kzalloc(sizeof(*dp52_onkey), GFP_KERNEL);
	if (!dp52_onkey)
		return -ENOMEM;

	dp52_onkey->dp52 = dp52;
	dp52_onkey->irq = irq;
	INIT_DELAYED_WORK(&dp52_onkey->work, dp52_poll_onkey);

	dp52_onkey->dev = input_allocate_device();
	if (!dp52_onkey->dev) {
		dev_err(&pdev->dev, "Can't allocate input dev\n");
		ret = -ENOMEM;
		goto err;
	}

	dp52_onkey->dev->evbit[0] = BIT_MASK(EV_KEY);
	dp52_onkey->dev->keybit[BIT_WORD(KEY_POWER)] = BIT_MASK(KEY_POWER);
	dp52_onkey->dev->name = "dp52-onkey";
	dp52_onkey->dev->phys = "dp52-onkey/input0";
	dp52_onkey->dev->dev.parent = &pdev->dev;

	ret = request_threaded_irq(irq, NULL, dp52_onkey_irq,
				   IRQF_SHARED, "dp52-onkey",
				   dp52_onkey);
	if (ret < 0) {
		dev_err(&pdev->dev, "Unable to request IRQ: %d\n", ret);
		goto err_input_dev;
	}

	ret = input_register_device(dp52_onkey->dev);
	if (ret) {
		dev_dbg(&pdev->dev, "Can't register input device: %d\n", ret);
		goto err_irq;
	}

	/* enable On/Off button interrupt */
	dp52_set_bits(dp52, DP52_PMU_INTMSK, 1 << 0);

	device_init_wakeup(&pdev->dev, 1);
	platform_set_drvdata(pdev, dp52_onkey);
	dev_info(&pdev->dev, "initialized\n");

	return 0;

err_irq:
	free_irq(irq, dp52_onkey);
err_input_dev:
	input_free_device(dp52_onkey->dev);
err:
	kfree(dp52_onkey);
	return ret;
}

static int __devexit dp52_onkey_remove(struct platform_device *pdev)
{
	struct dp52_onkey *dp52_onkey = platform_get_drvdata(pdev);

	free_irq(dp52_onkey->irq, dp52_onkey);
	cancel_delayed_work_sync(&dp52_onkey->work);
	input_unregister_device(dp52_onkey->dev);
	kfree(dp52_onkey);

	return 0;
}

static int dp52_onkey_suspend(struct device *dev)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct dp52_onkey *dp52_onkey = platform_get_drvdata(pdev);

	if (device_may_wakeup(dev))
		enable_irq_wake(dp52_onkey->irq);

	return 0;
}

static int dp52_onkey_resume(struct device *dev)
{
	struct platform_device *pdev = to_platform_device(dev);
	struct dp52_onkey *dp52_onkey = platform_get_drvdata(pdev);

	if (device_may_wakeup(dev))
		disable_irq_wake(dp52_onkey->irq);

	return 0;
}

static struct dev_pm_ops dp52_onkey_pm_ops = {
	.suspend	= dp52_onkey_suspend,
	.resume		= dp52_onkey_resume,
};

static struct platform_driver dp52_onkey_driver = {
	.probe		= dp52_onkey_probe,
	.remove		= __devexit_p(dp52_onkey_remove),
	.driver		= {
		.name	= "dp52-onkey",
		.owner	= THIS_MODULE,
		.pm	= &dp52_onkey_pm_ops,
	},
};

static int __init dp52_onkey_init(void)
{
	return platform_driver_register(&dp52_onkey_driver);
}
module_init(dp52_onkey_init);

static void __exit dp52_onkey_exit(void)
{
	platform_driver_unregister(&dp52_onkey_driver);
}
module_exit(dp52_onkey_exit);

MODULE_ALIAS("platform:dp52-onkey");
MODULE_LICENSE("GPL");

