/*
 *  linux/arch/arm/mach-dmw/pm.c
 *
 *  Copyright (C) 2011 DSP Group
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <asm/atomic.h>
#include <linux/clk.h>
#include <linux/cpuidle.h>
#include <linux/err.h>
#include <linux/gpio.h>
#include <linux/module.h>
#include <linux/platform_device.h>
#include <linux/pm.h>
#include <linux/suspend.h>
#include <mach/system.h>

#include "irq.h"
#include "pm.h"

#define param_lowpower_auto_enable_bits 5
#define param_lowpower_auto_enable_reg  22
#define param_lowpower_auto_enable_offset 0
#define param_lowpower_control_bits 5
#define param_lowpower_control_reg  20
#define param_lowpower_control_offset 8
#define param_lowpower_external_cnt_bits 16
#define param_lowpower_external_cnt_reg  21
#define param_lowpower_external_cnt_offset 16
#define param_lowpower_internal_cnt_bits 16
#define param_lowpower_internal_cnt_reg  22
#define param_lowpower_internal_cnt_offset 8
#define param_lowpower_power_down_cnt_bits 16
#define param_lowpower_power_down_cnt_reg  20
#define param_lowpower_power_down_cnt_offset 16
#define param_lowpower_refresh_enable_bits 2
#define param_lowpower_refresh_enable_reg  22
#define param_lowpower_refresh_enable_offset 24
#define param_lowpower_self_refresh_cnt_bits 16
#define param_lowpower_self_refresh_cnt_reg  21
#define param_lowpower_self_refresh_cnt_offset 0

/* defined by DSPG */
#define param_dram_type_bits 4
#define param_dram_type_reg  129
#define param_dram_type_offset 16

/* denali parameter access */
#define denali_readl(_regno)		readl(denali_base + (_regno) * 4)
#define denali_writel(_val, _regno)	writel((_val), denali_base + (_regno) * 4)

#define denali_set(_name, _val)                               \
({                                                            \
	uint32_t msk = (1 << param_##_name##_bits) - 1;       \
	uint32_t reg;                                         \
	                                                      \
	reg = denali_readl(param_##_name##_reg);              \
	reg &= ~(msk << param_##_name##_offset);              \
	reg |= (_val) << param_##_name##_offset;              \
	denali_writel(reg, param_##_name##_reg);              \
})

#define denali_get(_name)                                     \
({                                                            \
	uint32_t msk = (1 << param_##_name##_bits) - 1;       \
	uint32_t reg;                                         \
	                                                      \
	reg = denali_readl(param_##_name##_reg);              \
	reg >>= param_##_name##_offset;                       \
	reg & msk;                                            \
})

static void *denali_base;
static struct clk *cpu_clk;
static struct clk *cpu_slow_clk;
static atomic_t dmw_dram_lock = ATOMIC_INIT(0);
static int quirks;
static unsigned overdrive_gpio;
static struct cpuidle_device *dmw_idle_dev;

void dmw_dram_request(void)
{
	atomic_inc(&dmw_dram_lock);
}

void dmw_dram_release(void)
{
	int res = atomic_dec_return(&dmw_dram_lock);
	BUG_ON(res < 0);
}

int dmw_dram_reqlevel(void)
{
	return atomic_read(&dmw_dram_lock);
}

static int dmw_sleep(int req_mode, int safe_mode)
{
	unsigned long reg, ret = 0;
	int mode, gpio_val = 0;
	struct clk *cpu_fast_clk = NULL;

	req_mode |= quirks;
	safe_mode |= quirks;

	if (req_mode & SLEEP_CPU_SLOW) {
		cpu_fast_clk = clk_get_parent(cpu_clk);
		ret = clk_set_parent(cpu_clk, cpu_slow_clk);
		if (ret)
			return ret;
/* This code is disabled until boards can safely work with CPU overdrive disabled */
#if 0
		gpio_val = gpio_get_value(overdrive_gpio);
		gpio_set_value(overdrive_gpio, 0);
#endif	
	}

	/*
	 * Stay here until an IRQ is pending. FIQ's will wake us up too and
	 * will be processed but they will not cause us to leave.
	 */
	while (!dmw_irq_pending()) {
		local_fiq_disable();

		/* Enable automatic system clock control if possible */
		if (!clk_active_io_cnt) {
			reg = readl(IO_ADDRESS(DMW_CMU_BASE) + DMW_CMU_CPUCLKCNTRL);
			reg |= 1 << 18;
			writel(reg, IO_ADDRESS(DMW_CMU_BASE) + DMW_CMU_CPUCLKCNTRL);
		}

		if (req_mode & SLEEP_CHECK_BM && atomic_read(&dmw_dram_lock) > 0)
			mode = safe_mode;
		else
			mode = req_mode;

		if (mode & SLEEP_DDR_PLL_OFF) {
			dmw_sleep_selfrefresh(mode, denali_base, IO_ADDRESS(DMW_CMU_BASE));
		} else {
			int mode_flag = 1 << SLEEP_DDR_MODE(mode);

			denali_set(lowpower_auto_enable, mode_flag);
			denali_set(lowpower_control, mode_flag);
			cpu_do_idle(); /* just WFI */
			denali_set(lowpower_control, 0);
			denali_set(lowpower_auto_enable, 0);
		}

		/* Disable automatic system clock control if enabled previously*/
		if (!clk_active_io_cnt) {
			reg = readl(IO_ADDRESS(DMW_CMU_BASE) + DMW_CMU_CPUCLKCNTRL);
			reg &= ~(1 << 18);
			writel(reg, IO_ADDRESS(DMW_CMU_BASE) + DMW_CMU_CPUCLKCNTRL);
		}

		local_fiq_enable();
	}

	if (req_mode & SLEEP_CPU_SLOW) {
/* This code is disabled until boards can safely work with CPU overdrive disabled */
#if 0
		gpio_set_value(overdrive_gpio, gpio_val);
#endif
		ret = clk_set_parent(cpu_clk, cpu_fast_clk);
	}

	return ret;
}

static int dmw_idle_enter(struct cpuidle_device *dev, struct cpuidle_state *state)
{
	struct timespec ts_preidle, ts_postidle, ts_idle;

	/* Fall back to safe idle mode if DRAM access is required */
	if ((int)state->driver_data & SLEEP_CHECK_BM &&
	    (clk_active_bm_cnt || atomic_read(&dmw_dram_lock) > 0)) {
		dev->last_state = dev->safe_state;
		return dev->safe_state->enter(dev, dev->safe_state);
	}

	/* Used to keep track of the total time in idle */
	getnstimeofday(&ts_preidle);

	dmw_sleep((int)state->driver_data, (int)dev->safe_state->driver_data);

	/* calculate sleep time */
	getnstimeofday(&ts_postidle);

	local_irq_enable(); /* idle handler must re-enable IRQs on exit */

	ts_idle = timespec_sub(ts_postidle, ts_preidle);
	return ts_idle.tv_nsec / NSEC_PER_USEC + ts_idle.tv_sec * USEC_PER_SEC;
}

struct cpuidle_device dmw_idle_dev_lpddr2 = {
	.states = {
		{
			.name = "C0",
			.desc = "M4",
			.driver_data = (void *)SLEEP_DDR_MODE_4,
			.flags = CPUIDLE_FLAG_TIME_VALID,
			.exit_latency = 0, /* in US */
			.power_usage = 150, /* in mW */
			.target_residency = 0, /* in US */
			.enter = dmw_idle_enter,
		},
		{
			.name = "C1",
			.desc = "M4+PLL2",
			.driver_data = (void *)(SLEEP_DDR_MODE_4 |
						SLEEP_CPU_SLOW),
			.flags = CPUIDLE_FLAG_TIME_VALID,
			.exit_latency = 200, /* in US */
			.power_usage = 140, /* in mW */
			.target_residency = 400, /* in US */
			.enter = dmw_idle_enter,
		},
		{
			.name = "C2",
			.desc = "M4+PLL2+PLL3",
			.driver_data = (void *)(SLEEP_DDR_MODE_4 |
						SLEEP_CHECK_BM |
						SLEEP_DDR_PLL_OFF |
						SLEEP_CPU_SLOW),
			.flags = CPUIDLE_FLAG_TIME_VALID,
			.exit_latency = 400, /* in US */
			.power_usage = 5, /* in mW */
			.target_residency = 800, /* in US */
			.enter = dmw_idle_enter,
		},
	},
	.state_count = 3, /* static array -> cannot use ARRAY_SIZE! */
	.safe_state = &dmw_idle_dev_lpddr2.states[1],
};

struct cpuidle_device dmw_idle_dev_ddr2 = {
	.states = {
		{
			.name = "C0",
			.desc = "M1",
			.driver_data = (void *)SLEEP_DDR_MODE_1,
			.flags = CPUIDLE_FLAG_TIME_VALID,
			.exit_latency = 0, /* in US */
			.power_usage = 200, /* in mW */
			.target_residency = 0, /* in US */
			.enter = dmw_idle_enter,
		},
		{
			.name = "C1",
			.desc = "M3",
			.driver_data = (void *)(SLEEP_DDR_MODE_3 |
						SLEEP_CHECK_BM),
			.flags = CPUIDLE_FLAG_TIME_VALID,
			.exit_latency = 1, /* in US */
			.power_usage = 150, /* in mW */
			.target_residency = 2, /* in US */
			.enter = dmw_idle_enter,
		},
		{
			.name = "C2",
			.desc = "M3+PLL2",
			.driver_data = (void *)(SLEEP_DDR_MODE_3 |
						SLEEP_CHECK_BM |
						SLEEP_CPU_SLOW),
			.flags = CPUIDLE_FLAG_TIME_VALID,
			.exit_latency = 200, /* in US */
			.power_usage = 140, /* in mW */
			.target_residency = 400, /* in US */
			.enter = dmw_idle_enter,
		},
	},
	.state_count = 3, /* static array -> cannot use ARRAY_SIZE! */
	.safe_state = &dmw_idle_dev_ddr2.states[0],
};

struct cpuidle_driver dmw_idle_driver = {
	.name  = "dmw_idle",
	.owner = THIS_MODULE,
};


static int dmw_pm_prepare(void)
{
	disable_hlt();
	return 0;
}

static int dmw_pm_prepare_late(void)
{
	if (clk_active_io_cnt)
		printk(KERN_WARNING "%d active IO devices remaining!\n",
			clk_active_io_cnt);

	/* make sure there are no active bus masters */
	if (clk_active_bm_cnt) {
		printk(KERN_ERR "%d active bus masters remaining!"
			" Aborting suspend...\n", clk_active_bm_cnt);
		dmw_clk_print_active();
		return -EBUSY;
	}

	return 0;
}

static int dmw_pm_enter(suspend_state_t state)
{
	int ret = 0;
	struct cpuidle_device *dev = dmw_idle_dev;

	switch (state) {
	case PM_SUSPEND_STANDBY:
	case PM_SUSPEND_MEM:
		dmw_irq_suspend();
		ret = dmw_sleep((int)dev->states[dev->state_count-1].driver_data,
				(int)dev->safe_state->driver_data);
		dmw_irq_resume();
		break;
	default:
		ret = -EINVAL;
	}

	return ret;
}

static void dmw_pm_finish(void)
{
	enable_hlt();
}

static struct platform_suspend_ops dmw_pm_ops = {
	.prepare	= dmw_pm_prepare,
	.prepare_late	= dmw_pm_prepare_late,
	.enter		= dmw_pm_enter,
	.finish		= dmw_pm_finish,
	.valid		= suspend_valid_only_mem,
};

static unsigned long __init ns2clk(unsigned long ns, unsigned long f)
{
	/* Assume that f is in MHz range and ns < 1us */
	return ((f / 1000) * ns) / 1000000 + 1;
}

static int __init dmw_pm_probe(struct platform_device *pdev)
{
	struct dmw_pm_pdata *pdata = pdev->dev.platform_data;
	struct clk *dram_clk;
	unsigned long dram_freq;
	int ret = 0;
	unsigned long reg, flags;

	if (!pdata)
		return -EINVAL;

	overdrive_gpio = pdata->overdrive_gpio;

	denali_base = ioremap_nocache(DMW_MPMC_BASE, SZ_4K);
	if (!denali_base) {
		dev_err(&pdev->dev, "could not map Denali controller!\n");
		return -ENOMEM;
	}

	cpu_clk = clk_get(&pdev->dev, "cpu");
	if (IS_ERR(cpu_clk)) {
		dev_err(&pdev->dev, "cpu clock missing!\n");
		ret = PTR_ERR(cpu_clk);
		goto free_denali;
	}

	cpu_slow_clk = clk_get(&pdev->dev, "slow");
	if (IS_ERR(cpu_slow_clk)) {
		dev_err(&pdev->dev, "slow clock missing!\n");
		ret = PTR_ERR(cpu_slow_clk);
		goto put_cpu_clk;
	}

	dram_clk = clk_get(&pdev->dev, "dram");
	if (IS_ERR(dram_clk)) {
		dev_err(&pdev->dev, "dram clock missing!\n");
		ret = PTR_ERR(dram_clk);
		goto put_slow_clk;
	}
	dram_freq = clk_get_rate(dram_clk);
	clk_put(dram_clk);
	if (dram_freq <= 0) {
		dev_err(&pdev->dev, "unknown dram freq!\n");
		ret = -ENODEV;
		goto put_slow_clk;
	}

	/* apply quirks as necessary */
	if (dmw_get_chip_rev() == DMW_CHIP_DMW96_REV_A)
		quirks = SLEEP_QUIRK_DIS_INP;

	/* determine RAM type */
	switch (denali_get(dram_type)) {
	case 0x0c:
		dmw_idle_dev = &dmw_idle_dev_lpddr2;
		denali_set(lowpower_power_down_cnt, (2 + ns2clk(8, dram_freq)) * 3);
		denali_set(lowpower_external_cnt, (2 + ns2clk(140, dram_freq)) * 2);
		break;
	case 0x0f:
		dmw_idle_dev = &dmw_idle_dev_ddr2;
		denali_set(lowpower_power_down_cnt, (3+2) * 3);
		denali_set(lowpower_self_refresh_cnt, (3+200) * 2);
		break;
	default:
		dev_err(&pdev->dev, "unknown RAM type!\n");
		ret = -ENODEV;
		goto put_slow_clk;
	}

	/*
	 * Make sure we power down PLL1 in low power and run SYSCLK from 12MHz
	 */
	raw_local_irq_save(flags);
	local_fiq_disable();
	reg = readl(IO_ADDRESS(DMW_CMU_BASE) + DMW_CMU_CPUCLKCNTRL);
	reg &= ~(3ul << 20); /* Run from CPUICLK in low power */
	reg |= 1ul << 19;    /* power down PLL1 in low power */
	writel(reg, IO_ADDRESS(DMW_CMU_BASE) + DMW_CMU_CPUCLKCNTRL);
	raw_local_irq_restore(flags);

	suspend_set_ops(&dmw_pm_ops);

	ret = cpuidle_register_driver(&dmw_idle_driver);
	if (!ret)
		ret = cpuidle_register_device(dmw_idle_dev);

	if (ret)
		dev_warn(&pdev->dev, "idle driver not registered: %d\n", ret);

	return 0;

put_slow_clk:
	clk_put(cpu_slow_clk);
put_cpu_clk:
	clk_put(cpu_clk);
free_denali:
	iounmap(denali_base);
	return ret;
}

static struct platform_driver dmw_pm_plat_driver = {
	.driver = {
		.name	 = "dmw-pm",
		.owner	 = THIS_MODULE,
	},
};

static int __init dmw_pm_init(void)
{
	return platform_driver_probe(&dmw_pm_plat_driver, dmw_pm_probe);
}

late_initcall(dmw_pm_init);
