/*
 *  linux/arch/arm/mach-dmw/pm-cpufreq.c
 *
 *  Copyright (C) 2011 DSPG Technologies GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/cpufreq.h>
#include <linux/init.h>
#include <linux/err.h>
#include <linux/clk.h>
#include <linux/gpio.h>
#include <linux/platform_device.h>

#include <mach/cpufreq.h>


struct dmw_cpufreq {
	struct device *dev;
	struct cpufreq_frequency_table *freq_table;
	struct clk *cpu_clk;
	struct clk *cpu_fast_clk;
	struct clk *cpu_safe_clk;
};

static struct dmw_cpufreq cpufreq;

static int dmw_verify_speed(struct cpufreq_policy *policy)
{
	return cpufreq_frequency_table_verify(policy, cpufreq.freq_table);
}

static unsigned int dmw_getspeed(unsigned int cpu)
{
	return clk_get_rate(cpufreq.cpu_clk) / 1000;
}

static int dmw_target(struct cpufreq_policy *policy, unsigned int target_freq,
		      unsigned int relation)
{
	int ret = 0;
	long new_freq;
	unsigned int idx;
	unsigned long flags;
	struct cpufreq_freqs freqs;
	struct dmw_cpufreq_pdata *pdata = cpufreq.dev->platform_data;

	/*
	 * Ensure desired rate is within allowed range.  Some govenors
	 * (ondemand) will just pass target_freq=0 to get the minimum.
	 */
	if (target_freq < policy->cpuinfo.min_freq)
		target_freq = policy->cpuinfo.min_freq;
	if (target_freq > policy->cpuinfo.max_freq)
		target_freq = policy->cpuinfo.max_freq;

	freqs.old = dmw_getspeed(0);
	if (freqs.old == target_freq)
		return 0;

	new_freq = clk_round_rate(cpufreq.cpu_fast_clk, target_freq * 1000);
	if (new_freq < 0)
		return new_freq;

	freqs.new = new_freq / 1000;
	freqs.cpu = 0;
	if (freqs.old == freqs.new)
		return ret;

	cpufreq_debug_printk(CPUFREQ_DEBUG_DRIVER,
			dev_driver_string(cpufreq.dev),
			"transition: %u --> %u\n", freqs.old, freqs.new);

	ret = cpufreq_frequency_table_target(policy, cpufreq.freq_table,
						freqs.new, relation, &idx);
	if (ret)
		return -EINVAL;

	cpufreq_notify_transition(&freqs, CPUFREQ_PRECHANGE);

	/* if moving to higher frequency, up the voltage beforehand */
	if (freqs.new > freqs.old)
		gpio_set_value(pdata->overdrive_gpio, cpufreq.freq_table[idx].index);

	local_irq_save(flags);

	ret = clk_set_parent(cpufreq.cpu_clk, cpufreq.cpu_safe_clk);
	if (ret)
		goto done;

	ret = clk_set_rate(cpufreq.cpu_fast_clk, new_freq);
	if (ret) {
		clk_set_parent(cpufreq.cpu_clk, cpufreq.cpu_fast_clk);
		goto done;
	}

	ret = clk_set_parent(cpufreq.cpu_clk, cpufreq.cpu_fast_clk);

done:
	local_irq_restore(flags);

	/* if moving to lower freq, lower the voltage after lowering freq */
	if (freqs.new < freqs.old)
		gpio_set_value(pdata->overdrive_gpio, cpufreq.freq_table[idx].index);

	cpufreq_notify_transition(&freqs, CPUFREQ_POSTCHANGE);

	return ret;
}

static int dmw_cpu_init(struct cpufreq_policy *policy)
{
	int ret;

	ret = cpufreq_frequency_table_cpuinfo(policy, cpufreq.freq_table);
	if (ret == 0)
		cpufreq_frequency_table_get_attr(cpufreq.freq_table, policy->cpu);

	policy->min = policy->cpuinfo.min_freq;
	policy->max = policy->cpuinfo.max_freq;
	policy->cur = dmw_getspeed(0);
	policy->cpuinfo.transition_latency = 1000 * 1000; /* TODO: measure */

	return 0;
}

static int dmw_cpu_exit(struct cpufreq_policy *policy)
{
	cpufreq_frequency_table_put_attr(policy->cpu);
	return 0;
}

static struct freq_attr *dmw_cpufreq_attr[] = {
	&cpufreq_freq_attr_scaling_available_freqs,
	NULL,
};


static struct cpufreq_driver dmw_cpufreq_driver = {
	.flags		= CPUFREQ_STICKY,
	.verify		= dmw_verify_speed,
	.target		= dmw_target,
	.get		= dmw_getspeed,
	.init		= dmw_cpu_init,
	.exit		= dmw_cpu_exit,
	.name		= "dmw",
	.attr		= dmw_cpufreq_attr,
};

static struct cpufreq_frequency_table static_freq_table[] = {
	{ .frequency = 0, },
	{ .frequency = CPUFREQ_TABLE_END, },
};

static int __init dmw_cpufreq_probe(struct platform_device *pdev)
{
	struct dmw_cpufreq_pdata *pdata = pdev->dev.platform_data;
	int ret;

	if (!pdata)
		return -EINVAL;
	if (!pdata->freq_table)
		return -EINVAL;

	cpufreq.dev = &pdev->dev;
	cpufreq.freq_table = pdata->freq_table;

	ret = gpio_request(pdata->overdrive_gpio,
			   (char *)dev_driver_string(cpufreq.dev));
	if (ret) {
		dev_err(&pdev->dev, "Unable to get GPIO\n");
		return ret;
	}

	cpufreq.cpu_clk = clk_get(&pdev->dev, "cpu");
	if (IS_ERR(cpufreq.cpu_clk)) {
		dev_err(&pdev->dev, "Unable to get CPU clock\n");
		gpio_free(pdata->overdrive_gpio);
		return PTR_ERR(cpufreq.cpu_clk);
	}

	/*
	 * Frequency switching might not be possible. Fall back to a frequency
	 * table with just one entry with the current frequency.
	 */
	cpufreq.cpu_fast_clk = clk_get(&pdev->dev, "fast");
	cpufreq.cpu_safe_clk = clk_get(&pdev->dev, "safe");
	if (IS_ERR(cpufreq.cpu_fast_clk) || IS_ERR(cpufreq.cpu_safe_clk)) {
		static_freq_table[0].frequency = dmw_getspeed(0);
		cpufreq.cpu_fast_clk = NULL;
		cpufreq.cpu_safe_clk = NULL;
		cpufreq.freq_table = static_freq_table;
		dev_warn(&pdev->dev, "Frequency scaling disabled\n");
	}

	return cpufreq_register_driver(&dmw_cpufreq_driver);
}

static int __exit dmw_cpufreq_remove(struct platform_device *pdev)
{
	struct dmw_cpufreq_pdata *pdata = pdev->dev.platform_data;

	if (cpufreq.cpu_safe_clk)
		clk_put(cpufreq.cpu_safe_clk);
	if (cpufreq.cpu_fast_clk)
		clk_put(cpufreq.cpu_fast_clk);
	clk_put(cpufreq.cpu_clk);
	gpio_free(pdata->overdrive_gpio);

	return cpufreq_unregister_driver(&dmw_cpufreq_driver);
}

static struct platform_driver dmw_cpufreq_plat_driver = {
	.driver = {
		.name	 = "cpufreq-dmw",
		.owner	 = THIS_MODULE,
	},
	.remove = __exit_p(dmw_cpufreq_remove),
};

static int __init dmw_cpufreq_init(void)
{
	return platform_driver_probe(&dmw_cpufreq_plat_driver, dmw_cpufreq_probe);
}
late_initcall(dmw_cpufreq_init);

