/*
 *  linux/arch/arm/mach-dmw/irq.c
 *
 *  Copyright (C) 2010 DSPG Technologies GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/irq.h>
#include <linux/ioport.h>
#include <linux/gpio.h>

#include <asm/io.h>
#include <asm/irq.h>
#include <asm/fiq.h>

#include <mach/irqs.h>
#include <mach/hardware.h>

#define DMW_EXTINT_CFG		0x2A0
#define DMW_EXTINT_EDGE		0x2A4
#define DMW_EXTINT_STAT		0x2A8
#define DMW_EXTINT_EDGE_CLEAR	0x2AC

static void __iomem *dmw_plicu_base = (void __iomem *)IO_ADDRESS(DMW_PLICU_BASE);
static void __iomem *dmw_gpio_base;

static unsigned long extint_edge_both;
static unsigned long irq_wake_mask[2];
static unsigned long irq_enable_save[2];

static int irqnr_to_extint(unsigned int irqnr)
{
	unsigned int extint = -1;

	if ((irqnr >= DMW_IID_EXTERNAL_REQUEST_0) &&
	    (irqnr <= DMW_IID_EXTERNAL_REQUEST_6))
	    return irqnr - DMW_IID_EXTERNAL_REQUEST_0;
	if ((irqnr >= DMW_IID_EXTERNAL_REQUEST_7) &&
	    (irqnr <= DMW_IID_EXTERNAL_REQUEST_15))
	    return irqnr - DMW_IID_EXTERNAL_REQUEST_7 + 7;

	return extint;
}

static unsigned extint_to_gpio(int extint)
{
	return GPIO_PORTF(extint + 16);
}

static int is_sw_irq(unsigned int irqnr)
{
	switch (irqnr) {
	case DMW_IID_SW1:
	case DMW_IID_SW2:
	case DMW_IID_SW3:
	case DMW_IID_SW4:
		return 1;
	}
	return 0;
}

static void dmw_irq_ack(struct irq_data *data)
{
	unsigned long reg_clr;
	unsigned int irq = data->irq;

	if (irq > 31) {
		irq -= 32;
		reg_clr = DMW_PLICU_CH_CLR_INT2;
	} else {
		reg_clr = DMW_PLICU_CH_CLR_INT1;
	}

	/* clear interrupt */
	writel(1 << irq, dmw_plicu_base + reg_clr);
}

static int dmw_irq_set_type_extint(struct irq_data *data, unsigned int type);

static void dmw_irq_ack_extint(struct irq_data *data)
{
	int extint = irqnr_to_extint(data->irq);

	BUG_ON(extint == -1);

	writel(1 << extint, dmw_gpio_base + DMW_EXTINT_EDGE_CLEAR);
	dmw_irq_ack(data);

	if (test_bit(extint, &extint_edge_both)) {
		/* call set_type to update EDGE_BOTH emulation */
		dmw_irq_set_type_extint(data, IRQ_TYPE_EDGE_BOTH);
	}
}

static void dmw_irq_mask(struct irq_data *data)
{
	unsigned long reg_mask;
	unsigned int irq = data->irq;

	/* Beware the inverted semantic of 'masking' in the PLICU */
	if (irq > 31) {
		irq -= 32;
		reg_mask = DMW_PLICU_CLR_CH_MASK2;
	} else {
		reg_mask = DMW_PLICU_CLR_CH_MASK1;
	}

	writel(1 << irq, dmw_plicu_base + reg_mask);
}

static void dmw_irq_unmask(struct irq_data *data)
{
	unsigned long reg_mask;
	unsigned int irq = data->irq;

	/* Beware the inverted semantic of 'masking' in the PLICU */
	if (irq > 31) {
		irq -= 32;
		reg_mask = DMW_PLICU_SET_CH_MASK2;
	} else {
		reg_mask = DMW_PLICU_SET_CH_MASK1;
	}

	writel(1 << irq, dmw_plicu_base + reg_mask);
}

static int dmw_irq_set_type(struct irq_data *data, unsigned int flow_type)
{
	unsigned long idx, reg_edge, value;
	unsigned long flags;
	int ret = 0;

	idx = data->irq;
	if (idx > 31) {
		idx -= 32;
		reg_edge = DMW_PLICU_CH_EDGE_LEVEL2;
	} else {
		reg_edge = DMW_PLICU_CH_EDGE_LEVEL1;
	}

	local_irq_save(flags);

	value = readl(dmw_plicu_base + reg_edge);
	if ((flow_type & IRQ_TYPE_SENSE_MASK) == IRQ_TYPE_EDGE_RISING) {
		value |= 1 << idx;
		__irq_set_handler_locked(data->irq, handle_edge_irq);
	} else if ((flow_type & IRQ_TYPE_SENSE_MASK) == IRQ_TYPE_LEVEL_HIGH) {
		value &= ~(1 << idx);
		__irq_set_handler_locked(data->irq, handle_level_irq);
	} else
		ret = -EINVAL;
	writel(value, dmw_plicu_base + reg_edge);

	local_irq_restore(flags);

	return ret;
}

static int dmw_irq_set_type_extint(struct irq_data *data, unsigned int type)
{
	unsigned long value;
	unsigned long flags;
	int pol = 0; /* 0 = active low; 1 = active high */
	int mode = 0;     /* 0 = level; 1 = edge */
	int shift;
	int extint = irqnr_to_extint(data->irq);
	unsigned gpio = extint_to_gpio(extint);

	BUG_ON(extint == -1);
	BUG_ON(gpio_get_direction(gpio) == 0);

	type &= IRQ_TYPE_SENSE_MASK;

	if ((type & IRQ_TYPE_EDGE_BOTH) == IRQ_TYPE_EDGE_BOTH) {
		/*
		 * emulate EDGE_BOTH by setting LEVEL_HIGH or LEVEL_LOW
		 * depending on current gpio value
		 */
		pol = !gpio_get_value(gpio);
		set_bit(extint, &extint_edge_both);

		goto apply_type;
	}

	clear_bit(extint, &extint_edge_both);

	if (type & IRQ_TYPE_EDGE_BOTH) {
		/* specific edge type irq */
		mode = 1;

		if (type & IRQ_TYPE_EDGE_RISING)
			pol = 1;
	}

	if (type & IRQ_TYPE_LEVEL_HIGH)
		pol = 1;

apply_type:
	shift = extint << 1; /* extint * 2 */

	local_irq_save(flags);

	value = readl(dmw_gpio_base + DMW_EXTINT_CFG);
	value &= ~(0x3UL << shift);
	value |= ((mode <<1) | pol) << shift;
	writel(value, dmw_gpio_base + DMW_EXTINT_CFG);

	local_irq_restore(flags);

	return 0;
}

static int dmw_irq_set_wake(struct irq_data *data, unsigned int enable)
{
	if (enable)
		set_bit(data->irq, irq_wake_mask);
	else
		clear_bit(data->irq, irq_wake_mask);

	return 0;
}

static struct irq_chip plicu_chip = {
	.name     = "dmw-plicu",
	.irq_ack      = dmw_irq_ack,
	.irq_mask     = dmw_irq_mask,
	.irq_unmask   = dmw_irq_unmask,
	.irq_set_type = dmw_irq_set_type,
	.irq_set_wake = dmw_irq_set_wake,
};

static struct irq_chip extint_chip = {
	.name     = "dmw-extint",
	.irq_ack      = dmw_irq_ack_extint,
	.irq_mask     = dmw_irq_mask,
	.irq_unmask   = dmw_irq_unmask,
	.irq_set_type = dmw_irq_set_type_extint,
	.irq_set_wake = dmw_irq_set_wake,
};

static struct irq_chip sw_chip = {
	.name     = "dmw-swirq",
	.irq_ack      = dmw_irq_ack,
	.irq_mask     = dmw_irq_mask,
	.irq_unmask   = dmw_irq_unmask,
	.irq_set_wake = dmw_irq_set_wake,
};

void __init dmw_init_irq(void)
{
	unsigned int i;

	/* disable all interrupt sources */
	writel(0xffffffff, dmw_plicu_base + DMW_PLICU_CLR_CH_MASK1);
	writel(0xffffffff, dmw_plicu_base + DMW_PLICU_CLR_CH_MASK2);

	/* set priorities of all interrupts to 1 */
	for (i = 0; i < 8; i++)
		writel(0x11111111, dmw_plicu_base + DMW_PLICU_CH_PRIORITY1 + i*4);

	/* set all sw interrupts to edge trigger */
	writel(0 | (1 << DMW_IID_SW1) | (1 << DMW_IID_SW2),
	       dmw_plicu_base + DMW_PLICU_CH_EDGE_LEVEL1);
	writel(0 | (1 << (DMW_IID_SW3 - 32)) | (1 << (DMW_IID_SW4 - 32)),
	       dmw_plicu_base + DMW_PLICU_CH_EDGE_LEVEL2);

	for (i = 0; i < DMW_PLICU_NR_IRQS; i++) {
		if (is_sw_irq(i)) {
			irq_set_chip_and_handler(i, &sw_chip, handle_edge_irq);
		} else {
			if (irqnr_to_extint(i) != -1)
				irq_set_chip(i, &extint_chip);
			else
				irq_set_chip(i, &plicu_chip);

			irq_set_handler(i, handle_level_irq);
		}
		set_irq_flags(i, IRQF_VALID | IRQF_PROBE);
	}

	/* map GPIO */
	dmw_gpio_base = ioremap_nocache(DMW_GPIO_BASE, SZ_4K);
	if (!dmw_gpio_base)
		panic("Could not map GPIO!");
}

/*
 * Suspend the interrupt controller, leaving only wakeup capable interrupt
 * sources enabled.  Must be called with interrupts disabled.
 */
void dmw_irq_suspend(void)
{
	/* save mask of enabled interrupts */
	irq_enable_save[0] = readl(dmw_plicu_base + DMW_PLICU_CH_MASK1);
	irq_enable_save[1] = readl(dmw_plicu_base + DMW_PLICU_CH_MASK2);

	/* disable everything except wakeup capable sources */
	writel(~irq_wake_mask[0], dmw_plicu_base + DMW_PLICU_CLR_CH_MASK1);
	writel(~irq_wake_mask[1], dmw_plicu_base + DMW_PLICU_CLR_CH_MASK2);
}

void dmw_irq_resume(void)
{
	writel(irq_enable_save[0], dmw_plicu_base + DMW_PLICU_SET_CH_MASK1);
	writel(irq_enable_save[1], dmw_plicu_base + DMW_PLICU_SET_CH_MASK2);
}

int dmw_irq_pending(void)
{
	return readl(dmw_plicu_base + DMW_PLICU_CAUSE1) ||
		readl(dmw_plicu_base + DMW_PLICU_CAUSE2);
}

/*
 * sw irq support
 */

int dmw_trigger_sw_irq(unsigned int irq)
{
	unsigned long sw_ch_req = 0;

	if (irq == DMW_IID_SW1)
		sw_ch_req = 1 << 0;
	else if (irq == DMW_IID_SW2)
		sw_ch_req = 1 << 1;
	else if (irq == DMW_IID_SW3)
		sw_ch_req = 1 << 2;
	else if (irq == DMW_IID_SW4)
		sw_ch_req = 1 << 3;
	else
		return -EINVAL;
	writel(sw_ch_req, dmw_plicu_base + DMW_PLICU_SW_CH_REQ);
	return 0;
}
EXPORT_SYMBOL(dmw_trigger_sw_irq);

/*
 * fiq support
 */

/* this moves a requested irq to fiq and back (depending on type) */
int dmw_set_irq_fiq(unsigned int irq, unsigned int type)
{
	struct pt_regs r;
	unsigned long reg_fiq;
	unsigned long flags;
	unsigned long value;
	int ret = 0;

	if (irq > 31) {
		irq -= 32;
		reg_fiq = DMW_PLICU_FIQ_SEL2;
	} else {
		reg_fiq = DMW_PLICU_FIQ_SEL1;
	}

	local_irq_save(flags);

	if (type) {
		/* update fiq register r10 to contain the plicu base address */
		get_fiq_regs(&r);
		r.ARM_r10 = (u32)dmw_plicu_base;
		set_fiq_regs(&r);

		/* move to fiq */
		value = readl(dmw_plicu_base + reg_fiq);
		value |= (1 << irq);
		writel(value, dmw_plicu_base + reg_fiq);
	} else {
		/* move to irq */
		value = readl(dmw_plicu_base + reg_fiq);
		if (!(value & (1 << irq))) {
			ret = -1;
			goto out;
		}

		value &= ~(1 << irq);
		writel(value, dmw_plicu_base + reg_fiq);
	}

out:
	local_irq_restore(flags);

	return ret;
}
EXPORT_SYMBOL(dmw_set_irq_fiq);
