/*
 * arch/arm/mach-dmw/css/voice.h - voice/RTP character device
 *
 *  Copyright (C) 2007 NXP Semiconductors
 *  Copyright (C) 2008 - 2011 DSP Group Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef __VOICE_H
#define __VOICE_H

#include <linux/ioctl.h>

#ifdef __KERNEL__
#include <linux/in.h>
#else
#include <netinet/in.h>
#endif

#define MAX_NUM_OF_CODEC 6
#define MAX_DYNAMIC_CODEC_LEN 15
#define RTCP_MAX_SDES 50
#define MAX_SDES_VAL_LEN 50	
#define MAX_SDES_ITEMS 8

struct voice_cng_opts {
	char level_rx;
	int  mode_rx;
	char level_tx;
	int  mode_tx;
	int  max_sid_update;
	int  vad_detect_level;
	int  vad_hangover;
};

enum rtp_jib_type {
	RTP_JIB_TYPE_FIXED = 0,
	RTP_JIB_TYPE_ADAPTIVE,
};

struct jib_config {
	int max_len; /* in 10ms units */
	int min_len; /* in 10ms units */
	enum rtp_jib_type type;
};

struct rx_codec_list
{
   char CodecStr[MAX_DYNAMIC_CODEC_LEN];
   char rx_pt;
};

/*!
  \brief  SDP Parameter Change Indicator
*/
typedef enum
{
	VOICE_SDP_PARAM_CODEC, /*!< codec changed */
	VOICE_SDP_PARAM_MODE,  /*!< mode changed */
	VOICE_SDP_PARAM_PTIME, /*!< ptime changed */
	VOICE_SDP_PARAM_JIB /*!< jitter buffer changed */ 
} VOICE_SDP_CHANGE;

/*!
  \brief  Audio enumerations for Call HOLD/UNHOLD
*/
typedef enum
{
	VOICE_MODE_ACTIVE, /*!< audio configuration is Active */
	VOICE_MODE_REC_ONLY,  /*!< audio configuration is only for receive direction  */
	VOICE_MODE_SEND_ONLY, /*!< audio configuration is only for transmit direction */
	VOICE_MODE_INACTIVE /*!< audio configuration is Inactive */ 
} VOICE_MODE_UPDATE;

/*!
  \brief  Audio enumerations for RTP Media Looping
*/
typedef enum
{
	VOICE_LOOP_IP_LEVEL = 1, /*!< Loop at IP Level */
	VOICE_LOOP_DSP_LEVEL, /*!< Loop at DSP Level */
	VOICE_LOOP_RTP_LEVEL /*!< Loop at RTP Level */
} VOICE_MEDIA_LOOP_LEVEL;

struct t38_voice_config
{
  int lsRedundancy; // Level of redundancy of low speed packets
  int hsRedundancy; // Level of redundancy of high speed packets
  int ecnOn; 	     // ECN always on. Parameter ignored.
};

/** struct voice_codec - select a voip codec
 *
 * @rx_codec: rtp payload type expected for received packets
 * @rx_codec_event: rtp payload type for received dtmf events
 * @tx_codec: rtp payload type used to send packets
 * @tx_codec_event: rtp payload type used to send dtmf events
 * @duration: rtp packet size in milliseconds
 * @opts: options for this codec (use jitterbuffer, ...)
 */
struct voice_codec {
	char rx_codec_event;
	char tx_codec;
	char tx_codec_event;
	int duration;
	unsigned int current_time; 
	int opts;
	char CodecStr[MAX_DYNAMIC_CODEC_LEN];
	int Timestamp;
	int ssrc;
	struct voice_cng_opts cng;
	struct jib_config jb;
	struct rx_codec_list rx_list[MAX_NUM_OF_CODEC];
	uint32_t dtmf2833numEndPackets;   
	VOICE_MODE_UPDATE audio_mode;
	VOICE_MEDIA_LOOP_LEVEL media_loop_level;
	struct t38_voice_config t38_cfg;
};

/** struct voice_kernelmode - holds parameters for kernelmode rtp
 *
 * @sock_fd: file descriptor pointing to socket to be used for rtp rx/tx
 * @remote_addr: address of peer
 */
struct voice_kernelmode {
	int sock_fd_rtp;
	int sock_fd_rtcp;
	struct sockaddr_in remote_addr_rtp;
	struct sockaddr_in remote_addr_rtcp;
};

/** struct voice_kernelmode_rtcp - holds parameters for kernelmode rtcp
 *
 * @sock_fd: file descriptor pointing to socket to be used for rtcp rx/tx
 * @remote_addr: address of peer
 */
struct voice_kernelmode_rtcp {
	char sdesItem[MAX_SDES_ITEMS][MAX_SDES_VAL_LEN];
	int sock_fd;
	struct sockaddr_in remote_addr;
	int rtcp_interval;
	int opts;
};

/** struct voice_dtmf_event - holds information about a dtmf event
 *
 * XXX: This struct has to be the same as struct dtmf_event in librtp/rtp.h.
 *
 * @status: status of event
 * @event: type of event
 * @volume: volume of event
 * @duration: duration of event
 */
struct voice_dtmf_event {
	int status;
	char event;
	char volume;
	int duration;
	int EvtDuration;
	int MaxEvtDuration;
};

/* ioctl definitions */
#define VOICE_IOC_MAGIC     'V'

#define VOICE_IOCSETCODEC   _IOW(VOICE_IOC_MAGIC, 0, struct voice_codec *)
#define VOICE_IOCFLUSH      _IO (VOICE_IOC_MAGIC, 1)
#define VOICE_IOCKERNELMODE _IOW(VOICE_IOC_MAGIC, 2, struct voice_kernelmode *)
#define VOICE_IOCSENDDTMF   _IOW(VOICE_IOC_MAGIC, 3, struct voice_dtmf_event *)
#define VOICE_IOCGETDTMF    _IOR(VOICE_IOC_MAGIC, 4, struct voice_dtmf_event *)
#define VOICE_IOCRTCP       _IOW(VOICE_IOC_MAGIC, 5, struct voice_kernelmode_rtcp *)
#define VOICE_IOCUPDATE_CHAN  _IOW(VOICE_IOC_MAGIC, 6, struct voice_codec *)

#define VOICE_IOC_MAXNR     6

#endif
