/**
 *  css.c
 *
 *  Copyright (C) 2011 DSP Group
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/io.h>
#include <linux/err.h>
#include <linux/clk.h>
#include <linux/firmware.h>
#include <linux/slab.h>
#include <linux/delay.h>
#include <linux/platform_device.h>
#include <linux/sysfs.h>
#include <linux/dma-mapping.h>
#include <linux/stat.h>
#include <linux/debugfs.h>
#include <linux/uaccess.h>
#include <linux/vmalloc.h>
#include <linux/workqueue.h>

#include <mach/hardware.h>
#include <mach/platform.h>
#include <mach/css.h>
#include <mach/sec.h>
#include <mach/powerdomain.h>

#include "coma.h"
#include "../pm.h"

MODULE_AUTHOR("DSP Group Inc.");
MODULE_LICENSE("GPL");

#define CSS_FIRMWARE_NA				0
#define CSS_FIRMWARE_PENDING			1
#define CSS_FIRMWARE_LOADED			2
#define CSS_FIRMWARE_PANIC			3

static struct loader_private *p;

static void
coma_device_release(struct device *dev)
{
}

static struct platform_device coma_device = {
	.name	= "coma",
	.id	= 0,
	.dev	= {
		.release = coma_device_release,
	}
};

struct memory {
	void __iomem *data;
	unsigned int size;
	struct resource *res;
};

struct phys_memory {
	void *data;
	unsigned int size;
	dma_addr_t phys;
};

struct loader_private {
	struct memory itcm;
	struct memory ahbram;
	struct phys_memory dram;
	struct phys_memory dc;
	struct phys_memory cfifo;
#ifdef CONFIG_DEBUG_FS
	struct debugfs_blob_wrapper dram_blob;
#endif
	struct memory dtcm;
	struct clk *clk_css;
	struct clk *clk_css_fast;
	struct clk *clk_css_slow;
	struct clk *clk_css_arm;
	struct clk *clk_css_etm;
	struct css_pdata *pdata;
	struct device *dev;
	struct dentry *pdentry;
	struct sec_msg *panic;
	struct sec_msg *pm_dram;
	struct sec_msg *pm_pdwn;
	int state;
	int dram_requested;
};

#ifdef CONFIG_DEBUG_FS
static int css_debug_open(struct inode *inode, struct file *f)
{
	if (p->state != CSS_FIRMWARE_LOADED &&
	    p->state != CSS_FIRMWARE_PANIC)
		return -EIO;

	f->private_data = inode->i_private;

	return 0;
}

static int css_debug_cfifo_open(struct inode *inode, struct file *f)
{
	struct list_head *p_cfifo_list;
	cfifo_node_t *pnode;
	size_t data_size;

	if (p->state != CSS_FIRMWARE_LOADED &&
	    p->state != CSS_FIRMWARE_PANIC)
		return -EIO;

	p_cfifo_list = get_cfifo_list_head();

	data_size = 0;
	list_for_each_entry(pnode,p_cfifo_list,list){
		printk(KERN_ERR "cfifo buffer = %x, cfifo size = %d \n", (unsigned int)pnode->pbuf, (unsigned int)pnode->size);
		data_size+=pnode->size;
		
	}

	f->private_data = (void*)data_size;

	return 0;
}


static int css_debug_itcm_open(struct inode *inode, struct file *f)
{
	if (p->state != CSS_FIRMWARE_PANIC) {
		dev_err(p->dev, "only allowed in panic mode\n");
		return -EIO;
	}

	f->private_data = inode->i_private;

	clk_disable(p->clk_css_arm);
	p->pdata->reset_arm(1);
	clk_enable(p->clk_css_arm);
	mdelay(2);

	return 0;
}

static ssize_t css_debug_read(struct file *f, char __user *u, size_t size, loff_t *off)
{
	void *buf;
	struct memory *m = f->private_data;

	if (*off >= m->size)
		return 0;
	if (*off + size > m->size)
		size = m->size - *off;

	buf = vmalloc(size);
	if (!buf)
		return -ENOMEM;

	memcpy_fromio(buf, m->data + *off, size);

	if (copy_to_user(u, buf, size)) {
		vfree(buf);
		return -EFAULT;
	}

	*off += size;
	vfree(buf);
	return size;
}


static ssize_t css_debug_cfifo_read(struct file *f, char __user *u, size_t size, loff_t *off)
{
	void *buf;
	struct list_head *p_cfifo_list;
	cfifo_node_t *pnode;
	size_t data_size;
	unsigned int offset;

	data_size =(size_t) f->private_data;
	buf = vmalloc(data_size);
	if (!buf)
		return -ENOMEM;

	p_cfifo_list = get_cfifo_list_head();
	offset = 0;
	list_for_each_entry(pnode,p_cfifo_list,list){
		memcpy(buf+offset, pnode->pbuf, pnode->size);
		offset+=pnode->size;
	}
	
	if (*off >= data_size)
		return 0;
	if (*off + size > data_size)
		size = data_size - *off;

	if (copy_to_user(u, buf + *off, size)) {
		vfree(buf);
		return -EFAULT;
	}
	
	*off += size;
	vfree(buf);
	return size;
}



static struct file_operations debug_fops = {
	.open		= css_debug_open,
	.read		= css_debug_read,
};

static struct file_operations debug_itcm_fops = {
	.open		= css_debug_itcm_open,
	.read		= css_debug_read,
};


static struct file_operations debug_cfifo_fops = {
	.open		= css_debug_cfifo_open,
	.read		= css_debug_cfifo_read,
};


static void css_create_debugfs_files(void)
{
	struct dentry *pdentry;

	pdentry = debugfs_create_dir("css", NULL);

	debugfs_create_file("ahb-ram",
	                    S_IRUSR,
	                    pdentry,
	                    &p->ahbram,
	                    &debug_fops);
	debugfs_create_file("itcm",
	                    S_IRUSR,
	                    pdentry,
	                    &p->itcm,
	                    &debug_itcm_fops);
	debugfs_create_file("dtcm",
	                    S_IRUSR,
	                    pdentry,
	                    &p->dtcm,
	                    &debug_fops);

	p->dram_blob.data = p->dram.data;
	p->dram_blob.size = p->dram.size;

	debugfs_create_blob("dram",
	                    S_IRUSR,
	                    pdentry,
	                    &p->dram_blob);


	debugfs_create_file("cfifo",
	                    S_IRUSR,
	                    pdentry,
	                    NULL,
	                    &debug_cfifo_fops);
	
	p->pdentry = pdentry;
}

static void css_remove_debugfs_files(void)
{
	debugfs_remove_recursive(p->pdentry);
}
#else
#define css_create_debugfs_files()
#define css_remove_debugfs_files()
#endif

static void got_firmware(const struct firmware *fw, void *context)
{
	struct css_img_hdr *img_hdr;
	struct css_part_hdr *itcm_hdr;
	struct css_part_hdr *ahbram_hdr;
	struct css_part_hdr *dram_hdr;
	struct css_part_hdr *elf_hdr = NULL;
	uint32_t *virt_dc;
	void __iomem *virt_elf_md5;
	void __iomem *virt_boot_config;
	unsigned long csize = 0;
	int ret = 0;
	unsigned long expected_version;
	struct loader_private *p = (struct loader_private *)context;
	struct coma_config *dc;
	dma_addr_t dc_phys;
	dma_addr_t cfifo_phys;
	int cfifo_size;
	struct boot_config bootcfg;

	if (!fw) {
		p->state = CSS_FIRMWARE_NA;
		dev_err(p->dev, "Loading CSS firmware failed\n");
		return;
	}

	dmw_pwrdom_enable(DMW_SYSCFG_POWER_EN_CSS);

	p->pdata->reset_css(0);
	clk_enable(p->clk_css);
	mdelay(2);

	p->pdata->reset_etm(0);
	clk_enable(p->clk_css_etm);
	mdelay(2);

	clk_enable(p->clk_css_arm);

	/* clear memory */
	memset_io(p->itcm.data, 0, p->itcm.size);
	memset_io(p->ahbram.data, 0, p->ahbram.size);

	/* check image header */
	img_hdr = (struct css_img_hdr *)fw->data;

	if (img_hdr->parts < 3) {
		dev_err(p->dev, "invalid number of parts: %lu\n",
		       img_hdr->parts);
		ret = -EFAULT;
		goto out_err;
	}

	if (img_hdr->magic != CSS_MAGIC) {
		dev_err(p->dev, "invalid image magic\n");
		ret = -EFAULT;
		goto out_err;
	}

	expected_version = CSS_IMG_VERSION;
	if (img_hdr->img_version != expected_version) {
		dev_err(p->dev, "image version mismatch (got: v%lu, "
		                "kernel expected: v%lu)\n",
		                img_hdr->img_version, expected_version);
		ret = -EFAULT;
		goto out_err;
	}
	dev_info(p->dev, "loading %s\n", img_hdr->info);

	csize = sizeof(*img_hdr);
	itcm_hdr   = (struct css_part_hdr *)(fw->data + csize);
	csize += sizeof(*itcm_hdr) + itcm_hdr->size;
	if ((csize > fw->size) || (itcm_hdr->size > p->itcm.size)) {
		dev_err(p->dev, "ITCM size too big! sz=0x%08lX, max=0x%08X\n",
		        itcm_hdr->size, p->itcm.size);
		ret = -EFAULT;
		goto out_err;
	}

	ahbram_hdr = (struct css_part_hdr *)(fw->data + csize);
	csize += sizeof(*ahbram_hdr) + ahbram_hdr->size;
	if ((csize > fw->size) || (ahbram_hdr->size > p->ahbram.size)) {
		dev_err(p->dev, "AHBRAM size too big! sz=0x%08lX, max=0x%08X\n",
		        ahbram_hdr->size, p->ahbram.size);
		ret = -EFAULT;
		goto out_err;
	}

	dram_hdr   = (struct css_part_hdr *)(fw->data + csize);
	csize += sizeof(*dram_hdr) + dram_hdr->size;
	if ((csize > fw->size) || (dram_hdr->size > dram_hdr->rsize)) {
		dev_err(p->dev, "DRAM size too big! sz=0x%08lX, max=0x%08X\n",
		        dram_hdr->size, p->dram.size);
		ret = -EFAULT;
		goto out_err;
	}

	if (img_hdr->parts == 4) {
		elf_hdr = (struct css_part_hdr *)(fw->data + csize);
		csize += sizeof(*dram_hdr) + elf_hdr->size;
		if (csize > fw->size) {
			dev_err(p->dev, "ELF size too big! sz=0x%08lX\n",
			       elf_hdr->size);
			ret = -EFAULT;
			goto out_err;
		}
		dev_info(p->dev,
		         "MD5 sum of ELF file: 0x%2.2X%2.2X%2.2X%2.2X%2.2X%2.2X%2.2X%2.2X%2.2X%2.2X%2.2X%2.2X%2.2X%2.2X%2.2X%2.2X",
		         elf_hdr->md5[0],
		         elf_hdr->md5[1],
		         elf_hdr->md5[2],
		         elf_hdr->md5[3],
		         elf_hdr->md5[4],
		         elf_hdr->md5[5],
		         elf_hdr->md5[6],
		         elf_hdr->md5[7],
		         elf_hdr->md5[8],
		         elf_hdr->md5[9],
		         elf_hdr->md5[10],
		         elf_hdr->md5[11],
		         elf_hdr->md5[12],
		         elf_hdr->md5[13],
		         elf_hdr->md5[14],
		         elf_hdr->md5[15]);
	}

	if ((itcm_hdr->dest_addr < p->itcm.res->start) ||
	    (itcm_hdr->dest_addr > (p->itcm.res->start + p->itcm.size)) ||
	    ((itcm_hdr->dest_addr + itcm_hdr->size) > (p->itcm.res->start + p->itcm.size))) {
		dev_err(p->dev, "ITCM destination pointer out of range! ptr=0x%08lX\n",
		       itcm_hdr->dest_addr);
		ret = -EFAULT;
		goto out_err;
	}

	if ((ahbram_hdr->dest_addr < p->ahbram.res->start) ||
	    (ahbram_hdr->dest_addr > (p->ahbram.res->start + p->ahbram.size)) ||
	    ((ahbram_hdr->dest_addr + ahbram_hdr->size) > (p->ahbram.res->start + p->ahbram.size))) {
		dev_err(p->dev, "AHBRAM destination pointer out of range! ptr=0x%08lX\n",
		       ahbram_hdr->dest_addr);
		ret = -EFAULT;
		goto out_err;
	}

	if ((img_hdr->dc_addr > dram_hdr->size) ||
	    ((img_hdr->dc_addr + 4) > dram_hdr->size)) {
		dev_err(p->dev, "DC pointer out of range! ptr=0x%08lX\n",
		       img_hdr->dc_addr);
		ret = -EFAULT;
		goto out_err;
	}

	if ((img_hdr->elf_md5_addr > dram_hdr->size) ||
	    ((img_hdr->elf_md5_addr + 4) > dram_hdr->size)) {
		dev_err(p->dev, "Elf MD5 pointer out of range! ptr=0x%08lX\n",
		       img_hdr->elf_md5_addr);
		ret = -EFAULT;
		goto out_err;
	}

	p->dram.data = dma_alloc_coherent(p->dev,
	                                  dram_hdr->rsize,
	                                  &p->dram.phys,
	                                  GFP_KERNEL);
	if (p->dram.data == NULL) {
		dev_err(p->dev, "not enough DRAM memory for CSS: %lu\n",
		        dram_hdr->rsize);
		ret = -ENOMEM;
		goto out_err;
	}
	p->dram.size = dram_hdr->rsize;
	/* clear memory */
	memset(p->dram.data, 0, p->dram.size);

	dc = (struct coma_config *)dma_alloc_coherent(p->dev,
	                                              sizeof(*dc),
	                                              &dc_phys,
	                                              GFP_KERNEL);

	if (dc == NULL) {
		ret = -ENOMEM;
		goto out_free_dram;
	}

	cfifo_size = cfifo_alloc(1024, &dc->fifo_c2l, 1024, &dc->fifo_l2c, &cfifo_phys);
	if (cfifo_size < 0) {
		ret = -ENOMEM;
		goto out_free_dc;
	}

	dc->fifo_c2l_phys = (struct cfifo *)dc->fifo_c2l->self_phys;
	dc->fifo_l2c_phys = (struct cfifo *)dc->fifo_l2c->self_phys;

	p->dc.data = (void *)dc;
	p->dc.size = sizeof(*dc);
	p->dc.phys = dc_phys;

	p->cfifo.data = (void *)dc->fifo_c2l;
	p->cfifo.size = cfifo_size;
	p->cfifo.phys = cfifo_phys;

	/* copy CSS code and data */
	memcpy_toio(p->itcm.data + (itcm_hdr->dest_addr - p->itcm.res->start),
	            (void *)itcm_hdr + sizeof(*itcm_hdr),
	            itcm_hdr->size);

	memcpy_toio(p->ahbram.data + (ahbram_hdr->dest_addr - p->ahbram.res->start),
	            (void *)ahbram_hdr + sizeof(*ahbram_hdr),
	            ahbram_hdr->size);

	memcpy(p->dram.data + dram_hdr->dest_addr,
	       (void *)dram_hdr + sizeof(*dram_hdr),
	       dram_hdr->size);

	/* fill dc pointer of CSS */
	virt_dc = (uint32_t *)(p->dram.data + img_hdr->dc_addr);
	*virt_dc = dc_phys;

	if (img_hdr->parts == 4) {
		/* fill elf md5 sum of CSS */
		virt_elf_md5 = p->dram.data + img_hdr->elf_md5_addr;
		memcpy(virt_elf_md5, elf_hdr->md5, 16);
	}

	/* fill the boot config structure for the CSS */
	bootcfg.dram_addr = (uint32_t)p->dram.phys;
	bootcfg.dram_size = p->dram.size;
	bootcfg.cmu_sem = (struct ram_sem *)DMW_SRAM_CMU_SEM_BASE;
	virt_boot_config = p->itcm.data + img_hdr->boot_config;
	memcpy_toio(virt_boot_config, (void *)&bootcfg, sizeof(bootcfg));

	css_create_debugfs_files();

	/* make sure the DRAM is not switched off while the CSS is booting */
	dmw_dram_request();
	p->dram_requested = 1;

	clk_disable(p->clk_css_arm);
	p->pdata->reset_arm(0);
	clk_enable(p->clk_css_arm);
	mdelay(2);

	dev_info(p->dev, "successfully loaded CSS firmware\n");

	/* register COMA platform device */
	memset(&coma_device, 0, sizeof(coma_device));
	coma_device.name = "coma";
	coma_device.id = 0;
	coma_device.dev.release = coma_device_release;
	coma_device.dev.platform_data = (void *)dc;

	ret = platform_device_register(&coma_device);
	if (ret < 0)
		goto out_free;

	p->state = CSS_FIRMWARE_LOADED;

	goto out;
out_free:

	clk_disable(p->clk_css_arm);
	p->pdata->reset_arm(1);
	clk_enable(p->clk_css_arm);
	mdelay(2);
	clk_disable(p->clk_css_arm);

	cfifo_free(p->cfifo.data, p->cfifo.size, p->cfifo.phys);
out_free_dc:
	dma_free_coherent(p->dev, sizeof(*dc), dc, dc_phys);
out_free_dram:
	dma_free_coherent(p->dev, p->dram.size, p->dram.data, p->dram.phys);
out_err:
	dev_err(p->dev, "failed to load CSS firmware\n");

	clk_disable(p->clk_css_etm);
	p->pdata->reset_etm(1);
	clk_enable(p->clk_css_etm);
	mdelay(2);
	clk_disable(p->clk_css_etm);

	clk_disable(p->clk_css);
	p->pdata->reset_css(1);
	clk_enable(p->clk_css);
	mdelay(2);
	clk_disable(p->clk_css);


	p->state = CSS_FIRMWARE_NA;
out:
	release_firmware(fw);
	return;
}

static ssize_t set_load(struct device *dev, struct device_attribute *attr, const char *buf, size_t count)
{
	int ret = 0;
	unsigned int value = simple_strtoul(buf, NULL, 0);

	if ((value >= 1) && (p->state != CSS_FIRMWARE_PENDING)) {
		if (p->state == CSS_FIRMWARE_LOADED ||
		    p->state == CSS_FIRMWARE_PANIC) {

			css_remove_debugfs_files();

			clk_disable(p->clk_css_etm);
			p->pdata->reset_etm(1);
			clk_enable(p->clk_css_etm);
			mdelay(2);
			clk_disable(p->clk_css_etm);

			clk_disable(p->clk_css_arm);
			p->pdata->reset_arm(1);
			clk_enable(p->clk_css_arm);
			mdelay(2);
			clk_disable(p->clk_css_arm);

			clk_disable(p->clk_css);
			p->pdata->reset_css(1);
			clk_enable(p->clk_css);
			mdelay(2);
			clk_disable(p->clk_css);

			platform_device_unregister(&coma_device);

			cfifo_free(p->cfifo.data, p->cfifo.size, p->cfifo.phys);
			dma_free_coherent(p->dev, p->dc.size, p->dc.data, p->dc.phys);
			dma_free_coherent(p->dev, p->dram.size, p->dram.data, p->dram.phys);
		}

		p->state = CSS_FIRMWARE_PENDING;

		ret = request_firmware_nowait(THIS_MODULE,
		                              1,
		                              "css-loader",
		                              dev,
		                              GFP_KERNEL,
		                              (void *)p,
		                              got_firmware);
	}

	if (ret != 0)
		return 0;

	return count;
}

static ssize_t get_state(struct device *dev, struct device_attribute *attr, char *buf)
{
	return sprintf(buf, "%d\n", p->state);
}

static void
css_panic_work(struct work_struct *work)
{
	kobject_uevent(&p->dev->kobj, KOBJ_CHANGE);
}

DECLARE_WORK(css_panic, css_panic_work);

static void css_panic_handler(int id, unsigned long data, void *context)
{
	struct loader_private *p = context;

	dev_crit(p->dev, "CSS entered panic mode\n");
	p->state = CSS_FIRMWARE_PANIC;
	schedule_work(&css_panic);
}

static void css_pm_dram_handler(int id, unsigned long data, void *context)
{
	struct loader_private *p = context;

	/*
	 * From the deepest sleep state we need approx. 250us to be cack to
	 * C-code. With some slack time the threshold is somewhere around
	 * 300us.
	 */
	if (data < 300 && !p->dram_requested) {
		dmw_dram_request();
		p->dram_requested = 1;
	} else if (data >= 300 && p->dram_requested) {
		dmw_dram_release();
		p->dram_requested = 0;
	}
}

static void css_pm_pwdn_Handler(int id, unsigned long data, void *context)
{
	struct loader_private *p = context;

	/*
	 * From the deepest sleep state we need approx. 250us to be cack to
	 * C-code. Enabling PLL4 might also take 250us. With some slack time
	 * the threshold is somewhere around 550us.
	 */
	if (data < 550)
		clk_set_parent(p->clk_css, p->clk_css_fast);
	else
		clk_set_parent(p->clk_css, p->clk_css_slow);
}

static struct device_attribute css_attrs[] = {
	__ATTR(load, S_IWUSR, NULL, set_load),
	__ATTR(state, S_IRUSR, get_state, NULL),
};

static int
css_probe(struct platform_device *pdev)
{
	int ret;
	int i;
	struct resource *ahbram_res;
	struct resource *itcm_res;
	struct resource *dtcm_res;

	if (!pdev->dev.platform_data) {
		dev_err(&pdev->dev, "Missing platform data!\n");
		return -EINVAL;
	}

	ahbram_res = platform_get_resource(pdev, IORESOURCE_MEM, CSS_AHBRAM);
	if (!ahbram_res)
		return -EINVAL;

	itcm_res = platform_get_resource(pdev, IORESOURCE_MEM, CSS_ITCM);
	if (!itcm_res)
		return -EINVAL;

	dtcm_res = platform_get_resource(pdev, IORESOURCE_MEM, CSS_DTCM);
	if (!dtcm_res)
		return -EINVAL;

	if (!request_mem_region(ahbram_res->start,
	                        ahbram_res->end - ahbram_res->start + 1,
	                        pdev->name)) {
		dev_err(&pdev->dev, "failed to request CSS AHBRAM memory resource\n");
		return -EBUSY;
	}

	if (!request_mem_region(itcm_res->start,
	                        itcm_res->end - itcm_res->start + 1,
	                        pdev->name)) {
		dev_err(&pdev->dev, "failed to request CSS ITCM memory resource\n");
		ret = -EBUSY;
		goto out_release_ahbram_res;
	}

	if (!request_mem_region(dtcm_res->start,
	                        dtcm_res->end - dtcm_res->start + 1,
	                        pdev->name)) {
		dev_err(&pdev->dev, "failed to request CSS DTCM memory resource\n");
		ret = -EBUSY;
		goto out_release_itcm_res;
	}

	p = kzalloc(sizeof(*p), GFP_KERNEL);
	if (!p) {
		ret = -ENOMEM;
		goto out_release_dtcm_res;
	}

	p->ahbram.res = ahbram_res;
	p->itcm.res = itcm_res;
	p->dtcm.res = dtcm_res;

	p->ahbram.size = ahbram_res->end - ahbram_res->start + 1;
	p->itcm.size = itcm_res->end - itcm_res->start + 1;
	p->dtcm.size = dtcm_res->end - dtcm_res->start + 1;

	p->pdata = (struct css_pdata *)pdev->dev.platform_data;
	p->state = CSS_FIRMWARE_NA;

	p->ahbram.data = ioremap_nocache(ahbram_res->start, p->ahbram.size);
	if (!p->ahbram.data) {
		dev_err(&pdev->dev, "failed to map CSS AHBRAM\n");
		ret = -ENOMEM;
		goto out_free_mem;
	}

	p->itcm.data = ioremap_nocache(itcm_res->start, p->itcm.size);
	if (!p->itcm.data) {
		dev_err(&pdev->dev, "failed to map CSS ITCM\n");
		ret = -ENOMEM;
		goto out_unmap_ahbram;
	}

	p->dtcm.data = ioremap_nocache(dtcm_res->start, p->dtcm.size);
	if (!p->dtcm.data) {
		dev_err(&pdev->dev, "failed to map CSS DTCM\n");
		ret = -ENOMEM;
		goto out_unmap_itcm;
	}

	p->clk_css = clk_get(&pdev->dev, "css");
	p->clk_css_fast = clk_get(&pdev->dev, "fast");
	p->clk_css_slow = clk_get(&pdev->dev, "slow");
	p->clk_css_arm = clk_get(&pdev->dev, "arm");
	p->clk_css_etm = clk_get(&pdev->dev, "etm");
	p->dev = &pdev->dev;

	for (i = 0; i < ARRAY_SIZE(css_attrs); i++) {
		ret = device_create_file(&pdev->dev, &css_attrs[i]);
		if (ret) {
			dev_err(&pdev->dev, "failed to create css sysfs file\n");
			goto out_unmap_dtcm;
		}
	}

	p->panic = sec_msg_register(css_panic_handler, SEC_MSG_CSS_PANIC, 0, p);
	if (IS_ERR_OR_NULL(p->panic))
		dev_err(&pdev->dev, "failed to register CSS panic handler\n");

	p->pm_dram = sec_msg_register(css_pm_dram_handler, SEC_MSG_DRAM,
				      SEC_FIQ, p);
	if (IS_ERR_OR_NULL(p->pm_dram))
		dev_err(&pdev->dev, "failed to register CSS DRAM req handler\n");

	p->pm_pdwn = sec_msg_register(css_pm_pwdn_Handler, SEC_MSG_PWR_DOWN,
				      SEC_FIQ, p);
	if (IS_ERR_OR_NULL(p->pm_pdwn))
		dev_err(&pdev->dev, "failed to register CSS power down handler\n");

	ret = 0;
	goto out;

out_unmap_dtcm:
	iounmap(p->dtcm.data);
out_unmap_itcm:
	iounmap(p->itcm.data);
out_unmap_ahbram:
	iounmap(p->ahbram.data);
out_release_dtcm_res:
	release_mem_region(dtcm_res->start, dtcm_res->end - dtcm_res->start + 1);
out_release_itcm_res:
	release_mem_region(itcm_res->start, itcm_res->end - itcm_res->start + 1);
out_release_ahbram_res:
	release_mem_region(ahbram_res->start, ahbram_res->end - ahbram_res->start + 1);
out_free_mem:
	kfree(p);
out:
	return ret;
}


static int
css_remove(struct platform_device *pdev)
{
	int i;

	sec_msg_deregister(p->panic);

	css_remove_debugfs_files();

	for (i = 0; i < ARRAY_SIZE(css_attrs); i++)
		device_remove_file(&pdev->dev, &css_attrs[i]);

	if (p->state == CSS_FIRMWARE_LOADED) {
		clk_disable(p->clk_css);
		clk_disable(p->clk_css_arm);
		clk_disable(p->clk_css_etm);

		platform_device_unregister(&coma_device);

		cfifo_free(p->cfifo.data, p->cfifo.size, p->cfifo.phys);
		dma_free_coherent(p->dev, p->dc.size, p->dc.data, p->dc.phys);
		dma_free_coherent(p->dev, p->dram.size, p->dram.data, p->dram.phys);
	}

	clk_put(p->clk_css);
	clk_put(p->clk_css_arm);
	clk_put(p->clk_css_etm);
	p->pdata->reset_css(1);
	p->pdata->reset_arm(1);
	p->pdata->reset_etm(1);

	iounmap(p->dtcm.data);
	iounmap(p->itcm.data);
	iounmap(p->ahbram.data);
	release_mem_region(p->dtcm.res->start,
	                   p->dtcm.res->end - p->dtcm.res->start + 1);
	release_mem_region(p->itcm.res->start,
	                   p->itcm.res->end - p->itcm.res->start + 1);
	release_mem_region(p->ahbram.res->start,
	                   p->ahbram.res->end - p->ahbram.res->start + 1);

	kfree(p);

	return 0;
}

static struct platform_driver css_driver = {
	.driver = {
		.name = "css",
		.owner = THIS_MODULE,
	},
	.probe = css_probe,
	.remove = css_remove,
};

static int __init css_init(void)
{
	return platform_driver_register(&css_driver);
}
module_init(css_init);

static void __exit css_exit(void)
{
	platform_driver_unregister(&css_driver);
}
module_exit(css_exit);
