/*
 *  arch/arm/mach-dmw/css/coma-voice.c - voice/RTP character device
 *
 *  This driver registers one character device for each available voice channel.
 *  It allows only one reader and one writer at the same time.
 *
 *  Copyright (C) 2007 NXP Semiconductors
 *  Copyright (C) 2008 - 2011 DSP Group Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/* TODO:
 *  - if minor numbers dont start at zero, this code will fail
 *  - read/write several rtp packets at once
 *  - merge the two receiver threads (rtp & rtcp)
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/err.h>
#include <linux/cdev.h>
#include <linux/uio.h>
#include <linux/poll.h>
#include <linux/fs.h>
#include <linux/wait.h>
#include <linux/mutex.h>
#include <linux/kthread.h>
#include <linux/net.h>
#include <linux/file.h>
#include <linux/completion.h>
#include <asm/uaccess.h>
#include <linux/delay.h>
#include <asm/uaccess.h>
//#include <cordless/voice.h>

#include "coma.h"
#include "coma-service.h"
#include "cmsg.h"
#include "cfifo.h"
#include "voice.h"
#include "cmsg-voice.h"
#include "coma-voice.h"

/*
   Note:
   Currently we get the local IP address of the eth0 interface,
   If some customer is having more than one eth interface and using eth1 for RTP
   then he has to change the  static int ethernet_id = 0; to static int ethernet_id = 1;
   Or he can always send the dummy RTP/RTCP packet to loop back address
   insted of local IP,some customer who disabled loop back can send to local ip
   
*/

/* Ramu */
#if 1				/*  For getting local  IP Address */

#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/inetdevice.h>
#include <linux/netdevice.h>

#include <asm/types.h>

static int ethernet_id = 0;
module_param(ethernet_id, int, 0400);
MODULE_PARM_DESC(ethernet_id, "Ethernet device to display");

#endif
/* Ramu  */

DEFINE_MUTEX(coma_msg_mutex);

#define MAX_LEN_TEMPBUFF 172
unsigned char Tempbuf[MAX_LEN_TEMPBUFF];

MODULE_AUTHOR("DSP Group Inc.");
MODULE_LICENSE("GPL");

//extern  struct mutex coma_msg_mutex;
MODULE_AUTHOR("DSP Group Inc.");
MODULE_LICENSE("GPL");

#define SERVICE_NAME          "coma-voice"
#define SERVICE_VOICE         COMA_SERVICE_VOICE
//#define VOICE_MAJOR           253
#define FIFO_SIZE             4096
#define VOICE_CODEC_G711_ALAW 8
#define VOICE_CODEC_G722      9
#define DEFAULT_CODEC         VOICE_CODEC_G722
#define DEFAULT_DURATION      20

#define RTCP_PT_SR            200
#define RTCP_PT_RR            201
#define RTCP_PT_BYE           203

#define CFIFO_RTP_PACKET 1
#define CFIFO_RTCP_PACKET 2

static char voice_name[] = "coma-voice";

dev_t voice_dev;
static struct cdev voice_cdev;

static struct cfifo *voice_c2l;
static struct cfifo *voice_l2c;
static int cfifo_size;
static dma_addr_t cfifo_phys;

int voice_reg;

const unsigned int voice_numlines = CONFIG_CORDLESS_NUM_VOICELINES;

static DECLARE_COMPLETION(voice_reply_compl);
static DEFINE_MUTEX(voice_mutex);

static union cmsg_voice_params last_cmsg_voice_params;

static struct task_struct *kmode_thread;

enum voice_chan_status {
	CHANNEL_INVALID = 0,
	CHANNEL_INITIALIZED,
	CHANNEL_STARTED,
};

struct voice_kmode_rtcp {
	struct socket *sock;
	struct sockaddr_in remote_addr;
	struct task_struct *receiver;
	struct completion receiver_exit;
	struct completion dummyRTCP_exit;
	struct mutex dec_fifo_mutex;
	int rtcp_stop_flag;
};

/**
 * struct voice_kmode - all information for kernelmode of one audio channel
 *
 * @sock:        socket to be used for rtp session
 * @remote_addr: address of peer
 * @receiver:    receiver kthread (receive rtp data from peer)
 * @sender:      sender kthread (send rtp data to peer)
 *
 */
struct voice_kmode {
	struct socket *sock;
	struct sockaddr_in remote_addr;
	struct task_struct *receiver;
	struct task_struct *sender;
	struct completion receiver_exit;
	struct completion dummyRTP_exit;
	struct voice_kmode_rtcp *rtcp;
	int rtp_stop_flag;
};

/**
 * struct voice_chan - all the info needed to run one audio channel
 *
 * @enc_fifo:  pointer to cfifo used for the encoder
 * @dec_fifo:  pointer to cfifo used for the decoder
 * @reader:    pointer to the reading process
 * @writer:    pointer to the writing process
 *
 */
struct voice_chan {
	struct cfifo *enc_fifo;
	struct cfifo *dec_fifo;
	int cfifo_size;
	dma_addr_t cfifo_phys;
	struct file *reader;
	struct file *writer;
	wait_queue_head_t enc_wq;
	enum voice_chan_status status;
	int id;
	struct voice_codec codec;
	struct mutex codec_mutex;
	int flushed;
	struct socket *sock;
	struct sockaddr_in remote_addr;
	/* dtmf handling */
	struct voice_dtmf_event *dtmf_event;
	struct mutex dtmf_event_mutex;
	wait_queue_head_t dtmf_wq;
};

static struct voice_chan voice_channels[CONFIG_CORDLESS_NUM_VOICELINES];

static int voice_registered(void)
{
	return (voice_reg != 0);
}

static int voice_setup(void)
{
	voice_reg = 1;
	return 0;
}

static void voice_remove(void)
{
	voice_reg = 0;
}

static void voice_generic_reply(union cmsg_voice_params *params)
{
	last_cmsg_voice_params = *params;
	complete(&voice_reply_compl);
}

/*
 * voice_chan specific function
 */

static void voice_chan_reset(struct voice_chan *chan)
{
	mutex_lock(&voice_mutex);
	cfifo_reset(chan->enc_fifo);
	cfifo_reset(chan->dec_fifo);
	chan->reader = 0;
	chan->writer = 0;
	chan->status = CHANNEL_INVALID;
	chan->sock = NULL;
	mutex_unlock(&voice_mutex);
}

/*
 * coma handling
 */

int voice_process_message(struct cmsg *cmsg)
{
	int ret = 0;
	union cmsg_voice_params *params =
	    (union cmsg_voice_params *)cmsg->params;

	switch (cmsg->type) {
	case CMSG_VOICE_REPLY_GET_CHAN:
	case CMSG_VOICE_REPLY_SET_CHAN_FIFOS:
	case CMSG_VOICE_REPLY_START_CHAN:
	case CMSG_VOICE_REPLY_STOP_CHAN:
	case CMSG_VOICE_REPLY_FREE_CHAN:
	case CMSG_VOICE_REPLY_START_RTCP:
	case CMSG_VOICE_REPLY_STOP_RTCP:
	case CMSG_VOICE_REPLY_SEND_DTMF:
	case CMSG_VOICE_REPLY_UPDATE_CHAN:
		voice_generic_reply(params);
		break;
#if 0
	case CMSG_REPLY_REPORT_RTCP:
		rtcp_buf = cmsg->payload;
		rtcp_len = cmsg->payload_size;
		coma_generic_reply(params);
		break;

	case CMSG_VOICE_RECEIVE_DTMF:
		voice_receive_dtmf(params->receive_dtmf.id,
				   params->receive_dtmf.status,
				   params->receive_dtmf.event,
				   params->receive_dtmf.volume,
				   params->receive_dtmf.duration);
		break;
#endif

	default:
		ret = -1;
		break;
	}

	return ret;
}

static int
coma_create_voice_message(enum cmsg_voice_types type,
			  union cmsg_voice_params *params,
			  void *payload, unsigned int payload_size)
{
	return coma_create_message(SERVICE_VOICE, (int)type,
				   (void *)params, sizeof(*params),
				   payload, payload_size);
}

static int voice_request_get_chan(int id)
{
	int ret;
	union cmsg_voice_params params;

	if (!voice_registered())
		return -EFAULT;

	params.request_get_chan.id = id;
	ret =
	    coma_create_voice_message(CMSG_VOICE_REQUEST_GET_CHAN, &params,
				      NULL, 0);
	if (ret == 0)
		coma_signal(SERVICE_VOICE);

	return ret;
}

static int voice_get_chan(struct voice_chan *chan)
{
	int ret;

	mutex_lock(&voice_mutex);

	ret = voice_request_get_chan(chan->id);
	if (ret < 0) {
		mutex_unlock(&voice_mutex);
		return -EFAULT;
	}

	wait_for_completion(&voice_reply_compl);

	ret = last_cmsg_voice_params.reply_get_chan.result;
	mutex_unlock(&voice_mutex);

	return ret;

}

static int
voice_request_set_chan_fifos(int id, struct cfifo *enc, struct cfifo *dec)
{
	int ret;
	union cmsg_voice_params params;

	if (!voice_registered())
		return -EFAULT;

	params.request_set_chan_fifos.id = id;
	params.request_set_chan_fifos.enc = (struct cfifo *)enc->self_phys;
	params.request_set_chan_fifos.dec = (struct cfifo *)dec->self_phys;

	ret =
	    coma_create_voice_message(CMSG_VOICE_REQUEST_SET_CHAN_FIFOS,
				      &params, NULL, 0);
	if (ret == 0)
		coma_signal(SERVICE_VOICE);

	return ret;
}

static int voice_set_fifos(struct voice_chan *chan)
{
	int ret;

	mutex_lock(&voice_mutex);

	ret =
	    voice_request_set_chan_fifos(chan->id, chan->enc_fifo,
					 chan->dec_fifo);
	if (ret < 0) {
		mutex_unlock(&voice_mutex);
		return -EFAULT;
	}

	wait_for_completion(&voice_reply_compl);

	ret = last_cmsg_voice_params.reply_set_chan_fifos.result;
	mutex_unlock(&voice_mutex);

	return ret;

}

static int voice_request_start_chan(int id, struct voice_codec *codec)
{
	int ret, iLoop;
	union cmsg_voice_params params;

	if (!voice_registered())
		return -EFAULT;

	params.request_start_chan.id = id;
	for (iLoop = 0; iLoop < MAX_NUM_OF_CODEC; iLoop++) {
		params.request_start_chan.rx_list[iLoop].rx_pt =
		    codec->rx_list[iLoop].rx_pt;
		//strcpy(params.request_start_chan.rx_list[iLoop].CodecStr,codec->rx_list[iLoop].CodecStr);
		memcpy(params.request_start_chan.rx_list[iLoop].CodecStr,
		       codec->rx_list[iLoop].CodecStr,
		       sizeof(codec->rx_list[iLoop].CodecStr));
		printk(" In cordless Driver Codec %d and str %s  chan %d \n",
		       params.request_start_chan.rx_list[iLoop].rx_pt,
		       params.request_start_chan.rx_list[iLoop].CodecStr, id);
	}
	params.request_start_chan.rx_codec_event = codec->rx_codec_event;
	params.request_start_chan.tx_codec = codec->tx_codec;
	params.request_start_chan.tx_codec_event = codec->tx_codec_event;
	params.request_start_chan.duration = codec->duration;
	params.request_start_chan.opts = codec->opts;
	params.request_start_chan.current_time = codec->current_time;
	params.request_start_chan.cng.level_rx = codec->cng.level_rx;
	params.request_start_chan.cng.level_tx = codec->cng.level_tx;
	params.request_start_chan.cng.max_sid_update =
	    codec->cng.max_sid_update;
	params.request_start_chan.cng.mode_rx = codec->cng.mode_rx;
	params.request_start_chan.cng.mode_tx = codec->cng.mode_tx;
	params.request_start_chan.cng.vad_detect_level =
	    codec->cng.vad_detect_level;
	params.request_start_chan.cng.vad_hangover = codec->cng.vad_hangover;
	params.request_start_chan.jb.max_len = codec->jb.max_len;
	params.request_start_chan.jb.min_len = codec->jb.min_len;
	params.request_start_chan.jb.type = codec->jb.type;
	params.request_start_chan.Timestamp = codec->Timestamp;
	params.request_start_chan.ssrc = codec->ssrc;
	params.request_start_chan.audio_mode = codec->audio_mode;

//      strcpy(params.request_start_chan.CodecStr,codec->CodecStr);

#if 0
	memset(params.request_start_chan.cNameVal, 0,
	       sizeof(params.request_start_chan.cNameVal));
	memcpy(params.request_start_chan.cNameVal, codec->cNameVal,
	       sizeof(params.request_start_chan.cNameVal));
#endif

	memcpy(params.request_start_chan.CodecStr, codec->CodecStr,
	       strlen(codec->CodecStr));
	params.request_start_chan.CodecStr[strlen(codec->CodecStr)] = '\0';

	params.request_start_chan.dtmf2833numEndPackets =
	    codec->dtmf2833numEndPackets;
	params.request_start_chan.media_loop_level = codec->media_loop_level;
	params.request_start_chan.t38_cfg = codec->t38_cfg;

	ret =
	    coma_create_voice_message(CMSG_VOICE_REQUEST_START_CHAN, &params,
				      NULL, 0);
	if (ret == 0)
		coma_signal(SERVICE_VOICE);

	return ret;
}

static int voice_start_chan(struct voice_chan *chan)
{
	int ret;

	mutex_lock(&voice_mutex);

	ret = voice_request_start_chan(chan->id, &chan->codec);
	if (ret < 0) {
		mutex_unlock(&voice_mutex);
		return -EFAULT;
	}

	wait_for_completion(&voice_reply_compl);

	ret = last_cmsg_voice_params.reply_start_chan.result;
	mutex_unlock(&voice_mutex);

	return ret;
}

static int voice_request_stop_chan(int id)
{
	int ret;
	union cmsg_voice_params params;

	if (!voice_registered())
		return -EFAULT;

	params.request_stop_chan.id = id;

	ret =
	    coma_create_voice_message(CMSG_VOICE_REQUEST_STOP_CHAN, &params,
				      NULL, 0);
	if (ret == 0)
		coma_signal(SERVICE_VOICE);

	return ret;
}

static int voice_stop_chan(struct voice_chan *chan)
{
	int ret;

	mutex_lock(&voice_mutex);

	ret = voice_request_stop_chan(chan->id);
	if (ret < 0) {
		mutex_unlock(&voice_mutex);
		return -EFAULT;
	}

	wait_for_completion(&voice_reply_compl);

	ret = last_cmsg_voice_params.reply_stop_chan.result;
	mutex_unlock(&voice_mutex);

	return ret;
}

int voice_request_update_chan(int id, struct voice_codec *mediaParams)
{
	int ret;
	union cmsg_voice_params params;

	if (!voice_registered())
		return -EFAULT;

	params.request_update_chan.id = id;

	/* Audio Mode */
	params.request_update_chan.audio_mode = mediaParams->audio_mode;

	/* Rx Codec */
	params.request_update_chan.rx_list[0].rx_pt =
	    mediaParams->rx_list[0].rx_pt;
	strcpy(params.request_update_chan.rx_list[0].CodecStr,
	       mediaParams->rx_list[0].CodecStr);

	/* Tx Codec */
	params.request_update_chan.tx_codec = mediaParams->tx_codec;
	strcpy(params.request_update_chan.CodecStr, mediaParams->CodecStr);

	/* Rx Event */
	params.request_update_chan.rx_codec_event = mediaParams->rx_codec_event;

	/* Tx Event */
	params.request_update_chan.tx_codec_event = mediaParams->tx_codec_event;

	/* P - Time */
	params.request_update_chan.duration = mediaParams->duration;

	/* CNG */
	params.request_update_chan.cng.mode_rx = mediaParams->cng.mode_rx;
	params.request_update_chan.cng.mode_tx = mediaParams->cng.mode_tx;
	params.request_update_chan.cng.level_rx = mediaParams->cng.level_rx;
	params.request_update_chan.cng.level_tx = mediaParams->cng.level_tx;
	params.request_update_chan.cng.max_sid_update =
	    mediaParams->cng.max_sid_update;
	params.request_update_chan.cng.vad_detect_level =
	    mediaParams->cng.vad_detect_level;
	params.request_update_chan.cng.vad_hangover =
	    mediaParams->cng.vad_hangover;

	/* Codec Opts */
	params.request_update_chan.opts = mediaParams->opts;

	/* JIB */
	params.request_update_chan.jb.max_len = mediaParams->jb.max_len;
	params.request_update_chan.jb.min_len = mediaParams->jb.min_len;

	/* DTMF End Count */
	params.request_update_chan.dtmf2833numEndPackets =
	    mediaParams->dtmf2833numEndPackets;

	ret =
	    coma_create_voice_message(CMSG_VOICE_REQUEST_UPDATE_CHAN, &params,
				      NULL, 0);
	if (ret == 0)
		coma_signal(SERVICE_VOICE);

	return ret;

}

static
int voice_kmode_update_chan(struct voice_chan *chan, struct voice_codec *mediaParams)
{
	int ret;

	mutex_lock(&voice_mutex);
	ret = voice_request_update_chan(chan->id, mediaParams);
	wait_for_completion(&voice_reply_compl);

	ret = last_cmsg_voice_params.reply_stop_chan.result;
	mutex_unlock(&voice_mutex);

	return ret;

}

#if 0
static int
voice_request_start_rtcp(int id, struct voice_kernelmode_rtcp *km_rtcp)
{
	int ret;
        int i = -1;
	union cmsg_voice_params params;

	if (!voice_registered())
		return -EFAULT;

	params.request_start_rtcp.id = id;
	params.request_start_rtcp.rtcp_interval=km_rtcp->rtcp_interval;
	params.request_start_rtcp.opts=km_rtcp->opts;
 
	for (i = 0; i < MAX_SDES_ITEMS; i++)
	{	
		memset(params.request_start_rtcp.sdesItem[i], 0, sizeof(params.request_start_rtcp.sdesItem[i]));	
		
		if (km_rtcp->sdesItem[i][0] != 0)
		{					
			strncpy(params.request_start_rtcp.sdesItem[i], km_rtcp->sdesItem[i], MAX_SDES_VAL_LEN);				
		}	
	}	
	
	ret = coma_create_voice_message(CMSG_VOICE_REQUEST_START_RTCP, &params, NULL, 0);
	if (ret == 0)
		coma_signal(SERVICE_VOICE);

	return ret;
}

static int
voice_start_rtcp(struct voice_chan *chan, struct voice_kernelmode_rtcp *km_rtcp)
{
	int ret;

	mutex_lock(&voice_mutex);

	ret = voice_request_start_rtcp(chan->id, km_rtcp);
	if (ret < 0) {
		mutex_unlock(&voice_mutex);
		return -EFAULT;
	}

	wait_for_completion(&voice_reply_compl);

	ret = last_cmsg_voice_params.reply_start_rtcp.result;
	mutex_unlock(&voice_mutex);

	return ret;
}

static int voice_request_stop_rtcp(int id)
{
	int ret;
	union cmsg_voice_params params;

	if (!voice_registered())
		return -EFAULT;

	params.request_stop_rtcp.id = id;

	ret =
	    coma_create_voice_message(CMSG_VOICE_REQUEST_STOP_RTCP, &params,
				      NULL, 0);
	if (ret == 0)
		coma_signal(SERVICE_VOICE);

	return ret;
}

static int voice_stop_rtcp(struct voice_chan *chan)
{
	int ret;

	mutex_lock(&voice_mutex);

	ret = voice_request_stop_rtcp(chan->id);
	if (ret < 0) {
		mutex_unlock(&voice_mutex);
		return -EFAULT;
	}

	wait_for_completion(&voice_reply_compl);

	ret = last_cmsg_voice_params.reply_stop_rtcp.result;
	mutex_unlock(&voice_mutex);

	return ret;
}

static int voice_request_report_rtcp(int id)
{
	int ret;
	union cmsg_voice_params params;

	if (!voice_registered())
		return -EFAULT;

	params.request_report_rtcp.id = id;

	ret =
	    coma_create_voice_message(CMSG_VOICE_REQUEST_REPORT_RTCP, &params,
				      NULL, 0);
	if (ret == 0)
		coma_signal();

	return ret;
}

static int
voice_report_rtcp(struct voice_chan *chan, unsigned char *buf, int len)
{

	int ret;

	mutex_lock(&voice_mutex);

	rtcp_buf = NULL;
	rtcp_len = 0;
	ret = voice_request_report_rtcp(chan->id);
	if (ret < 0) {
		mutex_unlock(&voice_mutex);
		return -EFAULT;
	}

	wait_for_completion(&voice_reply_compl);

	ret = last_cmsg_voice_params.reply_report_rtcp.result;
	if ((rtcp_buf != NULL) && (rtcp_len > 0)) {
		if (rtcp_len > len)
			rtcp_len = len;
		memcpy(buf, rtcp_buf, rtcp_len);
	}
	mutex_unlock(&voice_mutex);

	return ret;

	return 0;
}
#endif

static int voice_request_free_chan(int id)
{
	int ret;
	union cmsg_voice_params params;

	if (!voice_registered())
		return -EFAULT;

	params.request_free_chan.id = id;

	ret =
	    coma_create_voice_message(CMSG_VOICE_REQUEST_FREE_CHAN, &params,
				      NULL, 0);
	if (ret == 0)
		coma_signal(SERVICE_VOICE);

	return ret;
}

static int voice_free_chan(struct voice_chan *chan)
{
	int ret;

	mutex_lock(&voice_mutex);

	ret = voice_request_free_chan(chan->id);
	if (ret < 0) {
		mutex_unlock(&voice_mutex);
		return -EFAULT;
	}

	wait_for_completion(&voice_reply_compl);

	ret = last_cmsg_voice_params.reply_free_chan.result;
	mutex_unlock(&voice_mutex);

	return ret;
}

static void voice_data_l2c(int id)
{
	coma_signal(SERVICE_VOICE);
}

static int voice_chan_init(struct voice_chan *chan)
{
	int ret = 0, i;

	init_waitqueue_head(&chan->enc_wq);

	for (i = 0; i < MAX_NUM_OF_CODEC; i++) {
		chan->codec.rx_list[i].rx_pt = 0xff;
		memset(chan->codec.rx_list[i].CodecStr, 0,
		       sizeof(chan->codec.rx_list[i].CodecStr));
	}
	chan->codec.rx_list[0].rx_pt = DEFAULT_CODEC;
	chan->codec.rx_codec_event = 0xff;
	chan->codec.tx_codec = DEFAULT_CODEC;
	chan->codec.tx_codec_event = 0xff;
	chan->codec.duration = DEFAULT_DURATION;
	chan->codec.opts = 0;
	mutex_init(&chan->codec_mutex);

	mutex_init(&chan->dtmf_event_mutex);
	init_waitqueue_head(&chan->dtmf_wq);

	ret = voice_get_chan(chan);
	if (ret != 0)
		return ret;

	ret = voice_set_fifos(chan);
	if (ret != 0)
		goto err_set_fifos;

	chan->status = CHANNEL_INITIALIZED;

	return 0;

      err_set_fifos:
	voice_free_chan(chan);

	return ret;
}

/*
 * kernelmode
 */

char kthread_buf[1024];

static int voice_kmode_thread(void *data)
{
	struct voice_chan *chan;
	unsigned char *enc_buf, *dec_buf;
	struct msghdr msg;
	struct iovec iov;
	int i, to_send, sent, recvd, ret, retryFifos = 1;
	struct task_struct *tsk = current;
	struct sched_param param = {.sched_priority = 1 };

	memset(&msg, 0, sizeof(msg));
	memset(&iov, 0, sizeof(iov));

	msg.msg_iov = &iov;
	msg.msg_iovlen = 1;

	sched_setscheduler(tsk, SCHED_FIFO, &param);

	while (!kthread_should_stop()) {
		while (retryFifos)
		{
			retryFifos = 0;
			for (i = 0; i < voice_numlines ; i++) {
				chan = &voice_channels[i];
				if (chan->status == CHANNEL_STARTED) {
					if (chan->sock)
					{
						// Kernelmode is on
						to_send = cfifo_get(chan->enc_fifo, (void **)&enc_buf);
						if (to_send > 0) {
							/* send data */
							/* For now assume that all data is RTP, */
							/* need to fix for RTCP - Naveen */
							to_send--;
							iov.iov_base = enc_buf+1;
							iov.iov_len = to_send;

							msg.msg_name = &chan->remote_addr;
							msg.msg_namelen = sizeof(chan->remote_addr);
							sent = sock_sendmsg(chan->sock, &msg, to_send);
							if (sent < 0)
									printk(KERN_ERR "%s(%d): sock_sendmsg() failed with"
										   "error %d\n", __FUNCTION__, chan->id, sent);
							else if (sent != to_send)
									printk(KERN_ERR "%s(%d): failed to send %d bytes\n",
										   __FUNCTION__, chan->id, to_send - sent);
				
							cfifo_processed(chan->enc_fifo);
							retryFifos = 1;
						}

						iov.iov_base = kthread_buf;
						iov.iov_len  = sizeof(kthread_buf);

						msg.msg_name = &chan->remote_addr;
						msg.msg_namelen = sizeof(chan->remote_addr);
						msg.msg_flags = MSG_DONTWAIT;
			      
						recvd = kernel_recvmsg(chan->sock, &msg,
						                       (struct kvec *)&iov, 1,
						                       sizeof(kthread_buf), msg.msg_flags);
						if (recvd > 0) {
							/* Added extra byte to distinguish RTP and RTCP */
							ret = cfifo_request(chan->dec_fifo, (void **)&dec_buf, recvd+1);
							if (ret > 0) {
								*dec_buf = CFIFO_RTP_PACKET;
								memcpy(dec_buf+1, kthread_buf, recvd);
								cfifo_commit(chan->dec_fifo, recvd+1);
							}
							//voice_data_l2c(chan->id); // No interrupt required now. Polling implementation
						}
					}
					else 
					{
						// Kernelmode is off. Check if the FIFO is empty and signal the poll
						if (!cfifo_empty(chan->enc_fifo))
						{
							wake_up(&chan->enc_wq);
						}

					}
					
				}
			}
		}
		msleep(2); retryFifos = 1;
	}

	return 0;
}

static int
voice_kmode_start(struct voice_chan *chan, struct voice_kernelmode *km)
{
	int ret = 0;

	/* check if kernelmode is already active */
	if (chan->sock) {
		printk("start: kmode active\n");
		return -1;
	}

	/* set up kmode structure */
	chan->remote_addr = km->remote_addr_rtp;
	chan->sock = sockfd_lookup(km->sock_fd_rtp, &ret);

	if (!chan->sock) {
		printk("start: socket invalid\n");
		return -1;
	}

	cfifo_reset(chan->enc_fifo);

	return ret;
}

static void voice_kmode_stop(struct voice_chan *chan)
{
	/* make sure kmode is active */
	if (!chan->sock)
		return;

	sockfd_put(chan->sock);

	chan->sock = NULL;
}

#if 0
static int
voice_send_dtmf(struct voice_chan *chan, struct voice_dtmf_event *new)
{
	return coma_send_dtmf(chan->id, new->status, new->event,
			      new->volume, new->duration, new->EvtDuration,
			      new->MaxEvtDuration);
}
#endif

static int
voice_request_send_dtmf(int id, int status, char event, char volume,
			int duration, int EvtDuration, int MaxEvtDuration)
{
	int ret;
	union cmsg_voice_params params;

	if (!voice_registered())
		return -EFAULT;

	params.request_send_dtmf.id = id;
	params.request_send_dtmf.status = status;
	params.request_send_dtmf.event = event;
	params.request_send_dtmf.volume = volume;
	params.request_send_dtmf.duration = duration;
	params.request_send_dtmf.EvtDuration = EvtDuration;
	params.request_send_dtmf.MaxEvtDuration = MaxEvtDuration;

	ret =
	    coma_create_voice_message(CMSG_VOICE_REQUEST_SEND_DTMF, &params,
				      NULL, 0);
	if (ret == 0)
		coma_signal(SERVICE_VOICE);

	return ret;
}

static int
voice_send_dtmf(struct voice_chan *chan, struct voice_dtmf_event *new)
{
	int ret;
	mutex_lock(&voice_mutex);

	ret =
	    voice_request_send_dtmf(chan->id, new->status, new->event,
				    new->volume, new->duration,
				    new->EvtDuration, new->MaxEvtDuration);
	if (ret < 0) {
		mutex_unlock(&voice_mutex);
		return -EFAULT;
	}

	wait_for_completion(&voice_reply_compl);

	ret = last_cmsg_voice_params.reply_send_dtmf.result;
	mutex_unlock(&voice_mutex);

	return ret;
}

/*
 * character device functions
 */

static long
voice_ioctl(struct file *file, unsigned int cmd, unsigned long arg)
{
	int size;
	int ret = 0;

	void __user *argp = (void __user *)arg;
	unsigned int minor = iminor(file->f_dentry->d_inode);
	struct voice_chan *chan = &voice_channels[minor];

	if (_IOC_TYPE(cmd) != VOICE_IOC_MAGIC)
		return -EINVAL;

	if (_IOC_NR(cmd) > VOICE_IOC_MAXNR)
		return -EINVAL;

	size = _IOC_SIZE(cmd);

	if (_IOC_DIR(cmd) & _IOC_READ)
		if (!access_ok(VERIFY_WRITE, argp, size))
			return -EFAULT;

	if (_IOC_DIR(cmd) & _IOC_WRITE)
		if (!access_ok(VERIFY_READ, argp, size))
			return -EFAULT;

	switch (cmd) {
	case VOICE_IOCSETCODEC:
		{
			struct voice_codec new;

			if (copy_from_user(&new, argp, sizeof(new)))
				return -EFAULT;

			/*
			 * TODO: switch codec in stack, this will also do
			 *       the sanity checking for us (result in reply)
			 */

			/* update our local info if codec switch went ok */
			if (chan->status == CHANNEL_INVALID)
				return -EFAULT;

			if (chan->status == CHANNEL_STARTED)
				voice_stop_chan(chan);

			mutex_lock(&chan->codec_mutex);
			chan->codec = new;
			mutex_unlock(&chan->codec_mutex);

			ret = voice_start_chan(chan);
			if (ret < 0)
				return -EFAULT;

			chan->status = CHANNEL_STARTED;
			return 0;
		}

	case VOICE_IOCSENDDTMF:
		{
			struct voice_dtmf_event dtmf;

			if (copy_from_user(&dtmf, argp, sizeof(dtmf)))
				return -EFAULT;

			ret = voice_send_dtmf(chan, &dtmf);
			if (ret != 0)
				return -EFAULT;

			return 0;
		}

	case VOICE_IOCGETDTMF:
		{
			if (wait_event_interruptible(chan->dtmf_wq,
						     (chan->dtmf_event !=
						      NULL))) {
				return -ERESTARTSYS;
			}

			mutex_lock(&chan->dtmf_event_mutex);
			if (copy_to_user(argp, chan->dtmf_event,
					 sizeof(*chan->dtmf_event))) {
				mutex_unlock(&chan->dtmf_event_mutex);
				return -EFAULT;
			}
			kfree(chan->dtmf_event);
			chan->dtmf_event = NULL;
			mutex_unlock(&chan->dtmf_event_mutex);

			return 0;
		}

	case VOICE_IOCFLUSH:
		cfifo_reset(chan->enc_fifo);
		return 0;

	case VOICE_IOCKERNELMODE:
		{
			struct voice_kernelmode km;
			mutex_lock(&chan->codec_mutex);

			/*
			 * if argp is set, start kernelmode, if its NULL,
			 * stop kernelmode
			 */
			if (argp) {
				if (copy_from_user(&km, argp, sizeof(km)))
					return -EFAULT;

				ret = voice_kmode_start(chan, &km);
			} else {
				voice_kmode_stop(chan);
			}
			mutex_unlock(&chan->codec_mutex);

			return ret;
		}

#if 0
	case VOICE_IOCRTCP:
		{
			struct voice_kernelmode_rtcp km_rtcp;

			/* if argp is set, start rtcp, if its NULL, stop rtcp */
			if (argp) {
				if (copy_from_user
				    (&km_rtcp, argp, sizeof(km_rtcp)))
					return -EFAULT;
				ret = voice_kmode_start_rtcp(chan, &km_rtcp);
			} else {
				voice_kmode_stop_rtcp(chan);
			}

			return 0;
		}

	case VOICE_IOCUPDATE_CHAN:
		{
			struct voice_codec new;

			if (copy_from_user(&new, argp, sizeof(new)))
				return -EFAULT;

			ret = voice_kmode_update_chan(chan, &new);
			if (ret < 0)
				return -EFAULT;

			return 0;
		}
#endif

	default:		/* redundant, as cmd was checked against MAXNR */
		return -EINVAL;
	}
}

static int voice_open(struct inode *inode, struct file *file)
{
	int ret = 0;
	unsigned int flags = file->f_flags & O_ACCMODE;
	unsigned int minor = iminor(inode);
	struct voice_chan *chan = &voice_channels[minor];

	if (!voice_registered())
		return -1;

	/* only allow one reader */
	if (flags == O_RDONLY || flags == O_RDWR) {
		if (chan->reader)
			return -EBUSY;
		else {
			chan->reader = file;
			file->private_data = chan;
		}
	}

	/* only allow one writer */
	if (flags == O_WRONLY || flags == O_RDWR) {
		if (chan->writer)
			return -EBUSY;
		else {
			chan->writer = file;
			file->private_data = chan;
		}
	}

	if (chan->status == CHANNEL_INVALID) {
		chan->id = minor;
		ret = voice_chan_init(chan);
	}

	return ret;
}

static int voice_release(struct inode *inode, struct file *file)
{
	int ret = 0;
	unsigned int minor = iminor(inode);
	struct voice_chan *chan = &voice_channels[minor];

	if (!voice_registered())
		return -1;

	if (chan->reader == file)
		chan->reader = NULL;

	if (chan->writer == file)
		chan->writer = NULL;

	if (!chan->reader && !chan->writer) {
		chan->status = CHANNEL_INVALID;

		if (chan->sock)
			voice_kmode_stop(chan);

		voice_stop_chan(chan);
		voice_free_chan(chan);
		voice_chan_reset(chan);
	}

	return ret;
}

static ssize_t
voice_read(struct file *file, char __user * buf, size_t count_want,
	   loff_t * f_pos)
{
	struct voice_chan *chan = (struct voice_chan *)file->private_data;
	unsigned char *enc_buf;
	size_t not_copied;
#if 0
	ssize_t count_remain = count_want;
#endif
	ssize_t to_copy;
	ssize_t ret;

	if (!voice_registered())
		return -1;

	if (chan->sock)
		return -EBUSY;

#if 1
//		cfifo_reset(chan->enc_fifo);

	if (wait_event_interruptible(chan->enc_wq,
				     ((ret =
				       cfifo_get(chan->enc_fifo,
						 (void **)&enc_buf))
				      > 0)))
		return -ERESTARTSYS;
#endif	

#if 0
	if ((ret = cfifo_get(chan->enc_fifo, (void **)&enc_buf)) > 0)
#endif		
	{

		/* copy one rtp packet at a time */
		to_copy = ret;
		not_copied = copy_to_user(buf, enc_buf, to_copy);
		cfifo_processed(chan->enc_fifo);
		return ret - not_copied;
	}
	return -ERESTARTSYS;
}

static ssize_t
voice_write(struct file *file, const char __user * buf, size_t count,
	    loff_t * f_pos)
{
	int ret = 0;
	struct voice_chan *chan = (struct voice_chan *)file->private_data;
	char *fifo_buf;

	if (!voice_registered())
		return -1;

	if (chan->sock)
		return -EBUSY;

	ret = cfifo_request(chan->dec_fifo, (void **)&fifo_buf, count);
	if (ret <= 0)
		return -ENOMEM;

	ret = copy_from_user(fifo_buf, buf, count);
	if (ret != 0)
		return -1;

	cfifo_commit(chan->dec_fifo, count);

	/* DATA */
	//voice_data_l2c(chan->id); // No intertrupt required now. Polling implementation

	return count;
}

static unsigned int voice_poll(struct file *file, poll_table * event_list)
{
	struct voice_chan *chan = (struct voice_chan *)file->private_data;
	unsigned int mask = 0;

	/* we always allow writing */
	mask |= POLLOUT | POLLWRNORM;

	if (!cfifo_empty(chan->enc_fifo))
		mask |= POLLIN | POLLRDNORM;

	poll_wait(file, &chan->enc_wq, event_list);
	mask |= POLLIN | POLLRDNORM;
	return mask;
}

static struct file_operations voice_fops = {
	.owner          = THIS_MODULE,
	.unlocked_ioctl = voice_ioctl,
	.open           = voice_open,
	.release        = voice_release,
	.read           = voice_read,
	.write          = voice_write,
	.poll           = voice_poll,
};

void voice_release_all(void)
{
	int i;
	struct voice_chan *chan;

	if (voice_registered()) {
		for (i = 0; i < CONFIG_CORDLESS_NUM_VOICELINES; i++) {
			chan = &voice_channels[i];

			if (chan->sock)
				voice_kmode_stop(chan);

			if (chan->status == CHANNEL_STARTED) {
				voice_stop_chan(chan);
				voice_free_chan(chan);
			}
			voice_chan_reset(chan);
		}
	}
}

int voice_init(void)
{
	int i, j, ret = 0;

	printk(KERN_ERR "%s: service init\n", SERVICE_NAME);

	ret = alloc_chrdev_region(&voice_dev, 0, voice_numlines, voice_name);
	if (ret) {
		printk("coma-voice: alloc of chrdev returned %d\n", ret);
		return ret;
	}

	cdev_init(&voice_cdev, &voice_fops);

	voice_cdev.owner = THIS_MODULE;
	voice_cdev.ops = &voice_fops;

	ret = cdev_add(&voice_cdev, voice_dev, voice_numlines);
	if (ret) {
		printk(KERN_ERR "%s: Error %d adding character device",
		       voice_name, ret);
		goto err_unreg_chrdev_reg;
	}

	cfifo_size =
	    cfifo_alloc(FIFO_SIZE, &voice_c2l, FIFO_SIZE, &voice_l2c,
			&cfifo_phys);
	if (cfifo_size < 0) {
		ret = -ENOMEM;
		goto err_unreg_chrdev_reg;
	}

	for (i = 0; i < CONFIG_CORDLESS_NUM_VOICELINES; i++) {

		voice_channels[i].cfifo_size =
		    cfifo_alloc(FIFO_SIZE, &voice_channels[i].enc_fifo,
				FIFO_SIZE, &voice_channels[i].dec_fifo,
				&voice_channels[i].cfifo_phys);
		if (voice_channels[i].cfifo_size < 0) {
			ret = -ENOMEM;
			goto err_cfifo_free;
		}

	}

	ret = coma_register(SERVICE_VOICE, SERVICE_NAME, voice_l2c, voice_c2l,
			    voice_process_message, voice_setup, voice_remove);
	if (ret < 0) {
		printk(KERN_ERR "%s: Registration failed: %d\n", voice_name,
		       ret);
		goto err_cfifo_free;
	}

	/* create sender thread */
	kmode_thread = kthread_create(voice_kmode_thread, 0, "kvoice");

	/* start the thread */
	wake_up_process(kmode_thread);

	printk(KERN_INFO "%s: character device initialized (major=%d)\n",
	       voice_name, MAJOR(voice_dev));

	return 0;

      err_cfifo_free:

	for (j = 0; j < CONFIG_CORDLESS_NUM_VOICELINES; j++) {
		if (voice_channels[i].cfifo_size > 0)
			cfifo_free((char *)voice_channels[j].enc_fifo,
				   voice_channels[j].cfifo_size,
				   voice_channels[j].cfifo_phys);

	}

	cfifo_free((char *)voice_c2l, cfifo_size, cfifo_phys);
      err_unreg_chrdev_reg:
	unregister_chrdev_region(voice_dev, voice_numlines);

	return ret;
}

void voice_exit(void)
{
	int i;

	voice_release_all();
	for (i = 0; i < CONFIG_CORDLESS_NUM_VOICELINES; i++) {
		cfifo_free((char *)voice_channels[i].enc_fifo,
			   voice_channels[i].cfifo_size,
			   voice_channels[i].cfifo_phys);
	}

	cfifo_free((char *)voice_c2l, cfifo_size, cfifo_phys);

	cdev_del(&voice_cdev);
	unregister_chrdev_region(voice_dev, voice_numlines);
}

EXPORT_SYMBOL(voice_init);
EXPORT_SYMBOL(voice_exit);
