/*
 *  linux/arch/arm/mach-dmw/clock.c
 *
 *  Copyright (C) 2010 DSP Group
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * version 2 as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA
 *
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/list.h>
#include <linux/errno.h>
#include <linux/err.h>
#include <linux/string.h>
#include <linux/clk.h>
#include <linux/clkdev.h>
#include <linux/mutex.h>
#include <linux/proc_fs.h>
#include <linux/io.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/seq_file.h>
#include <asm/div64.h>

#include <mach/hardware.h>
#include <mach/platform.h>
#include <mach/clock.h>
#include <mach/ramsem.h>

#include "pm.h"

static LIST_HEAD(clocks_list);
static struct ram_sem *cmu_sem = NULL;

int clk_active_bm_cnt = 0;
int clk_active_io_cnt = 0;

#define WRPR_UNLOCK_CLK_EN	(0x09 << 4)
#define WRPR_UNLOCK_RST_EN	(0x06 << 0)

#define cmu_read(_reg)		readl(IO_ADDRESS(DMW_CMU_BASE) + (_reg))
#define cmu_write(_val, _reg)	writel((_val), IO_ADDRESS(DMW_CMU_BASE) + (_reg))

#define WAIT_PLL_1	(1<<0)
#define WAIT_PLL_2	(1<<1)
#define WAIT_PLL_3	(1<<2)
#define WAIT_PLL_4	(1<<3)
#define WAIT_OSC_DECT	(1<<4)

static unsigned long get_rate_parent(struct clk *clk);
static unsigned long get_rate_fixed(struct clk *clk);
static int get_on_off_always_on(struct clk *clk);
static int get_on_off_gated(struct clk *clk);
static int set_on_off_gated(struct clk *clk, int on);

static int get_on_off_osc_dect(struct clk *clk);
static int set_on_off_osc_dect(struct clk *clk, int on);

static unsigned long get_rate_pll(struct clk *clk);
static int set_rate_pll(struct clk *clk, unsigned long rate);
static long round_rate_pll(struct clk *clk, unsigned long rate);
static int get_on_off_pllx(struct clk *clk);
static int set_on_off_pllx(struct clk *clk, int on);
static int set_on_off_pll1(struct clk *clk, int on);
static struct clk *get_parent_pll1(struct clk *clk);
static int set_parent_pll1(struct clk *clk, struct clk *parent);
static struct clk *get_parent_pll4src(struct clk *clk);
static int set_parent_pll4src(struct clk *clk, struct clk *parent);

static struct clk *get_parent_cpuiclk(struct clk *clk);
static int set_parent_cpuiclk(struct clk *clk, struct clk *parent);
static struct clk *get_parent_cortex(struct clk *clk);
static int set_parent_cortex(struct clk *clk, struct clk *parent);
static struct clk *get_parent_altsysclk(struct clk *clk);
static int set_parent_altsysclk(struct clk *clk, struct clk *parent);
static struct clk *get_parent_sysclk(struct clk *clk);
static int set_parent_sysclk(struct clk *clk, struct clk *parent);

static unsigned long get_rate_div(struct clk *clk);
static int set_rate_div(struct clk *clk, unsigned long rate);
static long round_rate_div(struct clk *clk, unsigned long rate);
static int set_rate_comaltdiv(struct clk *clk, unsigned long rate);

static struct clk *get_parent_pll4prediv(struct clk *clk);
static int set_parent_pll4prediv(struct clk *clk, struct clk *parent);
static struct clk *get_parent_lcdc(struct clk *clk);
static int set_parent_lcdc(struct clk *clk, struct clk *parent);
static struct clk *get_parent_tdm1(struct clk *clk);
static int set_parent_tdm1(struct clk *clk, struct clk *parent);
static struct clk *get_parent_tdm2(struct clk *clk);
static int set_parent_tdm2(struct clk *clk, struct clk *parent);
static struct clk *get_parent_tdm3(struct clk *clk);
static int set_parent_tdm3(struct clk *clk, struct clk *parent);

static struct clk *get_parent_dram(struct clk *clk);
static int set_parent_dram(struct clk *clk, struct clk *parent);
static struct clk *get_parent_css(struct clk *clk);
static int set_parent_css(struct clk *clk, struct clk *parent);

static unsigned long get_rate_hclk(struct clk *clk);
static int set_rate_hclk(struct clk *clk, unsigned long rate);
static long round_rate_hclk(struct clk *clk, unsigned long rate);

static int get_on_off_divsclk(struct clk *clk);
static int set_on_off_divsclk(struct clk *clk, int on);
static struct clk * get_parent_slowclk(struct clk *clk);
static int set_parent_slowclk(struct clk *clk, struct clk *parent);

static struct clk osc_32k = {
	.name = "osc_32k",
	.rate = 32768,
	.get_rate = get_rate_fixed,
	.get_on_off = get_on_off_always_on,
};


static struct clk osc_12m = {
	.name = "osc_12m",
	.rate = 12000000,
	.get_rate = get_rate_fixed,
	.get_on_off = get_on_off_always_on,
};

static struct clk osc_dect = {
	.name = "osc_dect",
	.rate = 13824000,
	.get_rate = get_rate_fixed,
	.get_on_off = get_on_off_osc_dect,
	.set_on_off = set_on_off_osc_dect,
};

static struct clk osc_20m = {
	.name = "osc_20m",
	.rate = 20000000,
	.get_rate = get_rate_fixed,
	.get_on_off = get_on_off_always_on,
};

static struct clk pll1 = {
	.name = "pll1",
	.priv.pll.reg = DMW_CMU_PLL1CONTROL,
	.priv.pll.waitflag = WAIT_PLL_1,
	.get_rate = get_rate_pll,
	.set_rate = set_rate_pll,
	.round_rate = round_rate_pll,
	.get_on_off = get_on_off_pllx,
	.set_on_off = set_on_off_pll1,
	.get_parent = get_parent_pll1,
	.set_parent = set_parent_pll1
};

static struct clk pll2 = {
	.name = "pll2",
	.priv.pll.reg = DMW_CMU_PLL2CONTROL,
	.priv.pll.waitflag = WAIT_PLL_2,
	.parent = &osc_12m,
	.get_rate = get_rate_pll,
	.set_rate = set_rate_pll,
	.round_rate = round_rate_pll,
	.get_on_off = get_on_off_pllx,
	.set_on_off = set_on_off_pllx,
};

static struct clk pll3 = {
	.name = "pll3",
	.priv.pll.reg = DMW_CMU_PLL3CONTROL,
	.priv.pll.waitflag = WAIT_PLL_3,
	.parent = &osc_12m,
	.get_rate = get_rate_pll,
	.set_rate = set_rate_pll,
	.round_rate = round_rate_pll,
	.get_on_off = get_on_off_pllx,
	.set_on_off = set_on_off_pllx,
};

static struct clk pll4src = {
	.name = "pll4src",
	.get_rate = get_rate_parent,
	.get_on_off = get_on_off_always_on,
	.get_parent = get_parent_pll4src,
	.set_parent = set_parent_pll4src,
};

static struct clk pll4 = {
	.name = "pll4",
	.parent = &pll4src,
	.priv.pll.reg = DMW_CMU_PLL4CONTROL,
	.priv.pll.waitflag = WAIT_PLL_4,
	.get_rate = get_rate_pll,
	.set_rate = set_rate_pll,
	.round_rate = round_rate_pll,
	.get_on_off = get_on_off_pllx,
	.set_on_off = set_on_off_pllx,
};

static struct clk cpuiclk = {
	.name = "cpuiclk",
	.get_rate = get_rate_parent,
	.get_parent = get_parent_cpuiclk,
	.set_parent = set_parent_cpuiclk,
	.get_on_off = get_on_off_always_on,
};

static struct clk cortex = {
	.name = "cortex",
	.flags = FLAG_ENABLE,
	.get_rate = get_rate_parent,
	.get_parent = get_parent_cortex,
	.set_parent = set_parent_cortex,
	.get_on_off = get_on_off_always_on,
};

static struct clk altsysclk = {
	.name = "altsysclk",
	.get_rate = get_rate_parent,
	.get_on_off = get_on_off_always_on,
	.get_parent = get_parent_altsysclk,
	.set_parent = set_parent_altsysclk,
};

static struct clk sysclk = {
	.name = "sysclk",
	.get_rate = get_rate_parent,
	.get_on_off = get_on_off_always_on,
	.get_parent = get_parent_sysclk,
	.set_parent = set_parent_sysclk,
};

static struct clk divsclk = {
	.name = "divsclk",
	.parent = &osc_12m,
	.flags = FLAG_ENABLE,
	.priv.gate.reg = DMW_CMU_CLKSWCNTRL,
	.priv.gate.shift = 4,
	.priv.gate.size = 11,
	.get_rate = get_rate_div,
	.set_rate = set_rate_div,
	.round_rate = round_rate_div,
	.get_on_off = get_on_off_divsclk,
	.set_on_off = set_on_off_divsclk,
};

static struct clk slowclk = {
	.name = "slowclk",
	.get_parent = get_parent_slowclk,
	.set_parent = set_parent_slowclk,
	.get_rate = get_rate_parent,
	.get_on_off = get_on_off_always_on,
};

static struct clk rtc = {
	.name = "rtc",
	.parent = &slowclk,
	.priv.gate.bit = 2,
	.get_rate = get_rate_parent,
	.get_on_off = get_on_off_gated,
	.set_on_off = set_on_off_gated,
};

#define CLK_DIV_SYS(_name, _bit, _reg, _shift, _size, _flags)	\
	static struct clk _name = {				\
		.name = __stringify(_name),			\
		.flags = (_flags),				\
		.parent = &sysclk,				\
		.priv.gate.bit = (_bit),			\
		.priv.gate.reg = (_reg),			\
		.priv.gate.shift = (_shift),			\
		.priv.gate.size = (_size),			\
		.get_rate = get_rate_div,			\
		.set_rate = set_rate_div,			\
		.round_rate = round_rate_div,			\
		.get_on_off = get_on_off_gated,			\
		.set_on_off = set_on_off_gated,			\
	}

/*          name    bit reg                 shift size flags */
CLK_DIV_SYS(gpu,    24, DMW_CMU_CLKDIVCNTRL2, 16,   4,   FLAG_BM);
CLK_DIV_SYS(ciu,    19, DMW_CMU_CLKDIVCNTRL1,  0,   8,   FLAG_BM | FLAG_IO);
CLK_DIV_SYS(sms,    12, DMW_CMU_CLKDIVCNTRL1, 16,   8,   FLAG_BM | FLAG_IO);
CLK_DIV_SYS(sdmmc,  13, DMW_CMU_CLKDIVCNTRL1, 24,   8,   FLAG_BM | FLAG_IO);
CLK_DIV_SYS(dp,     28, DMW_CMU_CLKDIVCNTRL3,  0,   8,   FLAG_IO);
CLK_DIV_SYS(videnc, 26, DMW_CMU_CLKDIVCNTRL2,  0,   4,   FLAG_BM);
CLK_DIV_SYS(viddec, 25, DMW_CMU_CLKDIVCNTRL2,  8,   8,   FLAG_BM);
CLK_DIV_SYS(sim,    27, DMW_CMU_CLKDIVCNTRL3,  8,   8,   FLAG_IO);

#define CLK_DIV_MUX(_name, _bit, _reg, _shift, _size, _flags)	\
	static struct clk _name = {				\
		.name = __stringify(_name),			\
		.flags = (_flags),				\
		.priv.gate.bit = (_bit),			\
		.priv.gate.reg = (_reg),			\
		.priv.gate.shift = (_shift),			\
		.priv.gate.size = (_size),			\
		.get_rate = get_rate_div,			\
		.set_rate = set_rate_div,			\
		.round_rate = round_rate_div,			\
		.get_on_off = get_on_off_gated,			\
		.set_on_off = set_on_off_gated,			\
		.get_parent = get_parent_##_name,		\
		.set_parent = set_parent_##_name,		\
	}

/*          name        bit reg               shift size flags */
CLK_DIV_MUX(pll4prediv, 51, DMW_CMU_CLKDIVCNTRL2, 24,    8,  0);
CLK_DIV_MUX(lcdc,       21, DMW_CMU_CLKDIVCNTRL1, 8,     8,  FLAG_BM | FLAG_IO | FLAG_DONT_DISABLE);
CLK_DIV_MUX(tdm1,        9, DMW_CMU_CLKDIVCNTRL4, 0,    12,  FLAG_IO);
CLK_DIV_MUX(tdm2,       10, DMW_CMU_CLKDIVCNTRL4, 12,   12,  FLAG_IO);
CLK_DIV_MUX(tdm3,       11, DMW_CMU_CLKDIVCNTRL5, 0,    12,  FLAG_IO);

static struct clk comaltdiv = {
	.name = "comaltdiv",
	.parent = &sysclk,
	.priv.gate.reg = DMW_CMU_COMCLKSEL,
	.priv.gate.shift = 0,
	.priv.gate.size = 4,
	.get_rate = get_rate_div,
	.set_rate = set_rate_comaltdiv,
	.round_rate = round_rate_div,
	.get_on_off = get_on_off_always_on,
};

static struct clk dram = {
	.name = "dram",
	.flags = FLAG_ENABLE,
	.priv.gate.bit = 44,
	.get_parent = get_parent_dram,
	.set_parent = set_parent_dram,
	.get_on_off = get_on_off_gated,
	.set_on_off = set_on_off_gated,
	.get_rate = get_rate_parent,
};

static struct clk css = {
	.name = "css",
	.priv.gate.bit = 23,
	.get_parent = get_parent_css,
	.set_parent = set_parent_css,
	.get_on_off = get_on_off_gated,
	.set_on_off = set_on_off_gated,
	.get_rate = get_rate_parent,
};

static struct clk css_arm = {
	.name = "css_arm",
	.priv.gate.bit = 1,
	.parent = &css,
	.get_on_off = get_on_off_gated,
	.set_on_off = set_on_off_gated,
	.get_rate = get_rate_parent,
};

static struct clk css_etm = {
	.name = "css_etm",
	.priv.gate.bit = 31,
	.parent = &css,
	.get_on_off = get_on_off_gated,
	.set_on_off = set_on_off_gated,
	.get_rate = get_rate_parent,
};

static struct clk usb1_phy = {
	.name = "usb1_phy",
	.parent = &osc_12m,
	.priv.gate.bit = 17,
	.get_on_off = get_on_off_gated,
	.set_on_off = set_on_off_gated,
	.get_rate = get_rate_parent,
};

static struct clk usb1_mac = {
	.name = "usb1_mac",
	.parent = &usb1_phy,
	.priv.gate.bit = 16,
	.get_on_off = get_on_off_gated,
	.set_on_off = set_on_off_gated,
	.rate = 480000000,
	.get_rate = get_rate_fixed,
};

static struct clk usb2_phy = {
	.name = "usb2_phy",
	.parent = &osc_12m,
	.priv.gate.bit = 49,
	.get_on_off = get_on_off_gated,
	.set_on_off = set_on_off_gated,
	.get_rate = get_rate_parent,
};

static struct clk usb2_mac = {
	.name = "usb2_mac",
	.parent = &usb2_phy,
	.priv.gate.bit = 48,
	.get_on_off = get_on_off_gated,
	.set_on_off = set_on_off_gated,
	.rate = 480000000,
	.get_rate = get_rate_fixed,
};

static struct clk hclk = {
	.name = "hclk",
	.flags = FLAG_ENABLE,
	.parent = &sysclk,
	.priv.gate.reg = DMW_CMU_CPUCLKCNTRL,
	.priv.gate.shift = 4,
	.priv.gate.size = 4,
	.get_on_off = get_on_off_always_on,
	.get_rate = get_rate_hclk,
	.set_rate = set_rate_hclk,
	.round_rate = round_rate_hclk,
};

#define CLK_HCLK(_name, _bit, _flags)		\
	static struct clk _name = {		\
		.name = __stringify(_name),	\
		.flags = (_flags),		\
		.parent = &hclk,		\
		.priv.gate.bit = (_bit),	\
		.get_rate = get_rate_parent,	\
		.get_on_off = get_on_off_gated,	\
		.set_on_off = set_on_off_gated,	\
	}

CLK_HCLK(coresight, 52, FLAG_DONT_DISABLE);
CLK_HCLK(dbm,       50, 0); /* DRAM bus monitor */
CLK_HCLK(clkout,    43, 0);
CLK_HCLK(smc,       42, 0); /* Static Memory Controller */
CLK_HCLK(ir_remctl, 41, FLAG_IO);
CLK_HCLK(slvmii,    40, FLAG_IO);
CLK_HCLK(spi2,      39, FLAG_IO);
CLK_HCLK(spi1,      38, FLAG_IO);
CLK_HCLK(i2c2,      37, FLAG_IO);
CLK_HCLK(i2c1,      36, FLAG_IO);
CLK_HCLK(timer4,    35, 0);
CLK_HCLK(timer3,    34, 0);
CLK_HCLK(timer2,    33, 0);
CLK_HCLK(timer1,    32, 0);
CLK_HCLK(sec,       22, 0);
CLK_HCLK(osdm,      20, FLAG_BM);
CLK_HCLK(wifi,      18, 0);
CLK_HCLK(pacp,      15, 0);
CLK_HCLK(nfc,       14, FLAG_IO | FLAG_BM);
CLK_HCLK(uart3,      8, FLAG_ENABLE); /* FIXME: see next FIXME for comment */
CLK_HCLK(uart2,      7, FLAG_ENABLE); /* FIXME: see next FIXME for comment */
CLK_HCLK(uart1,      6, FLAG_ENABLE); /* FIXME: see next FIXME for comment */
CLK_HCLK(secure,     4, FLAG_BM);
CLK_HCLK(ethmac,     3, FLAG_IO | FLAG_BM);
CLK_HCLK(gdmac,      0, FLAG_BM);

/* Deliberately left out due to unknown parent: RMII_IF_CLK_EN (bit 5) */

static struct clk *onchip_clks[] = {
	&osc_32k,
	&osc_12m,
	&osc_dect,
	&osc_20m,
	&pll1,
	&pll2,
	&pll3,
	&pll4src,
	&pll4,
	&cpuiclk,
	&cortex,
	&altsysclk,
	&sysclk,
	&gpu,
	&ciu,
	&sms,
	&sdmmc,
	&dp,
	&videnc,
	&viddec,
	&sim,
	&pll4prediv,
	&lcdc,
	&tdm1,
	&tdm2,
	&tdm3,
	&comaltdiv,
	&dram,
	&css,
	&css_arm,
	&css_etm,
	&usb1_phy,
	&usb1_mac,
	&usb2_phy,
	&usb2_mac,
	&hclk,
	&coresight,
	&dbm,
	&clkout,
	&smc,
	&ir_remctl,
	&slvmii,
	&spi2,
	&spi1,
	&i2c2,
	&i2c1,
	&timer4,
	&timer3,
	&timer2,
	&timer1,
	&sec,
	&osdm,
	&wifi,
	&pacp,
	&nfc,
	&uart3,
	&uart2,
	&uart1,
	&secure,
	&ethmac,
	&rtc,
	&gdmac,
	&divsclk,
	&slowclk,
};

/*****************************************************************************
 * Device clock mappings
 ****************************************************************************/

#define CLK_MAP_DEV(_devname, _clk)	\
	{				\
		.dev_id = (_devname),	\
		.clk = (_clk),		\
	}

#define CLK_MAP_DEV_CON(_devname, _conname, _clk)	\
	{						\
		.dev_id = (_devname),			\
		.con_id = (_conname),			\
		.clk = (_clk),				\
	}

static struct clk_lookup clk_mappings[] = {
	CLK_MAP_DEV("osc_dect",		&osc_dect),
	CLK_MAP_DEV("pll4src",		&pll4src),
	CLK_MAP_DEV("pll3",		&pll3),
	CLK_MAP_DEV("pll4",		&pll4),
	CLK_MAP_DEV("dbm.0",		&dbm),
	CLK_MAP_DEV("dmw-tdm",		&dp),
	CLK_MAP_DEV("dmw-tdm.0",	&tdm1),
	CLK_MAP_DEV("dmw-tdm.1",	&tdm2),
	CLK_MAP_DEV("dmw-tdm.2",	&tdm3),
	CLK_MAP_DEV("dmw96-spi.0",	&spi1),
	CLK_MAP_DEV("dmw96-spi.1",	&spi2),
	CLK_MAP_DEV("stmmaceth.0",	&ethmac),
	CLK_MAP_DEV("dmw96ciu.0",	&ciu),
	CLK_MAP_DEV("dmw96dma.0",	&gdmac),
	CLK_MAP_DEV("dmw96osdm.0",	&osdm),
	CLK_MAP_DEV("dmw_nand.0",	&nfc),
	CLK_MAP_DEV("dw-uart.0",	&uart1),
	CLK_MAP_DEV("dw-uart.1",	&uart2),
	CLK_MAP_DEV("dw-uart.2",	&uart3),
	CLK_MAP_DEV("dw-wifi",		&wifi),
	CLK_MAP_DEV("dw74fb.0",		&lcdc),
	CLK_MAP_DEV("dw_mmc.0",		&sdmmc),
	CLK_MAP_DEV("galcore.0",	&gpu),
	CLK_MAP_DEV("hx170dec.0",	&viddec),
	CLK_MAP_DEV("hx280enc.0",	&videnc),
	CLK_MAP_DEV("pnx-i2c.0",	&i2c1),
	CLK_MAP_DEV("pnx-i2c.1",	&i2c2),
	CLK_MAP_DEV("rtc",		&rtc),
	CLK_MAP_DEV("sec.0",		&sec),
	CLK_MAP_DEV("timer1",		&timer1),
	CLK_MAP_DEV("timer2",		&timer2),
	CLK_MAP_DEV("timer3",		&timer3),
	CLK_MAP_DEV("timer4",		&timer4),

	CLK_MAP_DEV_CON("cpufreq-dmw",	"cpu",		&cortex),
	CLK_MAP_DEV_CON("css.0",	"arm",		&css_arm),
	CLK_MAP_DEV_CON("css.0",	"css",		&css),
	CLK_MAP_DEV_CON("css.0",	"etm",		&css_etm),
	CLK_MAP_DEV_CON("css.0",	"fast",		&pll4),
	CLK_MAP_DEV_CON("css.0",	"slow",		&comaltdiv),
	CLK_MAP_DEV_CON("dwc_otg.0",	"mac",		&usb1_mac),
	CLK_MAP_DEV_CON("dwc_otg.0",	"phy",		&usb1_phy),
	CLK_MAP_DEV_CON("dwc_otg.1",	"mac",		&usb2_mac),
	CLK_MAP_DEV_CON("dwc_otg.1",	"phy",		&usb2_phy),
	CLK_MAP_DEV_CON("dmw-pm",	"cpu",		&cortex),
	CLK_MAP_DEV_CON("dmw-pm",	"dram",		&dram),
	CLK_MAP_DEV_CON("dmw-pm",	"slow",		&cpuiclk),
};

static struct clk_lookup clk_mappings_cpufreq[] = {
	CLK_MAP_DEV_CON("cpufreq-dmw",	"fast",		&pll2),
	CLK_MAP_DEV_CON("cpufreq-dmw",	"safe",		&cpuiclk),
};

/*****************************************************************************
 * Clock specific callbacks
 ****************************************************************************/

static unsigned long
get_rate_parent(struct clk *clk)
{
	return clk->parent->get_rate(clk->parent);
}

static unsigned long
get_rate_fixed(struct clk *clk)
{
	return clk->rate;
}

static int
get_on_off_always_on(struct clk *clk)
{
	return 1;
}

static int
get_on_off_gated(struct clk *clk)
{
	u32 clken = 0, bit = clk->priv.gate.bit;

	if (bit < 32) {
		clken = cmu_read(DMW_CMU_SWCLKENR1);
	} else if (bit < 64) {
		clken = cmu_read(DMW_CMU_SWCLKENR2);
		bit -= 32;
	}

	return !!(clken & (1 << bit));
}

static int
set_on_off_gated(struct clk *clk, int on)
{
	u32 clken, bit = clk->priv.gate.bit;

	if (cmu_sem)
		ram_sem_lock(cmu_sem);

	if (bit < 32) {
		clken = cmu_read(DMW_CMU_SWCLKENR1);
		if (on)
			clken |=  (1 << bit);
		else
			clken &= ~(1 << bit);
		cmu_write(WRPR_UNLOCK_CLK_EN, DMW_CMU_WRPR);
		cmu_write(clken, DMW_CMU_SWCLKENR1);
	} else if (bit < 64) {
		bit -= 32;
		clken = cmu_read(DMW_CMU_SWCLKENR2);
		if (on)
			clken |=  (1 << bit);
		else
			clken &= ~(1 << bit);
		cmu_write(WRPR_UNLOCK_CLK_EN, DMW_CMU_WRPR);
		cmu_write(clken, DMW_CMU_SWCLKENR2);
	}

	if (cmu_sem)
		ram_sem_unlock(cmu_sem);

	return 0;
}

static int
get_on_off_osc_dect(struct clk *clk)
{
	return !!(cmu_read(DMW_CMU_CLKOSCCR) & 2);
}

static int
set_on_off_osc_dect(struct clk *clk, int on)
{
	cmu_write(cmu_read(DMW_CMU_CLKOSCCR) | 2, DMW_CMU_CLKOSCCR);
	return WAIT_OSC_DECT;
}

static unsigned long
get_rate_pll(struct clk *clk)
{
	unsigned long in, prediv, mult, postdiv;
	unsigned long long tmp;
	u32 reg = cmu_read(clk->priv.pll.reg);

	in = clk->parent->get_rate(clk->parent);
	prediv  = ((reg >>  0) & 0x1f) + 1;
	mult    = ((reg >>  5) & 0x7f) + 1;
	postdiv = (reg >> 12) & 0x03;

	tmp = (unsigned long long)in * mult;
	do_div(tmp, prediv);

	return (unsigned long)tmp >> postdiv;
}

#define ABSDIFF(x, y)	((x) > (y) ? (x) - (y) : (y) - (x))

static long
round_rate_pll(struct clk *clk, unsigned long rate)
{
	unsigned long prediv, mult, postdiv;
	unsigned long match_prediv=0, match_mult=0, match_postdiv=0;
	unsigned long match_rate=0, match_diff=~0ul, match_vco=~0ul;

	unsigned long in = clk->parent->get_rate(clk->parent);

	/* loop through all settings finding best match */
	for (prediv=1; prediv<=32; prediv++) {
		unsigned long phaseclk = (in >> 10) / prediv;

		/* keep in phase detector reference frequency: 10..50MHz */
		if (phaseclk < 10000000>>10 || phaseclk > 50000000>>10)
			continue;

		for (mult=6; mult<=100; mult++) {
			unsigned long vcoclk = phaseclk * mult;

			/* keep in VCO frequency range: 300..1000MHz */
			if (vcoclk < 300000000>>10 || vcoclk > 1000000000>>10)
				continue;

			for (postdiv=0; postdiv<=3; postdiv++) {
				unsigned long diff, out = vcoclk >> postdiv;
				unsigned long long tmp;

				/* quick check if in target range */
				if (out-10000 > rate>>10 || out+10000 < rate>>10)
					continue;

				/* ok, do the precise calculation */
				tmp = (unsigned long long)in * mult;
				do_div(tmp, prediv);
				out = (unsigned long)tmp >> postdiv;

				/* better match? */
				diff = ABSDIFF(rate, out);
				if ((diff <  match_diff) ||
				    (diff == match_diff && vcoclk < match_vco)) {
					match_diff    = diff;
					match_vco     = vcoclk;
					match_rate    = out;
					match_prediv  = prediv;
					match_mult    = mult;
					match_postdiv = postdiv;
				}
			}
		}
	}

	if (!match_rate)
		return -EINVAL;

	/* cache setting */
	clk->priv.pll.round_rate     = match_rate;
	clk->priv.pll.round_setting  = (match_prediv  - 1) <<  0;
	clk->priv.pll.round_setting |= (match_mult    - 1) <<  5;
	clk->priv.pll.round_setting |= (match_postdiv    ) << 12;
	clk->priv.pll.round_setting |= match_vco > 600000000>>10 ? 1 << 14 : 0;

	return match_rate;
}

static int
set_rate_pll(struct clk *clk, unsigned long rate)
{
	u32 reg;

	/* See if rate is supported */
	if (rate != clk->priv.pll.round_rate) {
		long ret;
		if ((ret = round_rate_pll(clk, rate)) <= 0)
			return ret;
		if (ret != rate)
			return -EINVAL;
	}

	/* Make sure the PLL is not running */
	if (clk->get_on_off(clk))
		return -EBUSY;

	/* Apply settings */
	reg = cmu_read(clk->priv.pll.reg);
	reg &= ~0x7ffful;
	reg |= clk->priv.pll.round_setting;
	cmu_write(reg, clk->priv.pll.reg);

	ndelay(500);

	return 0;
}

static int
get_on_off_pllx(struct clk *clk)
{
	return !(cmu_read(clk->priv.pll.reg) & 0x8000);
}

static int
set_on_off_pllx(struct clk *clk, int on)
{
	u32 reg = cmu_read(clk->priv.pll.reg);
	if (on) {
		if ((reg & 0x8000) == 0)
			return 0;

		reg &= ~0x8000;
		cmu_write(reg, clk->priv.pll.reg);
		return clk->priv.pll.waitflag;
	} else {
		reg |= 0x8000;
		cmu_write(reg, clk->priv.pll.reg);
		return 0;
	}
}

static int
set_on_off_pll1(struct clk *clk, int on)
{
	u32 reg = cmu_read(DMW_CMU_CPUCLKCNTRL);
	if (on) {
		if ((reg & (1 << 11)) == 0)
			return 0;

		reg &= ~(1 << 11);
		cmu_write(reg, DMW_CMU_CPUCLKCNTRL);
		return WAIT_PLL_1;
	} else {
		reg |= 1 << 11;
		cmu_write(reg, DMW_CMU_CPUCLKCNTRL);
		return 0;
	}
}

static struct clk *
get_parent_pll1(struct clk *clk)
{
	u32 reg = cmu_read(DMW_CMU_PLL1CONTROL);

	switch ((reg >> 19) & 0x03) {
		case 0:
		default:
			return &osc_12m;
		case 1:
			return &osc_20m;
		case 2:
			return &osc_dect;
	}
}

static int
set_parent_pll1(struct clk *clk, struct clk *parent)
{
	u32 reg = cmu_read(DMW_CMU_PLL1CONTROL) & ~(0x03 << 19);

	if (parent == &osc_12m)
		reg |= 0 << 19;
	else if (parent == &osc_20m)
		reg |= 1 << 19;
	else if (parent == &osc_dect)
		reg |= 2 << 19;
	else
		return -EINVAL;

	cmu_write(reg, DMW_CMU_PLL1CONTROL);
	return 0;
}

static struct clk *
get_parent_pll4src(struct clk *clk)
{
	u32 reg = cmu_read(DMW_CMU_PLL4CONTROL);

	switch ((reg >> 19) & 0x03) {
		case 0:
		default:
			return &osc_12m;
		case 1:
			return &osc_20m;
		case 2:
			return &osc_dect;
		case 3:
			return &pll4prediv;
	}
}

static int
set_parent_pll4src(struct clk *clk, struct clk *parent)
{
	u32 reg = cmu_read(DMW_CMU_PLL4CONTROL) & ~(0x03 << 19);

	if (parent == &osc_12m)
		reg |= 0 << 19;
	else if (parent == &osc_20m)
		reg |= 1 << 19;
	else if (parent == &osc_dect)
		reg |= 2 << 19;
	else if (parent == &pll4prediv)
		reg |= 3 << 19;
	else
		return -EINVAL;

	cmu_write(reg, DMW_CMU_PLL4CONTROL);
	return 0;
}

static struct clk *
get_parent_cpuiclk(struct clk *clk)
{
	u32 reg = cmu_read(DMW_CMU_CPUCLKCNTRL);

	switch ((reg >> 16) & 0x03) {
		case 0:
		default:
			return &osc_12m;
		case 1:
			return &osc_20m;
		case 2:
			return &osc_dect;
	}
}

static int
set_parent_cpuiclk(struct clk *clk, struct clk *parent)
{
	u32 reg = cmu_read(DMW_CMU_CPUCLKCNTRL) & ~(0x03 << 16);

	if (parent == &osc_12m)
		reg |= 0 << 16;
	else if (parent == &osc_20m)
		reg |= 1 << 16;
	else if (parent == &osc_dect)
		reg |= 2 << 16;
	else
		return -EINVAL;

	cmu_write(reg, DMW_CMU_CPUCLKCNTRL);
	return 0;
}

static struct clk *
get_parent_cortex(struct clk *clk)
{
	u32 reg = cmu_read(DMW_CMU_CPUCLKCNTRL);

	switch ((reg >> 8) & 0x03) {
		case 0:
		default:
			return &cpuiclk;
		case 1:
			return &pll2;
		case 2:
			return &altsysclk;
	}
}

static int
set_parent_cortex(struct clk *clk, struct clk *parent)
{
	u32 reg = cmu_read(DMW_CMU_CPUCLKCNTRL) & ~(0x03 << 8);

	if (parent == &cpuiclk)
		reg |= 0 << 8;
	else if (parent == &pll2)
		reg |= 1 << 8;
	else if (parent == &altsysclk)
		reg |= 2 << 8;
	else
		return -EINVAL;

	cmu_write(reg, DMW_CMU_CPUCLKCNTRL);
	return 0;
}

static struct clk *
get_parent_altsysclk(struct clk *clk)
{
	u32 reg = cmu_read(DMW_CMU_CPUCLKCNTRL);

	switch ((reg >> 14) & 0x03) {
		case 0:
		default:
			return &osc_12m;
		case 1:
			return &osc_20m;
		case 2:
			return &osc_dect;
		case 3:
			return &osc_32k;
	}
}

static int
set_parent_altsysclk(struct clk *clk, struct clk *parent)
{
	u32 reg = cmu_read(DMW_CMU_CPUCLKCNTRL) & ~(0x03 << 14);

	if (parent == &osc_12m)
		reg |= 0 << 14;
	else if (parent == &osc_20m)
		reg |= 1 << 14;
	else if (parent == &osc_dect)
		reg |= 2 << 14;
	else if (parent == &osc_32k)
		reg |= 3 << 14;
	else
		return -EINVAL;

	cmu_write(reg, DMW_CMU_CPUCLKCNTRL);
	return 0;
}

static struct clk *
get_parent_sysclk(struct clk *clk)
{
	u32 reg = cmu_read(DMW_CMU_CPUCLKCNTRL);

	switch ((reg >> 12) & 0x03) {
		case 0:
		default:
			return &cpuiclk;
		case 1:
			return &altsysclk;
		case 2:
			return &pll1;
	}
}

static int
set_parent_sysclk(struct clk *clk, struct clk *parent)
{
	u32 reg = cmu_read(DMW_CMU_CPUCLKCNTRL) & ~(0x03 << 12);

	if (parent == &cpuiclk)
		reg |= 0 << 12;
	else if (parent == &altsysclk)
		reg |= 1 << 12;
	else if (parent == &pll1)
		reg |= 2 << 12;
	else
		return -EINVAL;

	cmu_write(reg, DMW_CMU_CPUCLKCNTRL);
	return 0;
}

static unsigned long
get_rate_div(struct clk *clk)
{
	unsigned long in = clk->parent->get_rate(clk->parent);
	u32 reg = cmu_read(clk->priv.gate.reg);
	u32 div = (reg >> clk->priv.gate.shift) & ((1 << clk->priv.gate.size)-1);
	return in / (div + 1);
}

static int
set_rate_div(struct clk *clk, unsigned long rate)
{
	unsigned long in = clk->parent->get_rate(clk->parent);
	unsigned long div, reg;

	if (!rate)
		return -EINVAL;

	div = in / rate;
	if (div == 0 || div > (1 << clk->priv.gate.size))
		return -EINVAL;
	if (in / div != rate)
		return -EINVAL;

	div--;

	reg = cmu_read(clk->priv.gate.reg);
	reg &= ~(((1ul << clk->priv.gate.size) - 1ul) << clk->priv.gate.shift);
	reg |= div << clk->priv.gate.shift;
	cmu_write(reg, clk->priv.gate.reg);

	return 0;
}

static long
round_rate_div(struct clk *clk, unsigned long rate)
{
	unsigned long in = clk->parent->get_rate(clk->parent);
	unsigned long div;

	if (!rate)
		return -EINVAL;

	div = (in + rate/2) / rate;
	if (div == 0)
		div = 1;
	if (div > (1 << clk->priv.gate.size))
		div = 1 << clk->priv.gate.size;

	return in / div;
}

static int
get_on_off_divsclk(struct clk *clk)
{
	return !!(cmu_read(DMW_CMU_CLKSWCNTRL) & (1 << 3));
}

static int
set_on_off_divsclk(struct clk *clk, int on)
{
	u32 reg = cmu_read(DMW_CMU_CLKSWCNTRL);

	if (on)
		reg |= 1 << 3;
	else
		reg &= ~(1 << 3);

	cmu_write(reg, DMW_CMU_CLKSWCNTRL);
	return 0;
}

static struct clk *
get_parent_slowclk(struct clk *clk)
{
	if (cmu_read(DMW_CMU_CLKSWCNTRL) & (1 << 16))
		return &osc_32k;
	else
		return &divsclk;
}

static int
set_parent_slowclk(struct clk *clk, struct clk *parent)
{
	u32 reg = cmu_read(DMW_CMU_CLKSWCNTRL) & ~(1 << 16);

	if (parent == &divsclk)
		reg |= 0 << 16;
	else if (parent == &osc_32k)
		reg |= 1 << 16;
	else
		return -EINVAL;

	cmu_write(reg, DMW_CMU_CLKSWCNTRL);
	return 0;
}

static int
set_rate_comaltdiv(struct clk *clk, unsigned long rate)
{
	int ret;

	if (cmu_sem)
		ram_sem_lock(cmu_sem);

	ret = set_rate_div(clk, rate);

	if (cmu_sem)
		ram_sem_unlock(cmu_sem);

	return ret;
}

static struct clk *
get_parent_pll4prediv(struct clk *clk)
{
	u32 reg = cmu_read(DMW_CMU_CLKDIVCNTRL2);

	switch ((reg >> 4) & 0x03) {
		case 0:
			return &pll2;
		case 1:
		default:
			return &pll1;
		case 2:
			return &pll3;
	}
}

static int
set_parent_pll4prediv(struct clk *clk, struct clk *parent)
{
	u32 reg = cmu_read(DMW_CMU_CLKDIVCNTRL2) & ~(0x03 << 4);

	if (parent == &pll2)
		reg |= 0 << 4;
	else if (parent == &pll1)
		reg |= 1 << 4;
	else if (parent == &pll3)
		reg |= 2 << 4;
	else
		return -EINVAL;

	cmu_write(reg, DMW_CMU_CLKDIVCNTRL2);
	return 0;
}

static struct clk *
get_parent_lcdc(struct clk *clk)
{
	u32 reg = cmu_read(DMW_CMU_CLKSWCNTRL);

	switch ((reg >> 17) & 0x03) {
		case 0:
		default:
			return &sysclk;
		case 1:
			return &pll3;
		case 2:
			return &pll4;
	}
}

static int
set_parent_lcdc(struct clk *clk, struct clk *parent)
{
	u32 reg = cmu_read(DMW_CMU_CLKSWCNTRL) & ~(0x03 << 17);

	if (parent == &sysclk)
		reg |= 0 << 17;
	else if (parent == &pll3)
		reg |= 1 << 17;
	else if (parent == &pll4)
		reg |= 2 << 17;
	else
		return -EINVAL;

	cmu_write(reg, DMW_CMU_CLKSWCNTRL);
	return 0;
}

static struct clk *
get_parent_tdm1(struct clk *clk)
{
	u32 reg = cmu_read(DMW_CMU_CLKSWCNTRL);

	switch ((reg >> 19) & 0x03) {
		case 0:
		default:
			return &sysclk;
		case 1:
			return &pll3;
		case 2:
			return &pll4;
	}
}

static int
set_parent_tdm1(struct clk *clk, struct clk *parent)
{
	u32 reg = cmu_read(DMW_CMU_CLKSWCNTRL) & ~(0x03 << 19);

	if (parent == &sysclk)
		reg |= 0 << 19;
	else if (parent == &pll3)
		reg |= 1 << 19;
	else if (parent == &pll4)
		reg |= 2 << 19;
	else
		return -EINVAL;

	cmu_write(reg, DMW_CMU_CLKSWCNTRL);
	return 0;
}

static struct clk *
get_parent_tdm2(struct clk *clk)
{
	u32 reg = cmu_read(DMW_CMU_CLKSWCNTRL);

	switch ((reg >> 21) & 0x03) {
		case 0:
		default:
			return &sysclk;
		case 1:
			return &pll3;
		case 2:
			return &pll4;
	}
}

static int
set_parent_tdm2(struct clk *clk, struct clk *parent)
{
	u32 reg = cmu_read(DMW_CMU_CLKSWCNTRL) & ~(0x03 << 21);

	if (parent == &sysclk)
		reg |= 0 << 21;
	else if (parent == &pll3)
		reg |= 1 << 21;
	else if (parent == &pll4)
		reg |= 2 << 21;
	else
		return -EINVAL;

	cmu_write(reg, DMW_CMU_CLKSWCNTRL);
	return 0;
}

static struct clk *
get_parent_tdm3(struct clk *clk)
{
	u32 reg = cmu_read(DMW_CMU_CLKSWCNTRL);

	switch ((reg >> 23) & 0x03) {
		case 0:
		default:
			return &sysclk;
		case 1:
			return &pll3;
		case 2:
			return &pll4;
	}
}

static int
set_parent_tdm3(struct clk *clk, struct clk *parent)
{
	u32 reg = cmu_read(DMW_CMU_CLKSWCNTRL) & ~(0x03 << 23);

	if (parent == &sysclk)
		reg |= 0 << 23;
	else if (parent == &pll3)
		reg |= 1 << 23;
	else if (parent == &pll4)
		reg |= 2 << 23;
	else
		return -EINVAL;

	cmu_write(reg, DMW_CMU_CLKSWCNTRL);
	return 0;
}

static struct clk *
get_parent_dram(struct clk *clk)
{
	u32 reg = cmu_read(DMW_CMU_CLKSWCNTRL);

	switch (reg & 0x01) {
		case 0:
		default:
			return &osc_12m;
		case 1:
			return &pll3;
	}
}

static int
set_parent_dram(struct clk *clk, struct clk *parent)
{
	u32 reg = cmu_read(DMW_CMU_CLKSWCNTRL) & ~0x01;

	if (parent == &osc_12m)
		reg |= 0;
	else if (parent == &pll3)
		reg |= 1;
	else
		return -EINVAL;

	cmu_write(reg, DMW_CMU_CLKSWCNTRL);
	return 0;
}

static struct clk *
get_parent_css(struct clk *clk)
{
	u32 reg = cmu_read(DMW_CMU_COMCLKSEL);

	switch ((reg >> 10) & 0x03) {
		case 0:
		default:
			return &comaltdiv;
		case 1:
			return &pll4;
		case 2:
			return &pll4src;
	}
}

static int
set_parent_css(struct clk *clk, struct clk *parent)
{
	u32 reg;
	int ret = 0;

	if (cmu_sem)
		ram_sem_lock(cmu_sem);

	reg = cmu_read(DMW_CMU_COMCLKSEL) & ~(0x03 << 10);

	if (parent == &comaltdiv)
		reg |= 0 << 10;
	else if (parent == &pll4)
		reg |= 1 << 10;
	else if (parent == &pll4src)
		reg |= 2 << 10;
	else {
		ret = -EINVAL;
		goto out;
	}

	cmu_write(reg, DMW_CMU_COMCLKSEL);

out:
	if (cmu_sem)
		ram_sem_unlock(cmu_sem);
	return ret;
}

static unsigned long
get_rate_hclk(struct clk *clk)
{
	return get_rate_div(clk) / 2;
}

static int
set_rate_hclk(struct clk *clk, unsigned long rate)
{
	return set_rate_div(clk, rate*2);
}

static long
round_rate_hclk(struct clk *clk, unsigned long rate)
{
	long ret = round_rate_div(clk, rate*2);
	if (ret > 0)
		ret /= 2;

	return ret;
}

/*****************************************************************************
 * Internal helper functions
 ****************************************************************************/

static int
clk_enable_locked(struct clk *clk)
{
	int ret = 0;

	clk->refcnt++;

	if (clk->refcnt == 1) {
		if (clk->parent)
			ret |= clk_enable_locked(clk->parent);

		if (!clk->get_on_off(clk) && clk->set_on_off)
			ret |= clk->set_on_off(clk, 1);

		if (clk->flags & FLAG_BM)
			clk_active_bm_cnt++;
		if (clk->flags & FLAG_IO)
			clk_active_io_cnt++;
	}

	return ret;
}

static void
clk_disable_locked(struct clk *clk)
{
	clk->refcnt--;
	BUG_ON(clk->refcnt < 0);

	if (clk->refcnt == 0) {
		if (clk->get_on_off(clk) && clk->set_on_off)
			clk->set_on_off(clk, 0);

		if (clk->parent)
			clk_disable_locked(clk->parent);

		if (clk->flags & FLAG_BM)
			clk_active_bm_cnt--;
		if (clk->flags & FLAG_IO)
			clk_active_io_cnt--;
	}
}

/*
 * Re-parenting is a bit more involved. It must be divided in two parts because
 * the new parent may have to be enabled before actually switching to it. This
 * requires enabling IRQs and FIQs to keep the latency impact to the system to
 * a minimum.
 */

static int
clk_reparent_locked_prepare(struct clk *clk, struct clk *new_parent)
{
	struct clk *old_parent = clk->parent;
	int ret = 0;

	if (clk->flags & FLAG_REPARENTING)
		return -EBUSY;

	if (old_parent == new_parent)
		goto done;

	if (!clk->set_parent)
		return -EINVAL;

	if (clk->refcnt) {
		ret = clk_enable_locked(new_parent);
		/* make sure we can't be switched off */
		if (ret >= 0)
			ret |= clk_enable_locked(clk);
	}

done:
	if (ret >= 0)
		clk->flags |= FLAG_REPARENTING;
	return ret;
}

static int
clk_reparent_locked_switch(struct clk *clk, struct clk *new_parent)
{
	struct clk *old_parent = clk->parent;
	int ret = 0;

	if (old_parent == new_parent)
		goto done;

	ret = clk->set_parent(clk, new_parent);
	if (ret < 0) {
		/* Revert clk_reparent_locked_prepare() actions */
		if (clk->refcnt) {
			clk_disable_locked(clk);
			clk_disable_locked(new_parent);
		}
		goto done;
	}

	list_move(&clk->child_node, &new_parent->childs);
	clk->parent = new_parent;

	if (clk->refcnt) {
		clk_disable_locked(clk);
		clk_disable_locked(old_parent);
	}

done:
	clk->flags &= ~FLAG_REPARENTING;
	return ret;
}

static int
clk_wait(int flags)
{
	int repeat;

	if (flags <= 0)
		return flags;

	do {
		repeat = 0;

		if (flags & WAIT_PLL_1)
			repeat |= !(cmu_read(DMW_CMU_PLL1CONTROL) & (1 << 18));
		if (flags & WAIT_PLL_2)
			repeat |= !(cmu_read(DMW_CMU_PLL2CONTROL) & (1 << 18));
		if (flags & WAIT_PLL_3)
			repeat |= !(cmu_read(DMW_CMU_PLL3CONTROL) & (1 << 18));
		if (flags & WAIT_PLL_4)
			repeat |= !(cmu_read(DMW_CMU_PLL4CONTROL) & (1 << 18));
		if (flags & WAIT_OSC_DECT)
			repeat |= !(cmu_read(DMW_CMU_CLKOSCCR) & (1 << 4));
	} while (repeat);

	return 0;
}

/*****************************************************************************
 * Public interface
 ****************************************************************************/

int
clk_enable(struct clk *clk)
{
	unsigned long flags;
	int ret = 0;

	if (!clk)
		return -ENODEV;

	raw_local_irq_save(flags);
	local_fiq_disable();

	ret = clk_enable_locked(clk);

	raw_local_irq_restore(flags);

	return clk_wait(ret);
}
EXPORT_SYMBOL(clk_enable);

void clk_disable(struct clk *clk)
{
	unsigned long flags;

	if (!clk)
		return;

	raw_local_irq_save(flags);
	local_fiq_disable();

	clk_disable_locked(clk);

	raw_local_irq_restore(flags);
}
EXPORT_SYMBOL(clk_disable);

long
clk_round_rate(struct clk *clk, unsigned long rate)
{
	unsigned long flags;

	if (!clk)
		return -ENODEV;

	if (clk->round_rate) {
		raw_local_irq_save(flags);
		local_fiq_disable();

		rate = clk->round_rate(clk, rate);

		raw_local_irq_restore(flags);
	}

	return rate;
}
EXPORT_SYMBOL(clk_round_rate);

unsigned long
clk_get_rate(struct clk *clk)
{
	unsigned long rate, flags;

	if (!clk)
		return -ENODEV;

	raw_local_irq_save(flags);
	local_fiq_disable();

	rate = clk->get_rate(clk);

	raw_local_irq_restore(flags);

	return rate;
}
EXPORT_SYMBOL(clk_get_rate);

int clk_set_rate(struct clk *clk, unsigned long rate)
{
	unsigned long flags;
	int ret = -EINVAL;

	if (!clk)
		return -ENODEV;

	if (clk->set_rate) {
		raw_local_irq_save(flags);
		local_fiq_disable();

		ret = clk->set_rate(clk, rate);

		raw_local_irq_restore(flags);
	}

	return clk_wait(ret);
}
EXPORT_SYMBOL(clk_set_rate);

struct clk *clk_get_parent(struct clk *clk)
{
	if (!clk)
		return ERR_PTR(-ENODEV);

	return clk->parent;
}
EXPORT_SYMBOL(clk_get_parent);

int clk_set_parent(struct clk *clk, struct clk *parent)
{
	unsigned long flags;
	int ret;

	if (!clk)
		return -ENODEV;

	raw_local_irq_save(flags);
	local_fiq_disable();
	ret = clk_reparent_locked_prepare(clk, parent);
	raw_local_irq_restore(flags);

	if (ret < 0)
		return ret;
	clk_wait(ret);

	raw_local_irq_save(flags);
	local_fiq_disable();
	ret = clk_reparent_locked_switch(clk, parent);
	raw_local_irq_restore(flags);

	return ret;
}
EXPORT_SYMBOL(clk_set_parent);


/* /proc/clocks information */

static void
clocks_show_clock(struct seq_file *f, struct clk *clk, int indent)
{
	for (; indent>0; indent--)
		seq_printf(f, "\t");

	if (clk->get_on_off(clk)) {
		unsigned long rate = clk_get_rate(clk);
		unsigned long int mhz = rate / 1000000;
		unsigned long int khz = (rate - (mhz * 1000000)) / 1000;

		seq_printf(f, "%s\t refcnt:%d", clk->name, clk->refcnt);
		if (mhz && khz)
			seq_printf(f, " rate:%lu.%03luMHz", mhz, khz);
		else if (mhz && !khz)
			seq_printf(f, " rate:%luMHz", mhz);
		else
			seq_printf(f, " rate:%lukHz", khz);

		if (clk->flags & FLAG_BM)
			seq_printf(f, " +BM");
		if (clk->flags & FLAG_IO)
			seq_printf(f, " +IO");
	} else {
		seq_printf(f, "(%s)\t", clk->name);
		if (clk->refcnt)
			seq_printf(f, " refcnt:%d", clk->refcnt);
	}

	if (clk->getcnt)
		seq_printf(f, " users:%d", clk->getcnt);

	seq_printf(f, "\n");
}

static void
clocks_show_list(struct seq_file *f, struct list_head *list, int indent)
{
	struct clk *clk;

	list_for_each_entry(clk, list, child_node) {
		clocks_show_clock(f, clk, indent);
		clocks_show_list(f, &clk->childs, indent+1);
	}
}

static int clocks_show(struct seq_file *f, void *unused)
{
	seq_printf(f, "Active bus masters: %d\n", clk_active_bm_cnt);
	seq_printf(f, "Active io devices:  %d\n", clk_active_io_cnt);
	seq_printf(f, "DRAM lock depth:    %d\n\n", dmw_dram_reqlevel());
	clocks_show_list(f, &clocks_list, 0);

	return 0;
}

static int clocks_open(struct inode *inode, struct file *file)
{
	return single_open(file, clocks_show, NULL);
}

static const struct file_operations clock_fops = {
	.owner = THIS_MODULE,
	.open = clocks_open,
	.read = seq_read,
	.llseek = seq_lseek,
	.release = single_release,
};

static void clk_init_node(struct clk *clk)
{
	struct clk *child;

	/* First init childs */
	list_for_each_entry(child, &clk->childs, child_node)
		clk_init_node(child);

	/* Enable if requested */
	if (clk->flags & FLAG_ENABLE) {
		/* enable statically */
		clk_enable_locked(clk);
	} else if (!clk->refcnt && clk->get_on_off(clk)) {
		if (clk->flags & FLAG_DONT_DISABLE) {
			/* propagate flag upwards */
			struct clk *parent = clk->parent;
			while (parent) {
				parent->flags |= FLAG_DONT_DISABLE;
				parent = parent->parent;
			}
		} else if (clk->set_on_off)
			clk->set_on_off(clk, 0);
	}
}

static void
reset_set_onoff(int id, int on)
{
	int tmp, bit = id, reg = DMW_CMU_SWCHWRST2;

	if ((id < 0) || (id > 63))
		return;
	else if (bit < 32)
		reg = DMW_CMU_SWCHWRST1;
	else
		bit -= 32;

	if (cmu_sem)
		ram_sem_lock(cmu_sem);

	tmp = cmu_read(reg);
	if (on)
		tmp |=  (1 << bit);
	else
		tmp &= ~(1 << bit);
	cmu_write(WRPR_UNLOCK_RST_EN, DMW_CMU_WRPR);
	cmu_write(tmp, reg);

	if (cmu_sem)
		ram_sem_unlock(cmu_sem);
}

void
reset_set(int id)
{
	reset_set_onoff(id, 1);
}
EXPORT_SYMBOL(reset_set);

void
reset_release(int id)
{
	reset_set_onoff(id, 0);
}
EXPORT_SYMBOL(reset_release);

int
dmw_get_wdt_reset_ind(void)
{
	return !!(cmu_read(DMW_CMU_RSTSR) & (1<<3));
}

void dmw_clk_print_active(void)
{
	int i;

	printk("Active IO/BM clocks:\n");

	for (i = 0; i < ARRAY_SIZE(onchip_clks); i++) {
		struct clk *clk = onchip_clks[i];

		if ((clk->flags & (FLAG_IO|FLAG_BM)) && clk->refcnt > 0)
			printk("  %s%s\n",
			       clk->name,
			       clk->flags & FLAG_BM ? " (BM)" : "");
	}
}

int __init dmw_clk_init(void)
{
	int i;
	struct clk *clk;
	unsigned long tmp;

	/* Reset SD/MMC block */
	tmp = cmu_read(DMW_CMU_SWCHWRST1);
	tmp |= (1<<13);
	cmu_write(WRPR_UNLOCK_RST_EN, DMW_CMU_WRPR);
	cmu_write(tmp, DMW_CMU_SWCHWRST1);
	udelay(1000);

	/* Release reset of all blocks (except for USB & CSS) */
	tmp = cmu_read(DMW_CMU_SWCHWRST1);
	tmp &= ~(0x0778ffdd);
	cmu_write(WRPR_UNLOCK_RST_EN, DMW_CMU_WRPR);
	cmu_write(tmp, DMW_CMU_SWCHWRST1);
	tmp = cmu_read(DMW_CMU_SWCHWRST2);
	tmp &= ~(0x003417f3);
	cmu_write(WRPR_UNLOCK_RST_EN, DMW_CMU_WRPR);
	cmu_write(tmp, DMW_CMU_SWCHWRST2);

	/* Round 1: set up parents */
	for (i = 0; i < ARRAY_SIZE(onchip_clks); i++) {
		clk = onchip_clks[i];

		/*
		 * FIXME: here we statically enable all bus masters and io
		 * devices to prevent a race condition with the other
		 * arm926 core (missing mutex when accessing CMU)
		 */
		if (clk->flags & (FLAG_BM | FLAG_IO))
			clk->flags |= FLAG_ENABLE;

		INIT_LIST_HEAD(&clk->childs);
		if (clk->get_parent)
			clk->parent = clk->get_parent(clk);
	}

	/* Round 2: set up child lists */
	for (i = 0; i < ARRAY_SIZE(onchip_clks); i++) {
		clk = onchip_clks[i];

		if (clk->parent)
			list_add(&clk->child_node, &clk->parent->childs);
		else
			list_add(&clk->child_node, &clocks_list);
	}

	/* Round 3: enable the clocks needed during early bootup, disable unneeded ones */
	list_for_each_entry(clk, &clocks_list, child_node)
		clk_init_node(clk);

	/* Add clock mappings for devices */
	for (i = 0; i < ARRAY_SIZE(clk_mappings); i++)
		clkdev_add(&clk_mappings[i]);

	/* The mappings for the cpufreq driver are only added if PLL4 is NOT
	 * running from PLL2! */
	if (!(pll4src.parent == &pll4prediv && pll4prediv.parent == &pll2)) {
		for (i = 0; i < ARRAY_SIZE(clk_mappings_cpufreq); i++)
			clkdev_add(&clk_mappings_cpufreq[i]);
	}

	return 0;
}

static int __init clk_init_late(void)
{
	cmu_sem = ioremap_nocache(DMW_SRAM_CMU_SEM_BASE, DMW_SRAM_CMU_SEM_SIZE);
	if (cmu_sem)
		ram_sem_init(cmu_sem);
	else
		printk(KERN_ERR "%s(): failed to map CMU semaphore\n", __func__);

	/* Create proc entry */
	proc_create_data("clocks", S_IRUGO, NULL, &clock_fops, NULL);

	return 0;
}

arch_initcall(clk_init_late);

