/*
 *  linux/arch/arm/mach-dmw/board-imh3.c
 *
 *  Copyright (C) 2011 DSPG Technologies GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/interrupt.h>
#include <linux/cpufreq.h>
#include <linux/delay.h>
#include <linux/dma-mapping.h>
#include <linux/dwf_platform.h>
#include <linux/gpio_vibrator.h>
#include <linux/i2c-pnx.h>
#include <linux/i2c.h>
#include <linux/i2c/ssd2531-ts.h>
#include <linux/init.h>
#include <linux/lis3lv02d.h>
#include <linux/mfd/core.h>
#include <linux/mfd/dp52/battery.h>
#include <linux/mfd/dp52/core.h>
#include <linux/mmc/dw_mmc.h>
#include <linux/mmc/host.h>
#include <linux/platform_device.h>
#include <linux/regulator/machine.h>
#include <linux/spi/spi.h>
#include <linux/spi/spi_dmw96.h>
#include <linux/usb/dwc_otg.h>
#include <mach/cpufreq.h>
#include <mach/gpio.h>
#include <mach/headsetdet.h>
#include <mach/irqs.h>
#include <mach/keypad.h>
#include <mach/reset.h>
#include <media/dmw96ciu.h>
#include <media/dmw96ciu.h>
#include <media/dmw96ciu_w.h>
#if defined (CONFIG_VIDEO_CAMERA_OV3640)
#include <media/ov3640.h>
#elif defined (CONFIG_VIDEO_CAMERA_OV2643)
#include <media/ov2643.h>
#endif
#include <media/v4l2-int-device.h>
#include <sound/dmw-tdm.h>
#include <sound/dmw-boards.h>
#include <video/amoled.h>
#include <video/dmw96osdm_common.h>
#include <video/dw74fb.h>
#if defined(CONFIG_USB_ANDROID) || defined(CONFIG_USB_ANDROID_MODULE)
#include <linux/usb/android_composite.h>
#endif
#include <linux/i2c/adv7525_hdmi.h>

#include "devices.h"
#include "gpio-setup.h"
#include "pm.h"

#define OVERDRIVE_GPIO (GPIO_PORTG(9))

static __initdata struct gpio_setup_t gpio_setup_default[] = {
	GPIO_SETUP(A, 0,  GPIO_DISABLE,  GPIO_PULL_UP      ), /* NFL - NFLD0 */
	GPIO_SETUP(A, 1,  GPIO_DISABLE,  GPIO_PULL_UP      ), /* NFL - NFLD1 */
	GPIO_SETUP(A, 2,  GPIO_DISABLE,  GPIO_PULL_UP      ), /* NFL - NFLD2 */
	GPIO_SETUP(A, 3,  GPIO_DISABLE,  GPIO_PULL_UP      ), /* NFL - NFLD3 */
	GPIO_SETUP(A, 4,  GPIO_DISABLE,  GPIO_PULL_UP      ), /* NFL - NFLD4 */
	GPIO_SETUP(A, 5,  GPIO_DISABLE,  GPIO_PULL_UP      ), /* NFL - NFLD5 */
	GPIO_SETUP(A, 6,  GPIO_DISABLE,  GPIO_PULL_UP      ), /* NFL - NFLD6 */
	GPIO_SETUP(A, 7,  GPIO_DISABLE,  GPIO_PULL_UP      ), /* NFL - NFLD7 */
	GPIO_SETUP(A, 8,  GPIO_DISABLE,  GPIO_PULL_UP      ), /* NFL - NFLD8 */
	GPIO_SETUP(A, 9,  GPIO_DISABLE,  GPIO_PULL_UP      ), /* NFL - NFLD9 */
	GPIO_SETUP(A, 10, GPIO_DISABLE,  GPIO_PULL_UP      ), /* NFL - NFLD10*/
	GPIO_SETUP(A, 11, GPIO_DISABLE,  GPIO_PULL_UP      ), /* NFL - NFLD11 */
	GPIO_SETUP(A, 12, GPIO_DISABLE,  GPIO_PULL_UP      ), /* NFL - NFLD12 */
	GPIO_SETUP(A, 13, GPIO_DISABLE,  GPIO_PULL_UP      ), /* NFL - NFLD13 */
	GPIO_SETUP(A, 14, GPIO_DISABLE,  GPIO_PULL_UP      ), /* NFL - NFLD14 */
	GPIO_SETUP(A, 15, GPIO_DISABLE,  GPIO_PULL_UP      ), /* NFL - NFLD15 */
	GPIO_SETUP(A, 16, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* NFL - NFLWR */
	GPIO_SETUP(A, 17, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* NFL - NFLRD */
	GPIO_SETUP(A, 19, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* NFL - NFLCLE */
	GPIO_SETUP(A, 20, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* NFL - NFLALE */
	GPIO_SETUP(A, 21, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* NFL - NFLCS0 */
	GPIO_SETUP(E, 31, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* NFL - NFLREADY2 */

	GPIO_SETUP(B, 0,  GPIO_DISABLE,  GPIO_PULL_UP      ), /* UART1 - TX */
	GPIO_SETUP(B, 1,  GPIO_DISABLE,  GPIO_PULL_DOWN    ), /* UART1 - RX */
	GPIO_SETUP(B, 4,  GPIO_DISABLE,  GPIO_PULL_UP      ), /* UART2 - TX */
	GPIO_SETUP(B, 5,  GPIO_DISABLE,  GPIO_PULL_DOWN    ), /* UART2 - RX */

	GPIO_SETUP(B, 12, GPIO_DISABLE,  GPIO_PULL_DOWN    ), /* TDM1 - SCLK */
	GPIO_SETUP(B, 13, GPIO_DISABLE,  GPIO_PULL_DOWN    ), /* TDM1 - Fsync */
	GPIO_SETUP(B, 14, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* TDM1 - TX */
	GPIO_SETUP(B, 15, GPIO_DISABLE,  GPIO_PULL_DOWN    ), /* TDM1 - RX */
	GPIO_SETUP(B, 16, GPIO_DISABLE,  GPIO_PULL_DOWN    ), /* TDM2 - SCLK */
	GPIO_SETUP(B, 17, GPIO_DISABLE,  GPIO_PULL_DOWN    ), /* TDM2 - Fsync */
	GPIO_SETUP(B, 18, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* TDM2 - TX */
	GPIO_SETUP(B, 19, GPIO_DISABLE,  GPIO_PULL_DOWN    ), /* TDM2 - RX */
	GPIO_SETUP(B, 20, GPIO_DISABLE,  GPIO_PULL_DOWN    ), /* TDM3 - SCLK */
	GPIO_SETUP(B, 21, GPIO_DISABLE,  GPIO_PULL_DOWN    ), /* TDM3 - Fsync */
	GPIO_SETUP(B, 22, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* TDM3 - TX */
	GPIO_SETUP(B, 23, GPIO_DISABLE,  GPIO_PULL_DOWN    ), /* TDM3 - RX */

	GPIO_SETUP(B, 28, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* SPI1 - DO */
	GPIO_SETUP(B, 29, GPIO_DISABLE,  GPIO_PULL_DOWN    ), /* SPI1 - DI */
	GPIO_SETUP(B, 30, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* SPI1 - CS1 */
	GPIO_SETUP(B, 31, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* SPI1 - CK */
	GPIO_SETUP(G, 21, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* SPI1 - CS2 */

	GPIO_SETUP(D, 0,  GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - data0 */
	GPIO_SETUP(D, 1,  GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - data1 */
	GPIO_SETUP(C, 0,  GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - data2 */
	GPIO_SETUP(C, 1,  GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - data3 */
	GPIO_SETUP(C, 2,  GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - data4 */
	GPIO_SETUP(C, 3,  GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - data5 */
	GPIO_SETUP(C, 4,  GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - data6 */
	GPIO_SETUP(C, 5,  GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - data7 */
	GPIO_SETUP(D, 2,  GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - data8 */
	GPIO_SETUP(D, 3,  GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - data9 */
	GPIO_SETUP(C, 6,  GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - data10 */
	GPIO_SETUP(C, 7,  GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - data11 */
	GPIO_SETUP(C, 8,  GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - data12 */
	GPIO_SETUP(C, 9,  GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - data13 */
	GPIO_SETUP(C, 10, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - data14 */
	GPIO_SETUP(C, 11, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - data15 */
	GPIO_SETUP(D, 4,  GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - data16 */
	GPIO_SETUP(D, 5,  GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - data17 */
	GPIO_SETUP(C, 12, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - data18 */
	GPIO_SETUP(C, 13, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - data19 */
	GPIO_SETUP(C, 14, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - data20 */
	GPIO_SETUP(C, 15, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - data21 */
	GPIO_SETUP(C, 16, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - data22 */
	GPIO_SETUP(C, 17, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - data23 */
	GPIO_SETUP(C, 18, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - pixel clock */
	GPIO_SETUP(C, 19, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - VSYNC */
	GPIO_SETUP(C, 20, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - HSYNC */
	GPIO_SETUP(C, 21, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* LCDC - ENABLE */

	GPIO_SETUP(C, 28, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* IIC1 - clock */
	GPIO_SETUP(C, 29, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* IIC1 - data */
	GPIO_SETUP(C, 30, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* IIC2 - clock */
	GPIO_SETUP(C, 31, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* IIC2 - data */

	GPIO_SETUP(D, 6,  GPIO_DISABLE,  GPIO_PULL_UP      ), /* CAM - data0 */
	GPIO_SETUP(D, 7,  GPIO_DISABLE,  GPIO_PULL_UP      ), /* CAM - data1 */
	GPIO_SETUP(D, 8,  GPIO_DISABLE,  GPIO_PULL_UP      ), /* CAM - data2 */
	GPIO_SETUP(D, 9,  GPIO_DISABLE,  GPIO_PULL_UP      ), /* CAM - data3 */
	GPIO_SETUP(D, 10, GPIO_DISABLE,  GPIO_PULL_UP      ), /* CAM - data4 */
	GPIO_SETUP(D, 11, GPIO_DISABLE,  GPIO_PULL_UP      ), /* CAM - data5 */
	GPIO_SETUP(D, 12, GPIO_DISABLE,  GPIO_PULL_UP      ), /* CAM - data6 */
	GPIO_SETUP(D, 13, GPIO_DISABLE,  GPIO_PULL_UP      ), /* CAM - data7 */
	GPIO_SETUP(D, 14, GPIO_DISABLE,  GPIO_PULL_UP      ), /* CAM - pixel clock */
	GPIO_SETUP(D, 15, GPIO_DISABLE,  GPIO_PULL_UP      ), /* CAM - hsync */
	GPIO_SETUP(D, 16, GPIO_DISABLE,  GPIO_PULL_UP      ), /* CAM - vsync */
	GPIO_SETUP(D, 17, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* CAM - 25MHz clock */
	GPIO_SETUP(G, 14, GPIO_OUT_LOW,  GPIO_PULL_DISABLE ), /* Camera 3MP - reset */
	GPIO_SETUP(F, 5,  GPIO_OUT_HIGH, GPIO_PULL_DISABLE ), /* Camera 3MP - PDN */
	GPIO_SETUP(F, 22, GPIO_OUT_LOW,  GPIO_PULL_DISABLE ), /* Camera 5MP - reset */
	GPIO_SETUP(F, 11, GPIO_OUT_HIGH, GPIO_PULL_DISABLE ), /* Camera 5MP - PDN */
	GPIO_SETUP(B, 27, GPIO_OUT_LOW,  GPIO_PULL_DISABLE ), /* Camera 5MP - autofocus PDN - active low */
	GPIO_SETUP(F, 13, GPIO_OUT_LOW,  GPIO_PULL_DISABLE ), /* Flash light - GPIO1 */
	GPIO_SETUP(F, 14, GPIO_OUT_LOW,  GPIO_PULL_DISABLE ), /* Flash light - GPIO2 */
	GPIO_SETUP(G, 18, GPIO_OUT_LOW,  GPIO_PULL_DISABLE ), /* Flash light - enable */

	GPIO_SETUP(D, 18, GPIO_DISABLE,  GPIO_PULL_UP      ), /* SDMMC - data0 */
	GPIO_SETUP(D, 19, GPIO_DISABLE,  GPIO_PULL_UP      ), /* SDMMC - data1 */
	GPIO_SETUP(D, 20, GPIO_DISABLE,  GPIO_PULL_UP      ), /* SDMMC - data2 */
	GPIO_SETUP(D, 21, GPIO_DISABLE,  GPIO_PULL_UP      ), /* SDMMC - data3 */
	GPIO_SETUP(D, 22, GPIO_DISABLE,  GPIO_PULL_UP      ), /* SDMMC - clock */
	GPIO_SETUP(D, 23, GPIO_DISABLE,  GPIO_PULL_UP      ), /* SDMMC - command */
	GPIO_SETUP(F, 19, GPIO_IN,       GPIO_PULL_DISABLE ), /* SDMMC - detect */

	GPIO_SETUP(D, 28, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* DP - PWM_HILO0 */
	GPIO_SETUP(D, 29, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* DP - PWM_HILO1 */
	GPIO_SETUP(D, 30, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* DP - clock */
	GPIO_SETUP(F, 18, GPIO_IN,       GPIO_PULL_DISABLE ), /* DP - INT */

	GPIO_SETUP(F, 0,  GPIO_DISABLE,  GPIO_PULL_UP      ), /* KPD - row 0 */
	GPIO_SETUP(F, 1,  GPIO_DISABLE,  GPIO_PULL_UP      ), /* KPD - row 1 */
	GPIO_SETUP(F, 2,  GPIO_DISABLE,  GPIO_PULL_UP      ), /* KPD - row 2 */
	GPIO_SETUP(F, 3,  GPIO_DISABLE,  GPIO_PULL_UP      ), /* KPD - row 3 */
	GPIO_SETUP(F, 8,  GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* KPD - col 0 */
	GPIO_SETUP(F, 9,  GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* KPD - col 1 */

	GPIO_SETUP(F, 24, GPIO_IN,       GPIO_PULL_DISABLE ), /* HDMI - interrupt */
	GPIO_SETUP(G, 11, GPIO_OUT_HIGH, GPIO_PULL_DISABLE ), /* HDMI - power */
	GPIO_SETUP(A, 25, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* HDMI - CEC clock */

	GPIO_SETUP(C, 26, GPIO_OUT_LOW,  GPIO_PULL_DISABLE ), /* Audio PA - shutdown */
	GPIO_SETUP(F, 15, GPIO_OUT_LOW,  GPIO_PULL_DISABLE ), /* Audio PA - gain select */

	GPIO_SETUP(F, 16, GPIO_IN,       GPIO_PULL_DOWN    ), /* Accelerometer - Int1 */
	GPIO_SETUP(F, 17, GPIO_IN,       GPIO_PULL_DOWN    ), /* Accelerometer - Int2 */

	GPIO_SETUP(G, 17, GPIO_OUT_HIGH, GPIO_PULL_DISABLE ), /* Touch panel - reset */
	GPIO_SETUP(F, 21, GPIO_IN,       GPIO_PULL_UP      ), /* Touch panel - interrupt */

	GPIO_SETUP(A, 26, GPIO_OUT_HIGH, GPIO_PULL_DISABLE ), /* PHY - reset */
	GPIO_SETUP(E, 0,  GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* MII - TXD0 */
	GPIO_SETUP(E, 1,  GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* MII - TXD1 */
	GPIO_SETUP(E, 2,  GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* MII - TXEN */
	GPIO_SETUP(E, 3,  GPIO_DISABLE,  GPIO_PULL_DOWN    ), /* MII - RXD0 */
	GPIO_SETUP(E, 4,  GPIO_DISABLE,  GPIO_PULL_DOWN    ), /* MII - RXD1 */
	GPIO_SETUP(E, 5,  GPIO_DISABLE,  GPIO_PULL_DOWN    ), /* MII - CRS_DV */
	GPIO_SETUP(E, 6,  GPIO_DISABLE,  GPIO_PULL_DOWN    ), /* MII - RXER */
	GPIO_SETUP(E, 7,  GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* MII - REFCK */
	GPIO_SETUP(E, 8,  GPIO_DISABLE,  GPIO_PULL_UP      ), /* MII - MDIO */
	GPIO_SETUP(E, 9,  GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* MII - MDC */
	GPIO_SETUP(E, 10, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* MII - TXER */
	GPIO_SETUP(E, 11, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* MII - TXD2 */
	GPIO_SETUP(E, 12, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* MII - TXD3 */
	GPIO_SETUP(E, 13, GPIO_DISABLE,  GPIO_PULL_DOWN    ), /* MII - RXD2 */
	GPIO_SETUP(E, 14, GPIO_DISABLE,  GPIO_PULL_DOWN    ), /* MII - RXD3 */
	GPIO_SETUP(E, 15, GPIO_DISABLE,  GPIO_PULL_DOWN    ), /* MII - RXCLK */
	GPIO_SETUP(E, 16, GPIO_DISABLE,  GPIO_PULL_DOWN    ), /* MII - RXCRS */
	GPIO_SETUP(E, 17, GPIO_DISABLE,  GPIO_PULL_DOWN    ), /* MII - COL */

	GPIO_SETUP(F, 26, GPIO_OUT_HIGH, GPIO_PULL_DISABLE ), /* USB - VBUS enable */
	GPIO_SETUP(F, 20, GPIO_IN,       GPIO_PULL_DISABLE ), /* USB - detect */
	GPIO_SETUP(A, 28, GPIO_IN,       GPIO_PULL_DOWN    ), /* USB - port type detect */
	GPIO_SETUP(A, 29, GPIO_IN,       GPIO_PULL_DOWN    ), /* USB - port type FET control */

	GPIO_SETUP(D, 31, GPIO_OUT_LOW,  GPIO_PULL_DISABLE ), /* DECT - power */

	GPIO_SETUP(F, 25, GPIO_OUT_LOW,  GPIO_PULL_DISABLE ), /* Vibrator - enable */

	GPIO_SETUP(G, 15, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* Wi-Fi Baseband/RF - SW1 */
	GPIO_SETUP(G, 16, GPIO_DISABLE,  GPIO_PULL_DISABLE ), /* Wi-Fi Baseband/RF - SW1B */
};

static struct resource dw_mmc_resource[] = {
	{
		.start = DMW_SDMMC_BASE,
		.end   = DMW_SDMMC_BASE + SZ_4K - 1,
		.flags = IORESOURCE_MEM,
	},
	{
		.start = DMW_IID_SDMMC_CONTROLLER,
		.flags = IORESOURCE_IRQ,
	},
};

static int dw_mmc_init(u32 slot_id, irq_handler_t isr, void *dev_id)
{
	enable_irq_wake(DMW_IID_EXTERNAL_REQUEST_3);
	return request_irq(DMW_IID_EXTERNAL_REQUEST_3, isr, IRQF_DISABLED |
	                   IRQF_TRIGGER_RISING | IRQF_TRIGGER_FALLING,
	                   "dw-mci cdetect", dev_id);
}

static int dw_mmc_get_ro(u32 slot_id)
{
	/* always read-write */
	return 0;
}

static int dw_mmc_get_cd(u32 slot_id)
{
	return gpio_get_value(GPIO_PORTF(19));
}

static struct dw_mci_board_slot dw_mmc_slots[] = {
	{
		.id = 0,
		.caps = MMC_CAP_4_BIT_DATA,
		.init = dw_mmc_init,
		.get_ro = dw_mmc_get_ro,
		.get_cd = dw_mmc_get_cd,
	},
};

static struct block_settings dw_mmc_blk_settings = {
	.max_segs = 64,
	.max_blk_size = 65536,
	.max_blk_count = 0xffffffff,
	.max_req_size = 0xffffffff,
	.max_seg_size = 4096,
};

static struct dw_mci_board dw_mmc_pdata = {
	.num_slots = 1,
	.slots = dw_mmc_slots,
	.detect_delay_ms = 500,
	.bus_hz = 50000000,
	.blk_settings = &dw_mmc_blk_settings,
	.fifo_depth = 0x20,
	.quirks = DW_MCI_QUIRK_SBE_NEED_STOP | DW_MCI_QUIRK_HIGHSPEED,
};

static struct platform_device sdmmc_device = {
	.name = "dw_mmc",
	.dev.platform_data = &dw_mmc_pdata,
	.dev.coherent_dma_mask = DMA_BIT_MASK(32),
	.num_resources = ARRAY_SIZE(dw_mmc_resource),
	.resource = dw_mmc_resource,
};

static struct gpio_vibrator_pdata gpio_vibrator_pdata = {
	.onoff = GPIO_PORTF(25),
};

static struct platform_device vibrator_device = {
	.name = "gpio-vibrator",
	.dev.platform_data = &gpio_vibrator_pdata,
};

static struct regulator_consumer_supply spare2_consumers[] = {
	{
		.dev = &sdmmc_device.dev,
		.supply = "vmmc0",
	},
};

static struct regulator_consumer_supply spare3_consumers[] = {
};

static struct regulator_init_data spare2_data = {
	.constraints = {
		.min_uV = 1500000,
		.max_uV = 3600000,
		.apply_uV = 1,
		.valid_modes_mask = REGULATOR_MODE_NORMAL,
		.valid_ops_mask = REGULATOR_CHANGE_STATUS |
		                  REGULATOR_CHANGE_VOLTAGE,
	},
	.num_consumer_supplies = ARRAY_SIZE(spare2_consumers),
	.consumer_supplies = spare2_consumers,
};

static struct regulator_init_data spare3_data = {
	.constraints = {
		.min_uV = 1800000,
		.max_uV = 1800000,
		.always_on = 1, /* connected to cameras */
		.apply_uV = 1,
		.valid_modes_mask = REGULATOR_MODE_NORMAL,
		.valid_ops_mask = REGULATOR_CHANGE_STATUS,
	},
	.num_consumer_supplies = ARRAY_SIZE(spare3_consumers),
	.consumer_supplies = spare3_consumers,
};

static struct resource dp52_rtc_resources[] = {
	{
		.name  = "rtc",
		.start = DP52_IRQ_RTC,
		.end   = DP52_IRQ_RTC,
		.flags = IORESOURCE_IRQ,
	},
};

static struct resource dp52_pmu_resources[] = {
	{
		.name  = "pmu",
		.start = DP52_IRQ_PMUEN,
		.end   = DP52_IRQ_PMUEN,
		.flags = IORESOURCE_IRQ,
	},
};

static struct resource dp52_bat_resources[] = {
	{
		.name  = "auxcomp",
		.start = DP52_IRQ_AUXCOMP,
		.end   = DP52_IRQ_AUXCOMP,
		.flags = IORESOURCE_IRQ,
	},
};

static struct dp52_bat_pdata dp52_bat_pdata = {
	.vbat_dcin	= 3,
	.vbat_r1	= 1000, /* 1MOhm */
	.vbat_r2	= 330,  /* 330kOhm */
	.ichg_dcin	= 1,
	.ichg_r		= 50, /* 50 mOhm */
	.temp_dcin	= 0,
	.temp_r		= 39000, /* 39kOhm */
};

static struct dwheadset_config dp52_hsdet_pdata = {
	.switch_platform_data = {
		.name = "h2w",
		.name_on = "ON",
		.name_off = "OFF",
		.state_on = NULL,
		.state_off = NULL,
	},
};

static struct mfd_cell dp52_cells[] = {
	{
		.name = "dp52-rtc",
		.num_resources = ARRAY_SIZE(dp52_rtc_resources),
		.resources = dp52_rtc_resources,
	},
	{
		.name = "dp52-onkey",
		.num_resources = ARRAY_SIZE(dp52_pmu_resources),
		.resources = dp52_pmu_resources,
	},
	{
		.name = "dp52-power",
		.num_resources = ARRAY_SIZE(dp52_pmu_resources),
		.resources = dp52_pmu_resources,
	},
	{
		.name = "dp52-battery",
		.num_resources = ARRAY_SIZE(dp52_bat_resources),
		.resources = dp52_bat_resources,
		.mfd_data = &dp52_bat_pdata,
	},
	{
		.name = "dp52-hsdet",
		.num_resources = ARRAY_SIZE(dp52_bat_resources),
		.resources = dp52_bat_resources,
		.mfd_data = &dp52_hsdet_pdata,
	},
};

static struct dp52_platform_data dp52_pdata = {
	.irq = DMW_IID_EXTERNAL_REQUEST_2,
	.irq_base = 64,

	.spare2 = &spare2_data,
	.spare3 = &spare3_data,

	.cells     = dp52_cells,
	.num_cells = ARRAY_SIZE(dp52_cells),
};

static struct amoled_platform_data amoled_pdata = {
	.reset = GPIO_PORTF(23),
};

static struct spi_board_info imh3_spi_devs[] = {
	{
		/* DP52 */
		.modalias	= "dp52",
		.chip_select	= 0,
		.max_speed_hz	= 5 * 1000 * 1000,
		.bus_num	= 0,
		.platform_data	= &dp52_pdata,
	},
	{
		/* AMOLED display */
		.modalias	= "dmw96_amoled_spi",
		.chip_select	= 1,
		.max_speed_hz	= 5 * 1000 * 1000,
		.bus_num	= 0,
		.platform_data	= &amoled_pdata,
	},
};

static struct dmw96_spi_info dmw96_spi_info = {
	.bus_num	= 0,
	.set_cs		= NULL,
};

static struct resource dmw96_spi_resources[] = {
	{
		.start  = DMW_SPI1_BASE,
		.end    = DMW_SPI1_BASE + 0x200 - 1,
		.flags  = IORESOURCE_MEM,
	},
	{
		.start  = DMW_IID_SPI1,
		.end    = DMW_IID_SPI1,
		.flags  = IORESOURCE_IRQ,
	},
	{
		.start  = 16, /* SPI1 RX */
		.end    = 16,
		.flags  = IORESOURCE_DMA,
	},
	{
		.start  = 15, /* SPI1 TX */
		.end    = 15,
		.flags  = IORESOURCE_DMA,
	},
};

static struct platform_device spi_device = {
	.name = "dmw96-spi",
	.id = 0,
	.num_resources = ARRAY_SIZE(dmw96_spi_resources),
	.resource = dmw96_spi_resources,
	.dev = {
		.platform_data = &dmw96_spi_info
	},
};

static struct dmw96cam_hw_config *hwc;

static struct dmw96cam_sensor_config ov_hwc = {
	.xclk = DMW96CAM_XCLK_B,
};

#if defined(CONFIG_VIDEO_CAMERA_OV3640) || defined(CONFIG_VIDEO_CAMERA_OV2643)
static int sensor_set_prv_data(void *priv)
{
	hwc = priv;
	hwc->u.sensor.xclk = ov_hwc.xclk;
	hwc->dev_index = 1;
	hwc->dev_minor = 4;
	hwc->dev_type = DMW96CAM_SLAVE_SENSOR;
	return 0;
}

static int sensor_power_set(enum v4l2_power power)
{
	static enum v4l2_power previous_power = V4L2_POWER_OFF;
	static char requested = 0;

	if (!requested) {
		gpio_request(GPIO_PORTF(5), "CAM_PWDN");
		gpio_request(GPIO_PORTG(14), "CAM_RESETB");
		requested = 1;
	}

	switch (power) {
	case V4L2_POWER_ON:
		ciu_configure_intreface();

		gpio_set_value(GPIO_PORTF(5), 0); /* power up */
		if (previous_power == V4L2_POWER_OFF) {
			/* reset */
			udelay(100);
			gpio_set_value(GPIO_PORTG(14), 0);
			udelay(100);
			gpio_set_value(GPIO_PORTG(14), 1);
		}
		break;

	case V4L2_POWER_OFF:
		/* Cannot power off on IMH3. Just assert power down. */
		gpio_set_value(GPIO_PORTF(5), 1);
		break;

	case V4L2_POWER_STANDBY:
		gpio_set_value(GPIO_PORTF(5), 1);
		break;

	}
	previous_power = power;

	return 0;
}

static u32 sensor_set_xclk(u32 xclkfreq)
{
	return 0;
}
#endif

#if defined (CONFIG_VIDEO_CAMERA_OV3640)
static struct ov3640_platform_data ov3640_platform_data = {
	.power_set	= sensor_power_set,
	.priv_data_set	= sensor_set_prv_data,
	.set_xclk	= sensor_set_xclk,
};
#elif defined (CONFIG_VIDEO_CAMERA_OV2643)
static struct ov2643_platform_data ov2643_platform_data = {
	.power_set	= sensor_power_set,
	.priv_data_set	= sensor_set_prv_data,
	.set_xclk	= sensor_set_xclk,
};
#endif

static struct resource dmw96ciu_resources[] = {
	{
		.start = DMW_CIU_BASE,
		.end   = DMW_CIU_BASE + SZ_4K - 1,
		.flags = IORESOURCE_MEM,
	},
	{
		.start = DMW_IID_CIU,
		.flags = IORESOURCE_IRQ,
		.name  = "dmw96ciu-irq",
	},
};

static void ciu_reset(int enable)
{
	if (enable)
		reset_set(RESET_CIU);
	else
		reset_release(RESET_CIU);
};

static struct ciu_pdata ciu_pdata = {
	.ciu_reset = ciu_reset,
};

static struct platform_device ciu_device = {
	.name = "dmw96ciu",
	.id		= 0,
	.dev		= {
		.platform_data = &ciu_pdata,
		.coherent_dma_mask = ~0,
	},
	.num_resources = ARRAY_SIZE(dmw96ciu_resources),
	.resource = dmw96ciu_resources,
};

static struct dw74fb_panel dmw_wvga_amoled_panel = {
	.name = "amoled",
	.xres = 480,
	.yres = 800,
	.bits_per_pixel = 18,
	.is_cpu_type = 0,
	.clock_rate = 27000000,
	.height_mm = 50,
	.width_mm = 80,
	.osdm_config = OSDM_DISPLAY_RGB24_888_RGB,
	.init_seq = {
		/* LCDC Disable */
		{LCDC_REG_LCDCCR, 0x0000},

		/* Display pannel */
		{LCDC_REG_INTER, 0x000c},
		{LCDC_REG_DISPIR, 0x0001}, /* DISPTYPE = 24bit TFT (and not 18) since 18bits of LCD are connected to MSB bits of DMW96 */

		{LCDC_REG_PANCSR, 0x000F},
		{LCDC_REG_DISPCR, 0x0001},
		{LCDC_REG_OSDMCR, 0x0000},

		/* Background color and brightness */
		{LCDC_REG_BACKCPR, 0x0000},
		{LCDC_REG_YCLPCR, 0xff00},
		{LCDC_REG_CCLPCR, 0xff00},
		{LCDC_REG_CLPER, 0x0000},
		{LCDC_REG_DISPIDXR, 0x0000},

		{LCDC_REG_PALR0, 0x0000},
		{LCDC_REG_PALR1, 0x0000},
		{LCDC_REG_PALR2, 0x0000},
		{LCDC_REG_PALR3, 0x0000},
		{LCDC_REG_ISTAR4, 0x4009},

		/* GP_SELR 0 - legacy mode,  1-gpiomode */
		{LCDC_REG_GPSELR, 0x0000},

		{LCDC_REG_PARUP, 0x0001},

		{LCDC_REG_INTR, 0x0001},

		/* Please note that AMOLED is connected to LCDC GPIO
		for boardtile only */
		{LCDC_REG_GP3CNTR, 0x1},

		/* LCDC Enable */
		{LCDC_REG_LCDCCR, 0x0001},

		{LCDC_SEQ_END, 0}
	},
	.dispir = 0x0021,
	.pancsr = 0x000f,
	.videomode = {
		"amoled", 0, 480, 800, 0, 8, 5, 5, 5, 2, 1,
		0, FB_VMODE_NONINTERLACED, FB_MODE_IS_STANDARD
	},
};

static struct resource lcdc_resource[] = {
	{
		.start	= DMW_LCDC_BASE,
		.end	= DMW_LCDC_BASE + SZ_4K - 1,
		.flags	= IORESOURCE_MEM,
	},
	{
		.start	= DMW_IID_LCD_CONTROLLER,
		.flags	= IORESOURCE_IRQ,
	},
};

static struct platform_device lcdc_device = {
	.name		= "dw74fb",
	.num_resources	= ARRAY_SIZE(lcdc_resource),
	.resource	= lcdc_resource,
	.dev		= {
		.platform_data = &dmw_wvga_amoled_panel,
		.coherent_dma_mask = DMA_BIT_MASK(32),
	},
};

static struct i2c_pnx_data i2c0_data = {
	.name	= "pnx-i2c0",
	.base	= DMW_I2C1_BASE,
	.irq	= DMW_IID_I2C1,
	.scl	= GPIO_PORTC(28),
	.sda	= GPIO_PORTC(29),
};

static struct i2c_pnx_data i2c1_data = {
	.name	= "pnx-i2c1",
	.base	= DMW_I2C2_BASE,
	.irq	= DMW_IID_I2C2,
	.scl	= GPIO_PORTC(30),
	.sda	= GPIO_PORTC(31),
};

static struct platform_device i2c0_pnx_device = {
	.name	= "pnx-i2c",
	.id	= 0,
	.dev	= {
		.platform_data = &i2c0_data,
	},
};

static struct platform_device i2c1_pnx_device = {
	.name	= "pnx-i2c",
	.id	= 1,
	.dev	= {
		.platform_data = &i2c1_data,
	},
};

static struct lis3lv02d_platform_data lis3lv02d_pdata = {
	/* .irq_cfg = 0x9, */ /* Interrupt 1 source OR interrupt 2 source */
};

static struct adv7525_platform_data adv7525_pdata = {
	.power_gpio = GPIO_PORTG(11),
};

static struct i2c_board_info __initdata i2c0_devices[] = {
#if defined (CONFIG_VIDEO_CAMERA_OV3640)
	{
		I2C_BOARD_INFO("ov3640", OV3640_I2C_ADDR),
		.platform_data = &ov3640_platform_data,
	},
#elif defined (CONFIG_VIDEO_CAMERA_OV2643)
	{
		I2C_BOARD_INFO("ov2643", OV2643_I2C_ADDR),
		.platform_data = &ov2643_platform_data,
	},
#endif
	{
		I2C_BOARD_INFO("lis3lv02d", 0x19),
		.irq = DMW_IID_EXTERNAL_REQUEST_0,
		.platform_data = &lis3lv02d_pdata,
	},
	{
		I2C_BOARD_INFO("adv7525", 0x39),
		.irq = DMW_IID_EXTERNAL_REQUEST_8,
		.platform_data = &adv7525_pdata,
	},
};

static struct i2c_board_info __initdata wm897X_i2c_dev = {
	I2C_BOARD_INFO("wm897X", 0x1A),
};

static struct ssd2531_platform_data ssd2531_platform_data = {
	.reset_gpio = GPIO_PORTG(17),
	.panel = {
		.remap = SSD2531_REMAP_FLIP_X | SSD2531_REMAP_FLIP_Y,
		.driveline_num = 15,
		.senseline_num = 10,
		.x_scale = 0x47,
		.y_scale = 0x44,
		.min_x = 0,
		.min_y = 0,
		.max_x = 224,
		.max_y = 352,
		.drivepin_slew_rates[0] = 0x3,
		.drivepin_slew_rates[1] = 0x4,
		.drivepin_slew_rates[2] = 0x5,
		.drivepin_slew_rates[3] = 0x6,
		.drivepin_slew_rates[4] = 0x7,
		.drivepin_slew_rates[5] = 0x8,
		.drivepin_slew_rates[6] = 0x9,
		.drivepin_slew_rates[7] = 0xa,
		.drivepin_slew_rates[8] = 0xb,
		.drivepin_slew_rates[9] = 0xc,
		.drivepin_slew_rates[10] = 0xd,
		.drivepin_slew_rates[11] = 0xe,
		.drivepin_slew_rates[12] = 0xf,
		.drivepin_slew_rates[13] = 0x10,
		.drivepin_slew_rates[14] = 0x11,
	},
	.screen = {
		.min_x = 0,
		.min_y = 0,
		.max_x = 319,
		.max_y = 479,
	}
};

static struct i2c_board_info __initdata i2c1_devices[] = {
	{
		I2C_BOARD_INFO("ssd2531", 0x5c),
		.platform_data = &ssd2531_platform_data,
		.irq = DMW_IID_EXTERNAL_REQUEST_5,
	},
};

static struct dmw_keypad_config dmw_keypad_config = {
	.rows = 4,
	.cols = 2,
};

static struct resource dmw_keypad_resources[] = {
	{
		.start	= DMW_GPIO_BASE,
		.end	= DMW_GPIO_BASE + 0x3fff,
		.flags	= IORESOURCE_MEM,
	},
	{
		.start	= DMW_IID_KEYPAD,
		.end	= DMW_IID_KEYPAD,
		.flags	= IORESOURCE_IRQ,
	},
};

static struct platform_device keypad_device = {
	.name		= "dmw-keypad",
	.id		= 0,
	.num_resources	= ARRAY_SIZE(dmw_keypad_resources),
	.resource	= dmw_keypad_resources,
	.dev		= {
		.platform_data = &dmw_keypad_config,
	},
};

static struct resource dmw_uart0_resource[] = {
	{
		.start	= DMW_UART1_BASE,
		.end	= DMW_UART1_BASE + SZ_4K - 1,
		.flags	= IORESOURCE_MEM,
	},
	{
		.start	= DMW_IID_UART1,
		.end	= DMW_IID_UART1,
		.flags	= IORESOURCE_IRQ,
	}
};

static struct platform_device uart0_device = {
	.name		= "dw-uart",
	.id		= 0,
	.num_resources	= ARRAY_SIZE(dmw_uart0_resource),
	.resource	= dmw_uart0_resource,
};

static struct resource dmw96_macg_resources[] = {
	{
		.start	= DMW_MAC_BASE,
		.end	= DMW_MAC_BASE + SZ_4K + SZ_256 - 1,
		.flags	= IORESOURCE_MEM,
	},
	{
		.start	= DMW_IID_8023MAC,
		.flags	= IORESOURCE_IRQ,
		.name	= "dmw96_gmac-irq",
	},
};

static struct platform_device macg_device = {
	.name		= "dmw96_gmac",
	.num_resources	= ARRAY_SIZE(dmw96_macg_resources),
	.resource	= dmw96_macg_resources,
};

static struct resource dmw_nand_resources[] = {
	{
		.start	= DMW_NAND_LCD_BASE,
		.end	= DMW_NAND_LCD_BASE + SZ_4K - 1,
		.flags	= IORESOURCE_MEM,
	},
	{
		.start  = DMW_IID_FLASH_CONTROLLER,
		.flags  = IORESOURCE_IRQ,
	},
};

static struct platform_device nand_device = {
	.name		= "dmw_nand",
	.dev		= {
		.coherent_dma_mask = ~0,
	},
	.num_resources	= ARRAY_SIZE(dmw_nand_resources),
	.resource	= dmw_nand_resources,
};

static void reset_tdm0(int enable)
{
	if (enable)
		reset_set(RESET_TDM1);
	else
		reset_release(RESET_TDM1);
}

static void reset_tdm1(int enable)
{
	if (enable)
		reset_set(RESET_TDM2);
	else
		reset_release(RESET_TDM2);
}

static void reset_tdm2(int enable)
{
	if (enable)
		reset_set(RESET_TDM3);
	else
		reset_release(RESET_TDM3);
}

static struct dmw_tdm_pdata tdm0_pdata = {
	.reset = reset_tdm0,
};

static struct dmw_tdm_pdata tdm1_pdata = {
	.reset = reset_tdm1,
};

static struct dmw_tdm_pdata tdm2_pdata = {
	.reset = reset_tdm2,
};

static struct resource tdm0_resources[] = {
	{
		.start	= DMW_IID_TDM1,
		.end	= DMW_IID_TDM1,
		.flags	= IORESOURCE_IRQ,
	},
	{
		.start	= DMW_TDM1_BASE,
		.end	= DMW_TDM1_BASE + (2*SZ_4K) - 1,
		.flags	= IORESOURCE_MEM,
	},
	{
		.start	= 8, /* capture -> TDM1 RX FIFO */
		.end	= 8,
		.flags	= IORESOURCE_DMA,
	},
	{
		.start	= 9, /* playback -> TDM1 TX FIFO */
		.end	= 9,
		.flags	= IORESOURCE_DMA,
	},
};

static struct resource tdm1_resources[] = {
	{
		.start	= DMW_IID_TDM2,
		.end	= DMW_IID_TDM2,
		.flags	= IORESOURCE_IRQ,
	},
	{
		.start	= DMW_TDM2_BASE,
		.end	= DMW_TDM2_BASE + (2*SZ_4K) - 1,
		.flags	= IORESOURCE_MEM,
	},
	{
		.start	= 10, /* capture -> TDM2 RX FIFO */
		.end	= 10,
		.flags	= IORESOURCE_DMA,
	},
	{
		.start	= 11, /* playback -> TDM2 TX FIFO */
		.end	= 11,
		.flags	= IORESOURCE_DMA,
	},
};

static struct resource tdm2_resources[] = {
	{
		.start	= DMW_IID_TDM3,
		.end	= DMW_IID_TDM3,
		.flags	= IORESOURCE_IRQ,
	},
	{
		.start	= DMW_TDM3_BASE,
		.end	= DMW_TDM3_BASE + (2*SZ_4K) - 1,
		.flags	= IORESOURCE_MEM,
	},
	{
		.start	= 12, /* capture -> TDM3 RX FIFO */
		.end	= 12,
		.flags	= IORESOURCE_DMA,
	},
	{
		.start	= 13, /* playback -> TDM3 TX FIFO */
		.end	= 13,
		.flags	= IORESOURCE_DMA,
	},
};

static struct platform_device tdm0_device = {
	.name		= "dmw-tdm",
	.id		= 0,
	.dev		= {
		.platform_data = &tdm0_pdata,
	},
	.num_resources	= ARRAY_SIZE(tdm0_resources),
	.resource	= tdm0_resources,
};

static struct platform_device tdm1_device = {
	.name		= "dmw-tdm",
	.id		= 1,
	.dev		= {
		.platform_data = &tdm1_pdata,
	},
	.num_resources	= ARRAY_SIZE(tdm1_resources),
	.resource	= tdm1_resources,
};

static struct platform_device tdm2_device = {
	.name		= "dmw-tdm",
	.id		= 2,
	.dev		= {
		.platform_data = &tdm2_pdata,
	},
	.num_resources	= ARRAY_SIZE(tdm2_resources),
	.resource	= tdm2_resources,
};

static struct dwc_otg_platform_data usb1_pdata = {
	.drvvbus_gpio		= -1,
	.overcurrent_gpio	= -1,
	.vbusdetect_gpio	= GPIO_PORTF(20),
	.busy			= dmw_dram_request,
	.idle			= dmw_dram_release,
};

static struct resource usb1_resources[] = {
	{
		.start	= DMW_IID_USB1OTG_MC_NINT,
		.end	= DMW_IID_USB1OTG_MC_NINT,
		.flags	= IORESOURCE_IRQ,
	},
	{
		.start	= DMW_IID_USB1OTG_SOF_PULSE,
		.end	= DMW_IID_USB1OTG_SOF_PULSE,
		.flags	= IORESOURCE_IRQ,
	},
	{
		.start	= DMW_USB1_OTG_BASE,
		.end	= DMW_USB1_OTG_BASE + SZ_1M - 1,
		.flags	= IORESOURCE_MEM,
	},
};

static struct platform_device usb1_device = {
	.name		= "dwc_otg",
	.id		= 0,
	.dev		= {
		.coherent_dma_mask = DMA_BIT_MASK(32),
		.platform_data = &usb1_pdata,
	},
	.num_resources	= ARRAY_SIZE(usb1_resources),
	.resource	= usb1_resources,
};

#if defined(CONFIG_USB_ANDROID) || defined(CONFIG_USB_ANDROID_MODULE)
static char *android_usb_functions[] = {
#ifdef CONFIG_USB_ANDROID_ADB
	"adb",
#endif
#ifdef CONFIG_USB_ANDROID_MASS_STORAGE
	"usb_mass_storage",
#endif
#ifdef CONFIG_USB_ANDROID_ACM
	"acm",
#endif
#ifdef CONFIG_USB_ANDROID_RNDIS
	"rndis",
#endif
};

static struct android_usb_platform_data android_usb_data = {
	.vendor_id = 0x18D1,
	.product_id = 1,
	.product_name = "DMW96-IMH3",
	.manufacturer_name = "DSPG",
	.serial_number = "DSPG-DMW96-IMH3",
	.num_functions = ARRAY_SIZE(android_usb_functions),
	.functions = android_usb_functions,
};

static struct platform_device android_usb = {
	.name			= "android_usb",
	.id			= -1,
	.dev			= {
		.platform_data	= &android_usb_data,
	},
};

#ifdef CONFIG_USB_ANDROID_MASS_STORAGE
static struct usb_mass_storage_platform_data android_mass_storage = {
	.nluns = 1,
};

static struct platform_device android_usb_mass_storage = {
	.name			= "usb_mass_storage",
	.id			= -1,
	.dev			= {
		.platform_data	= &android_mass_storage,
	},
};
#endif

#ifdef CONFIG_USB_ANDROID_RNDIS
static struct usb_ether_platform_data rndis_usb_data = {
	.ethaddr	= { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05 },
	.vendorID	= 0,
	.vendorDescr	= "DSPG",
};

static struct platform_device rndis = {
	.name			= "rndis",
	.id			= -1,
	.dev			= {
		.platform_data	= &rndis_usb_data,
	},
};
#endif

#endif /* CONFIG_USB_ANDROID || CONFIG_USB_ANDROID_MODULE */

static void wifi_20Mhz_osc(int on)
{
	unsigned int val = readl(IO_ADDRESS(DMW_CMU_BASE) + DMW_CMU_SWOVRCTL);

	if (on)
		writel(val | 0x16, IO_ADDRESS(DMW_CMU_BASE) + DMW_CMU_SWOVRCTL);
	else
		writel(val & ~0x16, IO_ADDRESS(DMW_CMU_BASE) + DMW_CMU_SWOVRCTL);
}

static void wifi_cmu_reset(void)
{
	reset_set(RESET_WIFI);
	reset_release(RESET_WIFI);
}

static struct dwf_platform_data wifi_pdata = {
	.chip = DMW_CHIP_DMW96,
	.board = 1, /* DMW_BOARD_IMH3 */
	.reset = wifi_cmu_reset,
};

static struct resource wifi_resources[] = {
	[DWF_RESOURCE_APB] = {
		.start	= DMW_WIFI_APB_BASE,
		.end	= DMW_WIFI_APB_BASE + 0xA000 - 1,
		.flags	= IORESOURCE_MEM,
	},
	[DWF_RESOURCE_AHB] = {
		.start	= DMW_WIFI_AHB_BASE,
		.end	= DMW_WIFI_AHB_BASE + 0x214 - 1,
		.flags	= IORESOURCE_MEM,
	},
	[DWF_RESOURCE_ONCHIP] = {
		.start	= DMW_SRAM_WIFI_BASE,
		.end	= DMW_SRAM_WIFI_BASE + DMW_SRAM_WIFI_SIZE - 1,
		.flags	= IORESOURCE_MEM,
	},
	[DWF_RESOURCE_IRQ] = {
		.start	= DMW_IID_MAC,
		.end	= DMW_IID_MAC,
		.flags	= IORESOURCE_IRQ,
	},
};

static struct platform_device wifi_device = {
	.name = "dw-wifi",
	.id = -1,
	.dev.platform_data = &wifi_pdata,
	.num_resources	= ARRAY_SIZE(wifi_resources),
	.resource	= wifi_resources,
};

static struct cpufreq_frequency_table dmw_freq_table[] = {
	{
		.frequency	= 756000,
		.index		= 0,
	},
	{
		.frequency	= 864000,
		.index		= 1,
	},
	{
		.frequency	= CPUFREQ_TABLE_END,
	},
};

static struct dmw_cpufreq_pdata cpufreq_pdata = {
	.freq_table = dmw_freq_table,
	.overdrive_gpio = OVERDRIVE_GPIO,
};

static struct platform_device cpufreq_device = {
	.name	= "cpufreq-dmw",
	.id	= -1,
	.dev	= {
		.platform_data	= &cpufreq_pdata,
	},
};

static struct dmw_pm_pdata pm_pdata = {
	.overdrive_gpio = OVERDRIVE_GPIO,
};

static struct platform_device pm_device = {
	.name	= "dmw-pm",
	.id	= -1,
	.dev	= {
		.platform_data	= &pm_pdata,
	},
};

/* Bluetooth "codec" chip */
static struct platform_device csr8811_device = {
	.name	= "BT-csr8811",
	.id	= -1,
};

/* kernel 2.6.39 ASoC changes: gdma snd pcm device */
static struct platform_device dmw_pcm_device = {
	.name	= "dmw-pcm-audio",
	.id	= -1,
};

/* IMH3 with WM8976 board audio specifics */
static struct dmw_board_pdata dmw_imh3_wm8976_board_pdata = {
	.out_ana_sw    	= -1,
	.in_ana_sw 	= GPIO_PORTB(2),	
	.pa_gain   	= GPIO_PORTF(15),
	.pa_shutdn 	= GPIO_PORTC(26),
};
/* IMH3 with WM8978 board audio specifics */
static struct dmw_board_pdata dmw_imh3_wm8978_board_pdata = {
	.out_ana_sw	= -1,
	.in_ana_sw	= -1,	
	.pa_gain   	= GPIO_PORTF(15),
	.pa_shutdn 	= GPIO_PORTC(26),
};
/* kernel 2.6.39 ASoC changes: */
struct platform_device imh3_mach_device = {
	.name = "snd-imh3-mach-drv",
	.id = -1,
	.dev = {
		.platform_data = &dmw_imh3_wm8976_board_pdata,
	},
};

static struct platform_device *imh3_platform_devs[] __initdata = {
	&cpufreq_device,
	&i2c0_pnx_device,
	&i2c1_pnx_device,
	&keypad_device,
	&lcdc_device,
	&macg_device,
	&nand_device,
	&pm_device,
	&sdmmc_device,
	&spi_device,
	&tdm0_device,
	&tdm1_device,
	&tdm2_device,
	
	/* kernel 2.6.39 ASoC changes: */
	&dmw_pcm_device,   /* ASoC Platform (PCM) driver */
	&imh3_mach_device, /* ASoC IMH3 Machine driver */

	&uart0_device,
	&usb1_device,
	&vibrator_device,
	&wifi_device,
	&csr8811_device, /* ASoC Machine driver */
};

static int __init dmw_imh3_dev_init(void)
{
	uint32_t reg;

	if (!dmw_board_is("imh3"))
		return -ENODEV;

	if (dmw_board_has_feature("wm8978")) 
		imh3_mach_device.dev.platform_data = &dmw_imh3_wm8978_board_pdata;
	/*
	 * Configure pads according to the used devices
	 */
	reg = readl(IO_ADDRESS(DMW_SYSCFG_BASE) + DMW_SYSCFG_SPCR1);
	reg |= DMW_SYSCFG_SPCR1_BB_PAD_EN;
	reg |= DMW_SYSCFG_SPCR1_FC_PAD_EN;
	reg |= DMW_SYSCFG_SPCR1_UART1_PAD_EN;
	reg |= DMW_SYSCFG_SPCR1_TDM1_PAD_EN;
	reg |= DMW_SYSCFG_SPCR1_TDM2_PAD_EN;
	reg |= DMW_SYSCFG_SPCR1_TDM3_PAD_EN;
	reg |= DMW_SYSCFG_SPCR1_SPI1_PAD_EN;
	reg |= DMW_SYSCFG_SPCR1_I2C_1_PAD_EN;
	reg |= DMW_SYSCFG_SPCR1_I2C_2_PAD_EN;
	reg |= DMW_SYSCFG_SPCR1_SD_PAD_EN;
	reg |= DMW_SYSCFG_SPCR1_EMACA_PAD_EN;
	reg |= DMW_SYSCFG_SPCR1_LCDC_PAD_EN;
	reg |= DMW_SYSCFG_SPCR1_LCDGP_PAD2_EN;
	reg |= DMW_SYSCFG_SPCR1_LCDGP_PAD1_EN;
	reg |= DMW_SYSCFG_SPCR1_LCDGP_PAD0_EN;
	writel(reg, IO_ADDRESS(DMW_SYSCFG_BASE) + DMW_SYSCFG_SPCR1);

	reg = readl(IO_ADDRESS(DMW_SYSCFG_BASE) + DMW_SYSCFG_SPCR2);
	reg |= DMW_SYSCFG_SPCR2_CIU_PAD_EN;
	reg |= DMW_SYSCFG_SPCR2_DP_CLK_PAD_EN;
	writel(reg, IO_ADDRESS(DMW_SYSCFG_BASE) + DMW_SYSCFG_SPCR2);

	/* Setup GPIOs */
	dmw_gpio_setup(gpio_setup_default, ARRAY_SIZE(gpio_setup_default));

	/*
	 * Release USB from reset.
	 */
	reset_set(RESET_USB1_PHY_POR);
	reset_set(RESET_USB1_PHY_PORT);
	reset_set(RESET_USB1_MAC_PHYIF);
	reset_set(RESET_USB1_MAC_RST);
	udelay(10); reset_release(RESET_USB1_PHY_POR);
	udelay(40); reset_release(RESET_USB1_PHY_PORT);
	udelay(10); reset_release(RESET_USB1_MAC_PHYIF);
	udelay(10); reset_release(RESET_USB1_MAC_RST);

	/*
	 * Register SPI and I2C devices.
	 */
	spi_register_board_info(imh3_spi_devs, ARRAY_SIZE(imh3_spi_devs));
	i2c_register_board_info(0, i2c0_devices, ARRAY_SIZE(i2c0_devices));
	i2c_register_board_info(1, i2c1_devices, ARRAY_SIZE(i2c1_devices));
	i2c_register_board_info(0, &wm897X_i2c_dev, 1);

	/* FIXME: remove once wifi driver manages clock itself */
	wifi_20Mhz_osc(true);

#if defined(CONFIG_USB_ANDROID) || defined(CONFIG_USB_ANDROID_MODULE)
	platform_device_register(&android_usb);
#ifdef CONFIG_USB_ANDROID_MASS_STORAGE
	platform_device_register(&android_usb_mass_storage);
#endif
#ifdef CONFIG_USB_ANDROID_RNDIS
	platform_device_register(&rndis);
#endif
#endif

	if (dmw_dev_reserve_dmamem(&ciu_device.dev, DMW_RESERVEMEM_CAMERA, 1) == 0)
		platform_device_register(&ciu_device);

	return platform_add_devices(imh3_platform_devs, ARRAY_SIZE(imh3_platform_devs));
}

arch_initcall(dmw_imh3_dev_init);
