/*
 *  linux/arch/arm/mach-dmw/arch.c
 *
 *  Copyright (C) 2010-2011 DSPG Technologies GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <linux/init.h>
#include <linux/device.h>
#include <linux/dma-mapping.h>
#include <linux/platform_device.h>
#include <linux/sysdev.h>
#include <linux/interrupt.h>
#include <linux/amba/bus.h>
#include <linux/amba/clcd.h>
#include <linux/delay.h>
#include <linux/ctype.h>
#include <linux/memblock.h>
#include <linux/regulator/machine.h>

#include <asm/system.h>
#include <mach/hardware.h>
#include <mach/clock.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/hardware/arm_timer.h>
#include <asm/mach-types.h>
#include <asm/gpio.h>
#include <asm/setup.h>
#include <asm/memory.h>

#include <asm/mach/arch.h>
#include <asm/mach/irq.h>
#include <asm/mach/time.h>
#include <asm/mach/map.h>
#include <mach/gxp22xx.h>
#include "arch.h"
#include "irq.h"
#include "devices.h"

#if defined (CONFIG_DMW_BOARD_UCM62XX) || (CONFIG_DMW_BOARD_GXE51XX)
static char *dmw_board = "ucm6xxx";
#elif defined (CONFIG_DMW_BOARD_GXP22XX)
static char *dmw_board = "gxp22xx";
#else
static char *dmw_board = "unknown";
#endif

static char *dmw_board_features = "";

static int __init dmw_detect_board_commandline(char *line)
{
	dmw_board = line;
	line = strchr(dmw_board, ':');
	if (line) {
		*line++ = '\0';
		dmw_board_features = line;
	}

	return 1;
}

__setup("board=", dmw_detect_board_commandline);

char *dmw_board_get_type(void)
{
	return dmw_board;
}
EXPORT_SYMBOL(dmw_board_get_type);

bool dmw_board_is(char *board)
{
	return strcmp(dmw_board, board) == 0;
}
EXPORT_SYMBOL(dmw_board_is);

bool dmw_board_has_feature(char *feature)
{
	char *spot = strstr(dmw_board_features, feature);
	int len;

	/* Not found at all? */
	if (!spot)
		return false;

	/* Must be delimited by comma or string start/end */
	len = strlen(feature);
	if ((spot[-1]  != '\0' && spot[-1]  != ',') ||
	    (spot[len] != '\0' && spot[len] != ','))
		return false;

	return true;
}
EXPORT_SYMBOL(dmw_board_has_feature);

static int dmw_chip;
static int dmw_chip_rev;

static int dmw_detect_chip_rev_96(void)
{
	int ret;
	void *rom;

	ret = readl(IO_ADDRESS(DMW_SYSCFG_BASE) + DMW_SYSCFG_CHIP_REV) & 0xffff;
	if (ret > 0)
		return ret + 1;

	/*
	 * Revision A and B share the same code in SYSCFG CHIP_REV. :( Look
	 * into the bootrom code to distinguis both.
	 */
	rom = ioremap_nocache(DMW_BOOTROM_BASE, SZ_128K);
	if (!rom)
		return -1;

	switch (readb(rom + 68)) {
	case 0x30: ret = DMW_CHIP_DMW96_REV_A; break;
	case 0x90: ret = DMW_CHIP_DMW96_REV_B; break;
	default:   ret = -1;
	}

	iounmap(rom);
	return ret;
}

static void dmw_detect_chip(void)
{
	unsigned id;

	if (dmw_chip)
		return;

	id = readl(IO_ADDRESS(DMW_SYSCFG_BASE) + DMW_SYSCFG_CHIP_ID);

	switch (id) {
	case 0x0960:
		dmw_chip = DMW_CHIP_DMW96;
		dmw_chip_rev = dmw_detect_chip_rev_96();
		break;
	default:
		dmw_chip = DMW_CHIP_UNKNOWN;
	}
}

int dmw_get_chip(void)
{
	dmw_detect_chip();
	return dmw_chip;
}
EXPORT_SYMBOL(dmw_get_chip);

int dmw_get_chip_rev(void)
{
	dmw_detect_chip();
	return dmw_chip_rev;
}
EXPORT_SYMBOL(dmw_get_chip_rev);

/*
 * All IO addresses are mapped onto VA 0xFFFx.xxxx, where x.xxxx
 * is the (PA >> 12).
 */
static struct map_desc dmw_io_desc[] __initdata = {
	{
		.virtual	= IO_ADDRESS(DMW_PLICU_BASE),
		.pfn		= __phys_to_pfn(DMW_PLICU_BASE),
		.length		= SZ_4K,
		.type		= MT_DEVICE
	}, {
		.virtual	= IO_ADDRESS(DMW_CMU_BASE),
		.pfn		= __phys_to_pfn(DMW_CMU_BASE),
		.length		= SZ_4K,
		.type		= MT_DEVICE
	}, {
		.virtual	= IO_ADDRESS(DMW_SYSCFG_BASE),
		.pfn		= __phys_to_pfn(DMW_SYSCFG_BASE),
		.length		= SZ_4K,
		.type		= MT_DEVICE
	},
};

static void __init dmw_map_io(void)
{
	iotable_init(dmw_io_desc, ARRAY_SIZE(dmw_io_desc));
}

static void __init dmw_early_init(void)
{
	if (dmw_clk_init())
		panic("Clock init failed!");
}

#if defined(CONFIG_EMMC_FLASH)
static int gs_storage = GS_STORAGE_EMMC;
#elif defined(CONFIG_NAND_FLASH)
static int gs_storage = GS_STORAGE_FLASH;
#else
static int gs_storage = GS_STORAGE_EMMC;
#endif
static int __init early_storage(char *p)
{
	switch(*p)
	{
	case 'n':
	case 'N':
		gs_storage = GS_STORAGE_FLASH;
		break;
	case 'e':
	case 'E':
		gs_storage = GS_STORAGE_EMMC;
		break;
	default:
		break;
	}
	return 0;
}
early_param("storage", early_storage);

int gs_get_storage()
{
	return gs_storage;
}
EXPORT_SYMBOL(gs_get_storage);

static const char *boot_mode[]=
{
	"NAND",
	"NAND(no ECC)",
	"JTAG",
	"Testing",
	"eMMC",
	"UART",
	"Reserved",
	"Reserved",
};

void gs_storage_from_ucfg(void)
{
	unsigned int value;
	value = readl(IO_ADDRESS(DMW_CMU_BASE) + DMW_CMU_RSTSR);
	value >>= 9;
	value &= 0x7;

	printk("Boot mode: %s\n", boot_mode[value]);
	switch(value)
	{
	case 0:
	case 1:
		gs_storage = GS_STORAGE_FLASH;
		break;
	case 3:
		gs_storage = GS_STORAGE_EMMC;
		break;
	default:
		break;
	}
	return;
}

static void __init dmw_init(void)
{
	WARN(strcmp(dmw_board, "unknown") == 0, "No valid board type specified");

	printk("Board: %s\n", dmw_board);
	printk("Board features: %s\n", dmw_board_features);

	gs_storage_from_ucfg();

	regulator_has_full_constraints();

	/*
	 * build-time check for correct CONSISTENT_DMA_SIZE; the SZ_4M is for
	 * the peripherals in dmw_io_desc[] (IO_ADDRESS() mappings start at
	 * VMALLOC_END).
	 */
	BUILD_BUG_ON(VMALLOC_END + (SZ_4M) > CONSISTENT_BASE);
}

extern struct sys_timer dmw_timer;

static phys_addr_t dmw_reservemem_base;
static phys_addr_t dmw_reservemem_next;
static size_t dmw_reservemem_size;

static size_t __init dmw_reservemem_remain(void)
{
	size_t used = dmw_reservemem_next - dmw_reservemem_base;
	return dmw_reservemem_size - used;
}

int __init dmw_reservemem_get(phys_addr_t *pa, size_t size)
{
	if (!size)
		return -EINVAL;

	size = roundup(size, SZ_1M);

	if (dmw_reservemem_remain() < size)
		return -ENOMEM;

	*pa = dmw_reservemem_next;
	dmw_reservemem_next += size;
	return 0;
}

static void __init dmw_reserve(void)
{
	dmw_reservemem_size =
		DMW_RESERVEMEM_GPU +
		DMW_RESERVEMEM_CAMERA +
		DMW_RESERVEMEM_MEMALLOC +
		DMW_RESERVEMEM_CSS;

	dmw_reservemem_base = dmw_reservemem_next =
		memblock_alloc(dmw_reservemem_size, SZ_1M);
	memblock_free(dmw_reservemem_base, dmw_reservemem_size);
	memblock_remove(dmw_reservemem_base, dmw_reservemem_size);
}

MACHINE_START(DMW96, "DSPG DMW96")
	.boot_params	= 0x40000100,
	.map_io		= dmw_map_io,
	.init_early	= dmw_early_init,
	.init_machine	= dmw_init,
	.init_irq	= dmw_init_irq,
	.timer		= &dmw_timer,
	.reserve	= dmw_reserve,
MACHINE_END

